import sys, time, random, copy
from settings import *

class GameState:
    # Initializer for the Connect4 GameState
    # Board is initialized to size width*height

    def __init__(self, rows, cols):
        self.__rows  = rows         # number of rows in the board
        self.__cols = cols          # number of columns in the board
        self.__pieces = [0]*cols    # __pieces[c] = number of pieces in a column c
        self.__player = 0           # the current player to move, 0 = Player One, 1 = Player Two
        self.__board   = [[PLAYER_NONE]*cols for r in range(rows)]

    # performs the given move, putting the piece into the appropriate column and swapping the player
    def do_move(self, move):
        if not self.is_legal(move): 
            print("DOING ILLEGAL MOVE: ", move)
            sys.exit()
        self.__board[self.pieces(move)][move] = self.player_to_move()
        self.__pieces[move] += 1
        self.__player = (self.__player + 1) % 2

    # takes a piece off the top of a pile
    def undo_move(self, move):
        if self.__pieces[move] == 0: return
        self.__pieces[move] -= 1
        self.__board[self.__pieces[move]][move] = PLAYER_NONE
        self.__player = (self.__player + 1) % 2
    
    def get(self, r, c):        return self.__board[r][c]   # piece type located at (r,c)
    def cols(self):             return self.__cols          # number of columns in board
    def rows(self):             return self.__rows          # number of rows in board
    def pieces(self, col):      return self.__pieces[col]   # number of pieces in a given column
    def total_pieces(self):     return sum(self.__pieces)   # total pieces on the board
    def player_to_move(self):   return self.__player        # the player to move next

    # a move (placing a piece into a given column) is legal if the column isn't full
    def is_legal(self, move):   return move >= 0 and move < self.cols() and self.__pieces[move] < self.rows()
    # returns a list of legal moves at this state (which columns aren't full yet)
    def get_legal_moves(self):  return [i for i in range(self.cols()) if self.is_legal(i)]

    # Student TODO: Implement
    #   Calculates a heuristic evaluation for the current GameState from the P.O.V. of a given player
    #
    #   Args:
    #
    #     player (int) - The player whose point of view you want to evaulate from
    #
    #   Returns:
    #     
    #     value  (int) - A heuristic evaluation of the current GameState
    #
    #     Suggested return values:
    #     Large positive value  = Player is winning the game (infinity if player has won)
    #     Larger negative value = Opponent is winning the game (-infinity if player has lost)
    #                             Infinity = Some large integer > non-win evaluations 
    def eval(self, player):
        winner = self.winner()
        if winner == player:           return  10000 - self.total_pieces()
        elif winner == ((player+1)%2): return -10000 + self.total_pieces()
        else: return self.possible_fours(player) - self.possible_fours((player + 1) % 2)

    def possible_fours(self, player):
        count = 0
        enemy = (player + 1) % 2
        # count the horizontal possibilities
        for c in range(self.cols()-3):
            for r in range(self.rows()):
                count += 1 if sum([1 for i in range(4) if self.get(r, c+i) != enemy]) == 4 else 0
        # count the vertical possibilities
        for c in range(self.cols()):
            for r in range(self.rows()-3):
                count += 1 if sum([1 for i in range(4) if self.get(r+i, c) != enemy]) == 4 else 0
        # diagonal up-right checks
        for c in range(self.cols()-3):
            for r in range(self.rows()-3):
                count += 1 if sum([1 for i in range(4) if self.get(r+i, c+i) != enemy]) == 4 else 0
        # diagonal down-right checks
        for c in range(self.cols()-3):
            for r in range(self.rows()-1, 2, -1):
                count += 1 if sum([1 for i in range(4) if self.get(r-1, c+i) != enemy]) == 4 else 0
        return count

    # Student TODO: Implement
    #   Calculates whether or not there is a winner on the current board and returns one of the following values
    #
    #   Return PLAYER_ONE  (0) - Player One has won the game 
    #   Return PLAYER_TWO  (1) - Player Two has won the game
    #   Return PLAYER_NONE (2) - There is no winner yet and the board isn't full
    #   Return DRAW        (3) - There is no winner and the board is full
    #
    #   A Player has won a connect 4 game if they have 4 pieces placed in a straight line or on a diagonal
    #   REMEMBER: The board rows and columns can be any size, make sure your checks acccount for this 
    #   TIP: Create 4 seprate loops to check win formations: horizontal, vertical, diagonal up, diagonal down 
    #        Be sure to test this function extensively, if you don't detect wins correctly it will be bad
    def winner(self):
        # horizontal win checks
        for c in range(self.cols()-3):
            for r in range(self.rows()):
                if self.get(r,c) == PLAYER_NONE: continue
                if self.get(r,c) == self.get(r, c+1) == self.get(r, c+2) == self.get(r, c+3): return self.get(r,c)
        # vertical win checks
        for c in range(self.cols()):
            for r in range(self.rows()-3):
                if self.get(r,c) == PLAYER_NONE: continue
                if self.get(r,c) == self.get(r+1,c) == self.get(r+2,c) == self.get(r+3,c): return self.get(r,c)
        # diagonal up-right checks
        for c in range(self.cols()-3):
            for r in range(self.rows()-3):
                if self.get(r,c) == PLAYER_NONE: continue
                if self.get(r,c) == self.get(r+1,c+1) == self.get(r+2,c+2) == self.get(r+3,c+3): return self.get(r,c)
        # diagonal down-right checks
        for c in range(self.cols()-3):
            for r in range(self.rows()-1, 2, -1):
                if self.get(r,c) == PLAYER_NONE: continue
                if self.get(r,c) == self.get(r-1,c+1) == self.get(r-2,c+2) == self.get(r-3,c+3): return self.get(r,c)
        # no winner has been found, so return DRAW if the board is full or PLAYER_NONE if not full
        return DRAW if (self.total_pieces() == self.rows() * self.cols()) else PLAYER_NONE


class Player_AlphaBeta:

    # Constructor for the Player_AlphaBeta class
    #
    # Ideally, this object should be constructed once per player, and then the get_move function will be
    # called once per turn to get the move the AI should do for a given state
    #
    # Args:
    #
    #  depth      (int) - Max depth for the AB search. If 0, no limit is used for depth
    #  time_limit (int) - Time limit (in ms) for the AB search. If 0, no limit is used for time
    #
    #  NOTE: One or both of depth or time_limit must be set to a value > 0
    def __init__(self, max_depth, time_limit):
        self.max_depth = max_depth
        self.current_max_depth = max_depth
        self.time_limit_ms = time_limit
        self.last_query_time = 0
        self.reset()

    def reset(self):
        self.timeout = 0
        self.best_move = -1
        self.temp_best_move = -1
        self.best_move_value = -1000000
        self.nodes = 0
        self.values = []

    # Student TODO: Implement this function
    #
    # This function calculates the move to be perfomed by the AI at a given state
    # This function will (ideally) call your alpha_beta recursive function from the the root node
    #
    # Args:
    #
    #   state (GameState) - The current state of the Connect4 game, with the AI next to move
    #
    # Returns:
    #
    #   move (int)        - The move the AI should do at this state. The move integer corresponds to
    #                       which column to place the next piece into (0 is the left-most column)
    def get_move(self, state):
        self.reset()
        self.player = state.player_to_move()
        self.last_query_time = time.clock()
        self.id_alpha_beta(state)
        return self.best_move

    def is_terminal(self, state, depth):
        if (self.current_max_depth > 0 and depth >= self.current_max_depth):
            return True
                
        return state.winner() != PLAYER_NONE

    def id_alpha_beta(self, state):
        max_d = self.max_depth if (self.max_depth > 0) else (state.rows() * state.cols() - state.total_pieces())
        for depth in range(1, max_d + 1):
            #print(depth, end = ' ')
            try:
                self.values = []
                self.current_max_depth = depth
                self.best_move_value = -1000000
                self.alpha_beta(copy.deepcopy(state), 0, -100000, 100000, True)
                self.best_move = self.temp_best_move
                #print(self.values, self.best_move)
            except:
                break
        #print ("Best Move: ", self.best_move)
        return self.best_move

    def alpha_beta(self, state, depth, alpha, beta, max_player):
        self.nodes += 1
        if self.is_terminal(state, depth):
            return state.eval(self.player)
        if (self.time_limit_ms > 0):
            elapsed = (time.clock() - self.last_query_time)*1000
            if (elapsed > self.time_limit_ms): raise Exception("TIMEOUT")
        for move in state.get_legal_moves():
            state.do_move(move)
            value = self.alpha_beta(state, depth+1, alpha, beta, not max_player)
            if (depth == 0): self.values.append(value)
            state.undo_move(move)
            if max_player and (value > alpha):
                if (depth == 0): self.temp_best_move = move
                alpha = value
            elif not max_player and (value < beta):  
                beta = value
            if alpha >= beta: break
        return alpha if max_player else beta
