#include "ReadWrite.h"
#include <random>

using namespace insanitybot;

insanitybot::ReadWrite::ReadWrite()
{
}

void insanitybot::ReadWrite::initialize()
{
	buildOrders[BWAPI::Races::Zerg] = { "Nuke", "BioDrops", "FiveFacGol" };
	buildOrders[BWAPI::Races::Protoss] = { "OneFacAllIn", "Mech", "GreedMech" };
	buildOrders[BWAPI::Races::Terran] = { "MechVT", "MechAllIn", "Mech", "GreedMech" };
	buildOrders[BWAPI::Races::Unknown] = { "OneFacAllIn", "Mech", "MechAllIn" };
}

std::vector<MatchData>  insanitybot::ReadWrite::readCompactMatchData() {
	std::string filePath = "bwapi-data/read/" + BWAPI::Broodwar->enemy()->getName() + ".txt";
	std::vector<MatchData> matches;

	std::ifstream inFile(filePath);
	if (inFile.is_open()) {
		std::string line;
		while (std::getline(inFile, line)) {
			std::istringstream iss(line);
			std::string mapName, startPositionStr, buildOrder, resultStr;

			if (iss >> mapName >> startPositionStr >> buildOrder >> resultStr) {
				MatchData match;
				// Parse map name
				match.mapName = mapName;

				// Parse starting position
				int x, y;
				sscanf(startPositionStr.c_str(), "%d,%d", &x, &y);
				match.startPosition = BWAPI::Position(x, y);

				// Parse build order
				match.buildOrder = buildOrder;

				// Parse result
				match.wonGame = (resultStr == "W");

				matches.push_back(match);
			}
		}
		inFile.close();
	}

	return matches;
}

void  insanitybot::ReadWrite::writeCompactMatchData(const std::string& opponentName, const std::string& mapName, const BWAPI::Position& startPosition, std::string buildOrder, bool wonGame) {
	std::string filePath = "bwapi-data/write/" + opponentName + ".txt";

	std::ofstream outFile(filePath, std::ios::app);
	if (outFile.is_open()) {
		// Write compact format: "mapId startPosition buildOrder result"
		outFile << mapName << " ";
		outFile << startPosition.x << "," << startPosition.y << " ";
		outFile << buildOrder << " ";
		outFile << (wonGame ? 'W' : 'L') << "\n";
		outFile.close();
	}
	else {
		BWAPI::Broodwar << "Error opening file for writing: " << filePath << std::endl;
	}
}

std::string  insanitybot::ReadWrite::selectBuildOrder(const std::vector<MatchData>& matchHistory, const std::string& currentMap, const BWAPI::Position& currentStartPosition) {
	std::unordered_map<std::string, int> buildOrderWinLoss;
	// Count wins and losses for each build order on the same map and start position
	for (const auto& match : matchHistory) {
		if (match.mapName == currentMap && match.startPosition == currentStartPosition) {
			if (match.wonGame) {
				buildOrderWinLoss[match.buildOrder]++;
			}
			else {
				buildOrderWinLoss[match.buildOrder]--;
			}
		}
	}

	// If no data, fall back to random build order
	if (buildOrderWinLoss.empty()) {
		return selectRandomBuildOrder();
	}

	// Build vectors for weighted random selection
	std::vector<std::string> buildOrders;
	std::vector<double> weights;

	// Calculate weights: Higher scores result in higher weight
	for (const auto& entry : buildOrderWinLoss) {
		buildOrders.push_back(entry.first);

		// A simple formula to convert win/loss score into weight
		// We use std::max(1, score) to ensure a non-zero weight (minimum 1)
		weights.push_back(static_cast<double>(std::max(1, entry.second)));
	}

	// Set up random generator and distribution
	std::random_device rd;
	std::mt19937 gen(rd());  // Mersenne Twister random generator
	std::discrete_distribution<> weightedDist(weights.begin(), weights.end());

	// Perform weighted random selection
	int selectedIndex = weightedDist(gen);
	BWAPI::Broodwar << "Read data detected, build " << buildOrders[selectedIndex] << " chosen." << std::endl;
	return buildOrders[selectedIndex];
}

std::string insanitybot::ReadWrite::selectRandomBuildOrder() {
	// Look up build orders for the opponent's race
	if (buildOrders.find(BWAPI::Broodwar->enemy()->getRace()) != buildOrders.end()) {
		const std::vector<std::string>& orders = buildOrders[BWAPI::Broodwar->enemy()->getRace()];
		// Select a random build order
		int randomIndex = std::rand() % orders.size();
		BWAPI::Broodwar << "Random build selected " << orders[randomIndex] << " chosen." << std::endl;
		return orders[randomIndex];
	}
	else {
		// Fallback case if opponent's race is unknown or not in the map
		return "mech";  // You can define a default build order
	}
}

std::string insanitybot::ReadWrite::getChosenBuildOrder(BWAPI::Position startPosition)
{
	return selectBuildOrder(readCompactMatchData(), BWAPI::Broodwar->mapFileName(), startPosition);
}


ReadWrite & ReadWrite::Instance()
{
	static ReadWrite instance;
	return instance;
}