#pragma once

// Overlay a coarser square grid over the map's tile grid.
// It's a template class that stores a pointer to a data record in each cell.
// All pointers are initially null.
// When the caller creates a data record and calls set(), Overgrid takes ownership
// of the record and will free it when appropriate.

#include <vector>
#include <BWAPI.h>

namespace UAlbertaBot
{
class Cell
{
public:
	Cell() {};
	~Cell() {};
};

template<class Cell>
class Overgrid
{
private:
	int _height;		// number of cells horizontally
	int _width;			// number of cells vertically
	int _size;			// side length of each cell in tiles

	std::vector< std::vector<Cell *> > _grid;

public:
	// Create an empty grid. All pointers are initially null.
	Overgrid(int size)
		: _size(size)
		, _height(BWAPI::Broodwar->mapHeight() / size + (BWAPI::Broodwar->mapHeight() % size > 0 ? 1 : 0))
		, _width(BWAPI::Broodwar->mapWidth() / size + (BWAPI::Broodwar->mapWidth() % size > 0 ? 1 : 0))
	{
		_grid = std::vector< std::vector<Cell *> >(_width, std::vector<Cell *>(_height, nullptr));
	}

	~Overgrid()
	{
		for (const std::vector<Cell *> row : _grid)
		{
			for (Cell * cell : row)
			{
				delete cell;
			}
		}
	}

	int width() const { return _width; }
	int height() const { return _height; }
	int size() const { return _size; }
	bool valid(int x, int y)
	{
		return x >= 0 && x < _width && y >= 0 && y < _height;
	}

	std::pair<int, int> toXY(const BWAPI::TilePosition & pos) const
	{
		return std::pair(pos.x / _size, pos.y / _size);
	}

	BWAPI::TilePosition fromXY(int x, int y) const
	{
		return BWAPI::TilePosition(x * _size + _size / 2, y * _size + _size / 2).makeValid();
	}

	// Once a cell is set, the Overgrid owns its memory and will delete it when appropriate.
	void set(int x, int y, Cell * item)
	{
		if (_grid[x][y])
		{
			delete _grid[x][y];
		}
		_grid[x][y] = item;
	}

	// The caller may freely alter the contents of the cell, but not delete it.
	// Once a cell is set, the Overgrid owns its memory.
	Cell * get(int x, int y) const
	{
		return _grid[x][y];
	}
};

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --
// Test objects.
// Unused unless you add code to use it for testing.

class TestCell : Cell
{
public:
	BWAPI::Color color;
};

class OvergridTest
{
	Overgrid<TestCell> g;

public:
	OvergridTest();
	void update();
};
}
