#pragma once

#include "ResourceInfo.h"
#include "UnitData.h"

namespace UAlbertaBot
{
class Zone;

class InformationManager
{
    BWAPI::Player	_self;
    BWAPI::Player	_enemy;

    bool			_weHaveCombatUnits;
	bool			_weHaveAirUnits;

    bool			_enemyHasCombatUnits;
    bool			_enemyHasStaticAntiAir;
    bool			_enemyHasAntiAir;
    bool			_enemyHasAirTech;
	bool			_enemyHasAirToGround;
    bool			_enemyHasCloakTech;
    bool            _enemyCloakedUnitsSeen;
    bool			_enemyHasMobileCloakTech;
    bool			_enemyHasAirCloakTech;
    bool			_enemyHasOverlordHunters;
    bool			_enemyHasStaticDetection;
    bool			_enemyHasMobileDetection;
	bool			_enemyHasTransport;

	bool			_enemyHasRestoration;
    bool			_enemyHasSiegeMode;
	bool            _enemyHasMines;
	bool            _enemyHasIrradiate;
	bool            _enemyHasEMP;
	bool            _enemyHasNuke;

	bool            _enemyHasStorm;
	bool            _enemyHasMaelstrom;
	bool            _enemyHasStasis;
	bool            _enemyHasRecall;

	bool			_enemyHasLurkers;
	bool			_enemyHasPlague;

	int             _enemyGasTiming;

	const int BumkerShouldFireRange = 5 * 32;		// used for telling whether an enemy bunker is loaded
	BWAPI::Unitset  _loadedBunkers;					// enemy bunkers that are loaded and may shoot at us

    std::map<BWAPI::Player, UnitData>                   _unitData;
    std::map<BWAPI::Player, std::set<const Zone *> >	_occupiedRegions;	// contains any building
    BWAPI::Unitset										_staticDefense;
    BWAPI::Unitset										_ourPylons;
    std::map<BWAPI::Unit, BWAPI::Unitset>				_yourTargets;		// our unit -> [enemy units targeting it]
    BWAPI::Unitset                                      _enemyScans;

    // Track a resource container (mineral patch or geyser) by its initial static unit.
    // A mineral patch unit will disappear when it is mined out. A geyser unit will change when taken.
	// The initial static units stay the same, and we remember them here.
    std::map<BWAPI::Unit, ResourceInfo> _resources;

    InformationManager();

    void initializeRegionInformation();
    void initializeResources();

    void maybeClearNeutral(BWAPI::Unit unit);

    void maybeAddStaticDefense(BWAPI::Unit unit);

    void updateUnit(BWAPI::Unit unit);
    void updateUnitInfo();
    void updateBaseLocationInfo();
    void updateOccupiedRegions(const Zone * zone, BWAPI::Player player);
    void updateGoneFromLastPosition();
    void updateYourTargets();
    void updateBullets();
    void updateResources();
    void updateEnemyGasTiming();
    void updateTerranEnemyAbilities();

	// Fill the given set with every enemy bunker within marine range of any of our units.
	void getEnemyBunkersInRange(BWAPI::Unitset & bunkers) const;

public:

    void                    initialize();

    void                    update();

    // event driven stuff
    void					onUnitShow(BWAPI::Unit unit)        { updateUnit(unit); }
    void					onUnitHide(BWAPI::Unit unit)        { updateUnit(unit); }
    void					onUnitCreate(BWAPI::Unit unit)		{ updateUnit(unit); }
    void					onUnitComplete(BWAPI::Unit unit)    { updateUnit(unit); maybeAddStaticDefense(unit); }
    void					onUnitMorph(BWAPI::Unit unit)       { updateUnit(unit); }
    void					onUnitRenegade(BWAPI::Unit unit)    { updateUnit(unit); maybeClearNeutral(unit); }
    void					onUnitDestroy(BWAPI::Unit unit);

    bool					isEnemyBuildingInRegion(const Zone * region);

    void                    getNearbyForce(std::vector<UnitInfo> & unitsOut, BWAPI::Position p, BWAPI::Player player, int radius) const;

    const UIMap &           getUnitInfo(BWAPI::Player player) const;

    std::set<const Zone *> &getOccupiedRegions(BWAPI::Player player);

    int						getAir2GroundSupply(BWAPI::Player player) const;

    bool					weHaveCombatUnits();
	bool					weHaveAirUnits();

	// For some spells, figure out whether the enemy has them by looking for their effects.
	void					updateEnemySpells();

    bool					enemyHasCombatUnits();
    bool					enemyHasStaticAntiAir();
    bool					enemyHasAntiAir();
    bool					enemyHasAirTech();
	bool					enemyHasAirToGround();
    bool                    enemyHasCloakTech();
    bool                    enemyCloakedUnitsSeen();
    bool                    enemyHasMobileCloakTech();
    bool                    enemyHasAirCloakTech();
    bool					enemyHasOverlordHunters();
    bool					enemyHasStaticDetection();
    bool					enemyHasMobileDetection();
	bool					enemyHasTransport();		// TODO does not work for zerg drop
	bool					enemyHasRestoration();		// TODO
    bool					enemyHasSiegeMode();
	bool					enemyHasMines();
	bool					enemyHasNuke();
	bool					enemyHasIrradiate() const { return _enemyHasIrradiate; };
	bool					enemyHasEMP() const { return _enemyHasEMP; };
	bool                    enemyHasStorm() const { return _enemyHasStorm; };
	bool					enemyHasMaelstrom() const { return _enemyHasMaelstrom; };
	bool					enemyHasStasis() const { return _enemyHasStasis; };
	bool					enemyHasRecall() const { return _enemyHasRecall; };		// TODO does not work, always false
	bool                    enemyHasLurkers();
	bool                    enemyHasPlague() const { return _enemyHasPlague; };
	int                     enemyGasTiming() const { return _enemyGasTiming; };

    bool                    weHaveCloakTech() const;

    void					enemySeenBurrowing();
    int                     getEnemyBuildingTiming(BWAPI::UnitType type) const;
    int                     remainingBuildTime(BWAPI::UnitType type) const;
    int                     getMySpireTiming() const;

    const BWAPI::Unitset &  getStaticDefense() const { return _staticDefense; };
    const BWAPI::Unitset &  getOurPylons() const { return _ourPylons; };
    const BWAPI::Unitset &  getEnemyScans() const { return _enemyScans; };

    BWAPI::Unit				nearestGroundStaticDefense(BWAPI::Position pos) const;
    BWAPI::Unit				nearestAirStaticDefense(BWAPI::Position pos) const;
    BWAPI::Unit				nearestShieldBattery(BWAPI::Position pos) const;

	bool					isEnemyBunkerLoaded(BWAPI::Unit bunker) const;

    int						nScourgeNeeded() const;           // zerg specific

    void                    drawExtendedInterface() const;
    void                    drawUnitInformation(int x,int y);
    void                    drawResourceAmounts() const;

    const UnitData &        getUnitData(BWAPI::Player player) const;
    const UnitInfo *        getUnitInfo(BWAPI::Unit unit) const;        // enemy units only
    const BWAPI::Unitset &	getEnemyFireteam(BWAPI::Unit ourUnit) const;
    int                     getResourceAmount(BWAPI::Unit resource) const;
    bool                    isMineralDestroyed(BWAPI::Unit resource) const;
    bool                    isGeyserTaken(BWAPI::Unit resource) const;

    static InformationManager & Instance();
};
}
