// AStar.cpp
// Short-distance pathfinding.

#include "AStar.h"

#include "Overgrid.h"
#include "The.h"

#include <queue>

using namespace UAlbertaBot;

// TODO _grid needs to be available throughout

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --
// For the Overgrid that keeps track of where we've looked.

class Place : public Cell
{
public:
	const Place * parent;
	const int x;
	const int y;
	const int distance;

	~Place() {};

	bool operator > (const Place & b) const { return distance > b.distance; };

	Place(int x0, int y0, int d)
		: parent(nullptr)
		, x(x0)
		, y(y0)
		, distance(d)
	{};

	Place(const Place * p, int x0, int y0, int d)
		: parent(nullptr)
		, x(x0)
		, y(y0)
		, distance(d)
	{};
};

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// Pretend the cluster has moved by the given delta.
// How much damage would it take in that location? Assume the units maintain
// the same relative positions.
int clusterAirDamage(const UnitCluster & cluster, int x, int y)
{
	Overgrid<Place> _grid(5);

	int damage = 0;

	// TODO fix - use real unit offsets from center

	for (BWAPI::Unit u : cluster.units)
	{
		BWAPI::TilePosition xy = _grid.fromXY(x, y);
		damage += the.airHits(xy.makeValid());
	}

	return damage;
}

// We found a goal. Trace back to the first move from the start--the daughter of the origin.
BWAPI::Position traceFirstMove(const Place * place)
{
	Overgrid<Place> _grid(5);

	const Place * p = place;
	while (p->parent && p->parent->parent)
	{
		p = p->parent;
	}
	return TileCenter(_grid.fromXY(p->x, p->y));
}

// Find an escape path for the air cluster that brings it to safety with
// the least possible damage, and return the first move to make on the path.
// If none, return BWAPI::Positions::None.
// There are many simplifications in the implementation.
BWAPI::Position UAlbertaBot::minimumDamageAirEscape(const UnitCluster & cluster)
{
	Overgrid<Place> _grid(5);

	std::vector<std::pair<int,int>> deltas =
	{ std::pair<int,int>(+1,  0)
	, std::pair<int,int>(+1,  0)
	, std::pair<int,int>( 0, +1)
	, std::pair<int,int>( 0, -1)
	, std::pair<int,int>(+1, +1)
	, std::pair<int,int>(+1, -1)
	, std::pair<int,int>(-1, +1)
	, std::pair<int,int>(-1, -1)
	};

	std::priority_queue<Place *, std::vector<Place *>, std::greater<Place *>> fringe;

	int damage = clusterAirDamage(cluster, 0 ,0);
	if (damage == 0)
	{
		return cluster.center;
	}
	std::pair<int, int> xy = _grid.toXY(BWAPI::TilePosition(cluster.center));
	Place * current = new Place(xy.first, xy.second, damage);
	fringe.push(current);

	// Outer loop: Choose the next node to expand.
	while (!fringe.empty())
	{
		current = fringe.top();

		// Inner loop: Expand the current node.
		for (std::pair<int, int> delta : deltas)
		{
			int x = current->x + delta.first;
			int y = current->y + delta.second;
			if (!_grid.valid(x, y))
			{
				continue;		// off the map
			}
			if (_grid.get(x, y))
			{
				continue;		// already looked here
			}

			damage = clusterAirDamage(cluster, x, y);
			if (damage == 0)
			{
				return traceFirstMove(current);
			}

			Place * neighbor = new Place(current, x, y, current->distance + damage);
			_grid.set(x, y, neighbor);
			fringe.push(neighbor);
		}

		fringe.pop();
	}

	return BWAPI::Positions::None;
}
