\
#include "The.h"

#include "Bases.h"
#include "GridTasks.h"
#include "InformationManager.h"
#include "MapGrid.h"
#include "OpeningTiming.h"
#include "OpponentModel.h"
#include "ParseUtils.h"
#include "ProductionManager.h"
#include "Random.h"
#include "StaticDefense.h"

using namespace UAlbertaBot;

The::The()
    : bases(Bases::Instance())
    , grid(MapGrid::Instance())
    , info(InformationManager::Instance())
    , openingTiming(OpeningTiming::Instance())
    , production(ProductionManager::Instance())
    , random(Random::Instance())
    , staticDefense(*new StaticDefense)
{
}

// -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --

// NOTE The skill kit is initialized in OpponentModel and updated in GameCommander.

void The::initialize()
{
    _selfRace = BWAPI::Broodwar->self()->getRace();

    // The order of initialization is important because of dependencies.
    partitions.initialize();
    inset.initialize();				// depends on partitions
    vWalkRoom.initialize();			// depends on edgeRange
    tileRoom.initialize();			// depends on vWalkRoom
    zone.initialize();				// depends on tileRoom
    map.initialize();

    bases.initialize();             // depends on map
    info.initialize();              // depends on bases
    placer.initialize();
    ops.initialize();

    // Read the opening timing file, to make opening decisions.
    // openingTiming.read();

    // Parse the bot's configuration file.
    // Change this file path (in config.cpp) to point to your config file.
    // Any relative path name will be relative to Starcraft installation folder
    // The config depends on the map and must be read after the map is analyzed.
    // This also reads the opponent model data and decides on the opening.
    ParseUtils::ParseConfigFile(Config::ConfigFile::ConfigFileLocation);

    // Sets the initial production queue to the book opening chosen above.
    production.initialize();

    // Add tasks that will run throughout the game.
    tasks.post(new GroundHitsFixedTask());
    tasks.post(new GroundHitsMobileTask());

    tasks.post(new AirHitsFixedTask());
    tasks.post(new AirHitsMobileTask());
}

// Hits from static defense, sieged tanks, and burrowed lurkers.
int The::staticHits(BWAPI::Unit unit, const BWAPI::TilePosition & tile) const
{
    return unit->isFlying()
        ? airHitsFixed.at(tile)
        : groundHitsFixed.at(tile);
}

// Hits from static defense, sieged tanks, and burrowed lurkers.
int The::staticHits(BWAPI::Unit unit) const
{
    return unit->isFlying()
        ? airHitsFixed.at(unit->getTilePosition())
        : groundHitsFixed.at(unit->getTilePosition());
}

// Hits from all enemies on ground units, static and mobile.
int The::groundHits(const BWAPI::TilePosition & tile) const
{
	return groundHitsFixed.at(tile) + groundHitsMobile.at(tile);
}

// Hits from all enemies on air units, static and mobile.
int The::airHits(const BWAPI::TilePosition & tile) const
{
	return airHitsFixed.at(tile) + airHitsMobile.at(tile);
}

// Air or ground hits.
int The::hits(bool air, const BWAPI::TilePosition & tile) const
{
	return air ? the.airHits(tile) > 0 : the.groundHits(tile);
}

// Air or ground hits.
int The::hits(bool air, const BWAPI::Position & pos) const
{
	return hits(air, BWAPI::TilePosition(pos));
}

// Hits from all enemies, static and mobile.
int The::hits(BWAPI::Unit unit, const BWAPI::TilePosition& tile) const
{
    return unit->isFlying()
        ? airHits(unit->getTilePosition())
        : groundHits(unit->getTilePosition());
}

// Hits from all enemies, static and mobile.
int The::hits(BWAPI::Unit unit) const
{
    return unit->isFlying()
        ? airHits(unit->getTilePosition())
        : groundHits(unit->getTilePosition());
}

// Start the ground path task: Somebody wants a safe path for a given base,
// so we need to update the paths for that base, and start the task if it's not done yet.
// The task goes on to update safe paths for every base (on different frames).
// This is typically called at the time of the first scout.
void The::startGroundPathTask(Base * base)
{
	Task * task = tasks.get("ground safe path");

	if (!task)
	{
		task = new GroundSafePathTask();
		tasks.post(task);
	}

	((GroundSafePathTask *)task)->updateBase(base);		// initial update so the results can be used immediately
	task->setNextUpdate(now() + 1);						// task will do future updates over time
}

// Start the air path task: Somebody wants a safe path for a given base,
// and we need to update the paths for that base.
// The task goes on to update safe paths for every base (on different frames).
// This is typically called when the first air unit comes out (immediately for zerg).
void The::startAirPathTask(Base * base)
{
	Task * task = tasks.get("air safe path");

	if (!task)
	{
		task = new AirSafePathTask();
		tasks.post(task);
	}

	((AirSafePathTask *)task)->updateBase(base);		// initial update so the results can be used immediately
	task->setNextUpdate(now() + 1);						// task will do future updates over time
}

void The::update()
{
    my.completed.takeSelf();
    my.all.takeSelfAll();
    your.seen.takeEnemy();
    your.ever.takeEnemyEver(your.seen);
    your.inferred.takeEnemyInferred(your.ever);

    ops.update();
    staticDefense.update();
    grid.update();
}

OpponentModel & The::oppModel()
{
	return OpponentModel::Instance();
}
