/**
 * Copyright (c) 2015-present, Facebook, Inc.
 * All rights reserved.
 *
 * This source code is licensed under the BSD-style license found in the
 * LICENSE file in the root directory of this source tree. An additional grant
 * of patent rights can be found in the PATENTS file in the same directory.
 */

#include "upcstorage.h"
#include "botcli-inl.h"

#include <torchcraft/state.h>
#include <torchcraft/client.h>

#define WIN32_LEAN_AND_MEAN
#include <Windows.h>
#undef min
#undef max
#include <BWAPI.h>

#include <chrono>
#include <memory>
#include <thread>
#include <optional>

using namespace torchcraft;

std::vector<uint8_t> groundHeightToVector()
{
  std::vector<uint8_t> v;
  for (int y = 0; y < BWAPI::Broodwar->mapHeight() * 4; ++y) {
    for (int x = 0; x < BWAPI::Broodwar->mapWidth() * 4; ++x) {
      v.push_back(BWAPI::Broodwar->getGroundHeight(x/4, y/4));
    }
  }
  return v;
}

std::vector<uint8_t> walkableToVector()
{
  std::vector<uint8_t> v;
  for (int y = 0; y < BWAPI::Broodwar->mapHeight() * 4; ++y) {
    for (int x = 0; x < BWAPI::Broodwar->mapWidth() * 4; ++x) {
      v.push_back(BWAPI::Broodwar->isWalkable(x, y));
    }
  }
  return v;
}

std::vector<uint8_t> buildableToVector()
{
  std::vector<uint8_t> v;
  for (int y = 0; y < BWAPI::Broodwar->mapHeight() * 4; ++y) {
    for (int x = 0; x < BWAPI::Broodwar->mapWidth() * 4; ++x) {
      v.push_back(BWAPI::Broodwar->isBuildable(x/4, y/4));
    }
  }
  return v;
}


void controllerInit(State* state) {
  auto bwl = bwapiLock();
  BWAPI::Broodwar->enableFlag(BWAPI::Flag::UserInput);
  BWAPI::Broodwar->setLatCom(false);

  state->lag_frames = BWAPI::Broodwar->getLatencyFrames();
  state->map_size[0] = BWAPI::Broodwar->mapWidth() * 4;
  state->map_size[1] = BWAPI::Broodwar->mapHeight() * 4;
  state->ground_height_data = groundHeightToVector();
  state->walkable_data = walkableToVector();
  state->buildable_data = buildableToVector();
  state->map_name = BWAPI::Broodwar->mapFileName();
  state->map_title = BWAPI::Broodwar->mapName();
  state->neutral_id = BWAPI::Broodwar->neutral()->getID();
  if (BWAPI::Broodwar->isReplay()) {
    state->replay = true;
  } else {
    state->replay = false;
    state->player_id = BWAPI::Broodwar->self()->getID();
  }
  state->battle_frame_count = 0;
  state->frame_from_bwapi = BWAPI::Broodwar->getFrameCount();
  for (auto loc : BWAPI::Broodwar->getStartLocations()) {
    BWAPI::WalkPosition walkPos(loc);
    state->start_locations.emplace_back(walkPos.x, walkPos.y);
  }
  for (const auto& p : BWAPI::Broodwar->getPlayers()) {
    State::PlayerInfo info;
    info.id = p->getID();
    auto bwrace = BW::Race::_from_integral_nothrow(p->getRace().getID());
    info.race = bwrace ? *bwrace : +BW::Race::Unknown;
    info.name = p->getName();
    info.is_enemy = p->isEnemy(BWAPI::Broodwar->self());
    info.has_left = false;
    state->player_info[info.id] = info;
  }
}


// clang-format off
enum Commands {
  // without args
  QUIT,
  RESTART,
  MAP_HACK,
  REQUEST_IMAGE,
  EXIT_PROCESS,
  NOOP,
  // one arg
  SET_SPEED,
  SET_LOG,
  SET_GUI,
  SET_FRAMESKIP,
  SET_CMD_OPTIM,
  SET_COMBINE_FRAMES, // one or two args actually
  SET_MAP,
  SET_MULTI,
  SET_BLOCKING,
  SET_MAX_FRAME_TIME_MS,
  // arguments are those of BWAPI::UnitCommand
  COMMAND_UNIT,
  COMMAND_UNIT_PROTECTED,
  // variable arguments
  COMMAND_USER, COMMAND_OPENBW,
  // BWAPI drawing routines
  DRAW_LINE, // x1, y1, x2, y2, color
  DRAW_UNIT_LINE, // uid1, uid2, color
  DRAW_UNIT_POS_LINE, // uid. x2, y2, color
  DRAW_CIRCLE, // x, y, radius, color
  DRAW_UNIT_CIRCLE, // uid, radius, color
  DRAW_TEXT, // x, y + text
  DRAW_TEXT_SCREEN, // x, y + text
  // last command id
  COMMAND_END
};
// clang-format on


enum CommandStatus : int8_t {
  SUCCESS = 0,
  // Positive numbers correspond to
  // BWAPI error codes from BWAPI::Errors::Enum | BWAPI_ERROR
  // (since an error code of 0 also signals an error in BWAPI)
  BWAPI_ERROR_MASK = 0x40,
  UNKNOWN_ERROR = -1,
  UNKNOWN_COMMAND = -2,
  MISSING_ARGUMENTS = -3,
  TOO_MANY_COMMANDS = -4,
  INVALID_UNIT = -5,
  PROTECTED = -6,
  OPENBW_NOT_IN_USE = -7,
  INVALID_PLAYER = -8,
  OPENBW_UNSUCCESSFUL_COMMAND = -9, // TODO reconsider whether we want this
};


const int pixelsPerWalkTile = 8;

BWAPI::Position getPositionFromWalkTiles(int x, int y) {
  return BWAPI::Position(pixelsPerWalkTile * x, pixelsPerWalkTile * y);
}

BWAPI::TilePosition getTilePositionFromWalkTiles(int x, int y) {
  return BWAPI::TilePosition(x / 4, y / 4);
}

void executeDrawCommands(const std::vector<std::pair<std::vector<int>, std::string>>& draw_cmds_) {
  for (const auto& cmdpair : draw_cmds_) {
    auto cmd = cmdpair.first;
    auto text = cmdpair.second;
    switch (cmd[0]) {
      case Commands::DRAW_LINE:
        BWAPI::Broodwar->drawLineMap(
            cmd.at(1), cmd.at(2), cmd.at(3), cmd.at(4), cmd.at(5));
        break;
      case Commands::DRAW_UNIT_LINE: {
        auto unit1 = BWAPI::Broodwar->getUnit(cmd.at(1));
        auto unit2 = BWAPI::Broodwar->getUnit(cmd.at(2));
        if (unit1 != nullptr && unit2 != nullptr && unit1->exists() &&
            unit2->exists()) {
          BWAPI::Broodwar->drawLineMap(
              unit1->getPosition(), unit2->getPosition(), cmd.at(3));
        }
        break;
      }
      case Commands::DRAW_UNIT_POS_LINE: {
        auto unit = BWAPI::Broodwar->getUnit(cmd.at(1));
        if (unit != nullptr && unit->exists()) {
          auto pos = unit->getPosition();
          BWAPI::Broodwar->drawLineMap(
              pos.x, pos.y, cmd.at(2), cmd.at(3), cmd.at(4));
        }
        break;
      }
      case Commands::DRAW_CIRCLE:
        BWAPI::Broodwar->drawCircleMap(
            cmd.at(1), cmd.at(2), cmd.at(3), cmd.at(4));
        break;
      case Commands::DRAW_UNIT_CIRCLE: {
        auto unit = BWAPI::Broodwar->getUnit(cmd.at(1));
        if (unit != nullptr && unit->exists()) {
          BWAPI::Broodwar->drawCircleMap(
              unit->getPosition(), cmd.at(2), cmd.at(3));
        }
        break;
      }
      case Commands::DRAW_TEXT:
        BWAPI::Broodwar->drawTextMap(cmd.at(1), cmd.at(2), text.c_str());
        break;
      case Commands::DRAW_TEXT_SCREEN:
        BWAPI::Broodwar->drawTextScreen(cmd.at(1), cmd.at(2), text.c_str());
        break;
    }
  }
}


int8_t handleCommand(
    int command,
    const std::vector<int>& args,
    const std::string& str) {
  int8_t status = CommandStatus::SUCCESS;
  auto check_args = [&](uint32_t n) {
    if (args.size() < n) {
      status = CommandStatus::MISSING_ARGUMENTS;
      return false;
    }
    return true;
  };
  auto check_unit = [&](int id) {
    auto res = BWAPI::Broodwar->getUnit(id);
    if (res == nullptr) {
      status = CommandStatus::INVALID_UNIT;
    }
    return res;
  };

  if (command <= Commands::NOOP) {
    switch (command) {
      case Commands::QUIT: // quit game
        BWAPI::Broodwar->leaveGame();
        return CommandStatus::SUCCESS;
//      case Commands::RESTART: // restart game
//        BWAPI::Broodwar->restartGame();
//        // Wait to finish game and start a new one if we're in the client...
//        if (BWAPI::BWAPIClient.isConnected()) {
//          while (BWAPI::Broodwar->isInGame()) {
//            BWAPI::BWAPIClient.update();
//            handleEvents();
//          }
//          while (!BWAPI::Broodwar->isInGame()) {
//            BWAPI::BWAPIClient.update();
//            handleEvents();
//          }
//        }
//        return CommandStatus::SUCCESS;
      case Commands::MAP_HACK: // remove fog of war, can only be done in onStart
                               // (at init)
        BWAPI::Broodwar->enableFlag(BWAPI::Flag::CompleteMapInformation);
        return CommandStatus::SUCCESS;
      case Commands::NOOP:
        return CommandStatus::SUCCESS;
    }
  } else if (command <= Commands::SET_MAX_FRAME_TIME_MS) {
    if (!check_args(1))
      return status;
    switch (command) {
      case Commands::SET_SPEED:
        BWAPI::Broodwar->setLocalSpeed(args[0]);
        return CommandStatus::SUCCESS;
      case Commands::SET_LOG:
        return CommandStatus::SUCCESS;
      case Commands::SET_GUI:
        BWAPI::Broodwar->setGUI(args[0] != 0);
        return CommandStatus::SUCCESS;
      case Commands::SET_FRAMESKIP:
        BWAPI::Broodwar->setFrameSkip(args[0]);
        // this->frameskips = args[0];
        return CommandStatus::SUCCESS;
      case Commands::SET_CMD_OPTIM:
        BWAPI::Broodwar->setCommandOptimizationLevel(args[0]);
        return CommandStatus::SUCCESS;
    }
  } else if (command <= Commands::COMMAND_UNIT_PROTECTED) {
    if (!check_args(2))
      return status;
    auto unit = check_unit(args[0]);
    if (unit == nullptr)
      return status;
    auto cmd_type = args[1];
    auto target =
        (args.size() >= 3 ? BWAPI::Broodwar->getUnit(args[2]) : nullptr);
    BWAPI::Position position = BWAPI::Positions::Invalid;
    BWAPI::TilePosition tposition = BWAPI::TilePositions::Invalid;
    if (args.size() >= 5) {
      // Some commands require tile position
      if (cmd_type == BWAPI::UnitCommandTypes::Build ||
          cmd_type == BWAPI::UnitCommandTypes::Land ||
          cmd_type == BWAPI::UnitCommandTypes::Build_Addon ||
          cmd_type == BWAPI::UnitCommandTypes::Place_COP)
        tposition = getTilePositionFromWalkTiles(args[3], args[4]);
      else
        position = getPositionFromWalkTiles(args[3], args[4]);
    }
    int x, y;
    if (position.isValid()) {
      x = position.x;
      y = position.y;
    } else if (tposition.isValid()) {
      x = tposition.x;
      y = tposition.y;
    } else {
      x = y = 0;
    }
    auto extra = (args.size() >= 6 ? args[5] : 0);
    switch (command) {
      case Commands::COMMAND_UNIT:
        if (!unit->issueCommand(
                BWAPI::UnitCommand(unit, cmd_type, target, x, y, extra))) {
          return CommandStatus::BWAPI_ERROR_MASK |
              BWAPI::Broodwar->getLastError().getID();
        }
        return CommandStatus::SUCCESS;
      case Commands::COMMAND_UNIT_PROTECTED:
        std::terminate();
    }
  } else if (command < Commands::COMMAND_END) {
    switch (command) {
      case Commands::DRAW_LINE:
      case Commands::DRAW_UNIT_LINE:
      case Commands::DRAW_UNIT_POS_LINE:
      case Commands::DRAW_CIRCLE:
      case Commands::DRAW_UNIT_CIRCLE:
      case Commands::DRAW_TEXT:
      case Commands::DRAW_TEXT_SCREEN: {
        static std::unordered_map<int, int> argcount = {
            {Commands::DRAW_LINE, 5},
            {Commands::DRAW_UNIT_LINE, 3},
            {Commands::DRAW_UNIT_POS_LINE, 4},
            {Commands::DRAW_CIRCLE, 4},
            {Commands::DRAW_UNIT_CIRCLE, 3},
            {Commands::DRAW_TEXT, 2},
            {Commands::DRAW_TEXT_SCREEN, 2},
        };
        if (!check_args(argcount[command]))
          return status;
        std::vector<int> cmd({command});
        cmd.insert(cmd.end(), args.begin(), args.end());
        executeDrawCommands({{cmd, str}});
        return CommandStatus::SUCCESS;
      }
    }
  }
  return CommandStatus::UNKNOWN_COMMAND;
}



/**
 * Pack some information about bullets.
 * The full list of BulletTypes is there
 * https://bwapi.github.io/namespace_b_w_a_p_i_1_1_bullet_types_1_1_enum.html
 */
void packBullets(replayer::Frame& f) {
  for (auto& b : BWAPI::Broodwar->getBullets()) {
    if (!b->getPosition().isValid())
      continue;
    f.bullets.push_back({b->getType(),
                         b->getPosition().x / pixelsPerWalkTile,
                         b->getPosition().y / pixelsPerWalkTile});
  }
}

/**
 * Pack information about resources.
 */
void packResources(replayer::Frame& f, BWAPI::PlayerInterface* p) {
  uint64_t upgrades = 0;
  uint64_t upgrades_level = 0;
  const auto NB_LVLABLE_UPGRADES = 16;
  for (auto up : BWAPI::UpgradeTypes::allUpgradeTypes()) {
    upgrades |= p->getUpgradeLevel(up) > 0 ? 1ll << up.getID() : 0;
    if (p->getUpgradeLevel(up) == 2)
      upgrades_level |= 1ll << up.getID();
    else if (p->getUpgradeLevel(up) == 3)
      upgrades_level |= 1ll << (up.getID() + NB_LVLABLE_UPGRADES);
  }
  uint64_t techs = 0;
  for (auto tt : BWAPI::TechTypes::allTechTypes()) {
    techs |= p->hasResearched(tt) ? 1ll << tt.getID() : 0;
  }

  f.resources[p->getID()] = {p->minerals(),
                             p->gas(),
                             p->supplyUsed(),
                             p->supplyTotal(),
                             upgrades,
                             upgrades_level,
                             techs};
}

/**
 * Packs the creep map
 */
void packCreep(replayer::Frame& f) {
  auto height = f.height / 4;
  auto width = f.width / 4;
  f.creep_map.resize(height * width / 8); // Only store build tiles
  for (unsigned y = 0, i = 0; y < height; ++y) {
    for (unsigned x = 0; x < width; ++x, ++i) {
      f.creep_map[i / 8] |= BWAPI::Broodwar->hasCreep(x, y) << (i % 8);
    }
  }
}


/**
 * Pack some information about the state of the unit:
 * unit ID: integer
 * position: (x, y) integers couple (in walk tiles)
 * enemy: 0/1
 * unit type: integer
 * current hit points: integer
 * current shield points: integer
 * ground weapon cooldown: integer
 * air weapon cooldown: integer
 * status flags: integer
 * is visible?: integer
 * some other stuff...
 * see http://bwapi.github.io/class_b_w_a_p_i_1_1_unit_interface.html for all
 * that's available
 */
void addUnit(
    BWAPI::Unit u,
    replayer::Frame& frame,
    BWAPI::PlayerInterface* player) {
  BWAPI::Position unitPosition = u->getPosition();
  if (!unitPosition.isValid())
    return;
  int x_wt = unitPosition.x / pixelsPerWalkTile;
  int y_wt = unitPosition.y / pixelsPerWalkTile;

  int pixel_size_x = u->getType().width();
  int pixel_size_y = u->getType().height();
  int unit_player = u->getPlayer()->getID();

  BWAPI::UnitType utype = u->getType();
  // could player->damage(WeaponType wpn) here but not sure of #attacks
  int ground_attack =
      (utype.groundWeapon().damageAmount() +
       utype.groundWeapon().damageBonus() *
           player->getUpgradeLevel(utype.groundWeapon().upgradeType())) *
      utype.maxGroundHits() * utype.groundWeapon().damageFactor();
  int air_attack =
      (utype.airWeapon().damageAmount() +
       utype.airWeapon().damageBonus() *
           player->getUpgradeLevel(utype.airWeapon().upgradeType())) *
      utype.maxAirHits() * utype.airWeapon().damageFactor();
  // store visibility from each player in a separate bit
  int32_t visible = 0;
  for (auto player : BWAPI::Broodwar->getPlayers()) {
    if (player->getID() >= 0) {
      visible |= (u->isVisible(player) << player->getID());
    }
  }
  int32_t buildTechUpgradeType = u->getBuildType().getID();
  if (buildTechUpgradeType == BWAPI::UnitTypes::None.getID()) {
    buildTechUpgradeType = u->getTech().getID();
    if (buildTechUpgradeType == BWAPI::TechTypes::None.getID()) {
      buildTechUpgradeType = u->getUpgrade().getID();
    }
  }

  int32_t associatedUnit = -1;
  if (u->getAddon() != nullptr)
    associatedUnit = u->getAddon()->getID();
  else if (u->getTransport() != nullptr)
    associatedUnit = u->getTransport()->getID();
  else if (u->getHatchery() != nullptr)
    associatedUnit = u->getHatchery()->getID();
  else if (u->getNydusExit() != nullptr)
    associatedUnit = u->getNydusExit()->getID();
  else if (u->getBuildUnit() != nullptr)
    associatedUnit = u->getBuildUnit()->getID();

  uint64_t flags = 0;
  flags |= u->isAccelerating() ? replayer::Unit::Flags::Accelerating : 0;
  flags |= u->isAttacking() ? replayer::Unit::Flags::Attacking : 0;
  flags |= u->isAttackFrame() ? replayer::Unit::Flags::AttackFrame : 0;
  flags |=
      u->isBeingConstructed() ? replayer::Unit::Flags::BeingConstructed : 0;
  flags |= u->isBeingGathered() ? replayer::Unit::Flags::BeingGathered : 0;
  flags |= u->isBeingHealed() ? replayer::Unit::Flags::BeingHealed : 0;
  flags |= u->isBlind() ? replayer::Unit::Flags::Blind : 0;
  flags |= u->isBraking() ? replayer::Unit::Flags::Braking : 0;
  flags |= u->isBurrowed() ? replayer::Unit::Flags::Burrowed : 0;
  flags |= u->isCarryingGas() ? replayer::Unit::Flags::CarryingGas : 0;
  flags |=
      u->isCarryingMinerals() ? replayer::Unit::Flags::CarryingMinerals : 0;
  flags |= u->isCloaked() ? replayer::Unit::Flags::Cloaked : 0;
  flags |= u->isCompleted() ? replayer::Unit::Flags::Completed : 0;
  flags |= u->isConstructing() ? replayer::Unit::Flags::Constructing : 0;
  flags |= u->isDefenseMatrixed() ? replayer::Unit::Flags::DefenseMatrixed : 0;
  flags |= u->isDetected() ? replayer::Unit::Flags::Detected : 0;
  flags |= u->isEnsnared() ? replayer::Unit::Flags::Ensnared : 0;
  flags |= u->isFlying() ? replayer::Unit::Flags::Flying : 0;
  flags |= u->isFollowing() ? replayer::Unit::Flags::Following : 0;
  flags |= u->isGatheringGas() ? replayer::Unit::Flags::GatheringGas : 0;
  flags |=
      u->isGatheringMinerals() ? replayer::Unit::Flags::GatheringMinerals : 0;
  flags |= u->isHallucination() ? replayer::Unit::Flags::Hallucination : 0;
  flags |= u->isHoldingPosition() ? replayer::Unit::Flags::HoldingPosition : 0;
  flags |= u->isIdle() ? replayer::Unit::Flags::Idle : 0;
  flags |= u->isInterruptible() ? replayer::Unit::Flags::Interruptible : 0;
  flags |= u->isInvincible() ? replayer::Unit::Flags::Invincible : 0;
  flags |= u->isIrradiated() ? replayer::Unit::Flags::Irradiated : 0;
  flags |= u->isLifted() ? replayer::Unit::Flags::Lifted : 0;
  flags |= u->isLoaded() ? replayer::Unit::Flags::Loaded : 0;
  flags |= u->isLockedDown() ? replayer::Unit::Flags::LockedDown : 0;
  flags |= u->isMaelstrommed() ? replayer::Unit::Flags::Maelstrommed : 0;
  flags |= u->isMorphing() ? replayer::Unit::Flags::Morphing : 0;
  flags |= u->isMoving() ? replayer::Unit::Flags::Moving : 0;
  flags |= u->isParasited() ? replayer::Unit::Flags::Parasited : 0;
  flags |= u->isPatrolling() ? replayer::Unit::Flags::Patrolling : 0;
  flags |= u->isPlagued() ? replayer::Unit::Flags::Plagued : 0;
  flags |= u->isPowered() ? replayer::Unit::Flags::Powered : 0;
  flags |= u->isRepairing() ? replayer::Unit::Flags::Repairing : 0;
  flags |= u->isResearching() ? replayer::Unit::Flags::Researching : 0;
  flags |= u->isSelected() ? replayer::Unit::Flags::Selected : 0;
  flags |= u->isSieged() ? replayer::Unit::Flags::Sieged : 0;
  flags |= u->isStartingAttack() ? replayer::Unit::Flags::StartingAttack : 0;
  flags |= u->isStasised() ? replayer::Unit::Flags::Stasised : 0;
  flags |= u->isStimmed() ? replayer::Unit::Flags::Stimmed : 0;
  flags |= u->isStuck() ? replayer::Unit::Flags::Stuck : 0;
  flags |= u->isTargetable() ? replayer::Unit::Flags::Targetable : 0;
  flags |= u->isTraining() ? replayer::Unit::Flags::Training : 0;
  flags |= u->isUnderAttack() ? replayer::Unit::Flags::UnderAttack : 0;
  flags |= u->isUnderDarkSwarm() ? replayer::Unit::Flags::UnderDarkSwarm : 0;
  flags |=
      u->isUnderDisruptionWeb() ? replayer::Unit::Flags::UnderDisruptionWeb : 0;
  flags |= u->isUnderStorm() ? replayer::Unit::Flags::UnderStorm : 0;
  flags |= u->isUpgrading() ? replayer::Unit::Flags::Upgrading : 0;

  frame.units[player->getID()].push_back({
      u->getID(),
      x_wt,
      y_wt,
      u->getHitPoints(),
      utype.maxHitPoints(),
      u->getShields(),
      utype.maxShields(),
      u->getEnergy(),
      player->weaponDamageCooldown(utype),
      u->getGroundWeaponCooldown(),
      u->getAirWeaponCooldown(),
      flags,
      visible,
      utype.getID(),
      player->armor(utype),
      player->getUpgradeLevel(BWAPI::UpgradeTypes::Protoss_Plasma_Shields),
      utype.size().getID(),
      unitPosition.x,
      unitPosition.y,
      pixel_size_x,
      pixel_size_y,
      ground_attack,
      air_attack,
      utype.groundWeapon().damageType().getID(),
      utype.airWeapon().damageType().getID(),
      player->weaponMaxRange(utype.groundWeapon()) / pixelsPerWalkTile,
      player->weaponMaxRange(utype.airWeapon()) / pixelsPerWalkTile,
      std::vector<replayer::Order>(),
      replayer::UnitCommand(),
      u->getVelocityX(),
      u->getVelocityY(),
      unit_player,
      u->getResources(),
      buildTechUpgradeType,
      u->getRemainingBuildTime() + u->getRemainingTrainTime(),
      u->getRemainingResearchTime() + u->getRemainingUpgradeTime(),
      u->getSpellCooldown(),
      associatedUnit,
      u->getScarabCount() + u->getSpiderMineCount() + u->getInterceptorCount() +
          u->hasNuke(),
  });

  // Add curent orders to order list
  // (we keep Orders::None orders as their timing marks the moment where
  //  previous order stops)
  int targetid = -1;
  if (u->getTarget()) {
    targetid = u->getTarget()->getID();
  } else if (u->getOrderTarget()) {
    targetid = u->getOrderTarget()->getID();
  }
  BWAPI::Position targetpos = u->getTargetPosition();

  frame.units[player->getID()].back().orders.push_back(
      {BWAPI::Broodwar->getFrameCount(), // first frame
       u->getOrder().getID(),
       targetid,
       targetpos.isValid() ? targetpos.x / pixelsPerWalkTile : -1,
       targetpos.isValid() ? targetpos.y / pixelsPerWalkTile : -1});

  if (u->getSecondaryOrder() != BWAPI::Orders::Nothing) {
    frame.units[player->getID()].back().orders.push_back({
        BWAPI::Broodwar->getFrameCount(),
        u->getSecondaryOrder().getID(),
        -1,
        -1,
        -1,
    });
  }

  // Set last command
  auto& command = frame.units[player->getID()].back().command;
  auto lastCommand = u->getLastCommand();
  targetpos = lastCommand.getTargetPosition();
  command.frame = u->getLastCommandFrame();
  command.type = lastCommand.type.getID();
  if (lastCommand.target) {
    command.targetId = lastCommand.target->getID();
  } else {
    command.targetId = -1;
  }
  command.targetX = targetpos.isValid() ? targetpos.x / pixelsPerWalkTile : -1;
  command.targetY = targetpos.isValid() ? targetpos.y / pixelsPerWalkTile : -1;
  command.extra = lastCommand.extra;
}


void packMyUnits(replayer::Frame& f) {
  auto self = BWAPI::Broodwar->self();
  for (auto& u : self->getUnits()) {
    // Ignore the unit if it no longer exists
    // Make sure to include this block when handling any Unit pointer!
    if (!u->exists())
      continue;

    addUnit(u, f, self); // TODO: only when the state changes

    if (u->getType().spaceProvided()) {
      for (auto x : u->getLoadedUnits()) {
        addUnit(x, f, self);
      }
    }
  }
}

void packTheirUnits(
    replayer::Frame& f,
    BWAPI::PlayerInterface* player) {
  if (player == nullptr)
    return;
  for (auto& u : player->getUnits()) {
    addUnit(u, f, player); // TODO: only when the state changes
  }
}

void packNeutral(replayer::Frame& f) {
  for (auto& u : BWAPI::Broodwar->getNeutralUnits()) {
    addUnit(
        u,
        f,
        BWAPI::Broodwar->neutral()); // TODO: only when the state changes
  }
}

void controllerUpdate(State* state) {
  auto bwl = bwapiLock();
  // Display the game frame rate as text in the upper left area of the screen
  BWAPI::Broodwar->drawTextScreen(200, 0, "FPS: %d", BWAPI::Broodwar->getFPS());
  BWAPI::Broodwar->drawTextScreen(
      200, 20, "Average FPS: %f", BWAPI::Broodwar->getAverageFPS());

  // Return if the game is paused
  if (BWAPI::Broodwar->isPaused())
    return;

  // Save frame state
  if (true) {
    //replayer::Frame* f = new replayer::Frame();
    auto* f = state->frame;
    *f = replayer::Frame();
    f->height = BWAPI::Broodwar->mapHeight() * 4;
    f->width = BWAPI::Broodwar->mapWidth() * 4;

    if (BWAPI::Broodwar->isReplay()) {
      for (auto player : BWAPI::Broodwar->getPlayers()) {
        if (!player->isNeutral()) {
          packTheirUnits(*f, player);
          packResources(*f, player);
        }
      }
      packNeutral(*f);
    } else {
      packResources(*f, BWAPI::Broodwar->self());
      packMyUnits(*f);
      packTheirUnits(*f, BWAPI::Broodwar->enemy());
      packNeutral(*f);
    }
    packBullets(*f);
    packCreep(*f);

  }

  state->numUpdates++;

  // Update units
  for (auto& us : state->units) {
    if (state->frame->units.find(us.first) == state->frame->units.end()) {
      // No more units from this team
      us.second.clear();
    }
  }
  for (const auto& fus : state->frame->units) {
    auto player = fus.first;
    if (state->units.find(player) == state->units.end()) {
      state->units.emplace(player, std::vector<Unit>());
    } else {
      state->units[player].clear();
    }

    std::copy_if(
        fus.second.begin(),
        fus.second.end(),
        std::back_inserter(state->units[player]),
        [state, player](const Unit& unit) {
          auto ut = torchcraft::BW::UnitType::_from_integral_nothrow(unit.type);
          return (
              // Unit is of known type (or a neutral unit)
              (player == state->neutral_id || ut) &&
              // Unit has not been marked dead
              std::find(state->deaths.begin(), state->deaths.end(), unit.id) == state->deaths.end());
        });
  }

  // Update alive units
  state->aliveUnits.clear();
  state->aliveUnitsConsidered.clear();
  for (const auto& us : state->units) {
    auto player = us.first;
    for (const auto& unit : us.second) {
      state->aliveUnits[unit.id] = player;
    }
  }

  state->frame_from_bwapi = BWAPI::Broodwar->getFrameCount();

}

extern void controllerExecute(const std::vector<torchcraft::Client::Command>& commands) {
  auto bwl = bwapiLock();
  for (auto& v : commands) {
    handleCommand(v.code, v.args, v.str);
  }

}

namespace  {

std::condition_variable cv;
std::condition_variable cv2;
std::mutex mut;
bool step = false;
bool stepDone = true;
bool inStep = false;
std::thread stepThread;
bool waitingForLock = false;
bool allDone = false;

}

std::unique_lock<std::mutex> bwapiLock() {
  std::unique_lock<std::mutex> l(mut);
  while (!inStep) {
    waitingForLock = true;
    cv.wait(l);
    waitingForLock = false;
  }
  return l;
}

class ExampleAIModule : public BWAPI::AIModule {
public:
  std::optional<cherrypi::Player> bot;
  virtual void onStart()  override {
    inStep = true;
    const char* args[] = {".", "-build", "tcheese", "-bandit", "none", "-nogame_history", "-nowarn_if_slow", "-noconsistency", "-notimers", nullptr};
    int argc = sizeof(args) / sizeof(args[0]) - 1;
    char** argv = (char**)args;
    cherrypi::init();
    google::InitGoogleLogging(argv[0]);
    gflags::ParseCommandLineFlags(&argc, &argv, true);
    if (FLAGS_seed >= 0) {
      common::Rand::setSeed(FLAGS_seed);
    }

    // We need to init the logging after we have parsed the command
    // line flags since it depends on flags set by it
    cherrypi::initLogging(argv[0], FLAGS_logsinkdir, FLAGS_logsinktostderr);

    try {
      using namespace cherrypi;

      auto client = std::make_shared<tc::Client>();
      std::vector<std::string> upd;
      client->init(upd);
      bot.emplace(client);
      setupPlayerFromCli(&*bot);

      // In normal playing mode we don't need to save UPC-related data longer than
      // necessary
      bot->state()->board()->upcStorage()->setPersistent(false);

      bot->init();

    } catch (std::exception& e) {
      LOG(FATAL) << "Exception: " << e.what();
    }

    inStep = false;

  }

  std::vector<int> deaths;

  virtual void onFrame() override {
    if (allDone) {
      return;
    }
    auto start = std::chrono::steady_clock::now();
    std::unique_lock<std::mutex> l(mut);
    inStep = true;
    auto timeout = start + std::chrono::milliseconds(30);
    if (waitingForLock) {
      cv.notify_all();
    }
    if (!stepDone) {
      while (std::chrono::steady_clock::now() < timeout && !stepDone) {
        cv2.wait_until(l, timeout);
      }
      if (!stepDone) {
        //VLOG(0) << " timed out waiting for previous step after " << std::chrono::duration_cast<std::chrono::milliseconds>(std::chrono::steady_clock::now() - start).count() << "ms";
        inStep = false;
        return;
      }
    }
    std::swap(deaths, bot->state()->tcstate()->deaths);
    deaths.clear();
    stepDone = false;
    step = true;
    cv.notify_all();

    if (!stepThread.joinable()) {
      stepThread = std::thread([this]() {
        std::unique_lock<std::mutex> l(mut);
        while (!bot->state()->gameEnded()) {
          while (!step) {
            if (bot->state()->gameEnded()) {
              return;
            }
            cv.wait(l);
          }
          step = false;
          l.unlock();
          bot->step();
          bot->state()->tcstate()->deaths.clear();
          l.lock();
          stepDone = true;
          cv2.notify_all();
        }
      });
    }

    while (std::chrono::steady_clock::now() < timeout && !stepDone) {
      cv2.wait_until(l, timeout);
    }
    if (!stepDone) {
      //VLOG(0) << " timed out waiting for step after " << std::chrono::duration_cast<std::chrono::milliseconds>(std::chrono::steady_clock::now() - start).count() << "ms";
      inStep = false;
      return;
    }
    inStep = false;
    //VLOG(0) << " step took " << std::chrono::duration_cast<std::chrono::milliseconds>(std::chrono::steady_clock::now() - start).count() << "ms";
  }
  virtual void onUnitDestroy(BWAPI::Unit unit) override {
    deaths.push_back(unit->getID());
  }
  virtual void onEnd(bool won) override {
    std::unique_lock<std::mutex> l(mut);
    inStep = true;
    while (!stepDone) {
      cv2.wait(l);
    }
    bot->state()->tcstate()->game_ended = true;
    bot->state()->tcstate()->game_won = won;

    std::swap(deaths, bot->state()->tcstate()->deaths);
    deaths.clear();
    stepDone = false;
    step = true;
    cv.notify_all();

    while (!stepDone) {
      cv2.wait(l);
    }

    stepThread.join();

    allDone = true;

    if (bot->state()->won()) {
      LOG(WARNING) << "Final result: Victory!!!";
    } else if (bot->state()->currentFrame() == 0) {
      LOG(WARNING) << "Game ended on frame 0";
      LOG(WARNING) << "Final result: Inconclusive???";
    } else {
      LOG(WARNING) << "Oh noes we lost :( -- with "
                   << bot->state()->unitsInfo().myBuildings().size()
                   << " buildings left";
      LOG(WARNING) << "Final result: Defeat!!!";
    }
  }
};

extern "C" __declspec(dllexport) void gameInit(BWAPI::Game* game) { BWAPI::BroodwarPtr = game; }
BOOL APIENTRY DllMain( HANDLE hModule, DWORD ul_reason_for_call, LPVOID lpReserved )
{
  switch (ul_reason_for_call)
  {
  case DLL_PROCESS_ATTACH:
    break;
  case DLL_PROCESS_DETACH:
    break;
  }
  return TRUE;
}

extern "C" __declspec(dllexport) BWAPI::AIModule* newAIModule()
{
  return new ExampleAIModule();
}


//void handleEvents() {
//  for (auto& e : BWAPI::Broodwar->getEvents()) {
//    switch (e.getType()) {
//      case BWAPI::EventType::UnitDestroy:
//        deaths.push_back(e.getUnit()->getID());
//        break;
//      case BWAPI::EventType::MatchEnd:
//        this->game_ended = true;
//        this->is_winner = e.isWinner();
//        this->battle_frame_count = 0;
//        break;
//      default:
//        break;
//    }
//  }
//}
