/*
 * Copyright (c) 2017-present, Facebook, Inc.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */

#include "behavior.h"
#include "agent.h"
#include "common/rand.h"
#include "squadtask.h"
#include <bwem/map.h>

#include <BWAPI.h>

namespace cherrypi {

bool canSiegeHere(State* state, Unit* unit) {
  for (Unit* u : state->unitsInfo().myBuildings()) {
    if (utils::intersects(u, unit)) {
      return false;
    }
  }
  return true;
}


DEFINE_bool(behavior_chase, false, "Toggles chasing behaviors");
DEFINE_bool(behavior_kite, false, "Toggles kiting behaviors");
void Behavior::perform(Agent& agent) {
  if (!agent.currentAction.isFinal) {
    //VLOG(0) << "unit " << agent.unit << " perform " << typeid(*this).name();
    agent.currentAction = onPerform(agent);
    if (agent.currentAction.isFinal) {
      //VLOG(0) << " action is final";
    }
  }
}

MicroAction BehaviorSeries::onPerform(Agent& agent) {
  for (auto& behavior : behaviors_) {
    behavior->perform(agent);
  }
  return agent.currentAction;
}

MicroAction BehaviorML::onPerform(Agent& agent) {
  for (auto& model : *(agent.task->models)) {
    auto action = model.second->decode(agent.unit);
    if (action.isFinal) {
      return action;
    }
  }
  return pass;
}

MicroAction BehaviorUnstick::onPerform(Agent& agent) {
  State* state = agent.state;
  Unit* unit = agent.unit;
  Unit* target = agent.target;
  auto* task = agent.task;

  auto findNewTarget = [&]() {
    Unit* nearestTarget = nullptr;
    Unit* furthestTarget = nullptr;
    float nearest = kfInfty;
    float furthest = 0.0f;
    int nInRange = 0;
    for (Unit* e : agent.legalTargets) {
      if (unit->canAttack(e) && e->type->isBuilding && (e->type->hasGroundWeapon || e->type == buildtypes::Terran_Bunker)) {
        float d = utils::distanceBB(unit, e);
        if (d <= 4.0f * 8) {
          if (d < nearest) {
            nearest = d;
            nearestTarget = e;
          }
          if (d > furthest) {
            furthest = d;
            furthestTarget = e;
          }
          if (e->inRangeOf(unit)) {
            ++nInRange;
          }
        }
      }
    }
    return nInRange < 2 || nearest <= 8.0f ? nearestTarget : furthestTarget;
  };

  if (unit->type == buildtypes::Zerg_Lurker) {
    if (!unit->unit.orders.empty() && unit->unit.orders.at(0).type == +tc::BW::Order::Burrowing) {
      return doNothing;
    }
    if (state->currentFrame() - agent.lastSiege < 5) {
      return doNothing;
    }
    if (agent.wantsToFight && !unit->burrowed()) {
      if (!target || utils::distanceBB(unit, target) > 8.0f) {
        Unit* newTarget = findNewTarget();
        if (newTarget) {
          target = newTarget;
        }
      }
    }
    bool isVeryClose = target && utils::distanceBB(unit, target) <= 8.0f;
    if (target && (agent.wantsToFight || isVeryClose)) {
      if (agent.targetInRange || isVeryClose) {
        if (!unit->burrowed() && state->currentFrame() - agent.lastSiege >= 20) {
          agent.lastSiege = state->currentFrame();
          auto bwl = bwapiLock();
          auto bwu = BWAPI::Broodwar->getUnit(unit->id);
          if (bwu) bwu->burrow();
        }
        return doNothing;
      }
    } else {
      target = nullptr;
    }
    if (unit->burrowed() && state->currentFrame() - agent.lastSiege >= 60 && state->currentFrame() - unit->lastFiredWeapon >= 60) {
      if (state->currentFrame() - agent.lastUnsiege >= 30) {
        agent.lastUnsiege = state->currentFrame();
        auto bwl = bwapiLock();
        auto bwu = BWAPI::Broodwar->getUnit(unit->id);
        if (bwu) bwu->unburrow();
      }
      return doNothing;
    }
    if (target) {
      if (unit->burrowed() || state->currentFrame() - agent.lastSiege < 60) {
        return doAction(agent.attack(target));
      }
      return doAction(agent.moveTo(target->pos()));
    }
    if (state->currentFrame() - agent.lastSiege < 60) {
      return doNothing;
    }
    BehaviorTravel x;
    return x.onPerform(agent);
  }

  if (agent.stuckFrames < agent.unstickTriggerFrames || agent.unit->flying()) {
    return pass;
  }
  VLOG(3) << utils::unitString(agent.unit) << " is unsticking";
  agent.postCommand(tc::BW::UnitCommandType::Stop);
  agent.stuckFrames = 0;
  return doNothing;
}

MicroAction BehaviorIfIrradiated::onPerform(Agent& agent) {
  auto* unit = agent.unit;

  if (unit->irradiated()) {
    std::vector<Unit*> units;
    for (auto u : unit->allyUnitsInSightRange) {
      if (utils::distance(u, unit) < 16) {
        units.push_back(u);
      }
    }
    if (!units.empty()) {
      auto centroid = Vec2(utils::centerOfUnits(units));
      auto pos = Vec2(unit) + (Vec2(unit) - Vec2(centroid)).normalize() * 10;
      return doAction(agent.moveTo(pos));
    }
  }
  return pass;
}

MicroAction BehaviorIfStormed::onPerform(Agent& agent) {
  auto& task = agent.task;
  auto* unit = agent.unit;

  for (auto stormLoc : task->storms_) {
    if (utils::distance(unit, stormLoc) > 16)
      continue;
    auto pos = Vec2(unit) + (Vec2(unit) - Vec2(stormLoc)).normalize() * 10;
    return doAction(agent.moveTo(pos));
  }
  return pass;
}

MicroAction BehaviorVsScarab::onPerform(Agent& agent) {
  State* state = agent.state;
  Unit* unit = agent.unit;
  Unit* target = agent.target;
  auto* task = agent.task;

  Vec2 move;
  for (Unit* u : task->squadUnits()) {
    if (!u->flying()) {
      for (Unit* a : unit->beingAttackedByEnemies) {
        if (a->type == buildtypes::Protoss_Scarab) {
          float d = std::max(utils::distanceBB(u, unit), 1.0f);
          if (d <= 8.0f) {
            move += (unit->posf() - u->posf()) / d;
          }
        }
      }
    }
  }
  if (move != Vec2()) {
    return doAction(agent.moveTo(unit->pos() + Position(move.normalize() * 12.0f)));
  }

  for (auto u : unit->beingAttackedByEnemies) {
    if (u->type == buildtypes::Protoss_Scarab) {
      //auto pos = Vec2(unit) + (Vec2(unit) - Vec2(u)).normalize() * 4.0f * 4;
      //return doAction(agent.moveTo(pos));
//      int latency = state->latencyFrames();
//      Vec2 myPos = unit->posf() + unit->velocity() * latency;
//      Vec2 targetPos = target ? target->posf() + target->velocity() * latency : myPos;

//      auto kiteVector = [&]() {
//        Vec2 adjustment;
//        for (Unit* u : agent.legalTargets) {
//          if (u->canAttack(unit)) {
//            float distance =
//                std::max(utils::distanceBB(unit, u), DFOASG(0.125f, 1.0f));
//            float maxDistance = DFOASG(4 * 10, 4 * 3);
//            if (distance <= maxDistance) {
//              adjustment += (myPos - u->posf()) * (maxDistance / distance);
//            }
//          }
//        }
//        for (Unit* u : task->squadUnits()) {
//          if (!u->flying()) {
//            float distance =
//                std::max(utils::distanceBB(unit, u), DFOASG(0.125f, 1.0f));
//            float maxDistance = DFOASG(4 * 10, 4 * 3);
//            if (distance <= maxDistance) {
//              adjustment += (myPos - u->posf()) * (maxDistance / distance);
//            }
//          }
//        }
//        Vec2 moveDir = (myPos - targetPos).normalize();
//        return unit->posf() + (moveDir + adjustment.normalize()).normalize() * 12.0;
//      };
//      return doAction(agent.moveTo(kiteVector()));

    }
  }
  return pass;
}

MicroAction BehaviorTravel::onPerform(Agent& agent) {
  State* state = agent.state;
  auto& task = agent.task;
  auto* unit = agent.unit;

//  if (unit->targetGroupPos != kInvalidPosition) {
//    Unit* nearestThreat = utils::getBestScoreCopy(agent.legalTargets, [&](Unit* u) {
//      if (!u->canAttack(unit) || u->type == buildtypes::Protoss_Interceptor) return kfInfty;
//      return utils::distance(u, unit) - (float)u->rangeAgainst(unit);
//    }, kfInfty);
//    if (!nearestThreat || !unit->inRangeOfPlus(nearestThreat, 4.0f * 6)) {
//      return doAction(agent.moveTo(unit->targetGroupPos));
//    }
//  }

//  if (utils::distance(unit->pos(), state->areaInfo().myStartLocation()) > 4.0f * 64) {
//    if (unit->type == buildtypes::Terran_Civilian) {
//      return pass;
//    }
//  }

  auto unsiege = [&]() {
    if (unit->type == buildtypes::Terran_Siege_Tank_Siege_Mode) {
      if (utils::distance(unit->pos(), Position(task->targetX, task->targetY)) > 8.0f) {
        agent.postCommand(tc::BW::UnitCommandType::Unsiege);
        return doNothing;
      }
    }
    return pass;
  };

  if (task->targetingLocation) {
    Position target(task->targetX, task->targetY);
    if (false && utils::distance(unit->pos(), target) <= unit->unit.groundRange) {
      if (unit->type == buildtypes::Terran_Siege_Tank_Tank_Mode) {
        if (state->hasResearched(buildtypes::Tank_Siege_Mode)) {
          bool siege = true;
          auto* area = state->areaInfo().tryGetArea(unit->pos());
          if (area) {
            for (auto& choke : area->area->ChokePoints()) {
              Position a(choke->Pos(BWEM::ChokePoint::node::middle));
              if (utils::distance(a, unit->pos()) <= 4.0f * 6) {
                siege = false;
                break;
              }
            }
          }
          if (siege && state->currentFrame() - agent.lastSiege >= 200 && canSiegeHere(state, unit)) {
            agent.lastSiege = state->currentFrame();
            agent.postCommand(tc::BW::UnitCommandType::Siege);
            return doNothing;
          }
        }
      } else if (unit->sieged() && state->currentFrame() - agent.lastSiege < 24 * 12) {
        return doNothing;
      }

      //return doAction(agent.moveTo(Vec2(target) + (unit->posf() - Vec2(target)).normalize() * unit->unit.groundRange));
      //return doAction(agent.moveTo(state->areaInfo().myStartLocation()));
    }
    auto tmp = unsiege();
    if (tmp.isFinal) {
      return tmp;
    }
    if (unit->type == buildtypes::Terran_Siege_Tank_Tank_Mode && false) {
      if (!unit->moving() && utils::distance(unit->pos(), Position(task->targetX, task->targetY)) <= 8.0f) {
        if (state->hasResearched(buildtypes::Tank_Siege_Mode)) {
          if (state->currentFrame() - agent.lastSiege >= 200 && canSiegeHere(state, unit)) {
            agent.lastSiege = state->currentFrame();
            agent.postCommand(tc::BW::UnitCommandType::Siege);
            return doNothing;
          }
        }
      }
    }
    return doAction(agent.moveTo(Position(task->targetX, task->targetY)));
  } else if (unit->threateningEnemies.empty()) {
    unsiege();
    return doAction(agent.smartMove(task->center_));
  }
  unsiege();
  return pass;
}

MicroAction BehaviorLeave::onPerform(Agent& agent) {
  auto* unit = agent.unit;

  if (unit->threateningEnemies.empty()) {
    return pass;
  }

  auto enemyCenter = utils::centerOfUnits(unit->threateningEnemies);
  auto fleePosition = (Vec2(unit) - Vec2(enemyCenter)) * 15;
  return doAction(agent.smartMove(Position(fleePosition)));
}

constexpr int kChaseHpThreshold = 20; // Magic: 0+
constexpr double kChaseDelProbThreshold = 1.0; // Magic: 0-1
constexpr double kChaseOvershoot = 4.0; // Magic: 0+
constexpr double kChaseLookahead = 4.0; // Magic: 0+
MicroAction BehaviorChase::onPerform(Agent& agent) {
  auto& task = agent.task;
  State* state = agent.state;
  Unit* unit = agent.unit;
  Unit* target = agent.target;

  if (!FLAGS_behavior_chase) {
    return pass;
  }
  if (unit->type == buildtypes::Zerg_Lurker) {
    return pass;
  }
  if (target == nullptr || target->gone) {
    return pass;
  }
  if (unit->unit.health < kChaseHpThreshold) {
    return pass;
  }
  if (task->delProb < kChaseDelProbThreshold) {
    return pass;
  }
  if (target->topSpeed <= 0 && !unit->threateningEnemies.empty()) {
    return pass;
  }

  VLOG(2) << utils::unitString(unit) << " chases " << utils::unitString(target);

  bool readyToShoot = unit->cd() <= state->latencyFrames();
  bool inRange = target->inRangeOf(unit);
  if (readyToShoot && inRange) {
    return doAction(agent.attack(target));
  } else {
    double overshoot = target->topSpeed > 0 ? kChaseOvershoot : 0;
    auto destination = Vec2(target) +
        (Vec2(target) - Vec2(unit)).normalize() * overshoot +
        target->velocity() * kChaseLookahead;
    return doAction(agent.moveTo(Position(destination)));
  }
}

constexpr int kKiteHpThreshold = 20; // Magic: 0+
constexpr double kKiteDelProbThreshold = 0.85; // Magic: 0-1
constexpr double kKiteRatioDefault = 0.25; // Magic: 0+
constexpr double kKiteRatioPunish = 1.0; // Magic: 0+
constexpr double kKiteRatioFallback = 0.5; // Magic: 0+
constexpr double kKiteRatioBreathe = 0.5; // Magic: 0+
constexpr int kKiteFrameMargin = 3; // Magic: 0+
constexpr int kKiteRangeMargin = 4; // Magic: 0+
MicroAction BehaviorKite::onPerform(Agent& agent) {
  auto& task = agent.task;
  State* state = agent.state;
  Unit* unit = agent.unit;
  Unit* target = agent.target;

  if (!FLAGS_behavior_kite) {
    return pass;
  }
  if (unit->type == buildtypes::Zerg_Lurker) {
    return pass;
  }
  if (unit->threateningEnemies.empty()) {
    return pass;
  }
  if (unit->unit.health > kKiteHpThreshold &&
      task->delProb > kKiteDelProbThreshold) {
    return pass;
  }
  if (std::max(unit->unit.groundRange, unit->unit.airRange) < 12) {
    return pass;
  }

  if (target == nullptr || target->gone) {
    return pass;
  }
  double kiteRange = unit->rangeAgainst(target);
  double kiteCd = unit->maxCdAgainst(target);

  // Punish:
  // * Hover at the edge of our range.
  // * Use if we outrange and outspeed them.
  // * Use even if retreating.
  //
  // Fallback:
  // * Shoot while retreating.
  // * Use if we outrange but don't outspeed them.
  // * Use only if retreating.
  //
  // Breathe:
  // * Back out while on cooldown (catching our breath)
  // * Use if this helps us trade more effectively
  // * Use only if fighting.
  //
  int countThreats = 0;
  int countCanPunish = 0;
  int countCanFallback = 0;
  int countCanBreathe = 0;
  for (auto* enemy : unit->threateningEnemies) {
    if (enemy->type == buildtypes::Terran_Siege_Tank_Siege_Mode ||
        enemy->type == buildtypes::Protoss_Reaver) {
      // Don't get cute while eating splash damage.
      return pass;
    }

    double rangeAgainstUs = enemy->rangeAgainst(unit);
    double maxCdAgainstUs = enemy->maxCdAgainst(unit);

    bool canPunish = true;
    bool canFallback = !agent.wantsToFight && target != nullptr;
    bool canBreathe = agent.wantsToFight;
    if (rangeAgainstUs >= kiteRange || enemy->topSpeed >= unit->topSpeed) {
      canPunish = false;
    }
    if (rangeAgainstUs >= kiteRange) {
      canFallback = false;
    }
    if (unit->topSpeed > enemy->topSpeed) {
      canFallback = false;
    }
    if (rangeAgainstUs * maxCdAgainstUs > kiteCd * task->delProb) {
      canBreathe = false;
    }
    countThreats += 1;
    countCanPunish += canPunish ? 1 : (agent.wantsToFight ? 0 : -1);
    countCanFallback += canFallback ? 1 : 0;
    countCanBreathe += canBreathe ? 1 : 0;
  }
  if (countThreats <= 0) {
    return pass;
  }
  double valueDefault = countThreats * kKiteRatioDefault;
  double valuePunish = countCanPunish * kKiteRatioPunish;
  double valueFallback = countCanFallback * kKiteRatioFallback;
  double valueBreathe = countCanBreathe * kKiteRatioBreathe;
  double valueBest = std::max(
      valueDefault,
      std::max(valuePunish, std::max(valueFallback, valueBreathe)));

  if (valueDefault >= valueBest) {
    return pass;
  }

  bool readyToShoot = unit->cd() < state->latencyFrames();
  double cdEffective = std::max(unit->cd(), (double)state->latencyFrames());
  double targetDistanceNow = utils::pxDistanceBB(unit, target);

  // Project distance at time we fire
  // Allow some time to turn
  double targetDistanceProjected =
      targetDistanceNow + target->topSpeed * (cdEffective + kKiteFrameMargin);
  bool targetEscaped = targetDistanceProjected > kiteRange;
  bool targetEscaping = targetDistanceProjected > kiteRange - kKiteRangeMargin;

  auto attack = [&]() { return doAction(agent.attack(target)); };
  auto runAway = [&]() {
    return doAction(agent.filterMove({movefilters::avoidThreatening()}));
  };

  if (valueFallback >= valueBest) {
    VLOG(2) << utils::unitString(unit) << " falls back from "
            << utils::unitString(target);

    if (readyToShoot) {
      return attack();
    }
    // The default behavior will have us flee; do so.
    return pass;
  } else if (valuePunish >= valueBest) {
    VLOG(2) << utils::unitString(unit) << " punishes "
            << utils::unitString(target);

    if ((readyToShoot && targetEscaping) || targetEscaped) {
      return attack();
    }
    return runAway();
  } else if (valueBreathe >= valueBest) {
    VLOG(2) << utils::unitString(unit) << " catches breath against "
            << utils::unitString(target);

    if (readyToShoot || targetEscaped) {
      return attack();
    }
    return runAway();
  }

  return pass;
}

MicroAction BehaviorFormation::onPerform(Agent& agent) {
  if (!agent.targetInRange && agent.formationPosition != kInvalidPosition) {
    return doAction(agent.moveTo(agent.formationPosition));
  }

  return pass;
}

MicroAction BehaviorDetect::onPerform(Agent& agent) {
  auto& task = agent.task;
  auto unit = agent.unit;
  auto unitType = unit->type;
  if (unitType != buildtypes::Terran_Science_Vessel &&
      unitType != buildtypes::Protoss_Observer &&
      unitType != buildtypes::Zerg_Overlord) {
    return pass;
  }
  return pass;

  Unit* cloakedTarget = utils::getBestScoreCopy(
      utils::filterUnits(
          task->targets_,
          [](Unit* e) { return e->cloaked() || e->burrowed(); }),
      [&](Unit* e) { return utils::distance(unit, e); },
      kfInfty);
  if (cloakedTarget) {
    Unit* ally = utils::getBestScoreCopy(
        task->squadUnits(),
        [&](Unit* u) {
          if (u == unit || !u->canAttack(cloakedTarget)) {
            return kfInfty;
          }
          return utils::distance(u, cloakedTarget);
        },
        kfInfty);
    if (ally && utils::distance(unit, cloakedTarget) < unit->sightRange - 4) {
      VLOG(3) << unit << " senses ally near cloaked target, moving to " << ally
              << " near cloaked " << cloakedTarget;
      return doAction(agent.smartMove(ally));
    }
  }
  if (!unit->threateningEnemies.empty()) {
    auto threat = unit->threateningEnemies[0];
    auto dir = Vec2(unit) - Vec2(threat);
    dir.normalize();
    VLOG(3) << unit << " senses threat, moving away from " << threat;
    return doAction(agent.smartMove(Position(Vec2(unit) + dir * 25)));
  }

  if (cloakedTarget) {
    VLOG(3) << unit << " senses cloaked target, moving to " << cloakedTarget;
    return doAction(agent.smartMove(cloakedTarget));
  }

  return pass;
}

MicroAction BehaviorEngageCooperatively::onPerform(Agent& agent) {
  Unit* unit = agent.unit;
  Unit* target = agent.target;
  State* state = agent.state;
  auto* task = agent.task;

  if (target == nullptr || unit->flying() || unit->burrowed()) {
    return pass;
  }

  // Avoid situations where some unit is stuck behind some other unit
  // and unable to attack the target.
  // Do this by simply moving the unit in front forwards between attacks.
  // This is most noticable with hydralisks attacking a static target
  // like cannons.

  if (unit->rangeAgainst(target) >= 8 &&
      target->inRangeOf(agent.unit, state->latencyFrames() + DFOASG(6, 3)) &&
      unit->cd() <= state->latencyFrames()) {
    Vec2 myPos = unit->posf();
    Vec2 targetPos = target->posf();
    Vec2 targetVector = targetPos - myPos;

    // TODO: check that we actually *can* move forward
    for (Unit* u : task->squadUnits()) {
      if ((myPos - u->posf()).dot(targetVector) >= 0 &&
          utils::distanceBB(u, unit) <= DFOASG(3.0f, 1.5f) &&
          !target->inRangeOf(u) &&
          u->rangeAgainst(target) >= unit->rangeAgainst(target)) {
        Vec2 moveTo = myPos + targetVector.normalize() * DFOASG(6, 3);
        return doAction(agent.moveTo(moveTo, false));
      }
    }
  }

  // Dodge splash!

  bool dodgeSplash = false;
  for (Unit* u : agent.legalTargets) {
    if (u->type == buildtypes::Terran_Valkyrie ||
        u->type == buildtypes::Protoss_Corsair ||
        u->type == buildtypes::Protoss_Archon ||
        u->type == buildtypes::Protoss_High_Templar ||
        u->type == buildtypes::Terran_Firebat ||
        u->type == buildtypes::Terran_Siege_Tank_Siege_Mode ||
        u->type == buildtypes::Zerg_Lurker ||
        u->type == buildtypes::Protoss_Reaver ||
        u->type == buildtypes::Terran_Vulture_Spider_Mine) {
      float range = u->rangeAgainst(unit);
      if (u->type == buildtypes::Protoss_High_Templar ||
          u->type == buildtypes::Protoss_Reaver) {
        range = 4.0f * 9;
      }
      if (u->type == buildtypes::Terran_Vulture_Spider_Mine) {
        range = 4.0f * 2;
      }
      if (u->canAttack(unit) &&
          utils::distanceBB(unit, u) <= range + DFOASG(6, 3)) {
        dodgeSplash = true;
        break;
      }
    }
  }
  if (dodgeSplash &&
      (unit->rangeAgainst(target) >= 8 || !agent.targetInRange) &&
      (!target->inRangeOf(unit, state->latencyFrames() + DFOASG(6, 3)) ||
       unit->cd() > state->latencyFrames())) {
    int latency = state->latencyFrames();

    Vec2 myPos = unit->posf() + unit->velocity() * latency;
    Vec2 targetPos = target->posf() + target->velocity() * latency;

    auto canMoveInDirection = [&](Vec2 dir,
                                  float distance = DFOASG(4.0f * 2, 4.0f)) {
      dir = dir.normalize();
      for (float d = 4.0f; d <= distance; d += 4.0f) {
        Position pos{myPos + dir * d};
        auto* tile = state->tilesInfo().tryGetTile(pos.x, pos.y);
        if (!tile || !tile->entirelyWalkable || tile->building) {
          return false;
        }
      }
      return true;
    };

    if (canMoveInDirection(
            targetPos - myPos,
            utils::distance(Position(myPos), Position(targetPos)))) {
      auto attackVector = [&]() {
        Vec2 adjustment;
        for (Unit* u : task->squadUnits()) {
          if (u != unit && u->flying() == unit->flying()) {
            float distance =
                std::max(utils::distanceBB(unit, u), DFOASG(0.125f, 1.0f));
            float maxDistance = DFOASG(4 * 3, 6);
            if (distance <= maxDistance) {
              adjustment += (myPos - u->posf()) * (maxDistance / distance);
            }
          }
        }
        if (adjustment == Vec2()) {
          return Vec2(kInvalidPosition);
        }
        Vec2 moveDir = (targetPos - myPos).normalize();
        return unit->posf() +
            (moveDir + moveDir + adjustment.normalize()).normalize() * 12.0f;
      };

      auto moveTo = attackVector();
      if (moveTo != Vec2(kInvalidPosition) &&
          canMoveInDirection(Vec2(moveTo) - myPos)) {
        utils::drawLine(state, unit, Position(moveTo));
        return doAction(agent.moveTo(Position(moveTo)));
      }
    }
  }

  if (!agent.targetInRange && agent.formationPosition != kInvalidPosition) {
    return doAction(agent.moveTo(agent.formationPosition));
  }

  return pass;
}

MicroAction BehaviorHideInBunkers::onPerform(Agent& agent) {
  Unit* unit = agent.unit;
  Unit* target = agent.target;
  State* state = agent.state;
  auto* task = agent.task;

  if (agent.wantsToFight) {
    return pass;
  }

  if (unit->type == buildtypes::Terran_Marine) {
    BWAPI::Unit bwu = BWAPI::BroodwarPtr->getUnit(unit->id);
    if (bwu && !bwu->isLoaded()) {
      Unit* nearestThreat = utils::getBestScoreCopy(agent.legalTargets, [&](Unit* u) {
        if (!u->canAttack(unit)) return kfInfty;
        return utils::distance(u, unit) - (float)u->rangeAgainst(unit);
      }, kfInfty);
      Unit* bunker = utils::getBestScoreCopy(state->unitsInfo().myCompletedUnitsOfType(buildtypes::Terran_Bunker), [&](Unit* u) {
        return utils::distance(u->pos(), target->pos());
      });
      if (nearestThreat && bunker) {
        float bunkerDistance = utils::distanceBB(unit, bunker);
        if ((!unit->inRangeOfPlus(nearestThreat, 4.0f) || bunkerDistance <= 8.0f) && bunkerDistance <= 4.0f * 8) {
          auto bwb = BWAPI::BroodwarPtr->getUnit(bunker->id);
          if (bwb) {
            int space = BWAPI::UnitType(bunker->type->unit).spaceProvided();
            int req = BWAPI::UnitType(unit->type->unit).spaceRequired();
            for (auto x : bwb->getLoadedUnits()) {
              space -= x->getType().spaceRequired();
            }
            if (space >= req) {
              if (unit->limitSpecial(state)) {
                bwu->rightClick(bwb);
              }
              return doNothing;
            }
          }
        }
      }
    }

  }

  return pass;
}

MicroAction BehaviorEngage::onPerform(Agent& agent) {
  State* state = agent.state;
  Unit* target = agent.target;
  Unit* unit = agent.unit;

//  if (agent.wantsToFight && unit->targetGroupPos != kInvalidPosition && (!target || utils::distanceBB(unit, target) > unit->rangeAgainst(target) + 4.0f)) {
//    return doAction(agent.moveTo(unit->targetGroupPos));
//  }

  if (!agent.wantsToFight || target == nullptr) {
    return pass;
  }

  VLOG(3) << utils::unitString(agent.unit) << " engages "
          << utils::unitString(target);

  if (utils::distanceBB(unit, target) > unit->rangeAgainst(target) + 4.0f * 6) {
    if (unit->flying()) return doAction(agent.moveTo(target->pos()));
    return doAction(agent.moveTo(movefilters::pathMoveTo(state, unit, target->pos())));
  }

  return doAction(agent.attack(target));
}

} // namespace cherrypi
