/*
 * Copyright (c) 2017-present, Facebook, Inc.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */

#include "bandit.h"
#include <fmt/format.h>

namespace cherrypi {

namespace model {

namespace {
const tc::BW::Race terran = tc::BW::Race::Terran;
const tc::BW::Race protoss = tc::BW::Race::Protoss;
const tc::BW::Race zerg = tc::BW::Race::Zerg;
const tc::BW::Race unknown = tc::BW::Race::Unknown;
const std::vector<tc::BW::Race> allRaces{terran, protoss, zerg, unknown};
} // namespace

// Update the tournament configuration by modifying this function
BuildOrderConfigurations buildOrdersForTournament(
    const std::string& rawOpponentName) {
  // Start with the default builds, so we know what's valid for Build Order
  // Switch.
  // Then disable all builds as openings, and re-enable them selectively.
  std::unordered_map<std::string, BuildOrderConfig> builds =
      buildOrdersForTraining();
  for (auto& build : builds) {
    build.second.validOpening(false).enemyRaces({});
  }

  auto opponentName = utils::stringToLower(rawOpponentName);
  auto enable = [&](std::vector<tc::BW::Race> addedRaces,
                    const char* buildName) {
    if (!utils::contains(builds, buildName)) {
      VLOG(0) << "WARNING: Trying to enable build not found in training "
                 "configuration: "
              << buildName;
    }
    auto& build = builds[buildName];
    build.validOpening(true);
    std::vector<tc::BW::Race> finalRaces(build.enemyRaces());
    for (auto addedRace : addedRaces) {
      finalRaces.push_back(addedRace);
    }
    build.enemyRaces(std::move(finalRaces));
  };
  auto enableAllRaces = [&](const char* buildName) {
    enable(allRaces, buildName);
  };
  auto isOpponent = [&](const char* name) {
    auto nameString = utils::stringToLower(std::string(name));
    bool output = opponentName.find(nameString) != std::string::npos;
    if (output) {
      VLOG(0) << fmt::format(
          "Found build configuration named {} matching opponent {}",
          name,
          rawOpponentName,
          opponentName);
    }
    return output;
  };

  // Default builds per race

  VLOG(0) << "Using default tournament bandit configuration";

  return builds;
}

// Update the training configuration by modifying this function
BuildOrderConfigurations buildOrdersForTraining() {
  std::unordered_map<std::string, BuildOrderConfig> builds;

  builds["ttest"]
      .validOpening(true)
      .ourRaces({terran});

  builds["tbunkerrush"]
      .validOpening(true)
      .ourRaces({terran});

  builds["prush"]
      .validOpening(true)
      .ourRaces({protoss});

  builds["pcannonrush"]
      .validOpening(true)
      .ourRaces({protoss});

  builds["pvtgreed"]
      .validOpening(true)
      .ourRaces({protoss});

  builds["pffe1"]
      .validOpening(true)
      .ourRaces({protoss});

  builds["pvpstd"]
      .validOpening(true)
      .ourRaces({protoss});

  builds["pml"]
      .validOpening(true)
      .ourRaces({protoss});

  builds["zmidmassling"]
      .validOpening(true)
      .ourRaces({zerg});

  builds["tvpjoyorush"]
      .validOpening(true)
      .ourRaces({terran});

  builds["trush"]
      .validOpening(true)
      .ourRaces({terran});

  builds["ptest"]
      .validOpening(true)
      .ourRaces({protoss});

  builds["tturtle"]
      .validOpening(true)
      .ourRaces({terran});

  builds["zcheese"]
      .validOpening(true)
      .ourRaces({zerg});

  builds["tcheese"]
      .validOpening(true)
      .ourRaces({terran});

  // clang-format on
  return builds;
}

} // namespace model

} // namespace cherrypi
