#include "EggBot.h"
#include <iostream>
#include <fstream>

using namespace BWAPI;
using namespace Filter;

void EggBot::onStart()
{
	homeBaseLocation = BWAPI::Broodwar->self()->getStartLocation();

	// Get all base locations.
	for (auto positions : BWAPI::Broodwar->getStartLocations())
	{
		if (positions != BWAPI::Broodwar->self()->getStartLocation())
		{
			baseLocations.push_back(positions);
		}
	}

	patrolLocations = baseLocations;

	// Set the command optimization level so that common commands can be grouped
	// and reduce the bot's APM (Actions Per Minute).
	Broodwar->setCommandOptimizationLevel(2);

	// Check if this is a replay
	if (Broodwar->isReplay())
	{
		// Announce the players in the replay
		Broodwar << "The following players are in this replay:" << std::endl;

		// Iterate all the players in the game using a std:: iterator
		Playerset players = Broodwar->getPlayers();
		for (auto p : players)
		{
			// Only print the player if they are not an observer
			if (!p->isObserver())
				Broodwar << p->getName() << ", playing as " << p->getRace() << std::endl;
		}
	}
	else // if this is not a replay
	{
		// Retrieve you and your enemy's races. enemy() will just return the first enemy.
		// If you wish to deal with multiple enemies then you must use enemies().
		if (Broodwar->enemy()) // First make sure there is an enemy
			Broodwar << "The matchup is " << Broodwar->self()->getRace() << " vs " << Broodwar->enemy()->getRace() << std::endl;
	}
}

// Commented out for tournament purposes, but this is used to log match results for testing

// void EggBot::onEnd(bool isWinner)
// {
// 	// This code logs the results of the match to a the record.txt file in the starcraft directory.
// 	// We later parse this file with a python script to compile win %s.
// 	std::ofstream record;
// 	record.open("record.txt", std::ios_base::app);
// 	if (isWinner)
// 	{
// 		record << "win " << Broodwar->enemy()->getRace() << "\n";
// 	}
// 	else
// 	{
// 		record << "loss " << Broodwar->enemy()->getRace() << "\n";
// 	}
// 	record.close();
// }

void EggBot::onFrame()
{
	// Called once every game frame

	// The give up timer is designed to leave the game after an hour. This results in a loss.
	// This timer is disabled for competition play purposes, but was used for testing.

	// giveUpTimer = BWAPI::Broodwar->getFrameCount();
	// if (giveUpTimer >= 86400)
	// {
	// 	Broodwar->leaveGame();
	// }

	// Display the game frame rate as text in the upper left area of the screen
	// Broodwar->drawTextScreen(200, 0, "FPS: %d", Broodwar->getFPS());
	// Broodwar->drawTextScreen(200, 20, "Average FPS: %f", Broodwar->getAverageFPS());
	// Return if the game is a replay or is paused
	if (Broodwar->isReplay() || Broodwar->isPaused() || !Broodwar->self())
		return;

	// Prevent spamming by only running our onFrame once every number of latency frames.
	// Latency frames are the number of frames before commands are processed.
	if (Broodwar->getFrameCount() % Broodwar->getLatencyFrames() != 0)
		return;

	// Count all the important units we have.
	numGateways = 0;
	numProbes = 0;
	numCannons = 0;
	numPylons = 0;
	numZealots = 0;
	for (auto& u : Broodwar->self()->getUnits())
	{
		if (u->getType() == UnitTypes::Protoss_Gateway)
		{
			numGateways = numGateways + 1;
			gateway = true;
		}
		if (u->getType() == UnitTypes::Protoss_Probe)
		{
			numProbes = numProbes + 1;
		}
		if (u->getType() == UnitTypes::Protoss_Photon_Cannon)
		{
			numCannons = numCannons + 1;
		}
		if (u->getType() == UnitTypes::Protoss_Pylon)
		{
			townPylon = true;
			numPylons = numPylons + 1;
			building = false;
		}
		if (u->getType() == UnitTypes::Protoss_Zealot)
		{
			numZealots = numZealots + 1;
		}
		if (u->getType() == UnitTypes::Protoss_Forge)
		{
			forge = true;
		}
	}

	// Iterate through all the units that we own
	for (auto& u : Broodwar->self()->getUnits())
	{
		// Ignore the unit if it no longer exists
		// Make sure to include this block when handling any Unit pointer!
		if (!u->exists())
			continue;

		// Ignore the unit if it has one of the following status ailments
		if (u->isLockedDown() || u->isMaelstrommed() || u->isStasised())
			continue;

		// Ignore the unit if it is in one of the following states
		if (u->isLoaded() || !u->isPowered() || u->isStuck())
			continue;

		// Ignore the unit if it is incomplete or busy constructing
		if (!u->isCompleted() || u->isConstructing())
			continue;

		// If the unit is a gateway, we should be training zealots with it.
		if (u == gatewayRef && (u->isIdle() && !u->train(UnitTypes::Protoss_Zealot)))
		{
			continue;
		}
		// If the unit is a worker unit
		if (u->getType() == UnitTypes::Protoss_Gateway)
		{
			gateway = true;
			gatewayRef = u;
		}
		else if (u->getType().isWorker())
		{
			bool built = false;
			// While there is bases to scout and we are currently not scouting, continue scouting.
			if (scouting == false && foundEnemy == false && baseLocations.size() != 0 )
			{
				if (scout == NULL)
				{
					// Our scout is our most important unit so we set it as a variable.
					scout = u;
				}
				// Move to the next base in our list of start locations.
				scout->move(BWAPI::Position(baseLocations[baseLocations.size() - 1]));
				scouting = true;
				baseLocations.pop_back();
			}
			// If our scout dies, we want to replace it with another probe.
			else if (foundEnemy == true && scout == NULL)
			{
				// Reset the scout if none exists.
				scout = u;
				scout->move(BWAPI::Position(enemyBaseLocation));
				Broodwar->sendText("replacement scout chosen");
			}
			// If the worker is not our scout and we don't have any pylons in our town, then build a pylon.
			else if (!townPylon && u != scout && numPylons < 1 && townTimer + 100 < Broodwar->getFrameCount() && (Broodwar->self()->minerals() >= UnitTypes::Protoss_Pylon.mineralPrice()) && BWAPI::Broodwar->self()->supplyUsed() + 10 >= BWAPI::Broodwar->self()->supplyTotal())
			{
				townTimer = Broodwar->getFrameCount();
				built = constructBuilding(u, UnitTypes::Protoss_Pylon);
				if (built)
				{
					Broodwar->sendText("townPylon");
				}
			}
			// If the worker is not our scout and we don't have a forge, build a forge.
			else if (u != scout && !forge && (Broodwar->self()->minerals() >= UnitTypes::Protoss_Forge.mineralPrice()) && forgeTimer + 100 < Broodwar->getFrameCount() && numPylons >= 1)
			{
				forgeTimer = Broodwar->getFrameCount();
				bool built = constructBuilding(u, UnitTypes::Protoss_Forge);
				if (built)
				{
					Broodwar->sendText("forge");
				}
			}

			// if our worker is idle
			if (u->isIdle())
			{
				if (u == scout)
				{
					scouting = false;
					// Randomly move our scout to possibly get unstuck if it has stopped moving.
					if (cannonTimer + 900 < Broodwar->getFrameCount())
					{
						u->move(BWAPI::Position(u->getPosition().x + rand() % 3 - 1, u->getPosition().y + rand() % 3 - 1));
						scoutIdleTimer = Broodwar->getFrameCount();
					}
					// If we haven't made a pylon in the enemy base, make one.
					if (u->getDistance((BWAPI::Position)homeBaseLocation) > 500)
					{
						if (numPylons == 1 && (Broodwar->self()->minerals() >= UnitTypes::Protoss_Pylon.mineralPrice()))
						{
							built = constructBuilding(u, UnitTypes::Protoss_Pylon);
							if (built)
							{
								rushPylonLocation = u->getPosition();
								Broodwar->sendText("rushPylon");
							}
							else
							{
								Broodwar->sendText("rushPylon build failed");
							}
						}
						// If we have a forge and less than 8 cannons, then build more cannons.
						else if (forge && numCannons < 8 && (Broodwar->self()->minerals() >= UnitTypes::Protoss_Photon_Cannon.mineralPrice()) && cannonTimer + 100 < Broodwar->getFrameCount())
						{
							cannonTimer = Broodwar->getFrameCount();
							built = constructBuilding(u, UnitTypes::Protoss_Photon_Cannon);
							if (built)
							{
								Broodwar->sendText("cannon");
							}
							else
							{
								Broodwar->sendText("cannon build failed");
							}
						}
						// If we don't have a gateway and we have at least 4 cannons, then build a gateway.
						else if (!gateway && numCannons >= 4 && forge && (Broodwar->self()->minerals() >= UnitTypes::Protoss_Gateway.mineralPrice()) && gatewayTimer + 300 < Broodwar->getFrameCount())
						{
							gatewayTimer = Broodwar->getFrameCount();
							built = constructBuilding(u, UnitTypes::Protoss_Gateway);
							if (built)
							{
								Broodwar->sendText("gateway");
							}
							else
							{
								Broodwar->sendText("gateway build failed");
							}
						}
					}
				}

				// Order workers carrying a resource to return them to the center,
				// otherwise find a mineral patch to harvest.
				else if (u->isCarryingGas() || u->isCarryingMinerals())
				{
					if (builder == u && !forge)
					{
						continue;
					}
					else
					{
						u->returnCargo();
					}
				}
				else if (!u->getPowerUp()) // The worker cannot harvest anything if it
				{						   // is carrying a powerup such as a flag
					// Harvest from the nearest mineral patch or gas refinery
					if (!u->gather(u->getClosestUnit(IsMineralField || IsRefinery)))
					{
						if (!gateway)
						{
							u->build(UnitTypes::Protoss_Gateway, homeBaseLocation);
						}
						u->move((BWAPI::Position)rushPylonLocation);
						// If the call fails, then print the last error message
						Broodwar << Broodwar->getLastError() << std::endl;
					}
				}
			}	 // closure: if idle
			else if (!seenEnemyUnit && u->isMoving())
			{
				BWAPI::Unit closestUnit = u->getClosestUnit();
				// If an enemy unit is near
				if (closestUnit->getPlayer() == Broodwar->enemy())
				{
					// and if they're within x units
					if (u->getDistance(closestUnit) < 200)
					{
						// then move slightly away and keep track that we saw them
						seenEnemyUnit = true;
						avoidUnit(u, closestUnit, 100);
					}
				}
			}
			else
			{
				if (u == scout)
				{
					scoutIdleTimer = 0;
				}
			}
		}
		else if (u->getType().isResourceDepot()) // A resource depot is a Command Center, Nexus, or Hatchery
		{
			// Order the depot to construct more workers! But only when it is idle.
			if (u->isIdle() && !building && (numProbes < 8 || forge) && numProbes < 8 && !u->train(u->getType().getRace().getWorker()))
			{
				// Error message text appears above the unit
				Position pos = u->getPosition();
				Error lastErr = Broodwar->getLastError();
				Broodwar->registerEvent([pos, lastErr](Game*) { Broodwar->drawTextMap(pos, "%c%s", Text::White, lastErr.c_str()); },
					nullptr,																					 
					Broodwar->getLatencyFrames());																  

				// Retrieve the supply provider type in the case that we have run out of supplies
				UnitType supplyProviderType = u->getType().getRace().getSupplyProvider();
				static int lastChecked = 0;

				// If we are supply blocked and haven't tried constructing more recently
				if (lastErr == Errors::Insufficient_Supply &&
					lastChecked + 400 < Broodwar->getFrameCount() &&
					Broodwar->self()->incompleteUnitCount(supplyProviderType) == 0)
				{
					lastChecked = Broodwar->getFrameCount();

					// Retrieve a unit that is capable of constructing the supply needed
					Unit supplyBuilder = u->getClosestUnit(GetType == supplyProviderType.whatBuilds().first &&
						(IsIdle || IsGatheringMinerals) &&
						IsOwned);
					// If a unit was found
					if (supplyBuilder)
					{
						if (supplyProviderType.isBuilding())
						{
							TilePosition targetBuildLocation = Broodwar->getBuildLocation(supplyProviderType, supplyBuilder->getTilePosition());
							if (targetBuildLocation)
							{

								// Register an event that draws the target build location
								Broodwar->registerEvent([targetBuildLocation, supplyProviderType](Game*) {
									Broodwar->drawBoxMap(Position(targetBuildLocation),
										Position(targetBuildLocation + supplyProviderType.tileSize()),
										Colors::Blue);
									},
									nullptr,							   // condition
									supplyProviderType.buildTime() + 100); // frames to run

								// Order the builder to construct the supply structure
								supplyBuilder->build(supplyProviderType, targetBuildLocation);
								townPylon = true;
								Broodwar->sendText("townPylon");
							}
						}
						else
						{
							// Train the supply provider (Overlord) if the provider is not a structure
							supplyBuilder->train(supplyProviderType);
						}
					}	// closure: supplyBuilder is valid
				}	// closure: insufficient supply
			}	// closure: failed to train idle unit
		}
		// If the unit is a Zealot and we have at least 5 Zealots, then attack the closest visible enemy.
		else if ((u->getType() == UnitTypes::Protoss_Zealot) && u->isIdle() && numZealots >= numZealotRush)
		{
			if (u->getClosestUnit(Filter::IsEnemy) != NULL)
			{
				u->attack(u->getClosestUnit(Filter::IsEnemy));
			}
			// If there are no visible enemies, attack an enemy position from our position map.
			else
			{
				u->move((Position)enemyBaseLocation);
				auto i = enemyUnits.begin();
				if (enemyUnits.size() != 0)
				{
					std::advance(i, rand() % enemyUnits.size());
				}
				u->attack(i->second);
			}
		}
	} // closure: unit iterator

	// iterate through enemy units
	for (auto& u : Broodwar->enemy()->getUnits())
	{
		if (foundEnemy == false)
		{
			enemyBaseLocation = u->getTilePosition();
			// Once we find the enemy base, we can clear all baseLocations as we no longer need to scout.
			foundEnemy = true;
			seenEnemyUnit = true;
			baseLocations.clear();

			if (!probeStopped && scout != NULL)
			{
				// If the enemy is Terran
				if (BWAPI::Broodwar->enemy()->getRace() == Races::Terran)
				{
					// Move slightly away from spotted enemy before beginning to build
					avoidUnit(scout, u, 100);
				}
				else
				{
					// Otherwise, stop
					scout->stop();
					probeStopped = true;
				}
			}
		}
		// Add enemy positions to our enemyUnit map to track units in the fog.
		enemyUnits.insert(std::make_pair(u->getID(), u->getPosition()));
	} // closure: enemy unit iterator
}

void EggBot::onSendText(std::string text)
{
	// Send the text to the game if it is not being processed.
	Broodwar->sendText("%s", text.c_str());
}

void EggBot::onReceiveText(BWAPI::Player player, std::string text)
{
	// Parse the received text
	Broodwar << player->getName() << " said \"" << text << "\"" << std::endl;
}


// void EggBot::onUnitCreate(BWAPI::Unit unit)
// {
// 	if (Broodwar->isReplay())
// 	{
// 		// If we are in a replay, then we will print out the build order of the structures
// 		if (unit->getType().isBuilding() && !unit->getPlayer()->isNeutral())
// 		{
// 			int seconds = Broodwar->getFrameCount() / 24;
// 			int minutes = seconds / 60;
// 			seconds %= 60;
// 			Broodwar->sendText("%.2d:%.2d: %s creates a %s", minutes, seconds, unit->getPlayer()->getName().c_str(), unit->getType().c_str());
// 		}
// 	}
// }

void EggBot::onUnitDestroy(BWAPI::Unit unit)
{
	if (unit->getPlayer()->isEnemy(Broodwar->self()))
	{
		enemyUnits.erase(unit->getID());
	}

	if (unit->getType() == UnitTypes::Protoss_Gateway)
	{
		gatewayRef = NULL;
		gateway = false;
		Broodwar->sendText("gateway destroyed");
	}

	// If dead unit was the scout, set flag to have it replaced
	if (unit == scout)
	{
		scout = NULL;
		seenEnemyUnit = false;
		Broodwar->sendText("scout has died");
	}
}

// Function to make a unit construct a building.
bool EggBot::constructBuilding(BWAPI::Unit unit, BWAPI::UnitType type)
{
	TilePosition targetBuildLocation = Broodwar->getBuildLocation(type, unit->getTilePosition());
	if (targetBuildLocation)
	{
		int radius = 150;
		int x = 0; int y = 0;
		for (x = 0; x < 10000; x += 200)
		{
			for (y = 0; y < 10000; y += 200)
			{
				if (Broodwar->getUnitsInRadius(targetBuildLocation.x + x, targetBuildLocation.y + y, radius).size() == 0
					&& (Broodwar->canBuildHere(targetBuildLocation + TilePosition(x, y), type)))
				{
					targetBuildLocation.x += x;
					targetBuildLocation.y += y;
					Broodwar->registerEvent([targetBuildLocation, type](Game*)
						{
							Broodwar->drawCircleMap(Position(targetBuildLocation), 150, Colors::Yellow, false);
						},
						nullptr,  // condition
						type.buildTime() + 100);  // frames to run

					goto exit;
				}
				else if (Broodwar->getUnitsInRadius(targetBuildLocation.x + x, targetBuildLocation.y - y, radius).size() == 0
					&& (Broodwar->canBuildHere(targetBuildLocation + TilePosition(x, y * (-1)), type)))
				{
					targetBuildLocation.x += x;
					targetBuildLocation.y -= y;
					goto exit;
				}
				else if (Broodwar->getUnitsInRadius(targetBuildLocation.x - x, targetBuildLocation.y + y, radius).size() == 0
					&& (Broodwar->canBuildHere(targetBuildLocation + TilePosition(x * (-1), y), type)))
				{
					targetBuildLocation.x -= x;
					targetBuildLocation.y += y;
					goto exit;
				}
				else if (Broodwar->getUnitsInRadius(targetBuildLocation.x - x, targetBuildLocation.y - y, radius).size() == 0
					&& (Broodwar->canBuildHere(targetBuildLocation - TilePosition(x, y), type)))
				{
					targetBuildLocation.x -= x;
					targetBuildLocation.y -= y;
					goto exit;
				}
				else
				{
					if (unit->isIdle())
					{
						unit->move(BWAPI::Position(unit->getPosition().x + rand() % 5 - 1, unit->getPosition().y + rand() % 5 - 1), true);
					}
					Broodwar->sendText("can't build here");
				}
			}
		}
	exit:
		// Register an event that draws the target build location
		Broodwar->registerEvent([targetBuildLocation, type](Game*)
			{
				Broodwar->drawBoxMap(Position(targetBuildLocation),
					Position(targetBuildLocation + type.tileSize()),
					Colors::Blue);
			},
			nullptr,  // condition
				type.buildTime() + 100);  // frames to run

				// Order the builder to construct the supply structure
		return unit->build(type, targetBuildLocation);
	}
}

void EggBot::avoidUnit(BWAPI::Unit target, BWAPI::Unit other, int distance)
{
	// Get position of both units
	auto targetPos = target->getPosition();
	auto otherPos = other->getPosition();

	auto moveX = (targetPos.x - otherPos.x) > 0 ? 1 : -1;
	auto moveY = (targetPos.y - otherPos.y) > 0 ? 1 : -1;

	// Move target away from the other unit
	target->move(BWAPI::Position(targetPos.x + (moveX * distance), targetPos.y + (moveY * distance)));
}
