/**
 * Copyright (c) 2015-present, Facebook, Inc.
 * All rights reserved.
 *
 * This source code is licensed under the BSD-style license found in the
 * LICENSE file in the root directory of this source tree. An additional grant
 * of patent rights can be found in the PATENTS file in the same directory.
 */

#include "state.h"

namespace torchcraft {

State::State(bool microBattles, std::set<BW::UnitType> onlyConsiderTypes)
    : RefCounted(),
      frame(new Frame()),
      microBattles_(microBattles),
      onlyConsiderTypes_(std::move(onlyConsiderTypes)) {
  reset();
}

State::State(const State& other)
    : RefCounted(),
      lag_frames(other.lag_frames),
      map_size{other.map_size[0], other.map_size[1]},
      ground_height_data(other.ground_height_data),
      walkable_data(other.walkable_data),
      buildable_data(other.buildable_data),
      map_name(other.map_name),
      map_title(other.map_title),
      start_locations(other.start_locations),
      player_info(other.player_info),
      player_id(other.player_id),
      neutral_id(other.neutral_id),
      replay(other.replay),
      frame(new Frame(other.frame)),
      deaths(other.deaths),
      frame_from_bwapi(other.frame_from_bwapi),
      battle_frame_count(other.battle_frame_count),
      game_ended(other.game_ended),
      game_won(other.game_won),
      battle_just_ended(other.battle_just_ended),
      battle_won(other.battle_won),
      waiting_for_restart(other.waiting_for_restart),
      last_battle_ended(other.last_battle_ended),
      img_mode(other.img_mode),
      screen_position{other.screen_position[0], other.screen_position[1]},
      visibility(other.visibility),
      visibility_size{other.visibility_size[0], other.visibility_size[1]},
      image(other.image),
      image_size{other.image_size[0], other.image_size[1]},
      aliveUnits(other.aliveUnits),
      aliveUnitsConsidered(other.aliveUnitsConsidered),
      units(other.units),
      numUpdates(other.numUpdates),
      microBattles_(other.microBattles_),
      onlyConsiderTypes_(other.onlyConsiderTypes_) {}

State::State(State&& other) : RefCounted(), frame(nullptr) {
  swap(*this, other);
}

State::~State() {
  if (frame) {
    frame->decref();
  }
}

State& State::operator=(State other) {
  swap(*this, other);
  return *this;
}

void swap(State& a, State& b) {
  using std::swap;
  swap(a.lag_frames, b.lag_frames);
  swap(a.map_size[0], b.map_size[0]);
  swap(a.map_size[1], b.map_size[1]);
  swap(a.ground_height_data, b.ground_height_data);
  swap(a.walkable_data, b.walkable_data);
  swap(a.buildable_data, b.buildable_data);
  swap(a.map_name, b.map_name);
  swap(a.map_title, b.map_title);
  swap(a.start_locations, b.start_locations);
  swap(a.player_info, b.player_info);
  swap(a.player_id, b.player_id);
  swap(a.neutral_id, b.neutral_id);
  swap(a.replay, b.replay);
  swap(a.frame, b.frame);
  swap(a.deaths, b.deaths);
  swap(a.frame_from_bwapi, b.frame_from_bwapi);
  swap(a.battle_frame_count, b.battle_frame_count);
  swap(a.game_ended, b.game_ended);
  swap(a.game_won, b.game_won);
  swap(a.battle_just_ended, b.battle_just_ended);
  swap(a.battle_won, b.battle_won);
  swap(a.waiting_for_restart, b.waiting_for_restart);
  swap(a.last_battle_ended, b.last_battle_ended);
  swap(a.img_mode, b.img_mode);
  swap(a.screen_position[0], b.screen_position[0]);
  swap(a.screen_position[1], b.screen_position[1]);
  swap(a.visibility, b.visibility);
  swap(a.visibility_size[0], b.visibility_size[0]);
  swap(a.screen_position[1], b.screen_position[1]);
  swap(a.image, b.image);
  swap(a.image_size[0], b.image_size[0]);
  swap(a.image_size[1], b.image_size[1]);
  swap(a.aliveUnits, b.aliveUnits);
  swap(a.aliveUnitsConsidered, b.aliveUnitsConsidered);
  swap(a.units, b.units);
  swap(a.numUpdates, b.numUpdates);
  swap(a.microBattles_, b.microBattles_);
  swap(a.onlyConsiderTypes_, b.onlyConsiderTypes_);
}

void State::reset() {
  replay = false;
  lag_frames = 0;
  map_size[0] = 0;
  map_size[1] = 0;
  ground_height_data.clear();
  walkable_data.clear();
  buildable_data.clear();
  map_name.clear();
  map_title.clear();
  start_locations.clear();
  player_info.clear();
  frame->clear();
  deaths.clear();
  frame_from_bwapi = 0;
  battle_frame_count = 0;
  game_ended = false;
  game_won = false;
  battle_just_ended = false;
  battle_won = false;
  waiting_for_restart = microBattles_;
  last_battle_ended = 0;
  img_mode.clear();
  screen_position[0] = -1;
  screen_position[1] = -1;
  image.clear(); // XXX invalidates existing tensors pointing to it
  image_size[0] = 0;
  image_size[1] = 0;
  aliveUnits.clear();
  aliveUnitsConsidered.clear();
  units.clear();

  numUpdates++;
}

} // namespace torchcraft
