/**
 * Copyright (c) 2015-present, Facebook, Inc.
 * All rights reserved.
 *
 * This source code is licensed under the BSD-style license found in the
 * LICENSE file in the root directory of this source tree. An additional grant
 * of patent rights can be found in the PATENTS file in the same directory.
 */

#include <iostream>
#include <mutex>
#include <random>
#include <sstream>

#include "torchcraft/client.h"
#include "torchcraft/state.h"

extern void controllerInit(torchcraft::State*);
extern void controllerExecute(const std::vector<torchcraft::Client::Command>& commands);
extern void controllerUpdate(torchcraft::State*);

namespace torchcraft {
class Connection {};
}

namespace {

std::string makeUid(size_t len = 6) {
  thread_local std::mt19937 rng = std::mt19937(std::random_device()());

  static const char alphanum[] =
      "0123456789"
      "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
      "abcdefghijklmnopqrstuvwxyz";
  std::uniform_int_distribution<int> dis(0, sizeof(alphanum) - 2);
  std::string s(len, 0);
  for (size_t i = 0; i < len; i++) {
    s[i] = alphanum[dis(rng)];
  }
  return s;
}

std::once_flag initFlag; // For protecting doInit(); see init() below
void doInit() {
  torchcraft::BW::data::init();
}

} // namespace

namespace torchcraft {

void init() {
  std::call_once(initFlag, doInit);
}

//============================= LIFECYCLE ====================================

Client::Client() : state_(new State()) {}

Client::~Client() {
  state_->decref();
}

//============================= OPERATIONS ===================================


bool Client::connect(
    const std::string& hostname,
    int port,
    int timeoutMs /* = -1 */) {
  return true;
}

bool Client::connect(const std::string& file_socket, int timeoutMs /* = -1 */) {
  return true;
}

bool Client::close() {
  clearError();
  return true;
}

bool Client::init(std::vector<std::string>& updates, const Options& opts) {

  state_->setMicroBattles(false);
  state_->setOnlyConsiderTypes({});
  controllerInit(state_);
  return true;
}

bool Client::send(const std::vector<Command>& commands) {
  clearError();

  controllerExecute(commands);

  sent_ = true;
  lastCommands_ = commands;
  lastCommandsStatus_.clear();
  return true;
}

namespace {
  template<typename T> const T* as(const void* msgData) {
    return reinterpret_cast<const T*>(msgData);
  }
}

bool Client::receive(std::vector<std::string>& updates) {
  if (!sent_) {
    send(std::vector<Command>());
  }

  controllerUpdate(state_);
  return true;
}

bool Client::poll(long timeout) {
  clearError();
  return false;
}

} // namespace torchcraft
