/*
 * Copyright (c) 2017-present, Facebook, Inc.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */

#include "behavior.h"
#include "agent.h"
#include "common/rand.h"
#include "fmt/format.h"
#include "squadtask.h"
#include "modules/tactics.h"
#include <bwem/map.h>

#include <BWAPI.h>

namespace cherrypi {

inline float moveAhead(Unit* unit) {
  if (unit->flying() || unit->type == buildtypes::Terran_Vulture) return 4.0f * 8;
  return 12.0f;
}

MicroAction BehaviorAsWorker::onPerform(Agent& agent) {
  auto& task = agent.task;
  State* state = agent.state;
  Unit* unit = agent.unit;
  Unit* target = agent.target;

  if (unit->type != buildtypes::Terran_SCV) {
    return pass;
  }

  if (!unit->unit.orders.empty() && unit->unit.orders.front().type == +tc::BW::Order::ConstructingBuilding) {
    return doNothing;
  }

  Position moveTo = kInvalidPosition;

  float averageHealth = 0.0f;
  int n = 0;
  for (Unit* u : task->squadUnits()) {
    averageHealth += u->unit.health / (float)u->unit.max_health;
    ++n;
  }
  if (n) {
    averageHealth /= n;
  }

  bool mineralWalking = state->currentFrame() < agent.mineralWalkUntil;

//  if (!mineralWalking && unit->unit.orders.front().type == +tc::BW::Order::MoveToMinerals) {
//    for (Unit* u : task->squadUnits()) {
//      if (u->unit.orders.front().type != +tc::BW::Order::MoveToMinerals) {
//        if (utils::intersects(unit, u)) {
//          mineralWalking = true;
//        }
//      }
//    }
//  }

  int nInRange = 0;
  float incomingDamage = 0.0f;
  int nEnemiesNearby = 0;
  Unit* threat = utils::getBestScoreCopy(
      task->targets_,
      [&](Unit* e) {
        if (!unit->canAttack(e)) {
          return kfInfty;
        }
        float d = utils::distance(unit, e->pos());
        if (d > e->unit.groundRange + 12.0f) {
          return kfInfty;
        }
        ++nEnemiesNearby;
        if (unit->inRangeOfPlus(e, mineralWalking ? 6.0f : 1.0f)) {
          int hp = 0;
          int shield = 0;
          e->computeDamageTo(unit, &hp, &shield);
          incomingDamage += hp + shield;
          ++nInRange;
        }
        return d;
      },
      kfInfty);

  if (threat && unit->canAttack(threat) && (!target || utils::distanceBB(unit, threat) < utils::distanceBB(unit, target))) {
    target = threat;
  }

  if (unit->carryingResources() && (!threat || utils::distanceBB(unit, threat) >= threat->rangeAgainst(unit) + 12.0)) {
    Unit* cc = utils::getBestScoreCopy(state->unitsInfo().myResourceDepots(), [&](Unit* u) {
      if (u->lifted()) {
        return kfInfty;
      }
      return utils::distance(u->pos(), unit->pos());
    }, kfInfty);
    if (cc && utils::distanceBB(unit, cc) <= 4.0f * 8) {
      if (unit->limitMine(state)) {
        agent.postCommand(tc::BW::UnitCommandType::Return_Cargo);
      }
      return doNothing;
    }
  }

  if (nInRange >= 4 && target) {
    int nAlliesNearby = 0;
    for (Unit* u : task->squadUnits()) {
      if (utils::distance(u, target) <= u->unit.groundRange + 12.0f) {
        ++nAlliesNearby;
        if (target->inRangeOfPlus(u, 1.0f)) {
          --nInRange;
        }
      }
    }
    if (nInRange >= 2 && nAlliesNearby < nEnemiesNearby + 2) {
      agent.mineralWalkUntil = state->currentFrame() + 6;
    }
  }

  int latency = state->latencyFrames();

  Vec2 myPos = unit->posf() + unit->velocity() * latency;
  Vec2 targetPos = target ? target->posf() + target->velocity() * latency : Vec2();
  float targetDistance = target ? utils::distanceBB(unit, myPos, target, targetPos) : 0.0f;

  float myRange = target ? unit->rangeAgainst(target) : 0.0f;
  float targetRange = target ? target->rangeAgainst(unit) : 0.0f;
  float maxRange = std::max(myRange, targetRange);

  //Unit* resTarget = nullptr;

  if (unit->combatWorkerAutoRetreat && target && (target->rangeAgainst(unit) <= 4.0f && (!threat || threat->rangeAgainst(unit) <= 4.0f))) {
    //if ((unit->unit.health - std::min(incomingDamage, 20.0f) <= 10 || mineralWalking) && state->resources().ore >= 10) {
    if (unit->unit.health - std::min(incomingDamage, 30.0f) <= 10 || mineralWalking) {
      //unit->combatWorker = false;

      Unit* allyWorker = utils::getBestScoreCopy(state->unitsInfo().myWorkers(), [&](Unit* u) {
        if (u->type != buildtypes::Terran_SCV || u == unit || !u->combatWorker) {
          return kfInfty;
        }
        return unit->flying() ? utils::distance(target->pos(), u->pos()) : state->areaInfo().walkPathLength(target->pos(), u->pos());
      }, kfInfty);

      if (allyWorker && utils::distance(allyWorker, unit) <= 4.0f * 128) {

        if (threat && (unit->inRangeOfPlus(threat, 6.0f) || mineralWalking)) {
        //if (threat && unit->inRangeOfPlus(threat, 9.0f)) {
          Unit* res = utils::getBestScoreCopy(state->unitsInfo().resourceUnits(), [&](Unit* u) {
            if (!u->type->isMinerals || !u->visible) {
              return kfInfty;
            }
            return utils::distance(u->pos(), state->areaInfo().myStartLocation());
          }, kfInfty);
          if (res) {
            agent.unstuckCounter = 0;
            if (state->currentFrame() >= agent.mineralWalkUntil) {
              agent.mineralWalkUntil = state->currentFrame() + 3;
            }
            return doAction(agent.repair(res));
//            resTarget = res;
          }
        } else {
          moveTo = allyWorker->pos();
        }
      }

    }
    if (unit->unit.health <= unit->unit.max_health / 2) {
      Unit* cc = utils::getBestScoreCopy(state->unitsInfo().myResourceDepots(), [&](Unit* u) {
        float d = utils::distance(u->pos(), unit->pos());
        if (d > 4.0f * 20) {
          return kfInfty;
        }
        return d;
      }, kfInfty);
      if (cc) {
        unit->combatWorker = false;
      }
    }
  }

  if (unit->stuck() && target) {
    agent.lastMove = 0;
    agent.lastMoveTo = kInvalidPosition;
    agent.lastAttack = 0;
    agent.attacking = nullptr;
    return doAction(agent.attack(target));
    //return doAction(agent.moveTo(target->posf()));
  }

  bool targetNear = target && utils::distance(unit, target) <= 4.0f * 10;

  bool protect = false;
  if (utils::distance(unit->pos(), state->areaInfo().myStartLocation()) > utils::distance(unit->pos(), state->areaInfo().enemyStartLocation())) {
    protect = true;
  }

  if (targetNear) {
    protect = true;
  }

  Unit* protectUnit = utils::getBestScoreCopy(state->unitsInfo().myUnits(), [&](Unit* u) {
    if (u->type != buildtypes::Terran_Siege_Tank_Siege_Mode && u->type != buildtypes::Terran_Siege_Tank_Tank_Mode && u->type != buildtypes::Terran_Bunker) {
      return kfInfty;
    }
    if (u == unit || !u->active()) {
      return kfInfty;
    }
    return unit->flying() ? utils::distance(unit->pos(), u->pos()) : state->areaInfo().walkPathLength(unit->pos(), u->pos());
  }, kfInfty);

  if (targetNear) {

    Unit* buildUnit = utils::getBestScoreCopy(state->unitsInfo().myBuildings(), [&](Unit* u) {
      if (u->type->race != +tc::BW::Race::Terran || u->type->builder != unit->type || u->completed() || u->associatedUnit) {
        return kfInfty;
      }
      if (utils::distance(unit->pos(), u->pos()) >= 4.0f * 8) {
        return kfInfty;
      }
      return unit->flying() ? utils::distance(unit->pos(), u->pos()) : state->areaInfo().walkPathLength(unit->pos(), u->pos());
    }, kfInfty);

    if (buildUnit) {
      return doAction(agent.repair(buildUnit));
    }
  }

  Unit* repair = utils::getBestScoreCopy(state->unitsInfo().myUnits(), [&](Unit* u) {
    if (!u->type->isMechanical || utils::distanceBB(unit, u) > 4.0f * 10 || u->unit.health >= u->unit.max_health || !u->active() || u->loaded()) {
      return kfInfty;
    }
    if (u == unit) {
      return kfInfty;
    }
    return utils::distanceBB(u, unit);
    //return (float)unit->unit.health;
  }, kfInfty);

  bool beingRepaired = false;
  for (Unit* u : unit->unitsInSightRange) {
    if (u->isMine && u->unit.orders.front().type == +tc::BW::Order::Repair && u->unit.orders.front().targetId == unit->id) {
      if (utils::distanceBB(unit, u) <= 2.0f) {
        beingRepaired = true;
        break;
      }
    }
  }

  bool attack = target && (targetDistance <= maxRange);

//  if (!attack && (!repair || utils::distanceBB(unit, repair) > 1.5f)) {
//    auto mineralWalking = [&](Unit* u) {
//      return u->unit.orders.front().type == +tc::BW::Order::MoveToMinerals;
//    };
//    if (!mineralWalking(unit)) {
//      for (Unit* u : task->squadUnits()) {
//        if (u == unit || !mineralWalking(u)) {
//          continue;
//        }
//        Vec2 uPos = u->posf() + u->velocity() * latency;
//        if (u->velocity().dot(unit->posf() - u->posf()) >= 0) {
//          if (utils::distanceBB(unit->type, myPos, u->type, uPos) <= 8.0f) {
//            Vec2 delta = (u->posf() - unit->posf()).normalize() * moveAhead(unit);
//            Vec2 a = unit->posf() + delta.rotateDegrees(90);
//            Vec2 b = unit->posf() + delta.rotateDegrees(-90);
//            if (utils::distancev(uPos, a) > utils::distancev(uPos, b)) {
//              return doAction(agent.moveTo(a));
//            } else {
//              return doAction(agent.moveTo(b));
//            }
//          }
//        }
//      }
//    }
//  }

//  if (beingRepaired && unit->stuck()) {
//    return doAction(agent.holdPosition());
//  }

//  if ((unit->moving() || unit->accelerating()) && unit->velocity() == Vec2()) {
//    if (state->currentFrame() - agent.lastStop >= 30) {
//      agent.lastStop = state->currentFrame();
//      agent.postCommand(tc::BW::UnitCommandType::Stop);
//      return doNothing;
//    }
//  }

  if (repair && state->resources().ore > 2 && (repair->type->isBuilding || !attack)) {
    agent.unstuckCounter = 0;
//    if (unit->velocity() == Vec2() && utils::pxDistanceBB(unit, repair) > 5) {
//      return doAction(agent.moveTo(unit->posf() + (repair->posf() - unit->posf()).normalize() * moveAhead(unit)));
//    }
    if (utils::pxDistanceBB(unit, repair) <= 5) {
      if (agent.lastRepairTarget != repair) {
        agent.lastRepairTarget = repair;
      } else if (state->currentFrame() - agent.lastRepairTargetAction >= 15) {
        if (state->currentFrame() - agent.lastStop >= 15) {
          agent.lastStop = state->currentFrame();
          agent.postCommand(tc::BW::UnitCommandType::Stop);
          return doNothing;
        }
      }
      if (repair->unit.health != agent.lastRepairTargetHp) {
        agent.lastRepairTargetAction = state->currentFrame();
      }
      agent.lastRepairTargetHp = repair->unit.health;
    } else {
      agent.lastRepairTarget = nullptr;
    }
    return doAction(agent.repair(repair));
  }

//  if (resTarget) {
//    agent.unstuckCounter = 0;
//    agent.mineralWalkUntil = state->currentFrame() + 24;
//    return doAction(agent.repair(resTarget));
//  }

  if (protectUnit && utils::distanceBB(protectUnit, unit) <= 4.0f * 10 && state->resources().ore > 10 && utils::pxDistanceBB(protectUnit, unit) > 8) {
    if (protectUnit->unit.health < protectUnit->unit.max_health) {
      return doAction(agent.moveTo(protectUnit->pos()));
    }
  }

  if (attack) {
    if (targetDistance <= myRange) {
      return doAction(agent.attack(target));
    } else {
      return doAction(agent.moveTo(target->posf()));
    }
  }

  if (beingRepaired) {
    return doAction(agent.holdPosition());
  }

  if (moveTo != kInvalidPosition) {
    return doAction(agent.moveTo(moveTo));
  }

  if (target && targetDistance <= myRange + 8.0f) {
    return doAction(agent.moveTo(target->posf()));
  }

  return pass;
}

MicroAction BehaviorTurtle::onPerform(Agent& agent) {
  auto& task = agent.task;
  State* state = agent.state;
  Unit* unit = agent.unit;
  Unit* target = agent.target;

  if (unit->type != buildtypes::Terran_Siege_Tank_Tank_Mode && unit->type != buildtypes::Terran_Siege_Tank_Siege_Mode && unit->type != buildtypes::Terran_Marine) {
    return pass;
  }

  if (unit->type == buildtypes::Terran_Siege_Tank_Tank_Mode && unit->unit.health < unit->unit.max_health) {
    Unit* repairer = utils::getBestScoreCopy(state->unitsInfo().myUnitsOfType(buildtypes::Terran_SCV), [&](Unit* u) {
      float d = utils::distanceBB(unit, u);
      if (d <= 2.0f) {
        if (u->unit.orders.front().type == +tc::BW::Order::Repair && u->unit.orders.front().targetId == unit->id) {
          return d;
        }
      }
      return kfInfty;
    }, kfInfty);
    if (repairer) {
      if (target && agent.targetInRange) {
        return doAction(agent.attack(target));
      }
      return doAction(agent.holdPosition());
    }
  }

  if (unit->type != buildtypes::Terran_Siege_Tank_Tank_Mode && state->currentFrame() < 24 * 60 * 8) {
    if (target && !agent.targetInRange && state->hasResearched(buildtypes::Tank_Siege_Mode)) {
      Unit* nearestTank = utils::getBestScoreCopy(task->squadUnits(), [&](Unit* u) {
        if (u->type != buildtypes::Terran_Siege_Tank_Siege_Mode && u->type != buildtypes::Terran_Siege_Tank_Tank_Mode && u->type != buildtypes::Protoss_Reaver && u->type != buildtypes::Zerg_Lurker) {
          return kfInfty;
        }
        if (u == unit) {
          return kfInfty;
        }
        return unit->flying() ? utils::distance(unit->pos(), u->pos()) : state->areaInfo().walkPathLength(unit->pos(), u->pos());
      }, kfInfty);
      if (!nearestTank) {
        Unit* bunker = utils::getBestScoreCopy(state->unitsInfo().myCompletedUnitsOfType(buildtypes::Terran_Bunker), [&](Unit* u) {
          return utils::distance(u->pos(), target->pos());
        });
        if (bunker && utils::distance(unit->pos(), bunker->pos()) <= 4.0f * 8) {
          if (target) {
            if (utils::distanceBB(target, unit) - 4.0f <= utils::distanceBB(target, bunker)) {
              Unit* cc = utils::getBestScoreCopy(state->unitsInfo().myResourceDepots(), [&](Unit* u) {
                return utils::distance(u->pos(), bunker->pos());
              });
              if (cc) {
                return doAction(agent.moveTo(cc->pos()));
              }
            }
          }
        }
      }
    }
  }

  if (state->currentFrame() >= 24 * 60 * 10 || state->areaInfo().numMyBases() >= 2 || !state->unitsInfo().myUnitsOfType(buildtypes::Terran_Bunker).empty()) {
    return pass;
  }

  bool turtle = false;

  if (unit->targetGroup && !unit->targetGroup->isAggressiveGroup) {
    turtle = true;
  }

  if (!turtle) {
    return pass;
  }

  if (target && agent.targetInRange) {
    return pass;
  }

  Position targetPos = unit->pos();
  if (target) {
    targetPos = target->pos();
  } else if (task->targetingLocation) {
    targetPos = Position(task->targetX, task->targetY);
  } else {
    return pass;
  }

  if (unit->type == buildtypes::Terran_Siege_Tank_Siege_Mode && state->currentFrame() - agent.lastSiege >= 24 * 8) {
    if (state->currentFrame() - unit->lastFiredWeapon >= 24 * 8) {
      agent.postCommand(tc::BW::UnitCommandType::Unsiege);
      return doNothing;
    }
  }

  Unit* threat = utils::getBestScoreCopy(
      task->targets_,
      [&](Unit* e) {
        if (!unit->canAttack(e)) {
          return kfInfty;
        }
        float d = utils::distance(unit, e->pos());
        if (d > e->unit.groundRange + 12.0f) {
          return kfInfty;
        }
        return d;
      },
      kfInfty);

  Position protect = targetPos;

  if (threat) {
    float protectD = utils::distance(targetPos, threat->pos());
    for (Unit* u : state->unitsInfo().myBuildings()) {
      float d = utils::distance(u->pos(), threat->pos());
      if (d < protectD) {
        protectD = d;
        protect = u->pos();
      }
    }
    for (Unit* u : state->unitsInfo().myWorkers()) {
      float d = utils::distance(u->pos(), threat->pos());
      if (d < protectD) {
        protectD = d;
        protect = u->pos();
      }
    }
  }

  bool haveSiegeMode = state->hasResearched(buildtypes::Tank_Siege_Mode);

  if (threat) {
    float range = unit->rangeAgainst(threat);
    if (unit->type == buildtypes::Terran_Siege_Tank_Tank_Mode && haveSiegeMode) {
      range = 4.0f * 12;
    }
    for (Unit* u : agent.task->squadUnits()) {
      if (u->inRangeOf(threat) || threat->inRangeOf(u)) {
        return pass;
      }
    }
    return doNothing;
  }

  float range = unit->unit.groundRange;
  if (unit->type == buildtypes::Terran_Siege_Tank_Tank_Mode && haveSiegeMode) {
    range = 4.0f * 12;
  }

  if (threat) {
    range -= std::max((float)threat->unit.groundRange, 4.0f * 6);
  } else {
    range -= 4.0f * 6;
  }

  Vec2 myPos = unit->posf();

  int n = 0;
  auto vector = [&]() {
    Vec2 adjustment;
    for (Unit* u : agent.task->squadUnits()) {
      if (!u->flying()) {
        float distance =
            std::max(utils::distanceBB(unit, u), DFOASG(0.125f, 1.0f));
        float maxDistance = DFOASG(4 * 10, 4 * 3);
        if (distance <= maxDistance) {
          ++n;
          adjustment += (myPos - u->posf()) * (maxDistance / distance);
        }
      }
    }
    Vec2 moveDir = (myPos - Vec2(targetPos)).normalize();
    return unit->posf() + (moveDir + adjustment.normalize()).normalize() * moveAhead(unit);
  };

  if (utils::distance(unit->pos(), protect) <= std::max(range, 4.0f * 4)) {

    if (state->areaInfo().walkPathLength(unit->pos(), state->areaInfo().myStartLocation()) > state->areaInfo().walkPathLength(protect, state->areaInfo().myStartLocation())) {
      return doAction(agent.moveTo(state->areaInfo().myStartLocation()));
    }

    if (unit->type == buildtypes::Terran_Siege_Tank_Tank_Mode && haveSiegeMode) {

      bool siege = true;
      auto* area = state->areaInfo().tryGetArea(unit->pos());
      if (area) {
        for (auto& choke : area->area->ChokePoints()) {
          Position a(choke->Pos(BWEM::ChokePoint::node::middle));
          if (utils::distance(a, unit->pos()) <= 4.0f * 3) {
            siege = false;
            break;
          }
        }
      }

      if (siege) {
        Vec2 dst = vector();

        if (n < 2 || state->currentFrame() - agent.lastSiege >= 280) {
          if (state->currentFrame() - agent.lastSiege >= 80 && canSiegeHere(state, unit)) {
            agent.lastSiege = state->currentFrame();
            agent.postCommand(tc::BW::UnitCommandType::Siege);
            return doNothing;
          }
        }

        return doAction(agent.moveTo(dst));
      }
    }

    return doAction(agent.moveTo(vector()));
  } else {
    return doAction(agent.moveTo(protect));
  }

  return pass;
}

MicroAction BehaviorAsZergling::onPerform(Agent& agent) {
  auto& task = agent.task;
  State* state = agent.state;
  Unit* unit = agent.unit;
  Unit* target = agent.target;

  if (unit->type != buildtypes::Zerg_Zergling && unit->type != buildtypes::Protoss_Zealot) {
    return pass;
  }

  for (auto u : unit->beingAttackedByEnemies) {
    if (u->type == buildtypes::Terran_Vulture_Spider_Mine) {
      // Suicide into enemy units
      Unit* target = utils::getBestScoreCopy(
          task->targets_,
          [&](Unit* target) {
            if (target->type == buildtypes::Terran_Vulture_Spider_Mine ||
                target->flying()) {
              return kdInfty;
            }
            return utils::distance(unit, target) -
                (target->type->mineralCost + target->type->gasCost) / 16;
          },
          kdInfty);
      if (target) {
        auto pos = Vec2(unit) + (Vec2(target) - Vec2(unit)).normalize() * 10;
        return doAction(agent.moveTo(pos));
      }
    }
  }

  if (target && agent.targetInRange && target->rangeAgainst(unit) > unit->rangeAgainst(target)) {
    return doAction(agent.attack(target));
  }

  if (target == nullptr || agent.targetInRange) {
    return pass;
  }

  if (unit->type == buildtypes::Protoss_Zealot) {
    int nDragoons = 0;
    int nZealots = 0;
    for (Unit* u : task->squadUnits()) {
      if (u->type == buildtypes::Protoss_Dragoon) ++nDragoons;
      else if (u->type == buildtypes::Protoss_Zealot) ++nZealots;
    }
    if (nDragoons > nZealots) {
      Unit* nearestDragoon = utils::getBestScoreCopy(task->squadUnits(), [&](Unit* u) {
        if (u->type != buildtypes::Protoss_Dragoon) {
          return kfInfty;
        }
        return utils::distance(target->pos(), u->pos());
      }, kfInfty);
      if (nearestDragoon && !target->inRangeOf(nearestDragoon) && utils::distanceBB(unit, nearestDragoon) <= 4.0f * 4) {
        return doAction(agent.moveTo(unit->pos() + Position((nearestDragoon->posf() - target->posf()).normalize() * 4.0f * 3)));
      }
    }
  }

  for (Unit* u : unit->unitsInSightRange) {
    if (u->type == buildtypes::Terran_Vulture_Spider_Mine) {
      if (u->attackingTarget) {
        if (utils::distance(unit, u) > 16) {
          continue;
        }
        if (unit != u->attackingTarget) {
          auto pos = Vec2(unit) +
              (Vec2(unit) - Vec2(u->attackingTarget)).normalize() * 10;
          return doAction(agent.moveTo(pos));
        }
      }
    }
  }

  Unit* u = unit;

  // Consider running away from Vultures.
  if (target->visible && target->type == buildtypes::Terran_Vulture &&
      utils::distance(u, target) > DFOASG(8, 4)) {
    int lings = 0;
    int nonLings = 0;
    bool anyInRange = false;
    for (Unit* ally : unit->allyUnitsInSightRange) {
      if (ally->type == buildtypes::Zerg_Zergling) {
        ++lings;
      } else {
        ++nonLings;
        if (target->inRangeOf(ally)) {
          anyInRange = true;
          break;
        }
      }
    }
    if (!anyInRange && lings < DFOASG(8, 4) &&
        nonLings >= lings / DFOASG(2, 1)) {
      Unit* moveToAlly = utils::getBestScoreCopy(
          unit->allyUnitsInSightRange,
          [&](Unit* ally) {
            float d = utils::distance(u, ally);
            if (ally->unit.groundRange <= 12) {
              return kfInfty;
            }
            return d;
          },
          kfInfty);
      if (moveToAlly) {
        if (utils::distance(u, moveToAlly) <= 8) {
          auto pos = Vec2(unit) + (Vec2(unit) - Vec2(target)).normalize() * 10;
          return doAction(agent.moveTo(pos));
        }
        return doAction(agent.moveTo(Position(moveToAlly)));
      }
    }
  }

  auto targetDistanceBB = utils::distanceBB(unit, target);
  auto canMoveInDirection = [&](
      Vec2 dir, float distance = DFOASG(4.0f * 2, 4.0f)) {
    dir = dir.normalize();
    for (float d = 4.0f; d <= distance; d += 4.0f) {
      Position pos = Position(unit->posf() + dir * d);
      auto* tile = state->tilesInfo().tryGetTile(pos.x, pos.y);
      if (!tile || !tile->entirelyWalkable || tile->building) {
        return false;
      }
    }
    return true;
  };

  if (target->type != buildtypes::Zerg_Zergling &&
      canMoveInDirection(target->posf() - unit->posf(), targetDistanceBB)) {
    // Try to surround the target when we have multiple zerglings -
    // The 2 zerglings nearest to the target can attack directly.
    // The other zerglings will move in the direction from the nearest
    // zergling to the target, until/unless they are on the other side
    // of the target.
    std::array<std::pair<Unit*, float>, 2> nearestN;
    nearestN.fill({nullptr, kfInfty});
    float targetd = utils::distanceBB(unit, target);
    for (Unit* u : agent.task->squadUnits()) {
      if (u->type == buildtypes::Zerg_Zergling) {
        float d = utils::distanceBB(u, target);
        if (utils::distanceBB(u, unit) > targetd) continue;
        for (size_t i = 0; i != nearestN.size(); ++i) {
          if (d < nearestN[i].second) {
            for (size_t i2 = nearestN.size() - 1; i2 != i; --i2) {
              nearestN[i2] = nearestN[i2 - 1];
            }
            nearestN[i] = {u, d};
            break;
          }
        }
      }
    }
    Position moveToPos = kInvalidPosition;
    if (targetDistanceBB > nearestN.back().second) {
      Unit* nearest = nearestN.front().first;
      if (nearest &&
          utils::distance(unit, nearest) <= DFOASG(4.0f * 10, 4.0f * 4)) {
        Vec2 targetpos =
            target->posf() + target->velocity() * state->latencyFrames();
        Vec2 myrel = targetpos - unit->posf();
        Vec2 nrel = targetpos - nearest->posf();

        Vec2 move;
        for (Unit* u : task->squadUnits()) {
          if (u->type == buildtypes::Zerg_Zergling) {
            float d = std::max(utils::distanceBB(u, unit), 1.0f);
            if (d <= 4.0f) {
              move += (unit->posf() - u->posf()) / d;
            }
          }
        }

        move = (move.normalize() * 0.25f + nrel.normalize());
        if (myrel.dot(nrel) > 0) {
          move = move.normalize() * 0.5f + myrel.normalize();
        }

        float moveDist = std::min(
            targetDistanceBB + DFOASG(4.0f, 4.0f), DFOASG(12.0f, 8.0f));
        if (canMoveInDirection(move, moveDist)) {
          moveToPos = Position(unit) + Position(move.normalize() * moveDist);
          if (utils::distance(target->pos(), moveToPos) > utils::distance(target->pos(), unit->pos())) {
            moveToPos = kInvalidPosition;
          }
        }

//        if (myrel.dot(nrel) > 0) {
//          float moveDist = std::min(
//              targetDistanceBB + DFOASG(4.0f, 4.0f), DFOASG(12.0f, 8.0f));
//          if (canMoveInDirection(nrel, moveDist)) {
//            moveToPos = Position(unit) + Position(nrel.normalize() * moveDist);
//          }
//        }
      }
    }

    if (moveToPos != kInvalidPosition) {
      return doAction(agent.moveTo(moveToPos, false));
    }
  }

  if (target->visible &&
      (target->type->isWorker || target->type == buildtypes::Terran_Vulture)) {
    auto shouldMoveTo = [&](Vec2& newPos) {
      int n = (int)(utils::distance(Position(target), Position(newPos)) / 4.0f);
      Vec2 step = (newPos - target->posf()).normalize() * 4;
      Vec2 pos = target->posf();
      for (int i = 0; i != n; ++i) {
        if (utils::distance(Position(pos), Position(u)) < DFOASG(8, 4)) {
          return false;
        }
        const Tile* tile =
            state->tilesInfo().tryGetTile((int)pos.x, (int)pos.y);
        if (!tile || !tile->entirelyWalkable) {
          return false;
        }
        pos += step;
      }
      return true;
    };

    if (target->topSpeed >= u->topSpeed * 0.66f && target->moving() &&
        !target->inRangeOf(u, DFOASG(4, 2))) {
      const float latency = state->latencyFrames() + DFOASG(0.0f, 2.0f);
      float weaponRange =
          target->flying() ? u->unit.groundRange : u->unit.airRange;
      auto targetVelocity = target->velocity();
      auto targetNextPos = target->posf() + targetVelocity * latency;
      auto myNextPos = u->posf() + u->velocity() * latency;
      float distance = std::min(
          utils::distanceBB(u, myNextPos, target, targetNextPos),
          utils::distanceBB(u, u->posf(), target, targetNextPos));
      if (distance > weaponRange) {
        float distance = utils::distance(u->x, u->y, target->x, target->y);
        if (utils::distance(u->x, u->y, targetNextPos.x, targetNextPos.y) >
            distance) {
          auto np =
              u->posf() + targetVelocity.normalize() * DFOASG(16.0f, 8.0f);
          if (shouldMoveTo(np)) {
            return doAction(agent.moveTo(np));
          }
        } else {
          auto np = target->posf() +
              targetVelocity.normalize() *
                  std::min(
                      std::max(
                          distance - DFOASG(4.0f, 4.0f), DFOASG(4.0f, 4.0f)),
                      DFOASG(20.0f, 8.0f));
          if (shouldMoveTo(np)) {
            return doAction(agent.moveTo(np));
          } else {
            auto np = target->posf() +
                targetVelocity.normalize() *
                    std::min(
                        std::max(
                            distance - DFOASG(4.0f, 4.0f), DFOASG(4.0f, 4.0f)),
                        DFOASG(12.0f, 8.0f));
            if (shouldMoveTo(np)) {
              return doAction(agent.moveTo(np));
            }
          }
        }
      }
    }
  }

  return pass;
}

MicroAction BehaviorAsShuttle::onPerform(Agent& agent) {
  State* state = agent.state;
  Unit* unit = agent.unit;
  Unit* target = agent.target;
  auto* task = agent.task;

  if (unit->type != buildtypes::Protoss_Shuttle) {
    return pass;
  }

  Unit* nearestThreat = utils::getBestScoreCopy(agent.legalTargets, [&](Unit* u) {
    if (!u->canAttack(unit) || u->type == buildtypes::Protoss_Interceptor) return kfInfty;
    return utils::distance(u, unit) - (float)u->rangeAgainst(unit);
  }, kfInfty);

  auto bwl = bwapiLock();
  auto bwu = BWAPI::Broodwar->getUnit(unit->id);
  if (!bwu) return pass;

  auto loadedUnits = bwu->getLoadedUnits();

  auto unload = [&]() {
    if (!unit->limitSpecial(state)) return;
    for (auto x : loadedUnits) {
      bwu->unload(x);
      break;
    }
  };

  bool noop = false;

  int space = BWAPI::UnitType(unit->type->unit).spaceProvided();
  for (auto x : loadedUnits) {
    space -= x->getType().spaceRequired();
    if (!state->unitsInfo().getUnit(x->getID())) {
      LOG(FATAL) << "loaded unit does not exist";
    }
    if (state->unitsInfo().getUnit(x->getID())->dead) {
      LOG(FATAL) << "loaded unit does is dead";
    }
    auto* u = state->unitsInfo().getUnit(x->getID());
    if (!u->isMine) {
      LOG(FATAL) << "loaded unit does is not mine!";
    }
    if (u->gone) {
      LOG(FATAL) << "loaded unit does is gone!";
    }
    bool found = false;
    for (Unit* z : state->unitsInfo().myUnits()) {
      if (z == u) found = true;
    }
    if (!found) {
      LOG(FATAL) << "loaded unit does is in myUnits!";
    }
    VLOG(0) << "loaded unit exists yey";

    //if (std::find(task->squadUnits().begin(), task->squadUnits().end(), u) == task->squadUnits().end()) {
    if (task->agents_->find(u) == task->agents_->end()) {
      if (unit->limitSpecial(state)) bwu->unload(x);
      noop = true;
    }
  }

  std::unordered_map<int, int> incomingBullets;

  for (auto& v : BWAPI::Broodwar->getBullets()) {
    if (v->getType() == BWAPI::BulletTypes::Phase_Disruptor) {
      auto target = v->getTarget();
      if (target) {
        ++incomingBullets[target->getID()];
      }
    }
  }

  Unit* pickup = space ? utils::getBestScoreCopy(state->unitsInfo().myUnits(), [&](Unit* u) {
    if (!u->active()) return kfInfty;
    if (u->flying() || u->loaded()) return kfInfty;
    if (task->agents_->find(u) == task->agents_->end()) return kfInfty;
    if (u->type != buildtypes::Protoss_Zealot && u->type != buildtypes::Protoss_Dragoon && u->type != buildtypes::Protoss_Reaver) return kfInfty;
    if (u->type != buildtypes::Protoss_Reaver) return kfInfty;
    int req = BWAPI::UnitType(u->type->unit).spaceRequired();
    if (space < req) return kfInfty;
    float r = utils::distance(unit->pos(), u->pos());
    if (r <= 4.0f * 7) {
      auto i = incomingBullets.find(u->id);
      if (i != incomingBullets.end()) {
        if (r <= 4.0f) {
          r -= 4.0f * 100;
        }
        r -= 4.0f * 10 * i->second;
      }
    }
    if (u->type == buildtypes::Protoss_Dragoon) r += 4.0f * 8;
    if (u->type == buildtypes::Protoss_Zealot) r += 4.0f * 16;
    return r;
  }, kfInfty) : nullptr;

  float incomingDpf = 0.0f;

  if (nearestThreat) {

    int latency = state->latencyFrames();

    Vec2 myPos = unit->posf() + unit->velocity() * latency;
    Vec2 targetPos = nearestThreat->posf() + nearestThreat->velocity() * latency;

    auto kiteVector = [&]() {
      Vec2 adjustment;
      for (Unit* u : agent.legalTargets) {
        if (u->canAttack(unit) && u->type != buildtypes::Protoss_Interceptor) {
          float distance =
              std::max(utils::distanceBB(unit, u), DFOASG(0.125f, 1.0f));
          float maxDistance = DFOASG(4 * 10, 4 * 3);
          if (distance <= maxDistance) {
            adjustment += (myPos - u->posf()) * (maxDistance / distance);
          }
        }
      }
      Vec2 moveDir = (myPos - targetPos).normalize();
      return unit->posf() + (moveDir + adjustment.normalize()).normalize() * moveAhead(unit);
    };

    for (Unit* e : agent.legalTargets) {
      if (unit->inRangeOfPlus(e, e->cd() <= latency + 4 ? 1.0f : 1.0f)) {
        incomingDpf += e->computeDamagePerFrame(unit);
      }
    }

    if ((unit->unit.shield + unit->unit.health) / incomingDpf < 24 * 2) {
      if (agent.wantsToFight) {
        unload();
      }
      return doAction(agent.moveTo(kiteVector()));
    }
  }

  if (noop) return doNothing;

  Position targetPos = kInvalidPosition;
  float targetDistance = kfInfty;

  bool wantsToFight = false;
  int nUnits = loadedUnits.size();
  for (auto x : bwu->getLoadedUnits()) {
    Unit* u = state->unitsInfo().getUnit(x->getID());
    if (u) {
      //VLOG(0) << " try loaded unit";
      auto& ua = task->agents_->at(u);
      //VLOG(0) << " loaded unit ok";
      //if (ua.wantsToFight && ua.target && ua.target->inRangeOfPlus(u, 4.0f + 4.0f * 2 * nUnits)) {
      if (ua.wantsToFight && ua.target && ua.target->inRangeOfPlus(u, 4.0f)) {
        if (unit->limitSpecial(state)) bwu->unload(x);
        break;
      }
      if (ua.target) {
        Position tpos = ua.target->pos();
        float d = utils::distance(unit->pos(), tpos);
        if (!ua.wantsToFight) d *= 4;
        if (d < targetDistance) {
          wantsToFight = ua.wantsToFight;
          targetDistance = d;
          targetPos = tpos;
        }
      }
    }
  }

  if (pickup && incomingBullets[pickup->id]) {
    float d = utils::distanceBB(unit, pickup);
    if (d <= 4.0f) {
      auto& ua = task->agents_->at(pickup);
      ua.pickupByUntil = state->currentFrame() + 15;
      ua.pickupBy = unit;
      auto* bwup = BWAPI::Broodwar->getUnit(pickup->id);
      if (bwup) {
        if (unit->limitSpecial(state)) bwu->load(bwup);
        return doNothing;
      }
    } else if (d <= 4.0 * 3 || incomingDpf == 0) {
      return doAction(agent.moveTo(Position(unit->posf() + (pickup->posf() - unit->posf()).normalize() * 4.0f * 12)));
    }
  }

  if (pickup && incomingDpf == 0) {
    //VLOG(0) << " try pickup";
    auto& pua = task->agents_->at(pickup);
    //VLOG(0) << " pickup ok";
    if (!pua.wantsToFight || !pua.target || !pua.target->inRangeOfPlus(pickup, 4.0f * 4)) {
      float d = utils::distanceBB(unit, pickup);
      if (d <= 4.0f) {
        pua.pickupByUntil = state->currentFrame() + 15;
        pua.pickupBy = unit;
        auto* bwup = BWAPI::Broodwar->getUnit(pickup->id);
        if (bwup) {
          if (unit->limitSpecial(state)) bwu->load(bwup);
          return doNothing;
        }
      } else if (targetPos == kInvalidPosition || utils::distance(unit->pos(), pickup->pos()) < targetDistance) {
        auto* bwup = BWAPI::Broodwar->getUnit(pickup->id);
        if (bwup) {
          if (unit->limitSpecial(state)) bwu->load(bwup);
          return doNothing;
        }
      }
    }
    return doAction(agent.moveTo(Position(unit->posf() + (pickup->posf() - unit->posf()).normalize() * 4.0f * 12)));
  }

  if (!wantsToFight) {
    auto r = BehaviorCircleAround().onPerform(agent);
    if (r.isFinal) {
      return r;
    }
    return pass;
  }

  if (!wantsToFight && nearestThreat && unit->inRangeOfPlus(nearestThreat, 4.0f * 5)) {
    auto r = BehaviorCircleAround().onPerform(agent);
    if (r.isFinal) {
      return r;
    }
    return pass;
  }

  if (targetPos != kInvalidPosition) {
    return doAction(agent.moveTo(unit->posf() + (Vec2(targetPos) - unit->posf()).normalize() * 24));
  }

  return pass;
}


MicroAction BehaviorAsDropship::onPerform(Agent& agent) {
  State* state = agent.state;
  Unit* unit = agent.unit;
  Unit* target = agent.target;
  auto* task = agent.task;

  if (unit->type != buildtypes::Terran_Dropship) {
    return pass;
  }

  Unit* nearestThreat = utils::getBestScoreCopy(agent.legalTargets, [&](Unit* u) {
    if (!u->canAttack(unit) || u->type == buildtypes::Protoss_Interceptor) return kfInfty;
    return utils::distance(u, unit) - (float)u->rangeAgainst(unit);
  }, kfInfty);

  auto bwl = bwapiLock();
  auto bwu = BWAPI::Broodwar->getUnit(unit->id);
  if (!bwu) return pass;

  auto loadedUnits = bwu->getLoadedUnits();

  auto unload = [&]() {
    if (!unit->limitSpecial(state)) return;
    for (auto x : loadedUnits) {
      bwu->unload(x);
      break;
    }
  };

  if (nearestThreat && unit->unit.health < unit->unit.max_health / 2 && unit->inRangeOfPlus(nearestThreat, 12.0f)) {
    //VLOG(0) << " unload because of threat!";
    unload();
  }

  if (true) {
    auto r = BehaviorGoForRepair().onPerform(agent);
    if (r.isFinal) {
      //VLOG(0) << " unload because of repair";
      unload();
      return r;
    }
  }

  bool noop = false;

  int medics = 0;
  int space = BWAPI::UnitType(unit->type->unit).spaceProvided();
  for (auto x : loadedUnits) {
    space -= x->getType().spaceRequired();
//    if (!state->unitsInfo().getUnit(x->getID())) {
//      LOG(FATAL) << "loaded unit does not exist";
//    }
//    if (state->unitsInfo().getUnit(x->getID())->dead) {
//      LOG(FATAL) << "loaded unit does is dead";
//    }
    auto* u = state->unitsInfo().getUnit(x->getID());
//    if (!u->isMine) {
//      LOG(FATAL) << "loaded unit does is not mine!";
//    }
//    if (u->gone) {
//      LOG(FATAL) << "loaded unit does is gone!";
//    }
//    bool found = false;
//    for (Unit* z : state->unitsInfo().myUnits()) {
//      if (z == u) found = true;
//    }
//    if (!found) {
//      LOG(FATAL) << "loaded unit does is in myUnits!";
//    }
//    VLOG(0) << "loaded unit exists yey";

    //if (std::find(task->squadUnits().begin(), task->squadUnits().end(), u) == task->squadUnits().end()) {
    if (task->agents_->find(u) == task->agents_->end()) {
      //VLOG(0) << " unload because agent not found!";
      if (unit->limitSpecial(state)) bwu->unload(x);
      noop = true;
    }
    if (u->type == buildtypes::Terran_Medic) {
      ++medics;
    }
  }

  std::unordered_map<int, int> incomingBullets;

  for (auto& v : BWAPI::Broodwar->getBullets()) {
    if (v->getType() == BWAPI::BulletTypes::Phase_Disruptor) {
      auto target = v->getTarget();
      if (target) {
        ++incomingBullets[target->getID()];
      }
    }
  }

  bool canLoadTanks = false;
  int nLoadedTanks = 0;
  int nTotalTanks = state->unitsInfo().myUnitsOfType(buildtypes::Terran_Siege_Tank_Tank_Mode).size() + state->unitsInfo().myUnitsOfType(buildtypes::Terran_Siege_Tank_Siege_Mode).size();
  for (Unit* u : state->unitsInfo().myUnitsOfType(buildtypes::Terran_Siege_Tank_Tank_Mode)) {
    if (u->loaded()) {
      ++nLoadedTanks;
    }
  }
  if (nLoadedTanks < (nTotalTanks - 2) / 4) {
    canLoadTanks = true;
  }

  Unit* pickup = space ? utils::getBestScoreCopy(state->unitsInfo().myUnits(), [&](Unit* u) {
    if (!u->active()) return kfInfty;
    if (u->flying() || u->loaded()) return kfInfty;
    if (task->agents_->find(u) == task->agents_->end()) return kfInfty;
    if ((u->type != buildtypes::Terran_Siege_Tank_Tank_Mode || !canLoadTanks) && u->type != buildtypes::Terran_Vulture && u->type != buildtypes::Terran_Marine && u->type != buildtypes::Terran_Medic) return kfInfty;
    //if (u->type != buildtypes::Terran_Vulture && u->type != buildtypes::Terran_Marine && u->type != buildtypes::Terran_Medic) return kfInfty;
    if (u->type == buildtypes::Terran_Medic && medics >= 2) return kfInfty;
    //if (u->type != buildtypes::Terran_Vulture) return kfInfty;
    int req = BWAPI::UnitType(u->type->unit).spaceRequired();
    if (space < req) return kfInfty;
    if (state->currentFrame() < u->noPickupUntil) return kfInfty;
    float r = utils::distance(unit->pos(), u->pos());
    if (r <= 4.0f * 7) {
      auto i = incomingBullets.find(u->id);
      if (i != incomingBullets.end()) {
        if (r <= 4.0f) {
          r -= 4.0f * 100;
        }
        r -= 4.0f * 10 * i->second;
      }
    }
    if (u->type == buildtypes::Protoss_Dragoon) r += 4.0f * 8;
    if (u->type == buildtypes::Protoss_Zealot) r += 4.0f * 16;
    return r;
  }, kfInfty) : nullptr;

  float incomingDpf = 0.0f;

  if (nearestThreat) {

    int latency = state->latencyFrames();

    Vec2 myPos = unit->posf() + unit->velocity() * latency;
    Vec2 targetPos = nearestThreat->posf() + nearestThreat->velocity() * latency;

    auto kiteVector = [&]() {
      Vec2 adjustment;
      for (Unit* u : agent.legalTargets) {
        if (u->canAttack(unit) && u->type != buildtypes::Protoss_Interceptor) {
          float distance =
              std::max(utils::distanceBB(unit, u), DFOASG(0.125f, 1.0f));
          float maxDistance = DFOASG(4 * 10, 4 * 3);
          if (distance <= maxDistance) {
            adjustment += (myPos - u->posf()) * (maxDistance / distance);
          }
        }
      }
      Vec2 moveDir = (myPos - targetPos).normalize();
      return unit->posf() + (moveDir + adjustment.normalize()).normalize() * moveAhead(unit);
    };

    for (Unit* e : agent.legalTargets) {
      if (unit->inRangeOfPlus(e, e->cd() <= latency + 4 ? 1.0f : 1.0f)) {
        incomingDpf += e->computeDamagePerFrame(unit);
      }
    }

    if ((unit->unit.shield + unit->unit.health) / incomingDpf < 24 * 2) {
      if (agent.wantsToFight) {
        //VLOG(0) << "unload because wants to fight and incomingDpf of " << incomingDpf;
        unload();
      }
      return doAction(agent.moveTo(kiteVector()));
    }
  }

  if (noop) return doNothing;

  Position targetPos = kInvalidPosition;
  float targetDistance = kfInfty;

  if (incomingDpf == 0) {
    targetPos = state->areaInfo().candidateEnemyStartLocations().front();
    auto dropScore = [&](Position at) {
      return -utils::distance(state->areaInfo().enemyStartLocation(), at) + utils::distance(unit->pos(), at) / 10.0f;
    };
    float s = dropScore(targetPos);
    for (Unit* u : state->unitsInfo().enemyUnits()) {
      if (!u->gone && u->type->isResourceDepot) {
        float d = dropScore(u->pos());
        if (d < s) {
          s = d;
          targetPos = u->pos();
        }
      }
    }
    for (auto& area : state->areaInfo().areas()) {
      for (auto& centerPos : area.baseLocations) {
        // Base locations are center-of-building -- move to top left instead
        Position pos = centerPos - Position(8, 6);

        auto* tile = state->tilesInfo().tryGetTile(pos.x, pos.y);
        if (tile && state->currentFrame() - tile->lastSeen >= 24 * 60 * 2) {
          //float d = utils::distance(unit->pos(), pos);
          float d = dropScore(pos);
//          for (Unit* e : state->unitsInfo().enemyUnits()) {
//            if (e->type->hasAirWeapon) {
//              d -= utils::distance(e, pos);
//            }
//          }
          if (d < s) {
            s = d;
            targetPos = pos;
          }
        }
      }
    }
  }

  bool wantsToFight = false;
  int nUnits = loadedUnits.size();
  for (auto x : bwu->getLoadedUnits()) {
    Unit* u = state->unitsInfo().getUnit(x->getID());
    if (u) {
      u->forceAggressive = true;
      u->autoResetForceAggressive = true;
      //VLOG(0) << " try loaded unit";
      auto& ua = task->agents_->at(u);
      //VLOG(0) << " loaded unit ok";
      //if (ua.wantsToFight && ua.target && ua.target->inRangeOfPlus(u, 4.0f + 4.0f * 2 * nUnits)) {
      if (ua.wantsToFight && ua.target && ua.target->inRangeOfPlus(u, 4.0f)) {
        //VLOG(0) << " unload because " << utils::unitString(u) << " wants to attack " << utils::unitString(ua.target);
        if (unit->limitSpecial(state)) bwu->unload(x);
        break;
      }
      if (ua.wantsToFight && u->type == buildtypes::Terran_Medic) {
        if (pickup && pickup->type == buildtypes::Terran_Marine && utils::distanceBB(unit, pickup) <= 4.0f * 2 && pickup->unit.health < pickup->unit.max_health) {
          //VLOG(0) << " unload because " << utils::unitString(u) << " wants to heal " << utils::unitString(pickup);
          u->noPickupUntil = state->currentFrame() + 24;
          if (unit->limitSpecial(state)) bwu->unload(x);
          break;
        }
      }
      if (ua.target) {
        Position tpos = ua.target->pos();
        float d = utils::distance(unit->pos(), tpos);
        if (!ua.wantsToFight) d *= 4;
        if (d < targetDistance) {
          wantsToFight = ua.wantsToFight;
          targetDistance = d;
          //targetPos = tpos;
        }
      }
    }
  }

  if (pickup && incomingBullets[pickup->id]) {
    float d = utils::distanceBB(unit, pickup);
    if (d <= 4.0f) {
      auto& ua = task->agents_->at(pickup);
      ua.pickupByUntil = state->currentFrame() + 15;
      ua.pickupBy = unit;
      auto* bwup = BWAPI::Broodwar->getUnit(pickup->id);
      if (bwup) {
        if (unit->limitSpecial(state)) bwu->load(bwup);
        return doNothing;
      }
    } else if (d <= 4.0 * 3 || incomingDpf == 0) {
      return doAction(agent.moveTo(Position(unit->posf() + (pickup->posf() - unit->posf()).normalize() * 4.0f * 12)));
    }
  }

  if (pickup && incomingDpf == 0) {
    //VLOG(0) << " try pickup";
    auto& pua = task->agents_->at(pickup);
    //VLOG(0) << " pickup ok";
    if ((!pua.wantsToFight || !pua.target || !pua.target->inRangeOfPlus(pickup, 4.0f * 4)) && (pickup->type != buildtypes::Terran_Medic || state->currentFrame() - pua.lastSpentEnergy >= 10)) {
      float d = utils::distanceBB(unit, pickup);
      if (d <= 4.0f) {
        pua.pickupByUntil = state->currentFrame() + 15;
        pua.pickupBy = unit;
        auto* bwup = BWAPI::Broodwar->getUnit(pickup->id);
        if (bwup) {
          if (unit->limitSpecial(state)) bwu->load(bwup);
          return doNothing;
        }
      } else if (targetPos == kInvalidPosition || utils::distance(unit->pos(), pickup->pos()) < targetDistance) {
        auto* bwup = BWAPI::Broodwar->getUnit(pickup->id);
        if (bwup) {
          if (unit->limitSpecial(state)) bwu->load(bwup);
          return doNothing;
        }
      }
    }
    return doAction(agent.moveTo(Position(unit->posf() + (pickup->posf() - unit->posf()).normalize() * 4.0f * 12)));
  }

  if (!wantsToFight && nearestThreat && unit->inRangeOfPlus(nearestThreat, 4.0f * 5)) {
    auto r = BehaviorCircleAround().onPerform(agent);
    if (r.isFinal) {
      return r;
    }
    return pass;
  }

  if (!wantsToFight) {
    auto r = BehaviorCircleAround().onPerform(agent);
    if (r.isFinal) {
      return r;
    }
    if (targetPos == kInvalidPosition) {
      return pass;
    }
  }

  if (targetPos != kInvalidPosition) {
    if (utils::distance(unit->pos(), targetPos) > 4.0f * 12) {
      float r = utils::distance(unit->pos(), targetPos) / utils::distance(state->areaInfo().myStartLocation(), targetPos);
      r *= r;
      Vec2 add = (Vec2(targetPos) - unit->posf()).normalize() * 24;
      Vec2 a = unit->posf() + add.rotateDegrees(std::min(90.0 * r, 75.0));
      Vec2 b = unit->posf() + add.rotateDegrees(std::max(-90.0 * r, -75.0));
      Position mapcenter(state->mapWidth() / 2, state->mapHeight() / 2);

      //bool goa = (unit->dropState == 0 && utils::distance(Position(a), mapcenter) > utils::distance(Position(b), mapcenter)) || unit->dropState == 1;
      //if (state->currentFrame() - unit->lastDecideDropState >= 24 * 60) {
      //  unit->lastDecideDropState = state->currentFrame();
      //  unit->dropState = common::Rand::sample(std::uniform_int_distribution<int>(0, 2));
      //}
      bool goa = utils::distance(Position(a), mapcenter) > utils::distance(Position(b), mapcenter);

      if (goa) {
        return doAction(agent.moveTo(a));
      } else {
        return doAction(agent.moveTo(b));
      }
    }
    return doAction(agent.moveTo(unit->posf() + (Vec2(targetPos) - unit->posf()).normalize() * 24));
  }

  return pass;
}

MicroAction BehaviorAsCarrier::onPerform(Agent& agent) {
  State* state = agent.state;
  Unit* unit = agent.unit;
  Unit* target = agent.target;
  auto* task = agent.task;

  if (unit->type != buildtypes::Protoss_Carrier) {
    return pass;
  }

  if (!agent.wantsToFight && unit->lastCombatScore < 0) {
    return pass;
  }

  if (unit->remainingBuildTrainTime <= state->latencyFrames() && unit->unit.associatedCount < 8) {
    if (unit->limitSpecial(state)) {
      state->board()->postCommand(
          tc::Client::Command(tc::BW::Command::CommandUnit, unit->id, tc::BW::UnitCommandType::Train, -1, 0, 0, buildtypes::Protoss_Interceptor->unit),
          task->upcId());
    }
  }

  if (!target) {
    return pass;
  }

  Unit* nearestThreat = utils::getBestScoreCopy(agent.legalTargets, [&](Unit* u) {
    if (!u->canAttack(unit) || u->type == buildtypes::Protoss_Interceptor) return kfInfty;
    return utils::distance(u, unit) - (float)u->rangeAgainst(unit);
  }, kfInfty);

  bool attacking = true;

  BWAPI::Unit bwu = BWAPI::BroodwarPtr->getUnit(unit->id);
  if (bwu) {
    for (auto u : bwu->getInterceptors()) {
      if (u->isCompleted() && u->getHitPoints() >= u->getType().maxHitPoints() && u->getOrder() != BWAPI::Orders::InterceptorAttack) {
        attacking = false;
        break;
      }
    }
  }

  if (true) {
    if (nearestThreat && utils::distanceBB(unit, nearestThreat) < utils::distanceBB(unit, target)) {
      target = nearestThreat;
    }

    float maxDistance = 4.0f * 8;

    int latency = state->latencyFrames();
    Vec2 myPos = unit->posf() + unit->velocity() * latency;
    Vec2 targetPos = target->posf() + target->velocity() * latency;

    auto kiteVector = [&]() {
      Vec2 adjustment;
      for (Unit* u : agent.legalTargets) {
        if (u->canAttack(unit) && u->type != buildtypes::Protoss_Interceptor) {
          float distance =
              std::max(utils::distanceBB(unit, u), DFOASG(0.125f, 1.0f));
          float maxDistance = DFOASG(4 * 10, 4 * 3);
          if (distance <= maxDistance) {
            adjustment += (myPos - u->posf()) * (maxDistance / distance);
          }
        }
      }
      Vec2 moveDir = (myPos - targetPos).normalize();
      return unit->posf() + (moveDir + adjustment.normalize()).normalize() * moveAhead(unit);
    };

    if (attacking) {
      if (utils::distancev(unit->posf(), targetPos) <= utils::distancev(unit->posf(), target->posf())) {
        maxDistance = 4.0f * 9;
      }
    }

    if (unit->associatedCount == 0) {
      maxDistance = 4.0f * 10;
    }

    if ((!nearestThreat || utils::distanceBB(unit, nearestThreat) >= 4.0f * 9) && utils::distancev(myPos, targetPos) > utils::distancev(myPos, target->posf())) {
      if (utils::distanceBB(unit, myPos, target, targetPos) > 4.0f * 6) {
        return doAction(agent.attack(target->pos()));
      }
    }

    if ((attacking || unit->associatedCount == 0) && utils::distanceBB(unit, target) <= maxDistance) {
      return doAction(agent.attack(Position(kiteVector())));
    }
    if (!agent.wantsToFight) {
      return pass;
    }
    if (utils::distanceBB(unit, myPos, target, targetPos) > 4.0f * 8.5f) {
      return doAction(agent.attack(target->pos()));
    } else {
      return doAction(agent.attack(target));
    }
  }

  if (agent.wantsToFight && !attacking) {
    if (nearestThreat && unit->inRangeOfPlus(nearestThreat, 6.0f) && !agent.targetInRange) {
      return doAction(agent.attack(nearestThreat));
    } else {
      return doAction(agent.attack(target));
    }
  }

  if (nearestThreat && (utils::distanceBB(nearestThreat, unit) <= std::min(4.0f * (attacking ? 9 : 7.75f), (float)nearestThreat->rangeAgainst(unit) + 4.0f) || !agent.wantsToFight)) {
    int latency = state->latencyFrames();
    Vec2 myPos = unit->posf() + unit->velocity() * latency;
    Vec2 targetPos = target->posf() + target->velocity() * latency;

    auto kiteVector = [&]() {
      Vec2 adjustment;
      for (Unit* u : agent.legalTargets) {
        if (u->canAttack(unit) && u->type != buildtypes::Protoss_Interceptor) {
          float distance =
              std::max(utils::distanceBB(unit, u), DFOASG(0.125f, 1.0f));
          float maxDistance = DFOASG(4 * 10, 4 * 3);
          if (distance <= maxDistance) {
            adjustment += (myPos - u->posf()) * (maxDistance / distance);
          }
        }
      }
      Vec2 moveDir = (myPos - targetPos).normalize();
      return unit->posf() + (moveDir + adjustment.normalize()).normalize() * moveAhead(unit);
    };
    return doAction(agent.attack(Position(kiteVector())));
  }

  if (utils::distanceBB(target, unit) <= 4.0f * 9) {
    return doAction(agent.attack(target));
  }

  return pass;
}

MicroAction BehaviorAsValkyrie::onPerform(Agent& agent) {
  State* state = agent.state;
  Unit* unit = agent.unit;
  Unit* target = agent.target;
  auto* task = agent.task;

  if (unit->type != buildtypes::Terran_Valkyrie) {
    return pass;
  }

  if (true) {
    auto r = BehaviorGoForRepair().onPerform(agent);
    if (r.isFinal) {
      return r;
    }
  }

  if (true) {
    auto r = BehaviorSupportTanks().onPerform(agent);
    if (r.isFinal) {
      return r;
    }
  }

  if (!target || !agent.wantsToFight) {
    return pass;
  }

  if (target->type == buildtypes::Protoss_Interceptor) {
    return pass;
  }

  Vec2 upos = unit->posf() + unit->velocity() * (state->latencyFrames() - 1);
  Vec2 tpos = target->posf() + target->velocity() * (state->latencyFrames() - 1);
  if (utils::distanceBB(unit, upos, target, tpos) <= unit->rangeAgainst(target)) {
    if (state->currentFrame() - agent.lastHealMove > 6 + state->latencyFrames() && unit->velocity().length() >= unit->topSpeed * 0.875f) {
      Position pos = Position(unit->posf() + (unit->posf() - target->posf()).rotateDegrees(30).normalize() * 4.0f * 12);
      agent.lastHealMove = state->currentFrame();
      auto bwl = bwapiLock();
      BWAPI::Unit bwu = BWAPI::Broodwar->getUnit(unit->id);
      if (bwu) bwu->patrol(BWAPI::Position(pos.x * 8, pos.y * 8));
//        state->board()->postCommand(
//            tc::Client::Command(tc::BW::Command::CommandUnit, unit->id, tc::BW::UnitCommandType::Patrol, -1, pos.x, pos.y, -1),
//            task->upcId());
      return doNothing;
    }
  } else {
    Position pos = Position(unit->posf() - (unit->posf() - target->posf()).normalize() * 4.0f * 12);
    return doAction(agent.moveTo(pos));
  }

  return pass;
}

MicroAction BehaviorUseShieldBattery::onPerform(Agent& agent) {
  State* state = agent.state;
  Unit* unit = agent.unit;
  Unit* target = agent.target;
  auto* task = agent.task;

  if (unit->unit.max_shield == 0) {
    return pass;
  }

  if (agent.usingShieldBattery && unit->unit.shield >= unit->unit.max_shield) {
    agent.usingShieldBattery = false;
  }

  if (unit->unit.shield <= unit->unit.max_shield / 3 || agent.usingShieldBattery || !agent.wantsToFight) {
    agent.usingShieldBattery = false;
    Unit* bat = utils::getBestScoreCopy(state->unitsInfo().myCompletedUnitsOfType(buildtypes::Protoss_Shield_Battery), [&](Unit* u) {
      if (u->unit.energy < 5 || !u->powered()) return kfInfty;
      return utils::distance(unit->pos(), u->pos());
    }, kfInfty);
    if (bat) {
      float d = utils::distanceBB(unit, bat);
      if (!agent.wantsToFight && d <= 4.0f * 2) agent.wantsToFight = true;
      if (d <= 4.0f * 4) {
        agent.usingShieldBattery = true;
        if (unit->unit.orders.empty() || unit->unit.orders.front().type != +tc::BW::Order::RechargeShieldsUnit) {
          if (unit->limitSpecial(state)) {
            state->board()->postCommand(
                tc::Client::Command(tc::BW::Command::CommandUnit, unit->id, tc::BW::UnitCommandType::Follow, bat->id, -1, -1),
                task->upcId());
          }
        }
        return doNothing;
      //} else if (d <= 4.0f * 9 && (!agent.targetInRange || unit->cd()) && agent.wantsToFight) {
      } else if (d <= 4.0f * 9 && (!agent.targetInRange || unit->cd())) {
        return doAction(agent.moveTo(bat->pos()));
      }
    }
  }

  return pass;
}

MicroAction BehaviorAsObserver::onPerform(Agent& agent) {
  State* state = agent.state;
  Unit* unit = agent.unit;
  Unit* target = agent.target;
  auto* task = agent.task;

  if (agent.pickupByUntil >= state->currentFrame() && agent.pickupBy) {
    return doAction(agent.moveTo(agent.pickupBy->pos()));
  }

  if (unit->type != buildtypes::Protoss_Observer && unit->type != buildtypes::Zerg_Overlord) {
    return pass;
  }

  Unit* nearestTarget = utils::getBestScoreCopy(agent.legalTargets, [&](Unit* u) {
    float d = utils::distance(u, unit);
    if (!u->cloaked() && !u->burrowed()) d += 4.0f * 64;
    return d;
  }, kfInfty);
  agent.target = target = nearestTarget;

  if (!target) return pass;

  Unit* nearestDetector = utils::getBestScoreCopy(agent.legalTargets, [&](Unit* u) {
    float d = utils::distance(u, unit);
    if (!u->type->isDetector) return kfInfty;
    return d;
  }, kfInfty);

  agent.target = target = nearestTarget;

  int latency = state->latencyFrames();
  Vec2 myPos = unit->posf() + unit->velocity() * latency;
  Vec2 targetPos = target ? target->posf() + target->velocity() * latency : myPos;

//  if (unit->targetGroupPos != kInvalidPosition) {
//    targetPos = Vec2(unit->targetGroupPos);
//  }

//  if (task->targetingLocation) {
//    targetPos.x = task->targetX;
//    targetPos.y = task->targetY;
//  }

  auto kiteVector = [&]() {
    if (unit->cloaked() && (!nearestDetector || utils::distanceBB(unit, nearestDetector) > nearestDetector->sightRange + 8.0f)) return targetPos;
    Vec2 adjustment;
    for (Unit* u : agent.legalTargets) {
      if (u->type->isDetector) {
        float distance =
            std::max(utils::distanceBB(unit, u), DFOASG(0.125f, 1.0f));
        float maxDistance = DFOASG(4 * 12, 4 * 3);
        if (distance <= maxDistance) {
          adjustment += (myPos - u->posf()) * (maxDistance / distance);
        }
      }
    }
    Vec2 moveDir = (myPos - targetPos).normalize();
    return unit->posf() + (moveDir + adjustment.normalize()).normalize() * moveAhead(unit);
  };

  Unit* nearestThreat = utils::getBestScoreCopy(agent.legalTargets, [&](Unit* u) {
    if (!u->canAttack(unit) || u->type == buildtypes::Protoss_Interceptor) return kfInfty;
    return utils::distance(u, unit) - (float)u->rangeAgainst(unit);
  }, kfInfty);

  if (nearestThreat) {
    if (!target || (!target->cloaked() && !target->burrowed())) {
      Unit* ally = utils::getBestScoreCopy(task->squadUnits(), [&](Unit* u) {
        if (!u->canAttack(nearestThreat)) return kfInfty;
        float d = utils::distance(u, unit);
        if (d > 4.0f * 12) return kfInfty;
        return d;
      }, kfInfty);
      if (ally && utils::distance(unit->pos(), ally->pos()) > 4.0f * 3) {
        return doAction(agent.moveTo(Position(unit->posf() + (ally->posf() - unit->posf()).normalize() * 24.0f)));
      }
    }
    float desiredRange = nearestThreat->rangeAgainst(unit) + 8.0f;
    if (unit->unit.health < unit->unit.health * 0.75f || !agent.wantsToFight) desiredRange += 8.0f;
    if (utils::distanceBB(unit, nearestThreat) < desiredRange) {
      return doAction(agent.moveTo(kiteVector()));
    }
  }
  if (utils::distance(unit->pos(), target->pos()) <= unit->sightRange - 16.0f) {
    return doAction(agent.moveTo(kiteVector()));
  }

  return doAction(agent.moveTo(targetPos));
}

MicroAction BehaviorAsGhost::onPerform(Agent& agent) {
  State* state = agent.state;
  Unit* unit = agent.unit;
  Unit* target = agent.target;
  auto* task = agent.task;

  if (unit->type != buildtypes::Terran_Ghost) {
    return pass;
  }

  if (unit->type == buildtypes::Terran_Ghost) {
    if (state->hasResearched(buildtypes::Lockdown) && unit->unit.energy >= buildtypes::Lockdown->energyCost) {
      float maxd = agent.targetInRange && unit->cd() <= state->latencyFrames() ? 4.0f * 9 : 4.0f * 12;
      Unit* target = utils::getBestScoreCopy(state->unitsInfo().visibleEnemyUnits(), [&](Unit* e) {
        if (!e->type->isMechanical) return kfInfty;
        if (!e->active() || e->type->isBuilding) return kfInfty;
        if (e->type == buildtypes::Protoss_Interceptor) return kfInfty;
        if (e->type->gasCost < 25 && unit->unit.energy < 200) return kfInfty;
        if (state->currentFrame() - e->lastDisable <= 15 && e->lastDisableSource != unit) {
          return kfInfty;
        }
        float incomingDamage = 0.0f;
        for (Unit* n : task->squadUnits()) {
          if (e->inRangeOf(n)) {
            int hp = 0;
            int shield = 0;
            n->computeDamageTo(e, &hp, &shield);
            incomingDamage += 10 + hp + shield;
          }
        }
//        bool bad = unit->lastCombatScore > -0.4f && incomingDamage > 0.0f;
//        if (unit->unit.health < 30) bad = true;
//        else if (unit->unit.health < 100) bad |= unit->lastCombatScore >= -0.4f;
//        else bad |= unit->lastCombatScore >= 0.4f;
//        if (e->type == buildtypes::Zerg_Defiler) bad = e->unit.health - incomingDamage < 0;
//        if (bad) {
//          return kfInfty;
//        }
        float d = utils::distanceBB(unit, e);
        if (d > maxd) {
          return kfInfty;
        }
        return std::max(d - unit->sightRange, 0.0f) - (e->unit.health - incomingDamage) / 0.42f;
      }, kfInfty);
      if (target && utils::distanceBB(unit, target) > 4.0f * 9) {
        return doAction(agent.moveTo(target->pos()));
      } else if (target) {
        target->lastDisable = state->currentFrame();
        target->lastDisableSource = unit;
        if (state->currentFrame() - agent.lastHealMove > 15 + state->latencyFrames()) {
          agent.lastHealMove = state->currentFrame();
          BWAPI::Unit bwu = BWAPI::BroodwarPtr->getUnit(unit->id);
          BWAPI::Unit bwtarget = BWAPI::BroodwarPtr->getUnit(target->id);
          if (bwu && target) {
            bwu->useTech(BWAPI::TechTypes::Lockdown, bwtarget);
          }
//          state->board()->postCommand(
//              tc::Client::Command(tc::BW::Command::CommandUnit, unit->id, tc::BW::UnitCommandType::Use_Tech_Unit, target->id, target->x, target->y, buildtypes::Spawn_Broodlings->tech),
//              task->upcId());
        }
        return doNothing;
      }
    }
  }

  return pass;
}

MicroAction BehaviorAsHighTemplar::onPerform(Agent& agent) {
  State* state = agent.state;
  Unit* unit = agent.unit;
  Unit* target = agent.target;
  auto* task = agent.task;

  if (unit->type != buildtypes::Protoss_High_Templar) {
    return pass;
  }

  Unit* nearestTarget = utils::getBestScoreCopy(agent.legalTargets, [&](Unit* u) {
    if (u->type->isBuilding || !u->active()) return kfInfty;
    return utils::distance(u, unit);
  }, kfInfty);
  agent.target = target = nearestTarget;

  if (!target) return pass;

  int latency = state->latencyFrames();
  Vec2 myPos = unit->posf() + unit->velocity() * latency;
  Vec2 targetPos = target ? target->posf() + target->velocity() * latency : myPos;

  if (task->targetingLocation) {
    targetPos.x = task->targetX;
    targetPos.y = task->targetY;
  }

  auto kiteVector = [&]() {
    Vec2 adjustment;
    for (Unit* u : agent.legalTargets) {
      if (u->canAttack(unit)) {
        float distance =
            std::max(utils::distanceBB(unit, u), DFOASG(0.125f, 1.0f));
        float maxDistance = DFOASG(4 * 10, 4 * 3);
        if (distance <= maxDistance) {
          adjustment += (myPos - u->posf()) * (maxDistance / distance);
        }
      }
    }
    for (Unit* u : task->squadUnits()) {
      if (!u->flying()) {
        float distance =
            std::max(utils::distanceBB(unit, u), DFOASG(0.125f, 1.0f));
        float maxDistance = DFOASG(4 * 10, 4 * 3);
        if (distance <= maxDistance) {
          adjustment += (myPos - u->posf()) * (maxDistance / distance);
        }
      }
    }
    Vec2 moveDir = (myPos - targetPos).normalize();
    return unit->posf() + (moveDir + adjustment.normalize()).normalize() * 6.0;
  };

  if (state->hasResearched(buildtypes::Psionic_Storm) && unit->unit.energy >= buildtypes::Psionic_Storm->energyCost) {
    float bestDamage = 0.0f;
    Vec2 bestPos;
    auto add = [&](Vec2 pos) {
      if (!agent.wantsToFight && utils::distancev(unit->posf(), pos) > 4.0f * 9) return;
      for (auto stormLoc : task->storms_) {
        if (utils::distancev(pos, Vec2(stormLoc)) < 4.0f * 2) {
          return;
        }
      }
      float damage = 0.0f;
      for (Unit* u : agent.legalTargets) {
        if (u->type->isBuilding || !u->active()) continue;
        if (utils::distanceBB(u, pos) <= 4.0f * 1.5f) {
          if (state->currentFrame() - u->lastFiredWeapon <= 8 || u->velocity().length() < 0.4f) {
            damage += std::min(u->unit.shield + u->unit.health, 80);
          }
        }
      }
      if (damage == 0) return;
      if (unit->lastCombatScore > -0.4f && damage < 150) return;
      if (unit->lastCombatScore > 0.25f && damage < 300) return;
      if (damage > bestDamage) {
        bestDamage = damage;
        bestPos = pos;
      }
    };

    for (Unit* e : agent.legalTargets) {
      if (!e->active() || e->type->isBuilding) continue;
      if (state->currentFrame() - e->lastDisable <= 15 && e->lastDisableSource != unit) {
        continue;
      }

      float d = utils::distance(unit, e);
      if (d > 4.0f * 12) {
        continue;
      }
      add(e->posf());
      add(e->posf() + Vec2(4.0f, 0.0f));
      add(e->posf() + Vec2(0.0f, 4.0f));
      add(e->posf() + Vec2(-4.0f, 0.0f));
      add(e->posf() + Vec2(0.0f, -4.0f));
    }
    if (bestPos != Vec2()) {
      target->lastDisable = state->currentFrame();
      target->lastDisableSource = unit;
      if (state->currentFrame() - agent.lastHealMove > 6 + state->latencyFrames()) {
        agent.lastHealMove = state->currentFrame();
        state->board()->postCommand(
            tc::Client::Command(tc::BW::Command::CommandUnit, unit->id, tc::BW::UnitCommandType::Use_Tech_Position, -1, (int)bestPos.x, (int)bestPos.y, buildtypes::Psionic_Storm->tech),
            task->upcId());
      }
      task->storms_.emplace_back(Position(bestPos));
      return doNothing;
    }
  }


  auto canMoveInDirection = [&](Vec2 odir,
                                float distance = DFOASG(4.0f * 2, 4.0f)) {
    Vec2 dir = odir.normalize();
    for (float d = 4.0f; ; d += 4.0f) {
      Position pos = d < distance ? Position(unit->posf() + dir * d) : Position(unit->posf() + odir);
      auto* tile = state->tilesInfo().tryGetTile(pos.x, pos.y);
      if (!tile || !tile->entirelyWalkable || tile->building) {
        return false;
      }
      if (d >= distance) break;
    }
    return true;
  };
  auto adjust = [&](Vec2 targetPos) {
    if (unit->flying()) return targetPos;
    Vec2 dir = targetPos - unit->posf();
    float d = dir.length();
    if (canMoveInDirection(dir, d)) return targetPos;
    for (float dg = 22.5f; dg <= 180.0f; dg += 22.5f) {
      if (dg > 90.0f) dg += 22.5f;
      Vec2 l = dir.rotateDegrees(dg);
      if (canMoveInDirection(l, d)) return unit->posf() + l;
      Vec2 r = dir.rotateDegrees(-dg);
      if (canMoveInDirection(r, d)) return unit->posf() + r;
    }
    return targetPos;
  };

  if (agent.wantsToFight) {
    Unit* nearestThreat = utils::getBestScoreCopy(agent.legalTargets, [&](Unit* u) {
      if (!u->canAttack(unit) || u->type == buildtypes::Protoss_Interceptor) return kfInfty;
      return utils::distance(u, unit) - (float)u->rangeAgainst(unit);
    }, kfInfty);

    if (nearestThreat) {
      float desiredRange = nearestThreat->rangeAgainst(unit) + 8.0f;
      if (unit->unit.health < unit->unit.health * 0.75f) desiredRange += 8.0f;
      if (utils::distanceBB(unit, nearestThreat) < desiredRange) {
        return doAction(agent.moveTo(adjust(kiteVector())));
      }
    }
  }

  return doAction(agent.moveTo(targetPos));
}

MicroAction BehaviorAsMutaliskMicro::onPerform(Agent& agent) {
  State* state = agent.state;
  Unit* unit = agent.unit;
  Unit* target = agent.target;
  auto* task = agent.task;

  if (unit->type != buildtypes::Zerg_Mutalisk && unit->type != buildtypes::Terran_Wraith && unit->type != buildtypes::Terran_Battlecruiser && unit->type != buildtypes::Terran_Science_Vessel && unit->type != buildtypes::Terran_Vulture && unit->type != buildtypes::Protoss_Scout && unit->type != buildtypes::Protoss_Corsair && unit->type != buildtypes::Zerg_Queen && unit->type != buildtypes::Zerg_Guardian) {
    return pass;
  }

  if (unit->type == buildtypes::Terran_Wraith && target) {
    if (unit->cd() > state->latencyFrames() &&
        (!target->flying() || !target->detected()) && agent.wantsToFight) {
      Unit* flyingTarget = utils::getBestScoreCopy(
          agent.legalTargets,
          [&](Unit* u) {
            if (!u->flying() || !u->detected()) {
              return kfInfty;
            }
            return utils::distance(u, unit);
          },
          kfInfty);
      if (flyingTarget) {
        return doAction(agent.attack(flyingTarget));
      }
    }

    Unit* nearestDetector = utils::getBestScoreCopy(agent.legalTargets, [&](Unit* u) {
      float d = utils::distance(u, unit);
      if (!u->type->isDetector) return kfInfty;
      return d;
    }, kfInfty);

    if (nearestDetector && nearestDetector->flying() && nearestDetector->detected() && !agent.targetInRange && utils::distanceBB(unit, nearestDetector) <= 4.0f * 12) {
      agent.target = target = nearestDetector;
    }

    bool hasCloaking = state->hasResearched(buildtypes::Cloaking_Field);

    bool shouldCloak = false;
    
    if (!unit->cloaked() && unit->unit.energy >= 40 && hasCloaking) {
      if (unit->unit.energy < 75 && unit->unit.health > unit->unit.max_health / 2) {
        for (Unit* u : task->squadUnits()) {
          if (u->cloaked() && utils::distance(u, unit) <= 4.0f * 16) {
            shouldCloak = true;
            break;
          }
        }
      } else {
        shouldCloak = true;
      }
    }

    bool getClose = false;
    if (((unit->cloaked() && unit->unit.energy > 2) || shouldCloak) &&
        (!nearestDetector ||
         utils::distanceBB(unit, nearestDetector) >
             nearestDetector->sightRange + 8.0f)) {
      agent.wantsToFight = true;
      getClose = true;
//      if (agent.targetInRange) {
//        return doAction(agent.attack(target));
//      }
    }

    Unit* nearestThreat = utils::getBestScoreCopy(agent.legalTargets, [&](Unit* u) {
      if (!u->canAttack(unit)) return kfInfty;
      return utils::distance(u, unit) - (float)u->rangeAgainst(unit);
    }, kfInfty);

    if (shouldCloak && nearestThreat &&
        unit->inRangeOfPlus(nearestThreat, 8.0f) && state->currentFrame() - unit->lastCloak >= 30 && (!target->type->isDetector || target->blind())) {
      BWAPI::Unit bwu = BWAPI::BroodwarPtr->getUnit(unit->id);
      if (bwu) {
        unit->lastCloak = state->currentFrame();
        bwu->cloak();
      }
    }

//    if (agent.targetInRange && unit->cd() <= state->latencyFrames() && (state->currentFrame() - agent.lastHealMove > 10 || state->currentFrame() - agent.lastHealMove <= 4) && unit->velocity().length() >= unit->topSpeed * 0.875f && unit->velocity().dot(target->posf() - unit->posf()) > 0) {
//      if (state->currentFrame() - agent.lastHealMove > 4) {
//        Position pos = Position(unit->posf() + -unit->velocity().rotateDegrees(45).normalize() * 4.0f * 12);
//        agent.lastHealMove = state->currentFrame();
//        BWAPI::Unit bwu = BWAPI::Broodwar->getUnit(unit->id);
//        if (bwu) bwu->patrol(BWAPI::Position(pos.x * 8, pos.y * 8));
//      }
//      return doNothing;
//    }

    if (agent.wantsToFight && target->type == buildtypes::Protoss_Interceptor && target->attackingTarget != unit) {
      getClose = true;
    }

    if (getClose) {
      if (target->type == buildtypes::Protoss_Interceptor) {
        Unit* carrier = utils::getBestScoreCopy(agent.legalTargets, [&](Unit* u) {
          if (u->type != buildtypes::Protoss_Carrier) return kfInfty;
          return utils::distance(u, unit);
        }, kfInfty);
        if (carrier) {
          agent.target = target = carrier;
        }
      }
      if (unit->cd() > state->latencyFrames() && utils::distance(unit, target) > unit->rangeAgainst(target) - 6) {
        return doAction(agent.moveTo(target->pos()));
      }
      return doAction(agent.attack(target));
      //return doAction(agent.moveTo(target->pos()));
    }

  }

  if (!unit->flying() && !unit->sieged() && !unit->burrowed()) {
    for (Unit* u : unit->unitsInSightRange) {
      if (u->type == buildtypes::Terran_Vulture_Spider_Mine) {
        if (u->attackingTarget && u->velocity() != Vec2()) {
          if (utils::distance(unit, u) > 16) {
            continue;
          }
          auto pos =
              Vec2(unit) + (Vec2(unit) - Vec2(u)).normalize() * moveAhead(unit);
          return doAction(agent.moveTo(pos));
        }
      }
    }
  }

  if (unit->type == buildtypes::Terran_Vulture && target && !agent.targetInRange) {
    if (agent.targetInRange && unit->cd() <= state->latencyFrames()) {
      return doAction(agent.attack(target));
    }
    Unit* nearestThreat = utils::getBestScoreCopy(
        agent.legalTargets,
        [&](Unit* u) {
          if (!u->canAttack(unit)) {
            return kfInfty;
          }
          return utils::distance(u, unit) - (float)u->rangeAgainst(unit);
        },
        kfInfty);
    float safeRange = unit->unit.health >= unit->unit.max_health * 0.75 ? 4.0f : 8.0f;
    if (!nearestThreat || !unit->inRangeOfPlus(nearestThreat, safeRange)) {
      auto canSafelyAttack = [&](Unit* e) {
        Vec2 at = e->posf() + (unit->posf() - e->posf()).normalize() * std::min((float)unit->rangeAgainst(e), utils::distance(unit->pos(), e->pos()));
        Position atp(at);
        Unit* threat = utils::getBestScoreCopy(
            agent.legalTargets,
            [&](Unit* u) {
              if (!u->canAttack(unit)) {
                return kfInfty;
              }
              return utils::distance(u->pos(), atp) - (float)u->rangeAgainst(unit);
            },
            kfInfty);
        if (!threat) {
          return false;
        }
        return utils::distanceBB(unit->type, at, e->type, e->posf()) > e->rangeAgainst(unit) + safeRange;
      };
      Unit* workerTarget = target->type->isWorker ? target : utils::getBestScoreCopy(
          agent.legalTargets,
          [&](Unit* u) {
            if (!u->type->isWorker) {
              return kfInfty;
            }
            return utils::distance(u, unit);
          },
          kfInfty);
      if (workerTarget && canSafelyAttack(workerTarget)) {
        agent.target = target = workerTarget;
      }
//      if (workerTarget && canSafelyAttack(workerTarget)) {
//        return doAction(agent.attack(workerTarget));
//      } else if (canSafelyAttack(target)) {
//        return doAction(agent.attack(target));
//      }
    }
  }

  if (unit->type == buildtypes::Terran_Battlecruiser) {
    if (unit->unit.energy >= buildtypes::Yamato_Gun->energyCost && state->hasResearched(buildtypes::Yamato_Gun)) {
      Unit* target = utils::getBestScoreCopy(agent.legalTargets, [&](Unit* e) {
        if (e->type != buildtypes::Terran_Siege_Tank_Siege_Mode) {
          return kfInfty;
        }
        if (!e->active()) return kfInfty;
        if (state->currentFrame() - e->lastDisable <= 15 && e->lastDisableSource != unit) {
          return kfInfty;
        }
        float d = utils::distance(unit, e);
        if (d > 4.0f * 11) {
          return kfInfty;
        }
        return d;
      }, kfInfty);
      if (target) {
        target->lastDisable = state->currentFrame();
        target->lastDisableSource = unit;
        if (state->currentFrame() - agent.lastHealMove > 6) {
          agent.lastHealMove = state->currentFrame();
          state->board()->postCommand(
              tc::Client::Command(tc::BW::Command::CommandUnit, unit->id, tc::BW::UnitCommandType::Use_Tech_Unit, target->id, target->x, target->y, buildtypes::Yamato_Gun->tech),
              task->upcId());
        }
        return doNothing;
      }
    }
  }

  if (unit->type == buildtypes::Zerg_Queen) {
    if (state->hasResearched(buildtypes::Spawn_Broodlings) && unit->unit.energy >= buildtypes::Spawn_Broodlings->energyCost) {
      Unit* target = utils::getBestScoreCopy(state->unitsInfo().visibleEnemyUnits(), [&](Unit* e) {
        if (e->type->isRobotic) return kfInfty;
        if (!e->active() || e->type->isBuilding) return kfInfty;
        if (e->flying()) return kfInfty;
        if (state->currentFrame() - e->lastDisable <= 15 && e->lastDisableSource != unit) {
          return kfInfty;
        }
        float incomingDamage = 0.0f;
        for (Unit* n : task->squadUnits()) {
          if (e->inRangeOf(n)) {
            int hp = 0;
            int shield = 0;
            n->computeDamageTo(e, &hp, &shield);
            incomingDamage += 10 + hp + shield;
          }
        }
//        bool bad = unit->lastCombatScore > -0.4f && incomingDamage > 0.0f;
//        if (unit->unit.health < 30) bad = true;
//        else if (unit->unit.health < 100) bad |= unit->lastCombatScore >= -0.4f;
//        else bad |= unit->lastCombatScore >= 0.4f;
//        if (e->type == buildtypes::Zerg_Defiler) bad = e->unit.health - incomingDamage < 0;
//        if (bad) {
//          return kfInfty;
//        }
        float d = utils::distance(unit, e);
        if (d > 4.0f * 20) {
          return kfInfty;
        }
        return std::max(d - unit->sightRange, 0.0f) - (e->unit.health - incomingDamage) / 0.42f;
      }, kfInfty);
      if (target) {
        target->lastDisable = state->currentFrame();
        target->lastDisableSource = unit;
        if (state->currentFrame() - agent.lastHealMove > 6 + state->latencyFrames()) {
          agent.lastHealMove = state->currentFrame();
          BWAPI::Unit bwu = BWAPI::BroodwarPtr->getUnit(unit->id);
          BWAPI::Unit bwtarget = BWAPI::BroodwarPtr->getUnit(target->id);
          if (bwu && target) {
            bwu->useTech(BWAPI::TechTypes::Spawn_Broodlings, bwtarget);
          }
//          state->board()->postCommand(
//              tc::Client::Command(tc::BW::Command::CommandUnit, unit->id, tc::BW::UnitCommandType::Use_Tech_Unit, target->id, target->x, target->y, buildtypes::Spawn_Broodlings->tech),
//              task->upcId());
        }
        return doNothing;
      }
    }
  }

  if (unit->type == buildtypes::Terran_Science_Vessel) {
    if (unit->unit.energy >= buildtypes::Defensive_Matrix->energyCost) {
      Unit* target = utils::getBestScoreCopy(task->squadUnits(), [&](Unit* e) {
        if (e->type != buildtypes::Terran_Siege_Tank_Siege_Mode && e->type != buildtypes::Terran_Siege_Tank_Tank_Mode && (e->type != buildtypes::Terran_Science_Vessel || e == unit || e->unit.health >= e->unit.max_health / 2) && e->type != buildtypes::Terran_Battlecruiser) {
          return kfInfty;
        }
        if (!e->active()) return kfInfty;
        if (state->currentFrame() - e->lastDisable <= 15 && e->lastDisableSource != unit) {
          return kfInfty;
        }
        if (e->flag(tc::Unit::Flags::DefenseMatrixed)) {
          return kfInfty;
        }
        float incomingDamage = 0.0f;
        for (Unit* n : agent.legalTargets) {
          if (e->inRangeOf(n)) {
            int hp = 0;
            int shield = 0;
            n->computeDamageTo(e, &hp, &shield);
            incomingDamage += 10 + hp + shield;
          }
        }
        bool bad = incomingDamage == 0.0f;
        if (unit->unit.health < unit->unit.max_health * 0.75f) bad |= incomingDamage < (e->unit.shield + e->unit.health) / 10;
        else bad |= incomingDamage < (e->unit.shield + e->unit.health) / 2;
        if (bad) {
          return kfInfty;
        }
        float d = utils::distance(unit, e);
        if (d > 4.0f * 12) {
          return kfInfty;
        }
        return std::max(d - unit->sightRange, 0.0f) + incomingDamage / 0.42f;
      }, kfInfty);
      if (target) {
        target->lastDisable = state->currentFrame();
        target->lastDisableSource = unit;
        if (state->currentFrame() - agent.lastHealMove > 6 + state->latencyFrames()) {
          agent.lastHealMove = state->currentFrame();
          state->board()->postCommand(
              tc::Client::Command(tc::BW::Command::CommandUnit, unit->id, tc::BW::UnitCommandType::Use_Tech_Unit, target->id, target->x, target->y, buildtypes::Defensive_Matrix->tech),
              task->upcId());
        }
        return doNothing;
      }
    }
    if (state->hasResearched(buildtypes::Irradiate) && unit->unit.energy >= buildtypes::Irradiate->energyCost) {
      Unit* target = utils::getBestScoreCopy(agent.legalTargets, [&](Unit* e) {
        if (!e->type->isBiological) return kfInfty;
        if (!e->active() || e->type->isBuilding) return kfInfty;
        if (e->irradiated()) return kfInfty;
        if (state->currentFrame() - e->lastDisable <= 15 && e->lastDisableSource != unit) {
          return kfInfty;
        }
        float incomingDamage = 0.0f;
        for (Unit* n : task->squadUnits()) {
          if (e->inRangeOf(n)) {
            int hp = 0;
            int shield = 0;
            n->computeDamageTo(e, &hp, &shield);
            incomingDamage += 10 + hp + shield;
          }
        }
        bool bad = unit->lastCombatScore > -0.4f && incomingDamage > 0.0f;
        if (unit->unit.health < 30) bad = true;
        else if (unit->unit.health < 100) bad |= unit->lastCombatScore >= -0.4f;
        else bad |= unit->lastCombatScore >= 0.4f;
        if (e->type == buildtypes::Zerg_Defiler) bad = e->unit.health - incomingDamage < 0;
        if (bad) {
          return kfInfty;
        }
        float d = utils::distance(unit, e);
        if (d > 4.0f * 12) {
          return kfInfty;
        }
        return std::max(d - unit->sightRange, 0.0f) - (e->unit.health - incomingDamage) / 0.42f;
      }, kfInfty);
      if (target) {
        target->lastDisable = state->currentFrame();
        target->lastDisableSource = unit;
        if (state->currentFrame() - agent.lastHealMove > 6 + state->latencyFrames()) {
          agent.lastHealMove = state->currentFrame();
          state->board()->postCommand(
              tc::Client::Command(tc::BW::Command::CommandUnit, unit->id, tc::BW::UnitCommandType::Use_Tech_Unit, target->id, target->x, target->y, buildtypes::Irradiate->tech),
              task->upcId());
        }
        return doNothing;
      }
    }
  }

  if (unit->type == buildtypes::Terran_Wraith || unit->type == buildtypes::Terran_Science_Vessel || unit->type == buildtypes::Protoss_Scout || unit->type == buildtypes::Protoss_Corsair) {
    auto r = BehaviorGoForRepair().onPerform(agent);
    if (r.isFinal) {
      return r;
    }
  }

  if (true) {
    auto r = BehaviorSupportTanks().onPerform(agent);
    if (r.isFinal) {
      return r;
    }
  }


  if (unit->type == buildtypes::Terran_Vulture) {
    if (target && unit->associatedCount > 0 && target->type != buildtypes::Protoss_Zealot && target->type != buildtypes::Zerg_Zergling && !target->type->isWorker && !target->type->isBuilding) {
      if (state->hasResearched(buildtypes::Spider_Mines)) {
        float d = utils::distanceBB(unit, target);
        //if (unit->lastCombatScore < 0.25f && utils::distanceBB(unit, target) < 4.0f * 16.0f) {
        if (d < 4.0f * 16.0f) {
          Unit* targetTarget = utils::getBestScoreCopy(task->squadUnits(), [&](Unit* u) {
            return utils::distanceBB(u, target);
          });
          if (targetTarget && (d <= utils::distanceBB(targetTarget, target) + 8.0f || d <= 4.0f * 6)) {
            //Position targetPos = Position(unit->posf() + unit->velocity() * state->latencyFrames());
            Position targetPos = unit->pos();
            int n = 0;
            for (Unit* u : state->unitsInfo().myUnitsOfType(buildtypes::Terran_Vulture_Spider_Mine)) {
              if (utils::distance(unit->pos(), u->pos()) < 4.0f) {
                ++n;
                //targetPos = u->pos();
              }
            }
            if (n <= 2) {
              n = 0;
              for (Unit* u : state->unitsInfo().myUnits()) {
                if (!u->flying() && u->type != buildtypes::Terran_Vulture && u->type != buildtypes::Terran_Vulture_Spider_Mine && utils::distance(targetPos, u->pos()) < 4.0f * 5) {
                  ++n;
                  if (u->type == buildtypes::Terran_Siege_Tank_Siege_Mode || u->type == buildtypes::Terran_Siege_Tank_Tank_Mode) {
                    n += 5;
                  }
                }
              }
              if (n < 5) {
                if (state->currentFrame() - unit->lastMine > 20 && unit->limitMine(state)) {
//                  state->board()->postCommand(
//                      tc::Client::Command(tc::BW::Command::CommandUnit, unit->id, tc::BW::UnitCommandType::Use_Tech_Position, -1, (int)targetPos.y, (int)targetPos.x, buildtypes::Spider_Mines->tech),
//                      task->upcId());
                  BWAPI::Unit bwu = BWAPI::BroodwarPtr->getUnit(unit->id);
                  if (bwu) {
                    bwu->useTech(BWAPI::TechTypes::Spider_Mines, BWAPI::Position(targetPos.x * 8, targetPos.y * 8));
                  }
                  return doNothing;
                } else if (state->currentFrame() - unit->lastMine < 10) {
                  return doNothing;
                }
              }
            }
          }
        }
      }
    }
  }


  if (unit->type == buildtypes::Protoss_Corsair) {
    //if ((!target || !unit->canAttack(target) && agent.poke) {
//    if (!target || !unit->canAttack(target)) {
//      Unit* newTarget = utils::getBestScoreCopy(state->unitsInfo().enemyUnits(), [&](Unit* u) {
//        if (u->gone || !unit->canAttack(u)) return kfInfty;
//        return utils::distance(u, unit);
//      }, kfInfty);
//      if (newTarget) {
//        return doAction(agent.moveTo(newTarget->pos()));
//      }
//    }
    //auto& allies = task->squadUnits(); // fixme squadunits is empty when fleeing
    auto& allies = state->unitsInfo().myCompletedUnitsOfType(unit->type);
    Unit* nearestTarget = utils::getBestScoreCopy(agent.legalTargets, [&](Unit* u) {
      if (!unit->canAttack(u) || u->type == buildtypes::Protoss_Interceptor) return kfInfty;
      return std::max(utils::distance(u, unit) - (float)u->rangeAgainst(unit), 0.0f) + (u->unit.shield + u->unit.health) / 100.0f;
    }, kfInfty);
    if (nearestTarget && (nearestTarget->cloaked() && !nearestTarget->detected())) target = nearestTarget;
    if (target && target->type == buildtypes::Protoss_Interceptor) {
      agent.target = target = nearestTarget;
      agent.targetInRange = nearestTarget->inRangeOf(unit);
    }
    bool snipe = false;
    if (target && target->flying() && target->type == buildtypes::Protoss_Observer) {
      Vec2 attackPos = nearestTarget->posf();
      float myDamage = 0.0f;
      float myHp = 0.0f;
      for (Unit* u : allies) {
        if (u->type == unit->type) {
          auto& ua = task->agents_->at(u);
          if (utils::distance(unit->pos(), u->pos()) <= 4.0f * 8) {
            myDamage += unit->computeDamagePerFrame(nearestTarget);
            myHp += unit->unit.shield + unit->unit.health;
          }
        }
      }
      float opDamage = 0.0f;
      for (Unit* u : agent.legalTargets) {
        if (utils::distanceBB(u, u->posf(), unit, attackPos) <= u->rangeAgainst(unit) + 16.0f) {
          opDamage += u->computeDamagePerFrame(unit);
        }
      }
      float opTtl = (target->unit.shield + target->unit.health) / myDamage;
      float myTtl = myHp / myDamage;
      if (myTtl >= 24 * 9 && myTtl > opTtl * 2) {
        snipe = true;
      }
    }
    if (target && target->flying() && (target->cloaked() && !target->detected())) {
      if (agent.wantsToFight || utils::distanceBB(unit, target) <= 4.0f * 4 || snipe) {
        int nCorsairs = 0;
        Unit* nearestCorsair = utils::getBestScoreCopy(allies, [&](Unit* u) {
          if (u->type != buildtypes::Protoss_Corsair) return kfInfty;
          if (!u->active()) return kfInfty;
          float d = utils::distanceBB(target, u->posf());
          if (d > 4.0f * 10) return kfInfty;
          ++nCorsairs;
          if (u->unit.health < 20) return kfInfty;
          return d;
        }, kfInfty);
        if (nCorsairs >= 2 && nearestCorsair) {
          Unit* splashTarget = utils::getBestScoreCopy(allies, [&](Unit* u) {
            if (u == unit || !u->flying()) return kfInfty;
            if (!u->active()) return kfInfty;
            float d = utils::distanceBB(target, u->posf());
            if (d > 12.5f) return kfInfty;
            if (d > 6.25f && u->type != buildtypes::Protoss_Corsair) return kfInfty;
            if (u->unit.health < 20) return kfInfty;
            return d;
          }, kfInfty);
          if (nearestCorsair == unit) {
            if (utils::distanceBB(target, unit->posf()) > 4.5f) {
              return doAction(agent.moveTo(unit->posf() + (target->posf() - unit->posf()).normalize() * 24.0f));
            } else if (!splashTarget) {
              return doAction(agent.moveTo(target->pos()));
            }
          }
          if (splashTarget) {
            agent.wantsToFight = true;
            target = agent.target = splashTarget;
            agent.targetInRange = splashTarget->inRangeOf(unit);
            //return doAction(agent.attack(splashTarget));
          }
        }
      }
    }
    if (snipe && target && ((!target->cloaked() && !target->burrowed()) || target->detected())) {
        agent.wantsToFight = true;
        agent.target = target;
        agent.targetInRange = target->inRangeOf(unit);
    }
  }

  if (!agent.wantsToFight && (unit->type == buildtypes::Terran_Wraith || unit->type == buildtypes::Protoss_Scout || unit->type == buildtypes::Protoss_Corsair)) {
    Unit* nearestThreat = utils::getBestScoreCopy(agent.legalTargets, [&](Unit* u) {
      if (!u->canAttack(unit)) return kfInfty;
      return utils::distance(u, unit) - (float)u->rangeAgainst(unit);
    }, kfInfty);

    Unit* nearestTarget = utils::getBestScoreCopy(agent.legalTargets, [&](Unit* u) {
      if (!unit->canAttack(u) || ((u->cloaked() || u->burrowed()) && !u->detected())) return kfInfty;
      return std::max(utils::distance(u, unit) - (float)u->rangeAgainst(unit), 0.0f) + (u->unit.shield + u->unit.health) / 100.0f;
    }, kfInfty);

    if (nearestThreat) {
      int latency = state->latencyFrames();
      Vec2 myPos = unit->posf() + unit->velocity() * latency;
      Vec2 threatPos = nearestThreat->posf() + nearestThreat->velocity() * latency;
      float desiredRange = (float)nearestThreat->rangeAgainst(unit) + 8.0f;
      if (unit->type == buildtypes::Protoss_Corsair) desiredRange += 8.0f;
      if (unit->unit.shield + unit->unit.health < (unit->unit.max_shield + unit->unit.health) * 0.75f) desiredRange += 8.0f;
      float d = utils::distanceBB(unit, myPos, nearestThreat, threatPos) - desiredRange;
      if (d > 0) {
        agent.wantsToFight = true;
        if (!agent.targetInRange) {
          agent.target = target = nearestTarget;
        }
      }
    } else {
      agent.wantsToFight = true;
    }

    if (nearestTarget && agent.poke) {
      float r = std::min((float)unit->rangeAgainst(nearestTarget), utils::distance(unit->pos(), nearestTarget->pos()));
      Vec2 attackPos = nearestTarget->posf() - (nearestTarget->posf() - unit->posf()).normalize() * r;
      float myDamage = 0.0f;
      //for (Unit* u : task->squadUnits()) {
      for (Unit* u : state->unitsInfo().myCompletedUnitsOfType(unit->type)) {
        //if (u->type == buildtypes::Terran_Wraith || u->type == buildtypes::Protoss_Scout || u->type == buildtypes::Protoss_Corsair) {
        if (true) {
          auto& ua = task->agents_->at(u);
          if (ua.poke && utils::distance(unit->pos(), u->pos()) <= 4.0f * 4) {
            myDamage += unit->computeDamagePerFrame(nearestTarget);
          }
        }
      }
      float opDamage = 0.0f;
      for (Unit* u : agent.legalTargets) {
        if (utils::distanceBB(u, u->posf(), unit, attackPos) <= u->rangeAgainst(unit) + 6.0f) {
          opDamage += u->computeDamagePerFrame(unit);
        }
      }
      if (myDamage > opDamage) {
        agent.wantsToFight = true;
        agent.target = target = nearestTarget;
      }
    }

  }

  if (!agent.wantsToFight || !target) {
    auto r = BehaviorCircleAround().onPerform(agent);
    if (r.isFinal) {
      return r;
    }
    if ((!target || !unit->canAttack(target)) && !task->targetingLocation) {
      Unit* nearestThreat = utils::getBestScoreCopy(agent.legalTargets, [&](Unit* u) {
        return utils::distance(u, unit) - u->rangeAgainst(unit);
      });
      if (!nearestThreat || !unit->inRangeOfPlus(nearestThreat, 8.0f)) {
        Unit* nearestTarget = utils::getBestScoreCopy(state->unitsInfo().enemyUnits(), [&](Unit* u) {
          if (u->gone || !unit->canAttack(u)) return kfInfty;
          return utils::distance(u, unit);
        }, kfInfty);
        if (nearestTarget) {
          agent.target = target = nearestTarget;
          agent.targetInRange = target->inRangeOf(unit);
        }
      }
      if (!target && (!nearestThreat || !unit->inRangeOfPlus(nearestThreat, 12.0f))) {
        if (utils::distance(unit->pos(), agent.scoutPos) <= 4.0f * 3 || state->currentFrame() - agent.lastSetScoutPos >= 24 * 30) {
          uint32_t v = common::Rand::rand();
          Unit* scoutTarget = utils::getBestScoreCopy(state->unitsInfo().enemyUnits(), [&](Unit* u) {
            if (u->gone) return kfInfty;
            v = v * 22695477 + 1;
            return (float)(v >> 16);
          }, kfInfty);
          if (scoutTarget) {
            agent.scoutPos = scoutTarget->pos();
          } else {
            agent.scoutPos.x = common::Rand::sample(std::uniform_int_distribution<int>(0, state->mapWidth() - 1));
            agent.scoutPos.y = common::Rand::sample(std::uniform_int_distribution<int>(0, state->mapHeight() - 1));
          }
          agent.lastSetScoutPos = state->currentFrame();
        }
        return doAction(agent.moveTo(agent.scoutPos));
      }
      if (!target) {
        auto kiteVector = [&]() {
          Vec2 adjustment;
          for (Unit* u : agent.legalTargets) {
            if (u->canAttack(unit) && u->type != buildtypes::Protoss_Interceptor) {
              float distance =
                  std::max(utils::distanceBB(unit, u), DFOASG(0.125f, 1.0f));
              float maxDistance = DFOASG(4 * 14, 4 * 3);
              if (distance <= maxDistance) {
                adjustment += (unit->posf() - u->posf()) * (maxDistance / distance);
              }
            }
          }
          if (nearestThreat) {
            adjustment = (nearestThreat->posf() - unit->posf()).normalize() * 0.15f + adjustment.normalize();
          }
          return unit->posf() + adjustment.normalize() * moveAhead(unit);
        };
        return doAction(agent.moveTo(kiteVector()));
      }
      if (target && !agent.wantsToFight) {
        return doAction(agent.moveTo(target->pos()));
      }
    }
  }

  if (!target || !agent.wantsToFight) {
    return pass;
  }

  int latency = state->latencyFrames();

  Vec2 myPos = unit->posf() + unit->velocity() * latency;
  Vec2 targetPos = target->posf() + target->velocity() * latency;
  float range = (float)unit->rangeAgainst(target);

  double cd = unit->cd();
  if (state->currentFrame() - agent.lastAttack < latency) {
    cd += 30;
  }
  float distance = utils::distanceBB(unit, myPos, target, targetPos);
  if (cd <= latency && agent.targetInRange) {
    return pass;
  }

  bool dodgeSplash = false;
  bool anyThreats = false;
  for (Unit* u : agent.legalTargets) {
    if (u->canAttack(unit) &&
        utils::distanceBB(unit, u) <=
            u->rangeAgainst(unit) + DFOASG(4 * 5, 8)) {
      anyThreats = true;
    }
    if (unit->flying()) {
      if (u->type == buildtypes::Terran_Valkyrie ||
          u->type == buildtypes::Protoss_Corsair ||
          u->type == buildtypes::Protoss_Archon ||
          u->type == buildtypes::Protoss_High_Templar) {
        if (utils::distance(unit, u) <= 4 * 8) {
          dodgeSplash = true;
          break;
        }
      }
    } else {
      if (u->type == buildtypes::Terran_Siege_Tank_Siege_Mode ||
          u->type == buildtypes::Protoss_Reaver ||
          u->type == buildtypes::Protoss_High_Templar ||
          u->type == buildtypes::Zerg_Lurker) {
        if (utils::distance(unit, u) <= 4 * 10) {
          dodgeSplash = true;
          break;
        }
      }
    }
  }

  auto attackVector = [&]() {
    if (!dodgeSplash || utils::distanceBB(unit, target) <= unit->rangeAgainst(target) + 6.0f) {
      return targetPos;
    }
    Vec2 adjustment;
    for (Unit* u : task->squadUnits()) {
      if (u != unit && u->flying() == unit->flying()) {
        float distance =
            std::max(utils::distanceBB(unit, u), DFOASG(0.125f, 1.0f));
        float maxDistance = DFOASG(4 * 3, 6);
        if (distance <= maxDistance) {
          adjustment += (myPos - u->posf()) * (maxDistance / distance);
        }
      }
    }
    Vec2 moveDir = (targetPos - myPos).normalize();
    if (adjustment == Vec2()) {
      adjustment = moveDir;
    }
    return unit->posf() +
        (moveDir + moveDir + adjustment.normalize()).normalize() * moveAhead(unit);
  };

  auto kiteVector = [&]() {
    Vec2 adjustment;
    for (Unit* u : agent.legalTargets) {
      if (u->canAttack(unit) && u->type != buildtypes::Protoss_Interceptor) {
        float distance =
            std::max(utils::distanceBB(unit, u), DFOASG(0.125f, 1.0f));
        float maxDistance = DFOASG(4 * 10, 4 * 3);
        if (distance <= maxDistance) {
          adjustment += (myPos - u->posf()) * (maxDistance / distance);
        }
      }
    }
    Vec2 moveDir = (myPos - targetPos).normalize();
    return unit->posf() + (moveDir + adjustment.normalize()).normalize() * moveAhead(unit);
  };

  auto willMoveIntoDanger = [&]() {
    if (!anyThreats) {
      return false;
    }
    Vec2 attackPos = targetPos + (myPos - targetPos).normalize() * range;
    for (Unit* u : agent.legalTargets) {
      if (u != target && u->canAttack(unit) &&
          utils::distance(u->pos(), Position(attackPos)) <=
              u->rangeAgainst(unit) + 6.0f) {
        if (u->velocity().length() < DFOASG(0.125, 0.25) ||
            u->velocity().dot(u->posf() - unit->posf()) <= 0) {
          return true;
        }
      }
    }
    return false;
  };

  if (unit->type == buildtypes::Terran_Science_Vessel || unit->type == buildtypes::Zerg_Queen) {
    Unit* nearestThreat = utils::getBestScoreCopy(agent.legalTargets, [&](Unit* u) {
      return utils::distance(u, unit) - u->rangeAgainst(unit);
    });
    float sf = 8.0f;
    if (unit->type == buildtypes::Zerg_Queen) sf = 16.0f;
    if (nearestThreat && utils::distanceBB(unit, nearestThreat) <= nearestThreat->rangeAgainst(unit) + sf) {
      return doAction(agent.moveTo(kiteVector()));
    }
  }

  if (!dodgeSplash && target->rangeAgainst(unit) > unit->rangeAgainst(target)) {
    if ((unit->cd() <= latency && agent.targetInRange) || utils::distanceBB(unit, target) <= 4.0f) {
      return pass;
    }
    return doAction(agent.moveTo(attackVector()));
  }

  if (target->type != buildtypes::Protoss_Interceptor) {
    if ((!anyThreats && !dodgeSplash) ||
        (target->velocity().length() >= DFOASG(0.125, 0.25) &&
         target->velocity().dot(target->posf() - unit->posf()) > 0)) {
      if (!willMoveIntoDanger()) {
        if (cd <= latency) {
          return pass;
        }
        return doAction(agent.moveTo(attackVector()));
      }
    } else if (target->velocity().length() < DFOASG(0.25, 0.25)) {
      if (target->topSpeed >= unit->topSpeed * DFOASG(0.66, 0.33)) {
        if (!willMoveIntoDanger()) {
          range /= DFOASG(4, 3);
        }
      }
    }
  }
  if (unit->type != buildtypes::Protoss_Corsair || target->type == buildtypes::Zerg_Scourge) {
    float tr = 128.0f / tc::BW::data::TurnRadius[unit->type->unit];
    if ((distance - range) / unit->topSpeed + tr < cd) {
      return doAction(agent.moveTo(kiteVector()));
    }
  }

  if ((unit->cd() <= latency && agent.targetInRange) || utils::distanceBB(unit, target) <= 4.0f) {
    return pass;
  }
  return doAction(agent.moveTo(attackVector()));
}

MicroAction BehaviorAsMutaliskVsScourge::onPerform(Agent& agent) {
  auto state = agent.state;
  auto unit = agent.unit;

  if (unit->type != buildtypes::Zerg_Mutalisk) {
    return pass;
  }

  if (agent.target == nullptr ||
      agent.target->type != buildtypes::Zerg_Scourge || !agent.targetInRange) {
    return pass;
  }

  auto u = agent.target;
  auto cd = unit->cd();
  auto scourgeVelo = u->velocity();
  auto myVelo = unit->velocity();
  auto dirToScourge = (Vec2(u) - Vec2(unit)).normalize();
  if (VLOG_IS_ON(2)) {
    utils::drawCircle(
        state, unit, unit->unit.airRange * tc::BW::XYPixelsPerWalktile);
  }

  scourgeVelo.normalize();
  myVelo.normalize();

  auto distBB = utils::distanceBB(u, unit);
  if (agent.mutaliskTurning ||
      (cd < 3 && distBB > 3 && myVelo.dot(dirToScourge) > 0)) {
    VLOG(3) << utils::unitString(unit) << " is launching a scourge attack ";
    utils::drawCircle(state, u, 25, tc::BW::Color::Red);
    agent.mutaliskTurning = false;
    return doAction(agent.attack(u));
  } else if (cd < 6 && distBB > 8) {
    VLOG(3) << utils::unitString(unit) << " is turning to face unit";
    utils::drawCircle(state, u, 25, tc::BW::Color::Red);
    agent.mutaliskTurning = true;
    return doAction(agent.moveTo(Vec2(unit) + dirToScourge * 20));
  } else if (myVelo.dot(scourgeVelo) < 0.1 || !u->atTopSpeed()) {
    VLOG(3) << utils::unitString(unit) << " is moving away from the scourge";
    return doAction(agent.moveTo(Vec2(unit) + dirToScourge * -20));
  } else {
    // http://liquipedia.net/starcraft/Mutalisk_vs._Scourge_Control#Method_2
    auto pos1 = Vec2(unit) + scourgeVelo.rotateDegrees(100) * 20;
    auto pos2 = Vec2(unit) + scourgeVelo.rotateDegrees(-200) * 20;
    auto pos = pos1.distanceTo(u) < pos2.distanceTo(u) ? pos2 : pos1;
    utils::drawCircle(state, unit, 25, tc::BW::Color::Blue);
    VLOG(3) << utils::unitString(unit)
            << " is using the triangle technique and moving to dir "
            << scourgeVelo;
    return doAction(agent.moveTo(pos));
  }
  return pass;
}

MicroAction BehaviorAsScourge::onPerform(Agent& agent) {
  auto& task = agent.task;
  auto unit = agent.unit;

  if (unit->type != buildtypes::Zerg_Scourge) {
    return pass;
  }

  if (agent.target == nullptr) {
    if (!unit->threateningEnemies.empty()) {
      auto centroid = utils::centerOfUnits(unit->threateningEnemies);
      auto pos = Vec2(unit) + (Vec2(unit) - Vec2(centroid)).normalize() * 10;
      return doAction(agent.moveTo(pos));
    } else {
      return doAction(agent.moveTo(task->center_));
    }
  }
  // Scourges wants to click past the target so to move at full speed, and
  // issue an attack command when they are right on top of the target.
  auto invalidUnit = [&](Unit const* u) {
    if (u->type == buildtypes::Protoss_Interceptor ||
        u->type == buildtypes::Zerg_Overlord || u->type->isBuilding) {
      return true;
    }
    auto it = task->enemyStates_->find(u);
    if (it == task->enemyStates_->end()) {
      return true;
    }
    if (u != agent.target &&
        it->second.damages > u->unit.health + u->unit.shield - 15) {
      return true;
    }
    return false;
  };
  if (invalidUnit(agent.target)) {
    agent.target = nullptr;
    for (auto u : unit->enemyUnitsInSightRange) {
      if (!invalidUnit(u)) {
        agent.target = u;
        break;
      }
    }
  }
  if (agent.target == nullptr) {
    return doNothing;
  }
  if (agent.target->inRangeOf(unit, 3)) {
    return doAction(agent.attack(agent.target));
  }
  auto dir = Vec2(agent.target) - Vec2(unit);
  dir.normalize();
  return doAction(agent.moveTo(Vec2(unit) + dir * 25));
}

constexpr int kLurkerBurrowFrames = 24; // Magic: 0+
constexpr int kLurkerUnburrowFrames = 12; // Magic: 0+
MicroAction BehaviorAsLurker::onPerform(Agent& agent) {
  if (agent.unit->type != buildtypes::Zerg_Lurker) {
    return pass;
  }

  return pass;
}

MicroAction BehaviorCircleAround::onPerform(Agent& agent) {
  State* state = agent.state;
  Unit* unit = agent.unit;
  auto* task = agent.task;
  Unit* target = agent.target;

  if (agent.wantsToFight || agent.targetInRange) {
    return pass;
  }

  Unit* nearestThreat = utils::getBestScoreCopy(agent.legalTargets, [&](Unit* u) {
    if (!u->canAttack(unit) || u->type == buildtypes::Protoss_Interceptor) return kfInfty;
    return utils::distance(u, unit) - (float)u->rangeAgainst(unit);
  }, kfInfty);

  if (nearestThreat) {
    float desiredRange = nearestThreat->rangeAgainst(unit) + 8.0f;
    if (unit->unit.health < unit->unit.health * 0.75f) desiredRange += 8.0f;
    if (unit->type == buildtypes::Terran_Dropship && !nearestThreat->type->isBuilding) desiredRange += 8.0f;
    float d = utils::distanceBB(unit, nearestThreat) - desiredRange;
    if (d > 0 && unit->targetGroupPos != kInvalidPosition) {
      return doAction(agent.moveTo(unit->targetGroupPos));
    }
    if (d <= -8.0f) {
      return pass;
    }
    Position targetPos = kInvalidPosition;
    if (d <= 0.0f) {
      Vec2 offset = (nearestThreat->posf() - unit->posf()).normalize() * moveAhead(unit);
      targetPos = unit->pos() - Position(offset);
    } else if (d <= 9.0f) {
      Vec2 offset = (nearestThreat->posf() - unit->posf()).normalize() * moveAhead(unit);
      if (d < 3.0f) {
        offset.rotateDegrees(agent.circleDirection > 0 ? 100 : -100);
      } else {
        offset.rotateDegrees(agent.circleDirection > 0 ? 80 : -80);
      }

      targetPos = unit->pos() + Position(offset);
    }
    if (targetPos != kInvalidPosition) {
      auto* tile = state->tilesInfo().tryGetTile(targetPos.x, targetPos.y);
      if (!tile || (!unit->flying() && (!tile->entirelyWalkable || tile->building))) {
        agent.circleDirection = -agent.circleDirection;
      }

      return doAction(agent.moveTo(targetPos));
    }
  }

  return pass;
}

MicroAction BehaviorGoForRepair::onPerform(Agent& agent) {
  State* state = agent.state;
  Unit* unit = agent.unit;
  auto* task = agent.task;
  Unit* target = agent.target;

  if (agent.targetInRange) {
    return pass;
  }

  Unit* nearestThreat = utils::getBestScoreCopy(agent.legalTargets, [&](Unit* u) {
    if (!u->canAttack(unit)) {
      return kfInfty;
    }
    return utils::distance(u, unit) - (float)u->rangeAgainst(unit);
  }, kfInfty);
  if (nearestThreat && unit->inRangeOfPlus(nearestThreat, 8.0f)) {
    return pass;
  }

  if (unit->unit.health < unit->unit.max_health / 2) {
    agent.goingForRepair = true;
  }
  if (unit->unit.health >= unit->unit.max_health) {
    agent.goingForRepair = false;
  }

  if (!agent.goingForRepair || state->resources().ore < 20 || state->resources().gas < 20) {
    return pass;
  }

  Unit* scv = utils::getBestScoreCopy(state->unitsInfo().myCompletedUnitsOfType(buildtypes::Terran_SCV), [&](Unit* u) {
    auto taskData = state->board()->taskDataWithUnit(u);
    if (taskData.task && taskData.owner &&
       (taskData.owner->name().find("AutoBuild") == std::string::npos)) {
      return kfInfty;
    }
    if (!u->unit.orders.empty() && u->unit.orders.front().type == +tc::BW::Order::Repair && u->unit.orders.front().targetId != unit->id) {
     return kfInfty;
    }
    return utils::distance(u->pos(), unit->pos());
  }, kfInfty);

  if (scv) {
    return doAction(agent.moveTo(scv->pos()));
  }

  return pass;
}

MicroAction BehaviorSupportTanks::onPerform(Agent& agent) {
  State* state = agent.state;
  Unit* unit = agent.unit;
  auto* task = agent.task;
  Unit* target = agent.target;

  if (target && !unit->canAttack(target)) target = nullptr;

  if (agent.wantsToFight && unit->type == buildtypes::Terran_Vulture && agent.targetInRange && unit->cd() <= state->latencyFrames()) {
    return pass;
  }

  if (unit->lastCombatScore < 1 && unit->lastCombatScore >= 0.97f) {
    return pass;
  }

  if (state->board()->get("gameIsWon", false)) {
    return pass;
  }

  if (unit->lastCombatScore >= 0.4f) {
    int marines = 0;
    int tanks = 0;
    for (Unit* u : task->squadUnits()) {
      if (utils::distance(u->pos(), unit->pos()) >= 4.0f * 26) {
        continue;
      }
      if (u->type == buildtypes::Terran_Siege_Tank_Siege_Mode || u->type == buildtypes::Terran_Siege_Tank_Tank_Mode || u->type == buildtypes::Protoss_Reaver || u->type == buildtypes::Zerg_Lurker) {
        ++tanks;
      } else if (u->type == buildtypes::Terran_Marine || u->type == buildtypes::Terran_Ghost) {
        ++marines;
      }
    }
    if (marines >= tanks * 12) return pass;
  }

  if (agent.wantsToFight && target && false) {
    if (!unit->flying()) {
      float tankScore = 0.0f;
      float otherScore = 0.0f;
      for (Unit* u : task->squadUnits()) {
        if (utils::distance(u->pos(), target->pos()) >= 4.0f * 16) {
          continue;
        }
        if (u->type == buildtypes::Terran_Siege_Tank_Siege_Mode || u->type == buildtypes::Terran_Siege_Tank_Tank_Mode || u->type == buildtypes::Protoss_Reaver || u->type == buildtypes::Zerg_Lurker) {
          tankScore += u->type->gScore;
        } else {
          otherScore += u->type->gScore;
        }
      }

      if (tankScore < otherScore / 2) {
        float myScore = 0.0f;
        float opScore = 0.0f;
        for (Unit* u : task->squadUnits()) {
          myScore += u->type->gScore;
        }
        for (Unit* u : agent.legalTargets) {
          if (u->type->hasGroundWeapon && !u->type->isWorker) {
            opScore += u->type->gScore;
          }
        }

        if (myScore >= opScore * 2) {
          if (unit->sieged()) {
            agent.postCommand(tc::BW::UnitCommandType::Unsiege);
            return doNothing;
          }
          return pass;
        }
      }

      float myScore = 0.0f;
      float opScore = 0.0f;
      for (Unit* u : task->squadUnits()) {
        myScore += u->type->gScore;
      }
      for (Unit* u : agent.legalTargets) {
        opScore += u->type->gScore;
      }

      if (myScore >= opScore * 4) {
        return pass;
      }
    }
  }
  
  auto findNewTarget = [&]() {
    Unit* r = nullptr;
    float best = kfInfty;
    for (Unit* u : task->squadUnits()) {
      if (u->type == buildtypes::Terran_Siege_Tank_Siege_Mode || u->type == buildtypes::Terran_Siege_Tank_Tank_Mode || u->type == buildtypes::Protoss_Reaver || u->type == buildtypes::Zerg_Lurker) {
        for (Unit* e : agent.legalTargets) {
          if (unit->canAttack(e) && !e->type->isBuilding) {
            float d = utils::distance(u->pos(), e->pos());
            if (d <= 4.0f * 12) {
              float d2 = utils::distance(unit->pos(), e->pos());
              float s = d * d + d2 * d2;
              if (s < best) {
                best = s;
                r = e;
              }
            }
          }
       }
      }
    }
    return r;
  };

  auto findScoutTarget = [&]() {
    Unit* r = nullptr;
    float best = kfInfty;
    for (Unit* u : task->squadUnits()) {
      if (u->type == buildtypes::Terran_Siege_Tank_Siege_Mode || u->type == buildtypes::Terran_Siege_Tank_Tank_Mode || u->type == buildtypes::Protoss_Reaver) {
        for (Unit* e : agent.legalTargets) {
          if (u->canAttack(e) && (!e->visible || !e->detected())) {
            float d = utils::distance(u->pos(), e->pos());
            if (d <= 4.0f * 16) {
              float d2 = utils::distance(unit->pos(), e->pos());
              float s = d * d + d2 * d2;
              if (s < best) {
                best = s;
                r = e;
              }
            }
          }
       }
      }
    }
    return r;
  };

  Unit* nearestTank = utils::getBestScoreCopy(task->squadUnits(), [&](Unit* u) {
    if (u->type != buildtypes::Terran_Siege_Tank_Siege_Mode && u->type != buildtypes::Terran_Siege_Tank_Tank_Mode && u->type != buildtypes::Protoss_Reaver && u->type != buildtypes::Zerg_Lurker) {
      return kfInfty;
    }
    return unit->flying() ? utils::distance(unit->pos(), u->pos()) : state->areaInfo().walkPathLength(unit->pos(), u->pos());
  }, kfInfty);
  if (nearestTank && target && (target->cloaked() || target->burrowed())) {
    if (utils::distanceBB(nearestTank, target) <= 4.0f * 12) {
      return pass;
    }
  }
  if (nearestTank && nearestTank->type == buildtypes::Protoss_Reaver && target) {
    float td = utils::distanceBB(nearestTank, target);
    if (td <= 4.0f * 9 || td < utils::distanceBB(unit, target) - 8.0f) {
      return pass;
    }
    if (unit->lastCombatScore >= 0.25) {
      return pass;
    }
  }
  if (nearestTank && (unit->flying() || state->areaInfo().walkPathLength(unit->pos(), nearestTank->pos()) < 4.0f * 24)) {
    Unit* tankTarget = agent.task->agents_->at(nearestTank).target;
    Unit* nearestTankThreat = utils::getBestScoreCopy(agent.legalTargets, [&](Unit* u) {
      if (!u->canAttack(nearestTank) || u->type->isWorker || !unit->canAttack(u) || ((u->cloaked() || u->burrowed()) && !u->detected())) {
        return kfInfty;
      }
      return utils::distance(u, nearestTank) - (float)u->rangeAgainst(nearestTank);
    }, kfInfty);
    Unit* nearestThreat = utils::getBestScoreCopy(agent.legalTargets, [&](Unit* u) {
      if (!u->canAttack(unit) || u->type->isWorker || u->type == buildtypes::Protoss_Arbiter) {
        return kfInfty;
      }
      return utils::distance(u, unit) - (float)u->rangeAgainst(unit);
    }, kfInfty);

    auto tryCircle = [&]() {
      if (nearestTank && nearestThreat && utils::distance(unit->pos(), nearestTank->pos()) >= 4.0f * 12) {
        if ((nearestThreat->posf() - nearestTank->posf()).rotate((nearestThreat->posf() - unit->posf()).normalize().invertAngle()).y >= 0) {
          agent.circleDirection = 1;
        } else {
          agent.circleDirection = -1;
        }
      }
      return BehaviorCircleAround().onPerform(agent);
    };

    auto isnearestunit = [&]() {
      Unit* nearest = utils::getBestScoreCopy(task->squadUnits(), [&](Unit* u) {
        if (u->flying() != unit->flying()) return kfInfty;
        if (u->type != buildtypes::Terran_Marine) {
          return kfInfty;
        }
        return utils::distance(u, target);
      }, kfInfty);
      return nearest == unit;
    };

    //if (target && (unit->type == buildtypes::Terran_Vulture || isnearestunit())) {
    if (target && isnearestunit()) {
      if (nearestTankThreat && unit->canAttack(nearestTankThreat)) {
        if (unit->cd() > state->latencyFrames() + 4) {
          if (utils::distanceBB(nearestTankThreat, nearestTank) < utils::distanceBB(nearestTankThreat, unit) + 12.0f) {
            return doAction(agent.moveTo(nearestTankThreat->posf()));
          }
        }
      }
      if (!nearestThreat || utils::distanceBB(unit, nearestThreat) > nearestThreat->rangeAgainst(unit) + 12.0f || state->currentFrame() - nearestThreat->lastSeen >= 24 * 14) {
        return pass;
      }
    }

    if (nearestTankThreat && nearestThreat) {
      float threatDistance = utils::distance(nearestTankThreat, nearestTank) - nearestTankThreat->rangeAgainst(nearestTank);
      for (Unit* u : state->unitsInfo().myUnits()) {
        if (u->type == buildtypes::Terran_Siege_Tank_Siege_Mode || u->type == buildtypes::Terran_Siege_Tank_Tank_Mode || u->type->isWorker || u->type == buildtypes::Protoss_Reaver || u->type == buildtypes::Zerg_Lurker) {
          if (utils::distance(u->pos(), unit->pos()) <= 4.0f * 12) {
            for (Unit* e : agent.legalTargets) {
              float d = utils::distance(u, e) - e->rangeAgainst(u);
              if (d < threatDistance) {
                threatDistance = d;
                nearestTank = u;
                nearestTankThreat = e;
              }
            }
          }
        }
      }
      float maxrange = 4.0f * 12;
      if (nearestTank->type == buildtypes::Zerg_Lurker) maxrange = 4.0f * 6;
      bool tankInDanger = utils::distanceBB(nearestTank, nearestTankThreat) < std::max(std::min((float)nearestTankThreat->rangeAgainst(nearestTank) + 8.0f, maxrange), std::min((float)unit->rangeAgainst(nearestTankThreat) + 8.0f, 4.0f * 8));
      Vec2 upos = unit->posf() + unit->velocity() * state->latencyFrames();
      Vec2 threatpos = nearestThreat->posf() + nearestThreat->velocity() * state->latencyFrames();
      auto flyingInDanger = [&]() {
        //if (!unit->flying() || (target && target->flying())) {
        if (!unit->flying()) {
          return false;
        }
        if (target && agent.targetInRange && tankInDanger) {
          if (utils::distanceBB(target, nearestTank) < 8.0f || !nearestTank->canAttack(target)) {
            return false;
          }
        }
        if (target && target->flying()) {
          Unit* nearestGroundThreat = utils::getBestScoreCopy(agent.legalTargets, [&](Unit* u) {
            if (u->flying() || !u->canAttack(unit)) {
              return kfInfty;
            }
            return utils::distance(u, unit) - (float)u->rangeAgainst(unit);
          }, kfInfty);
          if (!nearestGroundThreat || utils::distanceBB(unit, nearestGroundThreat) >= nearestGroundThreat->rangeAgainst(unit) + 12.0f) {
            return false;
          }
        }
        if (utils::distanceBB(nearestThreat, threatpos, unit, upos) >= nearestThreat->rangeAgainst(unit) + (unit->unit.health >= unit->unit.max_health * 0.75 ? 4.0f : 8.0f)) {
          return false;
        }
        return true;
        //return target->damageMultiplier(unit) > unit->damageMultiplier(target);
      };
      if (unit->type == buildtypes::Terran_Science_Vessel && !flyingInDanger()) {
        Unit* cloakedTarget = utils::getBestScoreCopy(agent.legalTargets, [&](Unit* e) {
          if (!(e->cloaked() || e->burrowed())) {
            return kfInfty;
          }
          Unit* detector = utils::getBestScoreCopy(task->squadUnits(), [&](Unit* u) {
            if (u->type != unit->type) {
              return kfInfty;
            }
            return utils::distance(u, e);
          }, kfInfty);
          if (detector != unit) {
            return kfInfty;
          }
          return utils::distance(unit, e);
        }, kfInfty);
        if (cloakedTarget) {
          return doAction(agent.moveTo(cloakedTarget->pos()));
        }
      }
      if (unit->flying() && !target && !unit->canAttack(nearestTankThreat) && !flyingInDanger()) {
        if (!tankTarget || (tankTarget->visible && !nearestTankThreat->visible)) tankTarget = nearestTankThreat;
        if (tankTarget->visible) {
          Unit* scoutTarget = findScoutTarget();
          if (scoutTarget) tankTarget = scoutTarget;
        }
        if (tankTarget && (!tankTarget->visible || utils::distanceBB(unit, tankTarget) > unit->sightRange - 8.0f)) {
          return doAction(agent.moveTo(tankTarget->posf() + (unit->posf() - tankTarget->posf()).normalize() * unit->sightRange));
        }
      }
      //if (tankInDanger || !unit->canAttack(nearestTankThreat) || flyingInDanger()) {
      if (unit->flying() && !unit->type->hasGroundWeapon ? !target || flyingInDanger() : !tankInDanger || !target) {
        if (utils::distanceBB(unit, upos, nearestThreat, threatpos) < nearestThreat->rangeAgainst(unit) + (unit->flying() ? 8.0f : 24.0f)) {
          //return doAction(agent.moveTo(unit->posf() + ((unit->posf() - nearestThreat->posf()).normalize() + (nearestTank->posf() - unit->posf()).normalize() * 0.5f).normalize() * 6.0f));
          auto kiteVector = [&]() {
            Vec2 adjustment;
            for (Unit* u : agent.legalTargets) {
              if (u->canAttack(unit) && u->type != buildtypes::Protoss_Interceptor) {
                float distance =
                    std::max(utils::distanceBB(unit, u), DFOASG(0.125f, 1.0f));
                float maxDistance = DFOASG(4 * 14, 4 * 3);
                if (distance <= maxDistance) {
                  //adjustment += (unit->posf() - u->posf()) * (maxDistance / distance);
                  adjustment += (unit->posf() - u->posf()).normalize();
                }
              }
            }
            //return unit->posf() + adjustment.normalize() * 6.0f;
            Vec2 adjustment2;
            if (!unit->flying()) {
              for (Unit* u : task->squadUnits()) {
                if (!u->flying()) {
                  float distance =
                      std::max(utils::distanceBB(unit, u), DFOASG(0.125f, 1.0f));
                  float maxDistance = DFOASG(4 * 3, 4 * 1.5);
                  if (distance <= maxDistance) {
                    adjustment2 += (unit->posf() - u->posf()) * (maxDistance / distance);
                  }
                }
              }
            }
            Vec2 moveDir = (nearestTank->posf() - unit->posf()).normalize();
            return unit->posf() + (moveDir * 0.33f + adjustment.normalize() * 1.5f + adjustment2.normalize() * 0.5f).normalize() * moveAhead(unit);
          };
          return doAction(agent.moveTo(kiteVector()));
        }
      }

      if (unit->flying() && utils::distanceBB(nearestThreat, threatpos, unit, upos) <= nearestThreat->rangeAgainst(unit) + 16.0f) {
        Unit* nearestTarget = utils::getBestScoreCopy(agent.legalTargets, [&](Unit* u) {
          if (!unit->canAttack(u) || ((u->cloaked() || u->burrowed()) && !u->detected())) return kfInfty;
          return utils::distance(unit->pos(), u->pos());
        }, kfInfty);
        if (nearestTarget && nearestTarget != target) {
          agent.target = nearestTarget;
          agent.targetInRange = utils::distanceBB(unit, upos, nearestTarget, nearestTarget->posf()) <= unit->rangeAgainst(nearestTarget);
        }

        auto r = tryCircle();
        if (r.isFinal) {
          return r;
        }
      }
      if (!agent.targetInRange) {
        Unit* retarget = findNewTarget();
        if (retarget) {
          agent.target = retarget;
          agent.targetInRange = utils::distanceBB(unit, upos, retarget, retarget->posf()) <= unit->rangeAgainst(retarget);
        }
      }
    } else if (nearestTankThreat) {
      if (unit->flying() && !unit->canAttack(nearestTankThreat) && (tankTarget || nearestTankThreat)) {
        if (!tankTarget || (tankTarget->visible && !nearestTankThreat->visible)) tankTarget = nearestTankThreat;
        if (tankTarget->visible) {
          Unit* scoutTarget = findScoutTarget();
          if (scoutTarget) tankTarget = scoutTarget;

          auto r = tryCircle();
          if (r.isFinal) {
            return r;
          }
        }
        return doAction(agent.moveTo(tankTarget->posf() + (unit->posf() - tankTarget->posf()).normalize() * unit->sightRange / 2));
      }
    } else {
      if (unit->flying() && !target) {
        return doAction(agent.moveTo(nearestTank->pos()));
      }
    }
  }

  return pass;

//  Unit* nearestTank = utils::getBestScoreCopy(task->squadUnits(), [&](Unit* u) {
//    if (u->type != buildtypes::Terran_Siege_Tank_Siege_Mode && u->type != buildtypes::Terran_Siege_Tank_Tank_Mode) {
//      return kfInfty;
//    }
//    return state->areaInfo().walkPathLength(unit->pos(), u->pos());
//  }, kfInfty);
//  if (nearestTank && (unit->flying() || state->areaInfo().walkPathLength(unit->pos(), nearestTank->pos()) < 4.0f * 24)) {
//    Unit* nearestThreat = utils::getBestScoreCopy(agent.legalTargets, [&](Unit* u) {
//      return utils::distance(u, nearestTank) - u->rangeAgainst(unit);
//    });
//    nearestTank = utils::getBestScoreCopy(task->squadUnits(), [&](Unit* u) {
//      if (u->type != buildtypes::Terran_Siege_Tank_Siege_Mode && u->type != buildtypes::Terran_Siege_Tank_Tank_Mode) {
//        return kfInfty;
//      }
//      return state->areaInfo().walkPathLength(nearestThreat->pos(), u->pos());
//    }, kfInfty);
//    Unit* nearestWorker = utils::getBestScoreCopy(state->unitsInfo().myWorkers(), [&](Unit* u) {
//      return state->areaInfo().walkPathLength(nearestThreat->pos(), u->pos());
//    }, kfInfty);
//    //if (nearestTank && nearestThreat && utils::distanceBB(nearestTank, nearestThreat) > std::max(nearestThreat->rangeAgainst(nearestTank), nearestThreat->rangeAgainst(unit)) && utils::distanceBB(unit, nearestThreat) > nearestThreat->rangeAgainst(unit)) {
//    if (nearestTank && nearestThreat && utils::distanceBB(nearestTank, nearestThreat) > std::max(nearestThreat->rangeAgainst(nearestTank), nearestThreat->rangeAgainst(unit)) && (utils::distanceBB(unit, nearestThreat) <= nearestThreat->rangeAgainst(unit) + 12.0f || utils::distance(unit, nearestThreat) + 4.0f * 3 < utils::distance(nearestTank, nearestThreat))) {
//      //if (nearestTank && nearestThreat && utils::distance(unit, nearestThreat) < utils::distance(nearestTank, nearestThreat) && (!nearestWorker || utils::distance(nearestThreat, nearestWorker) > std::max((float)nearestThreat->rangeAgainst(nearestTank), 6.0f))) {
//      //if (utils::distance(unit, nearestThreat) - 4.0f * 3 < utils::distance(nearestTank, nearestThreat) && (!nearestWorker || utils::distance(nearestThreat, nearestWorker) > std::max((float)nearestThreat->rangeAgainst(nearestWorker), 6.0f))) {
//      nearestWorker = nullptr;
//      if (!nearestWorker || utils::distance(nearestThreat, nearestWorker) > std::max((float)nearestThreat->rangeAgainst(nearestWorker), 6.0f)) {
//        //if (utils::distanceBB(unit, nearestTank) > 4.0f * 3) {
//        if (true) {
//          auto kiteVector = [&]() {
//            Vec2 adjustment;
//            for (Unit* u : agent.legalTargets) {
//              if (u->canAttack(unit)) {
//                float distance =
//                    std::max(utils::distanceBB(unit, u), DFOASG(0.125f, 1.0f));
//                float maxDistance = DFOASG(4 * 10, 4 * 3);
//                if (distance <= maxDistance) {
//                  adjustment += (unit->posf() - u->posf()) * (maxDistance / distance);
//                }
//              }
//            }
//            Vec2 adjustment2;
//            if (!unit->flying()) {
//              for (Unit* u : task->squadUnits()) {
//                if (!u->flying()) {
//                  float distance =
//                      std::max(utils::distanceBB(unit, u), DFOASG(0.125f, 1.0f));
//                  float maxDistance = DFOASG(4 * 3, 4 * 1.5);
//                  if (distance <= maxDistance) {
//                    adjustment2 += (unit->posf() - u->posf()) * (maxDistance / distance);
//                  }
//                }
//              }
//            }
//            Vec2 moveDir = (nearestTank->posf() - unit->posf()).normalize();
//            return unit->posf() + (moveDir * 0.33f + adjustment.normalize() * 1.5f + adjustment2.normalize()).normalize() * 6.0f;
//          };
//          if (utils::distanceBB(nearestTank, nearestThreat) > 4.0f * 10) {
//            return doAction(agent.moveTo(kiteVector()));
//          }
//        }
//      }
//    }
//  }
//  return pass;
}

MicroAction BehaviorAsHydralisk::onPerform(Agent& agent) {
  State* state = agent.state;
  Unit* unit = agent.unit;
  Unit* target = agent.target;
  auto* task = agent.task;

  if (unit->type == buildtypes::Terran_Marine) {
    BWAPI::Unit bwu = BWAPI::BroodwarPtr->getUnit(unit->id);
    if (bwu) {
      //bool aggressive = state->board()->get<bool>("TacticsAttack", true);
      bool aggressive = unit->lastCombatScore >= 0.5f;
      if (bwu->isLoaded()) {
        auto t = bwu->getTransport();
        if (t) {
          Unit* transport = state->unitsInfo().getUnit(t->getID());
          if (transport->type == buildtypes::Terran_Bunker) {
            if (state->currentFrame() - unit->lastFiredWeapon < 60) {
              return doNothing;
            }
//            if (!unit->limitSpecial(state)) {
//              return doNothing;
//            }
            bool unload = false;
            if (target) {
              Unit* targetTarget = utils::getBestScoreCopy(state->unitsInfo().myUnits(), [&](Unit* u) {
                return utils::distanceBB(u, target);
              });
              if (targetTarget && targetTarget != transport && targetTarget->inRangeOf(target)) {
                if (utils::distanceBB(targetTarget, transport) > 8.0f) {
                  unload = true;
                }
              }
              if (!unload && aggressive) {
                if (utils::distanceBB(unit, target) > unit->rangeAgainst(target) + 8.0f) {
                  unload = true;
                }
              }
            } else if (task->targetingLocation) {
              Position pos(task->targetX, task->targetY);
              if (utils::distance(unit->pos(), pos) > 4.0f * 12) {
                unload = true;
              }
            } else {
              unload = true;
            }
            if (!unload && target && !target->flying() && utils::distanceBB(transport, target) >= 4.0f) {
              Unit* cc = utils::getBestScoreCopy(state->unitsInfo().myResourceDepots(), [&](Unit* u) {
                return utils::distanceBB(u, target);
              });
              if (cc && state->areaInfo().walkPathLength(target->pos(), cc->pos()) < state->areaInfo().walkPathLength(unit->pos(), cc->pos())) {
                unload = true;
              }
            }
            if (unload) {
              if (unit->limitSpecial(state)) {
                t->unload(bwu);
              }
              return doNothing;
            }
          }
        }
      } else if (target) {
        Unit* bunker = utils::getBestScoreCopy(state->unitsInfo().myCompletedUnitsOfType(buildtypes::Terran_Bunker), [&](Unit* u) {
          return utils::distance(u->pos(), target->pos());
        });
        float d = bunker ? utils::distance(unit->pos(), bunker->pos()) : 0.0f;
        if (bunker && (d <= 4.0f * 8 || (d <= 4.0f * 20 && state->areaInfo().walkPathLength(unit->pos(), bunker->pos()) < state->areaInfo().walkPathLength(unit->pos(), target->pos())))) {
          Unit* cc = utils::getBestScoreCopy(state->unitsInfo().myResourceDepots(), [&](Unit* u) {
            return utils::distanceBB(u, target);
          });
          auto bwb = BWAPI::BroodwarPtr->getUnit(bunker->id);
          if (aggressive && !agent.targetInRange && agent.wantsToFight) {
            if (utils::distanceBB(unit, bunker->pos(), target, target->pos()) > unit->rangeAgainst(target) + 8.0f) {
              bwb = nullptr;
            }
          }
          if (cc && state->areaInfo().walkPathLength(target->pos(), cc->pos()) < state->areaInfo().walkPathLength(unit->pos(), cc->pos())) {
            bwb = nullptr;
          }
          if (bwb) {
            int space = BWAPI::UnitType(bunker->type->unit).spaceProvided();
            int req = BWAPI::UnitType(unit->type->unit).spaceRequired();
            for (auto x : bwb->getLoadedUnits()) {
              space -= x->getType().spaceRequired();
            }
            if (target && space >= req) {
              Unit* targetTarget = utils::getBestScoreCopy(state->unitsInfo().myUnits(), [&](Unit* u) {
                return utils::distance(u->pos(), target->pos());
              });
              if (targetTarget && (targetTarget == unit || target->inRangeOf(bunker))) {
                if (unit->limitSpecial(state)) {
                  bwu->rightClick(bwb);
                }
                return doNothing;
              }
            }
          }
        }
      }
    }
  }

  bool haveSiegeMode = state->hasResearched(buildtypes::Tank_Siege_Mode);
  if ((unit->type == buildtypes::Terran_Siege_Tank_Tank_Mode || unit->type == buildtypes::Terran_Siege_Tank_Siege_Mode || unit->type == buildtypes::Zerg_Lurker) && target) {
    int latency = state->latencyFrames();
    Vec2 myPos = unit->posf() + unit->velocity() * latency;
    Vec2 targetPos = target->posf() + target->velocity() * latency;
    float targetDistance = utils::distanceBB(unit, myPos, target, targetPos);
    bool sieged = unit->type == buildtypes::Terran_Siege_Tank_Siege_Mode || unit->burrowed();
    bool isLurker = unit->type == buildtypes::Zerg_Lurker;
    Unit* nearestThreat = utils::getBestScoreCopy(agent.legalTargets, [&](Unit* u) {
      return utils::distance(u, unit) - u->rangeAgainst(unit);
    });
    Unit* nearestTarget = utils::getBestScoreCopy(agent.legalTargets, [&](Unit* u) {
      if (!unit->canAttack(u) || ((u->cloaked() || u->burrowed()) && !u->detected())) return kfInfty;
      return utils::distance(u, unit);
    }, kfInfty);
    float nearestTargetDistance = nearestTarget ? utils::distanceBB(unit, nearestTarget) : kfInfty;
    //bool shouldSiege = nearestThreat && nearestThreat->type != buildtypes::Terran_Vulture && nearestThreat->type != buildtypes::Terran_Goliath;
    bool shouldSiege = nearestThreat != nullptr;
    float easyScore = 0.75f;
    if (nearestThreat && (nearestThreat->type == buildtypes::Terran_Siege_Tank_Tank_Mode || nearestThreat->type == buildtypes::Terran_Siege_Tank_Siege_Mode || nearestThreat->type == buildtypes::Zerg_Sunken_Colony || nearestThreat->type == buildtypes::Protoss_Photon_Cannon || nearestThreat->type == buildtypes::Terran_Bunker || nearestThreat->type == buildtypes::Protoss_Reaver)) {
      easyScore = 0.99f;
    }
    if (!unit->sieged() && unit->lastCombatScore < 1 && unit->lastCombatScore >= easyScore) {
      if (target->unit.groundRange < 4 * 8 && !isLurker) {
        shouldSiege = false;
      }
    }
    if (unit->type == buildtypes::Terran_Siege_Tank_Tank_Mode) {
      if (!haveSiegeMode && target && (!agent.targetInRange || unit->cd() > 0)) {
        for (Unit* u : state->unitsInfo().myCompletedUnitsOfType(buildtypes::Terran_Machine_Shop)) {
          if (u->upgrading() && u->upgradingType == buildtypes::Tank_Siege_Mode) {
            if (utils::distanceBB(unit, target) <= 11.0f) {
              return doAction(agent.moveTo(
                  unit->posf() +
                  (unit->posf() - target->posf()).normalize() * 4.0f));
            }
            break;
          }
        }
      }
      if (utils::distanceBB(unit, target) <= 8.0f) {
        shouldSiege = false;
      } else if (target->type == buildtypes::Protoss_Dark_Templar) {
        shouldSiege = false;
      } else {
        float targetTankDistance = kfInfty;
        Unit* targetTank = nullptr;
        float helpTankDistance = 4.0f * 10;
        Unit* helpTank = nullptr;
        int nTanks = 0;
        for (Unit* a : task->squadUnits()) {
          if (a->type == buildtypes::Terran_Siege_Tank_Tank_Mode || a->type == buildtypes::Terran_Siege_Tank_Siege_Mode || a->type == buildtypes::Zerg_Lurker) {
            ++nTanks;
            float d = utils::distanceBB(a, target);
            if (d < targetTankDistance) {
              targetTankDistance = d;
              targetTank = a;
            }
            if (a != unit) {
              d = utils::distanceBB(a, unit);
              if (d < helpTankDistance) {
                helpTankDistance = d;
                helpTank = a;
              }
            }
          }
        }
        if (nTanks >= 3 && targetTank == unit && !helpTank) {
          shouldSiege = false;
        }
        if (nTanks >= 3 && helpTank && helpTankDistance > 4.0f * 3) {
          bool backup = true;
          if (agent.tankBackupState == 0) {
            agent.tankBackupFrame = state->currentFrame();
            agent.tankBackupState = 1;
          } else {
            if (state->currentFrame() - agent.tankBackupFrame <= 24 * 2) {
            } else if (state->currentFrame() - agent.tankBackupFrame >= 24 * 4) {
              agent.tankBackupState = 0;
            } else {
              backup = false;
            }
          }
          if (backup) {
            shouldSiege = false;
            if (!agent.targetInRange || unit->cd() > latency) {
              return doAction(agent.moveTo(helpTank->pos()));
            }
          }
        }
      }
    }
    //if (haveSiegeMode && distance > 4.0f * 4 && state->currentFrame() - target->lastSeen < 24 * 10) {
    if (((haveSiegeMode && targetDistance > 4.0f * 4) || isLurker) && nearestTarget) {
      if (targetDistance <= (isLurker ? unit->rangeAgainst(target) : 4.0f * 12)) {
      //if (agent.targetInRange) {
        if (!sieged && shouldSiege && canSiegeHere(state, unit)) {
          agent.lastSiege = state->currentFrame();
          if (isLurker) {
            auto bwl = bwapiLock();
            auto bwu = BWAPI::Broodwar->getUnit(unit->id);
            if (bwu) bwu->burrow();
          } else agent.postCommand(isLurker ? tc::BW::UnitCommandType::Burrow : tc::BW::UnitCommandType::Siege);
          return doNothing;
        }
      } else if (nearestTargetDistance <= 4.0f * 16) {
        auto nearbyEnemyTank = [&]() {
          for (Unit* e : agent.legalTargets) {
            if (e->type == buildtypes::Terran_Siege_Tank_Tank_Mode || e->type == buildtypes::Terran_Siege_Tank_Siege_Mode) {
              if (utils::distance(unit->pos(), e->pos()) <= 4.0f * 16) {
                return true;
              }
            }
          }
          return false;
        };
        auto defendingBase = [&]() {
          for (Unit* e : agent.legalTargets) {
            for (Unit* u : state->unitsInfo().myBuildings()) {
              if (!u->flying() && u->inRangeOfPlus(e, 8.0f)) {
                return true;
              }
            }
            for (Unit* u : state->unitsInfo().myWorkers()) {
              if (!u->flying() && u->inRangeOfPlus(e, 8.0f)) {
                auto taskData = state->board()->taskDataWithUnit(u);
                if (taskData.task && taskData.owner &&
                   (taskData.owner->name().find("Gatherer") != std::string::npos || taskData.owner->name().find("Builder") != std::string::npos)) {
                  return true;
                }
              }
            }
          }
          return false;
        };
        auto push = [&]() {
          //return false;

          if (unit->lastCombatScore >= 0.5f) {
            return true;
          }
          if (target && target->type->isBuilding) {
            return true;
          }

          float myScore = 0.0f;
          float opScore = 0.0f;
          int mytanks = 0;
          int optanks = 0;
          int dmatrices = 0;
          for (Unit* u : task->squadUnits()) {
            if (utils::distance(u->pos(), nearestThreat->pos()) >= 4.0f * 16) {
              continue;
            }
            if (u->type == buildtypes::Terran_Science_Vessel && utils::distance(unit->pos(), u->pos()) <= 4.0f * 10) {
              if (u->unit.energy >= buildtypes::Defensive_Matrix->energyCost) ++dmatrices;
            }
            myScore += u->type->gScore;
            if (u->type == buildtypes::Terran_Siege_Tank_Tank_Mode || u->type == buildtypes::Terran_Siege_Tank_Siege_Mode || u->type == buildtypes::Zerg_Lurker) {
              ++mytanks;
            }
          }
          if (dmatrices >= 2) return true;
          for (Unit* u : agent.legalTargets) {
            if (u->type->hasGroundWeapon && !u->type->isWorker) {
              opScore += u->type->gScore;
            }
            if (u->type == buildtypes::Terran_Siege_Tank_Tank_Mode || u->type == buildtypes::Terran_Siege_Tank_Siege_Mode) {
              ++optanks;
            }
          }

          if (myScore >= opScore * 3) {
            return true;
          }

          float tankScore = 0.0f;
          float otherScore = 0.0f;
          for (Unit* u : task->squadUnits()) {
            if (utils::distance(u->pos(), target->pos()) >= 4.0f * 16) {
              continue;
            }
            if (u->type == buildtypes::Terran_Siege_Tank_Siege_Mode || u->type == buildtypes::Terran_Siege_Tank_Tank_Mode) {
              tankScore += u->type->gScore;
            } else {
              otherScore += u->type->gScore;
            }
          }

          if (tankScore < otherScore / 2) {
            if (myScore >= opScore * 2) {
              return true;
            }
          }

          //return false;
          return mytanks > std::max(optanks + 1, std::min(optanks * 2, optanks + 4)) && myScore > opScore;
        };
        //if (target->type == buildtypes::Terran_Siege_Tank_Tank_Mode || target->type == buildtypes::Terran_Siege_Tank_Siege_Mode) {
        if (nearbyEnemyTank()) {
          float nearestTankDistance = kfInfty;
          Unit* nearestTank = nullptr;
          for (Unit* a : task->squadUnits()) {
            if (a != unit && (a->type == buildtypes::Terran_Siege_Tank_Tank_Mode || a->type == buildtypes::Terran_Siege_Tank_Siege_Mode || a->type == buildtypes::Zerg_Lurker)) {
            //if (!a->flying()) {
              Unit* aNearestTarget = utils::getBestScoreCopy(agent.legalTargets, [&](Unit* u) {
                if (!a->canAttack(u) || ((u->cloaked() || u->burrowed()) && !u->detected())) return kfInfty;
                return utils::distance(u, a);
              }, kfInfty);
              if (aNearestTarget) {
                float d = utils::distanceBB(a, aNearestTarget);
                if (d < nearestTankDistance) {
                  nearestTankDistance = d;
                  nearestTank = a;
                }
              }
            }
          }
//          if (nearestTank && nearestThreat && utils::distanceBB(nearestThreat, nearestTank) >= 4.0f * 12 - nearestThreat->rangeAgainst(unit) - utils::distanceBB(unit, nearestTank)) {
//            nearestTankDistance = nearestTargetDistance;
//          }
          if (defendingBase() || push() || unit->flag(tc::Unit::Flags::DefenseMatrixed)) {
            nearestTankDistance = (isLurker ? unit->rangeAgainst(target) : 4.0f * 12) - 1.0f;
          }
          if (nearestTargetDistance <= nearestTankDistance + 0.5f) {
            if (!sieged && shouldSiege && canSiegeHere(state, unit)) {
              agent.lastSiege = state->currentFrame();
              if (isLurker) {
                auto bwl = bwapiLock();
                auto bwu = BWAPI::Broodwar->getUnit(unit->id);
                if (bwu) bwu->burrow();
              } else agent.postCommand(isLurker ? tc::BW::UnitCommandType::Burrow: tc::BW::UnitCommandType::Siege);
            }
            return doNothing;
          }
          if (nearestTargetDistance > nearestTankDistance + 1.0f) {
            if (sieged) {
              //if (state->currentFrame() - agent.lastSiege >= 24 * 5 && unit->cd() == 0) {
              if (!agent.targetInRange) {
                if (isLurker) {
                  auto bwl = bwapiLock();
                  auto bwu = BWAPI::Broodwar->getUnit(unit->id);
                  if (bwu && unit->limitSpecial(state)) bwu->unburrow();
                } else agent.postCommand(isLurker ? tc::BW::UnitCommandType::Unburrow : tc::BW::UnitCommandType::Unsiege);
                return doNothing;
              }
            }
          }
          if (!sieged && nearestTargetDistance <= nearestTankDistance + 6.0f) {
            auto pos = nearestTarget->posf() + (unit->posf() - nearestTarget->posf()).normalize() * nearestTankDistance;
            auto* tile = state->tilesInfo().tryGetTile((int)pos.x, (int)pos.y);
            if (tile && tile->entirelyWalkable) {
              return doAction(agent.moveTo(pos));
            } else {
              return doAction(agent.moveTo(nearestTarget->pos()));
            }
          }
        } else if (state->currentFrame() - agent.lastSiege >= 24 * 10) {
          if (!sieged && canSiegeHere(state, unit)) {
            for (Unit* u : task->squadUnits()) {
              if (u->type == buildtypes::Terran_Siege_Tank_Siege_Mode || u->type == buildtypes::Zerg_Lurker) {
                //if (target->inRangeOf(u) && shouldSiege && utils::distanceBB(unit, u) <= 11.0f - target->rangeAgainst(u)) {
                if (utils::distanceBB(u, target) <= 4.0f * 8 && shouldSiege && utils::distanceBB(unit, u) <= 11.0f - target->rangeAgainst(u)) {
                  agent.lastSiege = state->currentFrame();
                  if (isLurker) {
                    auto bwl = bwapiLock();
                    auto bwu = BWAPI::Broodwar->getUnit(unit->id);
                    if (bwu) bwu->burrow();
                  } else agent.postCommand(isLurker ? tc::BW::UnitCommandType::Burrow : tc::BW::UnitCommandType::Siege);
                  return doNothing;
                }
              }
            }
          }
        }
      }
    }
  }
  if (unit->type == buildtypes::Terran_Siege_Tank_Siege_Mode || (unit->type == buildtypes::Zerg_Lurker && unit->burrowed())) {
    if (!agent.targetInRange || !target || utils::distanceBB(unit, target) <= 4.0f * 2) {
      int nTanks = 0;
      int nSiegingTanks = 0;
      for (Unit* u : task->squadUnits()) {
        if (u->type == buildtypes::Terran_Siege_Tank_Tank_Mode || u->type == buildtypes::Terran_Siege_Tank_Siege_Mode || u->type == buildtypes::Zerg_Lurker) {
          if (utils::distance(unit, u) <= 4.0f * 12) {
            ++nTanks;
            if (!u->unit.orders.empty()) {
              auto o = u->unit.orders.front().type;
              if (o == tc::BW::Order::Sieging || o == tc::BW::Order::Unsieging || o == tc::BW::Order::Unburrowing) {
                ++nSiegingTanks;
              }
            }
          }
        }
      }
      if (nSiegingTanks <= nTanks / 2 && state->currentFrame() - agent.lastSiege >= 24 * 5 && unit->cd() == 0) {
        if (unit->burrowed()) {
          auto bwl = bwapiLock();
          auto bwu = BWAPI::Broodwar->getUnit(unit->id);
          if (bwu && unit->limitSpecial(state)) bwu->unburrow();
        } else agent.postCommand(unit->burrowed() ? tc::BW::UnitCommandType::Unburrow : tc::BW::UnitCommandType::Unsiege);
      }
      return doNothing;
    }
    if (!target) {
      return doNothing;
    }
    return doAction(agent.attack(target));
  }

  if ((unit->type == buildtypes::Terran_Siege_Tank_Tank_Mode || (unit->type == buildtypes::Zerg_Lurker && !unit->burrowed())) && !agent.targetInRange) {
    Unit* arbiter = utils::getBestScoreCopy(agent.legalTargets, [&](Unit* u) {
      if (u->type != buildtypes::Protoss_Arbiter) return kfInfty;
      return utils::distance(u, unit);
    }, kfInfty);
    if (arbiter && utils::distance(unit, arbiter) <= 4 * 20) {
      Vec2 move;
      for (Unit* u : task->squadUnits()) {
        if (u->type == buildtypes::Terran_Siege_Tank_Tank_Mode || u->type == buildtypes::Terran_Siege_Tank_Siege_Mode) {
          float d = std::max(utils::distanceBB(u, unit), 1.0f);
          if (d <= 8.0f) {
            move += (unit->posf() - u->posf()) / d;
          }
        }
      }
      if (move != Vec2()) {
        return doAction(agent.moveTo(unit->pos() + Position(move.normalize() * 12.0f)));
      }
    }
  }

  if (unit->type == buildtypes::Terran_Siege_Tank_Tank_Mode && target && agent.targetInRange) {
    //if (unit->inRangeOf(target) || utils::distanceBB(unit, target) > target->rangeAgainst(unit) + 8.0f) {
    if (utils::distanceBB(unit, target) > target->rangeAgainst(unit) + 8.0f) {
      return doAction(agent.attack(target));
    }
  }

  if (unit->type == buildtypes::Terran_Medic) {
    if (!unit->unit.orders.empty()) {
      auto o = unit->unit.orders.front().type;
      if (o == tc::BW::Order::MedicHeal || o == tc::BW::Order::MedicHealToIdle) {
        return doNothing;
      }
    }
    if (unit->unit.energy < agent.energy) {
      agent.lastSpentEnergy = state->currentFrame();
    }
    agent.energy = unit->unit.energy;
    if (state->currentFrame() - agent.lastSpentEnergy < 8) {
      return doNothing;
    }
    Unit* nearestThreat = utils::getBestScoreCopy(agent.legalTargets, [&](Unit* e) {
      if (!e->type->hasGroundWeapon) {
        return kfInfty;
      }
      return state->areaInfo().walkPathLength(unit->pos(), e->pos());
    }, kfInfty);

    auto len = [&](Position a, Position b) {
      float r = state->areaInfo().walkPathLength(a, b);
      if (r == kfInfty) {
        r = utils::distance(a, b) * 2;
      }
      return r;
    };

    target = utils::getBestScoreCopy(task->squadUnits(), [&](Unit* u) {
      if (u == unit || !u->type->isBiological || u->type == buildtypes::Terran_Medic) {
        return kfInfty;
      }
      float myDistance = len(unit->pos(), u->pos());
      auto& ua = task->agents_->at(u);
      if (ua.hasMedicFrame == state->currentFrame() && ua.medic && len(u->pos(), ua.medic->pos()) < myDistance) {
        myDistance += 4.0f * 24;
      }
      float damage = u->unit.max_health - u->unit.health;
      float r = -damage;
      float threatDistance = 0.0f;
      if (nearestThreat) {
        threatDistance = len(u->pos(), nearestThreat->pos());
      }
      r += myDistance*myDistance*0.25f + threatDistance*threatDistance*0.125f;
      return r;
    }, kfInfty);
    if (target) {
      auto& targetAgent = task->agents_->at(target);
      targetAgent.hasMedicFrame = state->currentFrame();
      targetAgent.medic = unit;
      if (state->currentFrame() - agent.lastHealMove > 6) {
        agent.lastHealMove = state->currentFrame();
        state->board()->postCommand(
            tc::Client::Command(tc::BW::Command::CommandUnit, unit->id, tc::BW::UnitCommandType::Use_Tech_Position, -1, target->x, target->y, buildtypes::Healing->tech),
            task->upcId());
      }
      //return doAction(agent.attack(Position(target)));
      return doNothing;
    }
    return pass;
  }

  if (unit->type == buildtypes::Terran_Marine || unit->type == buildtypes::Terran_Firebat) {
    if (!unit->stimmed() && agent.targetInRange && state->hasResearched(buildtypes::Stim_Packs)) {
      if (unit->limitSpecial(state)) {
        state->board()->postCommand(
            tc::Client::Command(tc::BW::Command::CommandUnit, unit->id, tc::BW::UnitCommandType::Use_Tech, -1, 0, 0, buildtypes::Stim_Packs->tech),
            task->upcId());
      }
    }
  }

//  unit->drawString = "boop";

//  if (unit->type == buildtypes::Terran_Goliath) {

//    float irange = kfInfty;
//    Unit* nearestInterceptor = nullptr;
//    for (Unit* u : state->unitsInfo().visibleEnemyUnits()) {
//      if (u->type == buildtypes::Protoss_Interceptor) {
//        float d = utils::distanceBB(unit, u);
//        if (d < irange) {
//          irange = d;
//          nearestInterceptor = u;
//        }
//      }
//    }
//    if (nearestInterceptor) {
//      unit->drawString = (target ? utils::unitString(target) : std::string("null")) + " " + std::to_string(agent.targetInRange) + " ? " + std::to_string(agent.wantsToFight) + " i " + utils::unitString(nearestInterceptor) + " " + std::to_string(irange);

//      if (nearestInterceptor->inRangeOf(unit)) {
//        return doAction(agent.attack(nearestInterceptor));
//      }
//    }
//  }

  if (unit->type == buildtypes::Terran_Goliath && agent.wantsToFight && target && agent.targetInRange) {
    return doAction(agent.attack(target));
  }

  if (unit->type == buildtypes::Terran_Goliath && target && target->type == buildtypes::Protoss_Interceptor && agent.targetInRange) {
    return doAction(agent.attack(target));
//    if (agent.wantsToFight || unit->cd() == 0) {
//      return doAction(agent.attack(target));
//    }
  }

  if (unit->type == buildtypes::Terran_Marine || unit->type == buildtypes::Terran_Firebat || unit->type == buildtypes::Terran_Vulture || unit->type == buildtypes::Terran_Goliath || unit->type == buildtypes::Protoss_Dragoon || unit->type == buildtypes::Protoss_Zealot || unit->type == buildtypes::Terran_Ghost) {
    if (!agent.targetInRange || true) {
      auto r = BehaviorSupportTanks().onPerform(agent);
      if (r.isFinal) {
        return r;
      }
    }
  }

  //if (unit->type != buildtypes::Zerg_Hydralisk && unit->type != buildtypes::Protoss_Dragoon && unit->type != buildtypes::Protoss_Reaver && unit->type != buildtypes::Terran_Marine && unit->type != buildtypes::Terran_Firebat && unit->type != buildtypes::Terran_Siege_Tank_Tank_Mode && unit->type != buildtypes::Terran_Vulture && unit->type != buildtypes::Terran_Goliath) {
  if (unit->type != buildtypes::Zerg_Hydralisk && unit->type != buildtypes::Protoss_Dragoon && unit->type != buildtypes::Protoss_Reaver && unit->type != buildtypes::Terran_Marine && unit->type != buildtypes::Terran_Ghost && unit->type != buildtypes::Terran_Firebat && unit->type != buildtypes::Terran_Siege_Tank_Tank_Mode && unit->type != buildtypes::Terran_Goliath) {
    return pass;
  }

//  if (agent.wantsToFight && target && unit->type == buildtypes::Terran_Vulture) {
//    Vec2 upos = unit->posf() + unit->velocity() * (state->latencyFrames() - 1);
//    Vec2 tpos = target->posf() + target->velocity() * (state->latencyFrames() - 1);
//    bool kite = false;
//    if (state->currentFrame() - unit->lastAttack < state->latencyFrames()) {
//      kite = true;
//    }
//    if (utils::distanceBB(unit, upos, target, tpos) <= unit->rangeAgainst(target)) {

//    }
//  }

  if (unit->type == buildtypes::Protoss_Reaver) {
    if (unit->remainingBuildTrainTime <= state->latencyFrames() && unit->unit.associatedCount < 5) {
      state->board()->postCommand(
          tc::Client::Command(tc::BW::Command::CommandUnit, unit->id, tc::BW::UnitCommandType::Train, -1, 0, 0, buildtypes::Protoss_Scarab->unit),
          task->upcId());
    }
    if (target && unit->cd() == 0 && agent.targetInRange && unit->unit.associatedCount >= 1) {
      return doAction(agent.attack(target));
    }
  }

//  if (unit->unit.flags & tc::Unit::AttackFrame) {
//    return doNothing;
//  }

  if (!agent.targetInRange && !agent.wantsToFight) {
    return pass;
  }

//  if (unit->type == buildtypes::Protoss_Dragoon && target && target->type == buildtypes::Protoss_Dragoon) {
//    if (agent.prevTargetInRange && state->currentFrame() - agent.lastAttack < 90 && unit->cd() <= state->latencyFrames()) {
//      return doNothing;
//    }
//    return doAction(agent.attack(target));
//  }

//  if (unit->type != buildtypes::Zerg_Hydralisk) {
//    if (agent.prevTargetInRange && state->currentFrame() - agent.lastAttack <= 6) {
//      return doNothing;
//    }
//  }

  if (unit->type != buildtypes::Terran_Marine && state->currentFrame() < 24 * 60 * 6 && false) {
    if (target && target->type == buildtypes::Zerg_Zergling) {
      Vec2 sum;
      int n = 0;
      for (Unit* u : state->unitsInfo().myWorkers()) {
        if (utils::distance(u->pos(), unit->pos()) <= 4.0f * 8) {
          sum += u->posf();
          ++n;
        }
      }
      if (n >= 2) {
        Vec2 pos = sum / n;
        Vec2 offset;
        for (Unit* e : agent.legalTargets) {
          if (utils::distance(e->pos(), Position(pos)) <= 16.0f) {
            offset += (pos - e->posf()).normalize();
          }
        }
        if (offset != Vec2()) {
          pos += offset.normalize() * 16.0f;
        }
        float d = utils::distance(unit->pos(), Position(pos));
        if (d > 8.0f && (!agent.targetInRange || unit->cd() > state->latencyFrames())) {
          bool ok = true;
//          for (Unit* e : agent.legalTargets) {
//            if (utils::distance(e->pos(), pos) <= d) {
//              ok = false;
//            }
//          }
          if (ok) {
            return doAction(agent.moveTo(pos));
          }
        }
      }
      return doAction(agent.attack(target));
    }
  }


  auto canMoveInDirection = [&](Vec2 odir,
                                float distance = DFOASG(4.0f * 2, 4.0f)) {
    Vec2 dir = odir.normalize();
    for (float d = 4.0f; ; d += 4.0f) {
      Position pos = d < distance ? Position(unit->posf() + dir * d) : Position(unit->posf() + odir);
      auto* tile = state->tilesInfo().tryGetTile(pos.x, pos.y);
      if (!tile || !tile->entirelyWalkable || tile->building) {
        return false;
      }
      if (d >= distance) break;
    }
    return true;
  };
  auto adjust = [&](Vec2 targetPos) {
    if (unit->flying()) return targetPos;
    Vec2 dir = targetPos - unit->posf();
    float d = dir.length();
    if (canMoveInDirection(dir, d)) return targetPos;
    for (float dg = 22.5f; dg <= 180.0f; dg += 22.5f) {
      if (dg > 90.0f) dg += 22.5f;
      Vec2 l = dir.rotateDegrees(dg);
      if (canMoveInDirection(l, d)) return unit->posf() + l;
      Vec2 r = dir.rotateDegrees(-dg);
      if (canMoveInDirection(r, d)) return unit->posf() + r;
    }
    return targetPos;
  };

  //VLOG(0) << " i am " << unit->type->name << " wantsToFight is " << agent.wantsToFight << " target is " << target;

  if ((unit->type == buildtypes::Protoss_Dragoon || unit->type == buildtypes::Protoss_Reaver || unit->type == buildtypes::Terran_Siege_Tank_Tank_Mode || unit->type == buildtypes::Terran_Vulture) && target) {

    int latency = state->latencyFrames();
    Vec2 myPos = unit->posf() + unit->velocity() * latency;
    Vec2 targetPos = target->posf() + target->velocity() * latency;
    float range = (float)unit->rangeAgainst(target);

    auto kiteVector = [&]() {
      Vec2 adjustment;
      for (Unit* u : agent.legalTargets) {
        if (u->canAttack(unit) && u->type != buildtypes::Protoss_Interceptor) {
          float distance =
              std::max(utils::distanceBB(unit, u), DFOASG(0.125f, 1.0f));
          float maxDistance = DFOASG(4 * 10, 4 * 3);
          if (distance <= maxDistance) {
            adjustment += (myPos - u->posf()) * (maxDistance / distance);
          }
        }
      }
      for (Unit* u : task->squadUnits()) {
        if (!u->flying()) {
          float distance =
              std::max(utils::distanceBB(unit, u), DFOASG(0.125f, 1.0f));
          float maxDistance = DFOASG(4 * 10, 4 * 3);
          if (distance <= maxDistance) {
            adjustment += (myPos - u->posf()) * (maxDistance / distance);
          }
        }
      }
      Vec2 moveDir = (myPos - targetPos).normalize();
      return unit->posf() + (moveDir + adjustment.normalize()).normalize() * 12.0;
    };

    if (unit->type == buildtypes::Protoss_Reaver && unit->unit.associatedCount == 0) {
      return doAction(agent.moveTo(adjust(kiteVector())));
    }

    if (unit->type == buildtypes::Protoss_Reaver) {
      Unit* nearestThreat = utils::getBestScoreCopy(agent.legalTargets, [&](Unit* u) {
        if (!u->canAttack(unit)) return kfInfty;
        return utils::distance(u, unit) - (float)u->rangeAgainst(unit);
      }, kfInfty);
      if (nearestThreat && unit->inRangeOfPlus(nearestThreat, 12.0f)) {
        return doAction(agent.moveTo(adjust(kiteVector())));
      }
    }


    if (target->type->isBuilding) {
      return pass;
    }

    if (unit->type == buildtypes::Protoss_Dragoon) {
      if (unit->cd() <= state->latencyFrames() && agent.prevTargetInRange && agent.attacking && state->currentFrame() - agent.lastAttack <= 15) {
        if (agent.attacking->inRangeOfPlus(unit, 8.0f)) {
          return doNothing;
        }
      }
      if (state->currentFrame() - unit->lastFiredWeapon < 10 - latency) {
        return doNothing;
      }

      if (agent.poke && false) {
        bool hasRange = unit->unit.groundRange > 4 * 5;
        Unit* nearestThreat = utils::getBestScoreCopy(agent.legalTargets, [&](Unit* u) {
          if (!u->canAttack(unit)) return kfInfty;
          if (u->type == buildtypes::Terran_Bunker && hasRange) return kfInfty;
          return utils::distance(u, unit) - (float)u->rangeAgainst(unit);
        }, kfInfty);
        Unit* nearestTarget = utils::getBestScoreCopy(agent.legalTargets, [&](Unit* u) {
          if (!unit->canAttack(u) || ((u->cloaked() || u->burrowed()) && !u->detected())) return kfInfty;
          return utils::distance(unit, u) - (float)unit->rangeAgainst(u);
        }, kfInfty);
        if (!nearestThreat || !unit->inRangeOfPlus(nearestThreat, 1.0f)) {
          float tr = nearestThreat ? nearestThreat->rangeAgainst(unit) + 1.0f : 0.0f;
          auto canMoveInDirection = [&](Vec2 dir,
                                        float distance = DFOASG(4.0f * 2, 4.0f)) {
            dir = dir.normalize();
            for (float d = 4.0f; d <= distance; d += 4.0f) {
              Vec2 posf = unit->posf() + dir * d;
              if (nearestThreat && utils::distanceBB(unit, posf, nearestThreat, nearestThreat->posf()) <= tr) return false;
              Position pos = Position(posf);
              auto* tile = state->tilesInfo().tryGetTile(pos.x, pos.y);
              if (!tile || !tile->entirelyWalkable || tile->building) {
                return false;
              }
            }
            return true;
          };
          auto tryPoke = [&](Unit* t) {
            if (t->type != buildtypes::Terran_Bunker && t->rangeAgainst(unit) >= unit->rangeAgainst(t)) return false;
            auto rel = t->posf() - unit->posf();
            if (nearestThreat && unit->inRangeOfPlus(nearestThreat, 8.0f) && !canMoveInDirection(rel, rel.length() - unit->rangeAgainst(t))) return false;
            return true;
          };
          if (tryPoke(target)) {
            agent.wantsToFight = true;
          } else if (nearestTarget && tryPoke(nearestTarget)) {
            agent.target = target = nearestTarget;
            agent.wantsToFight = true;
          }
        }
      }

      float distance = utils::distanceBB(unit, myPos, target, targetPos);
      double cd = unit->cd();
      float tr = 128.0f / tc::BW::data::TurnRadius[unit->type->unit];
      if (cd && (distance - range) / unit->topSpeed + tr < cd) {
        return agent.wantsToFight ? doAction(agent.moveTo(adjust(kiteVector()))) : pass;
      }
      if (agent.targetInRange && !agent.prevTargetInRange) {
        return doAction(agent.attack(target));
      }
      if (!agent.targetInRange && state->currentFrame() - agent.lastAttack > 9) {
        return doAction(agent.moveTo(targetPos));
      }
      return doAction(agent.attack(target));
      //return agent.wantsToFight ? pass : doAction(agent.attack(target));
    } else if (unit->type == buildtypes::Protoss_Reaver) {
      float distance = utils::distanceBB(unit, myPos, target, targetPos);
      if (unit->cd() <= state->latencyFrames() && unit->associatedCount >= 1) return agent.wantsToFight ? pass : doAction(agent.attack(target));
      else {
        double cd = unit->cd();
        float tr = 128.0f / tc::BW::data::TurnRadius[unit->type->unit];
        if ((unit->associatedCount == 0 && distance <= range + 4.0f) || (cd && (distance - range) / unit->topSpeed + tr < cd)) {
          return agent.wantsToFight ? doAction(agent.moveTo(adjust(kiteVector()))) : pass;
        }
        return agent.wantsToFight ? pass : doAction(agent.attack(target));
      }
    } else {
      if (agent.targetInRange && !agent.prevTargetInRange) {
        return doAction(agent.attack(target));
      }
  //    if (unit->attackingTarget != target) {
  //      return doAction(agent.attack(target));
  //    }
  //    return doNothing;

      if (unit->cd() >= unit->unit.maxCD - 9) {
        return doNothing;
      }
      if (unit->cd() <= state->latencyFrames() && agent.prevTargetInRange && state->currentFrame() - agent.lastAttack <= 6) {
        return doNothing;
      }
    }

    if (!target->type->isBuilding) {
      double cd = unit->cd();
//      if (state->currentFrame() - agent.lastAttack < latency) {
//        cd += 30;
//      }
      float distance = utils::distanceBB(unit, myPos, target, targetPos);
      float tr = 128.0f / tc::BW::data::TurnRadius[unit->type->unit];
      if ((distance - range) / unit->topSpeed + tr < cd) {
        return doAction(agent.moveTo(adjust(kiteVector())));
      }
    }
    return pass;
    //return doAction(agent.attack(target));
  }

  if (!target) {
    return pass;
  }

  if (((agent.prevTargetInRange && !agent.targetInRange) || (agent.unstuckCounter > 0 && agent.unstuckCounter <= 1)) &&
      unit->velocity() == Vec2()) {
    ++agent.unstuckCounter;
    if (agent.unstuckCounter >= 20) {
    } else if (agent.unstuckCounter >= 10) {
      return doAction(agent.moveTo(unit->pos() + Position(-8 + common::Rand::sample(std::uniform_int_distribution<int>(0, 16)), -8 + common::Rand::sample(std::uniform_int_distribution<int>(0, 16)))));
    } else if (agent.unstuckCounter >= 1) {
      agent.postCommand(tc::BW::UnitCommandType::Stop);
      return doNothing;
    }
  } else {
    agent.unstuckCounter = 0;
  }

  int latency = state->latencyFrames();

  Vec2 myPos = unit->posf() + unit->velocity() * latency;
  Vec2 targetPos = target->posf() + target->velocity() * latency;
  float range = (float)unit->rangeAgainst(target);
  float distance = utils::distanceBB(unit, myPos, target, targetPos);

  double cd = unit->cd();
  if (state->currentFrame() - agent.lastAttack < latency) {
    cd += 15;
    return doNothing;
  }
  if (unit->type == buildtypes::Terran_Marine && state->currentFrame() - agent.lastAttack <= 4) {
    return doNothing;
  }

  auto willMoveIntoDanger = [&]() {
    Vec2 attackPos = utils::distanceBB(unit, target) <= range ? myPos : targetPos + (myPos - targetPos).normalize() * range;
    for (Unit* u : agent.legalTargets) {
      if (u != target && u->canAttack(unit) &&
          utils::distance(u->pos(), Position(attackPos)) <=
              u->rangeAgainst(unit) + 6.0f) {
        if (u->velocity().length() < DFOASG(0.125, 0.25) ||
            u->velocity().dot(u->posf() - unit->posf()) <= 0) {
          return true;
        }
      }
    }
    return false;
  };

  if (target->type != buildtypes::Protoss_Interceptor && target->velocity().length() >= DFOASG(0.125, 0.25) &&
      target->velocity().dot(target->posf() - unit->posf()) > 0) {
    if (!willMoveIntoDanger()) {
      if (unit->topSpeed > target->topSpeed &&
          distance > std::max(range - 6.0f, 4.0f) &&
          canMoveInDirection(targetPos - myPos)) {
        return doAction(agent.moveTo(targetPos));
      }
      if (cd <= latency) {
        return pass;
      }
      if (distance > DFOASG(6, 3)) {
        return doAction(agent.moveTo(targetPos));
      } else {
        return pass;
      }
    }
  }

  float targetRange = target->rangeAgainst(unit);

  auto nearestThreat = [&]() {
    Unit* r = utils::getBestScoreCopy(agent.legalTargets, [&](Unit* u) {
      if (!u->canAttack(unit) || u->type == buildtypes::Protoss_Interceptor) return kfInfty;
      return utils::distance(u, unit) - (float)u->rangeAgainst(unit);
    }, kfInfty);
    return r ? r : target;
  };

  if (unit->type != buildtypes::Terran_Vulture) {
    Unit* threat = nearestThreat();
    if (threat && agent.targetInRange && !unit->inRangeOfPlus(threat, 8.0f)) {
      if (threat->velocity() == Vec2() || threat->velocity().dot(threat->posf() - unit->posf()) >= 0) {
        return doAction(agent.attack(target));
      }
    }
  }

  if (agent.targetInRange && target->canAttack(unit) && targetRange < range &&
      (distance <= targetRange + 12.0f || target->topSpeed >= unit->topSpeed)) {
    Unit* threat = nearestThreat();
    float threatD = utils::distanceBB(unit, nearestThreat());
    if (target->rangeAgainst(unit) < 6.0f && threat->rangeAgainst(unit) < 6.0f && threatD >= 12.0f) {
      return doAction(agent.attack(target));
    }
    float tr = 128.0f / tc::BW::data::TurnRadius[unit->type->unit];
    if (cd <= latency + tr) {
      //if (unit->type != buildtypes::Terran_Marine || target->type != buildtypes::Protoss_Zealot || utils::distanceBB(unit, nearestThreat()) > 12.0f || (unit->stimmed() && unit->cd() <= state->latencyFrames())) {
      if (unit->type != buildtypes::Terran_Marine || target->type != buildtypes::Protoss_Zealot || threatD > 12.0f || unit->stimmed()) {
        return doAction(agent.attack(target));
      }
    }
    auto kiteVector = [&]() {
      Vec2 adjustment;
      for (Unit* u : agent.legalTargets) {
        if (u->canAttack(unit) && u->type != buildtypes::Protoss_Interceptor) {
          float distance =
              std::max(utils::distanceBB(unit, u), 4.0f);
          float maxDistance = DFOASG(4 * 10, 4 * 3);
          if (distance <= maxDistance) {
            adjustment += (myPos - u->posf()) * (maxDistance / distance);
          }
        }
      }
      Vec2 moveDir = (myPos - targetPos).normalize();
      return unit->posf() +
          (moveDir + adjustment.normalize()).normalize() * 12.0;
    };
    return doAction(agent.moveTo(adjust(kiteVector())));
  } else if (
      distance <= range + 4.0f && distance > range - 4.0f &&
      targetRange > range) {
    if (cd <= latency) {
      return pass;
    }
    return doAction(agent.moveTo(targetPos));
  }
  return pass;
}

MicroAction BehaviorAsDefilerConsumeOnly::onPerform(Agent& agent) {
  auto* defiler = agent.unit;

  if (defiler->type != buildtypes::Zerg_Defiler) {
    return pass;
  }

  auto consumeValue = [&](Unit* const target) {
    if (!target->isMine)
      return -1.;
    double typeValue = 0;
    if (target->type == buildtypes::Zerg_Zergling) {
      typeValue = 1.0;
    }
    if (target->type == buildtypes::Zerg_Overlord) {
      typeValue = 0.1;
    }
    if (target->type == buildtypes::Zerg_Drone) {
      typeValue = 0.1;
    }
    if (target->type == buildtypes::Zerg_Hydralisk) {
      typeValue = 0.1;
    }
    if (typeValue <= 0) {
      return typeValue;
    }
    return typeValue / std::max(1.f, utils::distance(target, agent.unit));
  };
  auto energy = agent.unit->unit.energy;

  if (energy < 195) {
    auto upc = agent.tryCastSpellOnUnit(buildtypes::Consume, consumeValue, 0.0);
    if (upc)
      return doAction(upc);
  }

  return pass;
}

MicroAction BehaviorAsDefiler::onPerform(Agent& agent) {
  auto* defiler = agent.unit;

  if (defiler->type != buildtypes::Zerg_Defiler) {
    return pass;
  }

  // The cast range is 9, but we don't necessarily want to dive
  double range = 4 * (agent.wantsToFight ? 7 : 5);
  auto plagueValue = [&](Unit* const target) {
    if (target->plagued())
      return 0.0;
    if (target->unit.max_health <= 0)
      return 0.0;
    return target->type->subjectiveValue * target->unit.health /
        (target->unit.max_health + target->unit.max_shield) *
        (target->isEnemy ? 1.0 : -1.0) * range /
        std::max(range, double(utils::distance(defiler, target)));
  };
  auto swarmValue = [&](Unit* const target) {
    if (target->underDarkSwarm())
      return 0.0;
    if (!target->type->restrictedByDarkSwarm)
      return 0.0;

    return target->type->subjectiveValue * (target->isEnemy ? 1.0 : -1.0) *
        range / std::max(range, double(utils::distance(defiler, target)));
  };

  auto energy = agent.unit->unit.energy;
  if (energy >= 150) {
    auto upc = agent.tryCastSpellOnArea(
        buildtypes::Plague,
        16,
        16,
        plagueValue,
        3 * buildtypes::Zerg_Zergling->subjectiveValue);
    if (upc)
      return doAction(upc);
  }
  if (energy >= 100) {
    auto upc = agent.tryCastSpellOnArea(
        buildtypes::Dark_Swarm,
        24,
        24,
        swarmValue,
        3 * buildtypes::Zerg_Zergling->subjectiveValue,
        [&](auto p) {
          return p.project(defiler->pos(), agent.wantsToFight ? 16 : 8);
        });
    if (upc)
      return doAction(upc);
  }
  if (energy < 195) {
    return BehaviorAsDefilerConsumeOnly::onPerform(agent);
  }

  return pass;
}

MicroAction BehaviorAsOverlord::onPerform(Agent& agent) {
  auto& task = agent.task;
  auto unit = agent.unit;

  if (unit->type != buildtypes::Zerg_Overlord) {
    return pass;
  }

  // Stay away from other Overlords when there are no threats nearby.
  // This prevents Corsairs from murdering all our Overlords at once.
  Unit* repellant = nullptr;
  {
    auto enemies = unit->threateningEnemies;
    repellant = enemies.empty() ? nullptr : enemies[0];
    if (repellant == nullptr) {
      auto allies = unit->allyUnitsInSightRange;
      // Corsair max splash range = 100 pixels = 12.5 walktiles, then add a
      // bit of margin
      auto closestOverlordDistance = 15.0;
      std::for_each(allies.begin(), allies.end(), [&](Unit* ally) {
        if (ally != unit && ally->type == unit->type) {
          auto distance = utils::distance(unit, ally);
          if (distance < closestOverlordDistance) {
            repellant = ally;
            closestOverlordDistance = distance;
          }
        }
      });
    }
  }
  if (repellant != nullptr) {
    auto dir = Vec2(unit) - Vec2(repellant);
    dir.normalize();
    VLOG(3) << unit << " spreads away from " << repellant;
    return doAction(agent.smartMove(Position(Vec2(unit) + dir * 25)));
  }

  VLOG(3) << unit << " has no purpose in life, following the group";
  return doAction(agent.smartMove(task->center_));
}

} // namespace cherrypi
