/*
 * Copyright (c) 2017-present, Facebook, Inc.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 */

#include "base.h"

#include "src/controller.h"
#include "common/rand.h"

#include "bandit.h"

#include <bwem/map.h>

namespace cherrypi {

using namespace buildtypes;
using namespace autobuild;


class RushDefenceController : public SharedController {
 public:
  using SharedController::SharedController;

  std::vector<Unit*> units_;
  ABBOBase* bo = nullptr;

  struct UnitState {
    int lastCommand = 0;
    Position moveTo;
    Unit* attack = nullptr;
    int lastFind = 0;
    Position findPos = kInvalidPosition;
    bool mineralWalked = false;
    Unit* mineralWalkTarget = nullptr;
  };

  std::unordered_map<Unit*, UnitState> unitstates;

  bool saveForBunker = false;
  bool isFinding = false;

  bool allowBunkers = false;

  bool defend = false;

  void move(State* state, Unit* u, Position pos) {
    Unit* target = utils::getBestScoreCopy(
        state->unitsInfo().enemyUnits(),
        [&](Unit* e) {
          if (e->gone) return kfInfty;
//          if (e->pos() == pos && u->canAttack(e)) {
//            return float(e->unit.shield + e->unit.health) + 1000.0f;
//          }
          if ((e->type->isBuilding && utils::distance(pos, e->pos()) > 4) || !e->inRangeOf(u, std::max((int)u->cd() - 4, 0) + state->latencyFrames()) || !e->inRangeOfPlus(u, 3.0f)) {
            return kfInfty;
          }
          //if (e->type->isBuilding && !e->type->hasGroundWeapon && e->type != Terran_Bunker) return kfInfty;
          return float(e->unit.shield + e->unit.health);
        },
        kfInfty);
    auto& s = unitstates[u];
    if (target) {
      if (u->unit.flags & tc::Unit::Stuck) {
        Unit* mtarget = utils::getBestScoreCopy(state->unitsInfo().resourceUnits(), [&](Unit* e) {
          if (!e->visible) return kfInfty;
          if (!e->type->isMinerals) return kfInfty;
          float d = utils::distance(u->pos(), e->pos());
          if (d > 4.0f * 12) return kfInfty;
          if (d < 4.0f * 4) return kfInfty;
          if (utils::distance(target->pos(), e->pos()) > d) return kfInfty;
          return d;
        }, kfInfty);
        if (mtarget) {
          if (state->currentFrame() - s.lastCommand < 10 && s.attack == mtarget) return;
          s.moveTo = kInvalidPosition;
          s.attack = mtarget;
          s.lastCommand = state->currentFrame() + 30;
          state->board()->postCommand(
              tc::Client::Command(
                  tc::BW::Command::CommandUnit,
                  u->id,
                  tc::BW::UnitCommandType::Right_Click_Unit,
                  mtarget->id),
              SharedController::units_[u]);
          return;
        }
      }

      if (!target->inRangeOf(u) && u->type == Terran_SCV && state->resources().ore > 1) {
        Unit* rtarget = utils::getBestScoreCopy(state->unitsInfo().myUnits(), [&](Unit* e) {
          if (e->type != Terran_SCV && !e->type->isBuilding && e->type != Terran_Siege_Tank_Tank_Mode && e->type != Terran_Siege_Tank_Siege_Mode) return kfInfty;
          if (e == u || e->unit.health == e->unit.max_health) return kfInfty;
          if (!e->inRangeOf(u)) return kfInfty;
          return float(u->unit.health);
        }, kfInfty);
        if (rtarget) {
          if (state->currentFrame() - s.lastCommand < 100 && s.attack == rtarget) return;
          s.moveTo = kInvalidPosition;
          s.attack = rtarget;
          s.lastCommand = state->currentFrame();
          state->board()->postCommand(
              tc::Client::Command(
                  tc::BW::Command::CommandUnit,
                  u->id,
                  tc::BW::UnitCommandType::Repair,
                  rtarget->id),
              SharedController::units_[u]);
          return;
        }
      }

      if (state->currentFrame() - s.lastCommand < 100 && s.attack == target) return;
      s.moveTo = kInvalidPosition;
      s.attack = target;
      s.lastCommand = state->currentFrame();
      state->board()->postCommand(
          tc::Client::Command(
              tc::BW::Command::CommandUnit,
              u->id,
              tc::BW::UnitCommandType::Attack_Unit,
              target->id),
          SharedController::units_[u]);
//      float distance = utils::distanceBB(u, target);
//      float range = u->rangeAgainst(target);
//      float tr = 128.0f / tc::BW::data::TurnRadius[u->type->unit];
//      if ((distance - range) / u->topSpeed + tr >=
//          u->cd() - state->latencyFrames()) {
//        state->board()->postCommand(
//            tc::Client::Command(
//                tc::BW::Command::CommandUnit,
//                u->id,
//                tc::BW::UnitCommandType::Attack_Unit,
//                target->id),
//            SharedController::units_[u]);
//      } else {
//        pos = Position(
//            u->posf() + (u->posf() - target->posf()).normalize() * 12.0f);
//        state->board()->postCommand(
//            tc::Client::Command(
//                tc::BW::Command::CommandUnit,
//                u->id,
//                tc::BW::UnitCommandType::Move,
//                -1,
//                pos.x,
//                pos.y),
//            SharedController::units_[u]);
//      }
    } else {

      if (u->type == Terran_SCV && state->resources().ore > 1) {
        Unit* rtarget = utils::getBestScoreCopy(state->unitsInfo().myUnits(), [&](Unit* e) {
            if (e->type != Terran_SCV && !e->type->isBuilding && e->type != Terran_Siege_Tank_Tank_Mode && e->type != Terran_Siege_Tank_Siege_Mode) return kfInfty;
          if (e == u || e->unit.health == e->unit.max_health) return kfInfty;
          if (!e->inRangeOf(u)) return kfInfty;
          return float(u->unit.health);
        }, kfInfty);
        if (rtarget) {
          if (state->currentFrame() - s.lastCommand < 10 && s.attack == rtarget) return;
          s.moveTo = kInvalidPosition;
          s.attack = rtarget;
          s.lastCommand = state->currentFrame();
          state->board()->postCommand(
              tc::Client::Command(
                  tc::BW::Command::CommandUnit,
                  u->id,
                  tc::BW::UnitCommandType::Repair,
                  rtarget->id),
              SharedController::units_[u]);
          return;
        }
      }

      auto anyoneIsBuilding = [&](Unit* target) {
        for (Unit* u : state->unitsInfo().myWorkers()) {
          if (!u->unit.orders.empty() && u->unit.orders.front().targetId == target->id) {
            return true;
          }
        }
        return false;
      };

      Unit* resume = utils::getBestScoreCopy(state->unitsInfo().myBuildings(), [&](Unit* e) {
          if (!e->type->isBuilding || e->completed()) return kfInfty;
          if (anyoneIsBuilding(e)) return kfInfty;
          float d = utils::distanceBB(u, e);
          if (d > 4.0f * 6) return kfInfty;
          return d;
      }, kfInfty);
      if (resume) {
        if (state->currentFrame() - s.lastCommand < 60 && s.attack == resume) return;
        s.moveTo = kInvalidPosition;
        s.attack = resume;
        s.lastCommand = state->currentFrame();
        state->board()->postCommand(
            tc::Client::Command(
                tc::BW::Command::CommandUnit,
                u->id,
                tc::BW::UnitCommandType::Right_Click_Unit,
                resume->id),
            SharedController::units_[u]);
        return;
      }

      Unit* target = utils::getBestScoreCopy(
          state->unitsInfo().enemyUnits(),
          [&](Unit* e) {
            if (e->gone) return kfInfty;
            //if (!e->type->isBuilding) return kfInfty;
            if (e->type->isWorker || (!e->type->hasGroundWeapon && !e->type->isBuilding)) return kfInfty;
            //if (!e->type->isResourceDepot) return kfInfty;
            //if (utils::distance(e->pos(), state->areaInfo().enemyStartLocation()) > 4.0f * 8) return kfInfty;
            if (utils::distance(e->pos(), state->areaInfo().enemyStartLocation()) > 4.0f * 24) return kfInfty;
            if (utils::distanceBB(u, e) > 4.0f * 7) {
              return kfInfty;
            }
            return -float(e->unit.shield + e->unit.health);
          },
          kfInfty);

      if (target && allowBunkers) {
        const BuildType* t = buildtypes::Terran_Bunker;
        if (state->unitsInfo().myUnitsOfType(t).size() >= 2) {
          allowBunkers = false;
        }
        Position targetPos = u->pos();
        auto find = [&]() {
          if (isFinding) return kInvalidPosition;
          auto r = builderhelpers::findBuildLocation(state, {targetPos}, t, {}, [&](State* state, const BuildType* type, const Tile* tile) {
            Position pos = Position(tile) + Position(t->tileWidth * 2, t->tileHeight * 2);
            return utils::distance(pos, u->pos());
//            float r = utils::distanceBB(buildtypes::Terran_Bunker, pos, target->type, target->pos());
//            //if (nearestThreat) r = std::min(r, utils::distanceBB(buildtypes::Protoss_Photon_Cannon, pos, nearestThreat->type, nearestThreat->pos()));
//            r = std::abs(r - 4.0f * 4.0f);
//            r += utils::distance(pos, u->pos()) / 4.0f;
//            //if (nearestThreat && utils::distance(nearestThreat->pos(), pos) < utils::distance(u->pos(), pos)) r += 1600.0f;
//            for (Unit* e : state->unitsInfo().visibleUnits()) {
//              if (e != u && !e->flying() && !e->type->isBuilding) {
//                auto d = utils::distanceBB(e->type, e->pos(), t, pos);
//                if (d <= 2.0f) {
//                  r += 1600.0f;
//                }
//              }
//            }
//            for (Unit* e : state->unitsInfo().enemyUnits()) {
//              if (e->gone) continue;
//              if (utils::distanceBB(buildtypes::Terran_Bunker, pos, e->type, e->pos()) <= e->rangeAgainst(u)) r += 1600.0f;
//            }
//            return r;
          });
          isFinding = true;
          s.lastFind = state->currentFrame();
          s.findPos = r;
          return r;
        };
        Position bpos = state->currentFrame() - s.lastFind < 20 ? s.findPos : find();
        if (bpos != kInvalidPosition) {
          saveForBunker = true;
          if (state->resources().ore >= t->mineralCost - 10) {
            pos = bpos;
            if (utils::distanceBB(u->type, u->pos(), t, pos) <= 4.0f) {
              VLOG(0) << " building bunker!! ";
              state->board()->postCommand(
                  tc::Client::Command(
                      tc::BW::Command::CommandUnit,
                      u->id,
                      tc::BW::UnitCommandType::Build,
                      -1,
                      pos.x,
                      pos.y,
                      t->unit),
                  SharedController::units_[u]);
              return;
            }
          }
        }
      };

      if (state->currentFrame() - s.lastCommand < 10 && utils::distance(pos, s.moveTo) < 8.0f && utils::distance(pos, s.moveTo) < utils::distance(u->pos(), pos)) return;
      s.moveTo = pos;
      s.attack = nullptr;
      s.lastCommand = state->currentFrame();
      state->board()->postCommand(
          tc::Client::Command(
              tc::BW::Command::CommandUnit,
              u->id,
              tc::BW::UnitCommandType::Move,
              -1,
              pos.x,
              pos.y),
          SharedController::units_[u]);
    }
  }

  void repair(State* state, Unit* u, Unit* target) {
//    if (!u->unit.orders.empty() &&
//        u->unit.orders.front().type == +tc::BW::Order::Repair &&
//        u->unit.orders.front().targetId == target->id) {
//      return;
//    }
    Unit* attack = utils::getBestScoreCopy(
        state->unitsInfo().enemyUnits(),
        [&](Unit* e) {
          if (e->gone) return kfInfty;
          if (e->type->isBuilding || !e->inRangeOfPlus(u, 1.0f)) {
            return kfInfty;
          }
          if (e->type->isBuilding && !e->type->hasGroundWeapon && e->type != Terran_Bunker) return kfInfty;
          return float(e->unit.shield + e->unit.health);
        },
        kfInfty);
    auto& s = unitstates[u];
    if (attack) {
      if (state->currentFrame() - s.lastCommand < 120 && s.attack == attack) return;
      s.moveTo = kInvalidPosition;
      s.attack = attack;
      s.lastCommand = state->currentFrame();
      state->board()->postCommand(
          tc::Client::Command(
              tc::BW::Command::CommandUnit,
              u->id,
              tc::BW::UnitCommandType::Attack_Unit,
              attack->id),
          SharedController::units_[u]);
      return;
    }

    if (state->currentFrame() - s.lastCommand < 60 && s.attack == target) return;
    s.moveTo = kInvalidPosition;
    s.attack = target;
    s.lastCommand = state->currentFrame();
    state->board()->postCommand(
        tc::Client::Command(
            tc::BW::Command::CommandUnit,
            u->id,
            tc::BW::UnitCommandType::Repair,
            target->id),
        SharedController::units_[u]);
  }

  virtual void step(State* state) override {
    units_.clear();
    for (auto& v : SharedController::units_) {
      units_.push_back(v.first);
    }

    isFinding = false;

    for (Unit* u : state->unitsInfo().myBuildings()) {
      if (!u->completed()) {
        int damage = 0;
        for (Unit* e : state->unitsInfo().visibleEnemyUnits()) {
          if (u->inRangeOf(e) && e->attackingTarget == u) {
            int hp = 0;
            int shield = 0;
            e->computeDamageTo(u, &hp, &shield);
            damage += (shield + hp) * 2.0f;
          }
        }
        if (damage > 0 && damage >= u->unit.shield + u->unit.health - 10) {
          auto bwu = BWAPI::Broodwar->getUnit(u->id);
          if (bwu) bwu->cancelConstruction();
        }
      }
    }

    auto st = autobuild::getMyState(state);


    for (Unit* u : units_) {
      Unit* target = utils::getBestScoreCopy(
          state->unitsInfo().visibleEnemyUnits(),
          [&](Unit* e) {
            if (!u->canAttack(e)) return kfInfty;
            float r = state->areaInfo().walkPathLength(u->pos(), e->pos());
            if (!e->type->hasGroundWeapon && e->type != Terran_Bunker) {
              return kfInfty;
              r += 4.0f * 16;
            }
            if (!e->inRangeOfPlus(u, 2.0f) && !u->inRangeOfPlus(e, 2.0f) && state->currentFrame() - e->lastFiredWeapon > 30) r += 4.0f * 100;
            return r;
          },
          kfInfty);

      if (target) {
        move(state, u, target->pos());
      }

    }

    postUpcs(state);
  }
};


class RepairController : public SharedController {
 public:
  using SharedController::SharedController;

  std::vector<Unit*> units_;
  ABBOBase* bo = nullptr;

  struct UnitState {
    int lastCommand = 0;
    Position moveTo;
    Unit* attack = nullptr;
    int lastFind = 0;
    Position findPos = kInvalidPosition;
    bool mineralWalked = false;
    Unit* mineralWalkTarget = nullptr;

    int stuckCounter = 0;

    int lastFindExpo = 0;
    Position expoPos = kInvalidPosition;
  };

  std::unordered_map<Unit*, UnitState> unitstates;

  bool unstuck(State* state, Unit* u) {
    if (u->velocity() == Vec2()) {
      if (state->currentFrame() - u->lastUnstuck >= 45) {
        u->lastUnstuck = state->currentFrame();
        Position pos = u->pos();
        pos.x += -2 + common::Rand::sample(std::uniform_int_distribution<int>(0, 2));
        pos.y += -2 + common::Rand::sample(std::uniform_int_distribution<int>(0, 2));
        state->board()->postCommand(
            tc::Client::Command(
                tc::BW::Command::CommandUnit,
                u->id,
                tc::BW::UnitCommandType::Move,
                -1,
                pos.x,
                pos.y),
            SharedController::units_[u]);
        return true;
      }
      if (state->currentFrame() - u->lastUnstuck <= 6) {
        return true;
      }
    }
    return false;
  }

  void move(State* state, Unit* u, Position pos) {
    if (unstuck(state, u)) {
      return;
    }
    Unit* target = utils::getBestScoreCopy(
        state->unitsInfo().visibleEnemyUnits(),
        [&](Unit* e) {
          if (e->type->isBuilding || !e->inRangeOf(u, 3)) {
            return kfInfty;
          }
          return float(e->unit.shield + e->unit.health);
        },
        kfInfty);
    auto& s = unitstates[u];
    if (target) {
      float distance = utils::distanceBB(u, target);
      float range = u->rangeAgainst(target);
      float tr = 128.0f / tc::BW::data::TurnRadius[u->type->unit];
      if ((distance - range) / u->topSpeed + tr >=
          u->cd() - state->latencyFrames()) {
        if (state->currentFrame() - s.lastCommand < 100 && s.attack == target)
          return;
        s.moveTo = kInvalidPosition;
        s.attack = target;
        s.lastCommand = state->currentFrame();
        state->board()->postCommand(
            tc::Client::Command(
                tc::BW::Command::CommandUnit,
                u->id,
                tc::BW::UnitCommandType::Attack_Unit,
                target->id),
            SharedController::units_[u]);
      } else {
        pos = Position(
            u->posf() + (u->posf() - target->posf()).normalize() * 12.0f);
        if (state->currentFrame() - s.lastCommand < 10 &&
            utils::distance(pos, s.moveTo) < 8.0f &&
            utils::distance(pos, s.moveTo) < utils::distance(u->pos(), pos))
          return;
        s.moveTo = pos;
        s.attack = nullptr;
        s.lastCommand = state->currentFrame();
        state->board()->postCommand(
            tc::Client::Command(
                tc::BW::Command::CommandUnit,
                u->id,
                tc::BW::UnitCommandType::Move,
                -1,
                pos.x,
                pos.y),
            SharedController::units_[u]);
      }
    } else {
      if (state->currentFrame() - s.lastCommand < 10 &&
          utils::distance(pos, s.moveTo) < 8.0f &&
          utils::distance(pos, s.moveTo) < utils::distance(u->pos(), pos))
        return;
      s.moveTo = pos;
      s.attack = nullptr;
      s.lastCommand = state->currentFrame();
      state->board()->postCommand(
          tc::Client::Command(
              tc::BW::Command::CommandUnit,
              u->id,
              tc::BW::UnitCommandType::Move,
              -1,
              pos.x,
              pos.y),
          SharedController::units_[u]);
    }
  }

  void attack(State* state, Unit* u, Unit* target) {
    if (!u->unit.orders.empty() &&
        u->unit.orders.front().type == +tc::BW::Order::AttackUnit &&
        u->unit.orders.front().targetId == target->id) {
      Unit* target =
          state->unitsInfo().getUnit(u->unit.orders.front().targetId);
      if (target && target->inRangeOf(u)) {
        return;
      }
    }
    auto& s = unitstates[u];
    if (state->currentFrame() - s.lastCommand < 30 && s.attack == target) {
      return;
    }
    if (unstuck(state, u)) {
      return;
    }
    s.moveTo = kInvalidPosition;
    s.attack = target;
    s.lastCommand = state->currentFrame();
    state->board()->postCommand(
        tc::Client::Command(
            tc::BW::Command::CommandUnit,
            u->id,
            tc::BW::UnitCommandType::Attack_Unit,
            target->id),
        SharedController::units_[u]);
  }

  void repair(State* state, Unit* u, Unit* target) {
//    if (!u->unit.orders.empty() &&
//        u->unit.orders.front().type == +tc::BW::Order::Repair &&
//        u->unit.orders.front().targetId == target->id) {
//      return;
//    }
    if (!u->unit.orders.empty() &&
        u->unit.orders.front().type == +tc::BW::Order::Repair &&
        u->unit.orders.front().targetId == target->id) {
      Unit* target = state->unitsInfo().getUnit(u->unit.orders.front().targetId);
      if (target && utils::pxDistanceBB(u, target) <= 5) {
        return;
      }
    }
    auto& s = unitstates[u];
    if (state->currentFrame() - s.lastCommand < 30 && s.attack == target) return;
    if (unstuck(state, u)) {
      return;
    }
    s.moveTo = kInvalidPosition;
    s.attack = target;
    s.lastCommand = state->currentFrame();
    state->board()->postCommand(
        tc::Client::Command(
            tc::BW::Command::CommandUnit,
            u->id,
            target->isMine && target->type->isMechanical
                ? tc::BW::UnitCommandType::Repair
                : tc::BW::UnitCommandType::Right_Click_Unit,
            target->id),
        SharedController::units_[u]);
  }

  bool isFinding = false;
  bool isFindingExpo = false;

  bool buildTurrets = true;
  bool tryToExpand = false;

  virtual void step(State* state) override {
    units_.clear();
    for (auto& v : SharedController::units_) {
      Unit* u = v.first;
      if (!u->dead) {
        auto taskData = state->board()->taskDataWithUnit(u);
        if (taskData.owner == module_) {
          units_.push_back(v.first);
        }
      }
    }

    for (Unit* u : state->unitsInfo().myBuildings()) {
      if (!u->completed()) {
        int damage = 0;
        for (Unit* e : state->unitsInfo().visibleEnemyUnits()) {
          if (u->inRangeOf(e) && e->attackingTarget == u) {
            int hp = 0;
            int shield = 0;
            e->computeDamageTo(u, &hp, &shield);
            damage += (shield + hp) * 2.0f;
          }
        }
        if (damage > 0 && damage >= u->unit.shield + u->unit.health - 10) {
          auto bwu = BWAPI::Broodwar->getUnit(u->id);
          if (bwu) bwu->cancelConstruction();
        }
      }
    }

    auto st = autobuild::getMyState(state);

    std::unordered_map<Unit*, Position> goalPositions;

    bool attacking = state->board()->get<bool>("TacticsAttack", true);

    auto isMoveToTarget = [&](Unit* u) {
      return u->type == buildtypes::Terran_Siege_Tank_Tank_Mode ||
          u->type == buildtypes::Terran_Siege_Tank_Siege_Mode ||
          ((u->type == buildtypes::Terran_Bunker ||
          u->type == buildtypes::Terran_Missile_Turret) && (!attacking && st.frame < 24 * 60 * 7 || u->unit.health < u->unit.max_health));
    };

    std::vector<Unit*> targets;
    for (Unit* u : state->unitsInfo().myUnits()) {
       if (u->active() && !u->loaded() &&
           (u->type == buildtypes::Terran_Siege_Tank_Tank_Mode ||
            u->type == buildtypes::Terran_Siege_Tank_Siege_Mode ||
            u->type == buildtypes::Terran_Bunker ||
            u->type == buildtypes::Terran_Missile_Turret ||
            u->type == buildtypes::Terran_Battlecruiser ||
            u->type == buildtypes::Terran_Wraith ||
            u->type == buildtypes::Terran_Valkyrie ||
            u->type == buildtypes::Terran_Vulture ||
            u->type == buildtypes::Terran_Goliath ||
            u->type == buildtypes::Terran_Science_Vessel ||
            u->type == buildtypes::Terran_Dropship)) {
        targets.push_back(u);
      }
    }

    isFinding = false;

    for (Unit* u : units_) {
      auto& st = unitstates[u];
      Unit* nearestThreat = utils::getBestScoreCopy(
          state->unitsInfo().visibleEnemyUnits(),
          [&](Unit* e) {
            if (!e->type->hasGroundWeapon) {
              return kfInfty;
            }
            return state->areaInfo().walkPathLength(u->pos(), e->pos());
          },
          kfInfty);
      Unit* nearestTarget = utils::getBestScoreCopy(targets, [&](Unit* e) {
        float d = utils::distance(nearestThreat ? nearestThreat : u, e);
        float r = d < 4.0f * 6
            ? d
            : state->areaInfo().walkPathLength(u->pos(), e->pos());
        r -= (e->unit.max_health - e->unit.health) / 8.0f;
        if (state->currentFrame() < 24 * 60 * 9 && e->type == buildtypes::Terran_Bunker) {
          if (e->unit.health < e->unit.max_health) r -= 4.0f * 20;
        }
        if (!isMoveToTarget(e)) {
          if (utils::distanceBB(u, e) > 2.0f) r += 10000.0f;
        }
        return r;
      });
//      for (Unit* u : targets) {
//        if (u->type == buildtypes::Terran_Siege_Tank_Tank_Mode || u->type == buildtypes::Terran_Siege_Tank_Siege_Mode) {

//        }
//      }

      if (u->velocity().length() < 0.25f && (!nearestTarget || utils::distanceBB(u, nearestTarget) > 5)) {
        ++st.stuckCounter;
        if (st.stuckCounter >= 24) {
          st.stuckCounter = 0;
          move(state, u, u->pos() + Position(-16 + int(common::Rand::rand() % 32), -16 + int(common::Rand::rand() % 32)));
          continue;
        }
      } else {
        st.stuckCounter = 0;
      }

      Unit* escapeToRes = nullptr;
      for (Unit* a : u->beingAttackedByEnemies) {
        if (a->type == buildtypes::Protoss_Scarab) {
          Unit* res = nullptr;
          float resScore = kfInfty;
          for (Unit* r : state->unitsInfo().resourceUnits()) {
            if (r->type->isMinerals && r->visible) {
              float d = std::abs(4.0f * 24 - utils::distance(r, u));
              if (d < resScore) {
                resScore = d;
                res = r;
              }
            }
          }
          if (res) {
            escapeToRes = res;
            break;
          }
        }
      }
      if (escapeToRes) {
        repair(state, u, escapeToRes);
        continue;
      }

      if (nearestTarget) {
        //if (tryToExpand && state->resources().ore >= 400) {
        if (state->currentFrame() >= 24 * 60 * 9 && state->resources().ore >= 400) {
          if (state->currentFrame() - st.lastFindExpo >= 20 && !isFindingExpo) {
            st.lastFindExpo = state->currentFrame();
            isFindingExpo = true;
            st.expoPos = kInvalidPosition;
            float expoDistance = 4.0f * 16;
            for (auto& area : state->areaInfo().areas()) {
              for (auto& centerPos : area.baseLocations) {
                // Base locations are center-of-building -- move to top left
                // instead
                Position pos = centerPos - Position(8, 6);
                if (!builderhelpers::canBuildAt(
                        state, Terran_Command_Center, pos, true)) {
                  continue;
                }
                float d = utils::distance(u->pos(), pos);
                if (d < expoDistance) {
                  expoDistance = d;
                  st.expoPos = pos;
                }
              }
            }
          }
          if (st.expoPos != kInvalidPosition) {
            Position bpos = st.expoPos;
            bool okay = true;
            Unit* attackTarget = nullptr;
            for (Unit* a : state->unitsInfo().myUnits()) {
              if (a != u && utils::distanceBB(a->type, a->pos(), Terran_Command_Center, bpos) <= 0) {
                if (a->type == buildtypes::Terran_Vulture_Spider_Mine) {
                  attackTarget = a;
                } else {
                  okay = false;
                }
              }
            }
            if (attackTarget) {
              attack(state, u, attackTarget);
              continue;
            }
            if (okay) {
              if (utils::distanceBB(
                      u->type, u->pos(), Terran_Command_Center, bpos) <= 0.0f) {
                state->board()->postCommand(
                    tc::Client::Command(
                        tc::BW::Command::CommandUnit,
                        u->id,
                        tc::BW::UnitCommandType::Build,
                        -1,
                        bpos.x,
                        bpos.y,
                        Terran_Command_Center->unit),
                    SharedController::units_[u]);
                continue;
              } else {
                move(state, u, bpos);
                continue;
              }
            }
          }
        }
        if (buildTurrets && utils::distanceBB(u, nearestTarget) < 4.0f * 8) {
          const BuildType* t = buildtypes::Terran_Missile_Turret;
          Position targetPos = u->pos();
          auto find = [&]() {
            if (isFinding) return kInvalidPosition;
            auto r = builderhelpers::findBuildLocation(state, {targetPos}, t, {}, [&](State* state, const BuildType* type, const Tile* tile) {
              Position pos = Position(tile) + Position(t->tileWidth * 2, t->tileHeight * 2);
              return utils::distance(pos, u->pos());
            });
            isFinding = true;
            return r;
          };
          if (state->currentFrame() - st.lastFind >= 20 && !isFinding) {
            st.lastFind = state->currentFrame();
            float nDesiredTurrets = 0;
            for (Unit* e : state->unitsInfo().enemyUnits()) {
              if (!e->gone && utils::distance(e, u) <= 4.0f * 24) {
                if (e->type == buildtypes::Protoss_Carrier || u->type == buildtypes::Terran_Battlecruiser) {
                  nDesiredTurrets += 2;
                } else if (u->type == buildtypes::Protoss_Arbiter || u->type == buildtypes::Zerg_Queen) {
                  nDesiredTurrets += 1;
                } else if (u->type == buildtypes::Zerg_Mutalisk || u->type == buildtypes::Terran_Wraith || u->type == buildtypes::Protoss_Shuttle || u->type == buildtypes::Terran_Dropship) {
                  nDesiredTurrets += 0.5f;
                } else if (u->flying()) {
                  nDesiredTurrets += 0.25f;
                }
              }
            }
            for (Unit* a : state->unitsInfo().myBuildings()) {
              if (a->type == buildtypes::Terran_Missile_Turret && utils::distance(a, u) <= 4.0f * 16) {
                --nDesiredTurrets;
              }
            }
            if (nDesiredTurrets > 0) {
              st.findPos = find();
            } else {
              st.findPos = kInvalidPosition;
            }
          }
          Position bpos = st.findPos;
          if (bpos != kInvalidPosition) {
            if (state->resources().ore >= t->mineralCost - 10) {
              if (utils::distanceBB(u->type, u->pos(), t, bpos) <= 4.0f * 4) {
                state->board()->postCommand(
                    tc::Client::Command(
                        tc::BW::Command::CommandUnit,
                        u->id,
                        tc::BW::UnitCommandType::Build,
                        -1,
                        bpos.x,
                        bpos.y,
                        t->unit),
                    SharedController::units_[u]);
                continue;
              }
            }
          }
        }
        if (utils::distanceBB(u, nearestTarget) < 4.0f * 8 || true) {
          if (nearestTarget->unit.health >= nearestTarget->unit.max_health * 0.85) {
            if (nearestTarget->type == buildtypes::Terran_Bunker) {
              Unit* tank = nullptr;
              float tankDistance = kfInfty;
              for (Unit* a : state->unitsInfo().myUnitsOfType(Terran_Siege_Tank_Tank_Mode)) {
                float d = utils::distance(u->pos(), a->pos());
                if (d < tankDistance) {
                  tankDistance = d;
                  tank = a;
                }
              }
              for (Unit* a : state->unitsInfo().myUnitsOfType(Terran_Siege_Tank_Siege_Mode)) {
                float d = utils::distance(u->pos(), a->pos());
                if (d < tankDistance) {
                  tankDistance = d;
                  tank = a;
                }
              }
              if (tank && tankDistance <= 4.0f * 14) {
                Unit* tankThreat = utils::getBestScoreCopy(
                    state->unitsInfo().visibleEnemyUnits(),
                    [&](Unit* e) {
                      if (!e->type->hasGroundWeapon) {
                        return kfInfty;
                      }
                      return utils::distance(e->pos(), tank->pos());
                    },
                    kfInfty);
                if (tankThreat && u->canAttack(tankThreat) && tankThreat->detected() && (utils::distanceBB(tankThreat, nearestTarget) <= 4.0f * 4 || tank->inRangeOfPlus(tankThreat, 8.0f))) {
                  move(state, u, tankThreat->pos());
                  continue;
                }
              }
              if (nearestThreat && u->canAttack(nearestThreat) && nearestThreat->detected()) {
                Unit* cc = utils::getBestScoreCopy(state->unitsInfo().myResourceDepots(), [&](Unit* u) {
                  if (u->lifted()) {
                    return kfInfty;
                  }
                  return utils::distance(u->pos(), u->pos());
                }, kfInfty);
                if (cc && utils::distanceBB(u, cc) <= 4.0f * 16) {
                  float range = nearestThreat->rangeAgainst(nearestTarget);
                  if (range > 4.0f && utils::distanceBB(nearestThreat, nearestTarget) <= std::min(range / 2, 4.0f * 2)) {
                    move(state, u, nearestThreat->pos());
                    continue;
                  }
                  if (state->areaInfo().walkPathLength(nearestThreat->pos(), cc->pos()) < state->areaInfo().walkPathLength(nearestTarget->pos(), cc->pos())) {
                    move(state, u, nearestThreat->pos());
                    continue;
                  }
                }
                if (utils::distance(u->pos(), state->areaInfo().myStartLocation()) < utils::distance(u->pos(), state->areaInfo().enemyStartLocation()) / 2) {
                  if (state->areaInfo().walkPathLength(nearestThreat->pos(), state->areaInfo().myStartLocation()) < state->areaInfo().walkPathLength(nearestTarget->pos(), state->areaInfo().myStartLocation())) {
                    move(state, u, nearestThreat->pos());
                    continue;
                  }
                }
              }
            }
          }
        }
        if (utils::distanceBB(u, nearestTarget) < (nearestTarget->type->isBuilding ? 4 : 8)) {
          if (nearestTarget->unit.health >= nearestTarget->unit.max_health &&
              state->currentFrame() - nearestTarget->lastFiredWeapon >= 60) {
            if (nearestThreat && u->inRangeOfPlus(nearestThreat, 4.0f * 2)) {
              move(
                  state,
                  u,
                  Position(
                      u->posf() +
                      (u->posf() - nearestThreat->posf()).normalize() * 6.0f));
            } else {
              move(
                  state,
                  u,
                  Position(
                      u->posf() +
                      (u->posf() - nearestTarget->posf()).normalize() * 6.0f));
            }
          } else {
            if (utils::distanceBB(u, nearestTarget) < 3.0f) repair(state, u, nearestTarget);
            else move(state, u, nearestTarget->pos());
          }
        } else {
          move(state, u, nearestTarget->pos());
        }
      }
    }

    //    Unit* buildingTarget = nullptr;
    //    for (Unit* u : state->unitsInfo().enemyUnits()) {
    //      if (!u->gone && u->type->isResourceDepot && !u->completed()) {
    //        buildingTarget = u;
    //      }
    //    }

    //    for (Unit* u : units_) {
    //      for (Position pos : state->areaInfo().walkPath(u, goalPositions[u]))
    //      {
    //      }
    //    5

    postUpcs(state);
  }
};


class ABBOtcheese : public ABBOBase {
 public:
  using ABBOBase::ABBOBase;

  auto buildN(const BuildType* type, int n) {
    return AutoBuildTask::buildN(type, n);
  }
  auto buildN(const BuildType* type, int n, int simultaneous) {
    return AutoBuildTask::buildN(type, n, simultaneous);
  }
  auto buildN(const BuildType* type, int n, Position at) {
    return AutoBuildTask::buildN(type, n, at);
  }
  auto build(const BuildType* type, Position at) {
    return AutoBuildTask::build(type, at);
  }
  auto build(const BuildType* type) {
    return AutoBuildTask::build(type);
  }
  bool upgrade(const BuildType* type) {
    return AutoBuildTask::upgrade(type);
  }

  std::shared_ptr<RushDefenceController> controller;

  std::shared_ptr<RepairController> repaircontroller;

  bool attacking = false;
  bool wasRushed = false;

  bool rushed = false;
  bool earlyRushDefence = false;
  bool enemyVeryFastExpanded = false;
  bool maybeSneakyDrops = false;

  bool defendingMainBase = false;

  bool waitedForLarva = false;
  bool waitForLarva = false;

  Position proxyLocation = kInvalidPosition;
  Position sunkenLocation = kInvalidPosition;
  bool hasPulled = false;

  Position baseTurretPos = kInvalidPosition;
  Position antiMutaTurretPos = kInvalidPosition;
  Position baseBunkerPos = kInvalidPosition;
  Position nextBaseBunkerPos = kInvalidPosition;

  Position nextTurretPos = kInvalidPosition;

  Position defaultTurretPos = kInvalidPosition;

  Position firstBarracksPos = kInvalidPosition;
  Position firstFactoryPos = kInvalidPosition;

  Position proxyBunkerPos = kInvalidPosition;

  int lastFindBuildingPositions = 0;

  int lastFindTurretPositions1 = 0;
  int lastFindTurretPositions2 = 0;
  int lastFindBunkerPositions1 = 0;
  int lastFindBunkerPositions2 = 0;

  int scans = 0;

  bool haveBunkerAtHome = false;
  bool haveProxiedBunker = false;
  bool haveBunkerElsewhere = false;
  bool dontScout = false;
  bool hasPulledWorkers = false;

  bool cancelExpo = false;
  bool counterAttackingVultures = false;
  
  int inProdTurrets = 0;

  bool maybeNeedToRepair = false;
  bool weakRush = false;

  bool beingProxied = false;

  bool dirtyWorkerRush = false;

  bool cMassVulture = false;
  bool cGoBio = false;
  bool cTwoRax = false;
  bool cSiegeExpand = false;

  bool inited = false;

  virtual void preBuild2(autobuild::BuildState& st) override {
    using namespace buildtypes;
    using namespace autobuild;

    if (!inited) {
      inited = true;

      auto opening = state_->bandit().choose({"1rax fe", "mass vulture", "siege expand", "bio", "2rax mech", "2rax bio", "dirty worker rush"});

      if (opening == "1rax fe") {
      } if (opening == "mass vulture") {
        cMassVulture = true;
      } else if (opening == "siege expand") {
        cSiegeExpand = true;
      } else if (opening == "bio") {
        cGoBio = true;
      } else if (opening == "2rax mech") {
        cTwoRax = true;
        //cGoBio = true;
      } else if (opening == "2rax bio") {
        cTwoRax = true;
        cGoBio = true;
      } else if (opening == "dirty worker rush") {
        dirtyWorkerRush = true;
      }
    }

    if (!isSimulation && false) {
      if (!controller) {
        controller = std::make_shared<RushDefenceController>(module_);
        controller->bo = this;
      }

      //controller->allowBunkers = !scvRush;
      //controller->allowBunkers = false;

      if (st.frame < 24 * 60 * 6 && enemyArmySupplyInOurBase > armySupply) {
        int nRepairers =
            std::min((int)state_->unitsInfo().myWorkers().size() / 2, 5);
        int nToRemove = (int)controller->units_.size() - nRepairers;
        for (auto& task : state_->board()->tasksOfModule(module_)) {
          if (nToRemove <= 0) {
            break;
          }
          auto sct = std::dynamic_pointer_cast<SharedControllerTask>(task);
          if (sct && sct->controller() == controller) {
            task->cancel(state_);
            --nToRemove;
          }
        }
        while ((int)controller->units_.size() < nRepairers) {
          Unit* scv = utils::getBestScoreCopy(
              state_->unitsInfo().myWorkers(),
              [&](Unit* u) {
                if (u->type != buildtypes::Terran_SCV) {
                  return kfInfty;
                }
                //if (whitelist.find(u) != whitelist.end()) return kfInfty;
                auto taskData = state_->board()->taskDataWithUnit(u);
                if (taskData.task && taskData.owner &&
                    taskData.owner->name().find("Gatherer") ==
                        std::string::npos) {
                  return kfInfty;
                }
                if (u->unit.orders.front().type != tc::BW::Order::Move &&
                    u->unit.orders.front().type != tc::BW::Order::MoveToMinerals &&
                    u->unit.orders.front().type != tc::BW::Order::WaitForMinerals &&
                    !u->idle()) {
                  return kfInfty;
                }
                return state_->areaInfo().walkPathLength(
                    u->pos(), enemyBasePos);
              },
              kfInfty);

          if (scv) {
            auto id = state_->board()->postUPC(
                std::make_shared<UPCTuple>(), upcId(), module_);
            state_->board()->consumeUPC(id, module_);
            auto task = std::make_shared<SharedControllerTask>(
                id, std::unordered_set<Unit*>{scv}, state_, controller);
            state_->board()->postTask(task, module_, true);
          } else break;
        }
      } else {
        for (auto& task : state_->board()->tasksOfModule(module_)) {
          auto sct = std::dynamic_pointer_cast<SharedControllerTask>(task);
          if (sct && sct->controller() == controller) {
            task->cancel(state_);
          }
        }
      }
    }

    int tankCount = countPlusProduction(st, Terran_Siege_Tank_Tank_Mode) +
        countPlusProduction(st, Terran_Siege_Tank_Siege_Mode);

    if (!isSimulation) {
      if (!repaircontroller) {
        repaircontroller = std::make_shared<RepairController>(module_);
        repaircontroller->bo = this;
      }

      repaircontroller->tryToExpand = weArePlanningExpansion;

      // if (tankCount && tankCount < 5 && st.frame < 24 * 60 * 8) {
      //if (tankCount) {
      if (tankCount + countPlusProduction(st, Terran_Bunker) || bases >= 2) {
        int nRepairers =
            std::min((int)state_->unitsInfo().myWorkers().size() / 10, 4);
        if (enemyAttackingArmySupply >= armySupply / 2) {
          int nGoons = 0;
          for (const Unit* u : state_->unitsInfo().visibleEnemyUnits()) {
            if (u->type == Protoss_Dragoon && utils::distance(u, state_->areaInfo().myStartLocation()) < utils::distance(u, state_->areaInfo().enemyStartLocation())) {
              ++nGoons;
            }
          }
          nRepairers = std::max(nRepairers, std::max(4, std::min(nGoons, 8)));
        }
        if (!attacking || st.frame < 24 * 60 * 9 || state_->unitsInfo().myWorkers().size() < 30) {
          bool anyRepairableTanks = false;
          bool repairableAtHome = false;
          for (Unit* u : state_->unitsInfo().myUnits()) {
            if (anyRepairableTanks) {
              break;
            }
            if (u->type->isBuilding && utils::distance(u->pos(), homePosition) > utils::distance(u->pos(), enemyBasePos)) {
              continue;
            }
            if (u->type == buildtypes::Terran_Bunker && armySupply < 10 && bases >= 2 && state_->unitsInfo().myCompletedUnitsOfType(Terran_Command_Center).size() == 1) {
              if (!earlyRushDefence) {
                anyRepairableTanks = true;
              }
            }
            if (u->type == buildtypes::Terran_Siege_Tank_Tank_Mode ||
                u->type == buildtypes::Terran_Siege_Tank_Siege_Mode ||
                u->type == buildtypes::Terran_Bunker ||
                u->type == buildtypes::Terran_Missile_Turret ||
                u->type == buildtypes::Terran_Wraith ||
                u->type == buildtypes::Terran_Science_Vessel ||
                u->type == buildtypes::Terran_Valkyrie) {
              if (u->unit.health < u->unit.max_health) {
                anyRepairableTanks = true;
                if (utils::distance(u->pos(), homePosition) <= 4.0f * 8) {
                  repairableAtHome = true;
                }
                break;
              }
            }
            if (u->type == buildtypes::Terran_Siege_Tank_Tank_Mode ||
                u->type == buildtypes::Terran_Siege_Tank_Siege_Mode || u->type == buildtypes::Terran_Bunker) {
              if (!defendingMainBase) {
                for (Unit* e : state_->unitsInfo().visibleEnemyUnits()) {
                  if (utils::distance(u, e) <= 4.0f * 13) {
                    anyRepairableTanks = true;
                    break;
                  }
                }
                for (Unit* e : state_->unitsInfo().visibleEnemyUnits()) {
                  if (e->type == buildtypes::Protoss_Zealot || e->type == buildtypes::Zerg_Zergling) {
                    if (state_->areaInfo().walkPathLength(e->pos(), state_->areaInfo().myStartLocation()) <= state_->areaInfo().walkPathLength(u->pos(), state_->areaInfo().myStartLocation())) {
                      anyRepairableTanks = false;
                      break;
                    }
                  }
                }
              }
            }
          }
          if (!anyRepairableTanks) {
            nRepairers = 0;
          }
          if (earlyRushDefence && haveBunkerAtHome && !repairableAtHome && bases == 1) {
            nRepairers = 0;
          }
        }
//        bool anyCombatWorkers = false;
//        for (Unit* u : state_->unitsInfo().myWorkers()) {
//          if (u->combatWorker) {
//            anyCombatWorkers = true;
//            break;
//          }
//        }
//        if (anyCombatWorkers) {
//          nRepairers = 0;
//        }
//        if (state_->currentFrame() < 24 * 60 * 6 && defendingMainBase) {
//          nRepairers = 0;
//        }
        nRepairers = std::max(std::min(nRepairers, ((int)state_->unitsInfo().myWorkers().size() - 4) / 4), 0);
        int nToRemove = (int)repaircontroller->units_.size() - nRepairers;
        for (auto& task : state_->board()->tasksOfModule(module_)) {
          if (nToRemove <= 0) {
            break;
          }
          auto sct = std::dynamic_pointer_cast<SharedControllerTask>(task);
          if (sct && sct->controller() == repaircontroller) {
            task->cancel(state_);
            --nToRemove;
          }
        }
        if ((int)repaircontroller->units_.size() < nRepairers) {
          Unit* scv = utils::getBestScoreCopy(
              state_->unitsInfo().myWorkers(),
              [&](Unit* u) {
                if (u->type != buildtypes::Terran_SCV) {
                  return kfInfty;
                }
                if (!u->unit.orders.empty() && u->unit.orders.front().type == +tc::BW::Order::ConstructingBuilding) {
                  return kfInfty;
                }
                auto taskData = state_->board()->taskDataWithUnit(u);
                if (taskData.task && taskData.owner &&
                    taskData.owner->name().find("Gatherer") ==
                        std::string::npos) {
                  return kfInfty;
                }
                return state_->areaInfo().walkPathLength(
                    u->pos(), enemyBasePos);
              },
              kfInfty);

          if (scv) {
            auto id = state_->board()->postUPC(
                std::make_shared<UPCTuple>(), upcId(), module_);
            state_->board()->consumeUPC(id, module_);
            auto task = std::make_shared<SharedControllerTask>(
                id, std::unordered_set<Unit*>{scv}, state_, repaircontroller);
            state_->board()->postTask(task, module_, true);
          }
        }
      } else {
        for (auto& task : state_->board()->tasksOfModule(module_)) {
          auto sct = std::dynamic_pointer_cast<SharedControllerTask>(task);
          if (sct && sct->controller() == repaircontroller) {
            task->cancel(state_);
          }
        }
      }
    }

    if (state_->areaInfo().numMyBases() == 1 && enemyDragoonCount < 2) {
      counterAttackingVultures = true;
      for (Unit* u : state_->unitsInfo().myUnitsOfType(Terran_Vulture)) {
        u->allowAggressive = true;
      }
    } else if (counterAttackingVultures) {
      counterAttackingVultures = false;
      for (Unit* u : state_->unitsInfo().myUnitsOfType(Terran_Vulture)) {
        u->allowAggressive = false;
      }
    }

    if (state_->unitsInfo().myCompletedUnitsOfType(Zerg_Lurker).size() >= 4) {
      attacking = true;
    }

    bool attack = false;

    if (true) {
      //attack = weArePlanningExpansion || bases >= 3 || armySupply >= std::max(std::min(enemyArmySupply * 2, enemyArmySupply + 16.0f), enemyArmySupply + 8.0f);
      double es = enemyRace == +tc::BW::Race::Zerg ? enemyArmySupply : enemyGroundArmySupply;
      attack = bases >= 5 || armySupply >= std::max(std::min(es * 2, es + 16.0f), es + 8.0f);
      if (armySupply < 30.0 && st.frame < 24 * 60 * 6) {
        attack = false;
      }
      if (rushed && armySupply < enemyArmySupply * 2 && st.frame < 24 * 60 * 8) {
        attack = false;
      }
      if (enemyRace == +tc::BW::Race::Zerg && st.frame < 24 * 60 * 6) {
        if (!state_->unitsInfo().myCompletedUnitsOfType(Terran_Wraith).empty() && armySupply > enemyArmySupply && enemyMutaliskCount == 0) {
          attack = true;
        }
      }
      if (enemyRace == +tc::BW::Race::Terran) {
        attack = st.frame >= 24 * 60 * 7;
      }
      if (st.frame < 24 * 60 * 9 && maybeSneakyDrops) {
        attack = false;
      }
      //if (enemyRace == +tc::BW::Race::Protoss && armySupply < 20.0 && armySupply < enemyArmySupply * 1.5) attack = false;
      if (enemyRace == +tc::BW::Race::Protoss && armySupply < 40.0) attack = false;

      if (earlyRushDefence && !state_->unitsInfo().myUnitsOfType(Terran_Vulture).empty()) {
        attack = true;
      }

//      if (altattacklogic) {
//        if (!weArePlanningExpansion && armySupply < 40.0) {
//          attack = false;
//        }
//      }

      if (enemyArbiterCount || enemyCloakedUnitCount - enemyObserverCount >= 3) {
        if (countUnits(st, Terran_Science_Vessel) == 0 && armySupply < enemyArmySupply * 2) {
          attack = false;
        }
      }
      if (enemyDarkTemplarCount && countUnits(st, Terran_Science_Vessel) == 0 && scans == 0) {
        attack = false;
      }

      //if (attackearly && (tankCount >= 2 || armySupply > enemyArmySupply)) attack = true;

      //attack = armySupply >= 100.0;

      if (armySupply >= 20 && tankCount && countPlusProduction(st, Terran_Vulture) > countPlusProduction(st, Terran_Siege_Tank_Tank_Mode) * 4) {
        attack = true;
      }

      attacking = attack;
    }

    //attacking = true;

    postBlackboardKey("TacticsAttack", attacking);
    //postBlackboardKey(Blackboard::kMinScoutFrameKey, 24 * 60 * 2);
    postBlackboardKey(Blackboard::kMinScoutFrameKey, 24 * 60 * 60);

    if (st.frame >= 24 * 60 * 3 && st.workers < 10) {
      setScouts(0);
    } else if (dirtyWorkerRush) {
      if (st.frame >= 24 * 30 && state_->areaInfo().candidateEnemyStartLocations().size() != 1) {
        if (state_->areaInfo().candidateEnemyStartLocations().size() >= 2) {
          setScouts(2);
        } else {
          setScouts(1);
        }
      } else {
        setScouts(0);
      }
    } else if (st.frame >= 24 * 60 * 1 + 24 * 30) {
      if (dontScout || earlyRushDefence || enemyProxyGatewayCount) {
        setScouts(0);
      } else {
        if (enemyResourceDepots) {
          setScouts(1);
        } else {
          if (st.frame >= 24 * 60 * 2) {
            setScouts(state_->areaInfo().candidateEnemyStartLocations().size() > 1 ? 2 : 1);
          } else {
            setScouts(1);
          }
        }
      }
    }

    postBlackboardKey("dontBuildNear", naturalDefencePosDirection);

    //postBlackboardKey("TacticsRushBase", true);

    if (st.frame < 24 * 60 * 6 &&
        (enemyAttackingArmySupply > 0 && enemyArmySupply + enemyAttackingWorkerCount / 2 >
            std::max(armySupply, 3.0)) || (countUnits(st, Terran_Barracks) == 0 && enemyZerglingCount) || (st.frame < 24 * 60 * 3 && enemyZerglingCount)) {
      rushed = true;

      earlyRushDefence = enemyArmySupply == enemyZealotCount * 2 + enemyZerglingCount * 0.5 + enemyMarineCount;
    } else if ((st.frame >= 24 * 60 * 6 && enemyDragoonCount) || st.frame >= 24 * 60 * 9) {
      earlyRushDefence = false;
    }
    if (st.frame < 24 * 60 * 4 && enemyHasExpanded) {
      enemyVeryFastExpanded = true;
    }

    if (enemyCloakedUnitCount && enemyReaverCount + enemyShuttleCount) {
      maybeSneakyDrops = true;
    }

    if (st.frame < 24 * 60 * 4 && enemyAttackingArmySupply >= 4) {
      wasRushed = true;
    }

    defendingMainBase = false;
    for (Unit* u : state_->unitsInfo().visibleEnemyUnits()) {
      if (u->type->hasGroundWeapon && !u->type->isWorker) {
        if (utils::distance(u->pos(), state_->areaInfo().myStartLocation()) < 4.0f * 8) {
          defendingMainBase = true;
        }
      }
    }

    if (state_->currentFrame() -
            state_->board()->get<FrameNum>("last_scan", 0) >
        30) {
      Unit* comsat = nullptr;
      int comsatEnergy = 0;
      int scans = 0;
      for (Unit* u : state_->unitsInfo().myCompletedUnitsOfType(
               buildtypes::Terran_Comsat_Station)) {
        if (u->unit.energy >= 50) {
          ++scans;
        }
        if (u->unit.energy > comsatEnergy) {
          comsatEnergy = u->unit.energy;
          comsat = u;
        }
      }
      if (comsat && comsatEnergy >= 50) {
        float targetScore = 0.0f;
        Unit* target = nullptr;
        if (scans >= 5 || comsatEnergy >= 162 ||
            (scans >= (maybeSneakyDrops ? 4 : 2) && attacking)) {
          for (Unit* u : state_->unitsInfo().enemyUnits()) {
            if (u->gone || u->type->isBuilding || u->type->isWorker) {
              continue;
            }
            float score = 0.0f;
            for (Unit* e : state_->unitsInfo().enemyUnits()) {
              if (!e->gone && state_->currentFrame() - e->lastSeen >= 90 &&
                  (e->type->hasGroundWeapon || e->type->hasAirWeapon) &&
                  utils::distance(u->pos(), e->pos()) <= 4.0f * 6) {
                Tile* t = state_->tilesInfo().tryGetTile(e->x, e->y);
                if (t && !t->visible) {
                  score += e->type->subjectiveValue;
                }
              }
            }
            if (score >= 150) {
              if (score > targetScore) {
                targetScore = score;
                target = u;
              }
            }
          }
        }
        if (target) {
          state_->board()->post("last_scan", state_->currentFrame());
          state_->board()->postCommand(
              tc::Client::Command(
                  tc::BW::Command::CommandUnit,
                  comsat->id,
                  tc::BW::UnitCommandType::Use_Tech_Position,
                  -1,
                  target->x,
                  target->y,
                  buildtypes::Scanner_Sweep->tech),
              kRootUpcId);
        }
      }
    }
    scans = 0;
    for (Unit* u : state_->unitsInfo().myCompletedUnitsOfType(buildtypes::Terran_Comsat_Station)) {
      if (u->unit.energy >= 50) {
        ++scans;
      }
    }

    if (state_->currentFrame() - lastFindTurretPositions1 >= 90) {
      lastFindTurretPositions1 = state_->currentFrame();

      defaultTurretPos = getStaticDefencePos(state_, Terran_Missile_Turret);
      for (Unit* u : state_->unitsInfo().myUnitsOfType(Terran_Missile_Turret)) {
        if (utils::distance(u->pos(), defaultTurretPos) <= 4.0f * 5) {
          defaultTurretPos = kInvalidPosition;
          break;
        }
      }

      nextTurretPos = kInvalidPosition;
      float nextTurretPosScore = -1.0f;
      for (int i = state_->areaInfo().numMyBases(); i != 0;) {
        --i;
        Unit* depot = state_->areaInfo().myBase(i)->resourceDepot;
        if (depot) {
          Position pos = findSunkenPosNear(Terran_Missile_Turret, depot->pos(), true, Terran_Missile_Turret);
          if (pos != kInvalidPosition) {
            float score = 0.0f;
            for (Unit* u :  state_->unitsInfo().myBuildings()) {
              if (u->type != Terran_Missile_Turret) {
                continue;
              }
              float d = std::min(utils::distance(u, pos), utils::distanceBB(depot, u));
              score += d * d;
            }
            if (score > nextTurretPosScore) {
              nextTurretPosScore = score;
              nextTurretPos = pos;
            }
          }
        }
      }
    }
    if (state_->currentFrame() - lastFindTurretPositions2 + 5 >= 90) {
      lastFindTurretPositions2 = state_->currentFrame();

      baseTurretPos = kInvalidPosition;
      for (int i = state_->areaInfo().numMyBases(); i != 0;) {
        --i;
        Unit* depot = state_->areaInfo().myBase(i)->resourceDepot;
        if (depot) {
          baseTurretPos = findSunkenPosNear(Terran_Missile_Turret, depot->pos(), true, Terran_Missile_Turret);
          if (baseTurretPos != kInvalidPosition) {
            Unit* sunken = utils::getBestScoreCopy(
                state_->unitsInfo().myBuildings(),
                [&](Unit* u) {
                  if (u->type != Terran_Missile_Turret) {
                    return kfInfty;
                  }
                  float d = std::min(utils::distance(u, baseTurretPos), utils::distanceBB(depot, u));
                  if (d > 4 * 6) {
                    return kfInfty;
                  }
                  return d;
                },
                kfInfty);
            if (sunken) {
              baseTurretPos = kInvalidPosition;
            } else {
              break;
            }
          }
        }
      }

      if (enemyRace == +tc::BW::Race::Zerg) {
        antiMutaTurretPos = kInvalidPosition;
        int bestN = 100;
        for (int i = state_->areaInfo().numMyBases(); i != 0;) {
          --i;
          Unit* depot = state_->areaInfo().myBase(i)->resourceDepot;
          if (depot) {
            Position pos = findSunkenPosNear(Terran_Missile_Turret, depot->pos(), false, Terran_Missile_Turret, false, true);
            if (pos != kInvalidPosition) {
              int n = 0;
              Unit* sunken = utils::getBestScoreCopy(
                  state_->unitsInfo().myBuildings(),
                  [&](Unit* u) {
                    if (u->type != Terran_Missile_Turret) {
                      return kfInfty;
                    }
                    float d = std::min(utils::distance(u, baseTurretPos), utils::distanceBB(depot, u));
                    if (d > 4 * 6) {
                      return kfInfty;
                    }
                    ++n;
                    return d;
                  },
                  kfInfty);
              if (n < bestN) {
                bestN = n;
                antiMutaTurretPos = pos;
              }
            }
          }
        }
      }
    }
    if (state_->currentFrame() - lastFindBunkerPositions1 + 10 >= 90) {
      lastFindBunkerPositions1 = state_->currentFrame();

      baseBunkerPos = kInvalidPosition;
      auto findBaseBunkerPosNear = [&](Position near) {
        Position pos = findSunkenPosNear(Terran_Bunker, near, true, Terran_Bunker);
        if (pos != kInvalidPosition) {
          Unit* sunken = utils::getBestScoreCopy(
              state_->unitsInfo().myBuildings(),
              [&](Unit* u) {
                if (u->type != Terran_Bunker) {
                  return kfInfty;
                }
                float d = std::min(utils::distance(u, pos), utils::distance(near, u->pos()));
                if (d > 4 * 9) {
                  return kfInfty;
                }
                return d;
              },
              kfInfty);
          if (sunken || utils::distance(near, pos) > 4 * 8) {
            pos = kInvalidPosition;
          }
        }
        return pos;
      };
      nextBaseBunkerPos = findBaseBunkerPosNear(nextBase);
      for (int i = state_->areaInfo().numMyBases(); i != 0;) {
        --i;
        Unit* depot = state_->areaInfo().myBase(i)->resourceDepot;
        if (depot) {
          baseBunkerPos = findBaseBunkerPosNear(depot->pos());
          if (baseBunkerPos != kInvalidPosition) break;
        }
      }
    }

    auto backPos = [&](const BuildType* type) {
      auto& tilesInfo = state_->tilesInfo();
      auto copy = tilesInfo.tiles;
      size_t stride = TilesInfo::tilesWidth - tilesInfo.mapTileWidth();
      Tile* ptr = tilesInfo.tiles.data();
      for (unsigned tileY = 0; tileY != tilesInfo.mapTileHeight();
           ++tileY, ptr += stride) {
        for (unsigned tileX = 0; tileX != tilesInfo.mapTileWidth();
             ++tileX, ++ptr) {
          ptr->reservedAsUnbuildable = false;
        }
      }
      float maxd = state_->areaInfo().walkPathLength(homePosition, enemyBasePos);
      Position r = builderhelpers::findBuildLocation(
          state_,
          {homePosition},
          type,
          {},
          [&](State* state, const BuildType* type, const Tile* tile) {
            Position pos = Position(tile) + Position(4, 4);
            float r = 0.0f;
            r -= std::min(state->areaInfo().walkPathLength(pos, enemyBasePos), maxd + 4.0f * 8);
            r += std::max(utils::distance(pos, homePosition) - 4.0f * 6, 0.0f);
            return r;
          });
      state_->tilesInfo().tiles = copy;
      return r;
    };

    firstBarracksPos = kInvalidPosition;
    if (state_->unitsInfo().myUnitsOfType(Terran_Barracks).empty()) firstBarracksPos = backPos(Terran_Barracks);
    firstFactoryPos = kInvalidPosition;
    if (state_->unitsInfo().myUnitsOfType(Terran_Factory).empty()) firstFactoryPos = backPos(Terran_Factory);

    if (state_->unitsInfo().myUnitsOfType(Terran_Factory).empty() && armySupply < 8) {
      if (st.gas < 100 && st.gas < st.workers * 62.5 && (enemyArmySupplyInOurBase == 0 || st.minerals >= 150 || st.workers >= 16)) {
        state_->board()->post(Blackboard::kGathererMinGasWorkers, 60);
        state_->board()->post(Blackboard::kGathererMaxGasWorkers, 60);
      } else {
        state_->board()->post(Blackboard::kGathererMinGasWorkers, 0);
        state_->board()->post(Blackboard::kGathererMaxGasWorkers, 0);
      }
    }

    haveBunkerAtHome = false;
    haveProxiedBunker = false;
    haveBunkerElsewhere = false;
    for (Unit* u : state_->unitsInfo().myBuildings()) {
      if (u->type == buildtypes::Terran_Bunker) {
        float homed = utils::distance(u->pos(), homePosition);
        if (homed <= 4.0f * 8) {
          haveBunkerAtHome = true;
        } else if (
            utils::distance(u->pos(), state_->areaInfo().enemyStartLocation()) <
            homed) {
          haveProxiedBunker = true;
        }
      }
    }
    if (!state_->unitsInfo().myUnitsOfType(Terran_Bunker).empty()) {
      if (!haveBunkerAtHome && !haveProxiedBunker) {
        haveBunkerElsewhere = true;
      }
    }

    if (st.frame <= 24 * 60 * 2) {
      if (enemyResourceDepots && enemyPylonCount == 0 && enemyGatewayCount == 0 && enemyGasUnits == 0) {
        beingProxied = true;
      } else if (enemyGatewayCount) {
        beingProxied = false;
      }
      if (enemyProxyGatewayCount + enemyProxyBarracksCount) {
        beingProxied = true;
      }
    }
    
    auto findProxyPos = [&](const BuildType* type, Position targetpos, const BuildType* targettype) {
      auto& tilesInfo = state_->tilesInfo();
      auto copy = tilesInfo.tiles;
      size_t stride = TilesInfo::tilesWidth - tilesInfo.mapTileWidth();
      Tile* ptr = tilesInfo.tiles.data();
      for (unsigned tileY = 0; tileY != tilesInfo.mapTileHeight();
           ++tileY, ptr += stride) {
        for (unsigned tileX = 0; tileX != tilesInfo.mapTileWidth();
             ++tileX, ++ptr) {
          ptr->reservedAsUnbuildable = false;
        }
      }
      float maxd = state_->areaInfo().walkPathLength(homePosition, enemyBasePos);
      Position r = builderhelpers::findBuildLocation(
          state_,
          {targetpos},
          type,
          {},
          [&](State* state, const BuildType* type, const Tile* tile) {
            Position pos = Position(tile) + Position(4, 4);
            float r = 0.0f;
            r += state->areaInfo().walkPathLength(pos, targetpos);
            r += state->areaInfo().walkPathLength(pos, homePosition) / 40;
            //r -= std::min(state->areaInfo().walkPathLength(pos, enemyBasePos), maxd + 4.0f * 8);
            //r += std::max(utils::distance(pos, homePosition) - 4.0f * 6, 0.0f);
            if (utils::distanceBB(targettype, targetpos, type, pos) <= 4.0f * 4) {
              r += 4.0f * 8;
            }
            return r;
          });
      state_->tilesInfo().tiles = copy;
      return r;
    };

    auto isValidCombatWorker = [&](Unit* u) {
      auto taskData = state_->board()->taskDataWithUnit(u);
      if (taskData.task && taskData.owner &&
          taskData.owner->name().find("Gatherer") ==
              std::string::npos) {
        return false;
      }
      if (u->unit.orders.front().type != tc::BW::Order::Move &&
          u->unit.orders.front().type != tc::BW::Order::MoveToMinerals &&
          u->unit.orders.front().type != tc::BW::Order::WaitForMinerals &&
          !u->idle()) {
        return false;
      }
      return true;
    };

    bool isProxying = proxyBunkerPos != kInvalidPosition;
    proxyBunkerPos = kInvalidPosition;
    bool rush = false;
    if (enemyRace != +tc::BW::Race::Zerg && enemyVeryFastExpanded &&
        st.frame < 24 * 60 * 7 && enemyAttackingArmySupply == 0 && enemyAttackingWorkerCount < 3) {
      rush = true;
    }
//    if (enemyGasUnits == 0) {
//      rush = false;
//    }
    if (has(st, Terran_Barracks) &&
        state_->areaInfo().candidateEnemyStartLocations().size() == 1 &&
        armySupply >= enemyArmySupply - 2 && false) {
      if (enemyArmySupply == 0 && (enemyCyberneticsCoreCount || enemyGasUnits) && st.frame >= 24 * 60 * 2 + 24 * 30) {
        rush = true;
        //weakRush = true;
      }
    }
    if (!dirtyWorkerRush && rush && enemyStaticDefenceCount == 0) {
      if (dontScout) {
        Unit* target = nullptr;
        for (Unit* u : state_->unitsInfo().enemyUnits()) {
          if (u->gone) continue;
          if (u->type->isResourceDepot && utils::distance(u->pos(), state_->areaInfo().enemyStartLocation()) > 4.0f * 8) {
            target = u;
            break;
          }
        }
        if (!target && false) {
          target = utils::getBestScoreCopy(
              state_->unitsInfo().enemyBuildings(),
              [&](Unit* e) {
                if (e->gone) return kfInfty;
                return state_->areaInfo().walkPathLength(state_->areaInfo().myStartLocation(), e->pos());
              },
              kfInfty);
        }
        if (target) {
          proxyBunkerPos = findProxyPos(Terran_Bunker, target->pos(), target->type);
        } else {
          Position lpos = homePosition;
          Position llpos = homePosition;
          for (Position pos : state_->areaInfo().walkPath(homePosition, enemyBasePos)) {
            llpos = lpos;
            lpos = pos;
          }
          proxyBunkerPos = findProxyPos(Terran_Bunker, lpos, Protoss_Nexus);
          if (proxyBunkerPos == kInvalidPosition) {
            proxyBunkerPos = findProxyPos(Terran_Bunker, llpos, Protoss_Nexus);
          }
        }
      } else {
        dontScout = true;
        setScouts(0);
      }

      if (isProxying) {
        int na = 0;
        for (Unit* u : state_->unitsInfo().myUnits()) {
          if (u->type == buildtypes::Terran_Marine) {
            if (na++ % 2) {
              u->allowAggressive = true;
            } else {
              u->forceAggressive = true;
            }
          }
        }
        if (!hasPulledWorkers && !weakRush) {
          hasPulledWorkers = true;
          int n = 4;
          for (Unit* u : state_->unitsInfo().myUnits()) {
            if (u->type == buildtypes::Terran_SCV && isValidCombatWorker(u)) {
              if (n-- <= 0) {
                break;
              }
//              if (na++ % 2) {
//                u->allowAggressive = true;
//              } else {
//                u->forceAggressive = true;
//              }
              u->forceAggressive = true;
              u->combatWorker = true;
            }
          }
        }
      }
    }

    cancelExpo = false;
    if (!hasPulledWorkers && !dirtyWorkerRush) {

      int enemyHarassingWorkers = 0;
      if (state_->currentFrame() < 24 * 60 * 4) {
        for (Unit* u : state_->unitsInfo().visibleEnemyUnits()) {
          if (u->type->isWorker &&
              state_->currentFrame() - u->lastFiredWeapon <= 24 * 10) {
            auto& tilesInfo = state_->tilesInfo();
            auto* tilesData = tilesInfo.tiles.data();
            const Tile* tile = tilesInfo.tryGetTile(u->x, u->y);
            if (tile) {
              size_t index = tile - tilesData;
              if (inBaseArea[index]) {
                ++enemyHarassingWorkers;
              }
            }
          }
        }
      }

      bool noBunkerDefence = state_->unitsInfo().myCompletedUnitsOfType(Terran_Bunker).empty() && state_->currentFrame() < 24 * 60 * 8 && armySupply < 8;

      if ((enemyHarassingWorkers || earlyRushDefence || noBunkerDefence) && (enemyArmySupplyInOurBase + enemyHarassingWorkers) && armySupply <= enemyArmySupply && armySupply < 8 && enemyZerglingCount && st.frame < 24 * 60 * 4) {
        int n = std::min(int(enemyAttackingArmySupply + enemyHarassingWorkers - armySupply), (int)state_->unitsInfo().myUnitsOfType(Terran_SCV).size() / 2);
        if (enemyAttackingArmySupply || enemyHarassingWorkers >= 3) {
          if (countProduction(st, Terran_Command_Center) && countPlusProduction(st, Terran_SCV) >= 13) {
            n += 3;
          }
        }
//        if (haveBunkerAtHome) {
//          n = 0;
//        }
        for (Unit* u : state_->unitsInfo().myUnitsOfType(Terran_SCV)) {
          if (u->type == buildtypes::Terran_SCV && u->combatWorker) {
            --n;
          }
        }
        for (Unit* u : state_->unitsInfo().myUnitsOfType(Terran_SCV)) {
          if (n <= 0) {
            break;
          }
          if (u->type == buildtypes::Terran_SCV && !u->combatWorker && u->unit.health >= u->unit.max_health) {
            if (isValidCombatWorker(u)) {
              //u->forceAggressive = true;
              u->combatWorker = true;
              u->combatWorkerAutoRetreat = true;
              --n;
            }
          }
        }
        if (earlyRushDefence && ((haveBunkerAtHome && enemyArmySupply > armySupply) || enemyAttackingArmySupply > armySupply + 4 || (enemyAttackingArmySupply > armySupply && state_->currentFrame() < 24 * 60 * 3 + 24 * 30)) && enemyGasUnits == 0 && enemyCyberneticsCoreCount == 0) {
          cancelExpo = true;

          for (Unit* u : state_->unitsInfo().myBuildings()) {
            if (!u->completed() && u->type->isResourceDepot) {
              auto bwu = BWAPI::Broodwar->getUnit(u->id);
              if (bwu) bwu->cancelConstruction();
            }
          }
        }

      } else {
        for (Unit* u : state_->unitsInfo().myUnitsOfType(Terran_SCV)) {
          if (u->type == buildtypes::Terran_SCV && u->combatWorker) {
            u->combatWorker = false;
          }
        }
      }

    }

    if (dirtyWorkerRush) {

      postBlackboardKey("TacticsRushBase", true);
      postBlackboardKey("TacticsAttack", true);
      postBlackboardKey("TacticsAlwaysFight", true);

      if (st.frame >= 24 * 75) {

//        for (Unit* u : state_->unitsInfo().myWorkers()) {
//          auto taskData = state_->board()->taskDataWithUnit(u);
//          if (taskData.task && taskData.owner &&
//              taskData.owner->name().find("Scouting") !=
//                  std::string::npos) {
//            taskData.task->cancel(state_);
//          }
//        }

        std::vector<Unit*> workers;
        for (Unit* u : state_->unitsInfo().myWorkers()) {
          auto taskData = state_->board()->taskDataWithUnit(u);
          if (taskData.task && taskData.owner &&
              taskData.owner->name().find("Scouting") !=
                  std::string::npos) {
            continue;
          }
          workers.push_back(u);
        }
        std::sort(workers.begin(), workers.end(), [&](Unit* a, Unit* b) {
          return state_->areaInfo().walkPathLength(a->pos(), state_->areaInfo().enemyStartLocation()) < state_->areaInfo().walkPathLength(b->pos(), state_->areaInfo().enemyStartLocation());
        });
        for (int i = 0; i < (int)workers.size() - 1; ++i) {
          Unit* u = workers[i];
          u->combatWorker = true;
          u->combatWorkerAutoRetreat = true;
          //u->forceAggressive = true;
          u->allowAggressive = true;
        }

        int killedWorkers = 0;
        int liveWorkers = 0;
        for (Unit* u : state_->unitsInfo().allUnitsEver()) {
          if (u->isEnemy && u->type->isWorker) {
            if (u->dead) {
              ++killedWorkers;
            } else {
              ++liveWorkers;
            }
          }
        }

        bool done = false;
        if (liveWorkers <= 2 && killedWorkers >= 12) {
          done = true;
        } else if (liveWorkers == 0 && killedWorkers >= 9) {
          done = true;
        } else if (st.frame >= 24 * 60 * 6) {
          done = true;
        } else {
          for (Unit* u : state_->unitsInfo().visibleEnemyUnits()) {
            if (u->type == buildtypes::Protoss_Photon_Cannon && u->completed() && u->unit.health + u->unit.shield >= 10 + workers.size() * 5) {
              done = true;
            }
          }
        }

        if (done) {
          postBlackboardKey("TacticsRushBase", false);
          postBlackboardKey("TacticsAttack", false);
          postBlackboardKey("TacticsAlwaysFight", false);

          dirtyWorkerRush = false;
        }
      }

    }

//    for (Unit* u : state_->unitsInfo().myBuildings()) {
//      if (!u->completed() && u->type->isResourceDepot) {
//        bool ok = false;
//        for (auto& area : state_->areaInfo().areas()) {
//          for (auto& centerPos : area.baseLocations) {
//            if (utils::distance(centerPos, u->pos()) <= 8) {
//              ok = true;
//              break;
//            }
//          }
//        }
//        if (!ok) {
//          auto bwu = BWAPI::Broodwar->getUnit(u->id);
//          if (bwu) bwu->cancelConstruction();
//        }
//      }
//    }

    inProdTurrets = state_->unitsInfo()
            .myUnitsOfType(Terran_Missile_Turret)
            .size() -
        state_->unitsInfo()
            .myCompletedUnitsOfType(Terran_Missile_Turret)
            .size();


    maybeNeedToRepair = false;
    if (enemyRace == +tc::BW::Race::Protoss && countUnits(st, Terran_Bunker) && st.frame < 24 * 60 * 10) {
      if (countProduction(st, Terran_Factory) || enemyDragoonCount) {
        if (countUnits(st, Terran_Siege_Tank_Tank_Mode) == 0 && armySupply < 9.0) {
          maybeNeedToRepair = true;
        }
      }
    }

    if (cMassVulture) {
      for (Unit* u : state_->unitsInfo().myUnitsOfType(Terran_Vulture)) {
        u->allowAggressive = true;
      }
    }
  }

  virtual void buildStep2(autobuild::BuildState& st) override {
    using namespace buildtypes;
    using namespace autobuild;

    st.mineralIncomeMultiplier = maybeNeedToRepair ? 0.5 : 1.0;

    int scvCount = countPlusProduction(st, Terran_SCV);
    int tankCount = countPlusProduction(st, Terran_Siege_Tank_Tank_Mode) +
        countPlusProduction(st, Terran_Siege_Tank_Siege_Mode);
    int valkyrieCount = countPlusProduction(st, Terran_Valkyrie);
    int marineCount = countPlusProduction(st, Terran_Marine);
    int wraithCount = countPlusProduction(st, Terran_Wraith);
    int goliathCount = countPlusProduction(st, Terran_Goliath);
    int vultureCount = countPlusProduction(st, Terran_Vulture);

    st.autoBuildRefineries =
        (countPlusProduction(st, Terran_Refinery) == 0 && scvCount >= 13) ||
        st.frame >= 15 * 60 * 9 || (bases >= 2 && tankCount);

    bool massVulture = cMassVulture;
    bool goBio = cGoBio;
    bool twoRax = cTwoRax;

    if (dirtyWorkerRush) {
      st.preBuildSupply = false;
//      build(Terran_SCV);
//      build(Terran_Marine);
//      buildN(Terran_SCV, 8);
      if (st.frame < 24 * 75) {
        buildN(Terran_SCV, 10);
      }
      return;
    }

    if (earlyRushDefence && st.frame < 24 * 60 * 8 && bases == 1) {
      if (enemyRace == +tc::BW::Race::Zerg && countPlusProduction(st, Terran_Marine) == 0) {
        st.preBuildSupply = false;
        build(Terran_SCV);
        buildN(Terran_Bunker, 1, homePosition);
        build(Terran_Marine);
        return;
      }
      if (twoRax || (!massVulture && enemyRace == +tc::BW::Race::Zerg)) {
        buildN(Terran_Barracks, 2);
        build(Terran_Marine);
      } else if (massVulture) {
        upgrade(Spider_Mines) && upgrade(Ion_Thrusters);
        build(Terran_Vulture);
      } else {
        upgrade(Tank_Siege_Mode);
        build(Terran_Siege_Tank_Tank_Mode);
      }
      buildN(Terran_SCV, 40);

      if (!cancelExpo && !massVulture) {
        buildN(Terran_Command_Center, 2, nextBase);
      }
      if (vultureCount || enemyDragoonCount) {
        buildN(Terran_Bunker, haveBunkerAtHome || haveProxiedBunker ? 2 : 1, nextBunkerPos);
      }

      buildN(Terran_SCV, 30);

      if (enemyArmySupply >= armySupply) {
        if (twoRax) {
          buildN(Terran_Barracks, 2);
        }
        buildN(Terran_Marine, 12);
        buildN(Terran_Vulture, 5);
        if (!twoRax) {
          buildN(Terran_Machine_Shop, 1);
        }
        buildN(Terran_Vulture, 3);
      } else {
        if (twoRax) {
          buildN(Terran_Barracks, 2);
        } else {
          buildN(Terran_Machine_Shop, 1);
        }
        buildN(Terran_Vulture, 1);
        buildN(Terran_Factory, 1);
        buildN(Terran_Refinery, 1);
        buildN(Terran_Marine, 4);
      }
//      if (vultureCount >= 2 && enemyArmySupplyInOurBase == 0) {
//        buildN(Terran_Machine_Shop, 1);
//      }
      if (!haveBunkerAtHome) {
        buildN(Terran_Bunker, 1 + haveBunkerElsewhere, homePosition);
      }
      buildN(Terran_SCV, 12);
      buildN(Terran_Marine, 2);
      if (has(st, Terran_Factory)) {
        buildN(Terran_Vulture, 2);
      }
      return;
    }

    if (st.frame < 24 * 60 * 5 && (enemyVeryFastExpanded || proxyBunkerPos != kInvalidPosition) && armySupply > enemyAttackingArmySupply) {
      if (enemyRace != +tc::BW::Race::Zerg && st.minerals < 500) {
        if (bases < 2 || scvCount < 25) {
          buildN(
              Terran_Bunker,
              haveBunkerAtHome || haveProxiedBunker ? 2 : 1,
              nextBunkerPos);
          buildN(Terran_SCV, 40);
          if (twoRax) {
            buildN(Terran_Barracks, 2);
            build(Terran_Marine);
          } else if (massVulture) {
            upgrade(Spider_Mines) && upgrade(Ion_Thrusters);
            build(Terran_Vulture);
          } else {
            buildN(Terran_Siege_Tank_Tank_Mode, 3);
            upgrade(Tank_Siege_Mode);
            buildN(Terran_Siege_Tank_Tank_Mode, 1);
          }
          if (!cancelExpo && !massVulture) {
            buildN(Terran_Command_Center, 2, nextBase);
            if (enemyDragoonCount || countPlusProduction(st, Terran_Factory)) {
              buildN(
                  Terran_Bunker,
                  haveBunkerAtHome || haveProxiedBunker ? 2 : 1,
                  nextBunkerPos);
            }
          }
          buildN(Terran_SCV, 40);

          if (proxyBunkerPos != kInvalidPosition) {
            buildN(Terran_Bunker, 1, proxyBunkerPos);
            buildN(Terran_Marine, 4);
          }
          return;
        }
      }
    }

    if (st.frame < 24 * 60 * 5 && scvCount < 22 && state_->unitsInfo().myUnitsOfType(massVulture ? Terran_Vulture : Terran_Siege_Tank_Tank_Mode).empty() && state_->unitsInfo().myUnitsOfType(Terran_Siege_Tank_Siege_Mode).empty() && !wasRushed && state_->resources().ore < 600) {
      buildN(Terran_SCV, 26);
      if (enemyGatewayCount == 0 && enemyCyberneticsCoreCount == 0) {
        buildN(Terran_Marine, 4);
      }
      if (twoRax) {
        build(Terran_Marine);
        buildN(Terran_Medic, marineCount / 2);
        buildN(Terran_Marine, 6);
      } else {
        if (enemyRace == +tc::BW::Race::Zerg && !massVulture) {
          buildN(Terran_Starport, 1);
          build(Terran_Vulture);
          buildN(Terran_Factory, 1, firstFactoryPos);
        } else {
          build(massVulture && enemyDragoonCount == 0 ? Terran_Vulture : Terran_Siege_Tank_Tank_Mode);
          if (massVulture && (enemyGasUnits || armySupply >= enemyArmySupply)) {
            buildN(Terran_Machine_Shop, 1);
          }
          buildN(Terran_Factory, 1, firstFactoryPos);
        }
      }
      buildN(Terran_SCV, 15);
      bool maybeFe = enemyVeryFastExpanded ||
          enemyAttackingArmySupply + enemyAttackingWorkerCount / 5 +
                  enemyProxyGatewayCount + enemyProxyBarracksCount ==
              0;
      //bool fe = maybeFe && enemyGasCount + enemyGatewayCount + enemyCyberneticsCoreCount;
      //bool delayExpo = enemyZealotCount > 0 || enemyGatewayCount + enemyBarracksCount == 0 || enemyGatewayCount + enemyBarracksCount >= 2;
      bool delayExpo = enemyRace == +tc::BW::Race::Zerg;
      buildN(Terran_Marine, enemyRace != +tc::BW::Race::Zerg && enemyVeryFastExpanded ? 4 : enemyAttackingArmySupply + enemyAttackingWorkerCount / 5 + enemyProxyGatewayCount ? 8 : enemyGasCount || enemyGatewayCount || enemyCyberneticsCoreCount ? 1 : 4);
      if (enemyRace != +tc::BW::Race::Zerg && enemyHasExpanded) {
        if (!cancelExpo && !massVulture) {
          buildN(Terran_Command_Center, 2, nextBase);
        }
        buildN(Terran_SCV, 16);
      }
      bool siegeExpand = cSiegeExpand;

      if (maybeFe && !massVulture) {
        st.preBuildSupply = false;
        if (state_->currentFrame() >= 24 * 60 * 4 || countPlusProduction(st, Terran_Command_Center) >= 2) {
          buildN(
              Terran_Bunker,
              haveBunkerAtHome || haveProxiedBunker ? 2 : 1,
              nextBunkerPos);
        }
        buildN(Terran_SCV, 18);
        buildN(Terran_Marine, 1);
        buildN(Terran_Refinery, 1);
        buildN(Terran_Supply_Depot, 2);
        buildN(Terran_SCV, 16);
        if (enemyGasCount == 0 && !state_->unitsInfo().enemyBuildings().empty() && enemyCyberneticsCoreCount == 0) {
          buildN(Terran_Marine, 4);
        }
        if (!cancelExpo) {
          if (enemyRace != +tc::BW::Race::Zerg || (!twoRax && !massVulture)) {
            buildN(Terran_Command_Center, 2, nextBase);
          }
          if (enemyZealotCount || (enemyGatewayCount && enemyGasUnits == 0 && enemyCyberneticsCoreCount == 0)) {
            buildN(Terran_SCV, 17);
            buildN(Terran_Supply_Depot, 2);
            buildN(Terran_SCV, 16);
          }
          if (siegeExpand) {
            buildN(Terran_Refinery, 1);
          }
          if (enemyArmySupply >= 4 && enemyRace != +tc::BW::Race::Zerg) {
            buildN(Terran_Bunker, haveBunkerAtHome || haveProxiedBunker ? 2 : 1, nextBunkerPos);
          }

          //if (!state_->unitsInfo().myCompletedUnitsOfType(Terran_Refinery).empty() && (enemyArmySupply > armySupply || enemyGatewayCount + enemyBarracksCount >= 2)) {
          if (countPlusProduction(st, Terran_Refinery)) {
            if (!twoRax) {
              buildN(Terran_Factory, 1);
            }
            delayExpo = true;
          }
        }
        if (st.frame < 24 * 60 * 5 && enemyArmySupply == enemyZealotCount * 2 + enemyZerglingCount * 0.5 + enemyMarineCount && enemyArmySupply >= 6) {
          buildN(Terran_Marine, enemyArmySupply);
          if (has(st, Terran_Factory) || enemyArmySupply >= 8) {
            delayExpo = true;
            buildN(Terran_Vulture, (enemyArmySupply - marineCount) / 2);
          }
        }
        if (delayExpo) {
          if (twoRax) {
            buildN(Terran_Barracks, 2);
            buildN(Terran_Marine, 8);
          }
          buildN(Terran_SCV, 28);
          if (state_->currentFrame() >= 24 * 60 * 2 + 24 * 30 || enemyArmySupply) {
            if (earlyRushDefence) {
              buildN(Terran_Refinery, 1);
            }
            buildN(Terran_Supply_Depot, 2);
          }
          buildN(Terran_Marine, 4);
        } else {
          buildN(Terran_Marine, 1);
        }
        buildN(Terran_SCV, 15);
        //if (enemyGasCount == 0 && !state_->unitsInfo().enemyBuildings().empty()) {
        //  buildN(Terran_SCV, 17);
        //  buildN(Terran_Marine, 2);
        //  buildN(Terran_Supply_Depot, 2);
        //  buildN(Terran_SCV, 15);
        //}
      }
      if (enemyRace != +tc::BW::Race::Zerg && (enemyHasExpanded || (enemyStaticDefenceCount >= 2 && enemyProxyBuildingCount == 0))) {
        buildN(Terran_SCV, 20);
        buildN(Terran_Command_Center, 2, nextBase);
        buildN(Terran_SCV, 18);
      }
      if (!maybeFe) {
        buildN(Terran_SCV, 16);
        buildN(Terran_Marine, 4);
        buildN(Terran_Refinery, 1);
      }
      if (massVulture || siegeExpand) {
        buildN(Terran_Factory, 1);
        buildN(Terran_SCV, 15);
        buildN(Terran_Marine, 1);
        buildN(Terran_Refinery, 1);
      }
      if (beingProxied || earlyRushDefence) {
        if (enemyArmySupply > armySupply || countProduction(st, Terran_SCV)) {
          buildN(Terran_Marine, 4);
        }
        if (!haveBunkerAtHome) {
          buildN(Terran_Bunker, 1 + haveBunkerElsewhere, homePosition);
        }
      }
      if (enemyRace == +tc::BW::Race::Zerg) {
        buildN(Terran_SCV, 12);
        buildN(Terran_Barracks, 1, firstBarracksPos);
      } else {
        buildN(Terran_Barracks, 1, firstBarracksPos);
        buildN(Terran_SCV, 12);
      }
      buildN(Terran_Supply_Depot, 1);
      buildN(Terran_SCV, 9);

      if (enemyArmySupplyInOurBase && enemyGasCount == 0) {
        buildN(Terran_Marine, 2);
        //buildN(Terran_Bunker, 1, homePosition);
      }

      if (proxyBunkerPos != kInvalidPosition) {
        buildN(Terran_Bunker, 1, proxyBunkerPos);
        buildN(Terran_Marine, 4);
      }
      return;
    }

    bool wraithAntiAir = true;

//    if ((marineCount / 2 < enemyCarrierCount * 6 && enemyAirArmySupply >= enemyGroundArmySupply) || (marineCount / 2 < enemyCarrierCount * 8 + enemyStargateCount * 4 && tankCount * 2 > enemyGroundArmySupply) || (tankCount >= 6 && groundArmySupply > enemyGroundArmySupply && marineCount / 2 < tankCount)) {
//      goBio = true;
//    }

    if (goBio) {

      buildN(Terran_Barracks, 14);

      build(Terran_Marine);
      buildN(Terran_Medic, int(marineCount / 2.5));

      upgrade(U_238_Shells);
      upgrade(Stim_Packs);

    } else {
      //buildN(Terran_Machine_Shop, 8);
      if (countProduction(st, Terran_Factory) == 0 || st.minerals >= 400) {
        buildN(Terran_Factory, 10);
      }

      build(Terran_Vulture);
    }
    if (!massVulture || tankCount < (vultureCount - (enemyHydraliskCount + enemyLurkerCount + enemyDragoonCount + enemyTankCount ? 3 : 10) / 3)) {
      if ((goliathCount + marineCount / 2 + wraithCount < enemyCarrierCount * 6 && enemyAirArmySupply >= enemyGroundArmySupply) || (goliathCount + marineCount / 2 + wraithCount < enemyCarrierCount * 8 + enemyStargateCount * 4 && tankCount * 2 > enemyGroundArmySupply) || (tankCount >= 6 && groundArmySupply > enemyGroundArmySupply && goliathCount + marineCount / 2 + wraithCount < tankCount / 4)) {
        if (!goBio) {
          if (wraithAntiAir && wraithCount && (enemyCarrierCount || enemyAirArmySupply >= 10)) {
            buildN(Terran_Starport, 3);
          }
          build(wraithAntiAir ? Terran_Wraith : Terran_Goliath);
        }
      } else {
        bool buildTanks = true;
        if (enemyRace == +tc::BW::Race::Zerg) {
          if (enemySpireCount + enemyMutaliskCount && armySupply < 20 && enemyGroundArmySupply < enemyAirArmySupply / 2 && enemyLurkerCount == 0) {
            buildTanks = false;
          }
          double aa = marineCount + goliathCount * 2 + wraithCount * 2 + valkyrieCount * 2;
          if (enemyAirArmySupply > aa) {
            buildTanks = false;
          }
          if (groundArmySupply > enemyGroundArmySupply * 2 && (armySupply < 16 || tankCount)) {
            buildTanks = false;
          }
          if (enemyAirArmySupply > enemyGroundArmySupply && groundArmySupply > enemyGroundArmySupply && aa < enemyAirArmySupply + 16 && tankCount > enemyLurkerCount) {
            buildTanks = false;
          }
        }
        if (buildTanks) {
          if (enemyRace == +tc::BW::Race::Zerg && twoRax) {
            buildN(Terran_Barracks, 2);
            build(Terran_Marine);
            buildN(Terran_Medic, int(marineCount / 2.5));

            upgrade(U_238_Shells);
            upgrade(Stim_Packs);
          }
          buildN(Terran_Siege_Tank_Tank_Mode, 20);
          upgrade(Tank_Siege_Mode);
          if (!goBio) {
            buildN(Terran_Vulture, tankCount >= 3 ? tankCount + 2 : tankCount - 1);
          }
          if (countProduction(st, Terran_Siege_Tank_Tank_Mode) >= countUnits(st, Terran_Machine_Shop)) {
            buildN(Terran_Machine_Shop, countUnits(st, Terran_Factory));
          }
        } else {
          build(Terran_Goliath);
          if (goliathCount >= 2) {
            upgrade(Charon_Boosters);
          }

          if (enemyRace == +tc::BW::Race::Zerg && twoRax) {
            buildN(Terran_Barracks, 2);
            build(Terran_Marine);
            buildN(Terran_Medic, int(marineCount / 2.5));

            upgrade(U_238_Shells);
            upgrade(Stim_Packs);
          }
        }
      }
    } else {
      if (wraithCount < enemyAirArmySupply || (enemyRace == +tc::BW::Race::Zerg && groundArmySupply >= enemyGroundArmySupply)) {
        if (wraithAntiAir && wraithCount && (enemyCarrierCount || enemyAirArmySupply >= 10 || enemySpireCount || enemyMutaliskCount)) {
          buildN(Terran_Starport, 3);
        }
        build(wraithAntiAir ? Terran_Wraith : Terran_Goliath);
      }
    }

    if (wraithAntiAir && wraithCount && (wraithCount >= 4 || enemyAirArmySupply >= 10 || enemyCarrierCount)) {
      buildN(Terran_Starport, 2);
      upgrade(Cloaking_Field);
    }

    if (!rushed || st.minerals >= 400 || armySupply >= 24) {
      if (armySupply >= 60 && !goBio) {
        ////buildN(Terran_Wraith, 1 + tankCount / 6 + enemyArbiterCount + enemyShuttleCount);
        buildN(wraithAntiAir ? Terran_Wraith : Terran_Goliath, 1 + tankCount / 6 + enemyArbiterCount + enemyShuttleCount);
      }
      if (armySupply >= 30 && (bases >= 3 || armySupply > enemyArmySupply)) {
        if (enemyAirArmySupply) {
          ////buildN(Terran_Wraith, countPlusProduction(st, Terran_Dropship) / 2);
          buildN(wraithAntiAir ? Terran_Wraith : Terran_Goliath, countPlusProduction(st, Terran_Dropship) / 2);
        }
        buildN(Terran_Dropship, std::min(2, int(armySupply / 20)));
      }

      if (enemyArbiterCount + enemyCarrierCount + (wraithAntiAir ? armySupply >= 14 : enemyShuttleCount) && !goBio) {
        ////buildN(Terran_Starport, 2);
        ////buildN(Terran_Wraith, std::min(tankCount, enemyArbiterCount * 2 + enemyCarrierCount * 4 + enemyShuttleCount));
        buildN(wraithAntiAir ? Terran_Wraith : Terran_Goliath, std::min(tankCount, enemyArbiterCount * 2 + enemyCarrierCount * 4 + enemyShuttleCount));

        //if (countProduction(st, Terran_Missile_Turret) == 0) {
        //  buildN(Terran_Missile_Turret, 14);
        //}
      }

      if (st.minerals >= 400 && !goBio) {
        buildN(Terran_Factory, st.minerals >= 800 ? 10 : 4);
      }

      if (goliathCount) {
        upgrade(Charon_Boosters);
      }

      //buildN(Terran_Starport, 1);
    }
    if (!state_->unitsInfo().myUnitsOfType(massVulture ? Terran_Vulture : Terran_Siege_Tank_Tank_Mode).empty()) {
      if ((enemyAttackingArmySupply > armySupply && armySupply >= 8) || (wasRushed && armySupply >= 8) || countPlusProduction(st, Terran_Dropship)) {
        if (enemyCloakedUnitCount || armySupply > enemyArmySupply) {
          buildN(Terran_Missile_Turret, 1, nextBunkerPos);
        }
        buildN(
            Terran_Bunker,
            haveBunkerAtHome || haveProxiedBunker ? 2 : 1,
            nextBunkerPos);
      }
    }
    if (!goBio) {
      //buildN(Terran_Machine_Shop, 2);
      //buildN(Terran_Factory, 2);
    } else {
      buildN(Terran_Barracks, 2);
    }

    if (twoRax) {
      buildN(Terran_Barracks, 2);
      buildN(Terran_Medic, 2);
      buildN(Terran_Marine, 8);
    } else if (massVulture) {
      upgrade(Spider_Mines) && upgrade(Ion_Thrusters);
      buildN(Terran_Vulture, 2);
    } else if (enemyRace != +tc::BW::Race::Zerg) {
      buildN(Terran_Siege_Tank_Tank_Mode, 2);
      upgrade(Tank_Siege_Mode);
      buildN(Terran_Siege_Tank_Tank_Mode, 1);
    }

    int nWorkers = mineralFields * 2 + countPlusProduction(st, Terran_Refinery) * 3 + 3;

    auto expandLogic = [&]() {
      if (canExpand &&
          st.workers >= nWorkers &&
          !st.isExpanding && countProduction(st, Terran_Command_Center) == 0) {
        if (scans >= std::min(enemyLurkerCount + enemyDarkTemplarCount, 3) || has(st, Terran_Science_Vessel)) {
          if (mineralFields < 12 || armySupply >= std::min(std::max(std::max(22.0, (double)nWorkers), enemyArmySupply), 60.0)) {
            build(Terran_Command_Center, nextBase);
          }
          if (bases < 3 && armySupply >= enemyArmySupply + 16 - tankCount * 2 && state_->currentFrame() >= 24 * 60 * 9) {
            build(Terran_Command_Center, nextBase);
          }
        }
      }
    };

    bool prioExpand = false;

    if (armySupply >= 20 || bases >= 3 || st.frame >= 24 * 60 * 12) {
      buildN(Terran_SCV, std::min(nWorkers, 90), armySupply >= enemyArmySupply ? 5 : 2);
      if ((armySupply > enemyArmySupply && st.frame >= 24 * 60 * 10) || (mineralFields < 14 && armySupply >= 14.0)) {
        prioExpand = true;
      } else {
        expandLogic();
      }
    }

    if (enemyRace == +tc::BW::Race::Zerg) {
      if ((armySupply >= 20.0 && enemyZerglingCount * 0.7 + enemyHydraliskCount * 1.25 < enemyArmySupply) || armySupply >= enemyArmySupply + 10.0) {
        if (countPlusProduction(st, Terran_SCV) >= 55) {
          buildN(Terran_Control_Tower, countUnits(st, Terran_Starport));
          buildN(Terran_Starport, 2);
        }
        buildN(Terran_Science_Vessel, int(armySupply / 12));
        upgrade(Irradiate);
      }
      if (enemyLurkerCount) {
        upgrade(Irradiate);
        buildN(Terran_Science_Vessel, 1);
      }
    } else {
      if ((enemyRace != +tc::BW::Race::Terran || armySupply >= 40.0) && enemyCloakedUnitCount - enemyObserverCount + enemyLurkerCount * 2 + enemyArbiterCount * 4 >= (armySupply >= 42 ? 2 : 4)) {
        buildN(Terran_Science_Vessel, std::min(1 + (enemyCloakedUnitCount - enemyObserverCount / 4) / 4, int(armySupply / 20.0)));
      }
    }

    if (has(st, Terran_Starport)) {
      if (enemyRace == +tc::BW::Race::Terran) {
        if (armySupply >= 40.0 || enemyAirArmySupply > airArmySupply) {
          if (!goBio && enemyAirArmySupply - airArmySupply >= 6) {
            buildN(Terran_Starport, 3);
            buildN(Terran_Wraith, enemyAirArmySupply / 2);
          }
          buildN(Terran_Starport, 2);
        }
        buildN(Terran_Wraith, tankCount / 4 + (enemyWraithCount + 1) / 2);
      } else if (enemyRace == +tc::BW::Race::Zerg && !goBio) {
        if (enemyMutaliskCount + enemySpireCount && groundArmySupply > enemyGroundArmySupply) {
          buildN(Terran_Wraith, 2);
          buildN(Terran_Starport, 2);
          buildN(Terran_Valkyrie, 1);
        }
      }
      buildN(Terran_Valkyrie, enemyMutaliskCount / 3 + enemyWraithCount / 3 + enemyBattlecruiserCount);
    }

    if (armySupply >= 16 && (massVulture || has(st, Tank_Siege_Mode)) && !twoRax) {
      if (goBio) {
        upgrade(U_238_Shells);
        upgrade(Stim_Packs);
      } else {
        upgrade(Spider_Mines);
        upgrade(Ion_Thrusters);
      }
    }

    if (enemyRace == +tc::BW::Race::Zerg && (enemyMutaliskCount + enemySpireCount || enemyGroundArmySupply < groundArmySupply / 2)) {
      if (st.frame >= 24 * 60 * 5 + 24 * 30 || enemyAirArmySupply) {
        if (countProduction(st, Terran_Missile_Turret) == inProdTurrets || countProduction(st, Terran_Missile_Turret) < 3) {
          if (antiMutaTurretPos != kInvalidPosition) {
            buildN(Terran_Missile_Turret, std::min(std::max(enemyMutaliskCount * 3, 10), 20), antiMutaTurretPos);
          }
        }
      } else if (st.frame >= 24 * 60 * 5) {
        buildN(Terran_Engineering_Bay, 1);
      }
    }

    buildN(Terran_SCV, 60);

    if (!massVulture && (enemyArmySupply >= armySupply || tankCount < 2) && !twoRax) {
      if (enemyRace != +tc::BW::Race::Zerg) {
        buildN(Terran_Siege_Tank_Tank_Mode, 2);
        upgrade(Tank_Siege_Mode);
        buildN(Terran_Siege_Tank_Tank_Mode, 1);
      }
    } else if (massVulture && vultureCount && armySupply > enemyArmySupplyInOurBase) {
      if (enemyZealotCount < 6) {
        upgrade(Ion_Thrusters);
        upgrade(Spider_Mines);
      } else {
        upgrade(Spider_Mines);
        upgrade(Ion_Thrusters);
      }
    }

    if (tankCount || (massVulture && vultureCount >= 9)) {
      if (!cancelExpo) {
        buildN(Terran_Command_Center, 2, nextBase);
      }
      buildN(
          Terran_Bunker,
          haveBunkerAtHome || haveProxiedBunker ? 2 : 1,
          nextBunkerPos);
      if (!massVulture) {
        upgrade(Tank_Siege_Mode);
      }

      if (((massVulture ? vultureCount : tankCount) >= 8 || armySupply >= 24) && bases >= 3 && scvCount >= 40) {
        if (goBio || marineCount >= 16)  {

          if (countPlusProduction(st, Terran_Armory) >= 2) {
            upgrade(Terran_Vehicle_Plating_1) && upgrade(Terran_Vehicle_Plating_2) && upgrade(Terran_Vehicle_Plating_3);
          }
          upgrade(Terran_Vehicle_Weapons_1) && upgrade(Terran_Vehicle_Weapons_2) && upgrade(Terran_Vehicle_Weapons_3) && upgrade(Terran_Vehicle_Plating_1) && upgrade(Terran_Vehicle_Plating_2) && upgrade(Terran_Vehicle_Plating_3);

          if (has(st, Terran_Infantry_Armor_2)) {
            upgrade(Terran_Infantry_Armor_3);
          } else if (has(st, Terran_Infantry_Armor_1)) {
            upgrade(Terran_Infantry_Armor_2);
          }
          if (has(st, Terran_Infantry_Weapons_2)) {
            upgrade(Terran_Infantry_Weapons_3);
          } else if (has(st, Terran_Infantry_Weapons_1)) {
            upgrade(Terran_Infantry_Weapons_2);
          }
          upgrade(Terran_Infantry_Armor_1);
          upgrade(Terran_Infantry_Weapons_1);
          if (mineralFields >= 24 && st.workers >= 56) {
            buildN(Terran_Engineering_Bay, 2);
          }
        } else {
          upgrade(Terran_Vehicle_Plating_1) && upgrade(Terran_Vehicle_Plating_2) && upgrade(Terran_Vehicle_Plating_3);
          upgrade(Terran_Vehicle_Weapons_1) && upgrade(Terran_Vehicle_Weapons_2) && upgrade(Terran_Vehicle_Weapons_3);
          buildN(Terran_Armory, 2);
        }
      }

      if (armySupply > enemyAttackingArmySupply + 2) {
        buildN(Terran_SCV, 34);
      }
    }

    if (prioExpand) {
      expandLogic();
    }

    if (scvCount >= 48 || armySupply >= 32.0 || (armySupply >= 4.0 && enemyCloakedUnitCount + enemyTemplarArchivesCount + enemyLurkerCount && (armySupply >= 12.0 || enemyArmySupplyInOurBase))) {
      buildN(Terran_Comsat_Station, countUnits(st, Terran_Command_Center));
    }
    if (armySupply >= 16.0) {
      if (enemyArmySupplyInOurBase < armySupply / 10 && armySupply >= enemyArmySupply * 0.75) {
        buildN(Terran_Starport, 1);
        if (enemyArbiterCount || (has(st, Terran_Comsat_Station) && enemyDarkTemplarCount && enemyDarkTemplarCount >= scans)) {
          buildN(Terran_Science_Vessel, 1);
        }
      }
      buildN(Terran_Academy, 1);
    }
    
    //if (enemyRace != +tc::BW::Race::Terran && countPlusProduction(st, Terran_Factory) >= (rushed ? 2 : 1) && (!state_->unitsInfo().myCompletedUnitsOfType(Terran_Factory).empty() || enemyCloakedUnitCount || enemyTemplarArchivesCount)) {
    if (enemyRace != +tc::BW::Race::Terran &&
        countPlusProduction(st, Terran_Factory) >= (rushed ? 1 : 1) &&
        (!state_->unitsInfo().myCompletedUnitsOfType(Terran_Factory).empty() ||
         enemyCloakedUnitCount || enemyTemplarArchivesCount || (scvCount >= 24 && enemyDragoonCount <= 2 && enemyArmySupply <= 6))) {
      if (st.frame >= 24 * 60 * 4 + 24 * 40 + (std::max(enemyGroundArmySupply - 4, 0.0) - enemyDarkTemplarCount * 4) * 24 * 2 || tankCount + vultureCount / 3 >= 2) {
//        if (tankCount >= 3) {
//          buildN(Terran_Comsat_Station, 1);
//        }
        if (countProduction(st, Terran_Missile_Turret) == inProdTurrets) {
          if (baseTurretPos != kInvalidPosition) {
            //buildN(Terran_Missile_Turret, 2, baseTurretPos);
            build(Terran_Missile_Turret, baseTurretPos);
          }
          //buildN(Terran_Missile_Turret, 1, nextBunkerPos);

          if (countPlusProduction(st, Terran_Bunker) || bases >= 2) {
            if (defaultTurretPos != kInvalidPosition) {
              buildN(Terran_Missile_Turret, 5, defaultTurretPos);
            }
          }
        }
      }
      buildN(Terran_Engineering_Bay, 1);

      if ((enemyDarkTemplarCount && enemyArmySupplyInOurBase)) {
        buildN(Terran_Comsat_Station, countUnits(st, Terran_Command_Center));
      }
    }

    if (bases == 1 && (tankCount >= 1 || armySupply >= 12)) {
      if (!cancelExpo) {
        buildN(Terran_Command_Center, 2, nextBase);
      }
      buildN(
          Terran_Bunker,
          haveBunkerAtHome || haveProxiedBunker ? 2 : 1,
          nextBunkerPos);
    }

    if (enemyAttackingArmySupply && enemyDragoonCount == 0 && armySupply <= enemyArmySupply) {
      if (armySupply < 6 && enemyZealotCount && enemyDragoonCount == 0) {
        buildN(Terran_Vulture, enemyZealotCount);
      } else if (enemyZerglingCount + enemyMarineCount) {
        buildN(Terran_Vulture, 2);
      } else if (earlyRushDefence) {
        buildN(Terran_Vulture, 2);
      }
    }
    if (armySupply < 12.0) {
      if (bases >= 2 && tankCount == 0 && countProduction(st, Terran_SCV)) {
        buildN(Terran_Marine, 8);
      }
      if (twoRax) {
        buildN(Terran_Barracks, 2);
      } else {
        buildN(Terran_Factory, 1, firstFactoryPos);
      }
      buildN(Terran_SCV, 15);
      buildN(Terran_Marine, 4);
      //buildN(Terran_Refinery, bases >= 2 ? 2 : 1);
      buildN(Terran_Barracks, 1, firstBarracksPos);
      buildN(Terran_SCV, 10);
      buildN(Terran_Supply_Depot, 1);
      buildN(Terran_SCV, 9);

      if (has(st, Terran_Machine_Shop) && enemyAttackingArmySupply && !earlyRushDefence && enemyRace != +tc::BW::Race::Zerg) {
        if (massVulture) {
          upgrade(Spider_Mines);
          buildN(Terran_Vulture, 2);
        } else {
          upgrade(Tank_Siege_Mode);
          buildN(Terran_Siege_Tank_Tank_Mode, 1);
        }
      }
    }

    if (bases >= 2 && tankCount < 4) {
      buildN(
          Terran_Bunker,
          haveBunkerAtHome || haveProxiedBunker ? 2 : 1,
          nextBunkerPos);
    }

    if (wasRushed) {
      if (earlyRushDefence && enemyArmySupply > armySupply) {
        buildN(Terran_Marine, 16);
        if (has(st, Terran_Factory) && enemyGasUnits == 0) {
          buildN(Terran_Vulture, 10);
        }
      } else {
        buildN(Terran_Marine, std::min(4 + enemyZealotCount + enemyZerglingCount / 2, 8));
        if (has(st, Terran_Factory) && enemyGasUnits == 0) {
          buildN(Terran_Vulture, 1);
        }
      }
//      if (enemyArmySupply > armySupply) {
//        buildN(Terran_Bunker, 1, homePosition);
//      }
    }

    if (proxyBunkerPos != kInvalidPosition) {
      buildN(Terran_Bunker, 1, proxyBunkerPos);
      buildN(Terran_Marine, 4);
    }

    if (has(st, Terran_Machine_Shop) && enemyDragoonCount >= 3 && !twoRax) {
      buildN(Terran_Siege_Tank_Tank_Mode, 2);
      upgrade(Tank_Siege_Mode);
      buildN(Terran_Siege_Tank_Tank_Mode, 1);
    }

    if (tankCount) {
      upgrade(Tank_Siege_Mode);
    }

    if (twoRax && scvCount >= 20) {
      if (enemyRace == +tc::BW::Race::Zerg) {
        upgrade(U_238_Shells);
        upgrade(Stim_Packs);
      } else if (enemyRace == +tc::BW::Race::Protoss) {
        upgrade(U_238_Shells);
      }
    }
  }
};

REGISTER_SUBCLASS_3(ABBOBase, ABBOtcheese, UpcId, State*, Module*);
} // namespace cherrypi
