#include "BananaBrain.h"

int Strategy::opening_supply_count()
{
	int half_supply = (Broodwar->self()->supplyUsed() + 1) / 2;
	return half_supply + worker_manager.lost_worker_count();
}

bool Strategy::is_opponent_army_too_large()
{
	return is_opponent_army_too_large(spending_manager.training_cost_per_minute().supply);
}

bool Strategy::is_opponent_army_too_large(double build_supply_per_minute)
{
	if (build_supply_per_minute <= 0.0) return false;
	double build_supply_per_frame = build_supply_per_minute / (60.0 * 24.0);
	int current_army_supply = tactics_manager.army_supply();
	
	std::vector<std::pair<double,int>> incoming_units;
	
	for (auto& enemy_unit : information_manager.enemy_units()) {
		UnitType type = enemy_unit->type;
		if (type == UnitTypes::Terran_Siege_Tank_Siege_Mode) {
			type = UnitTypes::Terran_Siege_Tank_Tank_Mode;
		} else if (type == UnitTypes::Zerg_Lurker_Egg) {
			type = UnitTypes::Zerg_Lurker_Egg;
		}
		if (type.supplyRequired() > 0 && type.canMove()) {
			double frame = enemy_unit->base_distance / top_speed(type, enemy_unit->player);
			incoming_units.emplace_back(frame, type.supplyRequired());
		}
	}
	
	std::sort(incoming_units.begin(), incoming_units.end());
	int supply_sum = 0;
	for (auto& entry : incoming_units) {
		double frame = entry.first;
		supply_sum += entry.second;
		if (supply_sum >= 20) {
			double our_supply = current_army_supply + build_supply_per_frame * frame;
			if (supply_sum > our_supply) return true;
		}
	}
	return false;
}

bool Strategy::is_defending_rush()
{
	int current_army_supply_without_air_to_air = tactics_manager.army_supply() - tactics_manager.air_to_air_supply();
	return current_army_supply_without_air_to_air < 2 * 20 && current_army_supply_without_air_to_air <= tactics_manager.max_enemy_army_supply();
}

bool Strategy::is_contained()
{
	return tactics_manager.enemy_offense_supply() > tactics_manager.army_supply();
}

bool Strategy::dark_templars_without_mobile_detection()
{
	return (information_manager.enemy_count(UnitTypes::Protoss_Dark_Templar) >= 1 &&
			!tactics_manager.mobile_detection_exists());
}

bool Strategy::expect_lurkers()
{
	bool hydralisk_den_exists = information_manager.enemy_exists(UnitTypes::Zerg_Hydralisk_Den) || information_manager.enemy_seen(UnitTypes::Zerg_Hydralisk);
	bool lair_or_hive_exists = information_manager.enemy_exists(UnitTypes::Zerg_Lair) || information_manager.enemy_exists(UnitTypes::Zerg_Hive);
	return (hydralisk_den_exists && lair_or_hive_exists);
}

void Strategy::update_maxed_out()
{
	int supply = Broodwar->self()->supplyUsed();
	if (supply >= 2 * 192) {
		maxed_out_ = true;
	} else if (supply < 2 * 160) {
		maxed_out_ = false;
	}
}

void Strategy::attack_check_condition()
{
	int supply = tactics_manager.army_supply();
	int air_to_air_supply = tactics_manager.air_to_air_supply();
	if (air_to_air_supply > tactics_manager.enemy_air_supply()) supply -= (air_to_air_supply - tactics_manager.enemy_air_supply());
	
	int enemy_supply = tactics_manager.enemy_army_supply();
	if (supply >= std::max(attack_minimum_, 5 * enemy_supply / 4)) {
		attacking_ = true;
	} else if (supply <= 4 * enemy_supply / 5) {
		attacking_ = false;
	}
	
	if (initial_dark_templar_attack_ &&
		training_manager.unit_count_completed(UnitTypes::Protoss_Dark_Templar) == 0) {
		initial_dark_templar_attack_ = false;
	}
}

const BWEM::Base* Strategy::stage_base() const
{
	const BWEM::Base* base = base_state.main_base();
	if (base == nullptr) {
		key_value_vector<const BWEM::Base*,int> distance_from_center;
		for (auto& base : base_state.controlled_bases()) {
			distance_from_center.emplace_back(base, bwem_map.Center().getApproxDistance(base->Center()));
		}
		base = key_with_largest_value(distance_from_center);
	}
	return base;
}

void Strategy::update_stage_chokepoint()
{
	stage_chokepoint_ = nullptr;
	stage_component_areas_.clear();
	
	const BWEM::Base* base = stage_base();
	
	if (base != nullptr) {
		std::set<const BWEM::Area*> component_areas = base_state.connected_areas(base->GetArea(), base_state.controlled_and_planned_areas());
		Border border(component_areas);
		Position center = base->Center();
		key_value_vector<const BWEM::Area*,int> border_distances;
		for (auto& area : border.inside_areas()) {
			int distance = ground_distance(center, Position(area->Top()));
			if (distance >= 0) border_distances.emplace_back(area, distance);
		}
		const BWEM::Area* border_area = key_with_smallest_value(border_distances);
		if (border_area != nullptr) {
			std::vector<const BWEM::ChokePoint*> chokepoints;
			for (auto& area : border_area->AccessibleNeighbours()) {
				if (component_areas.count(area) == 0) {
					for (auto& cp : border_area->ChokePoints(area)) if (!cp.Blocked()) chokepoints.push_back(&cp);
				}
			}
			
			key_value_vector<const BWEM::ChokePoint*,int> chokepoint_width_map;
			for (auto& cp : chokepoints) chokepoint_width_map.emplace_back(cp, chokepoint_width(cp));
			
			stage_chokepoint_ = key_with_largest_value(chokepoint_width_map);
			stage_component_areas_ = component_areas;
		}
	}
}

void Strategy::update_stage_position()
{
	defense_decision_ = DefenseDecision::Minerals;
	stage_position_ = Positions::None;
	
	const BWEM::Base* base = stage_base();
	
	if (stage_chokepoint_ != nullptr) {
		const BWEM::ChokePoint* chokepoint = stage_chokepoint_;
		Position choke_center_position = chokepoint_center(chokepoint);
		Position walkable_choke_center_position = walkability_grid.walkable_tile_near(choke_center_position, 128);
		if (walkable_choke_center_position.isValid()) choke_center_position = walkable_choke_center_position;
		
		if (choke_center_position.isValid()) {
			defense_decision_ = determine_defense_decision(stage_component_areas_, choke_center_position);
			if (defense_decision_ == DefenseDecision::Minerals) {
				stage_position_ = calculate_mineral_center_base_stage_point(base);
			} else {
				stage_position_ = choke_center_position;
			}
		}
	}
	
	if (!stage_position_.isValid()) {
		if (base == nullptr) base = base_state.start_base();
		stage_position_ = calculate_mineral_center_base_stage_point(base);
	}
}

Strategy::DefenseDecision Strategy::determine_defense_decision(const std::set<const BWEM::Area*>& component_areas, Position choke_center_position)
{
	if (training_manager.unit_count_completed(UnitTypes::Protoss_Observer) == 0 &&
		building_manager.building_count(UnitTypes::Protoss_Photon_Cannon) > 0 &&
		std::any_of(information_manager.enemy_units().begin(), information_manager.enemy_units().end(),
					[](const auto& enemy_unit) {
						return enemy_unit->type == UnitTypes::Protoss_Dark_Templar;
					})) {
						return DefenseDecision::Minerals;
					}
	
	int worker_count = 0;
	bool at_minerals = false;
	bool zerg_opponent = (opponent_model.enemy_race() == Races::Zerg);
	bool dark_templar_outside = false;
	
	for (auto enemy_unit : information_manager.enemy_units()) {
		if (enemy_unit->unit->exists() &&
			enemy_unit->unit->isVisible() &&
			enemy_unit->type != UnitTypes::Terran_Medic &&
			(can_attack(enemy_unit->unit, false) || is_spellcaster(enemy_unit->type))) {
			const BWEM::Area* area = area_at(enemy_unit->position);
			if (area != nullptr && component_areas.count(area) > 0) {
				if (enemy_unit->type.isWorker()) {
					worker_count++;
					if (worker_count >= 3) at_minerals = true;
				} else {
					at_minerals = true;
				}
			} else {
				if (!enemy_unit->type.isWorker() && (!zerg_opponent || !is_melee(enemy_unit->type))) {
					bool can_attack_stage = false;
					int range = offense_max_range(enemy_unit->unit, false);
					if (range >= 0) {
						int distance = enemy_unit->unit->getDistance(choke_center_position);
						if (distance < 128 + range) can_attack_stage = true;
					}
					if (can_attack_stage) {
						at_minerals = true;
					}
				}
				if (enemy_unit->type == UnitTypes::Protoss_Dark_Templar) {
					dark_templar_outside = true;
				}
			}
		}
	}
	
	if (dark_templar_outside &&
		!tactics_manager.mobile_detection_exists()) {
		return DefenseDecision::BlockChokePointDarkTemplar;
	}
	
	return (at_minerals ? DefenseDecision::Minerals : DefenseDecision::BlockChokePoint);
}

Position Strategy::calculate_mineral_center_base_stage_point(const BWEM::Base* base)
{
	Position base_center = base->Center();
	Position mineral_center;
	int mineral_center_count = 0;
	for (auto& mineral : base->Minerals()) {
		mineral_center += edge_position(mineral->Type(), mineral->Pos(), base_center);
		mineral_center_count++;
	}
	
	Position result = Positions::None;
	if (mineral_center_count > 1) {
		mineral_center = mineral_center / mineral_center_count;
		Position base_edge_position = edge_position(Broodwar->self()->getRace().getResourceDepot(), base_center, mineral_center);
		Position candidate_position = (mineral_center + base_edge_position) / 2;
		result = walkability_grid.walkable_tile_near(candidate_position, 128);
	}
	
	if (!result.isValid()) result = walkability_grid.walkable_tile_near(base_center, 256);
	if (!result.isValid()) result = base_center;
	
	return result;
}

void Strategy::apply_stage_positions()
{
	update_stage_chokepoint();
	update_stage_position();
	
	Position stage_position = Positions::None;
	
	if (building_placement_manager.proxy_pylon_position().isValid()) {
		Position pylon_position = center_position_for(UnitTypes::Protoss_Pylon, building_placement_manager.proxy_pylon_position());
		stage_position = walkability_grid.walkable_tile_near(pylon_position, 256);
	}
	if (building_placement_manager.proxy_barracks_position().isValid()) {
		Position barracks_position = center_position_for(UnitTypes::Terran_Barracks, building_placement_manager.proxy_barracks_position());
		stage_position = walkability_grid.walkable_tile_near(barracks_position, 256);
	}
	
	if (!stage_position.isValid()) stage_position = stage_position_;
	
	for (auto& entry : micro_manager.combat_state()) {
		Position position = stage_position;
		if (type_specific_stage_.count(entry.first->getType()) > 0) {
			position = type_specific_stage_.at(entry.first->getType());
		}
		entry.second.set_stage_position(position);
		entry.second.set_block_chokepoint(false);
	}
	
	if (stage_position_.isValid() &&
		stage_chokepoint_ != nullptr &&
		(defense_decision_ == DefenseDecision::BlockChokePoint || defense_decision_ == DefenseDecision::BlockChokePointDarkTemplar)) {
		Gap gap(stage_chokepoint_);
		int gap_size = gap.size();
		std::vector<std::pair<Unit,int>> unit_distance_pairs;
		for (auto& entry : micro_manager.combat_state()) {
			int distance = entry.first->getDistance(stage_position_);
			if ((is_melee(entry.first->getType()) || defense_decision_ == DefenseDecision::BlockChokePointDarkTemplar) &&
				type_specific_stage_.count(entry.first->getType()) == 0 &&
				distance <= gap_size) {
				if (information_manager.all_units().at(entry.first).base_distance > 0) distance = -distance;
				unit_distance_pairs.emplace_back(entry.first, distance);
			}
		}
		
		UnitType tightness_unit_type;
		switch (opponent_model.enemy_race()) {
			case Races::Zerg:
				tightness_unit_type = UnitTypes::Zerg_Zergling;
				break;
			case Races::Terran:
				tightness_unit_type = UnitTypes::Terran_Marine;
				break;
			case Races::Protoss:
				tightness_unit_type = UnitTypes::Protoss_Zealot;
				break;
			default:
				tightness_unit_type = UnitTypes::Zerg_Zergling;
				break;
		}
		
		std::sort(unit_distance_pairs.begin(),
				  unit_distance_pairs.end(),
				  [](auto& a,auto& b){
					  return std::make_tuple(!is_melee(a.first->getType()), a.second, a.first->getID()) < std::make_tuple(!is_melee(b.first->getType()), b.second, b.first->getID());
				  });
		std::vector<UnitType> types;
		types.reserve(unit_distance_pairs.size());
		for (auto& entry : unit_distance_pairs) types.push_back(entry.first->getType());
		std::vector<Position> positions = gap.block_positions(types, tightness_unit_type);
		
		if (!positions.empty()) {
			Position d1 = gap.b() - gap.a();
			
			std::sort(unit_distance_pairs.begin(),
					  unit_distance_pairs.begin() + positions.size(),
					  [&d1](auto& a,auto& b){
						  const auto inner_product = [&d1](auto& a)
						  {
							  return d1.x * a.x + d1.y * a.y;
						  };
						  
						  return std::make_tuple(inner_product(a.first->getPosition()), a.first->getID()) < std::make_tuple(inner_product(b.first->getPosition()), b.first->getID());
					  });
			
			Position d(-d1.y, d1.x);
			Position p1 = scale_line_segment(stage_position_, stage_position_ - d, 64);
			Position p2 = scale_line_segment(stage_position_, stage_position_ + d, 64);
			Position near_stage_position = Positions::None;
			if (contains(base_state.controlled_and_planned_areas(), area_at(p1))) {
				near_stage_position = p1;
			} else {
				near_stage_position = p2;
			}
			if (near_stage_position.isValid()) {
				for (auto& entry : micro_manager.combat_state()) {
					if (type_specific_stage_.count(entry.first->getType()) == 0) {
						entry.second.set_stage_position(near_stage_position);
						entry.second.set_block_chokepoint(false);
					}
				}
			}
			
			for (size_t i = 0; i < positions.size(); i++) {
				CombatState& combat_state = micro_manager.combat_state().at(unit_distance_pairs[i].first);
				combat_state.set_stage_position(positions[i]);
				combat_state.set_block_chokepoint(true);
			}
		}
	}
}

void Strategy::attack_enemy()
{
	bool detection_missing = opponent_model.cloaked_or_mine_present() && !tactics_manager.mobile_detection_exists();
	bool desperation_attack = desperation_attack_condition();
	
	Position attack_position = Positions::None;
	bool always_advance = maxed_out_ || desperation_attack;
	
	if ((attacking_ && !detection_missing) || always_advance) {
		if (micro_manager.runby_possible()) {
			attack_position = tactics_manager.enemy_start_position();
		}
		if (!attack_position.isValid()) {
			attack_position = tactics_manager.enemy_base_attack_position();
		}
	}
	
	for (auto& entry : micro_manager.combat_state()) {
		entry.second.set_target_position(attack_position);
		entry.second.set_always_advance(always_advance);
		entry.second.set_near_target_only(attack_near_target_only_);
	}
	
	if (initial_dark_templar_attack_) {
		Position dark_templar_attack_position = tactics_manager.enemy_start_position();
		if (!dark_templar_attack_position.isValid()) {
			dark_templar_attack_position = tactics_manager.enemy_base_attack_position();
		}
		for (auto& entry : micro_manager.combat_state()) {
			if (entry.first->getType() == UnitTypes::Protoss_Dark_Templar) {
				entry.second.set_target_position(dark_templar_attack_position);
			}
		}
	}
	
	if (!attacking_) attack_near_target_only_ = false;
	if (attack_near_target_only_ && attack_position.isValid()) {
		if (Broodwar->isVisible(TilePosition(attack_position))) {
			bool unit_near_target = false;
			for (auto& enemy_unit : information_manager.enemy_units()) {
				if (enemy_unit->position.isValid() && attack_position.getDistance(enemy_unit->position) <= 320) {
					unit_near_target = true;
				}
			}
			if (!unit_near_target) attack_near_target_only_ = false;
		}
	}
}

bool Strategy::desperation_attack_condition()
{
	bool no_workers = worker_manager.worker_map().empty();
	bool supply_blocked = Broodwar->self()->supplyUsed() >= Broodwar->self()->supplyTotal();
	bool can_not_build_worker = base_state.controlled_and_planned_bases().empty() || !MineralGas(Broodwar->self()).can_pay(Broodwar->self()->getRace().getWorker());
	
	return no_workers && (supply_blocked || can_not_build_worker);
}

void Strategy::apply_result(bool win)
{
	result_store.apply_result(opening(), opponent_model.enemy_opening_info(), win);
}

void Strategy::frame()
{
	base_state.set_skip_mineral_only(false);
	worker_manager.set_limit_refinery_workers(-1);
	worker_manager.set_force_refinery_workers(false);
	building_manager.set_automatic_supply(true);
	training_manager.set_automatic_supply(true);
	training_manager.set_worker_production(true);
	training_manager.set_worker_cut(false);
	micro_manager.set_prevent_high_templar_from_archon_warp_count(0);
	micro_manager.set_force_archon_warp(false);
	type_specific_stage_.clear();
	
	frame_inner();
	
	apply_stage_positions();
	update_maxed_out();
	attack_enemy();
}
