#include "LurkerManager.h"
#include "UnitUtil.h"

using namespace UAlbertaBot;

LurkerManager::LurkerManager()
{
}

void LurkerManager::executeMicro(const BWAPI::Unitset & targets)
{
	const BWAPI::Unitset & lurkers = getUnits();

	// figure out targets
	BWAPI::Unitset LurkerTargets;
	std::copy_if(targets.begin(), targets.end(), std::inserter(LurkerTargets, LurkerTargets.end()),
		[](BWAPI::Unit u)
	{ 
		return 
			u->isVisible() &&
			u->isDetected() &&
			!u->isFlying() &&
			!u->isLifted() &&
			u->getPosition().isValid() &&
			u->getType() != BWAPI::UnitTypes::Zerg_Larva &&
			u->getType() != BWAPI::UnitTypes::Zerg_Egg &&
			!u->isStasised();
	});

	int lurkerRange = BWAPI::UnitTypes::Zerg_Lurker.groundWeapon().maxRange();

	for (auto & lurker : lurkers)
	{
		// Special micro eg. storm dodge
		if (Micro::CheckSpecialCases(lurker)) continue;

		// Check if lurker is near choke
		bool lurkerNearChokepoint = false;
		for (const auto & area : BWEMmap.Areas())
		{
			for (const BWEM::ChokePoint * choke : area.ChokePoints())
			{
				if (lurker->getDistance(BWAPI::Position(choke->Center())) < 64)
				{
					lurkerNearChokepoint = true;
					break;
				}
			}
		}

		// Ignore targets when regrouping, only regroup if lurkers are dead
		if (order.getType() == SquadOrderTypes::Regroup &&
			BWAPI::Broodwar->self()->deadUnitCount(BWAPI::UnitTypes::Zerg_Lurker) > 0)
		{
			if (lurker->canUnburrow())
			{
				if (BWAPI::Broodwar->getFrameCount() % 200 == 0)
				{
					lurker->unburrow();
				}
			}
			else
			{
				//Micro::SmartMovePath(lurker, order.getPosition());
				Micro::SmartMove(lurker, order.getPosition());
			}
			continue;
		}

		// if the order is to attack or defend
		if (order.getType() == SquadOrderTypes::Attack || 
			order.getType() == SquadOrderTypes::Defend ||
			order.getType() == SquadOrderTypes::Regroup)
		{

			// find the best target for this lurker
			BWAPI::Unit target = getTarget(lurker, LurkerTargets);

			// if there are targets
			if (target)
			{
				if (target && Config::Debug::DrawUnitTargetInfo)
				{
					BWAPI::Broodwar->drawLineMap(lurker->getPosition(), lurker->getTargetPosition(), BWAPI::Colors::Purple);
				}

				// if we are within range of a target, burrow
				int dist = lurker->getDistance(target);
				bool isThreat = !target->getType().isWorker() && UnitUtil::CanAttackGround(target);
				if ((isThreat && dist <= lurkerRange) || dist <= 64)
				{
					if (lurker->canBurrow() && !lurkerNearChokepoint)
					{
						lurker->burrow();
					}
				} 
				else 
				{
					if (lurker->canUnburrow()) 
					{
						if(BWAPI::Broodwar->getFrameCount() % 200 == 0)
						{
							lurker->unburrow();
						}
					}
					else
					{
						//Micro::SmartMovePath(lurker, target->getPosition());
						Micro::SmartMove(lurker, target->getPosition());
					}
				}

				// if we're burrowed, attack
				if (lurker->isBurrowed())
				{
					Micro::SmartAttackUnit(lurker, target);
				}
			}
			// if there are no targets
			else
			{
				// if we're not near the order position
				if (lurker->getDistance(order.getPosition()) > 96) 
				{
					if (lurker->canUnburrow()) 
					{
						if (BWAPI::Broodwar->getFrameCount() % 200 == 0)
						{
							lurker->unburrow();
						}
					} 
					else 
					{
						//Micro::SmartMovePath(lurker, order.getPosition());
						Micro::SmartMove(lurker, order.getPosition());
					}
				}
			}
		}
	}
}

BWAPI::Unit LurkerManager::getTarget(BWAPI::Unit lurker, const BWAPI::Unitset & targets)
{
	if (targets.empty())
	{
		return nullptr;
	}

	int lurkerRange = BWAPI::UnitTypes::Zerg_Lurker.groundWeapon().maxRange();

	BWAPI::Unitset targetsInRange;
	for (auto & target : targets)
	{
		if (target->getDistance(lurker) <= lurkerRange)
		{
			targetsInRange.insert(target);
		}
	}

	const BWAPI::Unitset & newTargets = targetsInRange.empty() ? targets : targetsInRange;

	int highPriority = 0;
	int closestDist = 99999;
	BWAPI::Unit bestTarget = nullptr;

	// check first for units that are in range of our attack that can cause damage
	// choose the highest priority one from them at the lowest health
	for (const auto & target : newTargets)
	{
		if (!UnitUtil::CanAttack(lurker, target))
		{
			continue;
		}

		int distance = lurker->getDistance(target);
		int priority = getAttackPriority(lurker, target);

		//BWAPI::Broodwar->drawTextMap(target->getPosition(), "%d", priority);

		if (!bestTarget || (priority > highPriority) || (priority == highPriority && distance < closestDist))
		{
			closestDist = distance;
			highPriority = priority;
			bestTarget = target;
		}
	}

	return bestTarget;
}

// get the attack priority of a type in relation to a zergling
int LurkerManager::getAttackPriority(BWAPI::Unit rangedUnit, BWAPI::Unit target)
{
	BWAPI::UnitType rangedType = rangedUnit->getType();
	BWAPI::UnitType targetType = target->getType();

	bool isThreat = rangedType.isFlyer() ? targetType.airWeapon() != BWAPI::WeaponTypes::None : targetType.groundWeapon() != BWAPI::WeaponTypes::None;

	if (target->getType().isWorker())
	{
		isThreat = false;
	}

	// Short circuit: Do not attack flying units or buildings
	if (target->isFlying() || target->isLifted())
	{
		return 0;
	}

	if (target->getType() == BWAPI::UnitTypes::Zerg_Larva || 
		target->getType() == BWAPI::UnitTypes::Zerg_Egg || 
		target->getType().isAddon())
	{
		return 0;
	}

	// if the target is building something near our base something is fishy
	BWAPI::Position ourBasePosition = BWAPI::Position(InformationManager::Instance().getMyMainBaseLocation()->Center());
	if (target->getType().isWorker() && 
		(target->isConstructing() || target->isRepairing()) && 
		target->getDistance(ourBasePosition) < 1200)
	{
		return 12;
	}
	if (target->getType().isBuilding() && 
		(target->isCompleted() || target->isBeingConstructed()) && 
		target->getDistance(ourBasePosition) < 1200)
	{
		return 12;
	}

	// highest priority is something that can attack us or aid in combat
	if (targetType == BWAPI::UnitTypes::Terran_Bunker || isThreat)
	{
		return 12;
	}
	// next priority is turrets
	else if (targetType == BWAPI::UnitTypes::Terran_Missile_Turret ||
		targetType == BWAPI::UnitTypes::Protoss_Forge)
	{
		return 10;
	}
	// next priority is worker
	if (targetType.isWorker())
	{
		if (target->isRepairing() || target->isConstructing() || unitNearChokepoint(target) ||
			target->isBraking() || !target->isMoving() || target->isGatheringMinerals() || target->isGatheringGas())
		{
			return 12;
		}
		return 12;
	}
	// next is special buildings
	else if (targetType == BWAPI::UnitTypes::Zerg_Spawning_Pool)
	{
		return 5;
	}
	// next is special buildings
	else if (targetType == BWAPI::UnitTypes::Protoss_Pylon)
	{
		return 5;
	}
	// next is buildings that cost gas
	else if (targetType.gasPrice() > 0)
	{
		return 4;
	}
	else if (targetType.mineralPrice() > 0)
	{
		return 3;
	}
	// then everything else
	else
	{
		return 1;
	}
}

