#pragma once

class Worker;
class WorkerOrder;

class WorkerAllocation
{
public:
	WorkerAllocation() {}
	WorkerAllocation(const std::vector<Unit>& minerals,
					 const std::vector<Unit>& refineries,
					 const std::map<Unit,Worker>& worker_map);
	bool minerals_not_saturated() const;
	Unit pick_mineral_for_worker(const Worker& worker);
	std::vector<Unit> minerals_with_worker_count(int worker_count);
	bool refineries_not_saturated() const;
	Unit pick_refinery_for_worker(const Worker& worker);
	std::vector<Unit> unsaturated_refineries();
	Unit WorkerAllocation::closest_unit(const std::vector<Unit>& units,Unit target_unit);
	int max_workers() const;
	double average_workers_per_mineral() const;
	int count_refinery_workers() const;
	
private:
	std::map<Unit,int> mineral_map_;
	std::map<Unit,int> refinery_map_;
};

class Worker {
public:
	Worker(Unit unit);
	
	Unit unit() const { return unit_; }
	const WorkerOrder* order() const { return order_.get(); }
	void apply_orders();
	void draw();
	
	void idle();
	void gather(Unit gather_target);
	void build(UnitType building_type,TilePosition building_position);
	void defend_base(const BWEM::Base *base);
	void defend_building(Unit building_unit);
	void defend_cannon_rush();
	void scout();
	void wait_at_proxy_location(Position position);
	void block_position(Position position);
private:
	Unit unit_;
	std::unique_ptr<WorkerOrder> order_;
};

class WorkerOrder {
public:
	WorkerOrder(Worker& worker) : worker_(worker), unit_(worker_.unit()) {}
	virtual ~WorkerOrder() {}
	
	virtual void apply_orders() = 0;
	virtual bool is_done() const = 0;
	virtual bool is_idle() const { return false; }
	virtual bool is_pullable() const { return false; }
	virtual bool is_defending() const { return false; }
	virtual bool is_scouting() const { return false; }
	virtual const BWEM::Base* scout_base() const { return nullptr; }
	virtual Unit gather_target() const { return nullptr; }
	virtual UnitType building_type() const { return UnitTypes::None; }
	virtual TilePosition building_position() const { return TilePositions::None; }
	virtual void draw() const {}
	virtual MineralGas reserved_resources() const { return MineralGas(); }
	virtual bool need_detection() const { return false; }
	virtual bool is_wait_at_proxy_location() const { return false; }
	virtual Position block_position() const { return Positions::None; }
	
protected:
	Worker& worker_;
	Unit unit_;
	
	bool repel_enemy_units() const;
};

class BlockPositionWorkerOrder : public WorkerOrder {
public:
	BlockPositionWorkerOrder(Worker& worker,Position position) : WorkerOrder(worker), position_(position) {}
	
	virtual void apply_orders() override;
	virtual bool is_done() const override { return false; }
	virtual Position block_position() const override { return position_; }
	
private:
	Position position_;
};

class BuildWorkerOrder : public WorkerOrder {
public:
	BuildWorkerOrder(Worker& worker,UnitType building_type,TilePosition building_position) : WorkerOrder(worker), building_type_(building_type), building_position_(building_position) {}
	
	virtual UnitType building_type() const override { return building_type_; }
	virtual TilePosition building_position() const override { return building_position_; }
	virtual bool is_done() const override;
	virtual void apply_orders() override;
	virtual void draw() const override;
	virtual MineralGas reserved_resources() const override;
	virtual bool need_detection() const override { return need_detection_; }
	
private:
	UnitType building_type_;
	TilePosition building_position_;
	int build_timeout_frame_ = -1;
	bool constructing_seen_true_ = false;
	bool failed_ = false;
	bool need_detection_ = false;
	
	std::vector<Unit> find_colliding_mines() const;
	bool building_exists() const;
	bool building_site_unobstructed() const;
};

class DefendBaseOrder : public WorkerOrder {
public:
	DefendBaseOrder(Worker& worker,const BWEM::Base* base) : WorkerOrder(worker), base_(base) {}
	
	virtual void apply_orders() override;
	virtual bool is_done() const override { return done_; }
	virtual bool is_defending() const override { return true; }
	
private:
	const BWEM::Base* base_;
	bool done_ = false;
};

class DefendBuildingOrder : public WorkerOrder {
public:
	DefendBuildingOrder(Worker& worker,Unit building_unit) : WorkerOrder(worker), building_unit_(building_unit) {}
	
	virtual void apply_orders() override;
	virtual bool is_done() const override { return done_; }
	virtual bool is_defending() const override { return true; }
	
private:
	Unit building_unit_;
	bool done_ = false;
};

class DefendCannonRushOrder : public WorkerOrder {
public:
	DefendCannonRushOrder(Worker& worker) : WorkerOrder(worker) {}
	
	virtual void apply_orders() override;
	virtual bool is_done() const override { return done_; }
	virtual bool is_defending() const override { return true; }
	
	static bool should_apply_cannon_rush_defense(const InformationUnit& information_unit);
private:
	bool done_ = false;
};

class GatherWorkerOrder : public WorkerOrder {
public:
	GatherWorkerOrder(Worker& worker,Unit gather_target) : WorkerOrder(worker), gather_target_(gather_target) {}
	
	virtual Unit gather_target() const override { return gather_target_; }
	virtual bool is_done() const override;
	virtual bool is_pullable() const override;
	virtual void apply_orders() override;
	
private:
	Unit gather_target_;
	bool return_started_ = false;
	
	bool is_carrying() const;
	bool is_gathering() const;
	bool is_base_valid() const;
};

class IdleWorkerOrder : public WorkerOrder {
public:
	IdleWorkerOrder(Worker& worker) : WorkerOrder(worker) {}
	~IdleWorkerOrder() {}
	virtual void apply_orders() override;
	virtual bool is_done() const override { return false; }
	virtual bool is_idle() const override { return true; }
	virtual bool is_pullable() const override { return true; }
	
private:
	bool move_to_closest_safe_base() const;
};

class ScoutWorkerOrder : public WorkerOrder {
public:
	ScoutWorkerOrder(Worker& worker);
	~ScoutWorkerOrder() {}
	virtual void apply_orders() override;
	virtual bool is_done() const override;
	virtual bool is_scouting() const override { return true; }
	virtual const BWEM::Base* scout_base() const override { return current_target_base_; }
private:
	enum class Mode
	{
		SearchBase,
		MoveAround,
		LookAtNatural,
		Done
	};
	
	static constexpr double kMoveAroundAngle = (15.0 / 180.0) * M_PI;
	static constexpr double kSafeMoveAroundAngle = (20.0 / 180.0) * M_PI;
	
	const BWEM::Base* current_target_base_;
	Mode mode_;
	bool clockwise_ = true;
	
	bool should_look_at_natural() const;
	bool enemy_has_non_start_resource_depot() const;
	bool other_scout_found_base() const;
	void move_around();
	bool walkable_line(Position position);
	std::tuple<const InformationUnit*,int> determine_danger(Position position);
	static bool is_defense_building(const InformationUnit* information_unit);
	bool safe_move(Position target_position);
	const BWEM::Base* closest_undiscovered_starting_base() const;
	std::set<const BWEM::Base*> undisovered_starting_bases() const;
};

class WaitAtProxyLocationWorkerOrder : public WorkerOrder {
public:
	WaitAtProxyLocationWorkerOrder(Worker& worker,Position proxy_location) : WorkerOrder(worker), proxy_location_(proxy_location) {}
	~WaitAtProxyLocationWorkerOrder() {}
	virtual void apply_orders() override;
	virtual bool is_done() const override { return false; }
	virtual bool is_pullable() const override { return true; }
	virtual bool is_wait_at_proxy_location() const override { return true; }
	
private:
	Position proxy_location_;
};

class WorkerManager : public Singleton<WorkerManager>
{
public:
	static constexpr int kWorkerCap = 70;
	
	void before();
	bool assign_building_request(UnitType building_type,TilePosition building_position);
	void send_initial_scout();
	void send_second_scout();
	void stop_scouting();
	bool is_scouting();
	void send_proxy_builder(Position proxy_position);
	void block_positions(const std::vector<Position>& positions);
	void end_block_positions();
	void defend_building(Unit building_unit,int max_defending_workers);
	void apply_worker_orders();
	void draw_for_workers();
	void onUnitLost(Unit unit);
	void cancel_building_construction_for_type(UnitType building_type);
	
	const std::map<Unit,Worker>& worker_map() { return worker_map_; }
	int determine_max_workers() const { return std::min(worker_allocation_.max_workers(), kWorkerCap); }
	bool minerals_not_saturated() const { return worker_allocation_.minerals_not_saturated(); }
	bool refineries_not_saturated() const { return worker_allocation_.refineries_not_saturated(); }
	double average_workers_per_mineral() const { return worker_allocation_.average_workers_per_mineral(); }
	bool has_idle_workers() const;
	MineralGas worker_reserved_mineral_gas() const;
	Position initial_scout_death_position() const { return initial_scout_death_position_; }
	bool allow_mining_from(const BWEM::Base* base) { return base_disallow_mining_until_frame_.count(base) == 0; }
	int limit_refinery_workers() const { return limit_refinery_workers_; }
	void set_limit_refinery_workers(int limit_refinery_workers) { limit_refinery_workers_ = limit_refinery_workers; }
	bool force_refinery_workers() const { return force_refinery_workers_; }
	void set_force_refinery_workers(bool force_refinery_workers) { force_refinery_workers_ = force_refinery_workers; }
	int lost_worker_count() const { return lost_worker_count_; }
	
private:
	std::map<Unit,Worker> worker_map_;
	WorkerAllocation worker_allocation_;
	bool sent_initial_scout_ = false;
	bool sent_second_scout_ = false;
	Position initial_scout_death_position_ = Positions::None;
	std::map<const BWEM::Base*,int> base_disallow_mining_until_frame_;
	std::map<const BWEM::Base*,int> base_unsafe_until_frame_;
	std::map<const BWEM::Base*,int> base_defended_until_frame_;
	int limit_refinery_workers_ = -1;
	bool force_refinery_workers_ = false;
	int lost_worker_count_ = 0;
	
	void register_new_workers();
	void done_workers_to_idle();
	void update_base_disallow_mining();
	void update_base_unsafety();
	void update_worker_allocation();
	void update_worker_gather_orders();
	void update_worker_defend_orders();
	bool assign_worker_to_gas();
	void defend_base_with_workers_if_needed();
	void defend_cannon_rush();
	bool send_scout();
	bool check_proxy_builder_time(Position worker_position,Position proxy_position);
	Worker* pick_worker_for_task();
	Worker* pick_worker_for_task(Position position);
};
