#include "BananaBrain.h"

WorkerAllocation::WorkerAllocation(const std::vector<Unit>& minerals,
								   const std::vector<Unit>& refineries,
								   const std::map<Unit,Worker>& worker_map)
{
	for (auto& unit : minerals) mineral_map_[unit] = 0;
	for (auto& unit : refineries) refinery_map_[unit] = 0;
	
	for (auto& entry : worker_map) {
		const Worker& worker = entry.second;
		Unit gather_target = worker.order()->gather_target();
		if (mineral_map_.count(gather_target) > 0) mineral_map_[gather_target]++;
		if (refinery_map_.count(gather_target) > 0) refinery_map_[gather_target]++;
	}
}

bool WorkerAllocation::minerals_not_saturated() const
{
	for (auto& entry : mineral_map_) if (entry.second < 3) return true;
	return false;
}

Unit WorkerAllocation::pick_mineral_for_worker(const Worker& worker)
{
	Unit worker_unit = worker.unit();
	Unit result = closest_unit(minerals_with_worker_count(0), worker_unit);
	if (result == nullptr) result = closest_unit(minerals_with_worker_count(1), worker_unit);
	if (result == nullptr) result = closest_unit(minerals_with_worker_count(2), worker_unit);
	if (result != nullptr) mineral_map_[result]++;
	return result;
}

std::vector<Unit> WorkerAllocation::minerals_with_worker_count(int worker_count)
{
	std::vector<Unit> result;
	for (auto& entry : mineral_map_) if (entry.second <= worker_count) result.push_back(entry.first);
	return result;
}

bool WorkerAllocation::refineries_not_saturated() const
{
	for (auto& entry : refinery_map_) if (entry.second < 3) return true;
	return false;
}

Unit WorkerAllocation::pick_refinery_for_worker(const Worker& worker)
{
	Unit worker_unit = worker.unit();
	Unit result = closest_unit(unsaturated_refineries(), worker_unit);
	if (result != nullptr) refinery_map_[result]++;
	return result;
}

std::vector<Unit> WorkerAllocation::unsaturated_refineries()
{
	std::vector<Unit> result;
	for (auto& entry : refinery_map_) if (entry.second < 3) result.push_back(entry.first);
	return result;
}

Unit WorkerAllocation::closest_unit(const std::vector<Unit>& units,Unit target_unit)
{
	std::map<Unit,int> distances;
	for (auto unit : units) {
		int distance = ground_distance(target_unit->getPosition(), unit->getPosition());
		if (distance > 0) distances[unit] = distance;
	}
	return key_with_smallest_value(distances);
}

int WorkerAllocation::max_workers() const
{
	return 3 * (mineral_map_.size() + refinery_map_.size());
}

double WorkerAllocation::average_workers_per_mineral() const
{
	if (mineral_map_.empty()) return 0.0;
	double sum = 0.0;
	for (auto& entry : mineral_map_) sum += entry.second;
	return sum / mineral_map_.size();
}

int WorkerAllocation::count_refinery_workers() const
{
	int count = 0;
	for (auto& entry : refinery_map_) count += entry.second;
	return count;
}

Worker::Worker(Unit unit) : unit_(unit), order_(new IdleWorkerOrder(*this))
{
}

void Worker::apply_orders()
{
	order_->apply_orders();
}

void Worker::draw()
{
	order_->draw();
}

void Worker::idle()
{
	order_.reset(new IdleWorkerOrder(*this));
}

void Worker::gather(Unit gather_target)
{
	order_.reset(new GatherWorkerOrder(*this, gather_target));
}

void Worker::build(UnitType building_type,TilePosition building_position)
{
	order_.reset(new BuildWorkerOrder(*this, building_type, building_position));
}

void Worker::defend_base(const BWEM::Base *base)
{
	order_.reset(new DefendBaseOrder(*this, base));
}

void Worker::defend_building(Unit building_unit)
{
	order_.reset(new DefendBuildingOrder(*this, building_unit));
}

void Worker::defend_cannon_rush()
{
	order_.reset(new DefendCannonRushOrder(*this));
}

void Worker::scout()
{
	order_.reset(new ScoutWorkerOrder(*this));
}

void Worker::wait_at_proxy_location(Position position)
{
	order_.reset(new WaitAtProxyLocationWorkerOrder(*this, position));
}

void Worker::block_position(Position position)
{
	order_.reset(new BlockPositionWorkerOrder(*this, position));
}

bool WorkerOrder::repel_enemy_units() const
{
	std::vector<Unit> all_enemy_units;
	for (auto& unit : Broodwar->enemy()->getUnits()) {
		if (unit->isVisible() && unit->isCompleted()) all_enemy_units.push_back(unit);
	}
	bool order_applied = unit_potential(unit_, [&all_enemy_units](UnitPotential& potential){
		potential.repel_units(all_enemy_units);
	});
	return order_applied;
}

void BlockPositionWorkerOrder::apply_orders()
{
	if (unit_->getPosition() != position_) {
		path_finder.execute_path(unit_, position_, [this](){
			unit_move(unit_, position_);
		});
	} else {
		if (!unit_->isHoldingPosition()) unit_->holdPosition();
	}
}

bool BuildWorkerOrder::is_done() const
{
	return failed_ || (constructing_seen_true_ && !unit_->isConstructing() && building_exists());
}

void BuildWorkerOrder::apply_orders()
{
	if (!constructing_seen_true_ && unit_->isConstructing()) constructing_seen_true_ = true;
	
	std::vector<Unit> mines = find_colliding_mines();
	Unit closest_mine = smallest_priority(mines, [this](Unit mine){ return unit_->getDistance(mine); });
	if (closest_mine != nullptr) {
		unit_attack(unit_, closest_mine);
	} else {
		if (!constructing_seen_true_) {
			Position center = center_position_for(building_type_, building_position_);
			int distance = unit_->getDistance(center);
			if (distance > unit_->getType().sightRange() / 2) {
				failed_ = !path_finder.execute_path(unit_, center, [this,center]() {
					unit_move(unit_, center);
				});
			} else {
				if (building_site_unobstructed()) {
					build_timeout_frame_ = -1;
					unit_->build(building_type_, building_position_);
				} else {
					if (build_timeout_frame_ == -1) {
						build_timeout_frame_ = Broodwar->getFrameCount() + BuildingManager::kBuildingPlacementFailedRetryFrames;
					} else {
						if (Broodwar->getFrameCount() >= build_timeout_frame_) failed_ = true;
					}
				}
			}
		}
		
		if (constructing_seen_true_ && !unit_->isConstructing() && !building_exists()) {
			if (building_site_unobstructed()) need_detection_ = true;
			constructing_seen_true_ = false;
		}
	}
}

void BuildWorkerOrder::draw() const
{
	Position position(building_position_);
	Broodwar->drawBoxMap(position.x, position.y, position.x + building_type_.tileWidth() * 32, position.y + building_type_.tileHeight() * 32, Colors::White);
}

MineralGas BuildWorkerOrder::reserved_resources() const
{
	return MineralGas(building_type_);
}

std::vector<Unit> BuildWorkerOrder::find_colliding_mines() const
{
	std::vector<Unit> result;
	
	Position center = center_position_for(building_type_, building_position_);
	int left = center.x - building_type_.dimensionLeft() - UnitTypes::Terran_Vulture_Spider_Mine.dimensionRight();
	int top = center.y - building_type_.dimensionUp() - UnitTypes::Terran_Vulture_Spider_Mine.dimensionDown();
	int right = center.x + building_type_.dimensionRight() + UnitTypes::Terran_Vulture_Spider_Mine.dimensionLeft();
	int bottom = center.y + building_type_.dimensionDown() + UnitTypes::Terran_Vulture_Spider_Mine.dimensionUp();
	
	for (auto& other_unit : Broodwar->enemy()->getUnits()) {
		if (other_unit->getType() == UnitTypes::Terran_Vulture_Spider_Mine &&
			other_unit->getPosition().x >= left && other_unit->getPosition().x <= right &&
			other_unit->getPosition().y >= top && other_unit->getPosition().y <= bottom) {
			result.push_back(other_unit);
		}
	}
	
	return result;
}

bool BuildWorkerOrder::building_exists() const
{
	return std::any_of(Broodwar->self()->getUnits().begin(), Broodwar->self()->getUnits().end(), [this](auto& unit) {
		return unit->getTilePosition() == building_position_ && unit->getType() == building_type_;
	});
}

bool BuildWorkerOrder::building_site_unobstructed() const
{
	return Broodwar->canBuildHere(building_position_, building_type_, unit_, true);
}

void DefendBaseOrder::apply_orders()
{
	Position center = center_position_for(Broodwar->self()->getRace().getResourceDepot(), base_->Location());
	Unitset unit_set = Broodwar->getUnitsInRadius(center, 320);
	std::map<Unit,int> distance_map;
	for (auto& target : unit_set) {
		if (target->getPlayer() == Broodwar->enemy() &&
			!target->isFlying() &&
			not_cloaked(target) &&
			(can_attack(target, false) || is_spellcaster(target->getType()))) {
			distance_map[target] = unit_->getDistance(target);
		}
	}
	Unit target = key_with_smallest_value(distance_map);
	if (target != nullptr) {
		unit_attack(unit_, target);
	} else {
		done_ = true;
	}
}

void DefendBuildingOrder::apply_orders()
{
	std::map<Unit,int> distance_map;
	for (auto& enemy_unit : Broodwar->enemy()->getUnits()) {
		if (building_unit_->getDistance(enemy_unit) <= 224 &&
			!enemy_unit->isFlying() &&
			not_cloaked(enemy_unit) &&
			(can_attack(enemy_unit, false) || is_spellcaster(enemy_unit->getType()))) {
			distance_map[enemy_unit] = unit_->getDistance(enemy_unit);
		}
	}
	Unit target = key_with_smallest_value(distance_map);
	if (target != nullptr) {
		unit_attack(unit_, target);
	} else {
		done_ = true;
	}
}

void DefendCannonRushOrder::apply_orders()
{
	Position center = center_position_for(Broodwar->self()->getRace().getResourceDepot(), Broodwar->self()->getStartLocation());
	std::map<const InformationUnit*,DistanceWithPriority> distance_position_map;
	for (auto& entry : information_manager.enemy_units()) {
		if (should_apply_cannon_rush_defense(entry.second)) {
			int distance = center.getApproxDistance(entry.second.position);
			bool completed = entry.second.unit->exists() && entry.second.unit->isCompleted();
			int priority = completed ? 0 : 1;
			distance_position_map[&entry.second] = DistanceWithPriority(distance, priority, entry.second.unit->getID());
		}
	}
	const InformationUnit* target = key_with_smallest_value(distance_position_map);
	if (target != nullptr) {
		path_finder.execute_path(unit_, target->position, [&,target]{
			if (target->unit->exists()) {
				unit_attack(unit_, target->unit);
			} else {
				unit_move(unit_, target->position);
			}
		});
	} else {
		done_ = true;
	}
}

bool DefendCannonRushOrder::should_apply_cannon_rush_defense(const InformationUnit& information_unit)
{
	Position center = center_position_for(Broodwar->self()->getRace().getResourceDepot(), Broodwar->self()->getStartLocation());
	return (information_unit.type == UnitTypes::Protoss_Photon_Cannon &&
			calculate_distance(UnitTypes::Protoss_Photon_Cannon, information_unit.position, center) <= 320 + WeaponTypes::STS_Photon_Cannon.maxRange() &&
			ground_height(information_unit.position) >= ground_height(center));
}

bool GatherWorkerOrder::is_done() const
{
	return (!gather_target_->exists() || return_started_ || !is_base_valid()) && !is_carrying();
}

bool GatherWorkerOrder::is_pullable() const
{
	return !is_carrying();
}

void GatherWorkerOrder::apply_orders()
{
	if (!is_carrying()) {
		if (!is_gathering() || !unit_has_target(unit_, gather_target_)) {
			path_finder.execute_path(unit_, gather_target_->getPosition(), [this](){
				if (!unit_has_target(unit_, gather_target_)) unit_->gather(gather_target_);
			});
		}
	} else {
		if (!is_gathering()) {
			unit_->returnCargo();
		}
		return_started_ = true;
	}
}

bool GatherWorkerOrder::is_carrying() const
{
	return unit_->isCarryingGas() || unit_->isCarryingMinerals();
}

bool GatherWorkerOrder::is_gathering() const
{
	return unit_->isGatheringGas() || unit_->isGatheringMinerals();
}

bool GatherWorkerOrder::is_base_valid() const
{
	for (auto base : base_state.controlled_bases()) {
		if (worker_manager.allow_mining_from(base)) {
			bool mineral_found = std::any_of(base->Minerals().begin(), base->Minerals().end(), [this](auto mineral){
				return mineral->Unit() == gather_target_;
			});
			if (mineral_found) return true;
			bool geyser_found = std::any_of(base->Geysers().begin(), base->Geysers().end(), [this](auto geyser){
				return geyser->Unit() == gather_target_;
			});
			if (geyser_found) return true;
		}
	}
	return false;
}

void IdleWorkerOrder::apply_orders()
{
	bool order_applied = repel_enemy_units();
	if (!order_applied) order_applied = move_to_closest_safe_base();
	if (!order_applied) random_move(unit_);
}

bool IdleWorkerOrder::move_to_closest_safe_base() const
{
	Position position = Positions::None;
	std::map<const BWEM::Base*,int> distance_map;
	for (auto base : base_state.controlled_bases()) {
		if (worker_manager.allow_mining_from(base)) {
			Position center = center_position_for(Broodwar->self()->getRace().getResourceDepot(),
												  base->Location());
			int distance = ground_distance(center, unit_->getPosition());
			if (distance >= 0) distance_map[base] = distance;
		}
	}
	const BWEM::Base* base = key_with_smallest_value(distance_map);
	if (base != nullptr) {
		position = center_position_for(Broodwar->self()->getRace().getResourceDepot(),
									   base->Location());
	}
	bool result = false;
	if (position.isValid()) {
		if (calculate_distance(unit_->getType(), unit_->getPosition(),
							   Broodwar->self()->getRace().getResourceDepot(), position) <= 32) {
			if (!unit_->isHoldingPosition()) unit_->holdPosition();
		} else {
			path_finder.execute_path(unit_, position, [this,position](){
				unit_move(unit_, position);
			});
		}
		result = true;
	}
	return result;
}

ScoutWorkerOrder::ScoutWorkerOrder(Worker& worker) : WorkerOrder(worker)
{
	current_target_base_ = closest_undiscovered_starting_base();
	mode_ = (current_target_base_ == nullptr) ? Mode::Done : Mode::SearchBase;
}

void ScoutWorkerOrder::apply_orders()
{
	switch (mode_) {
		case Mode::SearchBase:
			if (current_target_base_ == tactics_manager.enemy_start_base()) {
				mode_ = Mode::MoveAround;
			} else {
				if (undisovered_starting_bases().count(current_target_base_) == 0) {
					current_target_base_ = closest_undiscovered_starting_base();
				}
				
				if (current_target_base_ != nullptr) {
					Position position = center_position_for(Broodwar->self()->getRace().getResourceDepot(),
															current_target_base_->Location());
					path_finder.execute_path(unit_, position, [this,position](){
						unit_move(unit_, position);
					});
				} else {
					mode_ = Mode::Done;
				}
			}
			break;
		case Mode::MoveAround:
			move_around();
			if (should_look_at_natural()) mode_ = Mode::LookAtNatural;
			if (!is_hp_undamaged(unit_)) mode_ = Mode::Done;
			break;
		case Mode::LookAtNatural: {
			Position position = tactics_manager.enemy_natural_base()->Center();
			path_finder.execute_path(unit_, position, [this,position](){
				unit_move(unit_, position);
			});
			if (calculate_distance(opponent_model.enemy_race().getResourceDepot(), position, unit_->getType(), unit_->getPosition()) <= 16 && opponent_model.enemy_natural_sufficiently_scouted()) {
				mode_ = Mode::MoveAround;
			}
			if (opponent_model.enemy_opening() != EnemyOpening::Unknown) mode_ = Mode::MoveAround;
			if (!is_hp_undamaged(unit_)) mode_ = Mode::Done;
			break; }
		case Mode::Done:
			break;
	}
}

bool ScoutWorkerOrder::should_look_at_natural() const
{
	return (opponent_model.enemy_opening() == EnemyOpening::Unknown &&
			opponent_model.enemy_base_sufficiently_scouted() &&
			!opponent_model.enemy_natural_sufficiently_scouted() &&
			information_manager.enemy_count(UnitTypes::Terran_Barracks) <= 1 &&
			!information_manager.enemy_exists(UnitTypes::Terran_Factory) &&
			!information_manager.enemy_exists(UnitTypes::Protoss_Gateway) &&
			!information_manager.enemy_exists(UnitTypes::Protoss_Forge) &&
			!information_manager.enemy_exists(UnitTypes::Protoss_Cybernetics_Core) &&
			!information_manager.enemy_exists(UnitTypes::Zerg_Spawning_Pool) &&
			!enemy_has_non_start_resource_depot() &&
			Broodwar->getFrameCount() >= opponent_model.enemy_latest_expansion_frame());
}

bool ScoutWorkerOrder::enemy_has_non_start_resource_depot() const
{
	bool result = false;
	for (auto& entry : information_manager.enemy_units()) {
		const InformationUnit& unit = entry.second;
		if (unit.type.isResourceDepot()) {
			const BWEM::Base* base = base_state.base_for_tile_position(unit.tile_position());
			if (base == nullptr || !base->Starting()) {
				result = true;
				break;
			}
		}
	}
	return result;
}

bool ScoutWorkerOrder::is_done() const
{
	return (mode_ == Mode::Done ||
			other_scout_found_base() ||
			opponent_model.enemy_opening() == EnemyOpening::Z_4_5Pool ||
			opponent_model.enemy_opening() == EnemyOpening::Z_9PoolSpeed ||
			opponent_model.enemy_opening() == EnemyOpening::T_BBS ||
			opponent_model.enemy_opening() == EnemyOpening::P_CannonRush ||
			opponent_model.enemy_opening() == EnemyOpening::P_ProxyGate ||
			opponent_model.enemy_opening() == EnemyOpening::P_2GateFast);
}

bool ScoutWorkerOrder::other_scout_found_base() const
{
	const BWEM::Base* enemy_start_base = tactics_manager.enemy_start_base();
	return (enemy_start_base != nullptr && current_target_base_ != enemy_start_base);
}

void ScoutWorkerOrder::move_around()
{
	if (current_target_base_ == nullptr) return;
	
	if (debug_options.draw_enabled()) Broodwar->drawTextMap(unit_->getPosition(), clockwise_ ? "CW" : "CCW");
	
	Position center_position = current_target_base_->Center();
	Position current_position = unit_->getPosition();
	
	Position move_position = scale_line_segment(center_position, current_position, 40 * 8);
	for (int i = 40; i > 0; i--) {
		Position position = scale_line_segment(center_position, current_position, i * 8);
		if (walkable_line(position)) {
			move_position = position;
			break;
		}
	}
	
	Position target_position = rotate_around(center_position, move_position, clockwise_ ? kMoveAroundAngle : -kMoveAroundAngle);
	bool found = false;
	
	for (int i = 0; i < 40; i++) {
		Position position = scale_line_segment(target_position, center_position, -i * 8);
		if (!position.isValid()) continue;
		const InformationUnit* danger_unit;
		int danger_distance;
		std::tie(danger_unit, danger_distance) = determine_danger(position);
		if (danger_distance >= 24 && walkable_line(position) && check_collision(unit_, position)) {
			target_position = position;
			found = true;
			break;
		}
	}
	
	if (!found) {
		for (int i = 1; i <= 20; i++) {
			Position position = scale_line_segment(target_position, center_position, i * 8);
			if (!position.isValid()) continue;
			const InformationUnit* danger_unit;
			int danger_distance;
			std::tie(danger_unit, danger_distance) = determine_danger(position);
			if (danger_distance >= 24 && walkable_line(position) && check_collision(unit_, position)) {
				target_position = position;
				found = true;
				break;
			}
		}
	}
	
	if (found) {
		if (debug_options.draw_enabled()) Broodwar->drawCircleMap(target_position, 5, Colors::White, true);
		if (!safe_move(target_position)) {
			unit_move(unit_, target_position);
		}
	} else {
		clockwise_ = !clockwise_;
		if (!repel_enemy_units()) {
			if (!unit_->isIdle()) unit_->stop();
		}
	}
}

bool ScoutWorkerOrder::walkable_line(Position position)
{
	return position.isValid() && ::walkable_line(FastWalkPosition(unit_->getPosition()), FastWalkPosition(position));
}

std::tuple<const InformationUnit*,int> ScoutWorkerOrder::determine_danger(Position position)
{
	const InformationUnit* danger_unit = nullptr;
	int danger_distance = INT_MAX;
	
	for (auto& entry : information_manager.enemy_units()) {
		int max_range = weapon_max_range(entry.second.type, Broodwar->enemy(), false);
		if (max_range >= 0) {
			int distance = calculate_distance(unit_->getType(), unit_->getPosition(), entry.second.type, entry.second.position) - max_range;
			if (distance < danger_distance) {
				danger_unit = &entry.second;
				danger_distance = distance;
			}
		}
	}
	
	return std::make_tuple(danger_unit, danger_distance);
}

bool ScoutWorkerOrder::is_defense_building(const InformationUnit* information_unit)
{
	if (information_unit == nullptr) return false;
	UnitType type = information_unit->type;
	return (type == UnitTypes::Terran_Bunker ||
			type == UnitTypes::Protoss_Photon_Cannon ||
			type == UnitTypes::Zerg_Sunken_Colony);
}

bool ScoutWorkerOrder::safe_move(Position target_position)
{
	const InformationUnit* danger_unit;
	int danger_distance;
	std::tie(danger_unit, danger_distance) = determine_danger(target_position);
	
	if (danger_distance > 3 * 32 || danger_unit == nullptr) {
		unit_move(unit_, target_position);
		return true;
	}
	
	Position a = danger_unit->position - unit_->getPosition();
	Position b = target_position - unit_->getPosition();
	int inner_product = a.x * b.x + a.y * b.y;
	
	if (inner_product < 0) {
		unit_move(unit_, target_position);
		return true;
	}
	
	int max_range = weapon_max_range(danger_unit->type, Broodwar->enemy(), false);
	Position back_position = scale_line_segment(danger_unit->position, unit_->getPosition(), max_range + 3 * 32);
	Position move_position = rotate_around(danger_unit->position, back_position, clockwise_ ? kSafeMoveAroundAngle : -kSafeMoveAroundAngle);
	if (!walkable_line(move_position)) return false;
	
	unit_move(unit_, move_position);
	if (debug_options.draw_enabled()) Broodwar->drawCircleMap(move_position, 5, Colors::Red, true);
	return true;
}

const BWEM::Base* ScoutWorkerOrder::closest_undiscovered_starting_base() const
{
	UnitType center_type = Broodwar->self()->getRace().getResourceDepot();
	std::map<const BWEM::Base*,int> base_distance;
	for (auto& base : undisovered_starting_bases()) {
		int distance = ground_distance(center_position_for(center_type, base->Location()), unit_->getPosition());
		if (distance >= 0) base_distance[base] = distance;
	}
	return key_with_smallest_value(base_distance);
}

std::set<const BWEM::Base*> ScoutWorkerOrder::undisovered_starting_bases() const
{
	std::set<const BWEM::Base*> result;
	for (auto base : base_state.unexplored_start_bases()) result.insert(base);
	for (auto& entry : worker_manager.worker_map()) {
		const BWEM::Base* base = entry.second.order()->scout_base();
		if (base != nullptr && base != current_target_base_) result.erase(base);
	}
	return result;
}

void WaitAtProxyLocationWorkerOrder::apply_orders()
{
	path_finder.execute_path(unit_, proxy_location_, [this](){
		unit_move(unit_, proxy_location_);
	});
}

void WorkerManager::before()
{
	register_new_workers();
	done_workers_to_idle();
	update_base_disallow_mining();
	update_base_unsafety();
	update_worker_defend_orders();
	update_worker_allocation();
}

void WorkerManager::register_new_workers()
{
	for (Unit unit : Broodwar->self()->getUnits()) {
		if (unit->getType().isWorker() && unit->isCompleted() && worker_map_.count(unit) == 0) {
			worker_map_.emplace(unit, unit);
		}
	}
}

void WorkerManager::done_workers_to_idle()
{
	for (auto &entry : worker_map_) {
		Worker& worker = entry.second;
		if (worker.order()->is_done()) worker.idle();
	}
}

void WorkerManager::update_base_disallow_mining()
{
	for (auto& base : base_state.controlled_bases()) {
		Position center = center_position_for(Broodwar->self()->getRace().getResourceDepot(), base->Location());
		Unitset unit_set = Broodwar->getUnitsInRadius(center, 384);
		bool gathering_worker_under_attack = false;
		std::vector<Unit> gathering_workers;
		std::vector<Unit> resource_depots;
		std::vector<Unit> enemies;
		for (auto& unit : unit_set) {
			if (unit->getPlayer() == Broodwar->self()) {
				if (unit->getType().isWorker() && (unit->isGatheringMinerals() || unit->isGatheringGas())) {
					if (unit->isUnderAttack()) gathering_worker_under_attack = true;
					gathering_workers.push_back(unit);
				}
				if (unit->getType().isResourceDepot()) resource_depots.push_back(unit);
			} else if (unit->getPlayer() == Broodwar->enemy() &&
					   ((can_attack(unit, false) || is_spellcaster(unit->getType())) && unit->getType() != UnitTypes::Terran_Medic)) {
				enemies.push_back(unit);
			}
		}
		bool disallow_mining = gathering_worker_under_attack;
		if (!disallow_mining) {
			for (auto& resource_depot : resource_depots) {
				if (calculate_frames_to_live(resource_depot, enemies) <= 52.0) {
					disallow_mining = true;
					break;
				}
			}
		}
		if (!disallow_mining) {
			for (auto& enemy : enemies) {
				UnitType enemy_type = enemy->getType();
				if (enemy_type == UnitTypes::Protoss_High_Templar || enemy_type == UnitTypes::Protoss_Reaver ||
					enemy_type == UnitTypes::Terran_Siege_Tank_Tank_Mode || enemy_type == UnitTypes::Terran_Siege_Tank_Siege_Mode) {
					for (auto& worker : gathering_workers) {
						if (enemy->getDistance(worker) <= offense_max_range(enemy, false) + 32) {
							disallow_mining = true;
							break;
						}
					}
				}
			}
		}
		if (disallow_mining) base_disallow_mining_until_frame_[base] = Broodwar->getFrameCount() + 96;
	}
	
	std::vector<const BWEM::Base*> expired_base_disallow_mining;
	for (auto& entry : base_disallow_mining_until_frame_) if (entry.second <= Broodwar->getFrameCount()) expired_base_disallow_mining.push_back(entry.first);
	for (auto& base : expired_base_disallow_mining) base_disallow_mining_until_frame_.erase(base);
}

void WorkerManager::update_base_unsafety()
{
	for (auto& base : base_state.controlled_bases()) {
		Position center = center_position_for(Broodwar->self()->getRace().getResourceDepot(), base->Location());
		Unitset unit_set = Broodwar->getUnitsInRadius(center, 320);
		bool under_attack_found = false;
		bool enemy_found = false;
		for (auto& unit : unit_set) {
			if (unit->getPlayer() == Broodwar->self() && unit->isUnderAttack() &&
				(unit->getType().isWorker() || unit->getType() == UnitTypes::Protoss_Pylon || !is_hp_undamaged(unit))) under_attack_found = true;
			if (unit->getPlayer() == Broodwar->enemy() &&
				((can_attack(unit, false) || is_spellcaster(unit->getType())) && unit->getType() != UnitTypes::Terran_Medic)) enemy_found = true;
		}
		if (under_attack_found || (base_unsafe_until_frame_.count(base) > 0 && enemy_found)) base_unsafe_until_frame_[base] = Broodwar->getFrameCount() + 96;
	}
	
	std::vector<const BWEM::Base*> expired_base_unsafety;
	for (auto& entry : base_unsafe_until_frame_) if (entry.second <= Broodwar->getFrameCount()) expired_base_unsafety.push_back(entry.first);
	for (auto& base : expired_base_unsafety) base_unsafe_until_frame_.erase(base);
}

void WorkerManager::update_worker_allocation()
{
	std::map<TilePosition,Unit> tile_position_to_refinery_map;
	for (auto& unit : Broodwar->self()->getUnits()) {
		if (unit->getType().isRefinery() && unit->isCompleted()) {
			tile_position_to_refinery_map[unit->getTilePosition()] = unit;
		}
	}
	
	std::vector<Unit> minerals;
	std::vector<Unit> refineries;
	for (auto& base : base_state.controlled_bases()) {
		if (allow_mining_from(base)) {
			for (auto& mineral : base->Minerals()) {
				Unit unit = mineral->Unit();
				if (unit->exists()) minerals.push_back(unit);
			}
			for (auto& geyser : base->Geysers()) {
				auto it = tile_position_to_refinery_map.find(geyser->TopLeft());
				if (it != tile_position_to_refinery_map.end()) refineries.push_back(it->second);
			}
		}
	}
	worker_allocation_ = WorkerAllocation(minerals, refineries, worker_map_);
}

void WorkerManager::update_worker_gather_orders()
{
	for (auto& entry : worker_map_) {
		Worker& worker = entry.second;
		if (worker.order()->is_idle()) {
			Unit resource;
			if (assign_worker_to_gas()) {
				resource = worker_allocation_.pick_refinery_for_worker(worker);
				if (resource == nullptr) resource = worker_allocation_.pick_mineral_for_worker(worker);
			} else {
				resource = worker_allocation_.pick_mineral_for_worker(worker);
				if (resource == nullptr) resource = worker_allocation_.pick_refinery_for_worker(worker);
			}
			
			if (resource != nullptr) worker.gather(resource);
		}
	}
}

void WorkerManager::update_worker_defend_orders()
{
	defend_base_with_workers_if_needed();
	defend_cannon_rush();
}

void WorkerManager::defend_base_with_workers_if_needed()
{
	const BWEM::Base* main_base = base_state.main_base();
	
	for (auto base : base_state.controlled_bases()) {
		if (base_unsafe_until_frame_.count(base) > 0) {
			Position center = center_position_for(Broodwar->self()->getRace().getResourceDepot(), base->Location());
			Unitset unit_set = Broodwar->getUnitsInRadius(center, 320);
			bool defend_with_workers = false;
			int hostile_workers = 0;
			int hostile_attackable_combat_units = 0;
			int hostile_unattackable_combat_units = 0;
			Position hostile_position_sum = Position{0, 0};
			int hostile_position_count = 0;
			int defending_ground_units = 0;
			for (auto& target : unit_set) {
				if (target->getPlayer() == Broodwar->enemy() &&
					(can_attack(target, false) || is_spellcaster(target->getType()))) {
					if (target->getType().isWorker()) {
						hostile_workers++;
						hostile_position_sum += target->getPosition();
						hostile_position_count++;
					} else if (!target->isFlying() && not_cloaked(target) &&
							   (is_melee(target->getType()) || Broodwar->enemy()->topSpeed(target->getType()) <= Broodwar->self()->topSpeed(Broodwar->self()->getRace().getWorker()))) {
						hostile_attackable_combat_units++;
						hostile_position_sum += target->getPosition();
						hostile_position_count++;
					} else {
						hostile_unattackable_combat_units++;
					}
				}
				if (target->getPlayer() == Broodwar->self() &&
					target->isCompleted() &&
					!target->getType().isWorker() &&
					target->getType() != UnitTypes::Protoss_Corsair &&
					!target->getType().isBuilding()) {
					defending_ground_units++;
				}
			}
			bool multiple_defending_ground_units = (defending_ground_units > 1);
			if (base == main_base) {
				defend_with_workers = (hostile_workers + hostile_attackable_combat_units > 0 &&
									   hostile_unattackable_combat_units == 0 &&
									   !multiple_defending_ground_units);
			} else {
				defend_with_workers = (hostile_workers > 0 &&
									   hostile_attackable_combat_units == 0 &&
									   hostile_unattackable_combat_units == 0 &&
									   !multiple_defending_ground_units);
			}
			
			bool mark_base_defended = false;
			if (defend_with_workers) {
				int max_defending_workers = std::min(8, 4 * hostile_attackable_combat_units + (hostile_workers > 0 ? std::max(4, 1 + hostile_workers) : 0));
				mark_base_defended = true;
				int defending_workers = 0;
				for (auto& entry : worker_map_) {
					Worker& worker = entry.second;
					if (unit_set.count(worker.unit()) > 0 &&
						worker.order()->is_defending()) defending_workers++;
				}
				std::vector<Worker*> available_workers;
				for (auto& entry : worker_map_) {
					Worker& worker = entry.second;
					if (unit_set.count(worker.unit()) > 0 &&
						!worker.order()->is_defending() &&
						worker.order()->is_pullable()) {
						available_workers.push_back(&worker);
					}
				}
				if (hostile_position_count != 0) {
					Position hostile_position_average = Position{hostile_position_sum.x / hostile_position_count, hostile_position_sum.y / hostile_position_count};
					std::sort(available_workers.begin(), available_workers.end(), [hostile_position_average](auto& a,auto& b){
						return std::make_tuple(a->unit()->getDistance(hostile_position_average)) < std::make_tuple(b->unit()->getDistance(hostile_position_average));
					});
				}
				for (auto& worker : available_workers) {
					if (defending_workers >= max_defending_workers) break;
					worker->defend_base(base);
					defending_workers++;
				}
			} else if (multiple_defending_ground_units && base == main_base) {
				mark_base_defended = true;
			}
			
			if (mark_base_defended) base_defended_until_frame_[base] = Broodwar->getFrameCount() + 32;
		}
	}
	
	std::vector<const BWEM::Base*> expired_base_defense;
	for (auto& entry : base_defended_until_frame_) {
		base_disallow_mining_until_frame_.erase(entry.first);
		base_unsafe_until_frame_.erase(entry.first);
		if (entry.second <= Broodwar->getFrameCount()) expired_base_defense.push_back(entry.first);
	}
	for (auto& base : expired_base_defense) base_defended_until_frame_.erase(base);
}

void WorkerManager::defend_cannon_rush()
{
	int current_army_supply_without_corsairs = army_supply() - 2 * training_manager.unit_count_completed(UnitTypes::Protoss_Corsair);
	if (current_army_supply_without_corsairs <= 2 &&
		Broodwar->getFrameCount() <= 5 * UnitTypes::Protoss_Probe.buildTime() + UnitTypes::Protoss_Gateway.buildTime() + 2 * UnitTypes::Protoss_Zealot.buildTime()) {
		Position center = center_position_for(Broodwar->self()->getRace().getResourceDepot(), Broodwar->self()->getStartLocation());
		Unitset unit_set = Broodwar->getUnitsInRadius(center, 320);
		
		int proxy_count = 0;
		for (auto& entry : information_manager.enemy_units()) {
			if (DefendCannonRushOrder::should_apply_cannon_rush_defense(entry.second)) {
				proxy_count++;
			}
		}
		
		if (proxy_count > 0) {
			int max_defending_workers = std::min(8, 4 * proxy_count);
			int defending_workers = 0;
			for (auto& entry : worker_map_) {
				Worker& worker = entry.second;
				if (worker.order()->is_defending()) defending_workers++;
			}
			for (auto& entry : worker_map_) {
				Worker& worker = entry.second;
				if (!worker.order()->is_defending() && worker.order()->is_pullable() && unit_set.contains(worker.unit())) {
					worker.defend_cannon_rush();
					defending_workers++;
				}
				if (defending_workers >= max_defending_workers) break;
			}
		}
	}
}

bool WorkerManager::assign_worker_to_gas()
{
	if (limit_refinery_workers_ >= 0 && worker_allocation_.count_refinery_workers() >= limit_refinery_workers_) return false;
	if (force_refinery_workers_ || worker_allocation_.average_workers_per_mineral() >= 1.0) return true;
	
	// TODO Balance mineral/gas mining
	MineralGas spendable = spending_manager.spendable();
	if (spendable.minerals < 0 || spendable.gas < 0) return !spending_manager.spendable_out_of_minerals_first();
	
	CostPerMinute total_cost_per_minute = spending_manager.training_cost_per_minute();
	if (total_cost_per_minute.minerals == 0.0) return false;
	
	MineralGas income_per_minute = spending_manager.income_per_minute();
	double mineral_factor = income_per_minute.minerals / total_cost_per_minute.minerals;
	double gas_factor = total_cost_per_minute.gas == 0.0 ? INFINITY : income_per_minute.gas / total_cost_per_minute.gas;
	
	return gas_factor < mineral_factor;
}

bool WorkerManager::assign_building_request(UnitType building_type,TilePosition building_position)
{
	Worker* worker = pick_worker_for_task(center_position_for(building_type, building_position));
	if (worker != nullptr) {
		worker->build(building_type, building_position);
		return true;
	} else {
		return false;
	}
}

void WorkerManager::send_initial_scout()
{
	if (!sent_initial_scout_) {
		sent_initial_scout_ = send_scout();
	}
}

void WorkerManager::send_second_scout()
{
	if (!sent_second_scout_) {
		sent_second_scout_ = send_scout();
	}
}

bool WorkerManager::send_scout()
{
	bool result = false;
	UnitType center_type = Broodwar->self()->getRace().getResourceDepot();
	Position start_position = center_position_for(center_type, Broodwar->self()->getStartLocation());
	std::set<const BWEM::Base*> bases_already_being_scouted;
	for (auto &entry : worker_map_) {
		const BWEM::Base* base = entry.second.order()->scout_base();
		if (base != nullptr) bases_already_being_scouted.insert(base);
	}
	std::map<Position,int> undiscovered_base_distance_map;
	for (auto& base : base_state.unexplored_start_bases()) {
		if (bases_already_being_scouted.count(base) == 0) {
			Position position = center_position_for(center_type, base->Location());
			int distance = ground_distance(position, start_position);
			if (distance >= 0) undiscovered_base_distance_map[position] = distance;
		}
	}
	Position position = key_with_smallest_value(undiscovered_base_distance_map, Positions::None);
	if (position.isValid()) {
		Worker* worker = pick_worker_for_task(position);
		if (worker != nullptr) {
			worker->scout();
			result = true;
		}
	}
	return result;
}

void WorkerManager::stop_scouting()
{
	for (auto& entry : worker_map_) {
		Worker& worker = entry.second;
		if (worker.order()->is_scouting()) worker.idle();
	}
}

bool WorkerManager::is_scouting()
{
	if (!sent_initial_scout_) return false;
	for (auto& entry : worker_map_) {
		Worker& worker = entry.second;
		if (worker.order()->is_scouting()) return true;
	}
	return false;
}

void WorkerManager::send_proxy_builder(Position proxy_position)
{
	bool already_assigned = std::any_of(worker_map_.begin(), worker_map_.end(), [](auto& entry){
		return entry.second.order()->is_wait_at_proxy_location();
	});
	if (already_assigned) return;
	
	std::map<Worker*,int> distances;
	for (auto& entry : worker_map_) {
		Worker& worker = entry.second;
		if (worker.order()->is_pullable() ||
			worker.order()->building_position().isValid() ||
			worker.order()->block_position().isValid()) {
			int distance = ground_distance(proxy_position, worker.unit()->getPosition());
			if (distance >= 0) distances[&worker] = distance;
		}
	}
	Worker* worker = key_with_smallest_value(distances);
	if (worker != nullptr &&
		!worker->order()->building_position().isValid() &&
		check_proxy_builder_time(worker->unit()->getPosition(), proxy_position)) {
		worker->wait_at_proxy_location(proxy_position);
	}
}

void WorkerManager::block_positions(const std::vector<Position>& positions)
{
	std::set<Position> already_blocked;
	for (auto& entry : worker_map_) {
		Worker& worker = entry.second;
		Position position = worker.order()->block_position();
		if (position.isValid()) already_blocked.insert(position);
	}
	
	for (auto& position : positions) {
		if (already_blocked.count(position) == 0) {
			Worker* worker = pick_worker_for_task(position);
			if (worker != nullptr) {
				worker->block_position(position);
			} else {
				break;
			}
		}
	}
}

void WorkerManager::end_block_positions()
{
	for (auto& entry : worker_map_) {
		Worker& worker = entry.second;
		if (worker.order()->block_position().isValid()) worker.idle();
	}
}

void WorkerManager::defend_building(Unit building_unit,int max_defending_workers)
{
	Unitset unit_set = Broodwar->getUnitsInRadius(building_unit->getPosition(), 320);
	int defending_workers = 0;
	for (auto& entry : worker_map_) {
		Worker& worker = entry.second;
		if (unit_set.count(worker.unit()) > 0 &&
			worker.order()->is_defending()) defending_workers++;
	}
	std::vector<Worker*> available_workers;
	for (auto& entry : worker_map_) {
		Worker& worker = entry.second;
		if (unit_set.count(worker.unit()) > 0 &&
			!worker.order()->is_defending() &&
			worker.order()->is_pullable()) {
			available_workers.push_back(&worker);
		}
	}
	for (auto& worker : available_workers) {
		if (defending_workers >= max_defending_workers) break;
		worker->defend_building(building_unit);
		defending_workers++;
	}
}

bool WorkerManager::check_proxy_builder_time(Position worker_position,Position proxy_position)
{
	int arrive_frame = 4 * UnitTypes::Protoss_Probe.buildTime();
	int current_frame = Broodwar->getFrameCount();
	if (current_frame >= arrive_frame) return true;
	
	int distance = ground_distance(worker_position, proxy_position);
	if (distance < 0) return false;
	
	int travel_frames = (int)(1.15 * distance / UnitTypes::Protoss_Probe.topSpeed());
	return current_frame + travel_frames >= arrive_frame;
}

Worker* WorkerManager::pick_worker_for_task()
{
	for (auto& entry : worker_map_) {
		Worker& worker = entry.second;
		if (worker.order()->is_idle()) return &worker;
	}
	for (auto& entry : worker_map_) {
		Worker& worker = entry.second;
		if (worker.order()->is_pullable()) return &worker;
	}
	return nullptr;
}

Worker* WorkerManager::pick_worker_for_task(Position position)
{
	std::map<Worker*,int> distances;
	for (auto& entry : worker_map_) {
		Worker& worker = entry.second;
		if (worker.order()->is_pullable()) {
			int distance = ground_distance(position, worker.unit()->getPosition());
			if (distance >= 0) distances[&worker] = distance;
		}
	}
	return key_with_smallest_value(distances);
}

void WorkerManager::apply_worker_orders()
{
	update_worker_gather_orders();
	
	for (auto& entry : worker_map_) {
		Worker& worker = entry.second;
		worker.apply_orders();
	}
}

void WorkerManager::draw_for_workers()
{
	for (auto& entry : worker_map_) {
		Worker& worker = entry.second;
		worker.draw();
	}
}

void WorkerManager::onUnitLost(Unit unit)
{
	if (worker_map_.count(unit) > 0) {
		const WorkerOrder* order = worker_map_.at(unit).order();
		if (order->is_scouting()) initial_scout_death_position_ = unit->getPosition();
		lost_worker_count_++;
		for (auto &base : base_state.controlled_bases()) {
			Position center = center_position_for(Broodwar->self()->getRace().getResourceDepot(), base->Location());
			int distance = unit->getPosition().getApproxDistance(center);
			if (distance <= 256) {
				base_disallow_mining_until_frame_[base] = Broodwar->getFrameCount() + 96;
				if (order->gather_target() != nullptr && order->gather_target()->getDistance(unit) <= 256) {
					base_unsafe_until_frame_[base] = Broodwar->getFrameCount() + 96;
				}
			}
		}
		worker_map_.erase(unit);
	}
}

bool WorkerManager::has_idle_workers() const
{
	for (auto& entry : worker_map_) if (entry.second.order()->is_idle()) return true;
	return false;
}

MineralGas WorkerManager::worker_reserved_mineral_gas() const
{
	MineralGas result;
	for (auto& entry : worker_map_) {
		const Worker& worker = entry.second;
		result += worker.order()->reserved_resources();
	}
	return result;
}

void WorkerManager::cancel_building_construction_for_type(UnitType building_type)
{
	for (auto& entry : worker_map_) {
		Worker& worker = entry.second;
		if (worker.order()->building_type() == building_type) worker.idle();
	}
}
