#pragma once

struct EnemyCluster
{
	static constexpr int kEngagementDistance = 64;
	static constexpr int kFrontDistance = 256;
	
	std::vector<const InformationUnit*> units;
	std::vector<std::pair<Unit,int>> friendly_units_ascending_distance;
	int defense_supply;
	int offense_supply;
	int high_ground_supply;
	std::set<Unit> front_units;
	int front_supply;
	
	bool is_engaged() const;
	bool expect_win() const;
	bool expect_win(Unit unit) const;
	bool in_front(Unit unit) const;
	
private:
	bool expect_win(int enemy_supply) const;
	void determine_front();
	void calculate_high_ground_supply();
	bool higher_ground(Position a,Position b);
	int terrain_height(const BWEM::Area* area);
	
	friend class TacticsManager;
};

class TacticsManager : public Singleton<TacticsManager>
{
public:
	static constexpr int kClusterDistance = 512;
	
	void update();
	void draw();
	
	int enemy_worker_supply() const { return enemy_worker_supply_; }
	int enemy_army_supply() const { return enemy_army_supply_; }
	int enemy_defense_supply() const { return enemy_defense_supply_; }
	int enemy_air_supply() const { return enemy_air_supply_; }
	int max_enemy_army_supply() const { return max_enemy_army_supply_; }
	int enemy_offense_supply() const { return enemy_offense_supply_; }
	const std::vector<EnemyCluster>& clusters() const { return clusters_; }
	const EnemyCluster* cluster_for_unit(Unit unit) const { return get_or_default(cluster_map_, unit); }
	int cluster_engagement_distance(Unit unit,const EnemyCluster* cluster) const { return get_or_default(cluster_engagement_distance_, std::make_pair(unit, cluster), INT_MAX); }
	EnemyCluster* cluster_at_position(Position position);
	
	static int defense_supply_equivalent(Unit unit);
	static int defense_supply_equivalent(const InformationUnit& information_unit,bool runby_possible=false);
	static int defense_supply_equivalent(UnitType type,bool runby_possible=false);
	
	const BWEM::Base* enemy_start_base() const { return enemy_start_base_; }
	int enemy_start_base_found_at_frame() const { return enemy_start_base_found_at_frame_; }
	const BWEM::Base* enemy_natural_base() const { return enemy_natural_base_; }
	
	Position enemy_start_position() const;
	Position enemy_base_attack_position(bool guess_start_location=true) const;
	std::vector<const BWEM::Base*> possible_enemy_start_bases() const;
private:
	int enemy_worker_supply_ = 0;
	int enemy_army_supply_ = 0;
	int enemy_defense_supply_ = 0;
	int enemy_air_supply_ = 0;
	int max_enemy_army_supply_ = 0;
	int enemy_offense_supply_;
	std::vector<EnemyCluster> clusters_;
	std::map<Unit,EnemyCluster*> cluster_map_;
	std::map<std::pair<Unit,const EnemyCluster*>,int> cluster_engagement_distance_;
	const BWEM::Base* enemy_start_base_ = nullptr;
	int enemy_start_base_found_at_frame_ = -1;
	const BWEM::Base* enemy_natural_base_ = nullptr;
	
	void update_enemy_base();
	void update_enemy_supply();
	void update_clusters();
	void update_cluster_engagement_distances();
	Position deduce_enemy_start_position() const;
};
