#include "BananaBrain.h"

bool EnemyCluster::is_engaged() const
{
	return !friendly_units_ascending_distance.empty() && friendly_units_ascending_distance[0].second <= kEngagementDistance;
}

bool EnemyCluster::expect_win() const
{
	return expect_win(defense_supply);
}

bool EnemyCluster::expect_win(int enemy_supply) const
{
	bool result;
	
	if (enemy_supply == 0) {
		result = true;
	} else {
		if (is_engaged()) {
			result = (front_supply > 4 * enemy_supply / 5);
		} else {
			result = (front_supply > enemy_supply);
		}
	}
	
	return result;
}

bool EnemyCluster::expect_win(Unit unit) const
{
	bool ranged = !unit->getType().isWorker() && !is_melee(unit->getType());
	return ranged ? expect_win(high_ground_supply) : expect_win();
}

bool EnemyCluster::in_front(Unit unit) const
{
	return front_units.count(unit) > 0;
}

void EnemyCluster::determine_front()
{
	std::vector<std::pair<Unit,int>> units;
	for (auto& entry : friendly_units_ascending_distance) {
		if (can_attack(entry.first)) {
			units.push_back(entry);
		}
	}
	
	size_t start = 0;
	while (start < units.size() && units[start].first->getType().isBuilding()) start++;
	
	size_t end = start;
	if (start < units.size()) {
		int first_distance = units[start].second;
		if (first_distance < kFrontDistance) {
			int max_stride = 64;
			for (size_t i = start + 1; i <= units.size(); i++) {
				int distance = units[i - 1].second;
				if (distance == INT_MAX) break;
				int stride = distance - first_distance;
				if (stride <= max_stride) end = i;
				max_stride += 32;
			}
		}
		while (start > 0 && units[start].second - units[start - 1].second <= 64) start--;
	}
	
	int supply = 0;
	for (size_t i = start; i < end; i++) {
		Unit unit = units[i].first;
		front_units.insert(unit);
		if (!unit->getType().isWorker()) {
			supply += unit->getType().supplyRequired();
			supply += TacticsManager::defense_supply_equivalent(unit);
		}
	}
	front_supply = (supply + 1) / 2;
}

void EnemyCluster::calculate_high_ground_supply()
{
	size_t start = 0;
	while (start < friendly_units_ascending_distance.size() &&
		   !friendly_units_ascending_distance[start].first->getType().isBuilding() &&
		   !friendly_units_ascending_distance[start].first->getType().isWorker() &&
		   !is_melee(friendly_units_ascending_distance[start].first->getType())) start++;
	
	int supply = 0;
	if (start < friendly_units_ascending_distance.size()) {
		Position start_position = friendly_units_ascending_distance[start].first->getPosition();
		for (auto& enemy_unit : units) {
			if (enemy_unit->position.isValid() &&
				!enemy_unit->type.isWorker() &&
				!higher_ground(start_position, enemy_unit->position)) {
				supply += enemy_unit->type.supplyRequired();
				supply += TacticsManager::defense_supply_equivalent(*enemy_unit);
			}
		}
	}
	high_ground_supply = (supply + 1) / 2;
}

bool EnemyCluster::higher_ground(Position a,Position b)
{
	const BWEM::Area* area_a = area_at(a);
	const BWEM::Area* area_b = area_at(b);
	if (area_a == nullptr || area_b == nullptr) return false;
	int distance = -1;
	const BWEM::CPPath& path = bwem_map.GetPath(a, b, &distance);
	if (path.empty() || distance < 0) return false;
	
	const BWEM::Area* area_next = nullptr;
	const BWEM::ChokePoint* cp = path[0];
	if (cp->GetAreas().first == area_a) {
		area_next = cp->GetAreas().second;
	} else if (cp->GetAreas().second == area_a) {
		area_next = cp->GetAreas().first;
	}
	if (area_next == nullptr) return false;
	
	return terrain_height(area_a) > terrain_height(area_next);
}

int EnemyCluster::terrain_height(const BWEM::Area* area)
{
	if (area->VeryHighGroundPercentage() >= 90) return 3;
	if (area->VeryHighGroundPercentage() + area->HighGroundPercentage() >= 90) return 2;
	return 1;
}

void TacticsManager::update()
{
	update_enemy_base();
	update_enemy_supply();
	update_clusters();
	update_cluster_engagement_distances();
}

void TacticsManager::draw()
{
	int cluster_index = 0;
	for (auto& cluster : clusters_) {
		cluster_index++;
		for (auto& enemy_unit : cluster.units) {
			Broodwar->drawTextMap(enemy_unit->position + Position(0, 10), "C%d", cluster_index);
		}
		Broodwar->drawTextScreen(300, 10 + cluster_index * 10, "C%d: %d/%d/%d/%d/%s",
								 cluster_index,
								 cluster.defense_supply, cluster.offense_supply, cluster.front_supply, cluster.high_ground_supply,
								 cluster.expect_win() ? "advance" : "retreat");
	}
	
	Position position = enemy_start_position();
	if (position.isValid()) Broodwar->drawCircleMap(position, 20, Colors::White, true);
	if (enemy_natural_base_ != nullptr) draw_filled_diamond_map(enemy_natural_base_->Center(), 20, Colors::White);
}

void TacticsManager::update_enemy_base()
{
	if (enemy_start_base_ == nullptr) {
		for (auto& unit : Broodwar->enemy()->getUnits()) {
			if (unit->getType().isResourceDepot()) {
				const BWEM::Base* base = base_state.base_for_tile_position(unit->getTilePosition());
				if (base != nullptr && base->Starting()) {
					enemy_start_base_ = base;
					enemy_start_base_found_at_frame_ = Broodwar->getFrameCount();
					enemy_natural_base_ = base_state.natural_base_for_start_base(base);
				}
			}
		}
	}
}

void TacticsManager::update_enemy_supply()
{
	bool runby_possible = micro_manager.bunker_runby_possible();
	int worker_supply = 0;
	int army_supply = 0;
	int air_supply = 0;
	int defense_supply = 0;
	int offense_worker_supply = 0;
	int offense_army_supply = 0;
	for (auto& entry : information_manager.enemy_units()) {
		if (!entry.second.is_stasised()) {
			int supply = entry.second.type.supplyRequired();
			if (entry.second.type.isWorker()) worker_supply += supply; else army_supply += supply;
			if (entry.second.type.isFlyer()) air_supply += supply;
			if (entry.second.base_distance <= 320) {
				if (entry.second.type.isWorker()) offense_worker_supply += supply; else offense_army_supply += supply;
			}
			defense_supply = defense_supply_equivalent(entry.second, runby_possible);
		}
	}
	enemy_worker_supply_ = (worker_supply + 1) / 2;
	enemy_army_supply_ = (army_supply + 1) / 2;
	enemy_defense_supply_ = (army_supply + defense_supply + 1) / 2;
	enemy_air_supply_ = (air_supply + 1) / 2;
	max_enemy_army_supply_ = std::max(max_enemy_army_supply_, enemy_army_supply_);
	int offense_supply = offense_army_supply + std::max(0, offense_worker_supply - 2);
	enemy_offense_supply_ = (offense_supply + 1) / 2;
}

void TacticsManager::update_clusters()
{
	clusters_.clear();
	cluster_map_.clear();
	
	std::set<const InformationUnit*> unassigned;
	for (auto& entry : information_manager.enemy_units()) {
		if (!entry.second.is_stasised()) {
			unassigned.insert(&(entry.second));
		}
	}
	
	while (!unassigned.empty()) {
		const InformationUnit* first = *(unassigned.begin());
		unassigned.erase(first);
		
		EnemyCluster cluster;
		std::set<const InformationUnit*> todo;
		todo.insert(first);
		
		while (!todo.empty()) {
			const InformationUnit* current = *(todo.begin());
			todo.erase(current);
			unassigned.erase(current);
			
			for (auto& candidate : unassigned) {
				if (todo.count(candidate) == 0 &&
					current->position.getApproxDistance(candidate->position) <= kClusterDistance) todo.insert(candidate);
			}
			
			cluster.units.push_back(current);
		}
		
		clusters_.push_back(cluster);
	}
	
	bool runby_possible = micro_manager.bunker_runby_possible();
	for (auto& cluster : clusters_) {
		int worker_supply = 0;
		int army_supply = 0;
		int defense_supply = 0;
		for (auto& enemy_unit : cluster.units) {
			cluster_map_[enemy_unit->unit] = &cluster;
			int supply = enemy_unit->type.supplyRequired();
			if (enemy_unit->type.isWorker()) worker_supply += supply; else army_supply += supply;
			defense_supply += defense_supply_equivalent(*enemy_unit, runby_possible);
		}
		//cluster.offense_supply = (army_supply + std::max(0, worker_supply - 2) + 1) / 2;	// @ When do we include enemy worker supply?
		cluster.offense_supply = (army_supply + 1) / 2;
		cluster.defense_supply = (army_supply + defense_supply + 1) / 2;
	}
}

void TacticsManager::update_cluster_engagement_distances()
{
	cluster_engagement_distance_.clear();
	
	for (auto unit : Broodwar->self()->getUnits()) {
		if (unit->isCompleted() && unit->isVisible() && !unit->isStasised() && !unit->isLoaded()) {
			for (auto& cluster : clusters_) {
				int min_engagement_distance = INT_MAX;
				for (auto& enemy_unit : cluster.units) {
					if (!enemy_unit->position.isValid()) continue;
					int distance = calculate_distance(unit->getType(), unit->getPosition(), enemy_unit->type, enemy_unit->position);
					int enemy_unit_range = offense_max_range(enemy_unit->type, Broodwar->enemy(), unit->isFlying());
					if (enemy_unit_range >= 0) {
						int engagement_distance;
						if (enemy_unit->flying) {
							engagement_distance = distance - enemy_unit_range;
						} else {
							if (distance <= enemy_unit_range) {
								engagement_distance = 0;
							} else if (distance - enemy_unit_range > 512) {	// @ Performance workaround
								engagement_distance = distance - enemy_unit_range;
							} else {
								engagement_distance = ground_distance_to_bwcircle(enemy_unit->position, enemy_unit_range, unit->getPosition());
							}
						}
						min_engagement_distance = std::min(min_engagement_distance, std::max(0, engagement_distance));
					}
				}
				if (min_engagement_distance != INT_MAX) cluster_engagement_distance_[std::make_pair(unit, &cluster)] = min_engagement_distance;
			}
		}
	}
	
	for (auto& cluster : clusters_) {
		for (auto unit : Broodwar->self()->getUnits()) {
			cluster.friendly_units_ascending_distance.push_back(std::make_pair(unit, cluster_engagement_distance(unit, &cluster)));
		}
		std::sort(cluster.friendly_units_ascending_distance.begin(), cluster.friendly_units_ascending_distance.end(), [](auto& a,auto& b){
			return (std::make_tuple(a.second, a.first->getID()) <
					std::make_tuple(b.second, b.first->getID()));
		});
		cluster.determine_front();
		cluster.calculate_high_ground_supply();
	}
}

EnemyCluster* TacticsManager::cluster_at_position(Position position)
{
	int min_distance = INT_MAX;
	EnemyCluster* result = nullptr;
	
	for (auto& cluster : clusters_) {
		for (auto& enemy_unit : cluster.units) {
			int distance = position.getApproxDistance(enemy_unit->position);
			if (distance < min_distance) {
				min_distance = distance;
				result = &cluster;
			}
		}
	}
	
	return (min_distance <= kClusterDistance) ? result : nullptr;
}


int TacticsManager::defense_supply_equivalent(Unit unit)
{
	int result;
	if (unit->getType() == UnitTypes::Terran_Bunker) {
		result = 4 + 2 * information_manager.bunker_marines_loaded(unit);
	} else {
		result = defense_supply_equivalent(unit->getType());
	}
	return result;
}

int TacticsManager::defense_supply_equivalent(const InformationUnit& information_unit,bool runby_possible)
{
	int result;
	if (information_unit.type == UnitTypes::Terran_Bunker) {
		result = (runby_possible ? 0 : 4) + 2 * information_manager.bunker_marines_loaded(information_unit.unit);
	} else {
		result = defense_supply_equivalent(information_unit.type, runby_possible);
	}
	return result;
}

int TacticsManager::defense_supply_equivalent(UnitType type,bool runby_possible)
{
	int result;
	switch (type) {
		case UnitTypes::Protoss_Photon_Cannon:
		result = 3;
		break;
		case UnitTypes::Zerg_Sunken_Colony:
		result = 3;
		break;
		case UnitTypes::Terran_Bunker:
		result = runby_possible ? 2 : 6;
		break;
		case UnitTypes::Terran_Siege_Tank_Siege_Mode:
		result = 2;
		break;
		default:
		result = 0;
		break;
	}
	return result * 2;
}

Position TacticsManager::enemy_start_position() const
{
	if (enemy_start_base_ == nullptr) {
		Position position = deduce_enemy_start_position();
		if (position.isValid()) return position;
	} else {
		for (auto& entry : information_manager.enemy_units()) {
			if (entry.second.type.isResourceDepot() && !entry.second.flying &&
				entry.second.tile_position() == enemy_start_base_->Location()) {
				return entry.second.position;
			}
		}
	}
	
	return Positions::None;
}

Position TacticsManager::enemy_base_attack_position(bool guess_start_location) const
{
	for (auto& entry : information_manager.enemy_units()) {
		if (entry.second.type.isResourceDepot() && !entry.second.flying) return entry.second.position;
	}
	
	if (enemy_start_base_ == nullptr) {
		Position position = deduce_enemy_start_position();
		if (position.isValid()) return position;
	}
	
	for (auto& entry : information_manager.enemy_units()) {
		if (entry.second.type.isBuilding() && !entry.second.flying) return entry.second.position;
	}
	
	for (auto& entry : information_manager.enemy_units()) {
		if (entry.second.type.isBuilding()) return entry.second.position;
	}
	
	if (enemy_start_base_ == nullptr && guess_start_location) {
		for (auto& base : base_state.unexplored_start_bases()) {
			if (base_state.controlled_and_planned_bases().count(base) == 0) {
				return base->Center();
			}
		}
	}
	
	return Positions::Unknown;
}

std::vector<const BWEM::Base*> TacticsManager::possible_enemy_start_bases() const
{
	std::vector<const BWEM::Base*> result;
	
	if (enemy_start_base_ != nullptr) {
		result.push_back(enemy_start_base_);
	} else {
		for (auto& base : base_state.unexplored_start_bases()) {
			if (base_state.controlled_and_planned_bases().count(base) == 0) {
				result.push_back(base);
			}
		}
	}
	
	return result;
}

Position TacticsManager::deduce_enemy_start_position() const
{
	std::vector<const BWEM::Base*> start_bases;
	for (auto& base : base_state.bases()) {
		if (base->Starting()) {
			start_bases.push_back(base);
		}
	}
	
	std::set<const BWEM::Base*> candidate_bases;
	for (auto& base : base_state.unexplored_start_bases()) {
		if (base_state.controlled_and_planned_bases().count(base) == 0) {
			candidate_bases.insert(base);
		}
	}
	
	if (candidate_bases.empty()) return Positions::None;
	if (candidate_bases.size() == 1) return (*candidate_bases.begin())->Center();
	
	Position initial_scout_death_position = worker_manager.initial_scout_death_position();
	if (initial_scout_death_position.isValid()) {
		const BWEM::Base* base = smallest_priority(start_bases, [initial_scout_death_position](auto& base){
			return initial_scout_death_position.getApproxDistance(base->Center());
		});
		if (base != nullptr && candidate_bases.count(base) > 0) return base->Center();
	}
	
	std::map<Position,int> distances;
	for (auto& entry : information_manager.enemy_units()) {
		if (entry.second.type.isResourceDepot() && !entry.second.flying) {
			Position position = entry.second.position;
			const BWEM::Base* base = smallest_priority(start_bases, [position](auto& base){
				return position.getApproxDistance(base->Center());
			});
			if (base != nullptr && candidate_bases.count(base) > 0) distances[position] = position.getApproxDistance(base->Center());
		}
	}
	if (distances.empty()) {
		for (auto& entry : information_manager.enemy_units()) {
			if (entry.second.type.isBuilding() && !entry.second.flying) {
				Position position = entry.second.position;
				const BWEM::Base* base = smallest_priority(start_bases, [position](auto& base){
					return position.getApproxDistance(base->Center());
				});
				if (base != nullptr && candidate_bases.count(base) > 0) distances[position] = position.getApproxDistance(base->Center());
			}
		}
	}
	Position base_position_closest_to_building = key_with_smallest_value(distances, Positions::None);
	if (base_position_closest_to_building.isValid()) return base_position_closest_to_building;
	
	return Positions::None;
}
