#include "BananaBrain.h"

void DragoonState::update(Unit unit)
{
	unit_ = unit;
	if (last_attack_start_frame_ >= Broodwar->getFrameCount() ||
		Broodwar->getFrameCount() - last_attack_start_frame_ > kDragoonAttackFrames - Broodwar->getRemainingLatencyFrames()) {
		last_attack_start_frame_ = -1;
		if (unit_->isStartingAttack()) {
			last_attack_start_frame_ = Broodwar->getFrameCount();
		} else if (unit_->getLastCommand().getType() == UnitCommandTypes::Attack_Unit &&
				   unit_->getLastCommand().getTarget() != nullptr &&
				   unit_->getLastCommand().getTarget()->exists() &&
				   unit_->getLastCommand().getTarget()->isVisible()) {
			last_attack_start_frame_ = std::max(Broodwar->getFrameCount() + 1,
												std::max(unit_->getLastCommandFrame() + Broodwar->getRemainingLatencyFrames(),
														 Broodwar->getFrameCount() + unit_->getGroundWeaponCooldown()));
		}
	}
}

bool DragoonState::is_busy()
{
	int attack_frame_delta = Broodwar->getFrameCount() - last_attack_start_frame_;
	
	if (attack_frame_delta >= 0 && attack_frame_delta <= kDragoonAttackFrames - Broodwar->getRemainingLatencyFrames()) {
		return true;
	}
	
	if (attack_frame_delta < 0 && attack_frame_delta > -Broodwar->getRemainingLatencyFrames()) {
		Unit target = unit_->getLastCommand().getTarget();
		if (target == nullptr || !target->exists() || !target->isVisible() || !not_cloaked(target)) return false;
		
		Position unit_predicted_position = predict_position(unit_, -attack_frame_delta);
		Position target_predicted_position = predict_position(target, -attack_frame_delta);
		int predicted_distance = calculate_distance(unit_->getType(), unit_predicted_position, target->getType(), target_predicted_position);
		return predicted_distance <= weapon_max_range(WeaponTypes::Phase_Disruptor, Broodwar->self());
	}
	
	return false;
}

void UnstickState::update(Unit unit)
{
	unit_ = unit;
	if (!unit_->isMoving()) {
		stuck_since_ = -1;
	} else if (stuck_since_ != -1 && last_position_.isValid() && unit_->getPosition() != last_position_) {
		stuck_since_ = -1;
	} else if (stuck_since_ != -1 && unit_->getLastCommand().getType() == UnitCommandTypes::Stop &&
			   Broodwar->getRemainingLatencyFrames() == Broodwar->getLatencyFrames()) {
		stuck_since_ = -1;
	} else if (unit_->isAttackFrame()) {
		stuck_since_ = Broodwar->getFrameCount();
	}
	last_position_ = unit->getPosition();
}

bool UnstickState::is_stuck()
{
	if (unit_->isMoving() && unit_->isStuck()) return true;
	return (stuck_since_ != -1 && stuck_since_ < (Broodwar->getFrameCount() - Broodwar->getRemainingLatencyFrames() - 10));
}

bool CombatUnitTarget::should_update_target(Unit combat_unit) const
{
	bool result = false;
	if (Broodwar->getFrameCount() >= frame + kTargetUpdateFrames ||
		(combat_unit->getID() % kTargetUpdateFrames) == (Broodwar->getFrameCount() % kTargetUpdateFrames)) {
		result = true;
	} else if (unit != nullptr && (!unit->exists() || !unit->isVisible() || !not_cloaked(unit))) {
		result = true;
	}
	return result;
}

void MicroManager::prepare_combat()
{
	update_units_lists();
}

void MicroManager::apply_combat_orders()
{
	update_units_near_base();
	update_units_near_main_base();
	update_ignore_when_attacking();
	
	apply_shuttle_orders();
	apply_combat_unit_orders();
	apply_scout_orders();
	apply_corsair_orders();
	apply_observer_orders();
	apply_arbiter_orders();
	apply_high_templar_orders();
	apply_dark_archon_orders();
	
	expire_tentative_abilities();
}

void MicroManager::update_units_lists()
{
	combat_units_.clear();
	corsairs_.clear();
	observers_.clear();
	high_templars_.clear();
	dark_archons_.clear();
	shuttles_.clear();
	arbiters_.clear();
	scouts_.clear();
	std::set<Unit> controlled_units;
	for (auto& unit : Broodwar->self()->getUnits()) {
		if (!unit->getType().isWorker() && !unit->getType().isBuilding() && unit->isCompleted() &&
			unit->isVisible() && !unit->isStasised() && !unit->isLoaded()) {
			bool controlled = true;
			if (unit->getType() == UnitTypes::Protoss_Corsair) {
				corsairs_.push_back(unit);
			} else if (unit->getType() == UnitTypes::Protoss_Observer) {
				observers_.push_back(unit);
			} else if (unit->getType() == UnitTypes::Protoss_High_Templar) {
				high_templars_.push_back(unit);
			} else if (unit->getType() == UnitTypes::Protoss_Dark_Archon) {
				dark_archons_.push_back(unit);
			} else if (unit->getType() == UnitTypes::Protoss_Shuttle) {
				shuttles_.push_back(unit);
			} else if (unit->getType() == UnitTypes::Protoss_Arbiter) {
				arbiters_.push_back(unit);
			} else if (unit->getType() == UnitTypes::Protoss_Scout) {
				scouts_.push_back(unit);
			} else if (unit->getType() != UnitTypes::Protoss_Interceptor &&
					   (unit->getType().groundWeapon() != WeaponTypes::None ||
						unit->getType().airWeapon() != WeaponTypes::None ||
						unit->getType() == UnitTypes::Protoss_Carrier ||
						unit->getType() == UnitTypes::Protoss_Reaver)) {
				combat_units_.push_back(unit);
			} else {
				controlled = false;
			}
			if (controlled) {
				if (combat_state_.count(unit) == 0) combat_state_.emplace(unit, unit);
				controlled_units.insert(unit);
			}
		}
	}
	std::vector<Unit> uncontrolled_units;
	for (auto& entry : combat_state_) if (controlled_units.count(entry.first) == 0) uncontrolled_units.push_back(entry.first);
	for (auto& uncontrolled_unit : uncontrolled_units) combat_state_.erase(uncontrolled_unit);
	
	all_enemy_units_.clear();
	harassable_enemy_units_.clear();
	for (auto& unit : Broodwar->enemy()->getUnits()) {
		if (unit->isVisible() && not_incomplete(unit)) all_enemy_units_.push_back(unit);
	}
	for (auto& unit : all_enemy_units_) {
		if (not_cloaked(unit) && !unit->isStasised() && !unit->isInvincible()) {
			harassable_enemy_units_.push_back(unit);
		}
	}
}

int MicroManager::offense_distance_to_base(const EnemyCluster& cluster)
{
	int distance = INT_MAX;
	for (auto& enemy_unit : cluster.units) distance = std::min(distance, offense_distance_to_base(enemy_unit));
	return distance;
}

int MicroManager::offense_distance_to_base(const InformationUnit* enemy_unit)
{
	if (!enemy_unit->position.isValid()) return INT_MAX;
	
	int range = offense_max_range(enemy_unit->type, Broodwar->enemy(), false);
	if (range < 0) return INT_MAX;
	
	int distance = enemy_unit->base_distance;
	
	for (Unit unit : Broodwar->self()->getUnits()) {
		if (unit->getType().isBuilding()) {
			int building_distance = calculate_distance(enemy_unit->type, enemy_unit->position, unit->getType(), unit->getPosition());
			distance = std::min(distance, building_distance);
		}
	}
	
	return std::max(0, distance - range);
}

void MicroManager::update_units_near_base()
{
	for (auto& cluster : tactics_manager.clusters()) {
		int distance = offense_distance_to_base(cluster);
		if (distance <= 0) {
			for (auto& enemy_unit : cluster.units) units_near_base_.insert(enemy_unit->unit);
		} else if (distance >= 320) {
			for (auto& enemy_unit : cluster.units) units_near_base_.erase(enemy_unit->unit);
		}
	}
}

void MicroManager::update_units_near_main_base()
{
	const BWEM::Base* main_base = base_state.main_base();
	
	if (main_base == nullptr) {
		units_near_main_base_.clear();
	} else {
		for (auto entry : information_manager.enemy_units()) {
			int distance = std::min(distance_to_base(main_base, &entry.second), distance_to_proxy_pylon(&entry.second));
			if (distance <= 320) {
				units_near_main_base_.insert(entry.second.unit);
			} else if (distance > 384) {
				units_near_main_base_.erase(entry.second.unit);
			}
		}
	}
}

void MicroManager::update_ignore_when_attacking()
{
	ignore_when_attacking_.clear();
	for (auto& cluster : tactics_manager.clusters()) {
		if (cluster.units.size() == 1 || cluster.units.size() == 2) {
			bool only_workers = std::all_of(cluster.units.begin(), cluster.units.end(), [](auto& enemy_unit){
				return enemy_unit->type.isWorker();
			});
			if (only_workers) {
				for (auto& enemy_unit : cluster.units) ignore_when_attacking_.insert(enemy_unit->unit);
			}
		}
	}
}

void MicroManager::apply_shuttle_orders()
{
	loading_units_.clear();
	for (auto& shuttle_unit : shuttles_) {
		bool order_issued = false;
		
		ShuttleState& state = shuttle_state_[shuttle_unit];
		if (state.command == ShuttleCommand::LoadDarkTemplars && is_hp_undamaged(shuttle_unit)) {
			if (!order_issued) {
				order_issued = load_closest_unit_of_type(shuttle_unit, UnitTypes::Protoss_Dark_Templar, true);
			}
			if (!order_issued) {
				if (!shuttle_unit->getLoadedUnits().empty()) {
					state.position = tactics_manager.enemy_start_position();
					if (!state.position.isValid()) state.position = tactics_manager.enemy_base_attack_position();
					if (state.position.isValid()) state.command = ShuttleCommand::DropInEnemyBase;
				} else {
					state.command = ShuttleCommand::Default;
				}
			}
		} else if (state.command == ShuttleCommand::DropInEnemyBase) {
			if (shuttle_unit->getLoadedUnits().empty()) {
				state.command = ShuttleCommand::Default;
			} else {
				Position position = state.position;
				if (shuttle_unit->getDistance(position) <= 128 ||
					(shuttle_unit->isUnderAttack() && !is_hp_undamaged(shuttle_unit))) {
					Unit first_loaded_unit = *(shuttle_unit->getLoadedUnits().begin());
					shuttle_unit->unload(first_loaded_unit);
					order_issued = true;
				} else {
					order_issued = unit_potential(shuttle_unit, [this,position](UnitPotential& potential){
						potential.repel_units(all_enemy_units_);
						potential.repel_storms();
						if (potential.empty()) potential.add_potential(position, -0.1);
					});
					if (!order_issued) {
						Unit first_loaded_unit = *(shuttle_unit->getLoadedUnits().begin());
						shuttle_unit->unload(first_loaded_unit);
						order_issued = true;
					}
				}
			}
		} else if (state.command == ShuttleCommand::Default) {
			if (!order_issued) {
				Position tank_position = closest_sieged_tank(shuttle_unit->getPosition());
				if (tank_position.isValid()) {
					if (shuttle_unit->getLoadedUnits().empty()) {
						order_issued = load_closest_unit_of_type(shuttle_unit, UnitTypes::Protoss_Zealot);
					} else {
						Unit first_loaded_unit = *(shuttle_unit->getLoadedUnits().begin());
						Position shuttle_position = predict_position(shuttle_unit, Broodwar->getRemainingLatencyFrames());
						if (calculate_distance(UnitTypes::Terran_Siege_Tank_Siege_Mode, tank_position, first_loaded_unit->getType(), shuttle_position) <=
							first_loaded_unit->getType().groundWeapon().maxRange()) {
							shuttle_unit->unload(first_loaded_unit);
							order_issued = true;
						} else {
							unit_move(shuttle_unit, tank_position);
							order_issued = true;
						}
					}
				}
			}
		}
		
		CombatState& combat_state = combat_state_.at(shuttle_unit);
		if (!order_issued && combat_state.target_position().isValid()) {
			Unit closest_combat_unit = combat_unit_closest_to_target(combat_state.target_position());
			if (closest_combat_unit != nullptr) {
				order_issued = unit_potential(shuttle_unit, [this,closest_combat_unit](UnitPotential& potential){
					potential.repel_units(all_enemy_units_);
					potential.repel_storms();
					if (potential.empty()) potential.add_potential(closest_combat_unit->getPosition(), -0.1);
				});
			}
		}
		
		if (!order_issued) {
			if (combat_state.stage_position().isValid()) {
				unit_move(shuttle_unit, combat_state.stage_position());
			} else {
				if (!shuttle_unit->isHoldingPosition()) shuttle_unit->holdPosition();
			}
		}
	}
}

void MicroManager::apply_combat_unit_orders()
{
	std::pair<Unit,Unit> dark_archon_meld;
	bool need_dark_archons = false;
	if (training_manager.unit_count(UnitTypes::Protoss_Dark_Archon) < requested_dark_archon_count_) {
		need_dark_archons = true;
		std::vector<Unit> suitable_dark_templars;
		for (auto& unit : combat_units_) {
			if (unit->getType() == UnitTypes::Protoss_Dark_Templar && unit_in_safe_location(unit)) suitable_dark_templars.push_back(unit);
		}
		if (suitable_dark_templars.size() >= 2) {
			int smallest_distance = INT_MAX;
			for (size_t i = 0; i < suitable_dark_templars.size(); i++) {
				for (size_t j = i + 1; j < suitable_dark_templars.size(); j++) {
					int distance = ground_distance(suitable_dark_templars[i]->getPosition(), suitable_dark_templars[j]->getPosition());
					if (distance <= 128 && distance < smallest_distance) {
						smallest_distance = distance;
						dark_archon_meld.first = suitable_dark_templars[i];
						dark_archon_meld.second = suitable_dark_templars[j];
					}
				}
			}
		}
	}
	update_run_by();
	for (auto& combat_unit : combat_units_) {
		if (loading_units_.count(combat_unit) > 0) continue;
		UnitType unit_type = combat_unit->getType();
		
		if (unit_type == UnitTypes::Protoss_Dragoon) {
			DragoonState& state = dragoon_state_[combat_unit];
			state.update(combat_unit);
			if (state.is_busy()) continue;
		}
		
		if (unit_type == UnitTypes::Protoss_Dragoon ||
			unit_type == UnitTypes::Protoss_Zealot ||
			unit_type == UnitTypes::Protoss_Dark_Templar) {
			UnstickState& state = unstick_state_[combat_unit];
			state.update(combat_unit);
			if (state.is_stuck()) {
				combat_unit->stop();
				continue;
			}
		}
		
		if (combat_unit == dark_archon_meld.first) {
			if (!unit_has_target(dark_archon_meld.first, dark_archon_meld.second)) {
				dark_archon_meld.first->useTech(TechTypes::Dark_Archon_Meld, dark_archon_meld.second);
			}
			continue;
		}
		if (combat_unit == dark_archon_meld.second) {
			if (!unit_has_target(dark_archon_meld.second, dark_archon_meld.first)) {
				dark_archon_meld.second->useTech(TechTypes::Dark_Archon_Meld, dark_archon_meld.first);
			}
			continue;
		}
		
		if (recharge_at_shield_battery(combat_unit)) {
			continue;
		}
		
		CombatState& combat_state = combat_state_.at(combat_unit);
		Position target_position = combat_state.target_position();
		Position stage_position = combat_state.stage_position();
		
		Unit selected_enemy_unit = nullptr;
		bool enable_attack = true;
		
		Unit nearby_sieged_tank = determine_nearby_sieged_tank(combat_unit);
		if (nearby_sieged_tank != nullptr) {
			selected_enemy_unit = nearby_sieged_tank;
		} else if (unit_type == UnitTypes::Protoss_Dark_Templar) {
			if (!need_dark_archons) {
				selected_enemy_unit = select_enemy_unit_for_dark_templar(combat_unit);
			}
		} else {
			CombatUnitTarget& target = combat_unit_targets_[combat_unit];
			if (target.should_update_target(combat_unit)) {
				std::tie(target.unit, target.enable_attack) = select_enemy_unit_for_combat_unit(combat_unit);
				target.frame = Broodwar->getFrameCount();
			}
			selected_enemy_unit = target.unit;
			enable_attack = target.enable_attack;
		}
		
		if (selected_enemy_unit != nullptr) {
			bool order_issued = false;
			if (is_melee(combat_unit->getType()) || combat_unit->getType() == UnitTypes::Protoss_Carrier) {
				order_issued = unit_potential(combat_unit, [](UnitPotential& potential){
					potential.repel_storms();
				});
			}
			if (!is_melee(combat_unit->getType()) && combat_unit->getType() != UnitTypes::Protoss_Carrier && is_on_cooldown(combat_unit, selected_enemy_unit->isFlying())) {
				if (combat_unit->getType() == UnitTypes::Protoss_Dragoon && combat_unit->getLastCommand().getType() == UnitCommandTypes::Attack_Unit) {
					combat_unit->stop();
				} else {
					unit_potential(combat_unit, [this,selected_enemy_unit](UnitPotential& potential){
						potential.kite_units(all_enemy_units_);
						potential.repel_storms();
						potential.repel_buildings();
						potential.repel_terrain();
						if (potential.empty() && can_attack_in_range(potential.unit(), selected_enemy_unit)) {
							potential.add_potential(selected_enemy_unit->getPosition(), -0.1);
						}
					});
				}
				order_issued = true;
			}
			if (!order_issued) {
				Position intercept_position = calculate_interception_position(combat_unit, selected_enemy_unit);
				path_finder.execute_path(combat_unit, intercept_position, [this,combat_unit,selected_enemy_unit,intercept_position](){
					if (selected_enemy_unit->getPosition() == intercept_position) {
						unit_attack(combat_unit, selected_enemy_unit);
					} else {
						if (combat_unit->getType() == UnitTypes::Protoss_Dark_Templar) {
							move_with_blockade_breaking(combat_unit, intercept_position);
						} else {
							unit_move(combat_unit, intercept_position);
						}
					}
				});
			}
		} else if (running_by_.count(combat_unit) > 0 || desperados_.count(combat_unit) > 0) {
			move_with_blockade_breaking(combat_unit, run_by_target_position_);
		} else if (target_position == Positions::Unknown) {
			random_move(combat_unit);
		} else if (target_position.isValid() &&
				   enable_attack &&
				   (!need_dark_archons || unit_type != UnitTypes::Protoss_Dark_Templar)) {
			path_finder.execute_path(combat_unit, target_position, [target_position,combat_unit](){
				unit_move(combat_unit, target_position);
			});
		} else {
			bool moved = unit_potential(combat_unit, [this](UnitPotential& potential){
				potential.repel_storms();
			});
			if (!moved) {
				if (stage_position.isValid()) {
					move_retreat(combat_unit, stage_position);
				} else {
					if (!combat_unit->isHoldingPosition()) combat_unit->holdPosition();
				}
			}
		}
	}
}

void MicroManager::apply_scout_orders()
{
	for (auto& combat_unit : scouts_) {
		std::vector<Unit> repel_unit_list;
		if (!is_hp_undamaged(combat_unit)) {
			repel_unit_list.insert(repel_unit_list.end(), all_enemy_units_.begin(), all_enemy_units_.end());
		} else {
			std::copy_if(all_enemy_units_.begin(), all_enemy_units_.end(), std::back_inserter(repel_unit_list), [](auto& unit){
				return unit->getType() != UnitTypes::Terran_Marine && unit->getType() != UnitTypes::Zerg_Hydralisk;
			});
		}
		
		bool order_issued = unit_potential(combat_unit, [&repel_unit_list](UnitPotential& potential){
			potential.repel_units(repel_unit_list);
			potential.repel_storms();
		});
		if (!order_issued) {
			Unit selected_enemy_unit = select_enemy_unit_for_scout(combat_unit, scout_reached_base_);
			if (selected_enemy_unit != nullptr) {
				if (can_attack_in_range(combat_unit, selected_enemy_unit)) {
					unit_attack(combat_unit, selected_enemy_unit);
				} else {
					unit_potential(combat_unit, [&repel_unit_list,selected_enemy_unit](UnitPotential& potential){
						potential.repel_units(repel_unit_list);
						potential.repel_storms();
						if (potential.empty()) potential.add_potential(selected_enemy_unit->getPosition(), -0.1);
					});
				}
			} else {
				Position position = tactics_manager.enemy_start_position();
				if (!position.isValid()) position = tactics_manager.enemy_base_attack_position();
				if (position.isValid()) {
					unit_potential(combat_unit, [&repel_unit_list,position](UnitPotential& potential){
						potential.repel_units(repel_unit_list);
						potential.repel_storms();
						if (potential.empty()) potential.add_potential(position, -0.1);
					});
					if (combat_unit->getDistance(position) <= combat_unit->getType().sightRange()) {
						scout_reached_base_ = true;
					}
				} else {
					random_move(combat_unit);
					scout_reached_base_ = true;
				}
			}
		}
	}
}

void MicroManager::apply_corsair_orders()
{
	for (auto& combat_unit : corsairs_) {
		bool order_issued = false;
		if (!order_issued && !is_on_cooldown(combat_unit, true)) {
			bool faster_units_incoming = std::any_of(harassable_enemy_units_.begin(), harassable_enemy_units_.end(), [combat_unit](Unit unit) {
				return unit->getType().topSpeed() >= combat_unit->getType().topSpeed() && can_attack_in_range_with_prediction(unit, combat_unit);
			});
			if (faster_units_incoming) {
				std::vector<Unit> potential_targets;
				for (auto& unit : harassable_enemy_units_) {
					if (unit->getType().topSpeed() >= combat_unit->getType().topSpeed() &&
						can_attack_in_range_with_prediction(combat_unit, unit)) {
						potential_targets.push_back(unit);
					}
				}
				
				if (!potential_targets.empty()) {
					if (combat_unit->getLastCommand().getType() == UnitCommandTypes::Attack_Unit) {
						int turn_frames = turn_frames_needed(combat_unit, combat_unit->getLastCommand().getTarget());
						bool target_valid = false;
						Unit target = combat_unit->getLastCommand().getTarget();
						if (target != nullptr && target->exists() && target->isVisible() && not_cloaked(target)) {
							Position unit_predicted_position = predict_position(combat_unit, turn_frames);
							Position target_predicted_position = predict_position(target, turn_frames);
							int predicted_distance = calculate_distance(combat_unit->getType(), unit_predicted_position, target->getType(), target_predicted_position);
							target_valid = (predicted_distance <= weapon_max_range(WeaponTypes::Neutron_Flare, Broodwar->self()));
						}
						
						if (target_valid && turn_frames > Broodwar->getRemainingLatencyFrames()) {
							order_issued = true;
						}
					} else {
						Unit selected_enemy_unit = select_enemy_unit(combat_unit, potential_targets);
						if (selected_enemy_unit != nullptr) {
							unit_attack(combat_unit, selected_enemy_unit);
							order_issued = true;
						}
					}
				}
			}
		}
		if (!order_issued) {
			bool order_issued = unit_potential(combat_unit, [this](UnitPotential& potential){
				potential.repel_units(all_enemy_units_);
				potential.repel_storms();
			});
		}
		if (!order_issued) {
			Unit selected_enemy_unit = select_enemy_unit(combat_unit, harassable_enemy_units_);
			if (selected_enemy_unit != nullptr) {
				if (can_attack_in_range(combat_unit, selected_enemy_unit)) {
					unit_attack(combat_unit, selected_enemy_unit);
				} else {
					unit_potential(combat_unit, [this,selected_enemy_unit](UnitPotential& potential){
						potential.repel_units(all_enemy_units_);
						potential.repel_storms();
						if (potential.empty()) potential.add_potential(selected_enemy_unit->getPosition(), -0.1);
					});
				}
			} else {
				Position position = Positions::None;
				
				if (corsair_targets_.count(combat_unit) > 0) {
					CorsairTarget& target = corsair_targets_[combat_unit];
					if (Broodwar->getFrameCount() >= target.expire_frame || combat_unit->getDistance(target.position) < 32) {
						corsair_targets_.erase(combat_unit);
					} else {
						position = target.position;
					}
				}
				
				if (corsair_targets_.count(combat_unit) == 0) {
					position = pick_scout_location();
					if (position.isValid()) {
						CorsairTarget& target = corsair_targets_[combat_unit];
						target.position = position;
						int distance = combat_unit->getDistance(position);
						int frames = (int)(distance / combat_unit->getType().topSpeed());
						target.expire_frame = Broodwar->getFrameCount() + frames;
					}
				}
				
				if (position.isValid()) {
					unit_potential(combat_unit, [this,position](UnitPotential& potential){
						potential.repel_units(all_enemy_units_);
						potential.repel_storms();
						if (potential.empty()) potential.add_potential(position, -0.1);
					});
				}
			}
		}
	}
}

void MicroManager::apply_observer_orders()
{
	if (!observers_.empty()) {
		bool no_non_scouting_observer_exists = std::none_of(observers_.begin(), observers_.end(), [this](auto& unit){return !observer_targets_[unit].scouting;});
		bool want_non_scouting_observer = opponent_model.cloaked_or_mine_present() || observers_.size() >= 2;
		if (no_non_scouting_observer_exists && want_non_scouting_observer) observer_targets_[observers_[observers_.size() - 1]].scouting = false;
		apply_worker_need_detection();
		for (auto& special_unit : observers_) {
			bool order_issued = false;
			
			if (!order_issued) {
				Position position = Positions::None;
				ObserverTarget& target = observer_targets_[special_unit];
				
				if (!target.scouting) {
					Position position = determine_first_observer_location(special_unit);
					
					if (position.isValid()) {
						order_issued = unit_potential(special_unit, [this,position](UnitPotential& potential){
							bool detected = detected_or_almost_detected_by_enemy(potential.type(), potential.position());
							if (detected) potential.repel_units(all_enemy_units_);
							potential.repel_storms();
							if (potential.empty()) {
								potential.add_potential(position, -0.1);
								potential.repel_detectors(0.05);
							}
						});
					}
				} else {
					Position position = Positions::None;
					
					if (target.position.isValid()) {
						if (Broodwar->getFrameCount() >= target.expire_frame || special_unit->getDistance(target.position) < 32) {
							target.position = Positions::None;
						}
					}
					
					if (!target.position.isValid()) {
						Position scout_position = pick_scout_location();
						if (scout_position.isValid()) {
							target.position = scout_position;
							int distance = special_unit->getDistance(scout_position);
							int frames = (int)(distance / special_unit->getType().topSpeed());
							target.expire_frame = Broodwar->getFrameCount() + frames;
						}
					}
					
					if (target.position.isValid()) position = target.position;
					
					if (!position.isValid()) {
						Position stage_position = combat_state_.at(special_unit).stage_position();
						if (stage_position.isValid()) position = stage_position;
					}
					
					if (position.isValid()) {
						order_issued = unit_potential(special_unit, [this,position](UnitPotential& potential){
							potential.repel_detectors(1.0);
							potential.repel_storms();
							if (potential.empty()) potential.add_potential(position, -0.1);
						});
					}
				}
			}
			
			if (!order_issued) {
				if (!special_unit->isHoldingPosition()) special_unit->holdPosition();
			}
		}
	}
}

void MicroManager::apply_arbiter_orders()
{
	std::set<Unit> casting_arbiters;
	for (auto& tentative_stasis : tentative_stasises_) casting_arbiters.insert(tentative_stasis.unit);
	for (auto& special_unit : arbiters_) {
		CombatState& combat_state = combat_state_.at(special_unit);
		Position target_position = combat_state.target_position();
		Position stage_position = combat_state.stage_position();
		
		bool order_issued = false;
		bool stasis_possible = (Broodwar->self()->hasResearched(TechTypes::Stasis_Field) &&
								special_unit->getEnergy() >= TechTypes::Stasis_Field.energyCost() &&
								special_unit->getSpellCooldown() == 0);
		
		if (casting_arbiters.count(special_unit) > 0) order_issued = true;
		
		if (!order_issued && stasis_possible) {
			order_issued = stasis(special_unit);
		}
		
		if (!order_issued) {
			Position position = Positions::None;
			if (target_position.isValid()) {
				Unit closest_combat_unit = combat_unit_closest_to_target(target_position);
				if (closest_combat_unit != nullptr) position = closest_combat_unit->getPosition();
			}
			if (!position.isValid()) {
				Unit closest_combat_unit = combat_unit_closest_to_special_unit(special_unit);
				if (closest_combat_unit != nullptr) position = closest_combat_unit->getPosition();
			}
			if (!position.isValid()) {
				if (stage_position.isValid()) position = stage_position;
			}
			if (position.isValid()) {
				order_issued = unit_potential(special_unit, [this,position](UnitPotential& potential){
					potential.repel_units(all_enemy_units_);
					potential.repel_storms();
					potential.repel_emps();
					if (potential.empty()) {
						potential.add_potential(position, -0.1);
						for (auto& other_arbiter : arbiters_) {
							if (other_arbiter != potential.unit()) {
								potential.add_potential(other_arbiter, 0.1, 2 * WeaponTypes::EMP_Shockwave.outerSplashRadius());
							}
						}
					}
				});
			}
		}
		
		if (!order_issued) {
			if (!special_unit->isHoldingPosition()) special_unit->holdPosition();
		}
	}
}

void MicroManager::apply_high_templar_orders()
{
	std::pair<Unit,Unit> archon_warp;
	if (prevent_high_templar_from_archon_warp_count_ >= 0 &&
		high_templars_.size() >= (unsigned int)prevent_high_templar_from_archon_warp_count_ + 2) {
		std::vector<Unit> eligible_high_templars;
		for (auto& unit : high_templars_) {
			if (force_archon_warp_ || unit->getEnergy() < 50) eligible_high_templars.push_back(unit);
		}
		if (eligible_high_templars.size() >= 2) {
			int smallest_distance = INT_MAX;
			for (size_t i = 0; i < eligible_high_templars.size(); i++) {
				for (size_t j = i + 1; j < eligible_high_templars.size(); j++) {
					int distance = ground_distance(eligible_high_templars[i]->getPosition(), eligible_high_templars[j]->getPosition());
					if (distance <= 128 && distance < smallest_distance) {
						smallest_distance = distance;
						archon_warp.first = eligible_high_templars[i];
						archon_warp.second = eligible_high_templars[j];
					}
				}
			}
		}
	}
	std::set<Unit> storming_high_templars;
	for (auto& tentative_storm : tentative_storms_) storming_high_templars.insert(tentative_storm.unit);
	for (auto& special_unit : high_templars_) {
		CombatState& combat_state = combat_state_.at(special_unit);
		Position target_position = combat_state.target_position();
		Position stage_position = combat_state.stage_position();
		
		if (loading_units_.count(special_unit) > 0) continue;
		bool order_issued = false;
		bool storm_possible = (Broodwar->self()->hasResearched(TechTypes::Psionic_Storm) &&
							   special_unit->getEnergy() >= TechTypes::Psionic_Storm.energyCost() &&
							   special_unit->getSpellCooldown() == 0);
		
		if (storming_high_templars.count(special_unit) > 0) order_issued = true;
		
		if (!order_issued && special_unit == archon_warp.first) {
			if (!unit_has_target(archon_warp.first, archon_warp.second)) {
				archon_warp.first->useTech(TechTypes::Archon_Warp, archon_warp.second);
			}
			order_issued = true;
		}
		if (!order_issued && special_unit == archon_warp.second) {
			if (!unit_has_target(archon_warp.second, archon_warp.first)) {
				archon_warp.second->useTech(TechTypes::Archon_Warp, archon_warp.first);
			}
			order_issued = true;
		}
		
		if (!order_issued && storm_possible) {
			order_issued = storm(special_unit);
		}
		
		if (!order_issued && target_position == Positions::Unknown) {
			random_move(special_unit);
			order_issued = true;
		}
		
		if (!order_issued) {
			if (storm_possible && target_position.isValid()) {
				std::vector<Unit> repel_unit_list;
				for (auto& unit : all_enemy_units_) if (unit->getType().isBuilding()) repel_unit_list.push_back(unit);
				
				order_issued = unit_potential(special_unit, [&repel_unit_list](UnitPotential& potential){
					potential.repel_units(repel_unit_list);
					potential.repel_storms();
				});
				
				if (!order_issued) {
					order_issued = path_finder.execute_path(special_unit, target_position, [target_position,special_unit]{
						unit_move(special_unit, target_position);
					});
				}
			} else {
				order_issued = unit_potential(special_unit, [this](UnitPotential& potential){
					potential.repel_units(all_enemy_units_);
					potential.repel_storms();
				});
			}
		}
		
		if (!order_issued) {
			if (stage_position.isValid()) {
				path_finder.execute_path(special_unit, stage_position, [stage_position,special_unit](){
					unit_move(special_unit, stage_position);
				});
			} else {
				if (!special_unit->isHoldingPosition()) special_unit->holdPosition();
			}
		}
	}
}

void MicroManager::expire_tentative_abilities()
{
	std::set<Position> storm_positions;
	for (auto& bullet : Broodwar->getBullets()) if (bullet->getType() == BulletTypes::Psionic_Storm) storm_positions.insert(bullet->getPosition());
	remove_elements_in_place(tentative_storms_, [&storm_positions](const TentativeStorm& tentative_storm){
		return tentative_storm.expire_frame < Broodwar->getFrameCount() || storm_positions.count(tentative_storm.position) > 0;
	});
	remove_elements_in_place(tentative_mind_controls_, [](const TentativeMindControl& tentative_mind_control){
		return tentative_mind_control.expire_frame < Broodwar->getFrameCount() || !tentative_mind_control.target->exists() || tentative_mind_control.target->getPlayer() == Broodwar->self();
	});
	remove_elements_in_place(tentative_stasises_, [](const TentativeStasis& tentative_stasis){
		return tentative_stasis.expire_frame < Broodwar->getFrameCount();
	});
}

void MicroManager::apply_dark_archon_orders()
{
	std::set<Unit> mind_controlling_dark_archons;
	for (auto& tentative_mind_control : tentative_mind_controls_) mind_controlling_dark_archons.insert(tentative_mind_control.unit);
	for (auto& special_unit : dark_archons_) {
		CombatState& combat_state = combat_state_.at(special_unit);
		Position target_position = combat_state.target_position();
		Position stage_position = combat_state.stage_position();
		
		bool order_issued = false;
		bool mind_control_possible = (Broodwar->self()->hasResearched(TechTypes::Mind_Control) &&
									  special_unit->getEnergy() >= TechTypes::Mind_Control.energyCost() &&
									  special_unit->getSpellCooldown() == 0);
		
		if (mind_controlling_dark_archons.count(special_unit) > 0) order_issued = true;
		
		Unit mind_control_target = nullptr;
		if (mind_control_possible) {
			std::vector<Unit> potential_targets;
			for (auto& unit : all_enemy_units_) {
				if (not_cloaked(unit) && !unit->isStasised() && special_unit->getDistance(unit) <= 14 * 32 && (unit->getType() == UnitTypes::Protoss_Carrier || unit->getType() == UnitTypes::Protoss_Arbiter)) {
					potential_targets.push_back(unit);
				}
			}
			if (!potential_targets.empty()) {
				mind_control_target = smallest_priority(potential_targets, [special_unit](auto& unit) {
					return special_unit->getDistance(unit);
				});
			}
		}
		
		if (!order_issued && mind_control_target != nullptr) {
			if (special_unit->getDistance(mind_control_target) <= WeaponTypes::Mind_Control.maxRange()) {
				special_unit->useTech(TechTypes::Mind_Control, mind_control_target);
				int latency_frames = Broodwar->getRemainingLatencyFrames() + 24;
				tentative_mind_controls_.push_back(TentativeMindControl(special_unit, Broodwar->getFrameCount() + latency_frames, mind_control_target));
				order_issued = true;
			} else {
				order_issued = path_finder.execute_path(special_unit, mind_control_target->getPosition(), [special_unit,mind_control_target]{
					unit_move(special_unit, mind_control_target->getPosition());
				});
			}
		}
		
		if (!order_issued && target_position == Positions::Unknown) {
			random_move(special_unit);
			order_issued = true;
		}
		
		if (!order_issued) {
			if (mind_control_possible && target_position.isValid()) {
				std::vector<Unit> repel_unit_list;
				for (auto& unit : all_enemy_units_) if (unit->getType().isBuilding()) repel_unit_list.push_back(unit);
				
				order_issued = unit_potential(special_unit, [&repel_unit_list](UnitPotential& potential){
					potential.repel_units(repel_unit_list);
					potential.repel_storms();
				});
				
				if (!order_issued) {
					order_issued = path_finder.execute_path(special_unit, target_position, [target_position,special_unit]{
						unit_move(special_unit, target_position);
					});
				}
			} else {
				order_issued = unit_potential(special_unit, [this](UnitPotential& potential){
					potential.repel_units(all_enemy_units_);
					potential.repel_storms();
				});
			}
		}
		
		if (!order_issued) {
			if (stage_position.isValid()) {
				path_finder.execute_path(special_unit, stage_position, [stage_position,special_unit](){
					unit_move(special_unit, stage_position);
				});
			} else {
				if (!special_unit->isHoldingPosition()) special_unit->holdPosition();
			}
		}
	}
}

int MicroManager::distance_to_base(const BWEM::Base *base,const InformationUnit* enemy_unit)
{
	UnitType center_type = Broodwar->self()->getRace().getResourceDepot();
	int distance = calculate_distance(enemy_unit->type, enemy_unit->position, center_type, center_position_for(center_type, base->Location()));
	return distance;
}

int MicroManager::distance_to_proxy_pylon(const InformationUnit* enemy_unit)
{
	int distance = INT_MAX;
	TilePosition pylon_position = building_placement_manager.proxy_pylon_position();
	if (pylon_position.isValid()) {
		distance = calculate_distance(enemy_unit->type, enemy_unit->position,
									  UnitTypes::Protoss_Pylon, center_position_for(UnitTypes::Protoss_Pylon, pylon_position));
	}
	return distance;
}

Position MicroManager::calculate_interception_position(Unit combat_unit,Unit enemy_unit)
{
	Position result = enemy_unit->getPosition();
 
	if (!can_attack_in_range_with_prediction(combat_unit, enemy_unit)) {
		Position predicted_position = predict_position(enemy_unit, 8);
		Position delta_current = enemy_unit->getPosition() - combat_unit->getPosition();
		Position delta_predict = predicted_position - combat_unit->getPosition();
		int inner_product = delta_current.x * delta_predict.x + delta_current.y * delta_predict.y;
		if (inner_product > 0 && check_collision(enemy_unit, predicted_position)) result = predicted_position;
	}
	
	return result;
}

bool MicroManager::unit_in_safe_location(Unit unit)
{
	const BWEM::Area* area = area_at(unit->getPosition());
	if (area != nullptr && base_state.controlled_areas().count(area) > 0) return true;
	
	for (auto& cp : base_state.border().chokepoints()) {
		Position position = chokepoint_center(cp);
		if (unit->getDistance(position) <= 320) return true;
	}
	
	return false;
}

void MicroManager::perform_dark_templar_drop()
{
	for (Unit unit : Broodwar->self()->getUnits()) {
		if (unit->isCompleted() && !unit->isStasised() && unit->getType() == UnitTypes::Protoss_Shuttle) {
			ShuttleState& state = shuttle_state_[unit];
			if (state.command == ShuttleCommand::Default) {
				state.command = ShuttleCommand::LoadDarkTemplars;
				return;
			}
		}
	}
}

Position MicroManager::determine_first_observer_location(Unit special_unit)
{
	Position result = Positions::None;
	
	std::map<Position,int> distances;
	for (auto& unit : all_enemy_units_) {
		if (unit->isCloaked() || unit->getType() == UnitTypes::Zerg_Lurker) {
			if (information_manager.enemy_units().at(unit).base_distance == 0) {
				distances[unit->getPosition()] = unit->getDistance(special_unit);
			}
		}
	}
	result = key_with_smallest_value(distances, Positions::None);
	
	CombatState& combat_state = combat_state_.at(special_unit);
	Position target_position = combat_state.target_position();
	Position stage_position = combat_state.stage_position();
	
	if (!result.isValid() && target_position.isValid()) {
		Unit closest_combat_unit = combat_unit_closest_to_target(target_position);
		if (closest_combat_unit != nullptr) result = closest_combat_unit->getPosition();
	}
	if (!result.isValid()) {
		Unit closest_combat_unit = combat_unit_closest_to_special_unit(special_unit);
		if (closest_combat_unit != nullptr) result = closest_combat_unit->getPosition();
	}
	
	if (!result.isValid()) {
		if (stage_position.isValid()) {
			result = stage_position;
		}
	}
	
	return result;
}

Position MicroManager::closest_sieged_tank(Position position)
{
	std::vector<Position> tank_positions;
	
	for (auto& entry : information_manager.enemy_units()) {
		if (entry.second.type == UnitTypes::Terran_Siege_Tank_Siege_Mode &&
			(!entry.second.unit->exists() || !entry.second.unit->isStasised()) &&
			!melee_unit_near_sieged_tank(entry.second.position) &&
			entry.second.position.getApproxDistance(position) <= 500) {
			tank_positions.push_back(entry.second.position);
		}
	}
	
	if (tank_positions.empty()) {
		return Positions::None;
	} else {
		return *std::min_element(tank_positions.begin(), tank_positions.end(), [position](Position a,Position b){
			return a.getApproxDistance(position) < b.getApproxDistance(position);
		});
	}
}

bool MicroManager::melee_unit_near_sieged_tank(Position tank_position)
{
	for (auto& combat_unit : combat_units_) {
		if (is_melee(combat_unit->getType()) &&
			calculate_distance(UnitTypes::Terran_Siege_Tank_Siege_Mode, tank_position, combat_unit->getType(), combat_unit->getPosition()) <= 15) return true;
	}
	return false;
}

bool MicroManager::melee_unit_near_sieged_tank(Unit unit)
{
	if (is_melee(unit->getType())) {
		for (auto& siege_unit : all_enemy_units_) {
			if (siege_unit->getType() == UnitTypes::Terran_Siege_Tank_Siege_Mode &&
				!siege_unit->isStasised() &&
				siege_unit->getDistance(unit) < WeaponTypes::Arclite_Shock_Cannon.minRange()) {
				return true;
			}
		}
	}
	return false;
}

bool MicroManager::load_closest_unit_of_type(Unit shuttle_unit,UnitType load_type,bool ignore_distance)
{
	bool order_issued = false;
	
	if (load_type && shuttle_unit->getSpaceRemaining() >= load_type.spaceRequired()) {
		std::vector<Unit> candidate_load_targets;
		for (Unit unit : Broodwar->self()->getUnits()) {
			if (unit->getType() == load_type &&
				unit->isCompleted() && !unit->isStasised() && !unit->isLoaded() &&
				(ignore_distance || shuttle_unit->getDistance(unit) <= 500) &&
				!melee_unit_near_sieged_tank(unit)) candidate_load_targets.push_back(unit);
		}
		if (!candidate_load_targets.empty()) {
			Unit load_unit = *std::min_element(candidate_load_targets.begin(), candidate_load_targets.end(), [shuttle_unit](Unit a,Unit b) {
				return std::make_tuple(shuttle_unit->getDistance(a), a->getID()) < std::make_tuple(shuttle_unit->getDistance(b), b->getID());
			});
			if (!unit_has_target(shuttle_unit, load_unit)) shuttle_unit->load(load_unit);
			if (!unit_has_target(load_unit, shuttle_unit)) load_unit->follow(shuttle_unit);
			loading_units_.insert(load_unit);
			order_issued = true;
		}
	}
	
	return order_issued;
}

Position MicroManager::pick_scout_location()
{
	int resource_depot_count = 0;
	for (auto& entry : information_manager.enemy_units()) if (entry.second.type.isResourceDepot()) resource_depot_count++;
	int free_base_count = 0;
	for (auto& base : base_state.bases()) if (base_state.controlled_bases().count(base) == 0 && base_state.opponent_bases().count(base) == 0) free_base_count++;
	
	double sum = 0.0;
	std::vector<std::pair<Position,double>> distribution;
	for (auto &entry : information_manager.enemy_units()) {
		auto& enemy_unit = entry.second;
		if (enemy_unit.type.isResourceDepot()) {
			sum += 19.0 / resource_depot_count;
			distribution.push_back(std::pair<Position,double>(center_position_for(enemy_unit.type, enemy_unit.tile_position()), sum));
		}
	}
	for (auto& base : base_state.bases()) {
		if (base_state.controlled_bases().count(base) == 0 && base_state.opponent_bases().count(base) == 0) {
			sum += 1.0 / free_base_count;
			distribution.push_back(std::pair<Position,double>(center_position_for(UnitTypes::Protoss_Nexus, base->Location()), sum));
		}
	}
	
	if (distribution.empty()) return Positions::None;
	if (distribution.size() == 1) return distribution[0].first;
	
	double r = sum * ((double)rand() / (RAND_MAX + 1));
	for (size_t i = 0; i < distribution.size() - 1; i++) {
		if (r < distribution[i].second) return distribution[i].first;
	}
	return distribution[distribution.size() - 1].first;
}

void MicroManager::move_retreat(Unit unit,Position target_position)
{
	FastTilePosition start_tile_position(unit->getPosition());
	FastTilePosition target_tile_position(target_position);
	
	if (start_tile_position == target_tile_position) return;
	int divider = (unit->getPosition().getApproxDistance(target_position) <= 128) ? 24 : 3;
	if ((Broodwar->getFrameCount() % divider) != (unit->getID() % divider)) return;
	
	int start_component = threat_grid.component(start_tile_position, unit->isFlying());
	if (!unit->isFlying() && start_component == 0 && !walkability_grid.is_walkable(start_tile_position)) {
		for (auto delta : { FastTilePosition{1, 0}, FastTilePosition{-1, 0}, FastTilePosition{0, -1}, FastTilePosition{0, 1},
							FastTilePosition{1, 1}, FastTilePosition{-1, 1}, FastTilePosition{-1, -1}, FastTilePosition{1, -1} }) {
			int component = threat_grid.component(start_tile_position + delta, false);
			if (component != 0) {
				start_component = component;
				start_tile_position = start_tile_position + delta;
				break;
			}
		}
	}
	if (start_component == 0) {
		move_with_blockade_breaking(unit, target_position);
		return;
	}
	
	int target_component = threat_grid.component(target_tile_position, unit->isFlying());
	if (target_component != start_component) {
		move_with_blockade_breaking(unit, target_position);
		return;
	}
	
	struct Node
	{
		FastTilePosition tile_position;
		float f_score;
		
		bool operator<(const Node& o) const
		{
			// Implements > as we use this in a std::priority_queue and the lowest f_score has the highest priority
			return o.f_score < f_score || (o.f_score == f_score && o.tile_position < tile_position);
		}
	};
	
	enum class State : uint8_t
	{
		None = 0, Open, Closed
	};
	
	std::priority_queue<Node> priority_queue;
	TileGrid<State> state;
	TileGrid<float> g_score;
	TileGrid<FastTilePosition> came_from;
	came_from[target_tile_position] = target_tile_position;
	priority_queue.push(Node{ target_tile_position, (float)target_tile_position.getApproxDistance(start_tile_position) });
	state[target_tile_position] = State::Open;
	
	while (!priority_queue.empty()) {
		Node current_node = priority_queue.top();
		if (current_node.tile_position == start_tile_position) {
			FastTilePosition move_tile_position = came_from[start_tile_position];
			if (unit->getPosition().getApproxDistance(center_position(move_tile_position)) <= 64) move_tile_position = came_from[move_tile_position];
			unit_move(unit, center_position(move_tile_position));
			return;
		}
		priority_queue.pop();
		State& current_state = state[current_node.tile_position];
		if (current_state != State::Open) continue;
		current_state = State::Closed;
		float current_g_score = g_score[current_node.tile_position];
		
		for (auto delta : { FastTilePosition{1, 0}, FastTilePosition{-1, 0}, FastTilePosition{0, -1}, FastTilePosition{0, 1} }) {
			FastTilePosition next_position = current_node.tile_position + delta;
			if (threat_grid.component(next_position, unit->isFlying()) == start_component && state[next_position] != State::Closed) {
				State& next_state = state[next_position];
				FastTilePosition previous_position = came_from[current_node.tile_position];
				if (previous_position != current_node.tile_position && threat_grid.line_of_sight(previous_position, next_position, unit->isFlying())) {
					int dx = previous_position.x - next_position.x;
					int dy = previous_position.y - next_position.y;
					float tentative_g_score = g_score[previous_position] + std::sqrt(float(dx * dx + dy * dy));
					if (next_state == State::None) {
						next_state = State::Open;
					} else if (tentative_g_score >= g_score[next_position]) {
						continue;
					}
					g_score[next_position] = tentative_g_score;
					came_from[next_position] = previous_position;
					priority_queue.push(Node{ next_position, tentative_g_score + next_position.getApproxDistance(start_tile_position)});
				} else {
					float tentative_g_score = current_g_score + 1.0f;
					if (next_state == State::None) {
						next_state = State::Open;
					} else if (tentative_g_score >= g_score[next_position]) {
						continue;
					}
					g_score[next_position] = tentative_g_score;
					came_from[next_position] = current_node.tile_position;
					priority_queue.push(Node{ next_position, tentative_g_score + next_position.getApproxDistance(start_tile_position)});
				}
			}
		}
	}
}

void MicroManager::move_with_blockade_breaking(Unit unit,Position target_position)
{
	bool use_move = true;
	
	if (!unit->isFlying()) {
		int max_range = weapon_max_range(unit, false) + 32;
		FastPosition initial_position = unit->getPosition();
		int initial_ground_distance = ground_distance(initial_position, target_position);
		
		std::queue<FastPosition> queue;
		SparsePositionGrid<512,16,bool> visited;
		queue.push(initial_position);
		visited[FastPosition(0, 0)] = true;
		
		use_move = false;
		while (!queue.empty() && !use_move) {
			FastPosition current_position = queue.front();
			queue.pop();
			
			for (FastPosition delta_position : { FastPosition(-16, 0), FastPosition(16, 0), FastPosition(0, -16), FastPosition(0, 16) }) {
				FastPosition next_position = current_position + delta_position;
				if (next_position.isValid() && !visited[next_position - initial_position]) {
					if (check_collision(unit, next_position)) {
						if (next_position.getApproxDistance(target_position) <= 16) {
							// Reached the target position
							use_move = true;
							break;
						}
						if (next_position.getApproxDistance(initial_position) <= max_range) {
							queue.push(next_position);
						} else if (ground_distance(next_position, target_position) < initial_ground_distance) {
							// Found a border position closer to the target than the initial position
							use_move = true;
							break;
						}
					}
					visited[next_position - initial_position] = true;
				}
			}
		}
	}
	
	if (!use_move) {
		Unit target = smallest_priority(harassable_enemy_units_, [target_position,unit](Unit enemy_unit){
			double angle = -INFINITY;
			if (can_attack_in_range(unit, enemy_unit)) {
				Position target_delta = target_position - unit->getPosition();
				double target_delta_norm = std::sqrt(target_delta.x * target_delta.x + target_delta.y * target_delta.y);
				Position enemy_delta = enemy_unit->getPosition() - unit->getPosition();
				double enemy_delta_norm = std::sqrt(enemy_delta.x * enemy_delta.x + enemy_delta.y * enemy_delta.y);
				angle = (target_delta.x * enemy_delta.x + target_delta.y * enemy_delta.y) / (target_delta_norm * enemy_delta_norm);
			}
			return std::make_tuple(-angle, unit->getDistance(enemy_unit), target_tie_breaker(unit, enemy_unit));
		});
		if (target != nullptr && can_attack_in_range(unit, target)) {
			unit_attack(unit, target);
		} else {
			use_move = true;
		}
	}
	
	if (use_move) {
		path_finder.execute_path(unit, target_position, [this,unit,target_position](){
			unit_move(unit, target_position);
		});
	}
}

bool MicroManager::recharge_at_shield_battery(Unit unit)
{
	CombatState& combat_state = combat_state_.at(unit);
	
	if (combat_state.shield_battery() != nullptr) {
		Unit shield_battery_unit = combat_state.shield_battery();
		if (!shield_battery_unit->exists() ||
			shield_battery_unit->getEnergy() < 2 ||
			unit->getShields() >= unit->getType().maxShields()) {
			combat_state.set_shield_battery(nullptr);
		} else {
			if (!recharge_at_shield_battery(unit, shield_battery_unit)) {
				combat_state.set_shield_battery(nullptr);
			}
		}
	}
	
	if (combat_state.shield_battery() == nullptr) {
		if (unit->getShields() < unit->getType().maxShields() / 3) {
			std::map<Unit,int> distances;
			for (auto shield_battery_unit : Broodwar->self()->getUnits()) {
				if (shield_battery_unit->getType() == UnitTypes::Protoss_Shield_Battery &&
					shield_battery_unit->getEnergy() > 10) {
					int distance = unit->getDistance(shield_battery_unit);
					if (distance <= kShieldBatterySeekRange) distances[shield_battery_unit] = distance;
				}
			}
			std::vector<Unit> shield_battery_units = keys_sorted(distances);
			for (auto shield_battery_unit : shield_battery_units) {
				if (recharge_at_shield_battery(unit, shield_battery_unit)) {
					combat_state.set_shield_battery(shield_battery_unit);
					break;
				}
			}
		}
	}
	
	return combat_state.shield_battery() != nullptr;
}

bool MicroManager::recharge_at_shield_battery(Unit unit,Unit shield_battery_unit)
{
	FastPosition start_position(unit->getPosition());
	FastPosition target_position(shield_battery_unit->getPosition());
	
	struct Node
	{
		FastPosition position;
		float f_score;
		
		bool operator<(const Node& o) const
		{
			// Implements > as we use this in a std::priority_queue and the lowest f_score has the highest priority
			return o.f_score < f_score || (o.f_score == f_score && o.position < position);
		}
	};
	
	enum class State : uint8_t
	{
		None = 0, Open, Closed
	};
	
	std::priority_queue<Node> priority_queue;
	SparsePositionGrid<kShieldBatterySeekRange,16,State> state;
	SparsePositionGrid<kShieldBatterySeekRange,16,float> g_score;
	priority_queue.push(Node{ start_position, (float)start_position.getApproxDistance(target_position) });
	state[FastPosition(0, 0)] = State::Open;
	
	while (!priority_queue.empty()) {
		Node current_node = priority_queue.top();
		if (calculate_distance(unit->getType(), current_node.position, shield_battery_unit->getType(), target_position) <= kShieldBatteryRechargeRange) {
			unit_right_click(unit, shield_battery_unit);
			return true;
		}
		priority_queue.pop();
		State& current_state = state[current_node.position - start_position];
		if (current_state != State::Open) continue;
		current_state = State::Closed;
		float current_g_score = g_score[current_node.position - start_position];
		
		for (auto delta : { FastPosition(-16, 0), FastPosition(16, 0), FastPosition(0, -16), FastPosition(0, 16) }) {
			FastPosition next_position = current_node.position + delta;
			if (next_position.getApproxDistance(start_position) < kShieldBatterySeekRange &&
				check_collision(unit, next_position) &&
				state[next_position - start_position] != State::Closed) {
				State& next_state = state[next_position - start_position];
				float tentative_g_score = current_g_score + 1.0f;
				if (next_state == State::None) {
					next_state = State::Open;
				} else if (tentative_g_score >= g_score[next_position - start_position]) {
					continue;
				}
				g_score[next_position - start_position] = tentative_g_score;
				priority_queue.push(Node{ next_position, tentative_g_score + next_position.getApproxDistance(target_position)});
			}
		}
	}
	
	return false;
}

void MicroManager::draw()
{
	std::set<Position> target_positions;
	std::set<Position> stage_positions;
	for (auto& entry : combat_state_) {
		if (entry.second.target_position().isValid()) target_positions.insert(entry.second.target_position());
		if (entry.second.stage_position().isValid()) stage_positions.insert(entry.second.stage_position());
	}
	for (auto& target_position : target_positions) Broodwar->drawCircleMap(target_position, 20, Colors::Red, true);
	for (auto& stage_position : stage_positions) Broodwar->drawCircleMap(stage_position, 20, Colors::Green, true);
	if (run_by_bunker_ != nullptr) {
		const InformationUnit& enemy_bunker_unit = information_manager.enemy_units().at(run_by_bunker_);
		Broodwar->drawCircleMap(enemy_bunker_unit.position, 20, Colors::Blue, true);
		Broodwar->drawCircleMap(run_by_target_position_, 20, Colors::Blue, true);
		for (auto& unit : running_by_) Broodwar->drawCircleMap(unit->getPosition(), 10, Colors::Blue, true);
		for (auto& unit : desperados_) Broodwar->drawCircleMap(unit->getPosition(), 10, Colors::Blue, false);
	}
}

Unit MicroManager::determine_nearby_sieged_tank(Unit combat_unit)
{
	if (!information_manager.enemy_exists(UnitTypes::Terran_Siege_Tank_Siege_Mode)) return nullptr;
	std::vector<Unit> candidates;
	for (auto& unit : harassable_enemy_units_) {
		if (unit->getType() == UnitTypes::Terran_Siege_Tank_Siege_Mode &&
			can_attack_in_range(combat_unit, unit) &&
			combat_unit->getDistance(unit) < WeaponTypes::Arclite_Shock_Cannon.minRange()) {
			if (combat_unit->getTarget() == unit) return unit;
			candidates.push_back(unit);
		}
	}
	return smallest_priority(candidates, [combat_unit](Unit unit){
		return combat_unit->getDistance(unit);
	});
}

Unit MicroManager::combat_unit_closest_to_target(Position target_position)
{
	bool target_above_ground = has_area(WalkPosition(target_position));
	std::map<Unit,int> distances;
	for (Unit combat_unit : combat_units_) {
		if (!combat_unit->isFlying() && combat_unit->getType() != UnitTypes::Protoss_Dark_Templar) {
			int distance = target_above_ground ? ground_distance(combat_unit->getPosition(), target_position) : combat_unit->getDistance(target_position);
			if (distance >= 0) distances[combat_unit] = distance;
		}
	}
	
	return key_with_smallest_value(distances);
}

Unit MicroManager::combat_unit_closest_to_special_unit(Unit special_unit)
{
	std::map<Unit,int> distances;
	if (special_unit->isFlying()) {
		for (Unit combat_unit : combat_units_) {
			if (combat_unit->getType() != UnitTypes::Protoss_Dark_Templar) {
				distances[combat_unit] = special_unit->getDistance(combat_unit);
			}
		}
	} else {
		for (Unit combat_unit : combat_units_) {
			if (!combat_unit->isFlying() && combat_unit->getType() != UnitTypes::Protoss_Dark_Templar) {
				int distance = ground_distance(combat_unit->getPosition(), special_unit->getPosition());
				if (distance >= 0) distances[combat_unit] = distance;
			}
		}
	}
	
	return key_with_smallest_value(distances);
}

bool MicroManager::valid_target(Unit combat_unit,Unit enemy_unit)
{
	CombatState& combat_state = combat_state_.at(combat_unit);
	Position target_position = combat_state.target_position();
	bool ignore_when_attacking = (ignore_when_attacking_.count(enemy_unit) > 0 && !can_attack_in_range_with_prediction(combat_unit, enemy_unit));
	
	if (target_position.isValid()) {
		return !ignore_when_attacking;
	}
	
	if (building_placement_manager.proxy_pylon_position().isValid() && ignore_when_attacking) {
		return false;
	}
	
	const InformationUnit& enemy_information_unit = information_manager.enemy_units().at(enemy_unit);
	if (enemy_information_unit.base_distance == 0) return true;
	if (enemy_unit->getType() == UnitTypes::Protoss_Photon_Cannon && enemy_information_unit.base_distance <= 320) return true;
	
	for (Unit unit : Broodwar->self()->getUnits()) {
		if (unit->getType().isBuilding() &&
			can_attack_in_range(enemy_unit, unit)) {
			return true;
		}
	}
	
	for (auto& entry : worker_manager.worker_map()) {
		const Worker& worker = entry.second;
		if (!worker.order()->is_scouting()) {
			Unit worker_unit = worker.unit();
			if (can_attack_in_range(enemy_unit, worker_unit, 32)) return true;
		}
	}
	
	if (units_near_base_.count(enemy_unit) > 0) return true;
	
	return false;
}

Unit MicroManager::select_enemy_unit_for_scout(Unit combat_unit,bool scout_reached_base)
{
	std::map<Unit,DistanceWithPriority> map;
	for (auto& unit : harassable_enemy_units_) {
		int distance = combat_unit->getDistance(unit);
		int priority = 0;
		UnitType unit_type = unit->getType();
		if ((unit_type == UnitTypes::Terran_Marine || unit_type == UnitTypes::Zerg_Hydralisk) &&
			is_hp_undamaged(combat_unit) &&
			can_attack_in_range(unit, combat_unit)) {
			priority = 1;
		} else if (unit_type == UnitTypes::Zerg_Overlord) {
			priority = 2;
		} else if (scout_reached_base) {
			if (!unit->isCompleted() && unit_type == UnitTypes::Terran_Missile_Turret) {
				priority = 3;
			} else if (unit_type.isWorker() && (unit->isGatheringMinerals() || unit->isGatheringGas())) {
				priority = 4;
			} else if (unit_type.isWorker()) {
				priority = 5;
			}
		}
		if (priority > 0) map[unit] = DistanceWithPriority(distance, priority, unit->getID());
	}
	
	Unit result = key_with_smallest_value(map);
	
	if (result != nullptr &&
		result->getType().isWorker() &&
		(result->isGatheringMinerals() || result->isGatheringGas())) {
		std::vector<Unit> damaged_workers;
		for (auto& unit : harassable_enemy_units_) {
			if (unit->getType().isWorker() && (unit->isGatheringMinerals() || unit->isGatheringGas()) &&
				!is_undamaged(unit) && can_attack_in_range(combat_unit, unit)) {
				damaged_workers.push_back(unit);
			}
		}
		if (!damaged_workers.empty()) {
			result = smallest_priority(damaged_workers, [combat_unit](Unit unit){
				return std::make_tuple(!is_running_away(combat_unit, unit), unit->getHitPoints(), unit->getID());
			});
		}
	}
	
	if (result != nullptr &&
		result->getType().isBuilding() &&
		result->getBuildUnit() != nullptr &&
		result->getBuildUnit()->getType() == UnitTypes::Terran_SCV) result = result->getBuildUnit();
	
	return result;
}

std::pair<Unit,bool> MicroManager::select_enemy_unit_for_combat_unit(Unit combat_unit)
{
	CombatState& combat_state = combat_state_.at(combat_unit);
	Position target_position = combat_state.target_position();
	Position stage_position = combat_state.stage_position();
	
	Unit selected_enemy_unit = nullptr;
	bool enable_attack = true;
	
	if (running_by_.count(combat_unit) > 0 || desperados_.count(combat_unit) > 0) {
		if (desperados_.count(combat_unit) > 0) {
			const InformationUnit& enemy_bunker_unit = information_manager.enemy_units().at(run_by_bunker_);
			std::vector<Unit> enemy_units_not_near_bunker;	// @ Should probably be moved to a more global location
			for (auto& unit : harassable_enemy_units_) {
				// @ Take ignore_when_attacking_ into account?
				if (calculate_distance(enemy_bunker_unit.type, enemy_bunker_unit.position, unit->getType(), unit->getPosition()) > 6 * 32) {
					enemy_units_not_near_bunker.push_back(unit);
				}
			}
			if (enemy_units_not_near_bunker.empty()) {
				if (calculate_distance(UnitTypes::Protoss_Nexus, run_by_target_position_, combat_unit->getType(), combat_unit->getPosition()) < 32) {
					desperados_.erase(combat_unit);
				}
			} else {
				selected_enemy_unit = select_enemy_unit(combat_unit, enemy_units_not_near_bunker);
			}
		}
	} else {
		std::vector<Unit> enemy_units;
		for (auto& enemy_unit : harassable_enemy_units_) if (valid_target(combat_unit, enemy_unit)) enemy_units.push_back(enemy_unit);
		
		if (combat_state.always_advance()) {
			selected_enemy_unit = select_enemy_unit(combat_unit, enemy_units);
		} else {
			std::set<Unit> do_not_attack_units;
			for (auto& cluster : tactics_manager.clusters()) {
				if (!cluster.expect_win(combat_unit)) {
					if (cluster.in_front(combat_unit)) {
						enable_attack = false;
						break;
					} else {
						for (auto enemy_unit : cluster.units) {
							do_not_attack_units.insert(enemy_unit->unit);
						}
					}
				}
			}
			
			std::vector<Unit> attackable_enemy_units;
			if (!enable_attack) {
				for (auto unit : enemy_units) {
					if (units_near_main_base_.count(unit) > 0) {
						attackable_enemy_units.push_back(unit);
					}
				}
			} else {
				for (auto unit : enemy_units) {
					if (do_not_attack_units.count(unit) == 0 ||
						units_near_main_base_.count(unit) > 0 ||
						(target_position.isValid() && unit->getPosition() == target_position)) {
						attackable_enemy_units.push_back(unit);
					}
				}
			}
			selected_enemy_unit = select_enemy_unit(combat_unit, attackable_enemy_units);
		}
	}
	
	return std::make_pair(selected_enemy_unit, enable_attack);
}

Unit MicroManager::select_enemy_unit_for_dark_templar(Unit combat_unit)
{
	CombatState& combat_state = combat_state_.at(combat_unit);
	bool attacking = combat_state.target_position() != Positions::None;
	
	if (combat_unit->getOrderTarget() != nullptr &&
		combat_unit->getOrderTarget()->exists() &&
		combat_unit->getOrderTarget()->getType() == UnitTypes::Terran_Vulture_Spider_Mine) {
		return combat_unit->getOrderTarget();
	}
	std::vector<Unit> incoming_mines;
	for (auto& unit : harassable_enemy_units_) {
		if (unit->getType() == UnitTypes::Terran_Vulture_Spider_Mine &&
			(unit->getOrderTarget() == combat_unit || unit->getDistance(combat_unit) <= UnitTypes::Terran_Vulture_Spider_Mine.seekRange())) incoming_mines.push_back(unit);
	}
	Unit closest_incoming_mine = smallest_priority(incoming_mines, [combat_unit](Unit incoming_mine) {
		return combat_unit->getDistance(incoming_mine);
	});
	if (closest_incoming_mine != nullptr) return closest_incoming_mine;
	
	if (combat_unit->isStuck()) {
		std::vector<Unit> possible_targets;
		for (auto& unit : harassable_enemy_units_) {
			if (!unit->getType().isBuilding() &&
				can_attack_in_range(combat_unit, unit) &&
				is_facing(combat_unit, unit) &&
				!is_running_away(combat_unit, unit)) {
				possible_targets.push_back(unit);
			}
		}
		Unit result = smallest_priority(possible_targets, [combat_unit](Unit unit){
			return std::make_tuple(unit->getHitPoints() + unit->getShields(), unit->getID());
		});
		if (result != nullptr) return result;
	}
	
	const BWEM::Area* current_area = area_at(combat_unit->getPosition());
	Position target_position = combat_state.target_position().isValid() ? combat_state.target_position() : combat_state.stage_position();
	const BWEM::Area* target_area = nullptr;
	if (target_position.isValid()) {
		target_area = area_at(target_position);
	}
	Position attack_position = combat_state.target_position();
	
	auto const is_closer_to_target = [attack_position,combat_unit](Unit unit){
		bool result = true;
		if (attack_position.isValid()) {
			int distance_to_target = ground_distance(combat_unit->getPosition(), attack_position);
			if (distance_to_target >= 0) {
				int predicted_distance = ground_distance(predict_position(unit, 8), attack_position);
				if (predicted_distance >= 0) {
					result = (predicted_distance <= distance_to_target);
				} else {
					int distance = ground_distance(unit->getPosition(), attack_position);
					if (distance >= 0) {
						result = (distance <= distance_to_target);
					}
				}
			}
		}
		return result;
	};
	
	int component = connectivity_grid.component_for_position(combat_unit->getPosition());
	key_value_vector<Unit,DistanceWithPriority> map;
	for (auto& unit : harassable_enemy_units_) {
		if (attacking && ignore_when_attacking_.count(unit) > 0) continue;
		UnitType unit_type = unit->getType();
		if (!unit->isFlying() &&
			safe_to_attack_by_dark_templar(unit) && connectivity_grid.check_reachability_melee(component, unit)) {
			int distance = combat_unit->getDistance(unit);
			int base_priority;
			if (unit_type.isDetector()) {
				base_priority = unit->isCompleted() ? 1 : 2;
			} else if (unit_type == UnitTypes::Terran_Comsat_Station) {
				base_priority = unit->isCompleted() ? 3 : 4;
			} else if (unit_type.isWorker() && (unit->isGatheringMinerals() || unit->isGatheringGas())) {
				base_priority = 5;
			} else if (unit_type.isResourceDepot()) {
				base_priority = unit->isCompleted() ? 300 : 301;
			} else if (unit_type == UnitTypes::Protoss_Forge ||
					   unit_type == UnitTypes::Protoss_Robotics_Facility ||
					   unit_type == UnitTypes::Protoss_Observatory ||
					   unit_type == UnitTypes::Terran_Academy ||
					   unit_type == UnitTypes::Terran_Engineering_Bay ||
					   unit_type == UnitTypes::Terran_Science_Facility ||
					   unit_type == UnitTypes::Zerg_Evolution_Chamber) {
				base_priority = unit->isCompleted() ? 302 : 303;
			} else if (unit_type.isWorker()) {
				base_priority = 304;
			} else {
				base_priority = 600;
			}
			const BWEM::Area* area = information_manager.enemy_units().at(unit).area;
			int priority;
			if (current_area != nullptr && area == current_area) {
				priority = base_priority + 100;
			} else if (target_area != nullptr && area == target_area) {
				priority = base_priority + 200;
			} else {
				priority = base_priority + 300;
			}
			if ((target_area == nullptr || area != target_area) && base_priority >= 300 &&
				unit->getType().canMove() && !is_closer_to_target(unit)) {
				priority = INT_MAX;
			}
			if (priority < INT_MAX) {
				map.emplace_back(unit, DistanceWithPriority(distance, priority, unit->getID()));
			}
		}
	}
	Unit result = key_with_smallest_value(map);
	if (result == nullptr || result->getType() != UnitTypes::Terran_Missile_Turret || result->isCompleted()) {
		result = replace_by_repairing_scv(combat_unit, result);
	}
	return result;
}

bool MicroManager::safe_to_attack_by_dark_templar(Unit enemy_unit)
{
	bool near_comsat = false;
	bool near_enemy = false;
	
	for (auto& entry : information_manager.enemy_units()) {
		if (entry.second.type == UnitTypes::Spell_Scanner_Sweep) {
			if (calculate_distance(enemy_unit->getType(), enemy_unit->getPosition(), entry.second.position) <= 12 * 32) {
				near_comsat = true;
				break;
			}
		} else {
			WeaponType weapon_type = entry.second.type.groundWeapon();
			if (weapon_type != WeaponTypes::None) {
				int range = weapon_max_range(weapon_type, enemy_unit->getPlayer());
				if (calculate_distance(enemy_unit->getType(), enemy_unit->getPosition(), entry.second.type, entry.second.position) <= range + 32) {
					near_enemy = true;
				}
			}
		}
	}
	
	return !near_comsat || !near_enemy;
}

Unit MicroManager::select_enemy_unit(Unit combat_unit,const std::vector<Unit>& enemy_units)
{
	std::vector<Unit> possible_targets;
	for (auto& unit : enemy_units) {
		if (!unit->isStasised() && can_attack(combat_unit, unit)) possible_targets.push_back(unit);
	}
	int component = connectivity_grid.component_for_position(combat_unit->getPosition());
	if (is_melee(combat_unit->getType())) {
		remove_elements_in_place(possible_targets, [this,component](Unit unit){
			return !connectivity_grid.check_reachability_melee(component, unit);
		});
	} else if (!combat_unit->isFlying()) {
		remove_elements_in_place(possible_targets, [this,combat_unit,component](Unit unit){
			return !connectivity_grid.check_reachability_ranged(component, weapon_max_range(combat_unit, unit->isFlying()), unit);
		});
	}
	Unit result = nullptr;
	
	// Attack enemy based on the minimizing the amount of incoming damage
	if (result == nullptr) {
		struct Target
		{
			double incoming_dpf;
			double incoming_dpf_sum;
			Unit unit;
			Position approach_position;
			int approach_distance;
		};
		std::vector<Target> targets;
		std::map<Unit,std::pair<int,FastPosition>> approach_map = calculate_approach_distances_and_positions(combat_unit, possible_targets);
		for (auto& entry : approach_map) {
			Unit unit = entry.first;
			int approach_distance = entry.second.first;
			Position approach_position = entry.second.second;
			double incoming_dpf;
			const InformationUnit& enemy_unit = information_manager.enemy_units().at(unit);
			if (approach_distance == 0) {
				incoming_dpf = calculate_incoming_dpf(combat_unit, combat_unit->getPosition(), enemy_unit);
			} else {
				incoming_dpf = calculate_incoming_dpf(combat_unit, approach_position, enemy_unit);
			}
			if (approach_distance == 0 || (approach_distance >= 0 && incoming_dpf > 0.0)) {
				targets.push_back(Target{incoming_dpf, 0.0, unit, approach_position, approach_distance});
			}
		}
		double incoming_dpf_sum = calculate_incoming_dpf_sum(combat_unit, combat_unit->getPosition());
		for (auto& target : targets) {
			if (target.approach_distance > 0) {
				double destination_incoming_dpf_sum = calculate_incoming_dpf_sum(combat_unit, target.approach_position);
				target.incoming_dpf_sum = std::max(destination_incoming_dpf_sum, incoming_dpf_sum);
			} else {
				target.incoming_dpf_sum = incoming_dpf_sum;
			}
		}
		auto result_target = smallest_priority(targets, [combat_unit](Target target){
			Unit unit = target.unit;
			double travel_damage = 0.0;
			if (target.approach_distance > 0) {
				double travel_frames = target.approach_distance / Broodwar->self()->topSpeed(combat_unit->getType());
				if (!unit_has_target(combat_unit, unit)) travel_frames += Broodwar->getRemainingLatencyFrames();
				travel_damage = target.incoming_dpf_sum * travel_frames;
			}
			double expected_incoming_damage = calculate_kill_time(combat_unit, unit) * (target.incoming_dpf_sum - target.incoming_dpf);
			return std::make_tuple(is_low_priority_target(unit),
								   combat_unit->getType() == UnitTypes::Protoss_Carrier && target.approach_distance > 0,
								   travel_damage + expected_incoming_damage,
								   target_tie_breaker(combat_unit, unit));
		}, Target{0.0, 0.0, nullptr} );
		result = result_target.unit;
	}
	
	// Attack closest enemy
	if (result == nullptr) {
		result = smallest_priority(possible_targets, [combat_unit](Unit unit){
			return std::make_tuple(is_low_priority_target(unit),
								   combat_unit->getDistance(unit),
								   target_tie_breaker(combat_unit, unit));
		});
	}
	
	// When attacking a wall, attack the building with the least health points and largest perimeter
	if (result != nullptr && connectivity_grid.is_wall_building(result)) {
		std::vector<Unit> wall_buildings;
		for (auto& unit : possible_targets) if (connectivity_grid.is_wall_building(unit)) wall_buildings.push_back(unit);
		if (!wall_buildings.empty()) {
			result = smallest_priority(wall_buildings, [component,combat_unit](Unit unit) {
				return std::make_tuple(unit->getHitPoints() + unit->getShields(),
									   -connectivity_grid.wall_building_perimeter(unit, component),
									   target_tie_breaker(combat_unit, unit));
			});
		}
	}
	
	result = replace_by_repairing_scv(combat_unit, result);
	
	// (melee only) When attacking a building that can not attack the combat unit, attack a gathering worker in the same area instead
	if (is_melee(combat_unit->getType()) && result != nullptr && result->getType().isBuilding() && !can_attack(result, combat_unit) && !unit_in_safe_location(result)) {
		if (combat_unit->getOrderTarget() != nullptr && combat_unit->getOrderTarget()->getType().isWorker()) {
			result = combat_unit->getOrderTarget();
		} else {
			std::vector<Unit> targets;
			const BWEM::Area* area = area_at(combat_unit->getPosition());
			for (auto& unit : possible_targets) {
				if (unit->getType().isWorker() &&
					area == information_manager.enemy_units().at(unit).area) targets.push_back(unit);
			}
			Unit target = smallest_priority(targets, [combat_unit](Unit unit){
				return std::make_tuple(!unit->isGatheringMinerals() && !unit->isGatheringMinerals(),
									   combat_unit->getDistance(unit),
									   target_tie_breaker(combat_unit, unit));
			});
			if (target != nullptr) result = target;
		}
	}
	
	return result;
}

int MicroManager::target_tie_breaker(Unit combat_unit,Unit target_unit)
{
	int result;
	if (combat_unit->getOrderTarget() == target_unit) {
		result = INT_MIN;
	} else {
		result = target_unit->getID();
	}
	return result;
}

double MicroManager::calculate_kill_time(Unit combat_unit,Unit enemy_unit)
{
	double hit_chance = calculate_chance_to_hit(combat_unit, enemy_unit);
	if (hit_chance == 0.0) return INFINITY;
	DPF dpf = calculate_damage_per_frame(combat_unit, enemy_unit);
	if (!dpf) return INFINITY;
	return (enemy_unit->getShields() / dpf.shield + enemy_unit->getHitPoints() / dpf.hp) / hit_chance;
}

double MicroManager::calculate_incoming_dpf_sum(Unit combat_unit,Position position)
{
	double result = 0.0;
	for (auto& entry : information_manager.enemy_units()) {
		result += calculate_incoming_dpf(combat_unit, position, entry.second);
	}
	return result;
}

double MicroManager::calculate_incoming_dpf(Unit combat_unit,Position position,const InformationUnit& enemy_unit)
{
	if (enemy_unit.type == UnitTypes::Terran_Medic) return calculate_incoming_dpf_for_medic(combat_unit, position, enemy_unit);
	DPF dpf = calculate_damage_per_frame(enemy_unit.type, Broodwar->enemy(), combat_unit->getType(), combat_unit->getPlayer());
	int distance = calculate_distance(combat_unit->getType(), position, enemy_unit.type, enemy_unit.position);
	if (!dpf) {
		if (distance <= offense_max_range(enemy_unit.type, Broodwar->enemy(), combat_unit->isFlying()) &&
			is_spellcaster(enemy_unit.type)) return 100.0;
		return 0.0;
	}
	if (!is_suicidal_with_target(enemy_unit) &&
		!can_attack_in_range_at_positions(enemy_unit.type, enemy_unit.position, Broodwar->enemy(), combat_unit->getType(), position)) {
		if (enemy_unit.type.isWorker() || is_suicidal(enemy_unit.type)) {
			bool can_attack_enemy;
			if (combat_unit->getPosition() == position && enemy_unit.unit->exists()) {
				can_attack_enemy = can_attack_in_range_with_prediction(combat_unit, enemy_unit.unit);
			} else {
				can_attack_enemy = can_attack_in_range_at_positions(combat_unit->getType(), position, combat_unit->getPlayer(), enemy_unit.type, enemy_unit.position);
			}
			if (can_attack_enemy) return 1e-6;
		}
		return 0.0;
	}
	double base_dpf_value = (combat_unit->getShields() > 5) ? dpf.shield : dpf.hp;
	double result;
	if (combat_unit->getPosition() == position && enemy_unit.unit->exists()) {
		result = (base_dpf_value *
				  calculate_splash_factor(enemy_unit.unit, combat_unit) *
				  calculate_chance_to_hit(enemy_unit.unit, combat_unit));
	} else {
		result = (base_dpf_value *
				  calculate_splash_factor(enemy_unit.type, combat_unit->isFlying()) *
				  calculate_chance_to_hit(enemy_unit.type, enemy_unit.position, position));
	}
	return result;
	
}

double MicroManager::calculate_incoming_dpf_for_medic(Unit combat_unit,Position position,const InformationUnit& enemy_medic_unit)
{
	const EnemyCluster* cluster = tactics_manager.cluster_for_unit(enemy_medic_unit.unit);
	if (cluster == nullptr) return 0.0;
	
	double dpf_sum = 0.0;
	
	for (auto& enemy_unit : cluster->units) {
		if (enemy_unit->unit != enemy_medic_unit.unit &&
			calculate_distance(enemy_unit->type, enemy_unit->position, UnitTypes::Terran_Medic, enemy_medic_unit.position) <= kMedicHealRange) {
			if (enemy_unit->type == UnitTypes::Terran_Medic) return 0.0;
			if (enemy_unit->type.isOrganic()) dpf_sum += calculate_incoming_dpf(combat_unit, position, *enemy_unit);
		}
	}
	
	return dpf_sum;
}

bool MicroManager::is_suicidal_with_target(const InformationUnit& enemy_unit)
{
	return enemy_unit.unit->exists() && is_suicidal(enemy_unit.type) && enemy_unit.unit->getOrderTarget() != nullptr;
}

double MicroManager::calculate_chance_to_hit(Unit attacking_unit,Unit defending_unit)
{
	double result;
	
	if (is_melee(attacking_unit->getType())) {
		result = 1.0;
	} else if (defending_unit->isUnderDarkSwarm()) {
		result = 0.0;
	} else if (Broodwar->getGroundHeight(attacking_unit->getTilePosition()) <
			   Broodwar->getGroundHeight(defending_unit->getTilePosition())) {
		result = 0.53125;
	} else {
		result = 0.99609375;
	}
	
	return result;
}

double MicroManager::calculate_chance_to_hit(UnitType attacking_unit_type,Position attacking_unit_position,Position defending_unit_position)
{
	double result;
	
	if (is_melee(attacking_unit_type)) {
		result = 1.0;
	} else if (Broodwar->getGroundHeight(TilePosition(attacking_unit_position)) <
			   Broodwar->getGroundHeight(TilePosition(defending_unit_position))) {
		result = 0.53125;
	} else {
		result = 0.99609375;
	}
	
	return result;
}

double MicroManager::calculate_splash_factor(Unit attacking_unit,Unit defending_unit)
{
	return calculate_splash_factor(attacking_unit->getType(), defending_unit->isFlying());
}

double MicroManager::calculate_splash_factor(UnitType attacking_unit_type,bool defending_unit_flying)
{
	WeaponType weapon_type;
	if (attacking_unit_type == UnitTypes::Protoss_Reaver) {
		weapon_type = WeaponTypes::Scarab;
	} else {
		weapon_type = defending_unit_flying ? attacking_unit_type.airWeapon() : attacking_unit_type.groundWeapon();
	}
	
	double factor = 1.0;
	if (weapon_type != WeaponTypes::None && weapon_type != WeaponTypes::Unknown) {
		if (weapon_type.explosionType() == ExplosionTypes::Radial_Splash) factor = 2.5;
		else if (weapon_type.explosionType() == ExplosionTypes::Enemy_Splash) factor = 1.5;
	}
	return factor;
}

std::map<Unit,std::pair<int,FastPosition>> MicroManager::calculate_approach_distances_and_positions(Unit combat_unit,std::vector<Unit>& enemy_units)
{
	std::map<Unit,std::pair<int,FastPosition>> result;
	
	if (combat_unit->isFlying()) {
		for (auto& enemy_unit : enemy_units) {
			if (can_attack_in_range_with_prediction(combat_unit, enemy_unit)) {
				result[enemy_unit] = std::make_pair(0, combat_unit->getPosition());
			} else {
				int distance = combat_unit->getDistance(enemy_unit);
				int range = weapon_max_range(combat_unit, enemy_unit->isFlying());
				int approach_distance = std::max(8, distance - range);
				double t = clamp(0.0, double(range) / double(distance), 1.0);
				Position approach_position = lever(enemy_unit->getPosition(), combat_unit->getPosition(), t);
				result[enemy_unit] = std::make_pair(approach_distance, approach_position);
			}
		}
	} else {
		struct EnemyUnit
		{
			Unit unit;
			int span;
			FastPosition position;
		};
		
		std::vector<EnemyUnit> remaining_enemy_units;
		UnitType type = combat_unit->getType();
		for (auto& enemy_unit : enemy_units) {
			int range = weapon_max_range(combat_unit, enemy_unit->isFlying());
			int air_approach_distance = std::max(8, combat_unit->getDistance(enemy_unit) - range);
			if (air_approach_distance <= 320) {
				UnitType enemy_type = enemy_unit->getType();
				int horizontal = std::max(type.dimensionRight() + enemy_type.dimensionLeft(),
										  type.dimensionLeft() + enemy_type.dimensionRight());
				int vertical = std::max(type.dimensionUp() + enemy_type.dimensionDown(),
										type.dimensionDown() + enemy_type.dimensionUp());
				int span = 1 + range + std::max(horizontal, vertical);
				remaining_enemy_units.push_back(EnemyUnit{enemy_unit, span, enemy_unit->getPosition()});
			}
		}
		
		if (!remaining_enemy_units.empty()) {
			int max_distance = 0;
			for (auto& enemy_unit : remaining_enemy_units) {
				int range = weapon_max_range(combat_unit, enemy_unit.unit->isFlying());
				max_distance = std::max(max_distance, 320 - range);
			}
			
			FastPosition initial_position = combat_unit->getPosition();
			std::queue<FastPosition> queue;
			SparsePositionGrid<320,16,int> distances;
			queue.push(initial_position);
			distances[FastPosition(0, 0)] = 1;
			
			while (!queue.empty() && !remaining_enemy_units.empty()) {
				FastPosition current_position = queue.front();
				queue.pop();
				
				int current_distance = distances[current_position - initial_position];
				remove_elements_in_place(remaining_enemy_units, [&result,combat_unit,current_position,current_distance](auto& enemy_unit){
					bool remove_enemy = false;
					Unit unit = enemy_unit.unit;
					int span = enemy_unit.span;
					int delta = std::max(std::abs(current_position.x - enemy_unit.position.x),
										 std::abs(current_position.y - enemy_unit.position.y));
					if (delta <= span) {
						int shot_distance = calculate_distance(combat_unit->getType(), current_position, unit->getType(), enemy_unit.position);
						int range = weapon_max_range(combat_unit, unit->isFlying());
						if (shot_distance <= range) {
							result[unit] = std::make_pair(current_distance - 1, current_position);
							remove_enemy = true;
						}
					}
					return remove_enemy;
				});
				
				for (FastPosition delta_position : { FastPosition(-16, 0), FastPosition(16, 0), FastPosition(0, -16), FastPosition(0, 16) }) {
					FastPosition next_position = current_position + delta_position;
					int& distance = distances[next_position - initial_position];
					if (next_position.isValid() && distance == 0) {
						if (next_position.getApproxDistance(initial_position) <= max_distance && check_collision(combat_unit, next_position)) {
							queue.push(next_position);
							distance = current_distance + 16;
						} else {
							distance = -1;
						}
					}
				}
			}
		}
	}
	
	return result;
}

Unit MicroManager::replace_by_repairing_scv(Unit combat_unit,Unit target)
{
	Unit result = target;
	
	if (result != nullptr) {
		UnitType type = result->getType();
		if (!target->isCompleted() &&
			type.isBuilding() &&
			target->getBuildUnit() != nullptr &&
			target->getBuildUnit()->getType() == UnitTypes::Terran_SCV &&
			connectivity_grid.check_reachability(combat_unit, target->getBuildUnit())) {
			result = target->getBuildUnit();
		} else {
			if (type.getRace() == Races::Terran && type.isMechanical() && !type.isWorker()) {
				std::map<Unit,int> repair_scv_distances;
				for (auto& unit : harassable_enemy_units_) {
					if (unit->getType() == UnitTypes::Terran_SCV &&
						unit->isRepairing() &&
						unit->getOrderTarget() == target &&
						connectivity_grid.check_reachability(combat_unit, unit)) {
						repair_scv_distances[unit] = combat_unit->getDistance(unit);
					}
				}
				if (repair_scv_distances.size() >= 2) result = key_with_smallest_value(repair_scv_distances);
			}
		}
	}
	
	return result;
}

bool MicroManager::storm(Unit high_templar_unit)
{
	const int range = WeaponTypes::Psionic_Storm.maxRange();
	const int radius = WeaponTypes::Psionic_Storm.outerSplashRadius();
	const int threshold = should_emergency_storm(high_templar_unit) ? kStormThresholdUnderAttack : kStormThreshold;
	Position position = high_templar_unit->getPosition();
	
	std::vector<std::pair<Unit,int>> unit_scores;
	int sum_positive_scores = 0;
	for (auto& unit : Broodwar->getAllUnits()) {
		if (!unit->exists() || !unit->isVisible() || !unit->isCompleted() || unit->isStasised()) continue;
		UnitType unit_type = unit->getType();
		if (unit_type.isBuilding()) continue;
		
		int sign;
		if (unit->getPlayer() == Broodwar->self()) {
			sign = -1;
		} else if (unit->getPlayer() == Broodwar->enemy()) {
			sign = 1;
		} else {
			continue;
		}
		
		Position unit_position = unit->getPosition();
		if (unit_position.x < position.x - range - radius - unit_type.dimensionRight() ||
			unit_position.x > position.x + range + radius + unit_type.dimensionLeft() ||
			unit_position.y < position.y - range - radius - unit_type.dimensionDown() ||
			unit_position.y > position.y + range + radius + unit_type.dimensionUp()) {
			continue;
		}
		
		int score = sign * unit_type.supplyRequired();
		unit_scores.emplace_back(unit, score);
		if (score > 0) sum_positive_scores += score;
	}
	
	key_value_vector<Position,std::pair<int,int>> scores;
	if (sum_positive_scores >= threshold) {
		std::vector<Position> existing_storms = list_existing_storm_positions();
		for (int y = position.y - range; y <= position.y + range; y += 32) {
			for (int x = position.x - range; x <= position.x + range; x += 32) {
				Position candidate = Position(x, y);
				int distance = high_templar_unit->getDistance(candidate);
				if (distance < range && !bullet_postion_list_intersects(existing_storms, candidate, WeaponTypes::Psionic_Storm.outerSplashRadius())) {
					std::pair<int,int> score = storm_score(candidate, unit_scores);
					if (score.first >= threshold) scores.emplace_back(candidate, score);
				}
			}
		}
	}
	Position target = key_with_largest_value(scores, Positions::None);
	if (target.isValid()) {
		high_templar_unit->useTech(TechTypes::Psionic_Storm, target);
		int latency_frames = Broodwar->getRemainingLatencyFrames() + 24;
		tentative_storms_.push_back(TentativeStorm(high_templar_unit, Broodwar->getFrameCount() + latency_frames, target));
		return true;
	} else {
		return false;
	}
}

bool MicroManager::should_emergency_storm(Unit high_templar_unit)
{
	bool result = false;
	if (high_templar_unit->isUnderAttack()) {
		int storms_left = high_templar_unit->getEnergy() / TechTypes::Psionic_Storm.energyCost();
		if (storms_left >= 2) {
			result = true;
		} else if (storms_left >= 1) {
			double frames_to_live = calculate_frames_to_live(high_templar_unit);
			result = (frames_to_live <= Broodwar->getRemainingLatencyFrames() + 12);
		}
	}
	return result;
}

std::pair<int,int> MicroManager::storm_score(Position position,const std::vector<std::pair<Unit,int>>& unit_scores)
{
	int result = 0;
	int max_border_distance = 0;
	
	for (auto& entry : unit_scores) {
		Unit unit = entry.first;
		Position unit_position = predict_position(unit);
		int border_distance = WeaponTypes::Psionic_Storm.outerSplashRadius() - unit_position.getApproxDistance(position);
		if (border_distance >= 0) {
			result += entry.second;
			if (entry.second > 0) max_border_distance = std::max(max_border_distance, border_distance);
		}
	}
	
	return std::make_pair(result, max_border_distance);
}

bool MicroManager::stasis(Unit arbiter_unit)
{
	const int range = WeaponTypes::Stasis_Field.maxRange();
	const int radius = kStasisRadius;
	const int threshold = kStasisThreshold;
	Position position = arbiter_unit->getPosition();
	
	std::vector<std::pair<Unit,int>> unit_scores;
	for (auto& unit : Broodwar->getAllUnits()) {
		if (!unit->exists() || !unit->isVisible() || !unit->isCompleted() || unit->isStasised()) continue;
		UnitType unit_type = unit->getType();
		if (unit_type.isBuilding() || unit_type.isWorker() || unit_type == UnitTypes::Zerg_Overlord || is_low_priority_target(unit)) continue;
		
		int score;
		if (unit->getPlayer() == Broodwar->self()) {
			score = -1;
		} else if (unit->getPlayer() == Broodwar->enemy()) {
			score = (is_siege_tank(unit_type) || unit_type == UnitTypes::Terran_Science_Vessel) ? 2 : 1;
		} else {
			continue;
		}
		
		Position unit_position = unit->getPosition();
		if (unit_position.x < position.x - range - radius - unit_type.dimensionRight() ||
			unit_position.x > position.x + range + radius + unit_type.dimensionLeft() ||
			unit_position.y < position.y - range - radius - unit_type.dimensionDown() ||
			unit_position.y > position.y + range + radius + unit_type.dimensionUp()) {
		}
		
		unit_scores.emplace_back(unit, score);
	}
	
	std::map<Position,std::pair<int,int>> scores;
	std::vector<Position> existing_storms = list_existing_storm_positions();
	for (int y = position.y - range; y <= position.y + range; y += 32) {
		for (int x = position.x - range; x <= position.x + range; x += 32) {
			Position candidate = Position(x, y);
			int distance = arbiter_unit->getDistance(candidate);
			if (distance < range && !bullet_postion_list_intersects(existing_storms, candidate, WeaponTypes::Psionic_Storm.outerSplashRadius())) {
				std::pair<int,int> score = stasis_score(position, candidate, unit_scores);
				if (score.first >= threshold) scores[candidate] = score;
			}
		}
	}
	Position target = key_with_largest_value(scores, Positions::None);
	if (target.isValid()) {
		arbiter_unit->useTech(TechTypes::Stasis_Field, target);
		int latency_frames = Broodwar->getRemainingLatencyFrames() + 12;
		tentative_stasises_.push_back(TentativeStasis(arbiter_unit, Broodwar->getFrameCount() + latency_frames, target));
		return true;
	} else {
		return false;
	}
}

std::pair<int,int> MicroManager::stasis_score(Position arbiter_position,Position statis_position,const std::vector<std::pair<Unit,int>>& unit_scores)
{
	int result = 0;
	int min_distance = INT_MAX;
	
	for (auto& entry : unit_scores) {
		Unit unit = entry.first;
		Position unit_position = predict_position(unit);
		if (unit_position.getApproxDistance(statis_position) <= kStasisRadius) {
			int score = entry.second;
			if (score < 0) return std::make_pair(0, 0);
			result += score;
			min_distance = std::min(min_distance, arbiter_position.getApproxDistance(unit_position));
		}
	}
	
	return std::make_pair(result, min_distance);
}

std::vector<Position> MicroManager::list_existing_storm_positions()
{
	std::vector<Position> existing_storms;
	for (Bullet bullet : Broodwar->getBullets()) if (bullet->getType() == BulletTypes::Psionic_Storm) existing_storms.push_back(bullet->getPosition());
	for (TentativeStorm tentative_storm : tentative_storms_) existing_storms.push_back(tentative_storm.position);
	return existing_storms;
}

bool MicroManager::bullet_postion_list_intersects(const std::vector<Position>& positions,Position candidate,int size)
{
	return any_of(positions.begin(), positions.end(), [candidate,size](Position position) {
		return candidate.getApproxDistance(position) <= 2 * size;
	});
}

void MicroManager::apply_worker_need_detection()
{
	Position position = Positions::None;
	for (auto& entry : worker_manager.worker_map()) {
		const Worker& worker = entry.second;
		if (worker.order()->need_detection()) {
			position = worker.unit()->getPosition();
			break;
		}
	}
	
	if (position.isValid() && !observers_.empty()) {
		bool already_assigned = std::any_of(observers_.begin(), observers_.end(), [this,position](auto& unit){
			ObserverTarget& target = observer_targets_[unit];
			return target.scouting && target.position == position;
		});
		if (!already_assigned) {
			Unit closest_observer = smallest_priority(observers_, [position](auto& unit) {
				return unit->getDistance(position);
			});
			ObserverTarget& target = observer_targets_[closest_observer];
			target.scouting = true;
			target.position = position;
		}
	}
}

Position MicroManager::calculate_single_target_position()
{
	Position first_target_position = Positions::None;
	for (auto& entry : combat_state_) {
		if (entry.second.target_position().isValid()) {
			first_target_position = entry.second.target_position();
			break;
		}
	}
	Position target_position = Positions::None;
	if (first_target_position.isValid()) {
		bool all_match = std::all_of(combat_state_.begin(), combat_state_.end(), [first_target_position](auto& entry){
			Position position = entry.second.target_position();
			return !position.isValid() || position == first_target_position;
		});
		if (all_match) target_position = first_target_position;
	}
	return target_position;
}

void MicroManager::update_run_by()
{
	if (run_by_bunker_ == nullptr) {
		// @ Run-by now only works when there is a single target position, possibly not sufficient.
		Position target_position = calculate_single_target_position();
		
		if (!opponent_model.non_basic_combat_unit_seen() && target_position.isValid()) {
			Unit bunker = determine_bunker_for_run_by(target_position);
			
			if (bunker != nullptr) {
				const InformationUnit& bunker_enemy_unit = information_manager.enemy_units().at(bunker);
				std::set<Unit> units_near_bunker = determine_run_by_units_near_bunker(bunker_enemy_unit.position, target_position);
				
				if (check_run_by_damage(bunker, units_near_bunker)) {
					run_by_bunker_ = bunker;
					run_by_target_position_ = target_position;
					for (auto& unit : units_near_bunker) running_by_.insert(unit);
				}
			}
		}
	} else {
		if (information_manager.enemy_units().count(run_by_bunker_) == 0) {
			end_run_by();
		} else {
			const InformationUnit& bunker_enemy_unit = information_manager.enemy_units().at(run_by_bunker_);
			std::set<Unit> units_near_bunker = determine_run_by_units_near_bunker(bunker_enemy_unit.position, run_by_target_position_);
			
			if ((running_by_.empty() && check_run_by_damage(run_by_bunker_, units_near_bunker)) || !running_by_.empty()) {
				for (auto& unit : units_near_bunker) running_by_.insert(unit);
			}
			
			std::vector<Unit> remove_from_running_by;
			for (auto& unit : running_by_) {
				if (units_near_bunker.count(unit) == 0) {
					remove_from_running_by.push_back(unit);
					desperados_.insert(unit);
				} else if (!unit->exists() || unit->getPlayer() != Broodwar->self()) {
					remove_from_running_by.push_back(unit);
				}
			}
			for (Unit unit : remove_from_running_by) running_by_.erase(unit);
			
			std::vector<Unit> remove_from_desperados;
			for (auto& unit : desperados_) {
				if (!unit->exists() || unit->getPlayer() != Broodwar->self()) {
					remove_from_desperados.push_back(unit);
				}
			}
			for (Unit unit : remove_from_desperados) desperados_.erase(unit);
			
			if (running_by_.empty() && desperados_.empty()) end_run_by();
		}
	}
}

void MicroManager::end_run_by()
{
	run_by_bunker_ = nullptr;
	run_by_target_position_ = Positions::None;
	running_by_.clear();
	desperados_.clear();
}

std::set<Unit> MicroManager::determine_run_by_units_near_bunker(Position bunker_position,Position target_position)
{
	UnitType target_unit_type = UnitTypes::None;
	TilePosition target_tile_position = TilePositions::None;
	for (auto& entry : information_manager.enemy_units()) {
		if (entry.second.type.isBuilding() && !entry.second.flying && entry.second.position == target_position) {
			target_unit_type = entry.second.type;
			target_tile_position = entry.second.tile_position();
			break;
		}
	}
	
	int bunker_marine_range = weapon_max_range(WeaponTypes::Gauss_Rifle, Broodwar->enemy()) + 64;
	std::set<Unit> result;
	for (auto& unit : combat_units_) {
		if ((unit->getType() == UnitTypes::Protoss_Zealot || unit->getType() == UnitTypes::Protoss_Dragoon) &&
			calculate_distance(UnitTypes::Terran_Marine, bunker_position, unit->getType(), unit->getPosition()) <= bunker_marine_range) {
			int component = connectivity_grid.component_for_position(unit->getPosition());
			bool reachable;
			if (target_unit_type != UnitTypes::None) {
				reachable = connectivity_grid.building_has_component(target_unit_type, target_tile_position, component);
			} else {
				reachable = (connectivity_grid.component_for_position(target_position) == component);
			}
			if (reachable) result.insert(unit);
		}
	}
	return result;
}

bool MicroManager::bunker_runby_possible()
{
	bool result = false;
	Position target_position = tactics_manager.enemy_start_position();
	if (!opponent_model.non_basic_combat_unit_seen() && target_position.isValid()) {
		Unit bunker = determine_bunker_for_run_by(target_position);
		if (bunker != nullptr) {
			std::set<Unit> runby_units;
			for (auto& unit : Broodwar->self()->getUnits()) {
				if (unit->isCompleted() && unit->isVisible() && !unit->isStasised() && !unit->isLoaded() &&
					(unit->getType() == UnitTypes::Protoss_Zealot || unit->getType() == UnitTypes::Protoss_Dragoon)) {
					runby_units.insert(unit);
				}
			}
			result = check_run_by_damage(bunker, runby_units);
		}
	}
	return result;
}

Unit MicroManager::determine_bunker_for_run_by(Position target_position)
{
	int bunker_marine_range = weapon_max_range(WeaponTypes::Gauss_Rifle, Broodwar->enemy()) + 64;
	std::vector<Unit> bunkers;
	for (auto& entry : information_manager.enemy_units()) {
		if (entry.second.type == UnitTypes::Terran_Bunker &&
			(!entry.second.unit->exists() || entry.second.unit->isCompleted()) &&
			calculate_distance(UnitTypes::Terran_Marine, entry.second.position, target_position) > bunker_marine_range) {
			bunkers.push_back(entry.second.unit);
		}
	}
	return bunkers.size() == 1 ? bunkers[0] : nullptr;
}

bool MicroManager::check_run_by_damage(Unit bunker,const std::set<Unit>& units_near_bunker)
{
	if (units_near_bunker.empty()) return false;
	
	// @ Base distance on how close the unit is already to the bunker
	int distance = 6 * 32 + std::max(UnitTypes::Terran_Bunker.width(), UnitTypes::Terran_Bunker.height());
	int hp_sum = 0;
	double speed_sum = 0.0;
	double dpf_sum = 0.0;
	for (auto& unit : units_near_bunker) {
		DPF dpf = calculate_damage_per_frame(bunker, unit);
		int hp_and_shields = unit->getHitPoints() + unit->getShields();
		double hp_fract = double(unit->getHitPoints()) / double(hp_and_shields);
		double shield_fract = double(unit->getShields()) / double(hp_and_shields);
		dpf_sum += (hp_fract * dpf.hp + shield_fract * dpf.shield);
		hp_sum += hp_and_shields;
		speed_sum += unit->getType().topSpeed();
	}
	double speed = speed_sum / units_near_bunker.size();
	double dpf = dpf_sum / units_near_bunker.size();
	double frames = distance / speed;
	double damage = frames * dpf;
	return damage <= 0.5 * hp_sum;
}
