#include "BananaBrain.h"

int MineralGas::can_pay_times(const MineralGas& o) const
{
	if (o.minerals <= 0) return 0;
	int times = minerals / o.minerals;
	if (o.gas <= 0) {
		return std::max(0, times);
	} else {
		return std::max(0, std::min(times, gas / o.gas));
	}
}

bool TrainDistribution::is_nonempty() const
{
	for (auto& entry : weights_) {
		if (entry.second > 0.0) return true;
	}
	return false;
}

UnitType TrainDistribution::sample() const
{
	std::vector<std::pair<UnitType,double>> distribution;
	double sum = 0.0;
	for (auto& entry : weights_) {
		if (entry.second > 0.0) {
			sum += entry.second;
			distribution.push_back(std::pair<UnitType,double>(entry.first, sum));
		}
	}
	
	if (distribution.empty()) return UnitTypes::None;
	if (distribution.size() == 1) return distribution[0].first;
	
	double r = sum * ((double)rand() / (RAND_MAX + 1));
	for (size_t i = 0; i < distribution.size() - 1; i++) {
		if (r < distribution[i].second) return distribution[i].first;
	}
	return distribution[distribution.size() - 1].first;
}

CostPerMinute TrainDistribution::cost_per_minute(int producers) const
{
	if (is_empty() || producers == 0) return CostPerMinute();
	
	CostPerMinute result;
	double sum = 0.0;
	for (auto& entry : weights_) {
		if (entry.second > 0.0) {
			result += entry.second * cost_per_minute(entry.first);
			sum += entry.second;
		}
	}
	return (producers / sum) * result;
}

CostPerMinute TrainDistribution::cost_per_minute(UnitType unit_type)
{
	int frames = unit_type.buildTime();
	double units_per_minute = 24.0 * 60.0 / frames;
	int supply_required = unit_type.supplyRequired();
	if (unit_type.isTwoUnitsInOneEgg()) supply_required *= 2;
	return CostPerMinute(unit_type.mineralPrice() * units_per_minute,
						 unit_type.gasPrice() * units_per_minute,
						 supply_required * units_per_minute);
}

int TrainDistribution::additional_producers(bool this_train_distribution_only) const
{
	if (is_empty()) return 0;
	
	int times1 = spending_manager.spendable().can_pay_times(MineralGas(builder_type_));
	if (times1 <= 0) return 0;
	
	CostPerMinute cost = this_train_distribution_only ? (spending_manager.worker_training_cost_per_minute() + spending_manager.training_cost_per_minute(*this)) : cost_per_minute();
	MineralGas remainder = spending_manager.remainder() + spending_manager.spendable();
	MineralGas spend = MineralGas(builder_type_) + MineralGas((int)cost.minerals, (int)cost.gas);
	int times2 = remainder.can_pay_times(spend);
	
	return std::min(times1, times2);
}

void ResourceCounter::init(int value)
{
	for (size_t i = 0; i < values_.size(); i++) values_[i] = value;
}

void ResourceCounter::process_value(int value)
{
	int difference = value - values_[index_];
	values_[index_] = value;
	index_ = (index_ + 1) % values_.size();
	per_minute_ = 24.0 * 60.0 * difference / (double)values_.size();
}

void BuildingManager::update_supply_requests()
{
	if (automatic_supply_) {
		Player self = Broodwar->self();
		if (self->supplyTotal(Races::Protoss) < TrainingManager::kSupplyCap) {
			const UnitType supply_building = UnitTypes::Protoss_Pylon;
			double supply_build_minutes = (100 + supply_building.buildTime()) / (60.0 * 24.0);
			double supply_needed_per_minute = spending_manager.training_cost_per_minute().supply + spending_manager.worker_training_cost_per_minute().supply;
			double supply_requirement = supply_build_minutes * supply_needed_per_minute;
			double supply_shortage = self->supplyUsed(Races::Protoss) + supply_requirement - self->supplyTotal(Races::Protoss);
			int supply_building_shortage = (int) ceil(supply_shortage / supply_building.supplyProvided());
			set_requested_building_count_at_least(supply_building, building_count(supply_building) + supply_building_shortage, true);
		}
	}
}

void BuildingManager::init_building_count_map()
{
	building_count_.clear();
	
	std::set<TilePosition> building_worker_positions;
	for (auto& entry : worker_manager.worker_map()) {
		const Worker& worker = entry.second;
		UnitType unit_type = worker.order()->building_type();
		if (unit_type != UnitTypes::None) {
			building_count_[unit_type].planned++;
			building_worker_positions.insert(worker.order()->building_position());
		}
	}
	
	for (auto& unit : Broodwar->self()->getUnits()) {
		if (unit->getType().isBuilding()) {
			UnitType unit_type = unit->getType();
			BuildingCount& building_count = building_count_[unit_type];
			if (unit->isCompleted()) {
				building_count.actual++;
			} else {
				if (building_worker_positions.count(unit->getTilePosition()) == 0) {
					building_count.planned++;
					building_count.warping++;
				}
			}
		}
	}
}

void BuildingManager::init_base_defense_map()
{
	base_defense_.clear();
	
	std::set<TilePosition> planned_cannon_positions;
	std::set<TilePosition> completed_cannon_positions;
	std::vector<TilePosition> planned_pylon_positions;
	std::vector<TilePosition> completed_pylon_positions;
	for (auto& entry : worker_manager.worker_map()) {
		const Worker& worker = entry.second;
		if (worker.order()->building_type() == UnitTypes::Protoss_Photon_Cannon) {
			planned_cannon_positions.insert(worker.order()->building_position());
		} else if (worker.order()->building_type() == UnitTypes::Protoss_Pylon) {
			planned_pylon_positions.push_back(worker.order()->building_position());
		}
	}
	for (auto& unit : Broodwar->self()->getUnits()) {
		if (unit->getType() == UnitTypes::Protoss_Photon_Cannon) {
			if (unit->isCompleted()) {
				completed_cannon_positions.insert(unit->getTilePosition());
			} else {
				planned_cannon_positions.insert(unit->getTilePosition());
			}
		} else if (unit->getType() == UnitTypes::Protoss_Pylon) {
			if (unit->isCompleted()) {
				completed_pylon_positions.push_back(unit->getTilePosition());
			} else {
				planned_pylon_positions.push_back(unit->getTilePosition());
			}
		}
	}
	
	for (auto& entry : building_placement_manager.photon_cannon_positions()) {
		const BWEM::Base* base = entry.first;
		const std::vector<TilePosition>& positions = entry.second;
		for (auto& position : positions) {
			if (completed_cannon_positions.count(position) > 0) {
				base_defense_[base].cannons_actual++;
			} else if (planned_cannon_positions.count(position) > 0) {
				base_defense_[base].cannons_planned++;
			}
		}
		for (auto& position : positions) {
			bool position_powered = false;
			for (auto& pylon_position : completed_pylon_positions) {
				if (building_placement_manager.check_power(pylon_position, UnitTypes::Protoss_Photon_Cannon, position)) {
					base_defense_[base].pylon_exists = true;
					position_powered = true;
					break;
				}
			}
			if (!base_defense_[base].pylon_planned) {
				for (auto& pylon_position : planned_pylon_positions) {
					if (building_placement_manager.check_power(pylon_position, UnitTypes::Protoss_Photon_Cannon, position)) {
						base_defense_[base].pylon_planned = true;
						break;
					}
				}
			}
		}
	}
}

void BuildingManager::init_upgrade_and_research()
{
	upgrade_request_.clear();
	research_request_.clear();
}

void BuildingManager::set_requested_building_count_at_least(UnitType unit_type,int count,bool important)
{
	BuildingCount& building_count = building_count_[unit_type];
	int additional = std::max(0, count - building_count.actual - building_count.planned);
	building_count.additional = std::max(building_count.additional, additional);
	if (important) {
		building_count.additional_important = std::max(building_count.additional_important, additional);
	}
}

void BuildingManager::request_base_defense_pylon(const BWEM::Base* base)
{
	BaseDefense& base_defense = base_defense_[base];
	if (!base_defense.pylon_exists && !base_defense.pylon_planned) {
		base_defense.pylon_add = true;
	}
	building_count_[UnitTypes::Protoss_Pylon];
}

void BuildingManager::set_requested_base_defense_cannon_count_at_least(const BWEM::Base* base,int count)
{
	BaseDefense& base_defense = base_defense_[base];
	int additional = std::max(0, count - base_defense.cannons_actual - base_defense.cannons_planned);
	base_defense.cannons_additional = std::max(base_defense.cannons_additional, additional);
	building_count_[UnitTypes::Protoss_Photon_Cannon];
}

bool BuildingManager::required_buildings_exist_for_building(UnitType unit_type)
{
	for (auto& entry : unit_type.requiredUnits()) {
		UnitType required_type = entry.first;
		if (required_type.isBuilding() && building_count_[required_type].actual == 0) return false;
	}
	return true;
}

void BuildingManager::apply_building_requests(bool important)
{
	bool no_workers_for_building = false;
	
	for (auto& entry : building_count_) {
		UnitType unit_type = entry.first;
		if (!required_buildings_exist_for_building(unit_type)) continue;
		MineralGas unit_cost = MineralGas(unit_type);
		BuildingCount& building_count = entry.second;
		
		int additional_buildings_requested;
		if (important) {
			additional_buildings_requested = building_count.additional_important;
		} else {
			additional_buildings_requested = building_count.additional - building_count.additional_important;
		}
		
		if (!important && unit_type == UnitTypes::Protoss_Pylon) {
			for (auto& entry : base_defense_) {
				int& retry_frame = base_defense_retry_frame_[entry.first];
				if (no_workers_for_building || retry_frame > Broodwar->getFrameCount()) break;
				if (entry.second.pylon_add && spending_manager.try_spend(unit_cost)) {
					TilePosition tile_position = building_placement_manager.place_base_defense_pylon(entry.first);
					if (tile_position.isValid()) {
						bool success = worker_manager.assign_building_request(unit_type, tile_position);
						if (success) {
							entry.second.pylon_planned = true;
							if (additional_buildings_requested > 0) additional_buildings_requested--;
						}
						if (!success) no_workers_for_building = true;
					}
					if (!tile_position.isValid()) {
						retry_frame = Broodwar->getFrameCount() + kBuildingPlacementFailedRetryFrames;
						if (debug_options.draw_enabled()) Broodwar << Broodwar->getFrameCount() << ": Failed to place " << unit_type.getName() << std::endl;
					}
				}
			}
		}
		
		if (!important && unit_type == UnitTypes::Protoss_Photon_Cannon) {
			for (auto& entry : base_defense_) {
				int& retry_frame = base_defense_retry_frame_[entry.first];
				for (int i = 0; i < entry.second.cannons_additional && !no_workers_for_building && retry_frame <= Broodwar->getFrameCount(); i++) {
					if (spending_manager.try_spend(unit_cost)) {
						TilePosition tile_position = building_placement_manager.place_base_defense_photon_cannon(entry.first);
						if (tile_position.isValid()) {
							bool success = worker_manager.assign_building_request(unit_type, tile_position);
							if (success) {
								entry.second.cannons_planned++;
								if (additional_buildings_requested > 0) additional_buildings_requested--;
							}
							if (!success) no_workers_for_building = true;
						}
						if (!tile_position.isValid()) {
							retry_frame = Broodwar->getFrameCount() + kBuildingPlacementFailedRetryFrames;
							if (debug_options.draw_enabled()) Broodwar << Broodwar->getFrameCount() << ": Failed to place " << unit_type.getName() << std::endl;
						}
					}
				}
			}
		}
		
		if (additional_buildings_requested > 0) {
			if (unit_type == UnitTypes::Protoss_Assimilator) {
				std::vector<TilePosition> available_geyser_tile_positions = determine_available_geyser_tile_position();
				for (int i = 0; i < std::min<int>(available_geyser_tile_positions.size(), additional_buildings_requested); i++) {
					if (spending_manager.try_spend(unit_cost)) {
						bool success = worker_manager.assign_building_request(unit_type, available_geyser_tile_positions[i]);
						if (success) building_count.planned++;
					}
				}
			} else if (unit_type == UnitTypes::Protoss_Nexus) {
				std::vector<TilePosition> available_resource_depot_positions = determine_available_resource_depot_tile_positions(unit_type);
				for (int i = 0; i < std::min<int>(available_resource_depot_positions.size(), additional_buildings_requested); i++) {
					if (spending_manager.try_spend(unit_cost)) {
						bool success = worker_manager.assign_building_request(unit_type, available_resource_depot_positions[i]);
						if (success) building_count.planned++;
					}
				}
			} else {
				int& retry_frame = (unit_type == UnitTypes::Protoss_Pylon) ? pylon_retry_frame_ : non_pylon_building_retry_frame_;
				
				for (int i = 0; i < additional_buildings_requested && !no_workers_for_building && retry_frame <= Broodwar->getFrameCount(); i++) {
					if (spending_manager.try_spend(unit_cost)) {
						TilePosition tile_position = building_placement_manager.place_building(unit_type);
						if (tile_position.isValid()) {
							bool success = worker_manager.assign_building_request(unit_type, tile_position);
							if (success) building_count.planned++;
							if (!success) no_workers_for_building = true;
						}
						if (!tile_position.isValid()) {
							retry_frame = Broodwar->getFrameCount() + kBuildingPlacementFailedRetryFrames;
							if (debug_options.draw_enabled()) Broodwar << Broodwar->getFrameCount() << ": Failed to place " << unit_type.getName() << std::endl;
						}
					}
				}
			}
		}
	}
}

std::vector<TilePosition> BuildingManager::determine_available_geyser_tile_position()
{
	std::set<TilePosition> all_positions;
	for (auto& base : base_state.controlled_bases()) {
		for (auto& geyser : base->Geysers()) all_positions.insert(geyser->TopLeft());
	}
	
	for (auto& unit : Broodwar->getAllUnits()) if (unit->exists() && unit->getType().isRefinery()) all_positions.erase(unit->getTilePosition());
	
	for (auto& entry : worker_manager.worker_map()) {
		auto& worker = entry.second;
		if (worker.order()->building_type().isRefinery()) all_positions.erase(worker.order()->building_position());
	}
	
	std::vector<TilePosition> result;
	for (auto& position : all_positions) result.push_back(position);
	// TODO Order of results?
	return result;
}

std::vector<TilePosition> BuildingManager::determine_available_resource_depot_tile_positions(UnitType unit_type)
{
	std::vector<TilePosition> result;
	for (auto& base : base_state.next_available_bases()) {
		TilePosition tile_position = base->Location();
		result.push_back(tile_position);
	}
	return result;
}

void BuildingManager::apply_upgrades()
{
	for (Unit unit : Broodwar->self()->getUnits()) {
		if (unit->isPowered()) {
			for (UpgradeType upgrade_type : upgrade_request_) {
				int current_level = Broodwar->self()->getUpgradeLevel(upgrade_type);
				if (current_level < upgrade_type.maxRepeats() &&
					unit->isIdle() &&
					upgrade_type.whatUpgrades() == unit->getType() &&
					spending_manager.try_spend(MineralGas(upgrade_type, Broodwar->self()->getUpgradeLevel(upgrade_type) + 1))) {
					unit->upgrade(upgrade_type);
				}
			}
		}
	}
}

void BuildingManager::apply_research()
{
	for (Unit unit : Broodwar->self()->getUnits()) {
		if (unit->isPowered()) {
			for (TechType tech_type : research_request_) {
				if (!Broodwar->self()->hasResearched(tech_type) &&
					unit->isIdle() &&
					tech_type.whatResearches() == unit->getType() &&
					spending_manager.try_spend(MineralGas(tech_type))) {
					unit->research(tech_type);
				}
			}
		}
	}
}

void BuildingManager::cancel_buildings_of_type(UnitType building_type)
{
	for (auto& unit : Broodwar->self()->getUnits()) {
		if (unit->getType() == building_type && !unit->isCompleted()) unit->cancelConstruction();
	}
	worker_manager.cancel_building_construction_for_type(building_type);
}

void BuildingManager::cancel_doomed_buildings()
{
	for (Unit unit : Broodwar->self()->getUnits()) {
		if (unit->getType().isBuilding() && !unit->isCompleted() && unit->isUnderAttack()) {
			double frames_to_live = calculate_frames_to_live(unit);
			if (frames_to_live < 24.0) unit->cancelConstruction();
		}
	}
}

MineralGas SpendingManager::income_per_minute() const
{
	return MineralGas((int)mineral_counter_.per_minute(),
					  (int)gas_counter_.per_minute());
}

CostPerMinute SpendingManager::training_cost_per_minute() const
{
	return (training_cost_per_minute(training_manager.gateway_train_distribution()) +
			training_cost_per_minute(training_manager.robotics_facility_train_distribution()) +
			training_cost_per_minute(training_manager.stargate_train_distribution()));
}

CostPerMinute SpendingManager::worker_training_cost_per_minute() const
{
	if (training_manager.worker_count_including_unfinished() < worker_manager.determine_max_workers() && training_manager.probe_production()) {
		int count = 0;
		for (Unit unit : Broodwar->self()->getUnits()) {
			if (unit->isCompleted() && unit->getType() == UnitTypes::Protoss_Nexus) {
				const BWEM::Base* base = base_state.base_for_tile_position(unit->getTilePosition());
				if (worker_manager.allow_mining_from(base)) count++;
			}
		}
		
		return TrainDistribution::cost_per_minute(UnitTypes::Protoss_Probe) * count;
	} else {
		return CostPerMinute();
	}
}

CostPerMinute SpendingManager::total_cost_per_minute() const
{
	return training_cost_per_minute() + worker_training_cost_per_minute();
}

MineralGas SpendingManager::remainder() const
{
	CostPerMinute cost_per_minute = total_cost_per_minute();
	return income_per_minute() - MineralGas((int)cost_per_minute.minerals, (int)cost_per_minute.gas);
}

CostPerMinute SpendingManager::training_cost_per_minute(const TrainDistribution& train_queue)
{
	int active_building_count = 0;
	for (auto& building  : Broodwar->self()->getUnits()) {
		if (building->isCompleted() && building->isPowered() && building->getType() == train_queue.builder_type()) {
			active_building_count++;
		}
	}
	
	return train_queue.cost_per_minute(active_building_count);
}

void SpendingManager::init_resource_counters()
{
	mineral_counter_.init(Broodwar->self()->gatheredMinerals());
	gas_counter_.init(Broodwar->self()->gatheredGas());
}

void SpendingManager::init_spendable()
{
	spendable_ = MineralGas(Broodwar->self()) - worker_manager.worker_reserved_mineral_gas();
	spendable_out_of_minerals_first_ = false;
	spendable_out_of_gas_first_ = false;
	mineral_counter_.process_value(Broodwar->self()->gatheredMinerals());
	gas_counter_.process_value(Broodwar->self()->gatheredGas());
}

bool SpendingManager::try_spend(const MineralGas& mineral_gas)
{
	bool result = spendable_.can_pay(mineral_gas);
	spendable_ -= mineral_gas;
	if (!spendable_out_of_minerals_first_ && !spendable_out_of_gas_first_) {
		if (spendable_.minerals < 0) spendable_out_of_minerals_first_ = true;
		if (spendable_.gas < 0) spendable_out_of_gas_first_ = true;
	}
	return result;
}

void TrainingManager::apply_worker_train_orders()
{
	for (Unit unit : Broodwar->self()->getUnits()) {
		if (unit->isCompleted() && unit->getType() == UnitTypes::Protoss_Nexus) {
			const BWEM::Base* base = base_state.base_for_tile_position(unit->getTilePosition());
			if (unit->getTrainingQueue().empty() &&
				worker_count_including_unfinished() < worker_manager.determine_max_workers() &&
				worker_manager.allow_mining_from(base) &&
				can_train(UnitTypes::Protoss_Probe)) {
				unit->train(UnitTypes::Protoss_Probe);
			}
		}
	}
}

void TrainingManager::apply_train_orders()
{
	MineralGas reserved;
	
	apply_interceptor_train_orders();
	apply_scarab_train_orders();
	apply_train_orders(robotics_facility_train_distribution_, reserved);
	apply_train_orders(stargate_train_distribution_, reserved);
	apply_train_orders(gateway_train_distribution_, reserved);
	
	spending_manager.try_spend(reserved);
}

void TrainingManager::apply_interceptor_train_orders()
{
	int max_interceptor_count = (Broodwar->self()->getUpgradeLevel(UpgradeTypes::Carrier_Capacity) == 0) ? 4 : 8;
	
	for (Unit unit : Broodwar->self()->getUnits()) {
		if (unit->isCompleted() &&
			!unit->isStasised() &&
			unit->getType() == UnitTypes::Protoss_Carrier &&
			unit->getInterceptorCount() < max_interceptor_count &&
			unit->getTrainingQueue().empty() &&
			can_train(UnitTypes::Protoss_Interceptor)) {
			unit->train(UnitTypes::Protoss_Interceptor);
		}
	}
}

void TrainingManager::apply_scarab_train_orders()
{
	int max_scarab_count = 5;
	
	for (Unit unit : Broodwar->self()->getUnits()) {
		if (unit->isCompleted() &&
			!unit->isStasised() &&
			unit->getType() == UnitTypes::Protoss_Reaver &&
			unit->getScarabCount() < max_scarab_count &&
			unit->getTrainingQueue().empty() &&
			can_train(UnitTypes::Protoss_Scarab)) {
			unit->train(UnitTypes::Protoss_Scarab);
		}
	}
}

void TrainingManager::apply_train_orders(TrainDistribution& train_queue,MineralGas& reserved)
{
	for (auto& building : Broodwar->self()->getUnits()) {
		if (building->isCompleted() && building->isPowered() && building->getType() == train_queue.builder_type()) {
			if (train_queue.is_nonempty()) {
				UnitType unit_type = train_queue.sample();
				if (building->getTrainingQueue().empty() && can_train(unit_type)) {
					building->train(unit_type);
				} else if (!building->getTrainingQueue().empty()) {
					UnitType unit_type = building->getTrainingQueue()[0];
					double fraction_remaining = (double)building->getRemainingBuildTime() / (double)unit_type.buildTime();
					double fraction_finished = 1.0 - fraction_remaining;
					int minerals = (int)(fraction_finished * unit_type.mineralPrice() + 0.5);
					int gas = (int)(fraction_finished * unit_type.gasPrice() + 0.5);
					reserved += MineralGas(minerals, gas);
				}
			}
		}
	}
}

bool TrainingManager::can_train(UnitType type)
{
	Player self = Broodwar->self();
	int supply_left = self->supplyTotal(Races::Protoss) - self->supplyUsed(Races::Protoss);
	
	bool enough_supply = supply_left >= type.supplyRequired();
	bool enough_resources = spending_manager.try_spend(MineralGas(type));
	
	return enough_supply && enough_resources;
}

int TrainingManager::worker_count_including_unfinished()
{
	int result = 0;
	for (auto unit : Broodwar->self()->getUnits()) {
		if (unit->getType().isWorker()) result++;
	}
	return result;
}

void TrainingManager::init_unit_count_map()
{
	unit_count_.clear();
	for (Unit unit : Broodwar->self()->getUnits()) {
		if (!unit->getType().isBuilding()) {
			unit_count_[unit->getType()].count++;
			if (unit->isCompleted()) unit_count_[unit->getType()].count_completed++;
			if (unit->isLoaded()) unit_count_[unit->getType()].count_loaded++;
		}
	}
}

void TrainingManager::onUnitLost(Unit unit)
{
	if (unit->getPlayer() == Broodwar->self() &&
		unit->isCompleted() &&
		!unit->getType().isBuilding() &&
		!unit->getType().isWorker()) {
		lost_unit_count_++;
	}
}
