#pragma once

struct InformationUnit
{
	Unit unit;
	int first_frame = -1;
	int start_frame = -1;
	int stasis_end_frame = -1;
	int frame;
	UnitType type;
	Position position;
	bool flying;
	const BWEM::Area* area;
	int base_distance;	// Distance of this unit to a controlled or planned area, zero if inside
	
	bool is_current() const
	{
		return frame == Broodwar->getFrameCount();
	}
	
	bool is_stasised() const
	{
		if (unit->exists()) {
			return unit->isStasised();
		} else {
			return stasis_end_frame > Broodwar->getFrameCount();
		}
	}
	
	TilePosition tile_position() const
	{
		return TilePosition(Position(position.x - type.tileWidth() * 16,
									 position.y - type.tileHeight() * 16));
	}
	
	void update(Unit a_unit);
	int determine_base_distance();
	void clean_outdated_unit_position();
};

class InformationManager : public Singleton<InformationManager>
{
public:
	void update_units_and_buildings();
	void update_information();
	void draw();
	void onUnitDestroy(Unit unit);
	void onUnitDiscover(Unit unit);
	void onUnitEvade(Unit unit);
	
	const std::map<Unit,InformationUnit>& neutral_units() const { return neutral_units_; }
	const std::map<Unit,InformationUnit>& enemy_units() const { return enemy_units_; }
	bool enemy_building_seen() const;
	
	int enemy_count(UnitType unit_type) const { return get_or_default(enemy_count_, unit_type); }
	bool enemy_seen(UnitType unit_type) const { return enemy_seen_.count(unit_type) > 0; }
	bool enemy_exists(UnitType unit_type) const { return get_or_default(enemy_count_, unit_type) > 0; }
	int upgrade_level(Player player,UpgradeType upgrade_type) const;
	int bunker_marines_loaded(Unit bunker) const;
	
private:
	
	class MarineCountRingBuffer
	{
	public:
		MarineCountRingBuffer();
		void push(int value);
		int max() const;
		void decrement();
		void increment();
	private:
		std::array<int,24> values_;
		int counter_ = 0;
	};
	
	std::map<Unit,InformationUnit> neutral_units_;
	std::map<Unit,InformationUnit> enemy_units_;
	std::map<UnitType,int> enemy_count_;
	std::set<UnitType> enemy_seen_;
	std::map<Player,std::map<UpgradeType,int>> upgrades_;
	std::map<int,int> bullet_timestamps_;
	std::map<Unit,MarineCountRingBuffer> bunker_marines_loaded_;
	std::map<Unit,int> bunker_marines_delay_update_until_timestamp_;
	
	void clean_buildings_and_spells();
	void clean_outdated_unit_positions();
	void update_upgrades();
	void update_enemy_count();
	void update_bullet_timestamps();
	void update_bunker_marines_loaded();
	void update_bunker_marine_range();
};
