#include "BananaBrain.h"

void InformationUnit::update(Unit a_unit)
{
	unit = a_unit;
	if (first_frame == -1) first_frame = Broodwar->getFrameCount();
	if (start_frame == -1) start_frame = estimate_building_start_frame_based_on_health_points(unit);
	frame = Broodwar->getFrameCount();
	type = a_unit->getType();
	position = a_unit->getPosition();
	flying = a_unit->isFlying();
	stasis_end_frame = Broodwar->getFrameCount() + a_unit->getStasisTimer();
	area = area_at(position);
	base_distance = determine_base_distance();
}

int InformationUnit::determine_base_distance()
{
	int result = INT_MAX;
	
	if (area != nullptr && base_state.controlled_and_planned_areas().count(area) > 0) {
		result = 0;
	} else {
		for (auto& cp : base_state.border().chokepoints()) {
			Position cp_position = chokepoint_center(cp);
			int distance = calculate_distance(type, position, cp_position);
			result = std::min(result, distance);
		}
	}
	
	return result;
}

void InformationUnit::clean_outdated_unit_position()
{
	if (!type.isBuilding() && !is_current() && position.isValid()) {
		if (Broodwar->isVisible(TilePosition(position))) {
			position = Positions::Unknown;
			area = nullptr;
		}
	}
}

InformationManager::MarineCountRingBuffer::MarineCountRingBuffer()
{
	values_.fill(0);
}

void InformationManager::MarineCountRingBuffer::push(int value)
{
	values_[counter_++] = value;
	if (counter_ == values_.size()) counter_ = 0;
}

int InformationManager::MarineCountRingBuffer::max() const
{
	return *std::max_element(values_.begin(), values_.end());
}

void InformationManager::MarineCountRingBuffer::decrement()
{
	for (auto& element : values_) if (element > 0) --element;
}

void InformationManager::MarineCountRingBuffer::increment()
{
	for (auto& element : values_) if (element < 4) ++element;
}

void InformationManager::update_units_and_buildings()
{
	for (auto& unit : Broodwar->getNeutralUnits()) {
		if (unit->exists()) neutral_units_[unit].update(unit);
	}
	
	for (auto& unit : Broodwar->enemy()->getUnits()) {
		if (unit->exists()) enemy_units_[unit].update(unit);
	}
	clean_buildings_and_spells();
	clean_outdated_unit_positions();
}

void InformationManager::clean_buildings_and_spells()
{
	std::vector<Unit> neutral_buildings_to_remove;
	for (auto& entry : neutral_units_) {
		Unit unit = entry.first;
		InformationUnit& neutral_unit = entry.second;
		
		if (unit->exists() && unit->getPlayer() != Broodwar->neutral()) neutral_buildings_to_remove.push_back(entry.first);
		if (neutral_unit.type.isBuilding() && !unit->exists() && building_visible(neutral_unit.type, neutral_unit.tile_position())) neutral_buildings_to_remove.push_back(entry.first);
	}
	for (auto& unit : neutral_buildings_to_remove) {
		neutral_units_.erase(unit);
		bwem_handle_destroy_safe(unit);
	}
	
	std::vector<Unit> enemy_buildings_to_remove;
	for (auto& entry : enemy_units_) {
		Unit unit = entry.first;
		InformationUnit& enemy_unit = entry.second;
		
		if (unit->exists() && unit->getPlayer() != Broodwar->enemy()) enemy_buildings_to_remove.push_back(entry.first);
		if (enemy_unit.type.isBuilding() && !unit->exists() && building_visible(enemy_unit.type, enemy_unit.tile_position())) enemy_buildings_to_remove.push_back(entry.first);
		if (enemy_unit.type == UnitTypes::Spell_Scanner_Sweep &&
			Broodwar->getFrameCount() >= enemy_unit.first_frame + 240) enemy_buildings_to_remove.push_back(entry.first);
	}
	for (auto& unit : enemy_buildings_to_remove) {
		enemy_units_.erase(unit);
	}
}

void InformationManager::clean_outdated_unit_positions()
{
	for (auto& entry : neutral_units_) {
		InformationUnit& neutral_unit = entry.second;
		neutral_unit.clean_outdated_unit_position();
	}
	
	for (auto& entry : enemy_units_) {
		InformationUnit& enemy_unit = entry.second;
		enemy_unit.clean_outdated_unit_position();
	}
}

void InformationManager::update_information()
{
	update_enemy_count();
	update_upgrades();
	update_bullet_timestamps();
	update_bunker_marines_loaded();
	update_bunker_marine_range();
}

void InformationManager::update_upgrades()
{
	for (auto player : Broodwar->getPlayers()) {
		auto& upgrades = upgrades_[player];
		for (auto upgrade_type : UpgradeTypes::allUpgradeTypes()) {
			auto& level = upgrades[upgrade_type];
			level = std::max(level, player->getUpgradeLevel(upgrade_type));
		}
	}
}

void InformationManager::update_enemy_count()
{
	enemy_count_.clear();
	for (auto& entry : enemy_units_) {
		enemy_count_[entry.second.type]++;
		enemy_seen_.insert(entry.second.type);
	}
}

void InformationManager::draw()
{
	for (auto& entry : neutral_units_) {
		const InformationUnit& neutral_unit = entry.second;
		UnitType type = neutral_unit.type;
		if (!type.isMineralField() && type != UnitTypes::Resource_Vespene_Geyser && !neutral_unit.is_current()) {
			TilePosition tile_position = neutral_unit.tile_position();
			Broodwar->drawBoxMap(tile_position.x * 32,
								 tile_position.y * 32,
								 (tile_position.x + type.tileWidth()) * 32,
								 (tile_position.y + type.tileHeight()) * 32,
								 Colors::Blue);
			Broodwar->drawTextMap(Position(tile_position), "%s", type.c_str());
		}
	}
	
	for (auto& entry : enemy_units_) {
		const InformationUnit& enemy_unit = entry.second;
		UnitType type = enemy_unit.type;
		if (type.isBuilding()) {
			if (!enemy_unit.is_current()) {
				TilePosition tile_position = enemy_unit.tile_position();
				Broodwar->drawBoxMap(tile_position.x * 32,
									 tile_position.y * 32,
									 (tile_position.x + type.tileWidth()) * 32,
									 (tile_position.y + type.tileHeight()) * 32,
									 Colors::Orange);
				Broodwar->drawTextMap(Position(tile_position), "%s", type.c_str());
			}
		} else {
			Broodwar->drawBoxMap(enemy_unit.position - Position(enemy_unit.type.dimensionLeft(), enemy_unit.type.dimensionUp()),
								 enemy_unit.position + Position(enemy_unit.type.dimensionRight(), enemy_unit.type.dimensionDown()),
								 Colors::Red);
			Broodwar->drawTextMap(enemy_unit.position, "%s", enemy_unit.type.c_str());
		}
	}
	
	for (auto& entry : bunker_marines_loaded_) {
		Broodwar->drawTextMap(entry.first->getPosition() - Position(30, 0), "%d", entry.second.max());
	}
}

void InformationManager::onUnitDestroy(Unit unit)
{
	enemy_units_.erase(unit);
}

void InformationManager::onUnitDiscover(Unit unit)
{
	if (unit->getType() == UnitTypes::Terran_Marine) {
		Position position = unit->getPosition();
		for (auto& entry : bunker_marines_loaded_) {
			Unit bunker = entry.first;
			MarineCountRingBuffer& count = entry.second;
			Rect bunker_rect(UnitTypes::Terran_Bunker, bunker->getPosition());
			Rect below_rect(bunker_rect.left - 10 - UnitTypes::Terran_Marine.dimensionRight(),
							bunker_rect.bottom,
							bunker_rect.right + 10 + UnitTypes::Terran_Marine.dimensionLeft(),
							bunker_rect.bottom + 10 + UnitTypes::Terran_Marine.dimensionUp());
			if (below_rect.is_inside(position)) {
				count.decrement();
				bunker_marines_delay_update_until_timestamp_[bunker] = Broodwar->getFrameCount() + WeaponTypes::Gauss_Rifle.damageCooldown() + 1;
			}
		}
	}
}

void InformationManager::onUnitEvade(Unit unit)
{
	if (enemy_units_.count(unit) > 0) {
		InformationUnit& information_unit = enemy_units_.at(unit);
		if (information_unit.type == UnitTypes::Terran_Marine) {
			Position position = information_unit.position;
			for (auto& entry : bunker_marines_loaded_) {
				Unit bunker = entry.first;
				MarineCountRingBuffer& count = entry.second;
				int distance = calculate_distance(UnitTypes::Terran_Bunker, bunker->getPosition(), UnitTypes::Terran_Marine, position);
				if (distance == 0) count.increment();
			}
		}
	}
}

int InformationManager::upgrade_level(Player player,UpgradeType upgrade_type) const
{
	int level = 0;
	
	if (upgrades_.count(player) > 0) {
		auto& upgrades = upgrades_.at(player);
		if (upgrades.count(upgrade_type) > 0) {
			return upgrades.at(upgrade_type);
		}
	}
	
	return level;
}

void InformationManager::update_bullet_timestamps()
{
	std::set<int> existing_ids;
	for (auto& bullet : Broodwar->getBullets()) {
		if (bullet_timestamps_.count(bullet->getID()) == 0) {
			bullet_timestamps_[bullet->getID()] = Broodwar->getFrameCount();
		}
		existing_ids.insert(bullet->getID());
	}
	
	std::vector<int> remove_ids;
	for (auto& entry : bullet_timestamps_) if (existing_ids.count(entry.first) == 0) remove_ids.push_back(entry.first);
	for (auto& id : remove_ids) bullet_timestamps_.erase(id);
}

void InformationManager::update_bunker_marines_loaded()
{
	int bunker_marine_range = weapon_max_range(WeaponTypes::Gauss_Rifle, Broodwar->enemy()) + 64;
	std::vector<Unit> bunkers;
	std::map<Unit,int> bunker_marines_loaded;
	
	for (auto& enemy_unit : Broodwar->enemy()->getUnits()) {
		if (enemy_unit->isCompleted() && enemy_unit->getType() == UnitTypes::Terran_Bunker) {
			bool bunker_can_attack_us = false;
			for (auto& unit : Broodwar->self()->getUnits()) {
				if (not_incomplete(unit) && not_cloaked(unit)) {
					int distance = calculate_distance(UnitTypes::Terran_Marine, enemy_unit->getPosition(), unit->getType(), unit->getPosition());
					if (distance <= bunker_marine_range) bunker_can_attack_us = true;
				}
			}
			if (bunker_can_attack_us) {
				bunkers.push_back(enemy_unit);
				bunker_marines_loaded[enemy_unit] = 0;
			}
		}
	}
	
	if (!bunkers.empty()) {
		for (auto& bullet : Broodwar->getBullets()) {
			if (bullet->getType() == BulletTypes::Gauss_Rifle_Hit) {
				int timestamp = bullet_timestamps_.at(bullet->getID());
				if (bullet->getSource() == nullptr &&
					bullet->getTarget() != nullptr &&
					bullet->getTarget()->getPlayer() == Broodwar->self() &&
					Broodwar->getFrameCount() - timestamp < WeaponTypes::Gauss_Rifle.damageCooldown()) {
					std::map<Unit,int> distance_map;
					for (auto& bunker : bunkers) {
						int distance = calculate_distance(UnitTypes::Terran_Marine, bunker->getPosition(), bullet->getPosition());
						if (distance <= bunker_marine_range && bunker_marines_loaded[bunker] < 4) {
							distance_map[bunker] = distance;
						}
					}
					Unit bunker = key_with_smallest_value(distance_map);
					if (bunker != nullptr) bunker_marines_loaded[bunker]++;
				}
			}
		}
	}
	
	for (auto& entry : bunker_marines_loaded) {
		Unit unit = entry.first;
		if (bunker_marines_delay_update_until_timestamp_[unit] <= Broodwar->getFrameCount()) {
			int current_loaded = entry.second;
			bunker_marines_loaded_[unit].push(current_loaded);
		}
	}
	
	std::vector<Unit> remove_bunkers;
	for (auto& entry : bunker_marines_loaded_) if (!entry.first->exists()) remove_bunkers.push_back(entry.first);
	for (auto& bunker : remove_bunkers) bunker_marines_loaded_.erase(bunker);
}

void InformationManager::update_bunker_marine_range()
{
	if (opponent_model.enemy_race() == Races::Terran &&
		information_manager.upgrade_level(Broodwar->enemy(), UpgradeTypes::U_238_Shells) == 0) {
		for (auto& bullet : Broodwar->getBullets()) {
			if (bullet->getType() != BulletTypes::Gauss_Rifle_Hit) continue;
			int timestamp = bullet_timestamps_.at(bullet->getID());
			if (timestamp + 1 != Broodwar->getFrameCount()) continue;
			if (bullet->getSource() != nullptr || bullet->getTarget() == nullptr || !bullet->getTarget()->exists()) continue;
			Unit target = bullet->getTarget();
			std::map<Unit,int> distance_map;
			for (auto& unit : Broodwar->enemy()->getUnits()) {
				if (unit->isCompleted() && unit->getType() == UnitTypes::Terran_Bunker) {
					int distance = calculate_distance(UnitTypes::Terran_Marine, unit->getPosition(), bullet->getPosition());
					distance_map[unit] = distance;
				}
			}
			Unit bunker = key_with_smallest_value(distance_map);
			if (bunker != nullptr) {
				int distance = calculate_distance(UnitTypes::Terran_Marine, bunker->getPosition(), bullet->getPosition());
				if (distance >= 214 && distance < 246) {
					upgrades_[Broodwar->enemy()][UpgradeTypes::U_238_Shells] = 1;
					if (debug_options.draw_enabled()) Broodwar << "Detected U238 (bunker marine range)" << std::endl;
				}
			}
		}
	}
}

bool InformationManager::enemy_building_seen() const
{
	for (auto& entry : information_manager.enemy_units()) {
		if (entry.second.type.isBuilding() && !entry.second.flying) {
			return true;
		}
	}
	return false;
}

int InformationManager::bunker_marines_loaded(Unit bunker) const
{
	int result = 4;
	auto it = bunker_marines_loaded_.find(bunker);
	if (it != bunker_marines_loaded_.end()) result = it->second.max();
	return result;
}
