#pragma once

template <typename T>
class TileGrid
{
public:
	TileGrid() { clear(); }
	
	void clear()
	{
		data_.fill(T());
	}
	
	T& operator [](FastTilePosition tile_position)
	{
		return data_[tile_position.y * 256 + tile_position.x];
	}
	
	const T& operator [](FastTilePosition tile_position) const
	{
		return data_[tile_position.y * 256 + tile_position.x];
	}
	
	T get(FastTilePosition tile_position) const
	{
		if (tile_position.isValid()) {
			return data_[tile_position.y * 256 + tile_position.x];
		} else {
			return T();
		}
	}
	
	void set(FastTilePosition tile_position,T value)
	{
		if (tile_position.isValid()) data_[tile_position.y * 256 + tile_position.x] = value;
	}
private:
	std::array<T,256*256> data_;
};

template <int Size,int Granularity,typename T>
class SparsePositionGrid
{
public:
	SparsePositionGrid() { clear(); }
	
	static constexpr int Count = (Size / Granularity) * 2 + 1;
	
	void clear()
	{
		data_.fill(T());
	}
	
	T& operator [](FastPosition position)
	{
		int x = (position.x + Size) / Granularity;
		int y = (position.y + Size) / Granularity;
		return data_[y * Count + x];
	}
	
	const T& operator [](FastPosition position) const
	{
		int x = (position.x + Size) / Granularity;
		int y = (position.y + Size) / Granularity;
		return data_[y * Count + x];
	}
	
private:
	std::array<T,Count*Count> data_;
};

class WalkabilityGrid : public Singleton<WalkabilityGrid>
{
public:
	void init();
	void update();
	
	bool is_terrain_walkable(FastTilePosition tile_position) const { return terrain_walkable_.get(tile_position); }
	bool is_walkable(FastTilePosition tile_position) const { return walkable_.get(tile_position); }
	FastPosition walkable_tile_near(FastPosition position,int range) const;
private:
	TileGrid<bool> terrain_walkable_;
	TileGrid<bool> walkable_;
	
	static bool tile_fully_walkable(FastTilePosition tile_position);
};

class ConnectivityGrid : public Singleton<ConnectivityGrid>
{
public:
	void update();
	void invalidate() { valid_ = false; }
	
	int component_for_position(FastPosition position);
	bool check_reachability(Unit combat_unit,Unit enemy_unit);
	bool check_reachability_melee(int component,Unit enemy_unit);
	bool check_reachability_ranged(int component,int range,Unit enemy_unit);
	bool building_has_component(UnitType unit_type,FastTilePosition tile_position,int component);
	bool is_wall_building(Unit unit);
	int wall_building_perimeter(Unit unit,int component);
private:
	bool valid_ = false;
	TileGrid<int> component_;
};

class ThreatGrid : public Singleton<ThreatGrid>
{
public:
	void update();
	bool ground_threat(FastTilePosition tile_position) const { return ground_threat_.get(tile_position); }
	bool air_threat(FastTilePosition tile_position) const { return air_threat_.get(tile_position); }
	bool threat(FastTilePosition tile_position, bool air) const {
		return air ? air_threat_.get(tile_position) : ground_threat_.get(tile_position);
	}
	int component(FastTilePosition tile_position,bool air) const {
		return air ? air_component_.get(tile_position) : ground_component_.get(tile_position);
	}
	bool line_of_sight(FastTilePosition a,FastTilePosition b,bool air) const;
private:
	TileGrid<bool> ground_threat_;
	TileGrid<bool> air_threat_;
	TileGrid<int> ground_component_;
	TileGrid<int> air_component_;
	
	void apply(TileGrid<bool>& grid,FastPosition position,int range);
	void calculate_component(const TileGrid<bool>& threat_grid,TileGrid<int>& component_grid,bool air);
};

class UnitGrid : public Singleton<UnitGrid>
{
public:
	UnitGrid();
	void update();
	const std::vector<Unit>& units_in_tile(FastTilePosition tile_position) const;
	
private:
	std::array<std::vector<Unit>,256*256> grid_;
};
