#include "BananaBrain.h"

void WalkabilityGrid::init()
{
	for (int y = 0; y < Broodwar->mapHeight(); y++) {
		for (int x = 0; x < Broodwar->mapWidth(); x++) {
			FastTilePosition tile_position(x, y);
			if (tile_fully_walkable(tile_position)) terrain_walkable_[tile_position] = true;
		}
	}
}

bool WalkabilityGrid::tile_fully_walkable(FastTilePosition tile_position)
{
	FastWalkPosition walk_position(tile_position);
	for (int dy = 0; dy < 4; dy++) {
		for (int dx = 0; dx < 4; dx++) {
			if (!Broodwar->isWalkable(walk_position + FastWalkPosition(dx, dy))) return false;
		}
	}
	return true;
}

void WalkabilityGrid::update()
{
	walkable_ = terrain_walkable_;
	
	for (auto& unit : Broodwar->self()->getUnits()) {
		if (unit->getType().isBuilding() && !unit->isFlying()) {
			FastTilePosition tile_position(unit->getTilePosition());
			for (int dy = 0; dy < unit->getType().tileHeight(); dy++) {
				for (int dx = 0; dx < unit->getType().tileWidth(); dx++) {
					walkable_[tile_position + FastTilePosition(dx, dy)] = false;
				}
			}
		}
	}
	
	for (auto& entry : information_manager.neutral_units()) {
		const InformationUnit& neutral_unit = entry.second;
		if (neutral_unit.type.isBuilding() && !neutral_unit.flying) {
			FastTilePosition tile_position(neutral_unit.tile_position());
			for (int dy = 0; dy < neutral_unit.type.tileHeight(); dy++) {
				for (int dx = 0; dx < neutral_unit.type.tileWidth(); dx++) {
					walkable_[tile_position + FastTilePosition(dx, dy)] = false;
				}
			}
		}
	}
	
	for (auto& entry : information_manager.enemy_units()) {
		const InformationUnit& enemy_unit = entry.second;
		if (enemy_unit.type.isBuilding() && !enemy_unit.flying) {
			FastTilePosition tile_position(enemy_unit.tile_position());
			for (int dy = 0; dy < enemy_unit.type.tileHeight(); dy++) {
				for (int dx = 0; dx < enemy_unit.type.tileWidth(); dx++) {
					walkable_[tile_position + FastTilePosition(dx, dy)] = false;
				}
			}
		}
	}
}

FastPosition WalkabilityGrid::walkable_tile_near(FastPosition position,int range) const
{
	FastTilePosition tile_position(position);
	if (is_walkable(tile_position)) return position;
	FastPosition result;
	int result_distance = range + 1;
	int range_tiles = (range + 31) / 32;
	for (int y = tile_position.y - range_tiles; y <= tile_position.y + range_tiles; y++) {
		for (int x = tile_position.x - range_tiles; x <= tile_position.x + range_tiles; x++) {
			FastTilePosition current_tile(x, y);
			if (is_walkable(current_tile)) {
				FastPosition current = center_position(current_tile);
				int distance = current.getApproxDistance(position);
				if (current.getApproxDistance(position) < result_distance) {
					result = current;
					result_distance = distance;
				}
			}
		}
	}
	return result;
}

void ConnectivityGrid::update()
{
	if (valid_) return;
	
	component_.clear();
	int component_counter = 1;
	for (int y = 0; y < Broodwar->mapHeight(); y++) {
		for (int x = 0; x < Broodwar->mapWidth(); x++) {
			FastTilePosition tile_position(x, y);
			if (!walkability_grid.is_walkable(tile_position) || component_[tile_position] > 0) continue;
			
			int component = component_counter++;
			std::queue<FastTilePosition> queue;
			queue.push(tile_position);
			component_[tile_position] = component;
			
			while (!queue.empty()) {
				FastTilePosition tile_position = queue.front();
				queue.pop();
				
				for (auto& delta : { FastTilePosition(-1, 0), FastTilePosition(1, 0), FastTilePosition(0, -1), FastTilePosition(0, 1) }) {
					FastTilePosition new_tile_position = tile_position + delta;
					if (walkability_grid.is_walkable(new_tile_position) && component_[new_tile_position] == 0) {
						queue.push(new_tile_position);
						component_[new_tile_position] = component;
					}
				}
			}
		}
	}
	
	valid_ = true;
}

int ConnectivityGrid::component_for_position(FastPosition position)
{
	FastTilePosition tile_position(position);
	if (component_.get(tile_position) > 0) return component_[tile_position];
	
	int component = 0;
	int smallest_distance = INT_MAX;
	for (auto& delta : { FastTilePosition(-1, -1), FastTilePosition(0, -1), FastTilePosition(1, -1), FastTilePosition(-1, 0), FastTilePosition(1, 0), FastTilePosition(-1, 1), FastTilePosition(0, 1), FastTilePosition(1, 1) }) {
		FastTilePosition other_tile_position = tile_position + delta;
		int other_component = component_.get(other_tile_position);
		if (other_component != 0) {
			int distance = position.getApproxDistance(center_position(other_tile_position));
			if (distance < smallest_distance) {
				component = other_component;
				smallest_distance = distance;
			}
		}
	}
	
	return component;
}

bool ConnectivityGrid::check_reachability(Unit combat_unit,Unit enemy_unit)
{
	bool result = true;
	
	if (is_melee(combat_unit->getType())) {
		int component = component_for_position(combat_unit->getPosition());
		result = check_reachability_melee(component, enemy_unit);
	} else if (!combat_unit->isFlying()) {
		int component = component_for_position(combat_unit->getPosition());
		int range = weapon_max_range(combat_unit, enemy_unit->isFlying());
		result = check_reachability_ranged(component, range, enemy_unit);
	}
	
	return result;
}

bool ConnectivityGrid::check_reachability_melee(int component,Unit enemy_unit)
{
	if (component == 0) return true;
	
	UnitType unit_type = enemy_unit->getType();
	bool result;
	if (unit_type.isBuilding()) {
		result = building_has_component(unit_type, enemy_unit->getTilePosition(), component);
	} else {
		result = (component_for_position(enemy_unit->getPosition()) == component);
	}
	return result;
}

bool ConnectivityGrid::check_reachability_ranged(int component,int range,Unit enemy_unit)
{
	if (component == 0) return true;
	
	UnitType unit_type = enemy_unit->getType();
	if (!unit_type.isBuilding() && component_for_position(enemy_unit->getPosition()) == component) return true;
	
	int left = enemy_unit->getPosition().x - unit_type.dimensionLeft() - range;
	int top = enemy_unit->getPosition().y - unit_type.dimensionUp() - range;
	int right = enemy_unit->getPosition().x + unit_type.dimensionRight() + range;
	int bottom = enemy_unit->getPosition().y + unit_type.dimensionDown() + range;
	
	FastTilePosition top_left(FastPosition(left, top));
	FastTilePosition bottom_right(FastPosition(right, bottom));
	
	for (int y = top_left.y; y <= bottom_right.y; y++) {
		for (int x = top_left.x; x <= bottom_right.x; x++) {
			FastTilePosition tile_position(x, y);
			if (component_.get(tile_position) == component) {
				FastPosition position = center_position(tile_position);
				int distance = enemy_unit->getDistance(position);
				if (distance <= range) return true;
			}
		}
	}
	
	return false;
}

bool ConnectivityGrid::building_has_component(UnitType unit_type,FastTilePosition tile_position,int component)
{
	int left = tile_position.x - 1;
	int right = tile_position.x + unit_type.tileWidth();
	int top = tile_position.y - 1;
	int bottom = tile_position.y + unit_type.tileHeight();
	
	for (int x = left; x <= right; x++) {
		if (component_.get(FastTilePosition(x, top)) == component) return true;
		if (component_.get(FastTilePosition(x, bottom)) == component) return true;
	}
	for (int y = top; y <= bottom; y++) {
		if (component_.get(FastTilePosition(left, y)) == component) return true;
		if (component_.get(FastTilePosition(right, y)) == component) return true;
	}
	return false;
}

bool ConnectivityGrid::is_wall_building(Unit unit)
{
	UnitType unit_type = unit->getType();
	if (!unit_type.isBuilding()) return false;
	
	int left = unit->getTilePosition().x - 1;
	int right = unit->getTilePosition().x + unit_type.tileWidth();
	int top = unit->getTilePosition().y - 1;
	int bottom = unit->getTilePosition().y + unit_type.tileHeight();
	
	int component_seen = 0;
	
	for (int x = left; x <= right; x++) {
		int component;
		
		component = component_.get(FastTilePosition(x, top));
		if (component != 0) {
			if (component_seen == 0) component_seen = component; else if (component != component_seen) return true;
		}
		
		component = component_.get(FastTilePosition(x, bottom));
		if (component != 0) {
			if (component_seen == 0) component_seen = component; else if (component != component_seen) return true;
		}
	}
	
	for (int y = top; y <= bottom; y++) {
		int component;
		
		component = component_.get(FastTilePosition(left, y));
		if (component != 0) {
			if (component_seen == 0) component_seen = component; else if (component != component_seen) return true;
		}
		
		component = component_.get(FastTilePosition(right, y));
		if (component != 0) {
			if (component_seen == 0) component_seen = component; else if (component != component_seen) return true;
		}
	}
	
	return false;
}

int ConnectivityGrid::wall_building_perimeter(Unit unit,int component)
{
	UnitType unit_type = unit->getType();
	if (!unit_type.isBuilding()) return false;
	
	int left = unit->getTilePosition().x - 1;
	int right = unit->getTilePosition().x + unit_type.tileWidth();
	int top = unit->getTilePosition().y - 1;
	int bottom = unit->getTilePosition().y + unit_type.tileHeight();
	
	int count = 0;
	
	for (int x = left; x <= right; x++) {
		if (component_.get(FastTilePosition(x, top)) == component) count++;
		if (component_.get(FastTilePosition(x, bottom)) == component) count++;
	}
	
	for (int y = top + 1; y <= bottom - 1; y++) {
		if (component_.get(FastTilePosition(left, y)) == component) count++;
		if (component_.get(FastTilePosition(right, y)) == component) count++;
	}
	
	return count;
}

void ThreatGrid::update()
{
	ground_threat_.clear();
	air_threat_.clear();
	
	for (auto& entry : information_manager.enemy_units()) {
		apply(ground_threat_, entry.second.position, weapon_max_range(entry.second.type, Broodwar->enemy(), false));
		apply(air_threat_, entry.second.position, weapon_max_range(entry.second.type, Broodwar->enemy(), true));
	}
	
	calculate_component(ground_threat_, ground_component_, false);
	calculate_component(air_threat_, air_component_, true);
}

void ThreatGrid::apply(TileGrid<bool>& grid,FastPosition position,int range)
{
	FastPosition range_position(range, range);
	FastTilePosition top_left(position - range_position);
	FastTilePosition bottom_right(position + range_position);
	
	for (int y = top_left.y; y <= bottom_right.y; y++) {
		for (int x = top_left.x; x <= bottom_right.x; x++) {
			FastTilePosition tile_position(x, y);
			if (tile_position.isValid()) {
				FastPosition current_position = center_position(tile_position);
				if (current_position.getApproxDistance(position) <= range) {
					grid[tile_position] = true;
				}
			}
		}
	}
}

void ThreatGrid::calculate_component(const TileGrid<bool>& threat_grid,TileGrid<int>& component_grid,bool air)
{
	component_grid.clear();
	int component_counter = 1;
	for (int y = 0; y < Broodwar->mapHeight(); y++) {
		for (int x = 0; x < Broodwar->mapWidth(); x++) {
			FastTilePosition tile_position(x, y);
			if ((!air && !walkability_grid.is_walkable(tile_position)) || component_grid[tile_position] > 0) continue;
			
			int component = component_counter++;
			std::queue<FastTilePosition> queue;
			queue.push(tile_position);
			component_grid[tile_position] = component;
			
			while (!queue.empty()) {
				FastTilePosition tile_position = queue.front();
				queue.pop();
				
				for (auto& delta : { FastTilePosition(-1, 0), FastTilePosition(1, 0), FastTilePosition(0, -1), FastTilePosition(0, 1) }) {
					FastTilePosition new_tile_position = tile_position + delta;
					bool new_valid = air ? new_tile_position.isValid() : walkability_grid.is_walkable(new_tile_position);
					if (new_valid && component_grid[new_tile_position] == 0) {
						queue.push(new_tile_position);
						component_grid[new_tile_position] = component;
					}
				}
			}
		}
	}
}

bool ThreatGrid::line_of_sight(FastTilePosition a,FastTilePosition b,bool air) const
{
	if (a.x == b.x) {
		int x = a.x;
		int y1 = std::min(a.y, b.y);
		int y2 = std::max(a.y, b.y);
		for (int y = y1; y <= y2; y++) if (threat(FastTilePosition(x, y), air)) return false;
		return true;
	}
	
	if (a.y == b.y) {
		int y = a.y;
		int x1 = std::min(a.x, b.x);
		int x2 = std::max(a.x, b.x);
		for (int x = x1; x <= x2; x++) if (threat(FastTilePosition(x, y), air)) return false;
		return true;
	}
	
	if (abs(a.x - b.x) >= abs(a.y - b.y)) {
		if (a.x > b.x) std::swap(a, b);
		int x1 = a.x;
		int x2 = b.x;
		int y1 = a.y;
		int y2 = b.y;
		for (int x = x1; x <= x2; x++) {
			double t = double(x - x1) / double(x2 - x1);
			int y = int(t * (y2 - y1) + y1 + 0.5);
			if (threat(FastTilePosition(x, y), air)) return false;
		}
		return true;
	} else {
		if (a.y > b.y) std::swap(a, b);
		int x1 = a.x;
		int x2 = b.x;
		int y1 = a.y;
		int y2 = b.y;
		for (int y = y1; y <= y2; y++) {
			double t = double(y - y1) / double(y2 - y1);
			int x = int(t * (x2 - x1) + x1 + 0.5);
			if (threat(FastTilePosition(x, y), air)) return false;
		}
		return true;
	}
}

UnitGrid::UnitGrid()
{
	for (int y = 0; y < 256; y++) {
		for (int x = 0; x < 256; x++) {
			grid_[y * 256 + x].reserve(4);
		}
	}
}

void UnitGrid::update()
{
	for (int y = 0; y < Broodwar->mapHeight(); y++) {
		for (int x = 0; x < Broodwar->mapWidth(); x++) {
			grid_[y * 256 + x].clear();
		}
	}
	for (auto& unit : Broodwar->getAllUnits()) {
		int left = unit->getLeft();
		int top = unit->getTop();
		int right = unit->getRight();
		int bottom = unit->getBottom();
		FastTilePosition top_left(FastPosition(left, top));
		FastTilePosition bottom_right(FastPosition(right, bottom));
		for (int y = top_left.y; y <= bottom_right.y; y++) {
			for (int x = top_left.x; x <= bottom_right.x; x++) {
				grid_[y * 256 + x].push_back(unit);
			}
		}
	}
}

const std::vector<Unit>& UnitGrid::units_in_tile(FastTilePosition tile_position) const
{
	return grid_[tile_position.y * 256 + tile_position.x];
}
