#include "BananaBrain.h"
#include <iostream>
#include <sstream>
#include <climits>
#include <cstdio>
#define WIN32_LEAN_AND_MEAN
#include <windows.h>

void BananaBrain::onStart()
{
	// Enable the UserInput flag, which allows us to control the bot and type messages.
	//Broodwar->enableFlag(Flag::UserInput);

	if (!Broodwar->isReplay())
	{
		bool ok = true;
		
		if (Broodwar->self()->getRace() != Races::Protoss) {
			Broodwar->sendText("Error: This bot only plays Protoss");
			ok = false;
		}
		if (Broodwar->enemies().size() != 1 || Broodwar->allies().size() != 0) {
			Broodwar->sendText("Error: This bot only plays 1v1");
			ok = false;
		}
		
		if (ok) {
			debug_options.init();
			srand(static_cast<unsigned int>(time(nullptr)));
			bwem_map.Initialize(BroodwarPtr);
			bwem_map.FindBasesForStartingLocations();
			bwem_map.EnableAutomaticPathAnalysis();
			base_state.init_bases();
			path_finder.init();
			building_placement_manager.init();
			opponent_model.init();
			spending_manager.init_resource_counters();
			result_store.init();
			strategy.pick_strategy();
			walkability_grid.init();
			initialized_ = true;
		}
	}
}

void BananaBrain::onEnd(bool winner)
{
	if (!initialized_) return;
	strategy.apply_result(winner);
	result_store.store();
}

void BananaBrain::onFrame()
{
	if (!initialized_) return;
	if (Broodwar->isPaused() || !Broodwar->self()) return;
	
	LARGE_INTEGER start;
	QueryPerformanceCounter(&start);
	
	before();
	strategy.frame();
	after();
	surrender_if_hope_lost();
	
	if (debug_options.draw_enabled()) {
		draw();
		LARGE_INTEGER end,frequency;
		QueryPerformanceCounter(&end);
		QueryPerformanceFrequency(&frequency);
		double frequency_rec = 1000.0 / frequency.QuadPart;
		int duration = (int)((end.QuadPart - start.QuadPart) * frequency_rec);
		if (Broodwar->getFrameCount() == 0) {
			frame_zero_duration_ = duration;
		} else {
			max_duration_ = std::max(duration, max_duration_);
		}
		Broodwar->drawTextScreen(4, 16, "Frame duration: %d ms, max %d ms, frame zero %d ms", duration, max_duration_, frame_zero_duration_);
	}
}

void BananaBrain::onSendText(std::string text)
{
}

void BananaBrain::onReceiveText(BWAPI::Player player, std::string text)
{
}

void BananaBrain::onPlayerLeft(BWAPI::Player player)
{
}

void BananaBrain::onNukeDetect(BWAPI::Position target)
{
}

void BananaBrain::onUnitDiscover(BWAPI::Unit unit)
{
	if (unit->getType().isBuilding()) connectivity_grid.invalidate();
	information_manager.onUnitDiscover(unit);
}

void BananaBrain::onUnitEvade(BWAPI::Unit unit)
{
	information_manager.onUnitEvade(unit);
}

void BananaBrain::onUnitShow(BWAPI::Unit unit)
{
}

void BananaBrain::onUnitHide(BWAPI::Unit unit)
{
}

void BananaBrain::onUnitCreate(BWAPI::Unit unit)
{
	if (unit->getType().isBuilding()) connectivity_grid.invalidate();
}

void BananaBrain::onUnitDestroy(BWAPI::Unit unit)
{
	information_manager.onUnitDestroy(unit);
	training_manager.onUnitLost(unit);
	worker_manager.onUnitLost(unit);
	bwem_handle_destroy_safe(unit);
	if (unit->getType().isBuilding()) connectivity_grid.invalidate();
	building_placement_manager.onUnitDestroy(unit);
}

void BananaBrain::onUnitMorph(BWAPI::Unit unit)
{
}

void BananaBrain::onUnitRenegade(BWAPI::Unit unit)
{
	worker_manager.onUnitLost(unit);
}

void BananaBrain::onSaveGame(std::string gameName)
{
}

void BananaBrain::onUnitComplete(BWAPI::Unit unit)
{
}

void BananaBrain::draw() {
	draw_info();
	base_state.draw();
	information_manager.draw();
	tactics_manager.draw();
	micro_manager.draw();
	building_placement_manager.draw();
	worker_manager.draw_for_workers();
}

void BananaBrain::before()
{
	information_manager.update_units_and_buildings();
	walkability_grid.update();
	connectivity_grid.update();
	base_state.update_base_information();
	path_finder.close_small_chokepoints_if_needed();
	unit_grid.update();
	training_manager.init_unit_count_map();
	building_manager.init_building_count_map();
	building_manager.init_base_defense_map();
	building_manager.init_upgrade_and_research();
	building_manager.update_supply_requests();
	information_manager.update_information();
	tactics_manager.update();
	opponent_model.update();
	threat_grid.update();
	micro_manager.prepare_combat();
}

void BananaBrain::after()
{
	spending_manager.init_spendable();
	worker_manager.before();
	
	if (training_manager.probe_production() && !training_manager.probe_cut()) training_manager.apply_worker_train_orders();
	building_manager.apply_building_requests(true);
	if (training_manager.prioritize_training()) training_manager.apply_train_orders();
	building_manager.apply_building_requests(false);
	building_manager.apply_upgrades();
	building_manager.apply_research();
	if (!training_manager.prioritize_training()) training_manager.apply_train_orders();
	if (training_manager.probe_production() && training_manager.probe_cut()) training_manager.apply_worker_train_orders();
	
	worker_manager.apply_worker_orders();
	micro_manager.apply_combat_orders();
	building_manager.cancel_doomed_buildings();
}

void BananaBrain::surrender_if_hope_lost()
{
	if ((Broodwar->getFrameCount() % (2 * 24)) != 0) return;
	
	for (auto& unit : Broodwar->self()->getUnits()) {
		if (unit->getType().isResourceDepot() && MineralGas(Broodwar->self()).can_pay(Broodwar->self()->getRace().getWorker())) return;
		if (unit->getType().isBuilding() && !unit->getTrainingQueue().empty()) return;
		if (unit->getType().groundWeapon() != WeaponTypes::None ||
			unit->getType().airWeapon() != WeaponTypes::None ||
			unit->getType() == UnitTypes::Protoss_Carrier ||
			unit->getType() == UnitTypes::Protoss_Reaver) return;
	}
	
	Unitset enemy_units = Broodwar->enemies().getUnits();
	bool visible_attacker_found = std::any_of(enemy_units.begin(), enemy_units.end(), [](Unit unit){
		return (unit->getType().groundWeapon() != WeaponTypes::None ||
				unit->getType() == UnitTypes::Protoss_Carrier ||
				unit->getType() == UnitTypes::Protoss_Reaver);
	});
	if (!visible_attacker_found) return;
	
	Broodwar->leaveGame();
}

void BananaBrain::draw_info()
{
	Broodwar->drawTextScreen(4, 26, "Time %d:%02d (frame %d)", Broodwar->getFrameCount() / (24 * 60), (Broodwar->getFrameCount() / 24) % 60, Broodwar->getFrameCount());
	Broodwar->drawTextScreen(4, 36, "Playing: %s vs %s on %s", Broodwar->self()->getRace().getName().c_str(), opponent_model.enemy_race().getName().c_str(), Broodwar->mapFileName().c_str());
	Broodwar->drawTextScreen(4, 46, "Income: %d/%d ratio=%.1f",
							 spending_manager.income_per_minute().minerals,
							 spending_manager.income_per_minute().gas,
							 (double)spending_manager.income_per_minute().minerals / (double)spending_manager.income_per_minute().gas);
	Broodwar->drawTextScreen(4, 56, "Training: %.1f/%.1f/%.1f ratio=%.1f",
							 spending_manager.training_cost_per_minute().minerals,
							 spending_manager.training_cost_per_minute().gas,
							 spending_manager.training_cost_per_minute().supply,
							 spending_manager.training_cost_per_minute().minerals / spending_manager.training_cost_per_minute().gas);
	Broodwar->drawTextScreen(4, 66, "Worker training: %.1f/%.1f/%.1f", spending_manager.worker_training_cost_per_minute().minerals, spending_manager.worker_training_cost_per_minute().gas, spending_manager.worker_training_cost_per_minute().supply);
	Broodwar->drawTextScreen(4, 76, "Remainder: %d/%d", spending_manager.remainder().minerals, spending_manager.remainder().gas);
	Broodwar->drawTextScreen(4, 86, "Spendable: %d/%d", spending_manager.spendable().minerals, spending_manager.spendable().gas);
	Broodwar->drawTextScreen(4, 96, "Worker/Army supply: %d/%d opponent: %d/%d",
							 worker_supply(), army_supply(),
							 tactics_manager.enemy_worker_supply(), tactics_manager.enemy_army_supply());
	Broodwar->drawTextScreen(4, 106, "Enemy defense and offense supply: %d/%d",
							 tactics_manager.enemy_defense_supply(),
							 tactics_manager.enemy_offense_supply());
	Broodwar->drawTextScreen(4, 116, "Average #workers/mineral: %.1f", worker_manager.average_workers_per_mineral());
	Broodwar->drawTextScreen(4, 126, "Gateway distribution: %.2f, %.2f, %.2f, %.2f",
							 training_manager.gateway_train_distribution().get(UnitTypes::Protoss_Zealot),
							 training_manager.gateway_train_distribution().get(UnitTypes::Protoss_Dragoon),
							 training_manager.gateway_train_distribution().get(UnitTypes::Protoss_High_Templar),
							 training_manager.gateway_train_distribution().get(UnitTypes::Protoss_Dark_Templar));
	Broodwar->drawTextScreen(4, 136, "Mode: %s", strategy.mode().c_str());
	Broodwar->drawTextScreen(4, 146, "Opening: %s", strategy.opening().c_str());
	Broodwar->drawTextScreen(4, 156, "Late game carriers: %s", strategy.late_game_carriers() ? "Yes" : "No");
	Broodwar->drawTextScreen(4, 166, "Enemy opening: %s", opponent_model.enemy_opening_info().c_str());
	Broodwar->drawTextScreen(4, 176, "Lost units/workers: %d/%d",
							 training_manager.lost_unit_count(),
							 worker_manager.lost_worker_count());
}
