#include "StrategyManager.h"
#include "CombatCommander.h"
#include "MapTools.h"
#include "OpponentModel.h"
#include "ProductionManager.h"
#include "StrategyBossZerg.h"
#include "UnitUtil.h"
#include "Bases.h"


using namespace UAlbertaBot;

StrategyManager::StrategyManager() 
	: the(The::Root())
	, _selfRace(BWAPI::Broodwar->self()->getRace())
	, _enemyRace(BWAPI::Broodwar->enemy()->getRace())
    , _emptyBuildOrder(BWAPI::Broodwar->self()->getRace())
	, _openingGroup("")
	, _hasDropTech(false)
	, _highWaterBases(1)
	, _openingStaticDefenseDropped(false)
	, _lastModule("")
	, _buildSecondBaseIn(false)
{
}

StrategyManager & StrategyManager::Instance() 
{
	static StrategyManager instance;
	return instance;
}

const BuildOrder & StrategyManager::getOpeningBookBuildOrder() const
{
    auto buildOrderIt = _strategies.find(Config::Strategy::StrategyName);

    // look for the build order in the build order map
	if (buildOrderIt != std::end(_strategies))
    {
        return (*buildOrderIt).second._buildOrder;
    }
    else
    {
        UAB_ASSERT_WARNING(false, "Strategy not found: %s, returning empty initial build order", Config::Strategy::StrategyName.c_str());
        return _emptyBuildOrder;
    }
}

// This is used for terran and protoss.
const bool StrategyManager::shouldExpandNow() const
{
	int numCC = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Command_Center);
	int numVultures = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Vulture);
	int numVessels = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Science_Vessel);
	int numGoliaths = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Goliath);
	int numTanks = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode)
		+ UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Siege_Mode);
	int numAttackUnits = numVultures + numTanks + numGoliaths;

	// if there is no place to expand to, we can't expand
	// We check mineral expansions only.
	if (MapTools::Instance().getNextExpansion(false, true, false) == BWAPI::TilePositions::None)
	{
		return false;
	}

	// if we have idle workers then we need a new expansion
	if (WorkerManager::Instance().getNumIdleWorkers() > 3 && numCC > 1)
	{
		return true;
	}

    // if we have excess minerals, expand
	if (BWAPI::Broodwar->self()->minerals() > 600)
    {
		if (numCC == 1 && numAttackUnits > 7)
		{
			return true;
		}
		else if (numCC > 1)
		{
			return true;
		}
        return false;
    }

	size_t numDepots = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Command_Center)
		+ UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Protoss_Nexus)
		+ UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Hatchery)
		+ UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Lair)
		+ UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Hive);
	int frame = BWAPI::Broodwar->getFrameCount();
	int minute = frame / (24 * 60);

	// we will make expansion N after array[N] minutes have passed
	std::vector<int> expansionTimes = { 10, 14, 18, 22, 26, 30 };//{5, 9, 13, 17, 21, 25};

    for (size_t i(0); i < expansionTimes.size(); ++i)
    {
        if (numDepots < (i+2) && minute > expansionTimes[i])
        {
            return true;
        }
    }

	return false;
}

void StrategyManager::addStrategy(const std::string & name, Strategy & strategy)
{
    _strategies[name] = strategy;
}

// Set _openingGroup depending on the current strategy, which in principle
// might be from the config file or from opening learning.
// This is part of initialization; it happens early on.
void StrategyManager::setOpeningGroup()
{
	auto buildOrderItr = _strategies.find(Config::Strategy::StrategyName);

	if (buildOrderItr != std::end(_strategies))
	{
		_openingGroup = (*buildOrderItr).second._openingGroup;
	}
}

const std::string & StrategyManager::getOpeningGroup() const
{
	return _openingGroup;
}

const MetaPairVector StrategyManager::getBuildOrderGoal()
{
    if (_selfRace == BWAPI::Races::Protoss)
    {
        return getProtossBuildOrderGoal();
    }
	else if (_selfRace == BWAPI::Races::Terran && openingFinished())
	{
		//return getTerranBuildOrderGoal();
		if (_currentModule == "C1")
		{
			return updateModuleC1();
		}
		else if (_currentModule == "C2")
		{
		}

	}
	else if (_selfRace == BWAPI::Races::Zerg)
	{
		return getZergBuildOrderGoal();
	}

    return MetaPairVector();
}

const MetaPairVector StrategyManager::getProtossBuildOrderGoal()
{
	// the goal to return
	MetaPairVector goal;

	// These counts include uncompleted units.
	int numPylons = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Protoss_Pylon);
	int numNexusCompleted = BWAPI::Broodwar->self()->completedUnitCount(BWAPI::UnitTypes::Protoss_Nexus);
	int numNexusAll = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Protoss_Nexus);
	int numProbes = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Protoss_Probe);
	int numCannon = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Protoss_Photon_Cannon);
	int numObservers = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Protoss_Observer);
	int numZealots = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Protoss_Zealot);
	int numDragoons = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Protoss_Dragoon);
	int numDarkTemplar = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Protoss_Dark_Templar);
	int numReavers = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Protoss_Reaver);
	int numCorsairs = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Protoss_Corsair);
	int numCarriers = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Protoss_Carrier);

	bool hasStargate = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Protoss_Stargate) > 0;

	int maxProbes = WorkerManager::Instance().getMaxWorkers();

	BWAPI::Player self = BWAPI::Broodwar->self();

	if (_openingGroup == "zealots")
	{
		goal.push_back(MetaPair(BWAPI::UnitTypes::Protoss_Zealot, numZealots + 6));

		if (numNexusAll >= 3)
		{
			// In the end, switch to carriers; not so many dragoons.
			goal.push_back(MetaPair(BWAPI::UpgradeTypes::Carrier_Capacity, 1));
			goal.push_back(MetaPair(BWAPI::UnitTypes::Protoss_Carrier, numCarriers + 1));
			goal.push_back(MetaPair(BWAPI::UnitTypes::Protoss_Dragoon, numDragoons + 1));
		}
		else if (numNexusAll >= 2)
		{
			// Once we have a 2nd nexus, add dragoons.
			goal.push_back(MetaPair(BWAPI::UpgradeTypes::Singularity_Charge, 1));
			goal.push_back(MetaPair(BWAPI::UnitTypes::Protoss_Dragoon, numDragoons + 4));
		}

		// Once dragoons are out, get zealot speed.
		if (numDragoons > 0)
		{
			goal.push_back(MetaPair(BWAPI::UpgradeTypes::Leg_Enhancements, 1));
		}

		// Finally add templar archives.
		if (UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Protoss_Citadel_of_Adun) > 0)
		{
			goal.push_back(MetaPair(BWAPI::UnitTypes::Protoss_Templar_Archives, 1));
		}

		// If we have templar archives, make
		// 1. a small fixed number of dark templar to force a reaction, and
		// 2. an even number of high templar to merge into archons (so the high templar disappear quickly).
		if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Protoss_Templar_Archives) != 0)
		{
			goal.push_back(MetaPair(BWAPI::UnitTypes::Protoss_Dark_Templar, std::max(3, numDarkTemplar)));
			goal.push_back(MetaPair(BWAPI::UnitTypes::Protoss_High_Templar, 2));
		}
	}
	else if (_openingGroup == "dragoons")
	{
		goal.push_back(MetaPair(BWAPI::UpgradeTypes::Singularity_Charge, 1));
		goal.push_back(MetaPair(BWAPI::UnitTypes::Protoss_Dragoon, numDragoons + 6));

		// Once we have a 2nd nexus, add reavers.
		if (numNexusAll >= 2)
		{
			goal.push_back(MetaPair(BWAPI::UnitTypes::Protoss_Reaver, numReavers + 1));
		}

		// If we have templar archives, make a small fixed number of DTs to force a reaction.
		if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Protoss_Templar_Archives) != 0)
		{
			goal.push_back(MetaPair(BWAPI::UnitTypes::Protoss_Dark_Templar, std::max(3, numDarkTemplar)));
		}

	}
	else if (_openingGroup == "dark templar")
	{
		goal.push_back(MetaPair(BWAPI::UnitTypes::Protoss_Dark_Templar, numDarkTemplar + 2));

		// Once we have a 2nd nexus, add dragoons.
		if (numNexusAll >= 2)
		{
			goal.push_back(MetaPair(BWAPI::UpgradeTypes::Singularity_Charge, 1));
			goal.push_back(MetaPair(BWAPI::UnitTypes::Protoss_Dragoon, numDragoons + 4));
		}
	}
	else if (_openingGroup == "drop")
	{
		goal.push_back(MetaPair(BWAPI::UnitTypes::Protoss_Dark_Templar, numDarkTemplar + 2));

		// The drop prep is carried out entirely by the opening book.
		// Immediately transition into something else.
		_openingGroup = "dragoons";
	}
	else
	{
		UAB_ASSERT_WARNING(false, "Unknown Opening Group: %s", _openingGroup.c_str());
		_openingGroup = "dragoons";    // we're misconfigured, but try to do something
	}

	// If we're doing a corsair thing and it's still working, slowly add more.
	if (_enemyRace == BWAPI::Races::Zerg &&
		hasStargate &&
		numCorsairs < 6 &&
		self->deadUnitCount(BWAPI::UnitTypes::Protoss_Corsair) == 0)
	{
		goal.push_back(MetaPair(BWAPI::UnitTypes::Protoss_Corsair, numCorsairs + 1));
	}

	// Maybe get some static defense against air attack.
	const int enemyAirToGround =
		InformationManager::Instance().getNumUnits(BWAPI::UnitTypes::Terran_Wraith, BWAPI::Broodwar->enemy()) / 8 +
		InformationManager::Instance().getNumUnits(BWAPI::UnitTypes::Terran_Battlecruiser, BWAPI::Broodwar->enemy()) / 3 +
		InformationManager::Instance().getNumUnits(BWAPI::UnitTypes::Protoss_Scout, BWAPI::Broodwar->enemy()) / 5 +
		InformationManager::Instance().getNumUnits(BWAPI::UnitTypes::Zerg_Mutalisk, BWAPI::Broodwar->enemy()) / 6;
	if (enemyAirToGround > 0)
	{
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Protoss_Photon_Cannon, enemyAirToGround));
	}

	// Get observers if we have a second base, or if the enemy has cloaked units.
	if (numNexusCompleted >= 2 || InformationManager::Instance().enemyHasCloakTech())
	{
		goal.push_back(MetaPair(BWAPI::UnitTypes::Protoss_Robotics_Facility, 1));

		if (numObservers < 3 && self->completedUnitCount(BWAPI::UnitTypes::Protoss_Robotics_Facility) > 0)
		{
			goal.push_back(MetaPair(BWAPI::UnitTypes::Protoss_Observer, numObservers + 1));
		}
	}

	// Make more probes, up to a limit.
	goal.push_back(MetaPair(BWAPI::UnitTypes::Protoss_Probe, std::min(maxProbes, numProbes + 8)));

	// If the map has islands, get drop after we have 3 bases.
	if (Config::Macro::ExpandToIslands && numNexusCompleted >= 3 && MapTools::Instance().hasIslandBases())
	{
		goal.push_back(MetaPair(BWAPI::UnitTypes::Protoss_Shuttle, 1));
	}

	// if we want to expand, insert a nexus into the build order
	if (shouldExpandNow())
	{
		goal.push_back(MetaPair(BWAPI::UnitTypes::Protoss_Nexus, numNexusAll + 1));
	}

	return goal;
}

const MetaPairVector StrategyManager::getTerranBuildOrderGoal()
{
	// the goal to return
	std::vector<MetaPair> goal;

	// These counts include uncompleted units.
	int numSCVs			= UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_SCV);
    int numCC           = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Command_Center);            
    int numRefineries   = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Refinery);            
    int numMarines      = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Marine);
	int numMedics       = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Medic);
	int numWraith       = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Wraith);
    int numVultures     = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Vulture);
	int numVessels		= UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Science_Vessel);
	int numGoliaths		= UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Goliath);
    int numTanks        = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode)
                        + UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Siege_Mode);
	////////////
	int numBunker       = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Bunker);
	int numTurret       = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Missile_Turret);
	int numFactory      = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Factory);
	int numBarracks     = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Barracks);
	int numArmorys      = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Armory);
	int numSupplys      = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Supply_Depot);

	int numMinerals = BWAPI::Broodwar->self()->minerals();
	int numGas = BWAPI::Broodwar->self()->gas();

	/////////////
	bool hasEBay		= UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Engineering_Bay) > 0;
	bool hasAcademy		= UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Academy) > 0;
	bool hasArmory		= UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Armory) > 0;
	bool hasFactory     = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) > 0;
	bool hasMachineShop = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) > 0;

	int maxSCVs = WorkerManager::Instance().getMaxWorkers();

	bool makeVessel = false;

	BWAPI::Player self = BWAPI::Broodwar->self();

////////////////////////////////////////////////////////++	
	//ӵλ
	//õҪӵĵλ[0]Ϊǹ[1]׳[2]̹ˣ[3]
	std::vector<int> numAttackUnitAdd = StrategyManager::numAttackUnitAdd();
	//SCV
	//ǹ
	if (numAttackUnitAdd[0] && numBarracks > 0)
	{
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Marine, numAttackUnitAdd[0] + numMarines));
	}
	//׳
	if (numAttackUnitAdd[1] && hasFactory)
	{
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Vulture, numAttackUnitAdd[1] + numVultures));
	}
	//̹
	if (numAttackUnitAdd[2] && hasFactory)
	{
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode, numAttackUnitAdd[2] + numTanks));
	}
	//
	if (numAttackUnitAdd[3] && hasFactory && hasMachineShop)
	{
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Goliath, numAttackUnitAdd[3] + numGoliaths));
	}
	//Ƽ
	if (makeVessel || InformationManager::Instance().enemyHasCloakTech())
	{
		// Maintain 1 vessel to spot for the ground squad and 1 to go with the recon squad.
		if (numVessels < 2)
		{
			goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Science_Vessel, numVessels + 1));
		}
	}

	//
	//ӪֻҪһӪ
	if (numBarracks < 1)
	{
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Barracks, 1));
	}
	//壬һΪǰ
	if (shouldBuildEBay())
	{
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Engineering_Bay, 1));
	}
	//оԺһΪṇ̀˷ܼ
	if (!hasAcademy && hasFactory)
	{
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Academy, 1));
	}
	//2ͬʱͷ
	if (numArmorys == 0 && hasFactory)
	{
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Academy, numArmorys + 1));
	}
	else if (numArmorys == 1 && numMinerals > 400 && numGas > 200)
	{
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Academy, numArmorys + 1));
	}

	//ر1Ҫ·ڣ
	if (shouldBuildBunker())
	{
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Bunker, numBunker + 1));
	}
	// 
	//3Ͷελ
	//ÿ2
	//·2Ҫ2
	if (shouldBuildTurret())
	{
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Missile_Turret, numTurret + 1));
	}
	//е⣬̹˼֩
	if (hasFactory && !hasMachineShop)
	{
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Machine_Shop, 1));
	}
	//
	if (numCC == 1)
	{
		if (shouldBuildCenter())
		{
			goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Command_Center, numCC + 1));
		}
	}
	else if (shouldExpandNow() && numCC < 4)
	{
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Command_Center, numCC + 1));
	}
	//
	if (numFactory < 1)
	{
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Factory, 1));
	}

	//ѧվ2еλǰ
	if (self->getUpgradeLevel(BWAPI::UpgradeTypes::Terran_Vehicle_Weapons) == 1 &&
		UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Science_Facility) == 0)
	{
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Science_Facility, 1));
	}

	//Ƽ
	if (hasMachineShop)
	{
		int numAttackUnits = numVultures + numTanks + numGoliaths;

		if (!self->hasResearched(BWAPI::TechTypes::Spider_Mines) && numVultures >= 2)
		{
			goal.push_back(std::pair<MacroAct, int>(BWAPI::TechTypes::Spider_Mines, 1));
		}

		if (!self->hasResearched(BWAPI::TechTypes::Tank_Siege_Mode))
		{
			goal.push_back(std::pair<MacroAct, int>(BWAPI::TechTypes::Tank_Siege_Mode, 1));
		}
		if (self->getUpgradeLevel(BWAPI::UpgradeTypes::Ion_Thrusters) == 0 && numAttackUnits > 15)
		{
			goal.push_back(std::pair<MacroAct, int>(BWAPI::UpgradeTypes::Ion_Thrusters, 1));
		}
		if (self->getUpgradeLevel(BWAPI::UpgradeTypes::Charon_Boosters) == 0 && numAttackUnits > 15)
		{
			goal.push_back(std::pair<MacroAct, int>(BWAPI::UpgradeTypes::Charon_Boosters, 1));
		}
	}
	/*
	if (hasAcademy)
	{
		if (!self->hasResearched(BWAPI::TechTypes::Stim_Packs))
		{
			goal.push_back(std::pair<MacroAct, int>(BWAPI::TechTypes::Stim_Packs, 1));
		}
		if (self->getUpgradeLevel(BWAPI::UpgradeTypes::U_238_Shells) == 0)
		{
			goal.push_back(std::pair<MacroAct, int>(BWAPI::UpgradeTypes::U_238_Shells, 1));
		}
	}
	*/
	if (hasArmory && shouldUpgrade(BWAPI::UpgradeTypes::Terran_Vehicle_Weapons))
	{
		int weaponsUps = self->getUpgradeLevel(BWAPI::UpgradeTypes::Terran_Vehicle_Weapons);
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UpgradeTypes::Terran_Vehicle_Weapons, weaponsUps + 1));
	}
	if (hasArmory && shouldUpgrade(BWAPI::UpgradeTypes::Terran_Vehicle_Plating))
	{
		int platUps = self->getUpgradeLevel(BWAPI::UpgradeTypes::Terran_Vehicle_Plating);
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UpgradeTypes::Terran_Vehicle_Plating, platUps + 1));
	}

	// Make more SCVs, up to a limit. The anti-rush strategy makes its own SCVs.
	goal.push_back(MetaPair(BWAPI::UnitTypes::Terran_SCV, std::min(maxSCVs + 3, numSCVs + 2 * numCC + 1)));

	/*
	if (_openingGroup == "SafeExpend")
	{
		int numAttackUnits = numVultures + numTanks + numGoliaths;

		if (hasMachineShop)
		{
			if (!self->hasResearched(BWAPI::TechTypes::Spider_Mines) && numVultures >= 2)
			{
				goal.push_back(std::pair<MacroAct, int>(BWAPI::TechTypes::Spider_Mines, 1));
			}

			if (!self->hasResearched(BWAPI::TechTypes::Tank_Siege_Mode) && numTanks >= 1)
			{
				goal.push_back(std::pair<MacroAct, int>(BWAPI::TechTypes::Tank_Siege_Mode, 1));
			}
			if (self->getUpgradeLevel(BWAPI::UpgradeTypes::Ion_Thrusters) == 0 && numAttackUnits > 15)
			{
				goal.push_back(std::pair<MacroAct, int>(BWAPI::UpgradeTypes::Ion_Thrusters, 1));
			}
			if (self->getUpgradeLevel(BWAPI::UpgradeTypes::Charon_Boosters) == 0 && numAttackUnits > 15)
			{
				goal.push_back(std::pair<MacroAct, int>(BWAPI::UpgradeTypes::Charon_Boosters, 1));
			}
		}

		if (hasFactory)
		{
			if (numCC <= 2 && numGoliaths < 6)
			{
				goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Goliath, numGoliaths + 4));
			}
			else
			{
				if (numTanks < 8 )
				{
					goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode, numTanks + 4));
				}
				else if (numTanks >= 8 && numVultures < 15)
				{
					goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Vulture, numVultures + 4));
				}
				else if (numTanks <= 14  && numVultures >= 15)
				{
					goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode, numTanks + 4));
				}
				else
				{
					goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Vulture, numVultures + 4));
				}
				if (numGoliaths < 12)
				{
					goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Goliath, numGoliaths + 4));
				}
			}
		}
		else
		{
			goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Factory, 1));
		}

		if (!hasArmory && numAttackUnits > 20)
		{
			goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Armory, 1));
		}
		else if (hasArmory  && numAttackUnits > 20)
		{
			int weaponsUps = self->getUpgradeLevel(BWAPI::UpgradeTypes::Terran_Vehicle_Weapons);
			if (weaponsUps == 0 && !self->isUpgrading(BWAPI::UpgradeTypes::Terran_Vehicle_Weapons) && numAttackUnits > 20)
			{
				goal.push_back(std::pair<MacroAct, int>(BWAPI::UpgradeTypes::Terran_Vehicle_Weapons, 1));
			}
			else if (weaponsUps > 0 &&
				self->getUpgradeLevel(BWAPI::UpgradeTypes::Terran_Vehicle_Plating) == 0 &&
				!self->isUpgrading(BWAPI::UpgradeTypes::Terran_Vehicle_Plating) && numAttackUnits > 25)
			{
				goal.push_back(std::pair<MacroAct, int>(BWAPI::UpgradeTypes::Terran_Vehicle_Plating, 1));
			}
			else if (weaponsUps > 0 &&
				weaponsUps < 3 &&
				!self->isUpgrading(BWAPI::UpgradeTypes::Terran_Vehicle_Weapons) && numAttackUnits >35)
			{
				goal.push_back(std::pair<MacroAct, int>(BWAPI::UpgradeTypes::Terran_Vehicle_Weapons, weaponsUps + 1));
			}
		}

		
		if (numBunker <= 1)
		{
			goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Bunker, std::min(numBunker + 1, 2)));
		}
		
	}
	////////////////////////////////////////////////////////__

	else if (_openingGroup == "anti-rush")
	{
		int numRax = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Barracks);

		CombatCommander::Instance().setAggression(false);
		
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Marine, numMarines + 1));
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_SCV, std::min(maxSCVs, numSCVs + 1)));
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Bunker, 1));
		
		if (self->minerals() > 250)
		{
			goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Barracks, numRax + 1));
		}

		// If we survived long enough, transition to something more interesting.
		if (numMarines >= 10)
		{
			_openingGroup = "SafeExpend";
		}
	}
	/*
	else if (_openingGroup == "bio")
    {
	    goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Marine, numMarines + 8));

		if (numMarines >= 10)
		{
			goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Academy, 1));
			if (numRefineries == 0)
			{
				goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Refinery, 1));
			}
		}
		if (hasAcademy)
		{
			// 1 medic for each 5 marines.
			int medicGoal = std::max(numMedics, numMarines / 5);
			goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Medic, medicGoal));
			if (!self->hasResearched(BWAPI::TechTypes::Stim_Packs))
			{
				goal.push_back(std::pair<MacroAct, int>(BWAPI::TechTypes::Stim_Packs, 1));
			}
			else
			{
				goal.push_back(std::pair<MacroAct, int>(BWAPI::UpgradeTypes::U_238_Shells, 1));
			}
		}
        if (numMarines > 16)
        {
            goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Engineering_Bay, 1));
        }
		if (hasEBay)
		{
			int weaponsUps = self->getUpgradeLevel(BWAPI::UpgradeTypes::Terran_Infantry_Weapons);
			if (weaponsUps == 0 &&
				!self->isUpgrading(BWAPI::UpgradeTypes::Terran_Infantry_Weapons))
			{
				goal.push_back(std::pair<MacroAct, int>(BWAPI::UpgradeTypes::Terran_Infantry_Weapons, 1));
			}
			else if (weaponsUps > 0 &&
				self->getUpgradeLevel(BWAPI::UpgradeTypes::Terran_Infantry_Armor) == 0 &&
				!self->isUpgrading(BWAPI::UpgradeTypes::Terran_Infantry_Armor))
			{
				goal.push_back(std::pair<MacroAct, int>(BWAPI::UpgradeTypes::Terran_Infantry_Armor, 1));
			}
			else if (weaponsUps > 0 &&
				weaponsUps < 3 &&
				!self->isUpgrading(BWAPI::UpgradeTypes::Terran_Infantry_Weapons) &&
				numVessels > 0)
			{
goal.push_back(std::pair<MacroAct, int>(BWAPI::UpgradeTypes::Terran_Infantry_Weapons, weaponsUps + 1));
			}
		}

		// Add in tanks if they're useful.
		const int enemiesCounteredByTanks =
			InformationManager::Instance().getNumUnits(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode, BWAPI::Broodwar->enemy()) +
			InformationManager::Instance().getNumUnits(BWAPI::UnitTypes::Terran_Siege_Tank_Siege_Mode, BWAPI::Broodwar->enemy()) +
			InformationManager::Instance().getNumUnits(BWAPI::UnitTypes::Protoss_Dragoon, BWAPI::Broodwar->enemy()) +
			InformationManager::Instance().getNumUnits(BWAPI::UnitTypes::Protoss_Reaver, BWAPI::Broodwar->enemy()) +
			InformationManager::Instance().getNumUnits(BWAPI::UnitTypes::Zerg_Lurker, BWAPI::Broodwar->enemy()) +
			InformationManager::Instance().getNumUnits(BWAPI::UnitTypes::Zerg_Ultralisk, BWAPI::Broodwar->enemy());
		const bool enemyHasStaticDefense =
			InformationManager::Instance().getNumUnits(BWAPI::UnitTypes::Terran_Bunker, BWAPI::Broodwar->enemy()) > 0 ||
			InformationManager::Instance().getNumUnits(BWAPI::UnitTypes::Protoss_Photon_Cannon, BWAPI::Broodwar->enemy()) > 0 ||
			InformationManager::Instance().getNumUnits(BWAPI::UnitTypes::Zerg_Sunken_Colony, BWAPI::Broodwar->enemy()) > 0;
		if (enemiesCounteredByTanks > 0 || enemyHasStaticDefense)
		{
			int nTanksWanted;
			if (enemiesCounteredByTanks > 0)
			{
				nTanksWanted = std::min(numMarines / 4, enemiesCounteredByTanks);
				nTanksWanted = std::min(nTanksWanted, numTanks + 2);
			}
			else
			{
				nTanksWanted = numTanks;
				if (numTanks < 2)
				{
					nTanksWanted = numTanks + 1;
				}
			}
			goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode, nTanksWanted));
			goal.push_back(std::pair<MacroAct, int>(BWAPI::TechTypes::Tank_Siege_Mode, 1));
		}
	}
	else if (_openingGroup == "vultures")
	{
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Vulture, numVultures + 3));
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UpgradeTypes::Ion_Thrusters, 1));

		if (numVultures >= 6)
		{
			// The rush is over, transition out on the next call.
			_openingGroup = "tanks";
		}
	}
	else if (_openingGroup == "tanks")
	{
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Vulture, numVultures + 4));
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode, numTanks + 2));
		goal.push_back(std::pair<MacroAct, int>(BWAPI::TechTypes::Tank_Siege_Mode, 1));

		if (numVultures > 0)
		{
			goal.push_back(std::pair<MacroAct, int>(BWAPI::UpgradeTypes::Ion_Thrusters, 1));
		}
		if (numTanks >= 6)
		{
			goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Goliath, numGoliaths + 4));
		}
		if (numGoliaths >= 4)
		{
			goal.push_back(std::pair<MacroAct, int>(BWAPI::UpgradeTypes::Charon_Boosters, 1));
		}
		if (self->hasResearched(BWAPI::TechTypes::Tank_Siege_Mode))
		{
			makeVessel = true;
		}
	}
	else if (_openingGroup == "drop")
	{
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UpgradeTypes::Ion_Thrusters, 1));
		goal.push_back(MetaPair(BWAPI::UnitTypes::Terran_Vulture, numVultures + 1));

		// The drop prep is carried out entirely by the opening book.
		// Immediately transition into something else.
		if (_enemyRace == BWAPI::Races::Zerg)
		{
			_openingGroup = "bio";
		}
		else
		{
			_openingGroup = "tanks";
		}
	}
	else
	{
		BWAPI::Broodwar->printf("Unknown Opening Group: %s", _openingGroup.c_str());
		_openingGroup = "bio";       // we're misconfigured, but try to do something
	}
	//////////////

	if (numCC > 1 || InformationManager::Instance().enemyHasCloakTech())
	{
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Academy, 1));
		if (numRefineries == 0)
		{
			goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Refinery, 1));
		}
	}
	
	const int enemyAirToGround =
		InformationManager::Instance().getNumUnits(BWAPI::UnitTypes::Terran_Wraith, BWAPI::Broodwar->enemy()) / 6 +
		InformationManager::Instance().getNumUnits(BWAPI::UnitTypes::Terran_Battlecruiser, BWAPI::Broodwar->enemy()) / 2 +
		InformationManager::Instance().getNumUnits(BWAPI::UnitTypes::Protoss_Scout, BWAPI::Broodwar->enemy()) / 3 +
		InformationManager::Instance().getNumUnits(BWAPI::UnitTypes::Zerg_Mutalisk, BWAPI::Broodwar->enemy()) / 4;
	if (enemyAirToGround > 0)
	{
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Missile_Turret, enemyAirToGround * 2));
	}
	
	if (numCC > 0 && hasAcademy)
	{
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Comsat_Station, UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Command_Center)));
	}

	if (makeVessel || InformationManager::Instance().enemyHasCloakTech())
	{
		// Maintain 1 vessel to spot for the ground squad and 1 to go with the recon squad.
		if (numVessels < 2)
		{
			goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Science_Vessel, numVessels + 1));
		}
	}

	// Make more SCVs, up to a limit. The anti-rush strategy makes its own SCVs.
	if (_openingGroup != "anti-rush")
	{
		goal.push_back(MetaPair(BWAPI::UnitTypes::Terran_SCV, std::min(maxSCVs + 3, numSCVs + 2 * numCC + 3)));
	}

	// If the map has islands, get drop after we have 3 bases.
	if (Config::Macro::ExpandToIslands && numCC >= 3 && MapTools::Instance().hasIslandBases())
	{
		goal.push_back(MetaPair(BWAPI::UnitTypes::Terran_Dropship, 1));
	}

	if (shouldExpandNow() && numCC < 4)
	{
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Command_Center, numCC + 1));
	}
	*/
	return goal;
}

// BOSS method of choosing a zerg production plan. UNUSED!
// See freshProductionPlan() for the current method.
const MetaPairVector StrategyManager::getZergBuildOrderGoal() const
{
	// the goal to return
	std::vector<MetaPair> goal;

	// These counts include uncompleted units.
	int nLairs = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Lair);
	int nHives = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Hive);
	int nHatches = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Hatchery)
		+ nLairs + nHives;
	int nDrones = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Drone);
	int nHydras = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Zerg_Hydralisk);

	const int droneMax = 48;             // number of drones not to exceed

	// Simple default strategy as an example in case you want to use this method.
	goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Zerg_Hydralisk, nHydras + 12));
	if (shouldExpandNow())
	{
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Zerg_Hatchery, nHatches + 1));
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Zerg_Drone, std::min(droneMax, nDrones + 10)));
	}
	else
	{
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Zerg_Drone, std::min(droneMax, nDrones + 2)));
	}

	return goal;
}

void StrategyManager::handleUrgentProductionIssues(BuildOrderQueue & queue)
{
	// This is the enemy plan that we have seen in action.
	OpeningPlan enemyPlan = OpponentModel::Instance().getEnemyPlan();

	// For all races, if we've just discovered that the enemy is going with a heavy macro opening,
	// drop any static defense that our opening build order told us to make.
	if (!ProductionManager::Instance().isOutOfBook() && !_openingStaticDefenseDropped)
	{
		// We're in the opening book and haven't dropped static defenses yet. Should we?
		if (enemyPlan == OpeningPlan::Turtle ||
			enemyPlan == OpeningPlan::SafeExpand)
			// enemyPlan == OpeningPlan::NakedExpand && _enemyRace != BWAPI::Races::Zerg) // could do this too
		{
			// 1. Remove upcoming defense buildings from the queue.
			queue.dropStaticDefenses();
			// 2. Cancel unfinished defense buildings.
			for (BWAPI::Unit unit : BWAPI::Broodwar->self()->getUnits())
			{
				if (UnitUtil::IsComingStaticDefense(unit->getType()) && unit->canCancelConstruction())
				{
					the.micro.Cancel(unit);
				}
			}
			// 3. Never do it again.
			_openingStaticDefenseDropped = true;
		}
	}

	// All other considerations are handled separately by zerg.
	if (_selfRace == BWAPI::Races::Zerg)
	{
		StrategyBossZerg::Instance().handleUrgentProductionIssues(queue);
	}
	else
	{
		// Count resource depots.
		const BWAPI::UnitType resourceDepotType = _selfRace == BWAPI::Races::Terran
			? BWAPI::UnitTypes::Terran_Command_Center
			: BWAPI::UnitTypes::Protoss_Nexus;
		const int numDepots = UnitUtil::GetAllUnitCount(resourceDepotType);

		// If we need to cope with an extreme emergency, don't do anything else.
		// If we have no resource depot, we can do nothing; that case is dealt with below.
		if (numDepots > 0 && handleExtremeEmergency(queue))
		{
			return;
		}

		// If there are no workers, many reactions can't happen.
		const bool anyWorkers =
			UnitUtil::GetAllUnitCount(_selfRace == BWAPI::Races::Terran
			? BWAPI::UnitTypes::Terran_SCV
			: BWAPI::UnitTypes::Protoss_Probe) > 0;
		/*
		//7.17.2019//д滻
		// detect if there's a supply block once per second
		if ((BWAPI::Broodwar->getFrameCount() % 24 == 1) && detectSupplyBlock(queue) && anyWorkers)
		{
			if (Config::Debug::DrawBuildOrderSearchInfo)
			{
				BWAPI::Broodwar->printf("Supply block, building supply!");
			}

			queue.queueAsHighestPriority(MacroAct(BWAPI::Broodwar->self()->getRace().getSupplyProvider()));
		}
		*/
		const MacroAct * nextInQueuePtr = queue.isEmpty() ? nullptr : &(queue.getHighestPriorityItem().macroAct);

		// If we need gas, make sure it is turned on.
		int gas = BWAPI::Broodwar->self()->gas();
		if (nextInQueuePtr)
		{
			if (nextInQueuePtr->gasPrice() > gas)
			{
				WorkerManager::Instance().setCollectGas(true);
			}
		}

		// If we're protoss and building is stalled for lack of space,
		// schedule a pylon to make more space where buildings can be placed.
		if (BuildingManager::Instance().getStalledForLackOfSpace())
		{
			if (_selfRace == BWAPI::Races::Protoss && 
				(!nextInQueuePtr || !nextInQueuePtr->isBuilding() || nextInQueuePtr->getUnitType() != BWAPI::UnitTypes::Protoss_Pylon) &&
				!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Protoss_Pylon))
			{
				queue.queueAsHighestPriority(BWAPI::UnitTypes::Protoss_Pylon);
				return;				// and call it a day
			}
		}

		// If we have collected too much gas, turn it off.
		if (ProductionManager::Instance().isOutOfBook() &&
			gas > 400 &&
			gas > 4 * BWAPI::Broodwar->self()->minerals())
		{
			int queueMinerals, queueGas;
			queue.totalCosts(queueMinerals, queueGas);
			if (gas >= queueGas)
			{
				WorkerManager::Instance().setCollectGas(false);
			}
		}

		// If they have mobile cloaked units, get some static detection.
		if (InformationManager::Instance().enemyHasMobileCloakTech() && anyWorkers)
		{
			if (_selfRace == BWAPI::Races::Protoss)
			{
				if (BWAPI::Broodwar->self()->allUnitCount(BWAPI::UnitTypes::Protoss_Photon_Cannon) < 2 &&
					!queue.anyInQueue(BWAPI::UnitTypes::Protoss_Photon_Cannon) &&
					!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Protoss_Photon_Cannon))
				{
					queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Protoss_Photon_Cannon));
					queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Protoss_Photon_Cannon));

					if (BWAPI::Broodwar->self()->allUnitCount(BWAPI::UnitTypes::Protoss_Forge) == 0 &&
						!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Protoss_Forge))
					{
						queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Protoss_Forge));
					}
				}
			}
			else if (_selfRace == BWAPI::Races::Terran)
			{
				if (BWAPI::Broodwar->self()->allUnitCount(BWAPI::UnitTypes::Terran_Missile_Turret) < 3 &&
					!queue.anyInQueue(BWAPI::UnitTypes::Terran_Missile_Turret) &&
					!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Missile_Turret))
				{
					queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Missile_Turret));
					queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Missile_Turret));
					queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Missile_Turret));

					if (BWAPI::Broodwar->self()->allUnitCount(BWAPI::UnitTypes::Terran_Engineering_Bay) == 0 &&
						!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Engineering_Bay))
					{
						queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Engineering_Bay));
					}
				}
			}
		}

		// This is the enemy plan that we have seen, or if none yet, the expected enemy plan.
		// Some checks can use the expected plan, some are better with the observed plan.
		OpeningPlan likelyEnemyPlan = OpponentModel::Instance().getBestGuessEnemyPlan();

		// If the opponent is rushing, make some defense.
		if (likelyEnemyPlan == OpeningPlan::Proxy ||
			likelyEnemyPlan == OpeningPlan::WorkerRush ||
			likelyEnemyPlan == OpeningPlan::FastRush ||
			enemyPlan == OpeningPlan::HeavyRush)           // we can react later to this
		{
			// If we are terran and have marines, make a bunker.
			if (_selfRace == BWAPI::Races::Terran)
			{
				if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Bunker) &&
					UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Marine) > 0 &&          // usefulness requirement
					UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Barracks) > 0 &&  // tech requirement for a bunker
					UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Bunker) == 0 &&
					!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Bunker) &&
					anyWorkers)
				{
					queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Bunker, MacroLocation::Front));
				}
			}

			// If we are protoss, make a shield battery.
			// NOTE This works, but is turned off because protoss can't use the battery yet.
			/*
			else if (_selfRace == BWAPI::Races::Protoss)
			{
			if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Protoss_Pylon) > 0 &&    // tech requirement
			UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Protoss_Gateway) > 0 &&  // tech requirement
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Protoss_Shield_Battery) == 0 &&
			!queue.anyInQueue(BWAPI::UnitTypes::Protoss_Shield_Battery) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Protoss_Shield_Battery) &&
			anyWorkers)
			{
			queue.queueAsHighestPriority(BWAPI::UnitTypes::Protoss_Shield_Battery);
			}
			}
			*/
		}

		if (numDepots > _highWaterBases)
		{
			_highWaterBases = numDepots;
		}
		bool makeResourceDepot = false;

		// If there is no resource depot, order one if we can afford it.
		// NOTE Does not check whether we have a worker to build it.
		if (numDepots == 0 && BWAPI::Broodwar->self()->minerals() >= 400)
		{
			makeResourceDepot = true;
		}

		// If the opponent fast expanded and we haven't taken the natural yet, do that immediately.
		// Not if the enemy is zerg, though. Zerg can be ahead in expansions.
		if (enemyPlan == OpeningPlan::SafeExpand || enemyPlan == OpeningPlan::NakedExpand)
		{
			// Use _highWaterBases instead of numDepots so we don't try to remake a destroyed natural.
			if (_highWaterBases == 1 && BWAPI::Broodwar->enemy()->getRace() != BWAPI::Races::Zerg)
			{
				makeResourceDepot = true;
			}
		}

		// We only care about the next item in the queue, not possible later resource depots in the queue.
		// This should be after other rules that may add something, so that no other emegency reaction
		// pushes down the resource depot in the queue. Otherwise the rule will fire repeatedly.
		if (makeResourceDepot &&
			anyWorkers &&
			(!nextInQueuePtr || !nextInQueuePtr->isUnit() || nextInQueuePtr->getUnitType() != resourceDepotType) &&
			!BuildingManager::Instance().isBeingBuilt(resourceDepotType))
		{
			queue.queueAsHighestPriority(MacroAct(resourceDepotType));
			return;    // and don't do anything else just yet
		}
	}
}

// Return true if we're supply blocked and should build supply.
// NOTE This understands zerg supply but is not used when we are zerg.
bool StrategyManager::detectSupplyBlock(BuildOrderQueue & queue) const
{
	// If the _queue is empty or supply is maxed, there is no block.
	if (queue.isEmpty() || BWAPI::Broodwar->self()->supplyTotal() >= 400)
	{
		return false;
	}

	// If supply is being built now, there's no block. Return right away.
	// Terran and protoss calculation:
	if (BuildingManager::Instance().isBeingBuilt(BWAPI::Broodwar->self()->getRace().getSupplyProvider()))
	{
		return false;
	}

	// Terran and protoss calculation:
	int supplyAvailable = BWAPI::Broodwar->self()->supplyTotal() - BWAPI::Broodwar->self()->supplyUsed();

	// Zerg calculation:
	// Zerg can create an overlord that doesn't count toward supply until the next check.
	// To work around it, add up the supply by hand, including hatcheries.
	if (BWAPI::Broodwar->self()->getRace() == BWAPI::Races::Zerg) {
		supplyAvailable = -BWAPI::Broodwar->self()->supplyUsed();
		for (auto unit : BWAPI::Broodwar->self()->getUnits())
		{
			if (unit->getType() == BWAPI::UnitTypes::Zerg_Overlord)
			{
				supplyAvailable += 16;
			}
			else if (unit->getType() == BWAPI::UnitTypes::Zerg_Egg &&
				unit->getBuildType() == BWAPI::UnitTypes::Zerg_Overlord)
			{
				return false;    // supply is building, return immediately
				// supplyAvailable += 16;
			}
			else if ((unit->getType() == BWAPI::UnitTypes::Zerg_Hatchery && unit->isCompleted()) ||
				unit->getType() == BWAPI::UnitTypes::Zerg_Lair ||
				unit->getType() == BWAPI::UnitTypes::Zerg_Hive)
			{
				supplyAvailable += 2;
			}
		}
	}

	int supplyCost = queue.getHighestPriorityItem().macroAct.supplyRequired();
	// Available supply can be negative, which breaks the test below. Fix it.
	supplyAvailable = std::max(0, supplyAvailable);

	// if we don't have enough supply, we're supply blocked
	if (supplyAvailable < supplyCost)
	{
		// If we're zerg, check to see if a building is planned to be built.
		// Only count it as releasing supply very early in the game.
		if (_selfRace == BWAPI::Races::Zerg
			&& BuildingManager::Instance().buildingsQueued().size() > 0
			&& BWAPI::Broodwar->self()->supplyTotal() <= 18)
		{
			return false;
		}
		return true;
	}

	return false;
}

// This tries to cope with 1 kind of severe emergency: We have desperately few workers.
// The caller promises that we have a resource depot, so we may be able to make more.
bool StrategyManager::handleExtremeEmergency(BuildOrderQueue & queue)
{
	const int minWorkers = 3;
	const BWAPI::UnitType workerType = _selfRace.getWorker();
	const int nWorkers = UnitUtil::GetAllUnitCount(workerType);

	// NOTE This doesn't check whether the map has resources remaining!
	//      If not, we should produce workers only if needed for another purpose.
	// NOTE If we don't have enough minerals to make a worker, then we don't
	//      have enough minerals to make anything (since we're not zerg and can't make scourge).
	//      So don't bother.
	if (nWorkers < minWorkers && BWAPI::Broodwar->self()->minerals() >= 50)
	{
		// 1. If the next item in the queue is a worker, we're good. Otherwise, clear the queue.
		// This is a severe emergency and it doesn't make sense to continue business as usual.
		// But if we don't have enough 
		if (queue.size() > 0)
		{
			const MacroAct & act = queue.getHighestPriorityItem().macroAct;
			if (act.isUnit() && act.getUnitType() == workerType)
			{
				return false;
			}
			queue.clearAll();
		}
		// 2. Queue the minimum number of workers.
		for (int i = nWorkers; i < minWorkers; ++i)
		{
			queue.queueAsHighestPriority(workerType);
		}
		return true;
	}

	return false;
}

// Called to refill the production queue when it is empty.
void StrategyManager::freshProductionPlan()
{
	if (_selfRace == BWAPI::Races::Zerg)
	{
		ProductionManager::Instance().setBuildOrder(StrategyBossZerg::Instance().freshProductionPlan());
	}
	else
	{
		performBuildOrderSearch();
	}
}

void StrategyManager::performBuildOrderSearch()
{
	if (!canPlanBuildOrderNow())
	{
		return;
	}

	BuildOrder & buildOrder = BOSSManager::Instance().getBuildOrder();

	if (buildOrder.size() > 0)
	{
		ProductionManager::Instance().setBuildOrder(buildOrder);
		BOSSManager::Instance().reset();
	}
	else
	{
		if (!BOSSManager::Instance().isSearchInProgress())
		{
			BOSSManager::Instance().startNewSearch(getBuildOrderGoal());
		}
	}
}

// this will return true if any unit is on the first frame of its training time remaining
// this can cause issues for the build order search system so don't plan a search on these frames
bool StrategyManager::canPlanBuildOrderNow() const
{
	for (const auto unit : BWAPI::Broodwar->self()->getUnits())
	{
		if (unit->getRemainingTrainTime() == 0)
		{
			continue;
		}

		BWAPI::UnitType trainType = unit->getLastCommand().getUnitType();

		if (unit->getRemainingTrainTime() == trainType.buildTime())
		{
			return false;
		}
	}

	return true;
}

// Do we expect or plan to drop at some point during the game?
bool StrategyManager::dropIsPlanned() const
{
	// Don't drop in ZvZ.
	if (_selfRace == BWAPI::Races::Zerg && BWAPI::Broodwar->enemy()->getRace() == BWAPI::Races::Zerg)
	{
		return false;
	}

	// Otherwise plan drop if the opening says so, or if the map has islands to take.
	return getOpeningGroup() == "drop" ||
		Config::Macro::ExpandToIslands && MapTools::Instance().hasIslandBases();
}

// Whether we have the tech and transport to drop.
bool StrategyManager::hasDropTech()
{
	if (_selfRace == BWAPI::Races::Zerg)
	{
		// NOTE May be slow drop.
		return BWAPI::Broodwar->self()->getUpgradeLevel(BWAPI::UpgradeTypes::Ventral_Sacs) > 0 &&
			UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Zerg_Overlord) > 0;
	}
	if (_selfRace == BWAPI::Races::Protoss)
	{
		return UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Protoss_Shuttle) > 0;
	}
	if (_selfRace == BWAPI::Races::Terran)
	{
		return UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Dropship) > 0;
	}

	return false;
}

//8.2.2019//жǷҪӵλ
std::vector<int> StrategyManager::isMechanicalUnitAdd()
{
	int numGoliaths = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Goliath);
	int numTanks = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode)
		+ UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Siege_Mode);
	int numVultures = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Vulture);
	int numMinerals = BWAPI::Broodwar->self()->minerals();
	int numGas = BWAPI::Broodwar->self()->gas();
	std::vector<int> unitAdd;
	bool shouldAddForGoliath = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Armory) > 0;//Ϊtrue1.Armory 2.зпеλ

	//Step1_㵱ǰжٸ

	int numFactorys = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory);
	//ӸĹλҪȥ
	int numMachineBeingBuilt = BuildingManager::Instance().numBeingBuilt(BWAPI::UnitTypes::Terran_Machine_Shop);
	//еĹ
	int numUncompletedUnits = UnitUtil::GetUncompletedUnitCount(BWAPI::UnitTypes::Terran_Vulture) +
		UnitUtil::GetUncompletedUnitCount(BWAPI::UnitTypes::Terran_Goliath) +
		UnitUtil::GetUncompletedUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode);
	//й
	int numFreeSpace = numFactorys - numMachineBeingBuilt - numUncompletedUnits;
	//ûпйӵλ
	if (numFreeSpace == 0)
	{
		unitAdd.push_back(0);//׳
		unitAdd.push_back(0);//
		unitAdd.push_back(0);//̹
		return unitAdd;
	}

	//step2_жǷҪĿ굥λ

	//Ӧ
	//Էеλ࣬˵30
	if (enemyFlyArmyGroup() && numGoliaths < 30 && shouldAddForGoliath)
	{
		unitAdd.push_back(0);//׳
		unitAdd.push_back(1);//
		unitAdd.push_back(0);//̹
		return unitAdd;
	}
	else if (numMinerals > 1200 && numGas < 200)
	{
		//ֻ׳
		unitAdd.push_back(1);//׳
		unitAdd.push_back(0);//
		unitAdd.push_back(0);//̹
		return unitAdd;
	}
	//0׶ΣһմBлC
	//Ŀ꣺5׳1̹ˣ7
	else if (numVultures < 4 || numTanks < 1 || (numGoliaths < 6 && shouldAddForGoliath))
	{
		//Ȳ˵10
		if (numGoliaths < 6 && shouldAddForGoliath)
		{
			unitAdd.push_back(0);//׳
			unitAdd.push_back(1);//
			unitAdd.push_back(0);//̹
			return unitAdd;
		}
		else if (numVultures < 4)
		{
			unitAdd.push_back(1);//׳
			unitAdd.push_back(0);//
			unitAdd.push_back(0);//̹
			return unitAdd;
		}
		else
		{
			unitAdd.push_back(0);//׳
			unitAdd.push_back(0);//
			unitAdd.push_back(1);//̹
			return unitAdd;
		}
	}
	//1׶Σڴ·ڴ
	//Ŀ꣺8׳5̹ˣ12
	else if (numVultures < 9 || numTanks < 5 || (numGoliaths < 12 && shouldAddForGoliath))
	{
		//TANK
		if (numTanks < 5)
		{
			unitAdd.push_back(0);//׳
			unitAdd.push_back(0);//
			unitAdd.push_back(1);//̹
			return unitAdd;
		}
		else if (numGoliaths < 12 && shouldAddForGoliath)
		{
			unitAdd.push_back(0);//׳
			unitAdd.push_back(1);//
			unitAdd.push_back(0);//̹
			return unitAdd;
		}
		else
		{
			unitAdd.push_back(1);//׳
			unitAdd.push_back(0);//
			unitAdd.push_back(0);//̹
			return unitAdd;
		}
	}
	//2׶Σ1ƽ
	//Ŀ꣺15׳9̹ˣ18
	else if (numTanks < 14 || (numGoliaths < 22 && shouldAddForGoliath))
	{
		//̹
		if (numTanks < 9)
		{
			unitAdd.push_back(0);//׳
			unitAdd.push_back(0);//
			unitAdd.push_back(1);//̹
			return unitAdd;
		}
		else if (numGoliaths < 17 && shouldAddForGoliath)
		{
			unitAdd.push_back(0);//׳
			unitAdd.push_back(1);//
			unitAdd.push_back(0);//̹
			return unitAdd;
		}		
		else if (numTanks < 14)
		{
			unitAdd.push_back(0);//׳
			unitAdd.push_back(0);//
			unitAdd.push_back(1);//̹
			return unitAdd;
		}
		else
		{
			unitAdd.push_back(0);//׳
			unitAdd.push_back(1);//
			unitAdd.push_back(0);//̹
			return unitAdd;
		}
	}
	//3׶Σڴ13ƽ
	//Ŀ꣺18׳12̹ˣ25
	else if (numTanks < 18 || (numGoliaths < 27 && shouldAddForGoliath))
	{
		if (numTanks < 18)
		{
			unitAdd.push_back(0);//׳
			unitAdd.push_back(0);//
			unitAdd.push_back(1);//̹
			return unitAdd;
		}	    
		else
		{
			unitAdd.push_back(0);//׳
			unitAdd.push_back(1);//
			unitAdd.push_back(0);//̹
			return unitAdd;
		}
	}
	//4׶Σ3פأֱ
	//Ŀ꣺21׳15̹ˣ28
	else if (numTanks < 23 || (numGoliaths < 33 && shouldAddForGoliath))
	{
		if (numTanks < 23)
		{
			unitAdd.push_back(0);//׳
			unitAdd.push_back(0);//
			unitAdd.push_back(1);//̹
			return unitAdd;
		}
		else
		{
			unitAdd.push_back(0);//׳
			unitAdd.push_back(1);//
			unitAdd.push_back(0);//̹
			return unitAdd;
		}
	}
	else
	{
		unitAdd.push_back(0);//׳
		unitAdd.push_back(0);//
		unitAdd.push_back(1);//̹
		return unitAdd;
	}
}


//////////////////////////////////////////////++6.22
//㵱ǰҪеĵλ
std::vector<int> StrategyManager::numAttackUnitAdd()
{
	int numMarines = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Marine);
	int numGoliaths = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Goliath);
	int numTanks = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode)
		+ UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Siege_Mode);
	int numVultures = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Vulture);
	int numMinerals = BWAPI::Broodwar->self()->minerals();
	int numGas = BWAPI::Broodwar->self()->gas();
	int numFactorys = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory);

	bool shouldAddTank = false;
	bool shouldAddGoliath = false;
	bool shouldAddVulture = false;
	std::vector<int> unitAdd;

	//ǹ
	if (!CombatCommander::Instance().getAggression())
	{
		//ౣ8ǹ4أ4ǰߵĵر
		if (numMarines >= 4)
		{
			unitAdd.push_back(0);
		}
		else if (numMarines < 4)
		{
			//Դ¼ͬʱĻǹ
			if (numMinerals <= 200)
			{
				unitAdd.push_back(1);
			}
			else if (numMinerals <= 500)
			{
				unitAdd.push_back(2);
			}
			else
			{
				unitAdd.push_back(4);
			}
		}
	}
	else
	{
		unitAdd.push_back(0);//ܹҪĻǹ
	}
	//жǷҪ׳̹ˣ
	if (enemyFlyArmyGroup())
	{
		if (numGoliaths < 20)
		{
			shouldAddVulture = false;
			shouldAddTank = false;
			shouldAddGoliath = true;
		}
		else
		{
			shouldAddVulture = true;
			shouldAddTank = true;
			shouldAddGoliath = true;
		}
		if (numVultures < 12)
		{
			shouldAddVulture = true;
		}
		if (numTanks < 10)
		{
			shouldAddTank = true;
		}
	}
	else
	{
		if (numGoliaths < 10)
		{
			shouldAddVulture = false;
			shouldAddTank = false;
			shouldAddGoliath = true;
		}
		else if (numGoliaths >= 10 && numVultures < 4)
		{
			shouldAddVulture = true;
			shouldAddTank = false;
			shouldAddGoliath = false;
		}
		else if (numGoliaths >= 8 && numVultures >= 4 && numTanks < 8)
		{
			shouldAddVulture = false;
			shouldAddTank = true;
			shouldAddGoliath = false;
		}
		else if (numTanks < 14)
		{
			shouldAddVulture = true;
			shouldAddTank = true;
			shouldAddGoliath = true;
		}
		else
		{
			shouldAddVulture = true;
			shouldAddTank = false;
			shouldAddGoliath = true;
		}
		if (numTanks < 2)
		{
			shouldAddTank = true;
		}
	}
	//ûκελҪӣ
	if (!shouldAddVulture && !shouldAddTank && !shouldAddGoliath && unitAdd[0] == 0)
	{
		shouldAddGoliath = true;
	}
	//Ҫӣож
	int numType = 4 - (shouldAddVulture + shouldAddTank + shouldAddGoliath);//1-4ֵֻ֮Ҫһ֣Ӵ

	if (shouldAddVulture)
	{
		if (numFactorys < 2)
		{
			unitAdd.push_back(int(1.5 * numType));
		}
		else if (numFactorys < 3)
		{
			unitAdd.push_back(2 * numType);
		}
		else if (numFactorys < 4)
		{
			unitAdd.push_back(3 * numType);
		}
		else
		{
			unitAdd.push_back(4 * numType);
		}
	}
	else
	{
		unitAdd.push_back(0);
	}
	if (shouldAddTank)
	{
		if (numFactorys < 2)
		{
			unitAdd.push_back(int(1.5 * numType));
		}
		else if (numFactorys < 3)
		{
			unitAdd.push_back(2 * numType);
		}
		else if (numFactorys < 4)
		{
			unitAdd.push_back(3 * numType);
		}
		else
		{
			unitAdd.push_back(4 * numType);
		}
	}
	else
	{
		unitAdd.push_back(0);
	}

	if (shouldAddGoliath)
	{
		if (numFactorys < 2)
		{
			unitAdd.push_back(int(1.5 * numType));
		}
		else if (numFactorys < 3)
		{
			unitAdd.push_back(2 * numType);
		}
		else if (numFactorys < 4)
		{
			unitAdd.push_back(3 * numType);
		}
		else
		{
			unitAdd.push_back(4 * numType);
		}
	}
	else
	{
		unitAdd.push_back(0);
	}

	return unitAdd;
}

//жϵзǷΪеλ
bool StrategyManager::enemyFlyArmyGroup()
{
	int numEnemyUnits = 0;
	int numEnemyFlyUnits = 0;

	for (BWAPI::Unit unit : BWAPI::Broodwar->enemy()->getUnits())
	{
		if (unit->getType().isWorker()){ continue; }

		if (unit->isFlying())
		{
			numEnemyFlyUnits += 1;
		}
		numEnemyUnits += 1;
	}
	//зķеλг40%жΪо
	if (numEnemyFlyUnits / (numEnemyUnits + 1) > 0.4)
	{
		return true;
	}
	return false;
}

bool StrategyManager::shouldBuildEBay()
{
	bool shouldBuildEBay = false;

	bool hasEBay = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Engineering_Bay) > 0;
	bool hasFactory = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) > 0;

	if (!hasEBay && hasFactory)
	{
		shouldBuildEBay = true;
	}
	return shouldBuildEBay;
}

bool StrategyManager::shouldBuildBunker()
{
	bool shouldBuildBunker = false;
	int numTanks = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode)
		+ UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Siege_Mode);

	BWAPI::Position defenseLine;
	if (_currentModule == "C1")
	{
		defenseLine = CombatCommander::Instance().getDefenseLine();//ǰߵλ
	}
	else
	{
		defenseLine = InformationManager::Instance().getPosition("frontLine3");
	}

	//ر޽Ŀص
	BWAPI::Position chockPoint2;

	chockPoint2 = InformationManager::Instance().getPosition("chokePoint2");
	
	BWAPI::Position defenseLine1 = 
		CombatCommander::Instance().getNearPosition(InformationManager::Instance().getPosition("frontLine1"), 
		InformationManager::Instance().getPosition("frontLine2"), false, 60);
	BWAPI::Position defenseLine2 = 
		CombatCommander::Instance().getNearPosition(InformationManager::Instance().getPosition("frontLine2"),
		InformationManager::Instance().getPosition("frontLine3"), false, 60);
	BWAPI::Position defenseLine3;
	if (Bases::Instance().enemyStart() &&
		Bases::Instance().getSecondBase(1))
	{
		defenseLine3 = CombatCommander::Instance().getNearPosition(InformationManager::Instance().getPosition("frontLine3"),
			Bases::Instance().getSecondBase(1)->getPosition(), false, 60);
	}
	else
	{
		defenseLine3 = InformationManager::Instance().getPosition("frontLine3");
	}

	BWAPI::Position basePosition = Bases::Instance().myStartingBase()->getPosition();

	bool            baseHasBunker = false;
	bool            chokePoint2HasBunker = false;
	bool            line1HasBunker = false;
	bool            line2HasBunker = false;
	bool            line3HasBunker = false;

	//жϼĿλǷѾ޽˵ر
	for (const auto unit : BWAPI::Broodwar->self()->getUnits())
	{
		if (unit->getType() != BWAPI::UnitTypes::Terran_Bunker){ continue; }
		if (unit->getDistance(basePosition) < 400)
		{
			baseHasBunker = true;
		}
		if (unit->getDistance(chockPoint2) < 400)
		{
			chokePoint2HasBunker = true;
		}
		if (unit->getDistance(defenseLine1) < 600)
		{
			line1HasBunker = true;
		}
		if (unit->getDistance(defenseLine2) < 600)
		{
			line2HasBunker = true;
		}
		if (unit->getDistance(defenseLine3) < 600)
		{
			line3HasBunker = true;
		}
	}

	//жû޽رλǷҪ޽ر
	//
	if (numTanks >= 2 && chokePoint2HasBunker && !baseHasBunker &&
		Config::Strategy::StrategyName == "AntiRush")
	{
		shouldBuildBunker = true;
		BuildingPlacer::Instance().setBunkerBuildPlace(basePosition);
	}
	//·2
	else if (numTanks >= 2 && !chokePoint2HasBunker)
	{
		shouldBuildBunker = true;
		BWAPI::Position POS = CombatCommander::Instance().getNearPosition(chockPoint2, defenseLine1, true, 60);//ر΢޽һ
		BuildingPlacer::Instance().setBunkerBuildPlace(POS);
	}
	//1
	else if (defenseLine == InformationManager::Instance().getPosition("frontLine1") && !line1HasBunker)
	{
		shouldBuildBunker = true;
		BuildingPlacer::Instance().setBunkerBuildPlace(defenseLine1);
	}
	/*
	//2
	else if (defenseLine == InformationManager::Instance().getPosition("frontLine2") && !line2HasBunker)
	{
		shouldBuildBunker = true;
		BuildingPlacer::Instance().setBunkerBuildPlace(defenseLine2);
	}
	//3
	else if (defenseLine == InformationManager::Instance().getPosition("frontLine3") && !line3HasBunker)
	{
		shouldBuildBunker = true;
		BuildingPlacer::Instance().setBunkerBuildPlace(defenseLine3);
	}
	*/
	return shouldBuildBunker;
}

void StrategyManager::buildBaseTurret(BuildOrderQueue & queue)
{
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Command_Center) < 3){ return; }

	for (auto * base : Bases::Instance().getBases())
	{
		if (base->getOwner() != BWAPI::Broodwar->self()){ continue; }
		if (base == Bases::Instance().myStartingBase()){ continue; }
		if (base == Bases::Instance().getSecondBase(0)){ continue; }

		BWAPI::Unitset turrets = BWAPI::Broodwar->getUnitsInRadius(
			base->getPosition(), 360,
			BWAPI::Filter::IsOwned && BWAPI::Filter::GetType == BWAPI::UnitTypes::Terran_Missile_Turret);

		if (turrets.size() > 2){ continue; }

		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Missile_Turret) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Missile_Turret))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Missile_Turret));
			BuildingPlacer::Instance().setTurretBuildPlace(base->getPosition());
		}
	}
}

void StrategyManager::buildTurret(BuildOrderQueue & queue, int aimNumOfBaseTurret, int numChoke1, int numChoke2)
{
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Engineering_Bay) == 0){ return; }
	
	BWAPI::Position basePos = Bases::Instance().myStartingBase()->getPosition();
	BWAPI::Position choke1Pos = InformationManager::Instance().getPosition("barracksPoint");
	BWAPI::Position choke2Pos = InformationManager::Instance().getPosition("chokePoint2");

	int numBaseTurret = 0, numChoke1Turret = 0, numChoke2Turret = 0;
	//еĸصķ
	for (BWAPI::Unit unit : BWAPI::Broodwar->self()->getUnits())
	{
		if (unit->getType() != BWAPI::UnitTypes::Terran_Missile_Turret) { continue; }

		if (unit->getDistance(basePos) < 360){ numBaseTurret += 1; }
	    if (unit->getDistance(choke1Pos) < 200){ numChoke1Turret += 1; }
		if (unit->getDistance(choke2Pos) < 260){ numChoke2Turret += 1; }
	}

	//·ڶرǷ޽
	BWAPI::Unit choke2Bunker = BWAPI::Broodwar->getClosestUnit(
		InformationManager::Instance().getPosition("chokePoint2"),
		BWAPI::Filter::IsOwned && BWAPI::Filter::GetType == BWAPI::UnitTypes::Terran_Bunker,
		260);

	if (numChoke2Turret < numChoke2 &&
		UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Command_Center) > 1)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Missile_Turret) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Missile_Turret))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Missile_Turret));
			BuildingPlacer::Instance().setTurretBuildPlace(choke2Pos);
		}
	}
	else if (aimNumOfBaseTurret > numBaseTurret)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Missile_Turret) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Missile_Turret))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Missile_Turret));
			BuildingPlacer::Instance().setTurretBuildPlace(basePos);
		}
	}
	else if (numChoke1Turret < numChoke1)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Missile_Turret) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Missile_Turret))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Missile_Turret));
			//ѡȡλ
			std::vector<BWAPI::TilePosition> tiles = BWTA::getShortestPath(BWAPI::TilePosition(InformationManager::Instance().getPosition("barracksPoint")),
				BWAPI::TilePosition(Bases::Instance().myStartingBase()->getPosition()));

			if (!tiles.empty() && tiles.size() > 4)
			{
				BuildingPlacer::Instance().setTurretBuildPlace(BWAPI::Position(tiles[3]));
			}
			else
			{
				BWAPI::Position choke1 = BWTA::getNearestChokepoint(basePos)->getCenter();
				BWAPI::Position aimPos = CombatCommander::Instance().getNearPosition(InformationManager::Instance().getPosition("barracksPoint"),
					choke1, true, 60);
				BuildingPlacer::Instance().setTurretBuildPlace(aimPos);
			}
		}
	}

	/*
	BWAPI::Unit baseTurret = BWAPI::Broodwar->getClosestUnit(
		Bases::Instance().myStartingBase()->getPosition(),
		BWAPI::Filter::IsOwned && BWAPI::Filter::GetType == BWAPI::UnitTypes::Terran_Missile_Turret,
		300);
	std::vector<BWAPI::TilePosition> tiles = BWTA::getShortestPath(BWAPI::TilePosition(InformationManager::Instance().getPosition("barracksPoint")),
		BWAPI::TilePosition(Bases::Instance().myStartingBase()->getPosition()));

	if (tiles.empty() || tiles.size() < 6){ return; }

	BWAPI::Unit choke1Turret = BWAPI::Broodwar->getClosestUnit(
		BWAPI::Position(tiles[3]),
		BWAPI::Filter::IsOwned && BWAPI::Filter::GetType == BWAPI::UnitTypes::Terran_Missile_Turret,
		160);
	BWAPI::Unit choke2Turret = BWAPI::Broodwar->getClosestUnit(
		InformationManager::Instance().getPosition("chokePoint2"),
		BWAPI::Filter::IsOwned && BWAPI::Filter::GetType == BWAPI::UnitTypes::Terran_Missile_Turret,
		120);
	BWAPI::Unit choke2Bunker = BWAPI::Broodwar->getClosestUnit(
		InformationManager::Instance().getPosition("chokePoint2"),
		BWAPI::Filter::IsOwned && BWAPI::Filter::GetType == BWAPI::UnitTypes::Terran_Bunker,
		120);
	if (BuildingPlacer::Instance().getBaseTurretPosition() != BWAPI::Positions::None)//choke12
	{
		BuildingPlacer::Instance().allowBuildTuttet = true;

		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Missile_Turret) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Missile_Turret))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Missile_Turret));
			BuildingPlacer::Instance().setTurretBuildPlace(BuildingPlacer::Instance().getBaseTurretPosition());
		}
	}
	else if (choke2Bunker && !choke2Turret &&
		UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Command_Center) > 1)
	{
		BuildingPlacer::Instance().allowBuildTuttet = true;

		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Missile_Turret) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Missile_Turret))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Missile_Turret));
			BuildingPlacer::Instance().setTurretBuildPlace(InformationManager::Instance().getPosition("chokePoint2"));
		}
	}
	else if (!choke1Turret)
	{
		BuildingPlacer::Instance().allowBuildTuttet = true;

		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Missile_Turret) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Missile_Turret))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Missile_Turret));

			BuildingPlacer::Instance().setTurretBuildPlace(BWAPI::Position(tiles[3]));
		}
	}
	*/

}

bool StrategyManager::shouldBuildTurret()
{
	bool shouldBuildTurret = false;
	int numTanks = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode)
		+ UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Siege_Mode);
	int numMinerals = BWAPI::Broodwar->self()->minerals();

	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Engineering_Bay) == 0) { return shouldBuildTurret; }

	BWAPI::Position defenseLine;
	if (_currentModule == "C1")
	{
		defenseLine = CombatCommander::Instance().getDefenseLine();//ǰߵλ
	}
	else
	{
		defenseLine = InformationManager::Instance().getPosition("frontLine3");
	}

	//޽Ŀص
	BWAPI::Position chockPoint2 = InformationManager::Instance().getPosition("chokePoint2");
	BWAPI::Position defenseLine1 = InformationManager::Instance().getPosition("frontLine1");
	BWAPI::Position defenseLine2 = InformationManager::Instance().getPosition("frontLine2");
	BWAPI::Position defenseLine3 = InformationManager::Instance().getPosition("frontLine3");
	BWAPI::Position basePosition = Bases::Instance().myStartingBase()->getPosition();

	int            numBaseTurret = 0;
	int            numChokePoint2Turret = 0;
	int            numLine1Turret = 0;
	int            numLine2Turret = 0;
	int            numLine3Turret = 0;

	//жϼĿλѾ޽ķ
	for (const auto unit : BWAPI::Broodwar->self()->getUnits())
	{
		if (unit->getType() != BWAPI::UnitTypes::Terran_Missile_Turret){ continue; }
		if (unit->getDistance(basePosition) < 600)
		{
			numBaseTurret += 1;
		}
		if (unit->getDistance(chockPoint2) < 400)
		{
			numChokePoint2Turret += 1;
		}
		if (unit->getDistance(defenseLine1) < 400)
		{
			numLine1Turret += 1;
		}
		if (unit->getDistance(defenseLine2) < 400)
		{
			numLine2Turret += 1;
		}
		if (unit->getDistance(defenseLine3) < 400)
		{
			numLine3Turret += 1;
		}
	}

	BWAPI::Unit choke1Turret = BWAPI::Broodwar->getClosestUnit(
		InformationManager::Instance().getPosition("barracksPoint"),
		BWAPI::Filter::IsOwned && BWAPI::Filter::GetType == BWAPI::UnitTypes::Terran_Missile_Turret,
		160);

	//3
	if (numBaseTurret < 3)
	{
		shouldBuildTurret = true;
		BuildingPlacer::Instance().setTurretBuildPlace(basePosition);
	}
	//·2
	else if (defenseLine == chockPoint2 && numChokePoint2Turret < 2)
	{
		shouldBuildTurret = true;
		BuildingPlacer::Instance().setTurretBuildPlace(chockPoint2);
	}
	else if (!choke1Turret)
	{
		shouldBuildTurret = true;

		std::vector<BWAPI::TilePosition> tiles = BWTA::getShortestPath(BWAPI::TilePosition(InformationManager::Instance().getPosition("barracksPoint")),
			BWAPI::TilePosition(Bases::Instance().myStartingBase()->getPosition()));

		if (!tiles.empty() && tiles.size() > 5)
		{
			BuildingPlacer::Instance().setTurretBuildPlace(BWAPI::Position(tiles[3]));
		}
		else
		{
			BWAPI::Position choke1 = BWTA::getNearestChokepoint(basePosition)->getCenter();
			BWAPI::Position aimPos = CombatCommander::Instance().getNearPosition(InformationManager::Instance().getPosition("barracksPoint"),
				choke1, true, 60);
			BuildingPlacer::Instance().setTurretBuildPlace(aimPos);
		}
	}
	//1
	else if (defenseLine == InformationManager::Instance().getPosition("frontLine1") && numLine1Turret < 3)
	{
		shouldBuildTurret = true;
		if (!BWAPI::Broodwar->getClosestUnit(
			CombatCommander::Instance().getSquadPosition("51"),
			BWAPI::Filter::IsOwned && BWAPI::Filter::GetType == BWAPI::UnitTypes::Terran_Missile_Turret,
			160))
		{
			BuildingPlacer::Instance().setTurretBuildPlace(CombatCommander::Instance().getSquadPosition("51"));
		}
		else if (!BWAPI::Broodwar->getClosestUnit(
			CombatCommander::Instance().getSquadPosition("52"),
			BWAPI::Filter::IsOwned && BWAPI::Filter::GetType == BWAPI::UnitTypes::Terran_Missile_Turret,
			160))
		{
			BuildingPlacer::Instance().setTurretBuildPlace(CombatCommander::Instance().getSquadPosition("52"));
		}
		else
		{
			BuildingPlacer::Instance().setTurretBuildPlace(defenseLine1);
		}
	}
	//2
	else if (defenseLine == InformationManager::Instance().getPosition("frontLine2") && numLine2Turret < 3)
	{
		shouldBuildTurret = true;
		if (!BWAPI::Broodwar->getClosestUnit(
			CombatCommander::Instance().getSquadPosition("51"),
			BWAPI::Filter::IsOwned && BWAPI::Filter::GetType == BWAPI::UnitTypes::Terran_Missile_Turret,
			160))
		{
			BuildingPlacer::Instance().setTurretBuildPlace(CombatCommander::Instance().getSquadPosition("51"));
		}
		else if (!BWAPI::Broodwar->getClosestUnit(
			CombatCommander::Instance().getSquadPosition("52"),
			BWAPI::Filter::IsOwned && BWAPI::Filter::GetType == BWAPI::UnitTypes::Terran_Missile_Turret,
			160))
		{
			BuildingPlacer::Instance().setTurretBuildPlace(CombatCommander::Instance().getSquadPosition("52"));
		}
		else
		{
			BuildingPlacer::Instance().setTurretBuildPlace(defenseLine2);
		}
	}
	//3
	else if (defenseLine == InformationManager::Instance().getPosition("frontLine3") && numLine3Turret < 3)
	{
		shouldBuildTurret = true;
		if (!BWAPI::Broodwar->getClosestUnit(
			CombatCommander::Instance().getSquadPosition("51"),
			BWAPI::Filter::IsOwned && BWAPI::Filter::GetType == BWAPI::UnitTypes::Terran_Missile_Turret,
			160))
		{
			BuildingPlacer::Instance().setTurretBuildPlace(CombatCommander::Instance().getSquadPosition("51"));
		}
		else if (!BWAPI::Broodwar->getClosestUnit(
			CombatCommander::Instance().getSquadPosition("52"),
			BWAPI::Filter::IsOwned && BWAPI::Filter::GetType == BWAPI::UnitTypes::Terran_Missile_Turret,
			160))
		{
			BuildingPlacer::Instance().setTurretBuildPlace(CombatCommander::Instance().getSquadPosition("52"));
		}
		else
		{
			BuildingPlacer::Instance().setTurretBuildPlace(defenseLine3);
		}
	}
	return shouldBuildTurret;
}

bool StrategyManager::shouldBuildCenter()
{
	bool shouldBuildCenter = false;
	//Ϸʱ
	int minute = BWAPI::Broodwar->getFrameCount() / (24 * 60);
	
	//2ֱӽ»
	if (UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Command_Center) < 2)
	{
		shouldBuildCenter = true;
	}
	else if (CombatCommander::Instance().getDefenseLine() == InformationManager::Instance().getPosition("frontLine1"))
	{
		if (UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Command_Center) < 3 &&
			BWAPI::Broodwar->self()->minerals() > 400)
		{
			shouldBuildCenter = true;
		}
		else if (UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Command_Center) < 4 &&
			BWAPI::Broodwar->self()->minerals() > 1000)
		{
			shouldBuildCenter = true;
		}
	}
	else if (CombatCommander::Instance().getDefenseLine() != InformationManager::Instance().getPosition("chokePoint2"))
	{
		if (UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Command_Center) < 3 &&
			BWAPI::Broodwar->self()->minerals() > 400)
		{
			shouldBuildCenter = true;
		}
		else if (UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Command_Center) < 4 && 
			(BWAPI::Broodwar->self()->minerals() > 800 || WorkerManager::Instance().getNumIdleWorkers() > 5))
		{
			shouldBuildCenter = true;
		}
		else if (UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Command_Center) < 5 &&
			BWAPI::Broodwar->self()->minerals() > 1200)
		{
			shouldBuildCenter = true;
		}
		else if (BWAPI::Broodwar->self()->minerals() > 1600)
		{
			shouldBuildCenter = true;
		}
	}

	//ʣ㣬ֻ
	if (!shouldBuildCenter)
	{
		int remainingMinerals = 0;
		for (auto * base : Bases::Instance().getBases())
		{
			if (base->getOwner() != BWAPI::Broodwar->self()){ continue; }

			remainingMinerals += int(base->getMinerals().size());
		}
		if (remainingMinerals < 8)
		{
			shouldBuildCenter = true;
		}
	}

	return shouldBuildCenter;
}

bool StrategyManager::shouldBuildFactory()
{
	bool shouldBuildFactory = false;
	//˿ʹ
	int supplyTotal = BWAPI::Broodwar->self()->supplyTotal();
	int supplyUsed = BWAPI::Broodwar->self()->supplyUsed();

	int numFactory = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Factory);
	int idealFactoryNum = (supplyUsed - 50) / 20 + 3;

	if (numFactory < 3)
	{
		shouldBuildFactory = true;
	}
	else if (idealFactoryNum > numFactory)
	{
		shouldBuildFactory = true;
	}
	return shouldBuildFactory;
}

void StrategyManager::executeStrategy(BuildOrderQueue & queue)
{
	//ͬԼл
	shouldSwitchModule();

	if (BWAPI::Broodwar->getFrameCount() / 24 < 10)
	{
		_buildSecondBase = false;
		_eBayIsLifted = false;
		if (Config::Strategy::StrategyName == "A3-B5-C1")
		{
			_currentModule = "A3";
		}
		else if (Config::Strategy::StrategyName == "A1-B1-B2-C2")
		{
			_currentModule = "A1";
		}
		else if (Config::Strategy::StrategyName == "A3-B7-C1")
		{
			_currentModule = "A3";
		}
		else if (Config::Strategy::StrategyName == "A4-B2-C1")
		{
			_currentModule = "A4";
		}
		else if (Config::Strategy::StrategyName == "A1-B3-C2")
		{
			_currentModule = "A1";
		}
	}

	if (BWAPI::Broodwar->getFrameCount() / (24 * 60) < 1)
	{
		//_currentModule = Config::Strategy::StrategyName;
		_startModuleC = false;
	}
	if (_currentModule == "A1")
	{
		updateModuleA1(queue);
	}
	else if (_currentModule == "A2")
	{
		updateModuleA2(queue);
	}
	else if (_currentModule == "A3")
	{
		updateModuleA3(queue);
	}
	else if (_currentModule == "A4")
	{
		updateModuleA4(queue);
	}	
	else if (_currentModule == "B1")
	{
		updateModuleB1(queue);
	}
	else if (_currentModule == "B2")
	{
		updateModuleB2(queue);
	}
	else if (_currentModule == "B3")
	{
		updateModuleB3(queue);
	}
	else if (_currentModule == "B4")
	{
		updateModuleB4(queue);
	}
	else if (_currentModule == "B5")
	{
		updateModuleB5(queue);
	}
	else if (_currentModule == "B6")
	{
		updateModuleB6(queue);
	}
	else if (_currentModule == "B7")
	{
		updateModuleB7(queue);
	}
	else if (_currentModule == "C1" ||
		_currentModule == "C2")
	{
		updateModuleC(queue);
	}
	return;
}

void StrategyManager::updateAntiRush(BuildOrderQueue & queue)
{
	int numSCVs = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_SCV);
	int numMarines = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Marine);
	int numBunker = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Bunker);
	int numBarracks = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Barracks);
	int numSupplys = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Supply_Depot);
	int numFactorys = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Factory);
	int numTanks = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode)
		+ UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Siege_Mode);
	int numMinerals = BWAPI::Broodwar->self()->minerals();
	int numGas = BWAPI::Broodwar->self()->gas();
	BWAPI::Player self = BWAPI::Broodwar->self();
	bool hasEBay = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Engineering_Bay) > 0;
	int numTurrets = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Missile_Turret);
	int numGoliaths = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Goliath);
	int supplyTotal = BWAPI::Broodwar->self()->supplyTotal();
	int supplyUsed = BWAPI::Broodwar->self()->supplyUsed();


	//SCV
	if (numSCVs < 5 || numMarines > 5)
	{
		if (!queue.anyInNextN(BWAPI::UnitTypes::Terran_SCV, 2) &&
			!queue.anyInQueue(BWAPI::UnitTypes::Terran_Goliath) &&
			!queue.anyInQueue(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode))
		{
			queue.queueAsHighestPriority(BWAPI::UnitTypes::Terran_SCV);
		}
	}

	//Ӫ֮رǹ
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Barracks) >= 1)
	{
		if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Barracks) > 0 &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Marine) < 8 &&
			!queue.anyInNextN(BWAPI::UnitTypes::Terran_Marine, 2) &&
			(BWAPI::Broodwar->self()->minerals() >= 100 || UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_SCV) > 3))
		{
			queue.queueAsHighestPriority(BWAPI::UnitTypes::Terran_Marine);
		}
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Bunker) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Bunker) &&
			numBunker < 1)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Bunker));
			BuildingPlacer::Instance().setBunkerBuildPlace(InformationManager::Instance().getPosition("bunkerPoint"));
		}
	}

	//ˣˣǿ
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) > 0 &&
		UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Armory) > 0 &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Goliath) &&
		numMinerals >= 100 && numGas >= 50)
	{
		queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Goliath));
	}

	//ֻصĵر
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) >= 1 && numGoliaths > 6)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Bunker) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Bunker) &&
			numBunker < 2 && numMinerals > 100)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Bunker));
			BuildingPlacer::Instance().setBunkerBuildPlace(InformationManager::Instance().getPosition("chokePoint2"));
		}
	}

	//ֻ
	BWAPI::Unit choke2Bunker = BWAPI::Broodwar->getClosestUnit(
		InformationManager::Instance().getPosition("chokePoint2"),
		BWAPI::Filter::IsOwned && BWAPI::Filter::GetType == BWAPI::UnitTypes::Terran_Bunker,
		6 * 32);
	if (choke2Bunker && numMarines >= 5)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Command_Center) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Command_Center) &&
			numMinerals > 400)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Command_Center));
		}
	}

	//;е⽨ú̹
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) > 0 &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode) &&
		numMinerals >= 150 && numGas >= 100 &&
		numTanks < 1 && numGoliaths >= 5)
	{
		queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode));
	}

	//5SCVӪ
	if (numSCVs >= 5 && numBarracks == 0)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Barracks) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Barracks) &&
			numMinerals > 140)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Barracks));
		}
	}

	//˿ڵ8ʱʼ޵һ
	if (numSCVs >= 5 && numMarines >= 4)
	{
		BuildingPlacer::Instance().allowBuildSupply = true;
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Supply_Depot) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Supply_Depot) && 
			numSupplys == 0 && numMinerals > 90)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Supply_Depot));
		}
	}

	//
	if (numMarines >= 8)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Engineering_Bay) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Engineering_Bay) &&
			!hasEBay && numMinerals > 125)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Engineering_Bay));
		}
	}
	//޺̽죬
	if (hasEBay && numTurrets < 5)
	{
		BuildingPlacer::Instance().allowBuildTuttet = true;

		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Missile_Turret) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Missile_Turret))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Missile_Turret));
			BuildingPlacer::Instance().setTurretBuildPlace(Bases::Instance().myStartingBase()->getPosition());
		}
	}
	else if (hasEBay && numTurrets >= 4 && numTurrets < 7)//choke13
	{
		BuildingPlacer::Instance().allowBuildTuttet = true;

		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Missile_Turret) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Missile_Turret))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Missile_Turret));
			BuildingPlacer::Instance().setTurretBuildPlace(InformationManager::Instance().getPosition("chokePoint1"));
		}
	}

	//Ͳ
	if (numMarines >= 6 && numTurrets >= 2)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Factory) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Factory) && 
			numFactorys == 0 &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Refinery) >= 1)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Factory));
		}
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Refinery) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Refinery) && 
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Refinery) < 1)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Refinery));
		}
	}

	//ڶ
	if (numGoliaths > 1)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Factory) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Factory) &&
			numFactorys < 2 &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Refinery) >= 1)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Factory));
		}
	}

	//
	if (numGoliaths > 2)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Factory) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Factory) &&
			numFactorys < 3 &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Refinery) >= 1)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Factory));
		}
	}

	//˵ǰ
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) >= 1)
	{
		if (queue.numInQueue(BWAPI::UnitTypes::Terran_Armory) < 1 &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Armory) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Armory) < 1)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Armory));
		}
	}

	//е
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) >= 1)
	{
		if (queue.numInQueue(BWAPI::UnitTypes::Terran_Machine_Shop) < 1 &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Machine_Shop) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) < 1)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Machine_Shop));
		}
	}
	//̹˵ļ
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) >= 1 &&
		!queue.anyInNextN(BWAPI::TechTypes::Tank_Siege_Mode, 15) &&
		!self->hasResearched(BWAPI::TechTypes::Tank_Siege_Mode) &&
		!self->isResearching(BWAPI::TechTypes::Tank_Siege_Mode) && 
		numGoliaths > 4)
	{
		queue.queueAsHighestPriority(MacroAct(BWAPI::TechTypes::Tank_Siege_Mode));
	}

	//еӵķ
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Armory) >= 1 &&
		self->getUpgradeLevel(BWAPI::UpgradeTypes::Terran_Vehicle_Plating) == 0 &&
		!self->isUpgrading(BWAPI::UpgradeTypes::Terran_Vehicle_Plating) &&
		numMinerals > 100 && numGas > 100)
	{
		queue.queueAsHighestPriority(MacroAct(BWAPI::UpgradeTypes::Terran_Vehicle_Plating));
	}

	//˵
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) >= 1 &&
		self->getUpgradeLevel(BWAPI::UpgradeTypes::Charon_Boosters) == 0 &&
		!self->isUpgrading(BWAPI::UpgradeTypes::Charon_Boosters) &&
		numMinerals > 100 && numGas > 100 &&
		numGoliaths > 3)
	{
		queue.queueAsHighestPriority(MacroAct(BWAPI::UpgradeTypes::Charon_Boosters));
	}

	//
	if ((supplyTotal - supplyUsed) < 5 && supplyTotal < 200 && numSupplys >= 1)
	{
		BuildingPlacer::Instance().allowBuildSupply = true;

		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Supply_Depot) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Supply_Depot))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Supply_Depot));
		}
	}
}

//6.29
void StrategyManager::updateDefault(BuildOrderQueue & queue)
{
	int numSCVs = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_SCV);
	int numMarines = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Marine);
	int numBunker = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Bunker);
	int numBarracks = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Barracks);
	int numSupplys = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Supply_Depot);
	int numFactorys = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Factory);
	int numTanks = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode)
		+ UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Siege_Mode);
	int numMinerals = BWAPI::Broodwar->self()->minerals();
	int numGas = BWAPI::Broodwar->self()->gas();
	BWAPI::Player self = BWAPI::Broodwar->self();

	//8SCVһ
	if (numSCVs >= 8)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Supply_Depot) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Supply_Depot) && numSupplys == 0)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Supply_Depot));
		}
	}
	//SCVͣ
	if (!queue.anyInNextN(BWAPI::UnitTypes::Terran_SCV, 2) && numSCVs < 30 && 
		!queue.anyInNextN(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode, 2))
	{
		queue.queueAsHighestPriority(BWAPI::UnitTypes::Terran_SCV);
	}
	//10SCVӪ
	if (numSCVs >= 10 && numBarracks == 0)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Barracks) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Barracks))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Barracks));
		}
	}
	//11SCV
	if (numSCVs >= 11)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Refinery) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Refinery) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Refinery) < 1)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Refinery));
		}
	}
	//Ӫ8ǹر
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Barracks) >= 1)
	{
		if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Barracks) > 0 &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Marine) < 8 &&
			!queue.anyInNextN(BWAPI::UnitTypes::Terran_Marine, 2) &&
			(BWAPI::Broodwar->self()->minerals() >= 100 || UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_SCV) > 3))
		{
			queue.queueAsHighestPriority(BWAPI::UnitTypes::Terran_Marine);
		}
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Bunker) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Bunker) &&
			numBunker < 1)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Bunker));
			BuildingPlacer::Instance().setBunkerBuildPlace(InformationManager::Instance().getPosition("bunkerPoint"));
		}
	}
	//ڶ
	if (numSCVs + numMarines >= 16)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Supply_Depot) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Supply_Depot) && numSupplys <= 1)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Supply_Depot));
		}
	}
	//
	if (numSCVs + numMarines >= 15)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Factory) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Factory) &&
			numFactorys == 0 &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Refinery) >= 1)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Factory));
		}
	}
	//
	if (numSCVs + numMarines >= 24)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Supply_Depot) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Supply_Depot) && numSupplys <= 2)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Supply_Depot));
		}
	}
	//е
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) >= 1)
	{
		if (queue.numInQueue(BWAPI::UnitTypes::Terran_Machine_Shop) < 1 &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Machine_Shop) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) < 1)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Machine_Shop));
		}
	}
	//̹˼
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) >= 1 &&
		!queue.anyInNextN(BWAPI::TechTypes::Tank_Siege_Mode, 15) &&
		!self->hasResearched(BWAPI::TechTypes::Tank_Siege_Mode) &&
		!self->isResearching(BWAPI::TechTypes::Tank_Siege_Mode))
	{
		queue.queueAsHighestPriority(MacroAct(BWAPI::TechTypes::Tank_Siege_Mode));
	}
	//̹
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) > 0 &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode) &&
		numMinerals >= 150 && numGas >= 100)
	{
		queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode));
	}
}

//7.1
void StrategyManager::update8BB(BuildOrderQueue & queue)
{
	int numSCVs = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_SCV);
	int numMarines = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Marine);
	int numBunker = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Bunker);
	int numBarracks = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Barracks);
	int numSupplys = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Supply_Depot);
	int numFactorys = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Factory);
	int numTanks = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode)
		+ UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Siege_Mode);
	int numMinerals = BWAPI::Broodwar->self()->minerals();
	int numGas = BWAPI::Broodwar->self()->gas();
	BWAPI::Player self = BWAPI::Broodwar->self();
	int supplyTotal = BWAPI::Broodwar->self()->supplyTotal();
	int supplyUsed = BWAPI::Broodwar->self()->supplyUsed();

	//9SCVһ
	if (numSCVs >= 9)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Supply_Depot) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Supply_Depot) && numSupplys == 0)
		{
			BuildingPlacer::Instance().allowBuildSupply = true;
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Supply_Depot));
		}
	}
	//SCVͣ
	if (!queue.anyInNextN(BWAPI::UnitTypes::Terran_SCV, 2) && numSCVs < 30 &&
		!queue.anyInNextN(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode, 2) &&
		!queue.anyInNextN(BWAPI::UnitTypes::Terran_Bunker, 2) &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Barracks))
	{
		queue.queueAsHighestPriority(BWAPI::UnitTypes::Terran_SCV);
	}
	//8SCVӪ
	if (numSCVs >= 8 && numBarracks == 0)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Barracks) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Barracks))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Barracks));
		}
	}
	//11SCV
	if (numSCVs >= 11)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Refinery) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Refinery) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Refinery) < 1 &&
			numMinerals > 100)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Refinery));
		}
	}
	//Ӫôﵽ14˿ʱ6ǹر
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Barracks) >= 1)
	{
		if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Barracks) > 0 &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Marine) < 6 &&
			!queue.anyInNextN(BWAPI::UnitTypes::Terran_Marine, 2))
		{
			queue.queueAsHighestPriority(BWAPI::UnitTypes::Terran_Marine);
		}
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Bunker) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Bunker) &&
			numBunker < 1 &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Marine) > 0)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Bunker));
			BuildingPlacer::Instance().setBunkerBuildPlace(InformationManager::Instance().getPosition("bunkerPoint"));
		}
	}
	//ֻصĵر
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) >= 1 && numMarines >= 3)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Bunker) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Bunker) &&
			numBunker < 2 && numMinerals > 100)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Bunker));
			BuildingPlacer::Instance().setBunkerBuildPlace(InformationManager::Instance().getPosition("chokePoint2"));
		}
	}

	//ֻ
	BWAPI::Unit choke2Bunker = BWAPI::Broodwar->getClosestUnit(
		InformationManager::Instance().getPosition("chokePoint2"),
		BWAPI::Filter::IsOwned && BWAPI::Filter::GetType == BWAPI::UnitTypes::Terran_Bunker,
		6 * 32);
	if (choke2Bunker && numMarines >= 3)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Command_Center) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Command_Center) &&
			numMinerals > 400)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Command_Center));
		}
	}
	//
	if (numSCVs + numMarines >= 15)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Factory) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Factory) &&
			numFactorys == 0 &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Refinery) >= 1 &&
			numMinerals > 200 && numGas > 100)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Factory));
		}
	}
	//е
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) >= 1)
	{
		if (queue.numInQueue(BWAPI::UnitTypes::Terran_Machine_Shop) < 1 &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Machine_Shop) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) < 1)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Machine_Shop));
		}
	}
	//̹˼
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) >= 1 &&
		!queue.anyInNextN(BWAPI::TechTypes::Tank_Siege_Mode, 15) &&
		!self->hasResearched(BWAPI::TechTypes::Tank_Siege_Mode) &&
		!self->isResearching(BWAPI::TechTypes::Tank_Siege_Mode))
	{
		queue.queueAsHighestPriority(MacroAct(BWAPI::TechTypes::Tank_Siege_Mode));
	}
	//̹
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) > 0 &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode) &&
		numMinerals >= 150 && numGas >= 100)
	{
		queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode));
	}
	
	//
	if ((supplyTotal - supplyUsed) < 5 && supplyTotal < 200 && numSupplys >= 1)
	{
		BuildingPlacer::Instance().allowBuildSupply = true;
		
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Supply_Depot) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Supply_Depot))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Supply_Depot));
		}		
	}	
}

//6.24
bool StrategyManager::openingFinished()
{
	return _startModuleC;
}
//////////////////////////////////////////////__6.22
//7.4.2019//ݵǰжǷҪеλĹͷ
bool StrategyManager::shouldUpgrade(BWAPI::UpgradeType type)
{
	bool shouldUpgtade = false;

	int numMinerals = BWAPI::Broodwar->self()->minerals();
	int numGas = BWAPI::Broodwar->self()->gas();
	BWAPI::Player self = BWAPI::Broodwar->self();
	int weaponsUps = self->getUpgradeLevel(BWAPI::UpgradeTypes::Terran_Vehicle_Weapons);
	int platUps = self->getUpgradeLevel(BWAPI::UpgradeTypes::Terran_Vehicle_Plating);
	int numFreeSpace = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Armory) -
		int(self->isUpgrading(BWAPI::UpgradeTypes::Terran_Vehicle_Weapons)) -
		int(self->isUpgrading(BWAPI::UpgradeTypes::Terran_Vehicle_Plating));
	int numSupplyUsed = BWAPI::Broodwar->self()->supplyUsed() / 2;

	bool hasScience = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Science_Facility) > 0;

	if (numFreeSpace <= 0){ return false; }

	if (type == BWAPI::UpgradeTypes::Terran_Vehicle_Weapons)
	{
		if (weaponsUps == 0 && !self->isUpgrading(BWAPI::UpgradeTypes::Terran_Vehicle_Weapons))
		{
			if (numMinerals > 100 && numGas > 100 && _numFreeFactory == 0)
			{
				shouldUpgtade = true;
			}
		}
		else if (weaponsUps == 1 && hasScience &&
			!self->isUpgrading(BWAPI::UpgradeTypes::Terran_Vehicle_Weapons))
		{
			if ((numMinerals > 175 && numGas > 175 && _numFreeFactory == 0) ||
				numSupplyUsed > 140)
			{
				shouldUpgtade = true;
			}
		}
		else if (weaponsUps < self->getMaxUpgradeLevel(BWAPI::UpgradeTypes::Terran_Vehicle_Weapons) &&
			hasScience &&
			!self->isUpgrading(BWAPI::UpgradeTypes::Terran_Vehicle_Weapons))
		{
			if ((numMinerals > 250 && numGas > 250) ||
				numSupplyUsed > 170)
			{
				shouldUpgtade = true;
			}
		}
	}
	else if (type == BWAPI::UpgradeTypes::Terran_Vehicle_Plating)
	{
		if (platUps == 0 && !self->isUpgrading(BWAPI::UpgradeTypes::Terran_Vehicle_Plating))
		{
			if (numMinerals > 100 && numGas > 100 && _numFreeFactory == 0)
			{
				shouldUpgtade = true;
			}
		}
		else if (platUps == 1 && hasScience &&
			!self->isUpgrading(BWAPI::UpgradeTypes::Terran_Vehicle_Plating))
		{
			if ((numMinerals > 300 && numGas > 300 && _numFreeFactory == 0) ||
				numSupplyUsed > 150)
			{
				shouldUpgtade = true;
			}
		}	
		else if (platUps < self->getMaxUpgradeLevel(BWAPI::UpgradeTypes::Terran_Vehicle_Plating) && 
			hasScience &&
			!self->isUpgrading(BWAPI::UpgradeTypes::Terran_Vehicle_Plating))
		{
			if ((numMinerals > 500 && numGas > 500) ||
				numSupplyUsed > 180)
			{
				shouldUpgtade = true;
			}
		}	
	}
	return shouldUpgtade;
}

//7.5//
void StrategyManager::updateCommonState(BuildOrderQueue & queue)
{
	int supplyTotal = BWAPI::Broodwar->self()->supplyTotal() / 2;
	int supplyUsed = BWAPI::Broodwar->self()->supplyUsed() / 2;
	int numSupplys = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Supply_Depot);
	int seconds = BWAPI::Broodwar->getFrameCount() / 24;

	if (UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Command_Center) < 1 &&
		UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Engineering_Bay) > 0)
	{
		BWAPI::Broodwar->leaveGame();
	}


	//
	if ((supplyTotal - supplyUsed) < 6 && supplyTotal < 30 && numSupplys >= 1)
	{
		BuildingPlacer::Instance().allowBuildSupply = true;

		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Supply_Depot) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Supply_Depot))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Supply_Depot));
		}
	}
	else if ((supplyTotal - supplyUsed) < 10 && supplyTotal < 52 && supplyTotal >= 30)
	{
		BuildingPlacer::Instance().allowBuildSupply = true;

		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Supply_Depot) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Supply_Depot))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Supply_Depot));
		}
	}
	else if ((supplyTotal - supplyUsed) < 10 && supplyTotal < 90 && supplyTotal >= 52)
	{
		BuildingPlacer::Instance().allowBuildSupply = true;

		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Supply_Depot) &&
			BuildingManager::Instance().numBeingBuilt(BWAPI::UnitTypes::Terran_Supply_Depot) < 2)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Supply_Depot));
		}
	}
	else if ((supplyTotal - supplyUsed) < 26 && supplyTotal < 150 && supplyTotal >= 90)
	{
		BuildingPlacer::Instance().allowBuildSupply = true;

		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Supply_Depot) &&
			BuildingManager::Instance().numBeingBuilt(BWAPI::UnitTypes::Terran_Supply_Depot) < 2)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Supply_Depot));
		}
	}
	else if ((supplyTotal - supplyUsed) < 36 && supplyTotal < 200 && supplyTotal >= 150)
	{
		BuildingPlacer::Instance().allowBuildSupply = true;

		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Supply_Depot) &&
			BuildingManager::Instance().numBeingBuilt(BWAPI::UnitTypes::Terran_Supply_Depot) < 3)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Supply_Depot));
		}
	}
	
	//еĹ
	int numFactorys = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory);
	//ӸĹλҪȥ
	int numMachineBeingBuilt = BuildingManager::Instance().numBeingBuilt(BWAPI::UnitTypes::Terran_Machine_Shop);
	//еĹ
	int numUncompletedUnits = UnitUtil::GetUncompletedUnitCount(BWAPI::UnitTypes::Terran_Vulture) +
		UnitUtil::GetUncompletedUnitCount(BWAPI::UnitTypes::Terran_Goliath) +
		UnitUtil::GetUncompletedUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode);
	//й
	_numFreeFactory = numFactorys - numMachineBeingBuilt - numUncompletedUnits;


	//ƼӵУͻῨסλʱѡն
	if (BWAPI::Broodwar->self()->getUpgradeLevel(BWAPI::UpgradeTypes::Terran_Vehicle_Weapons) == 
		BWAPI::Broodwar->self()->getMaxUpgradeLevel(BWAPI::UpgradeTypes::Terran_Vehicle_Weapons) &&
		queue.anyInQueue(BWAPI::UpgradeTypes::Terran_Vehicle_Weapons))
	{
		queue.clearAll();
	}
	else if (BWAPI::Broodwar->self()->getUpgradeLevel(BWAPI::UpgradeTypes::Terran_Vehicle_Plating) == 
		BWAPI::Broodwar->self()->getMaxUpgradeLevel(BWAPI::UpgradeTypes::Terran_Vehicle_Plating) &&
		queue.anyInQueue(BWAPI::UpgradeTypes::Terran_Vehicle_Plating))
	{
		queue.clearAll();
	}
	else if (queue.anyInQueue(BWAPI::UpgradeTypes::Terran_Vehicle_Weapons) &&
		BWAPI::Broodwar->self()->isUpgrading(BWAPI::UpgradeTypes::Terran_Vehicle_Weapons))
	{
		queue.clearAll();
	}
	else if (queue.anyInQueue(BWAPI::UpgradeTypes::Terran_Vehicle_Plating) &&
		BWAPI::Broodwar->self()->isUpgrading(BWAPI::UpgradeTypes::Terran_Vehicle_Plating))
	{
		queue.clearAll();
	}

	//𹤳
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Engineering_Bay) > 0 && 
		!_eBayIsLifted)
	{
		CombatCommander::Instance().liftAndMoveEbay();
	}
	//طɵӦλ
	CombatCommander::Instance().updateSecondBase();

	if (BWAPI::Broodwar->getFrameCount() % (24 * 3) == 1)
	{
		buildBaseTurret(queue);
	}

	//C1ԵSquad
	if (_currentModule == "C1" || _currentModule == "C2")
	{
		int frame16 = BWAPI::Broodwar->getFrameCount() % 16;
		if (frame16 == 1)
		{
			CombatCommander::Instance().updateStartingBaseDefenseSquad();
			CombatCommander::Instance().updateBunkerSquad();
			CombatCommander::Instance().updateChokeSquads();
			CombatCommander::Instance().updateTankSquads();
			CombatCommander::Instance().updateAttackSquads();
			CombatCommander::Instance().updateResidentBaseSquad();
		}
		CombatCommander::Instance().updatePioneerSquad();
		CombatCommander::Instance().updateGoliathSquad();
		CombatCommander::Instance().updateBaseAttackSquad();


		//ǰ
		if (frame16 == 6)
		{
			if (enemyUnitsFrontPosition(InformationManager::Instance().getPosition("chokePoint2"), InformationManager::Instance().getPosition("frontLine1"), false) ||
				CombatCommander::Instance().getDefenseLine() == InformationManager::Instance().getPosition("chokePoint2"))
			{
				BWAPI::Position aimPos = CombatCommander::Instance().getNearPosition(InformationManager::Instance().getPosition("chokePoint2"),
					InformationManager::Instance().getPosition("frontLine1"), false, 300);
				CombatCommander::Instance().updateMineAsRowFrontSquad(aimPos, InformationManager::Instance().getPosition("chokePoint2"), 60, 200, 0);
			}
			else if (CombatCommander::Instance().getDefenseLine() == InformationManager::Instance().getPosition("frontLine1"))
			{
				BWAPI::Position aimPos = CombatCommander::Instance().getNearPosition(InformationManager::Instance().getPosition("frontLine1"),
					InformationManager::Instance().getPosition("frontLine2"), false, 260);
				CombatCommander::Instance().updateMineAsRowFrontSquad(aimPos, InformationManager::Instance().getPosition("chokePoint2"), 60, 200, 1);
			}
			else if (CombatCommander::Instance().getDefenseLine() == InformationManager::Instance().getPosition("frontLine2"))
			{
				BWAPI::Position aimPos = CombatCommander::Instance().getNearPosition(InformationManager::Instance().getPosition("frontLine2"),
					InformationManager::Instance().getPosition("frontLine3"), false, 260);
				CombatCommander::Instance().updateMineAsRowFrontSquad(aimPos, InformationManager::Instance().getPosition("frontLine1"), 60, 200, 1);
			}
			else if (CombatCommander::Instance().getDefenseLine() == InformationManager::Instance().getPosition("frontLine3") &&
				Bases::Instance().getSecondBase(1))
			{
				BWAPI::Position aimPos = CombatCommander::Instance().getNearPosition(InformationManager::Instance().getPosition("frontLine3"),
					Bases::Instance().getSecondBase(1)->getPosition(), false, 260);
				CombatCommander::Instance().updateMineAsRowFrontSquad(aimPos, InformationManager::Instance().getPosition("frontLine2"), 60, 100, 1);
			}
			else
			{
				CombatCommander::Instance().clearSquad("mineRowFrontSquad");
			}
		}
		if (frame16 == 11)
		{
			//ߺ
			if (CombatCommander::Instance().getDefenseLine() == InformationManager::Instance().getPosition("frontLine2"))
			{
				BWAPI::Position aimPos = CombatCommander::Instance().getNearPosition(InformationManager::Instance().getPosition("frontLine2"),
					InformationManager::Instance().getPosition("frontLine1"), false, 100);
				CombatCommander::Instance().updateMineAsRowBackSquad(aimPos, InformationManager::Instance().getPosition("frontLine1"), 100, 400);
			}
			else if (CombatCommander::Instance().getDefenseLine() == InformationManager::Instance().getPosition("frontLine3") &&
				!CombatCommander::Instance().getAggression())
			{
				BWAPI::Position aimPos = CombatCommander::Instance().getNearPosition(InformationManager::Instance().getPosition("frontLine3"),
					InformationManager::Instance().getPosition("frontLine2"), false, 100);
				CombatCommander::Instance().updateMineAsRowBackSquad(aimPos, InformationManager::Instance().getPosition("frontLine2"), 100, 400);
			}
			else
			{
				CombatCommander::Instance().clearSquad("mineRowBackSquad");
			}
		}
	}
}

////////////7.17//ģ黯

//¼ģı仯,Ļ
void StrategyManager::drawModuleInformation()
{
	//ģб仯¼ڱ_previousModule
	if (_previousModule.back() != _currentModule)
	{
		_previousModule.push_back(_currentModule);
	}
	//Ϸа˳֮ǰʹùĲģ
	int x = 300;
	int y = 100;
	for (int i = 0; i < _previousModule.size(); i++)
	{
		std::string module = _previousModule[i];
		BWAPI::Broodwar->drawTextScreen(x, y, module.c_str());
		y += 10;
	}
}

//A1//AntiRush//5SCV//
void StrategyManager::updateModuleA1(BuildOrderQueue & queue)
{
	int numSCVs = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_SCV);
	int numBarracks = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Barracks);
	int numMarines = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Marine);
	int numBunker = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Bunker);
	int numMinerals = BWAPI::Broodwar->self()->minerals();
	int numGas = BWAPI::Broodwar->self()->gas();
	int seconds = BWAPI::Broodwar->getFrameCount() / 24;
	int numSupplys = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Supply_Depot);
	bool hasEBay = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Engineering_Bay) > 0;
	int numTurrets = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Missile_Turret);

	//SCVʧ5
	if (numSCVs < 5 ||
		(numSCVs < 7 && numBarracks > 0 && !UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Barracks)))
	{
		if (!queue.anyInNextN(BWAPI::UnitTypes::Terran_SCV, 2))
		{
			queue.queueAsHighestPriority(BWAPI::UnitTypes::Terran_SCV);
		}
	}
	//5SCVʱֹͣSCVǮ޽Ӫ
	if (numSCVs >= 5 && numBarracks == 0)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Barracks) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Barracks) &&
			numMinerals > 150)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Barracks));
		}
	}
	//ǹ͵ر
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Barracks) >= 1)
	{
		if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Barracks) > 0 &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Marine) < 5 &&
			!queue.anyInNextN(BWAPI::UnitTypes::Terran_Marine, 2) &&
			(BWAPI::Broodwar->self()->minerals() >= 100 || UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_SCV) > 3))
		{
			queue.queueAsHighestPriority(BWAPI::UnitTypes::Terran_Marine);
		}
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Bunker) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Bunker) &&
			numBunker < 1)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Bunker));
			BuildingPlacer::Instance().setBunkerBuildPlace(InformationManager::Instance().getPosition("bunkerPoint"));
		}
	}
	//һ
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_SCV) + 
		UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Marine) >= 8)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Supply_Depot) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Supply_Depot) && numSupplys == 0)
		{
			BuildingPlacer::Instance().allowBuildSupply = true;
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Supply_Depot));
		}
	}

	////
	if (numMarines > 2 && numSupplys > 0)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Engineering_Bay) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Engineering_Bay) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Engineering_Bay) == 0 &&
			numMinerals > 125)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Engineering_Bay));
		}
	}

	//𣬻ǹɶУchoke1
	CombatCommander::Instance().updateBunkerSquad();

	//жA1ģǷﵽA1ѧϰģ߳һģ
	if ((numMarines > 4 && numBunker > 0 && UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Engineering_Bay) > 0) || seconds > 240)
	{
		doSwitchModule();
	}
}
//A2//8SCV޹ر·//
void StrategyManager::updateModuleA2(BuildOrderQueue & queue)
{
	int numSCVs = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_SCV);
	int numMarines = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Marine);
	int numBunker = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Bunker);
	int numBarracks = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Barracks);
	int numSupplys = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Supply_Depot);
	int numFactorys = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Factory);
	int numTanks = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode)
		+ UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Siege_Mode);
	int numMinerals = BWAPI::Broodwar->self()->minerals();
	int numGas = BWAPI::Broodwar->self()->gas();
	BWAPI::Player self = BWAPI::Broodwar->self();
	int supplyTotal = BWAPI::Broodwar->self()->supplyTotal();
	int supplyUsed = BWAPI::Broodwar->self()->supplyUsed();
	bool hasEBay = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Engineering_Bay) > 0;

	//8SCVһ
	if (numSCVs >= 8)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Supply_Depot) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Supply_Depot) && numSupplys == 0)
		{
			BuildingPlacer::Instance().allowBuildSupply = true;
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Supply_Depot));
		}
	}
	//SCVͣ
	if (!queue.anyInNextN(BWAPI::UnitTypes::Terran_SCV, 2) && numSCVs < 30 &&
		!queue.anyInNextN(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode, 2) &&
		UnitUtil::GetUncompletedUnitCount(BWAPI::UnitTypes::Terran_SCV) == 0)
	{
		queue.queueAsHighestPriority(BWAPI::UnitTypes::Terran_SCV);
	}
	//10SCVӪ
	if (numSCVs >= 10 && numBarracks == 0)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Barracks) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Barracks))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Barracks));
		}
	}
	//11SCV
	if (numSCVs >= 11)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Refinery) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Refinery) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Refinery) < 1 &&
			numMinerals > 100)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Refinery));
		}
	}
	//Ӫ,2ǹر
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Barracks) >= 1)
	{
		if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Barracks) > 0 &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Marine) < 2 &&
			!queue.anyInNextN(BWAPI::UnitTypes::Terran_Marine, 2) &&
			UnitUtil::GetUncompletedUnitCount(BWAPI::UnitTypes::Terran_Marine) == 0)
		{
			queue.queueAsHighestPriority(BWAPI::UnitTypes::Terran_Marine);
		}
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Bunker) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Bunker) &&
			numBunker < 1 &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Marine) > 0)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Bunker));
			BuildingPlacer::Instance().setBunkerBuildPlace(InformationManager::Instance().getPosition("bunkerPoint"));
		}
	}
	//
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Barracks) >= 1)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Factory) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Factory) &&
			numFactorys == 0 &&
			numMinerals > 200 && numGas > 100)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Factory));
		}
	}
	//ֹͣ
	if (numGas - numMinerals  > 100)
	{
		WorkerManager::Instance().setCollectGas(false);
	}
	else if (numGas - numMinerals  < 0)
	{
		WorkerManager::Instance().setCollectGas(true);
	}

	//////////սλ/////////////
	CombatCommander::Instance().updateBunkerSquad();

	//////////ģл///////////
	if (UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Factory) >= 1)
	{
		doSwitchModule();
	}
}
void StrategyManager::updateModuleA3(BuildOrderQueue & queue)
{
	int numSCVs = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_SCV);
	int numMarines = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Marine);
	int numBunker = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Bunker);
	int numBarracks = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Barracks);
	int numSupplys = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Supply_Depot);
	int numFactorys = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Factory);
	int numTanks = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode)
		+ UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Siege_Mode);
	int numMinerals = BWAPI::Broodwar->self()->minerals();
	int numGas = BWAPI::Broodwar->self()->gas();
	BWAPI::Player self = BWAPI::Broodwar->self();
	int supplyTotal = BWAPI::Broodwar->self()->supplyTotal();
	int supplyUsed = BWAPI::Broodwar->self()->supplyUsed();
	bool hasEBay = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Engineering_Bay) > 0;

	//8SCVһ
	if (numSCVs >= 8)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Supply_Depot) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Supply_Depot) && numSupplys == 0)
		{
			BuildingPlacer::Instance().allowBuildSupply = true;
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Supply_Depot));
		}
	}
	//SCVͣ
	if (!queue.anyInNextN(BWAPI::UnitTypes::Terran_SCV, 2) && numSCVs < 30 &&
		!queue.anyInNextN(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode, 2) &&
		UnitUtil::GetUncompletedUnitCount(BWAPI::UnitTypes::Terran_SCV) == 0)
	{
		queue.queueAsHighestPriority(BWAPI::UnitTypes::Terran_SCV);
	}
	//10SCVӪ
	if (numSCVs >= 10 && numBarracks == 0)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Barracks) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Barracks))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Barracks));
		}
	}
	//11SCV
	if (numSCVs >= 11)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Refinery) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Refinery) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Refinery) < 1 &&
			numMinerals > 100)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Refinery));
		}
	}
	////Ӫ,1ǹ
	//if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Barracks) >= 1)
	//{
	//	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Barracks) > 0 &&
	//		UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Marine) < 1 &&
	//		!queue.anyInNextN(BWAPI::UnitTypes::Terran_Marine, 2) &&
	//		UnitUtil::GetUncompletedUnitCount(BWAPI::UnitTypes::Terran_Marine) == 0)
	//	{
	//		queue.queueAsHighestPriority(BWAPI::UnitTypes::Terran_Marine);
	//	}
	//}
	//
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Barracks) >= 1)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Factory) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Factory) &&
			numFactorys == 0 &&
			numMinerals > 200 && numGas > 100)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Factory));
		}
	}
	//ֹͣ
	if (numGas - numMinerals  > 100)
	{
		WorkerManager::Instance().setCollectGas(false);
	}
	else if (numGas - numMinerals  < 0)
	{
		WorkerManager::Instance().setCollectGas(true);
	}

	//////////սλ/////////////
	CombatCommander::Instance().updateBunkerSquad();

	//////////ģл///////////
	if (UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Factory) >= 1)
	{
		doSwitchModule();
	}
}
void StrategyManager::updateModuleA4(BuildOrderQueue & queue)
{
	int numSCVs = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_SCV);
	int numMarines = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Marine);
	int numBunker = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Bunker);
	int numBarracks = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Barracks);
	int numSupplys = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Supply_Depot);
	int numFactorys = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Factory);
	int numTanks = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode)
		+ UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Siege_Mode);
	int numMinerals = BWAPI::Broodwar->self()->minerals();
	int numGas = BWAPI::Broodwar->self()->gas();
	BWAPI::Player self = BWAPI::Broodwar->self();
	int supplyTotal = BWAPI::Broodwar->self()->supplyTotal();
	int supplyUsed = BWAPI::Broodwar->self()->supplyUsed();
	
	//8SCVӪ
	if (numSCVs >= 8 && numBarracks == 0)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Barracks) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Barracks))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Barracks));
		}
	}
	//9SCVһ
	if (numSCVs >= 9)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Supply_Depot) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Supply_Depot) && numSupplys == 0)
		{
			BuildingPlacer::Instance().allowBuildSupply = true;
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Supply_Depot));
		}
	}
	//SCVͣ
	if (!queue.anyInNextN(BWAPI::UnitTypes::Terran_SCV, 2) && numSCVs < 30 &&
		UnitUtil::GetUncompletedUnitCount(BWAPI::UnitTypes::Terran_SCV) == 0 &&
		supplyUsed < supplyTotal &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Barracks) &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Supply_Depot))
	{
		queue.queueAsHighestPriority(BWAPI::UnitTypes::Terran_SCV);
	}

	//////////ģл///////////
	if (numSCVs >= 11)
	{
		doSwitchModule();
	}
}
//B1//ǹRush//
void StrategyManager::updateModuleB1(BuildOrderQueue & queue)
{
	int numSCVs = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_SCV);
	int numMarines = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Marine);
	int numBunker = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Bunker);
	int numBarracks = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Barracks);
	int numSupplys = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Supply_Depot);
	int numFactorys = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Factory);
	int numTanks = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode)
		+ UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Siege_Mode);
	int numMinerals = BWAPI::Broodwar->self()->minerals();
	int numGas = BWAPI::Broodwar->self()->gas();
	BWAPI::Player self = BWAPI::Broodwar->self();
	bool hasEBay = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Engineering_Bay) > 0;
	int numTurrets = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Missile_Turret);
	int numGoliaths = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Goliath);
	int numVultures = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Vulture);
	int numCenters = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Command_Center);
	int supplyTotal = BWAPI::Broodwar->self()->supplyTotal();
	int supplyUsed = BWAPI::Broodwar->self()->supplyUsed();
	int minute = BWAPI::Broodwar->getFrameCount() / (24 * 60);
	int numFreeBarracks = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Barracks) -
		UnitUtil::GetUncompletedUnitCount(BWAPI::UnitTypes::Terran_Marine) -
		UnitUtil::GetUncompletedUnitCount(BWAPI::UnitTypes::Terran_Medic);

	////////////////ͽ/////////////////

	//ͣSCV30ʱֹͣ
	if (numSCVs < 20 &&
		UnitUtil::GetUncompletedUnitCount(BWAPI::UnitTypes::Terran_SCV) == 0)
	{
		if (!queue.anyInNextN(BWAPI::UnitTypes::Terran_SCV, 12))
		{
			queue.queueAsLowestPriority(BWAPI::UnitTypes::Terran_SCV);
		}
	}
	//Ӫݻ٣޽
	if (numBarracks < 5)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Barracks) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Barracks) &&
			numMinerals > 135)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Barracks));
		}
	}
	//һֱǹ
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Barracks) > 0 &&
		numFreeBarracks > 0 &&
		numMinerals > 100 &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Marine))
	{
		queue.queueAsLowestPriority(BWAPI::UnitTypes::Terran_Marine);
	}

	//ҽƱ
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Academy) > 0 &&
		numFreeBarracks > 0 &&
		UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Medic) < numMarines / 4 &&
		numMinerals > 50 && numGas > 25 &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Medic))
	{

		queue.queueAsLowestPriority(BWAPI::UnitTypes::Terran_Medic);
	}

	//
	if (numSCVs + numMarines >= 9)
	{
		BuildingPlacer::Instance().allowBuildSupply = true;
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Supply_Depot) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Supply_Depot) &&
			numSupplys == 0 && numMinerals > 90)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Supply_Depot));
		}
	}
	//оԺ//ǹƼ//ҽƱǰ//
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Barracks) >= 1 && 
		numMinerals > 150)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Academy) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Academy) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Academy) == 0)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Academy));
		}
	}
	////ҽƱҪ//
	if (numMarines > 5 && numMinerals > 100)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Refinery) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Refinery) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Refinery) < 1)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Refinery));
		}
	}

	//ʵͣ//
	if (numGas - numMinerals > 100)
	{
		WorkerManager::Instance().setCollectGas(false);
	}
	else if (numGas - numMinerals < 0)
	{
		WorkerManager::Instance().setCollectGas(true);
	}

	//////////սλı////////////

	CombatCommander::Instance().updateAttackSquads();
	//
	if (numMarines > 10)
	{
		CombatCommander::Instance().setAggression(true);
	}
	//////////ģл////////////
	if (minute > 10)
	{
		doSwitchModule();
	}
}


//B2//˫//
void StrategyManager::updateModuleB2(BuildOrderQueue & queue)
{
	int numSCVs = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_SCV);
	int numMarines = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Marine);
	int numBunker = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Bunker);
	int numBarracks = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Barracks);
	int numSupplys = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Supply_Depot);
	int numFactorys = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Factory);
	int numTanks = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode)
		+ UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Siege_Mode);
	int numMinerals = BWAPI::Broodwar->self()->minerals();
	int numGas = BWAPI::Broodwar->self()->gas();
	BWAPI::Player self = BWAPI::Broodwar->self();
	bool hasEBay = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Engineering_Bay) > 0;
	int numTurrets = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Missile_Turret);
	int numGoliaths = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Goliath);
	int numVultures = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Vulture);
	int numCenters = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Command_Center);
	int supplyTotal = BWAPI::Broodwar->self()->supplyTotal() / 2;
	int supplyUsed = BWAPI::Broodwar->self()->supplyUsed() / 2;

	////////////////ͽ/////////////////

	if (_lastModule == "A4")
	{
		//·һĵر
		BWAPI::Unit choke1Bunker = BWAPI::Broodwar->getClosestUnit(
			InformationManager::Instance().getPosition("bunkerPoint"),
			BWAPI::Filter::IsOwned && BWAPI::Filter::GetType == BWAPI::UnitTypes::Terran_Bunker,
			100);
		if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Barracks) > 0 && numCenters < 2)
		{
			if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Bunker) &&
				!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Bunker) &&
				!choke1Bunker)
			{
				queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Bunker));
				BuildingPlacer::Instance().setBunkerBuildPlace(InformationManager::Instance().getPosition("bunkerPoint"));
			}

			if (!choke1Bunker && numMinerals < 180) { return; }
		}
	}

	//SCV10ֹͣһȲSCV
	//10ͣSCV50ʱֹͣ
	if (numSCVs < 10)
	{
		if (!queue.anyInNextN(BWAPI::UnitTypes::Terran_SCV, 12))
		{
			queue.queueAsLowestPriority(BWAPI::UnitTypes::Terran_SCV);
		}
		return;
	}
	if (numSCVs < 45 &&
		UnitUtil::GetUncompletedUnitCount(BWAPI::UnitTypes::Terran_SCV) == 0)
	{
		if (!queue.anyInNextN(BWAPI::UnitTypes::Terran_SCV, 12))
		{
			queue.queueAsLowestPriority(BWAPI::UnitTypes::Terran_SCV);
		}
	}

	//Ӫݻ٣޽
	if (numBarracks == 0)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Barracks) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Barracks) &&
			numMinerals > 135)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Barracks));
		}
	}
	//ǹ䵽5
	//ÿ301ǹֱҷܹеλֹԷܱ
	if ((UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Barracks) >= 1 && numMarines < 2) || 
		(UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Barracks) >= 1 && numMarines < 5 && numFactorys > 0))
	{
		if (!queue.anyInNextN(BWAPI::UnitTypes::Terran_Marine, 12) &&
			UnitUtil::GetUncompletedUnitCount(BWAPI::UnitTypes::Terran_Marine) == 0)
		{
			queue.queueAsLowestPriority(BWAPI::UnitTypes::Terran_Marine);
		}
	}
	//
	if (numSCVs + numMarines >= 9)
	{
		BuildingPlacer::Instance().allowBuildSupply = true;
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Supply_Depot) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Supply_Depot) &&
			numSupplys == 0 && numMinerals > 90)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Supply_Depot));
		}
	}

	////ֹݻ
	if (numSCVs >= 11 && numMinerals > 100)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Refinery) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Refinery) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Refinery) < 1)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Refinery));
		}
	}
	////ͬ//A1A4û޹ﲹ
	if (numFactorys == 0 && numGas > 100)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Factory) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Factory))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Factory));
		}
	}
	//ڶ
	if (numFactorys == 1 && numMinerals > 200 && numGas > 100 && 
		UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Command_Center) > 1)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Factory) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Factory))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Factory));
		}
	}
	//е//޺úӻе
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) >= 1)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Machine_Shop) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Machine_Shop) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) < 1)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Machine_Shop));
		}
	}

	//׳//̹2󣬿ʼ׳
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) > 0 &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Vulture) &&
		numMinerals >= 75 && numVultures < 4)
	{
		queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Terran_Vulture));
	}
	//׳֩
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) >= 1)
	{
		//B2Ϊ֩
		if (!self->hasResearched(BWAPI::TechTypes::Spider_Mines) &&
			!self->isResearching(BWAPI::TechTypes::Spider_Mines) &&
			numMinerals > 100 && numGas > 100 &&
			!queue.anyInNextN(BWAPI::TechTypes::Spider_Mines, 10))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::TechTypes::Spider_Mines));
		}
	}
	////Armory//˵ǰ//
	if (UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Command_Center) > 1)
	{
		if (!queue.numInQueue(BWAPI::UnitTypes::Terran_Armory) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Armory) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Armory) < 1 &&
			numMinerals > 100 && numGas > 50)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Armory));
		}
	}
	//еλ
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Armory) > 0 &&
		shouldUpgrade(BWAPI::UpgradeTypes::Terran_Vehicle_Plating))
	{
		if (!self->isUpgrading(BWAPI::UpgradeTypes::Terran_Vehicle_Plating) &&
			!queue.anyInQueue(BWAPI::UpgradeTypes::Terran_Vehicle_Plating) &&
			!queue.anyInQueue(BWAPI::UpgradeTypes::Terran_Vehicle_Weapons))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UpgradeTypes::Terran_Vehicle_Plating));
		}
	}
	////޺úʼ//ڷ//
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Armory) > 0 &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Goliath) &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode) &&
		numVultures > 3 &&
		UnitUtil::GetUncompletedUnitCount(BWAPI::UnitTypes::Terran_Goliath) == 0 &&
		numMinerals >= 100 && numGas >= 50)
	{
		queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Terran_Goliath));
	}
	////ڶ
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) >= 1 &&
		self->getUpgradeLevel(BWAPI::UpgradeTypes::Charon_Boosters) == 0 &&
		!self->isUpgrading(BWAPI::UpgradeTypes::Charon_Boosters) &&
		!queue.anyInQueue(BWAPI::UpgradeTypes::Charon_Boosters) &&
		numMinerals > 100 && numGas > 100 &&
		numGoliaths > 0)
	{
		queue.queueAsHighestPriority(MacroAct(BWAPI::UpgradeTypes::Charon_Boosters));
	}
	//̹˼//
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) >= 1 &&
		!self->hasResearched(BWAPI::TechTypes::Tank_Siege_Mode) &&
		!self->isResearching(BWAPI::TechTypes::Tank_Siege_Mode) &&
		!queue.anyInNextN(BWAPI::TechTypes::Tank_Siege_Mode, 12) &&
		numTanks > 0)
	{
		queue.queueAsHighestPriority(MacroAct(BWAPI::TechTypes::Tank_Siege_Mode));
	}
	//̹//Ϊ̹//
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) > 0 &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode) &&
		numMinerals >= 150 && numGas >= 100 &&
		numTanks < 1 && numGoliaths > 3)
	{
		queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode));
	}

	////
	if ((numFactorys > 0 && BWAPI::Broodwar->enemy()->getRace() != BWAPI::Races::Terran) ||
		UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Command_Center) > 1)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Engineering_Bay) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Engineering_Bay) &&
			!hasEBay && numMinerals > 125)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Engineering_Bay));
		}
	}

	//̨
	if (UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Command_Center) > 1)
	{
		buildTurret(queue, 3, 1, 2);
	}

	/*
	////
	BWAPI::Unit baseTurret = BWAPI::Broodwar->getClosestUnit(
		Bases::Instance().myStartingBase()->getPosition(),
		BWAPI::Filter::IsOwned && BWAPI::Filter::GetType == BWAPI::UnitTypes::Terran_Missile_Turret,
		300);
	std::vector<BWAPI::TilePosition> tiles = BWTA::getShortestPath(BWAPI::TilePosition(InformationManager::Instance().getPosition("barracksPoint")),
		BWAPI::TilePosition(Bases::Instance().myStartingBase()->getPosition()));
	BWAPI::Unit choke1Turret = BWAPI::Broodwar->getClosestUnit(
		BWAPI::Position(tiles[3]),
		BWAPI::Filter::IsOwned && BWAPI::Filter::GetType == BWAPI::UnitTypes::Terran_Missile_Turret,
		160);
	BWAPI::Unit choke2Turret = BWAPI::Broodwar->getClosestUnit(
		InformationManager::Instance().getPosition("chokePoint2"),
		BWAPI::Filter::IsOwned && BWAPI::Filter::GetType == BWAPI::UnitTypes::Terran_Missile_Turret,
		120);
	BWAPI::Unit choke2Bunker = BWAPI::Broodwar->getClosestUnit(
		InformationManager::Instance().getPosition("chokePoint2"),
		BWAPI::Filter::IsOwned && BWAPI::Filter::GetType == BWAPI::UnitTypes::Terran_Bunker,
		120);

	if (hasEBay && choke2Bunker && !choke2Turret &&
		UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Command_Center) > 1)
	{
		BuildingPlacer::Instance().allowBuildTuttet = true;

		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Missile_Turret) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Missile_Turret))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Missile_Turret));
			BuildingPlacer::Instance().setTurretBuildPlace(InformationManager::Instance().getPosition("chokePoint2"));
		}
	}
	else if (hasEBay && !choke1Turret)
	{
		BuildingPlacer::Instance().allowBuildTuttet = true;

		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Missile_Turret) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Missile_Turret))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Missile_Turret));

			BuildingPlacer::Instance().setTurretBuildPlace(BWAPI::Position(tiles[3]));
		}
	}
	else if (hasEBay &&
		BuildingPlacer::Instance().getBaseTurretPosition() != BWAPI::Positions::None)//choke12
	{
		BuildingPlacer::Instance().allowBuildTuttet = true;

		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Missile_Turret) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Missile_Turret))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Missile_Turret));
			BuildingPlacer::Instance().setTurretBuildPlace(BuildingPlacer::Instance().getBaseTurretPosition());
		}
	}
	*/
	//оԺ//̨ǰ//
	if (UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Command_Center) > 1 &&
		numMinerals > 150)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Academy) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Academy) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Academy) == 0)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Academy));
		}
	}
	//̨//з//
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Academy) >= 1)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Comsat_Station) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Comsat_Station) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Comsat_Station) < 1 &&
			numMinerals > 50 && numGas > 50)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Comsat_Station));
		}
	}

	//ֻغͷֻصĵر//
	BWAPI::Unit choke2Bunker = BWAPI::Broodwar->getClosestUnit(
		InformationManager::Instance().getPosition("chokePoint2"),
		BWAPI::Filter::IsOwned && BWAPI::Filter::GetType == BWAPI::UnitTypes::Terran_Bunker,
		120);

	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) > 0)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Bunker) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Bunker) &&
			!choke2Bunker && numMinerals > 100)
		{
			_buildSecondBase = true;
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Bunker));
			BuildingPlacer::Instance().setBunkerBuildPlace(InformationManager::Instance().getPosition("chokePoint2"));
		}
	}
	if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Command_Center) &&
		!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Command_Center) &&
		choke2Bunker && numMinerals > 400 &&
		UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) > 0)
	{
		queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Command_Center));
	}
	//ʵͣ
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) == 0)
	{
		if (numGas - numMinerals > 100)
		{
			WorkerManager::Instance().setCollectGas(false);
		}
		else if (numGas - numMinerals < 0)
		{
			WorkerManager::Instance().setCollectGas(true);
		}
	}
	else
	{
		WorkerManager::Instance().setCollectGas(true);
	}
	//////////սλı////////////

	CombatCommander::Instance().updateBunkerSquad();
	CombatCommander::Instance().updateChokeSquads();
	CombatCommander::Instance().updateAttackSquads();
	//ڼſڲ
	if (numVultures > 0 && choke2Bunker)
	{
		BWAPI::Position aimPos = CombatCommander::Instance().getNearPosition(InformationManager::Instance().getPosition("chokePoint2"),
			InformationManager::Instance().getPosition("frontLine1"), false, 300);
		CombatCommander::Instance().updateMineAsRowFrontSquad(aimPos, InformationManager::Instance().getPosition("chokePoint2"), 60, 200, 0);
	}
	else
	{
		CombatCommander::Instance().clearSquad("mineRowFrontSquad");
	}
	//һֱ
	CombatCommander::Instance().setAggression(false);

	//////////ģл////////////
	if (numCenters > 1 || supplyUsed > 100)
	{
		doSwitchModule();
	}
}


/*
//B2//˫//
void StrategyManager::updateModuleB2(BuildOrderQueue & queue)
{
	int numSCVs = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_SCV);
	int numMarines = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Marine);
	int numBunker = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Bunker);
	int numBarracks = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Barracks);
	int numSupplys = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Supply_Depot);
	int numFactorys = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Factory);
	int numTanks = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode)
		+ UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Siege_Mode);
	int numMinerals = BWAPI::Broodwar->self()->minerals();
	int numGas = BWAPI::Broodwar->self()->gas();
	BWAPI::Player self = BWAPI::Broodwar->self();
	bool hasEBay = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Engineering_Bay) > 0;
	int numTurrets = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Missile_Turret);
	int numGoliaths = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Goliath);
	int numVultures = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Vulture);
	int numCenters = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Command_Center);
	int supplyTotal = BWAPI::Broodwar->self()->supplyTotal() / 2;
	int supplyUsed = BWAPI::Broodwar->self()->supplyUsed() / 2;

	////////////////ͽ/////////////////

	//·һĵر
	BWAPI::Unit choke1Bunker = BWAPI::Broodwar->getClosestUnit(
		InformationManager::Instance().getPosition("bunkerPoint"),
		BWAPI::Filter::IsOwned && BWAPI::Filter::GetType == BWAPI::UnitTypes::Terran_Bunker,
		100);
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Barracks) > 0 && numCenters < 2)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Bunker) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Bunker) &&
			!choke1Bunker)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Bunker));
			BuildingPlacer::Instance().setBunkerBuildPlace(InformationManager::Instance().getPosition("bunkerPoint"));
		}

		if (!choke1Bunker && numMinerals < 180) { return;}
	}

	//SCV10ֹͣһȲSCV
	//10ͣSCV50ʱֹͣ
	if (numSCVs < 10)
	{
		if (!queue.anyInNextN(BWAPI::UnitTypes::Terran_SCV, 2))
		{
			queue.queueAsHighestPriority(BWAPI::UnitTypes::Terran_SCV);
		}
		return;
	}
	if (numSCVs < 45 &&
		UnitUtil::GetUncompletedUnitCount(BWAPI::UnitTypes::Terran_SCV) == 0)
	{
		if (!queue.anyInNextN(BWAPI::UnitTypes::Terran_SCV, 2))
		{
			queue.queueAsHighestPriority(BWAPI::UnitTypes::Terran_SCV);
		}
	}

	//Ӫݻ٣޽
	if (numBarracks == 0)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Barracks) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Barracks) &&
			numMinerals > 135)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Barracks));
		}
	}
	//ǹ䵽5
	//ÿ301ǹֱҷܹеλֹԷܱ
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Barracks) >= 1 &&
		(numMarines < 5 || (BWAPI::Broodwar->getFrameCount() % (24 * 50) == 1 && numFactorys < 1)) &&
		UnitUtil::GetUncompletedUnitCount(BWAPI::UnitTypes::Terran_Marine) == 0)
	{
		if (!queue.anyInNextN(BWAPI::UnitTypes::Terran_Marine, 2))
		{
			queue.queueAsHighestPriority(BWAPI::UnitTypes::Terran_Marine);
		}
	}
	//
	if (numSCVs + numMarines >= 9)
	{
		BuildingPlacer::Instance().allowBuildSupply = true;
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Supply_Depot) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Supply_Depot) &&
			numSupplys == 0 && numMinerals > 90)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Supply_Depot));
		}
	}

	////ֹݻ
	if (numSCVs >= 11 && numMinerals > 100)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Refinery) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Refinery) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Refinery) < 1)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Refinery));
		}
	}
	////ͬ//A1A4û޹ﲹ
	if (numFactorys == 0 && numMinerals > 200 && numGas > 100)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Factory) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Factory))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Factory));
		}
	}
	//ڶ
	if (numFactorys == 1 && numMinerals > 200 && numGas > 100 && 
		UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Command_Center) > 1)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Factory) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Factory))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Factory));
		}
	}
	//е//޺úӻе
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) >= 1)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Machine_Shop) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Machine_Shop) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) < 1)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Machine_Shop));
		}
	}

	//׳//̹2󣬿ʼ׳
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) > 0 &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Vulture) &&
		numMinerals >= 75 && numVultures < 4)
	{
		queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Vulture));
	}
	//׳֩
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) >= 1)
	{
		//B2Ϊ֩
		if (!self->hasResearched(BWAPI::TechTypes::Spider_Mines) &&
			!self->isResearching(BWAPI::TechTypes::Spider_Mines) &&
			numMinerals > 100 && numGas > 100 &&
			!queue.anyInNextN(BWAPI::TechTypes::Spider_Mines, 10))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::TechTypes::Spider_Mines));
		}
	}
	////Armory//˵ǰ//
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) >= 1)
	{
		if (!queue.numInQueue(BWAPI::UnitTypes::Terran_Armory) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Armory) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Armory) < 1 &&
			numMinerals > 100 && numGas > 50)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Armory));
		}
	}
	//еλ
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Armory) > 0 && 
		shouldUpgrade(BWAPI::UpgradeTypes::Terran_Vehicle_Plating))
	{
		if (!self->isUpgrading(BWAPI::UpgradeTypes::Terran_Vehicle_Plating) &&
			!queue.anyInQueue(BWAPI::UpgradeTypes::Terran_Vehicle_Plating) &&
			!queue.anyInQueue(BWAPI::UpgradeTypes::Terran_Vehicle_Weapons))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UpgradeTypes::Terran_Vehicle_Plating));
		}
	}
	////޺úʼ//ڷ//
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Armory) > 0 &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Goliath) &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode) &&
		numVultures > 3 && 
		UnitUtil::GetUncompletedUnitCount(BWAPI::UnitTypes::Terran_Goliath) == 0 &&
		numMinerals >= 100 && numGas >= 50)
	{
		queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Goliath));
	}	
	////ڶ
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) >= 1 &&
		self->getUpgradeLevel(BWAPI::UpgradeTypes::Charon_Boosters) == 0 &&
		!self->isUpgrading(BWAPI::UpgradeTypes::Charon_Boosters) &&
		!queue.anyInQueue(BWAPI::UpgradeTypes::Charon_Boosters) &&
		numMinerals > 100 && numGas > 100 &&
		numGoliaths > 0)
	{
		queue.queueAsHighestPriority(MacroAct(BWAPI::UpgradeTypes::Charon_Boosters));
	}	
	//̹˼//
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) >= 1 &&
		!self->hasResearched(BWAPI::TechTypes::Tank_Siege_Mode) &&
		!self->isResearching(BWAPI::TechTypes::Tank_Siege_Mode) &&
		!queue.anyInNextN(BWAPI::TechTypes::Tank_Siege_Mode, 12) &&
		numTanks > 0)
	{
		queue.queueAsHighestPriority(MacroAct(BWAPI::TechTypes::Tank_Siege_Mode));
	}
	//̹//Ϊ̹//
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) > 0 &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode) &&
		numMinerals >= 150 && numGas >= 100 &&
		numTanks < 1 && numGoliaths > 3)
	{
		queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode));
	}

	////
	if (numFactorys > 0)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Engineering_Bay) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Engineering_Bay) &&
			!hasEBay && numMinerals > 125)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Engineering_Bay));
		}
	}
	////
	if (hasEBay && numTurrets < 3 && numFactorys > 0)
	{
		BuildingPlacer::Instance().allowBuildTuttet = true;

		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Missile_Turret) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Missile_Turret))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Missile_Turret));
			BuildingPlacer::Instance().setTurretBuildPlace(Bases::Instance().myStartingBase()->getPosition());
		}
	}
	else if (hasEBay && numTurrets >= 2 && numTurrets < 4 && numFactorys > 0)//choke12
	{
		BuildingPlacer::Instance().allowBuildTuttet = true;

		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Missile_Turret) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Missile_Turret))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Missile_Turret));
			BuildingPlacer::Instance().setTurretBuildPlace(InformationManager::Instance().getPosition("chokePoint1"));
		}
	}
	//оԺ//̨ǰ//
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) >= 1 &&
		numMinerals > 150)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Academy) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Academy) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Academy) == 0)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Academy));
		}
	}
	//̨//з//
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Academy) >= 1)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Comsat_Station) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Comsat_Station) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Comsat_Station) < 1 &&
			numMinerals > 50 && numGas > 50)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Comsat_Station));
		}
	}

	//ֻغͷֻصĵر//
	BWAPI::Unit choke2Bunker = BWAPI::Broodwar->getClosestUnit(
		InformationManager::Instance().getPosition("chokePoint2"),
		BWAPI::Filter::IsOwned && BWAPI::Filter::GetType == BWAPI::UnitTypes::Terran_Bunker,
		6 * 32);
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Vulture) > 2)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Bunker) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Bunker) &&
			!choke2Bunker && numMinerals > 100)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Bunker));
			BuildingPlacer::Instance().setBunkerBuildPlace(InformationManager::Instance().getPosition("chokePoint2"));
		}
	}
	if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Command_Center) &&
		!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Command_Center) &&
		choke2Bunker && numMinerals > 400)
	{
		queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Command_Center));
	}
	//ʵͣ
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) == 0)
	{
		if (numGas - numMinerals > 100)
		{
			WorkerManager::Instance().setCollectGas(false);
		}
		else if (numGas - numMinerals < 0)
		{
			WorkerManager::Instance().setCollectGas(true);
		}
	}
	else
	{
		WorkerManager::Instance().setCollectGas(true); 
	}
	//////////սλı////////////

	CombatCommander::Instance().updateBunkerSquad();
	CombatCommander::Instance().updateChokeSquads();
	CombatCommander::Instance().updateAttackSquads();
	//ڼſڲ
	if (numVultures > 0 && choke2Bunker)
	{
		BWAPI::Position aimPos = CombatCommander::Instance().getNearPosition(InformationManager::Instance().getPosition("chokePoint2"),
			InformationManager::Instance().getPosition("frontLine1"), false, 200);
		CombatCommander::Instance().updateMineAsRowFrontSquad(aimPos, InformationManager::Instance().getPosition("chokePoint2"), 20, 200);
	}
	else if (numVultures > 0 && !choke2Bunker &&
		self->hasResearched(BWAPI::TechTypes::Spider_Mines))
	{
		BWAPI::Position aimPos = CombatCommander::Instance().getNearPosition(InformationManager::Instance().getPosition("barracksPoint"),
			Bases::Instance().myStartingBase()->getPosition(), true, 100);
		CombatCommander::Instance().updateMineAsRowFrontSquad(aimPos, Bases::Instance().myStartingBase()->getPosition(), 20, 100);
	}
	else
	{
		CombatCommander::Instance().clearSquad("mineRowFrontSquad");
	}
	//һֱ
	CombatCommander::Instance().setAggression(false);

	//////////ģл////////////
	if (numCenters > 1 || supplyUsed > 100)
	{
		doSwitchModule();
	}
}
*/


//B3//2VFѹ//
void StrategyManager::updateModuleB3(BuildOrderQueue & queue)
{
	int numSCVs = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_SCV);
	int numMarines = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Marine);
	int numBunker = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Bunker);
	int numBarracks = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Barracks);
	int numSupplys = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Supply_Depot);
	int numFactorys = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Factory);
	int numTanks = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode)
		+ UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Siege_Mode);
	int numMinerals = BWAPI::Broodwar->self()->minerals();
	int numGas = BWAPI::Broodwar->self()->gas();
	BWAPI::Player self = BWAPI::Broodwar->self();
	bool hasEBay = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Engineering_Bay) > 0;
	int numTurrets = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Missile_Turret);
	int numGoliaths = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Goliath);
	int numVultures = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Vulture);
	int numCenters = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Command_Center);
	int supplyTotal = BWAPI::Broodwar->self()->supplyTotal() / 2;
	int supplyUsed = BWAPI::Broodwar->self()->supplyUsed() / 2;

	////////////////ͽ/////////////////

	//SCV10ֹͣһȲSCV
	//SCVͣ30ʱֹͣ
	//ʱ¿ʼSCVֱ60
	if ((numSCVs < 30 || 
		(numSCVs < 60 && (queue.anyInQueue(BWAPI::UnitTypes::Terran_Command_Center) || UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Command_Center) == 2))) &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Vulture) &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Marine) &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Goliath) &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode))
	{
		if (!queue.anyInNextN(BWAPI::UnitTypes::Terran_SCV, 12))
		{
			queue.queueAsLowestPriority(BWAPI::UnitTypes::Terran_SCV);
		}
	}
	//Ӫݻ٣޽
	if (numSCVs >= 10 && numBarracks == 0)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Barracks) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Barracks) &&
			numMinerals > 135)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Barracks));
		}
	}
	//ǹ䵽5
	//ÿ301ǹֱҷܹеλֹԷܱ
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Barracks) >= 1 &&
		(numMarines < 5 || (BWAPI::Broodwar->getFrameCount() % (24 * 50) == 1 && numFactorys < 1)))
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Marine))
		{
			queue.queueAsLowestPriority(BWAPI::UnitTypes::Terran_Marine);
		}
	}
	//
	if (numSCVs + numMarines >= 9)
	{
		BuildingPlacer::Instance().allowBuildSupply = true;
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Supply_Depot) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Supply_Depot) &&
			numSupplys == 0 && numMinerals > 90)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Supply_Depot));
		}
	}

	////A2,A3,A4Ѿ޺ãֻA1ûޣҪﲹ
	if (numSCVs >= 11 && numMinerals > 100)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Refinery) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Refinery) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Refinery) < 1)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Refinery));
		}
	}
	////ͬϣֻA1Ҫ//
	if (numFactorys == 0 && numMinerals > 200 && numGas > 100)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Factory) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Factory))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Factory));
		}
	}
	//е//޺úӻе
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) >= 1)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Machine_Shop) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Machine_Shop) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) < 1)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Machine_Shop));
		}
	}
	//׳//е޺ú׳
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) > 0 &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Vulture) &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Goliath) &&
		_numFreeFactory > 0 &&
		numMinerals >= 75 && numVultures < 4)
	{
		queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Terran_Vulture));
	}
	//׳ٶȺ֩
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) >= 1)
	{
		//ٶ
		if (self->getUpgradeLevel(BWAPI::UpgradeTypes::Ion_Thrusters) == 0 &&
			!self->isUpgrading(BWAPI::UpgradeTypes::Ion_Thrusters) &&
			!queue.anyInQueue(BWAPI::UpgradeTypes::Ion_Thrusters) &&
			numMinerals > 100 && numGas > 100)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UpgradeTypes::Ion_Thrusters));
		}
		//Ȼ֩
		else if (!self->hasResearched(BWAPI::TechTypes::Spider_Mines) &&
			!self->isResearching(BWAPI::TechTypes::Spider_Mines) &&
			numMinerals > 100 && numGas > 100 &&
			!queue.anyInNextN(BWAPI::TechTypes::Spider_Mines, 2) &&
			!self->isUpgrading(BWAPI::UpgradeTypes::Ion_Thrusters) &&
			!queue.anyInQueue(BWAPI::UpgradeTypes::Ion_Thrusters))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::TechTypes::Spider_Mines));
		}
	}
	//Ǯ޵ڶ
	if (numFactorys == 1 && numMinerals > 200 && numGas > 100 && 
		UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) > 0)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Factory) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Factory))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Factory));
		}
	}
	//̹//ڶ޺ú2̹
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) > 1 &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode) &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Goliath) &&
		numMinerals >= 150 && numGas >= 100 && 
		numTanks < 1 && numVultures > 3)
	{
		queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode));
	}
	////Armory//˵ǰ//
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) >= 2)
	{
		if (!queue.numInQueue(BWAPI::UnitTypes::Terran_Armory) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Armory) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Armory) < 1 &&
			numMinerals > 100 && numGas > 50)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Armory));
		}
	}
	////޺úʼ//ڷ//
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Armory) > 0 &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Goliath) &&
		_numFreeFactory > 0 &&
		numVultures > 3 &&
		numMinerals >= 100 && numGas >= 50)
	{
		queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Terran_Goliath));
	}
	//
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) >= 1 &&
		self->getUpgradeLevel(BWAPI::UpgradeTypes::Charon_Boosters) == 0 &&
		!self->isUpgrading(BWAPI::UpgradeTypes::Charon_Boosters) &&
		!queue.anyInQueue(BWAPI::UpgradeTypes::Charon_Boosters) &&
		numMinerals > 100 && numGas > 100 &&
		numGoliaths > 0)
	{
		queue.queueAsHighestPriority(MacroAct(BWAPI::UpgradeTypes::Charon_Boosters));
	}
	//̹˼//һЩ
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) >= 1 &&
		UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Armory) >= 1 &&
		!self->hasResearched(BWAPI::TechTypes::Tank_Siege_Mode) &&
		!self->isResearching(BWAPI::TechTypes::Tank_Siege_Mode) &&
		!queue.anyInNextN(BWAPI::TechTypes::Tank_Siege_Mode, 10) &&
		numMinerals > 150 && numGas > 150)
	{
		queue.queueAsHighestPriority(MacroAct(BWAPI::TechTypes::Tank_Siege_Mode));
	}
	////
	if (numFactorys > 0)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Engineering_Bay) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Engineering_Bay) &&
			!hasEBay && numMinerals > 125)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Engineering_Bay));
		}
	}

	//̨
	if (numFactorys > 0)
	{
		buildTurret(queue, 3, 1, 2);
	}
	//оԺ//̨ǰ//
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) >= 2 &&
		numMinerals > 150)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Academy) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Academy) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Academy) == 0)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Academy));
		}
	}
	//̨//з//
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Academy) >= 1)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Comsat_Station) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Comsat_Station) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Comsat_Station) < 1 &&
			numMinerals > 50 && numGas > 50)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Comsat_Station));
		}
	}

	//ֻغͷֻصĵر//
	BWAPI::Unit choke2Bunker = BWAPI::Broodwar->getClosestUnit(
		InformationManager::Instance().getPosition("chokePoint2"),
		BWAPI::Filter::IsOwned && BWAPI::Filter::GetType == BWAPI::UnitTypes::Terran_Bunker,
		120);

	if (numGoliaths > 3 || numMinerals > 600)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Bunker) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Bunker) &&
			!choke2Bunker && numMinerals > 100)
		{
			_buildSecondBase = true;
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Bunker));
			BuildingPlacer::Instance().setBunkerBuildPlace(InformationManager::Instance().getPosition("chokePoint2"));
		}
	}
	if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Command_Center) &&
		!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Command_Center) &&
		choke2Bunker && numMinerals > 400)
	{
		queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Command_Center));
	}
	//ʵͣ
	if (numGas - numMinerals  > 100)
	{
		WorkerManager::Instance().setCollectGas(false);
	}
	else if (numGas - numMinerals  < 0)
	{
		WorkerManager::Instance().setCollectGas(true);
	}

	//////////սλı////////////

	CombatCommander::Instance().updateB3Squad();
	//ڶԷſ
	if (numVultures > 0 &&
		CombatCommander::Instance().getAggression() &&
		Bases::Instance().enemyStart())
	{
		CombatCommander::Instance().updateMineAsRowFrontSquad(InformationManager::Instance().getPosition("frontLine3"),
			InformationManager::Instance().getPosition("frontLine2"), 60, 260, 1);
	}
	else
	{
		CombatCommander::Instance().clearSquad("mineRowFrontSquad");
	}
	//㹻ʼѹƶԷ
	if (numVultures > 3 && numTanks > 0 && numMarines > 3)
	{
		CombatCommander::Instance().setAggression(true);
	}
	else
	{
		CombatCommander::Instance().setAggression(false);
	}

	//////////ģл////////////
	if (supplyUsed > 58 || BWAPI::Broodwar->getFrameCount() / (24 * 60) > 6)
	{
		doSwitchModule();
	}
}

void StrategyManager::updateModuleB5(BuildOrderQueue & queue)
{
	int numSCVs = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_SCV);
	int numMarines = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Marine);
	int numBunker = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Bunker);
	int numBarracks = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Barracks);
	int numSupplys = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Supply_Depot);
	int numFactorys = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Factory);
	int numTanks = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode)
		+ UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Siege_Mode);
	int numMinerals = BWAPI::Broodwar->self()->minerals();
	int numGas = BWAPI::Broodwar->self()->gas();
	BWAPI::Player self = BWAPI::Broodwar->self();
	bool hasEBay = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Engineering_Bay) > 0;
	int numTurrets = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Missile_Turret);
	int numGoliaths = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Goliath);
	int numVultures = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Vulture);
	int numCenters = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Command_Center);
	int supplyTotal = BWAPI::Broodwar->self()->supplyTotal() / 2;
	int supplyUsed = BWAPI::Broodwar->self()->supplyUsed() / 2;

	////////////////ͽ/////////////////

	//SCV10ֹͣһȲSCV
	//10ͣSCV33ʱֹͣ
	if (numSCVs < 10)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_SCV))
		{
			queue.queueAsLowestPriority(BWAPI::UnitTypes::Terran_SCV);
		}
		return;
	}
	if (numSCVs < 33 &&
		UnitUtil::GetUncompletedUnitCount(BWAPI::UnitTypes::Terran_SCV) == 0)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_SCV))
		{
			queue.queueAsLowestPriority(BWAPI::UnitTypes::Terran_SCV);
		}
	}

	//Ӫݻ٣޽
	if (numBarracks == 0)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Barracks) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Barracks) &&
			numMinerals > 135)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Barracks));
		}
	}

	//ǹ䵽2
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Barracks) >= 1 && numMarines < 1 && numFactorys > 0)
	{
		if (!queue.anyInNextN(BWAPI::UnitTypes::Terran_Marine, 12) &&
			UnitUtil::GetUncompletedUnitCount(BWAPI::UnitTypes::Terran_Marine) == 0)
		{
			queue.queueAsLowestPriority(BWAPI::UnitTypes::Terran_Marine);
		}
	}
	//䵽4
	else if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) >= 1 && numMarines < 4 && numMinerals > 200)
	{
		if (!queue.anyInNextN(BWAPI::UnitTypes::Terran_Marine, 12) &&
			UnitUtil::GetUncompletedUnitCount(BWAPI::UnitTypes::Terran_Marine) == 0)
		{
			queue.queueAsLowestPriority(BWAPI::UnitTypes::Terran_Marine);
		}
	}

	//
	if (numSCVs + numMarines >= 9)
	{
		BuildingPlacer::Instance().allowBuildSupply = true;
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Supply_Depot) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Supply_Depot) &&
			numSupplys == 0 && numMinerals > 90)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Supply_Depot));
		}
	}

	////ֹݻ
	if (numSCVs >= 11 && numMinerals > 100)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Refinery) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Refinery) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Refinery) < 1)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Refinery));
		}
	}
	////ͬ//A1A4û޹ﲹ
	if (numFactorys == 0 && numGas > 100)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Factory) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Factory))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Factory));
		}
	}
	static bool ticket = true;
	//ڶ
	if (numFactorys == 1 && numMinerals > 200 && numGas > 100 && ticket)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Factory))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Factory));
			ticket = false;
		}
	}
	//
	if (numFactorys == 2 &&
		numMinerals > 300 && numGas > 100 && 
		UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Command_Center) > 1)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Factory) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Factory))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Factory));
		}
	}

	//
	if (UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Command_Center) > 1 &&
		UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Engineering_Bay) == 0 &&
		numMinerals > 125)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Engineering_Bay) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Engineering_Bay))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Engineering_Bay));
		}
	}
	//·ڶظһ
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Engineering_Bay) > 0)
	{
		buildTurret(queue, 1, 0, 1);
	}

	//е//޺ú,2׳Ȼӻе
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Vulture) > 1 &&
		UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) < 2)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Machine_Shop) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Machine_Shop))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Machine_Shop));
		}
	}
	else if (UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) < UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) &&
		numFactorys > 2)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Machine_Shop) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Machine_Shop))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Machine_Shop));
		}
	}

	//׳//
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) > 0 &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Vulture) &&
		_numFreeFactory > 0 &&
		numMinerals >= 75 && numVultures < 2)
	{
		queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Terran_Vulture));
	}
	else if(UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) > 0 &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Vulture) &&
		_numFreeFactory > 0 &&
		numMinerals >= 75 && numVultures < 6)
	{
		queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Terran_Vulture));
	}
	else if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) > 0 &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Vulture) &&
		_numFreeFactory > 0 &&
		numMinerals >= 75 && numGas < 120)
	{
		queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Terran_Vulture));
	}

	//׳֩
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) > 0)
	{
		//֩
	    if (!self->hasResearched(BWAPI::TechTypes::Spider_Mines) &&
			!self->isResearching(BWAPI::TechTypes::Spider_Mines) &&
			numMinerals > 100 && numGas > 100 &&
			!queue.anyInNextN(BWAPI::TechTypes::Spider_Mines, 12))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::TechTypes::Spider_Mines));
		}
		//Ȼٶ
		else if (self->getUpgradeLevel(BWAPI::UpgradeTypes::Ion_Thrusters) == 0 &&
			!self->isUpgrading(BWAPI::UpgradeTypes::Ion_Thrusters) &&
			!queue.anyInQueue(BWAPI::UpgradeTypes::Ion_Thrusters) &&
			numMinerals > 100 && numGas > 100 &&
			!queue.anyInNextN(BWAPI::TechTypes::Spider_Mines, 12) &&
			!self->isResearching(BWAPI::TechTypes::Spider_Mines))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UpgradeTypes::Ion_Thrusters));
		}
	}
	//̹˼//
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) >= 1 &&
		!self->hasResearched(BWAPI::TechTypes::Tank_Siege_Mode) &&
		!self->isResearching(BWAPI::TechTypes::Tank_Siege_Mode) &&
		!queue.anyInNextN(BWAPI::TechTypes::Tank_Siege_Mode, 12) &&
		numTanks > 0)
	{
		queue.queueAsHighestPriority(MacroAct(BWAPI::TechTypes::Tank_Siege_Mode));
	}
	//̹//Ϊ̹//
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) > 0 &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode) &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Vulture) &&
		numMinerals >= 150 && numGas >= 100 &&
		_numFreeFactory > 0)
	{
		queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode));
	}

	//оԺ//̨ǰ//
	if (UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Command_Center) > 1 &&
		numMinerals > 150)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Academy) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Academy) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Academy) == 0)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Academy));
		}
	}
	//̨//з//
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Academy) >= 1)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Comsat_Station) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Comsat_Station) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Comsat_Station) < 1 &&
			numMinerals > 50 && numGas > 50)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Comsat_Station));
		}
	}

	//ֻغͷֻصĵر//
	BWAPI::Unit choke2Bunker = BWAPI::Broodwar->getClosestUnit(
		InformationManager::Instance().getPosition("chokePoint2"),
		BWAPI::Filter::IsOwned && BWAPI::Filter::GetType == BWAPI::UnitTypes::Terran_Bunker,
		120);

	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) > 0 &&
		UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Command_Center) < 2)
	{
		_buildSecondBaseIn = true;

		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Command_Center) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Command_Center) &&
			numMinerals > 200 &&
			UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) > 1)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Command_Center));
		}
	}

	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Command_Center) > 1)
	{
		_buildSecondBase = true;
		//ֻصر
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Bunker) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Bunker) &&
			!choke2Bunker && numMinerals > 300)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Bunker));
			BuildingPlacer::Instance().setBunkerBuildPlace(InformationManager::Instance().getPosition("chokePoint2"));
		}
		//ڶ
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Refinery) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Refinery) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Refinery) < 2 &&
			numMinerals > 100)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Refinery));
		}
	}

	//ʵͣ
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) == 0)
	{
		if (numGas - numMinerals > 100)
		{
			WorkerManager::Instance().setCollectGas(false);
		}
		else if (numGas - numMinerals < 0)
		{
			WorkerManager::Instance().setCollectGas(true);
		}
	}
	else
	{
		if (numGas - numMinerals > 200)
		{
			WorkerManager::Instance().setCollectGas(false);
		}
		else if (numGas - numMinerals < 0)
		{
			WorkerManager::Instance().setCollectGas(true);
		}
	}
	//////////սλı////////////

	BWAPI::Unitset enemyUnits = BWAPI::Broodwar->getUnitsInRadius(
		InformationManager::Instance().getPosition("chokePoint2"), 600, BWAPI::Filter::IsEnemy);

	//ɧģʽ
	if (numVultures > 2 &&
		enemyUnits.size() < 4 &&
		Bases::Instance().enemyStart() &&
		Bases::Instance().getSecondBase(1))
	{
		//ڶԷſ
		BWAPI::Position aimPos = CombatCommander::Instance().getNearPosition(Bases::Instance().getSecondBase(1)->getPosition(),
			InformationManager::Instance().getPosition("frontLine3"), false, 200);

		CombatCommander::Instance().updateMineAsRowFrontSquad(aimPos,
			InformationManager::Instance().getPosition("frontLine3"), 60, 260, 1);

		//ɧŵз
		CombatCommander::Instance().updateHarassSquad();

		//飬ȫɧ
		CombatCommander::Instance().clearSquad("bunkerSquad2");
		CombatCommander::Instance().clearSquad("chokeSquad1");
		CombatCommander::Instance().clearSquad("chokeSquad2");
		CombatCommander::Instance().clearSquad("chokeSquad3");
		CombatCommander::Instance().clearSquad("chokeSquad4");
	}
	//ģʽ
	else
	{
		CombatCommander::Instance().updateBunkerSquad();
		CombatCommander::Instance().updateChokeSquads();
		CombatCommander::Instance().updateAttackSquads();

		if (numVultures > 0 && _buildSecondBase)
		{
			BWAPI::Position aimPos = CombatCommander::Instance().getNearPosition(InformationManager::Instance().getPosition("chokePoint2"),
				InformationManager::Instance().getPosition("frontLine1"), false, 300);
			CombatCommander::Instance().updateMineAsRowFrontSquad(aimPos, InformationManager::Instance().getPosition("chokePoint2"), 60, 200, 0);
		}

		CombatCommander::Instance().clearSquad("harassSquad");
	}


	CombatCommander::Instance().setAggression(false);

	//////////ģл////////////
	if (supplyUsed > 68 || BWAPI::Broodwar->getFrameCount() / (24 * 60) > 7)
	{
		doSwitchModule();
	}
}

//
void StrategyManager::updateModuleB7(BuildOrderQueue & queue)
{
	int numSCVs = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_SCV);
	int numMarines = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Marine);
	int numBunker = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Bunker);
	int numBarracks = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Barracks);
	int numSupplys = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Supply_Depot);
	int numFactorys = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Factory);
	int numTanks = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode)
		+ UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Siege_Mode);
	int numMinerals = BWAPI::Broodwar->self()->minerals();
	int numGas = BWAPI::Broodwar->self()->gas();
	BWAPI::Player self = BWAPI::Broodwar->self();
	bool hasEBay = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Engineering_Bay) > 0;
	int numTurrets = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Missile_Turret);
	int numGoliaths = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Goliath);
	int numVultures = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Vulture);
	int numCenters = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Command_Center);
	int supplyTotal = BWAPI::Broodwar->self()->supplyTotal() / 2;
	int supplyUsed = BWAPI::Broodwar->self()->supplyUsed() / 2;

	////////////////ͽ/////////////////

	//SCV10ֹͣһȲSCV
	//10ͣSCV33ʱֹͣ
	if (numSCVs < 10)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_SCV))
		{
			queue.queueAsLowestPriority(BWAPI::UnitTypes::Terran_SCV);
		}
		return;
	}
	if (numSCVs < 33 &&
		UnitUtil::GetUncompletedUnitCount(BWAPI::UnitTypes::Terran_SCV) == 0)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_SCV))
		{
			queue.queueAsLowestPriority(BWAPI::UnitTypes::Terran_SCV);
		}
	}

	//Ӫݻ٣޽
	if (numBarracks == 0)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Barracks) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Barracks) &&
			numMinerals > 135)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Barracks));
		}
	}

	//ǹ䵽1
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Barracks) >= 1 && numMarines < 1 && numFactorys > 0)
	{
		if (!queue.anyInNextN(BWAPI::UnitTypes::Terran_Marine, 12) &&
			UnitUtil::GetUncompletedUnitCount(BWAPI::UnitTypes::Terran_Marine) == 0)
		{
			queue.queueAsLowestPriority(BWAPI::UnitTypes::Terran_Marine);
		}
	}
	//䵽6
	else if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) >= 1 && numMarines < 6 && numMinerals > 200)
	{
		if (!queue.anyInNextN(BWAPI::UnitTypes::Terran_Marine, 12) &&
			UnitUtil::GetUncompletedUnitCount(BWAPI::UnitTypes::Terran_Marine) == 0)
		{
			queue.queueAsLowestPriority(BWAPI::UnitTypes::Terran_Marine);
		}
	}

	//
	if (numSCVs + numMarines >= 9)
	{
		BuildingPlacer::Instance().allowBuildSupply = true;
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Supply_Depot) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Supply_Depot) &&
			numSupplys == 0 && numMinerals > 90)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Supply_Depot));
		}
	}

	////ֹݻ
	if (numSCVs >= 11 && numMinerals > 100)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Refinery) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Refinery) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Refinery) < 1)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Refinery));
		}
	}
	////ͬ//A1A4û޹ﲹ
	if (numFactorys == 0 && numGas > 100)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Factory) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Factory))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Factory));
		}
	}
	static bool ticket = true;
	//ڶ
	if (numFactorys == 1 && numMinerals > 200 && numGas > 100 && ticket)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Factory))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Factory));
			ticket = false;
		}
	}
	////
	//if (numFactorys == 2 &&
	//	numMinerals > 300 && numGas > 100 &&
	//	UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Command_Center) > 1)
	//{
	//	if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Factory) &&
	//		!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Factory))
	//	{
	//		queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Factory));
	//	}
	//}
		//
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Command_Center) > 1 &&
		UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Engineering_Bay) == 0 &&
		numMinerals > 125)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Engineering_Bay) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Engineering_Bay))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Engineering_Bay));
		}
	}
	//·ڶظһ
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Engineering_Bay) > 0)
	{
		buildTurret(queue, 1, 0, 1);
	}

	//Armory
	if (numFactorys == 2 &&
		numMinerals > 100 && numGas > 50 &&
		UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Command_Center) > 1 &&
		UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Armory) == 0)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Armory) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Armory))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Armory));
		}
	}

	//е
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Armory) > 0 &&
		shouldUpgrade(BWAPI::UpgradeTypes::Terran_Vehicle_Plating))
	{
		if (!self->isUpgrading(BWAPI::UpgradeTypes::Terran_Vehicle_Plating) &&
			!queue.anyInQueue(BWAPI::UpgradeTypes::Terran_Vehicle_Plating) &&
			!queue.anyInQueue(BWAPI::UpgradeTypes::Terran_Vehicle_Weapons))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UpgradeTypes::Terran_Vehicle_Plating));
		}
	}

	//е//޺ú,2׳Ȼӻе
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Vulture) > 1 &&
		UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) < 2)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Machine_Shop) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Machine_Shop))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Machine_Shop));
		}
	}
	else if (UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) < UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) &&
		numFactorys > 2)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Machine_Shop) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Machine_Shop))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Machine_Shop));
		}
	}

	//׳//
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) > 0 &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Vulture) &&
		_numFreeFactory > 0 &&
		numMinerals >= 75 && numVultures < 2)
	{
		queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Terran_Vulture));
	}
	else if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) > 0 &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Vulture) &&
		_numFreeFactory > 0 &&
		numMinerals >= 75 && numGas < 120 &&
		UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Armory) == 0)
	{
		queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Terran_Vulture));
	}
	else if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) > 0 &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Vulture) &&
		_numFreeFactory > 0 &&
		numMinerals >= 75 && numGas < 120 &&
		numGoliaths > 4)
	{
		queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Terran_Vulture));
	}

	//׳֩
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) > 0)
	{
		//֩
		if (!self->hasResearched(BWAPI::TechTypes::Spider_Mines) &&
			!self->isResearching(BWAPI::TechTypes::Spider_Mines) &&
			numMinerals > 100 && numGas > 100 &&
			!queue.anyInNextN(BWAPI::TechTypes::Spider_Mines, 12))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::TechTypes::Spider_Mines));
		}
	}
	////޺úʼ//ڷ//
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Armory) > 0 &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Goliath) &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode) &&
		numMinerals >= 100 && numGas >= 50 &&
		_numFreeFactory > 0 && numGoliaths < 8)
	{
		queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Terran_Goliath));
	}
	////ڶ
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) >= 1 &&
		self->getUpgradeLevel(BWAPI::UpgradeTypes::Charon_Boosters) == 0 &&
		!self->isUpgrading(BWAPI::UpgradeTypes::Charon_Boosters) &&
		!queue.anyInQueue(BWAPI::UpgradeTypes::Charon_Boosters) &&
		numMinerals > 100 && numGas > 100 &&
		numGoliaths > 0)
	{
		queue.queueAsHighestPriority(MacroAct(BWAPI::UpgradeTypes::Charon_Boosters));
	}

	//̹˼//
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) >= 1 &&
		!self->hasResearched(BWAPI::TechTypes::Tank_Siege_Mode) &&
		!self->isResearching(BWAPI::TechTypes::Tank_Siege_Mode) &&
		!queue.anyInNextN(BWAPI::TechTypes::Tank_Siege_Mode, 12) &&
		(self->getUpgradeLevel(BWAPI::UpgradeTypes::Charon_Boosters) != 0 || self->isUpgrading(BWAPI::UpgradeTypes::Charon_Boosters))&&
		numMinerals > 150 && numGas > 150)
	{
		queue.queueAsHighestPriority(MacroAct(BWAPI::TechTypes::Tank_Siege_Mode));
	}
	//̹//Ϊ̹//
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) > 0 &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode) &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Vulture) &&
		numMinerals >= 150 && numGas >= 100 &&
		_numFreeFactory > 0 && 
		(numGoliaths > 4 || UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Armory) == 0))
	{
		queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode));
	}

	//оԺ//̨ǰ//
	if (UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Command_Center) > 1 &&
		numMinerals > 150)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Academy) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Academy) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Academy) == 0)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Academy));
		}
	}
	//̨//з//
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Academy) >= 1)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Comsat_Station) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Comsat_Station) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Comsat_Station) < 1 &&
			numMinerals > 50 && numGas > 50)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Comsat_Station));
		}
	}

	//ֻغͷֻصĵر//
	BWAPI::Unit choke2Bunker = BWAPI::Broodwar->getClosestUnit(
		InformationManager::Instance().getPosition("chokePoint2"),
		BWAPI::Filter::IsOwned && BWAPI::Filter::GetType == BWAPI::UnitTypes::Terran_Bunker,
		120);

	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) > 0 &&
		UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Command_Center) < 2)
	{
		_buildSecondBaseIn = true;

		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Command_Center) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Command_Center) &&
			numMinerals > 200 &&
			UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) > 1)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Command_Center));
		}
	}

	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Command_Center) > 1)
	{
		_buildSecondBase = true;
		//ֻصر
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Bunker) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Bunker) &&
			!choke2Bunker && numMinerals > 300)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Bunker));
			BuildingPlacer::Instance().setBunkerBuildPlace(InformationManager::Instance().getPosition("chokePoint2"));
		}
		//ڶ
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Refinery) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Refinery) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Refinery) < 2 &&
			numMinerals > 100)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Refinery));
		}
	}

	//ʵͣ
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) == 0)
	{
		if (numGas - numMinerals > 100)
		{
			WorkerManager::Instance().setCollectGas(false);
		}
		else if (numGas - numMinerals < 0)
		{
			WorkerManager::Instance().setCollectGas(true);
		}
	}
	else
	{
		if (numGas - numMinerals > 200)
		{
			WorkerManager::Instance().setCollectGas(false);
		}
		else if (numGas - numMinerals < 0)
		{
			WorkerManager::Instance().setCollectGas(true);
		}
	}
	//////////սλı////////////

	BWAPI::Unitset enemyUnits = BWAPI::Broodwar->getUnitsInRadius(
		InformationManager::Instance().getPosition("chokePoint2"), 600, BWAPI::Filter::IsEnemy);

	//ɧģʽ
	if (numVultures > 2 &&
		enemyUnits.size() < 4 &&
		Bases::Instance().enemyStart() &&
		Bases::Instance().getSecondBase(1))
	{
		//ڶԷſ
		BWAPI::Position aimPos = CombatCommander::Instance().getNearPosition(Bases::Instance().getSecondBase(1)->getPosition(),
			InformationManager::Instance().getPosition("frontLine3"), false, 200);

		CombatCommander::Instance().updateMineAsRowFrontSquad(aimPos,
			InformationManager::Instance().getPosition("frontLine3"), 60, 260, 1);

		//ɧŵз
		CombatCommander::Instance().updateHarassSquad();

		//飬ȫɧ
		CombatCommander::Instance().clearSquad("bunkerSquad2");
		CombatCommander::Instance().clearSquad("chokeSquad1");
		CombatCommander::Instance().clearSquad("chokeSquad2");
		CombatCommander::Instance().clearSquad("chokeSquad3");
		CombatCommander::Instance().clearSquad("chokeSquad4");
	}
	//ģʽ
	else
	{
		CombatCommander::Instance().updateBunkerSquad();
		CombatCommander::Instance().updateChokeSquads();
		CombatCommander::Instance().updateAttackSquads();

		if (numVultures > 0 && _buildSecondBase)
		{
			BWAPI::Position aimPos = CombatCommander::Instance().getNearPosition(InformationManager::Instance().getPosition("chokePoint2"),
				InformationManager::Instance().getPosition("frontLine1"), false, 300);
			CombatCommander::Instance().updateMineAsRowFrontSquad(aimPos, InformationManager::Instance().getPosition("chokePoint2"), 60, 200, 0);
		}

		CombatCommander::Instance().clearSquad("harassSquad");
	}

	CombatCommander::Instance().setAggression(false);

	//////////ģл////////////
	if (supplyUsed > 68 || BWAPI::Broodwar->getFrameCount() / (24 * 60) > 7)
	{
		doSwitchModule();
	}
}


void StrategyManager::updateModuleB6(BuildOrderQueue & queue)
{
	int numSCVs = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_SCV);
	int numMarines = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Marine);
	int numBunker = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Bunker);
	int numBarracks = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Barracks);
	int numSupplys = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Supply_Depot);
	int numFactorys = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Factory);
	int numTanks = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode)
		+ UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Siege_Mode);
	int numMinerals = BWAPI::Broodwar->self()->minerals();
	int numGas = BWAPI::Broodwar->self()->gas();
	BWAPI::Player self = BWAPI::Broodwar->self();
	bool hasEBay = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Engineering_Bay) > 0;
	int numTurrets = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Missile_Turret);
	int numGoliaths = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Goliath);
	int numVultures = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Vulture);
	int numCenters = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Command_Center);
	int supplyTotal = BWAPI::Broodwar->self()->supplyTotal() / 2;
	int supplyUsed = BWAPI::Broodwar->self()->supplyUsed() / 2;

	////////////////ͽ/////////////////

	if (_lastModule == "A4")
	{
		//·һĵر
		BWAPI::Unit choke1Bunker = BWAPI::Broodwar->getClosestUnit(
			InformationManager::Instance().getPosition("bunkerPoint"),
			BWAPI::Filter::IsOwned && BWAPI::Filter::GetType == BWAPI::UnitTypes::Terran_Bunker,
			100);
		if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Barracks) > 0 && numCenters < 2)
		{
			if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Bunker) &&
				!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Bunker) &&
				!choke1Bunker)
			{
				queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Bunker));
				BuildingPlacer::Instance().setBunkerBuildPlace(InformationManager::Instance().getPosition("bunkerPoint"));
			}

			if (!choke1Bunker && numMinerals < 180) { return; }
		}
	}

	//SCV10ֹͣһȲSCV
	//10ͣSCV50ʱֹͣ
	if (numSCVs < 10)
	{
		if (!queue.anyInNextN(BWAPI::UnitTypes::Terran_SCV, 12))
		{
			queue.queueAsLowestPriority(BWAPI::UnitTypes::Terran_SCV);
		}
		return;
	}
	if (numSCVs < 33 &&
		UnitUtil::GetUncompletedUnitCount(BWAPI::UnitTypes::Terran_SCV) == 0)
	{
		if (!queue.anyInNextN(BWAPI::UnitTypes::Terran_SCV, 12))
		{
			queue.queueAsLowestPriority(BWAPI::UnitTypes::Terran_SCV);
		}
	}

	//Ӫݻ٣޽
	if (numBarracks == 0)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Barracks) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Barracks) &&
			numMinerals > 135)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Barracks));
		}
	}
	//ǹ䵽5
	//ÿ301ǹֱҷܹеλֹԷܱ
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Barracks) >= 1 && numMarines < 2)
	{
		if (!queue.anyInNextN(BWAPI::UnitTypes::Terran_Marine, 12) &&
			UnitUtil::GetUncompletedUnitCount(BWAPI::UnitTypes::Terran_Marine) == 0)
		{
			queue.queueAsLowestPriority(BWAPI::UnitTypes::Terran_Marine);
		}
	}
	//
	if (numSCVs + numMarines >= 9)
	{
		BuildingPlacer::Instance().allowBuildSupply = true;
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Supply_Depot) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Supply_Depot) &&
			numSupplys == 0 && numMinerals > 90)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Supply_Depot));
		}
	}

	////ֹݻ
	if (numSCVs >= 11 && numMinerals > 100)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Refinery) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Refinery) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Refinery) < 1)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Refinery));
		}
	}
	////ͬ//A1A4û޹ﲹ
	if (numFactorys == 0 && numGas > 100)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Factory) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Factory))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Factory));
		}
	}
	//ڶ
	if (numFactorys == 1 && numMinerals > 200 && numGas > 100)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Factory))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Factory));
		}
	}
	//е//޺úӻе
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) >
		UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) &&
		UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Vulture) > 1)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Machine_Shop) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Machine_Shop))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Machine_Shop));
		}
	}

	//׳//̹2󣬿ʼ׳
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) > 0 &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Vulture) &&
		_numFreeFactory > 0 &&
		numMinerals >= 75 && numVultures < 4)
	{
		queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Terran_Vulture));
	}
	//׳֩
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) >= 1)
	{
		//B2Ϊ֩
		if (!self->hasResearched(BWAPI::TechTypes::Spider_Mines) &&
			!self->isResearching(BWAPI::TechTypes::Spider_Mines) &&
			numMinerals > 100 && numGas > 100 &&
			!queue.anyInNextN(BWAPI::TechTypes::Spider_Mines, 10))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::TechTypes::Spider_Mines));
		}
	}
	////Armory//˵ǰ//
	if (UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Factory) > 1)
	{
		if (!queue.numInQueue(BWAPI::UnitTypes::Terran_Armory) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Armory) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Armory) < 1 &&
			numMinerals > 100 && numGas > 50)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Armory));
		}
	}
	//еλ
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Armory) > 0 &&
		shouldUpgrade(BWAPI::UpgradeTypes::Terran_Vehicle_Plating) &&
		UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Command_Center) > 0)
	{
		if (!self->isUpgrading(BWAPI::UpgradeTypes::Terran_Vehicle_Plating) &&
			!queue.anyInQueue(BWAPI::UpgradeTypes::Terran_Vehicle_Plating) &&
			!queue.anyInQueue(BWAPI::UpgradeTypes::Terran_Vehicle_Weapons))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UpgradeTypes::Terran_Vehicle_Plating));
		}
	}
	////޺úʼ//ڷ//
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Armory) > 0 &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Goliath) &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode) &&
		numVultures > 3 &&
		_numFreeFactory > 0 &&
		UnitUtil::GetUncompletedUnitCount(BWAPI::UnitTypes::Terran_Goliath) == 0 &&
		numMinerals >= 100 && numGas >= 50)
	{
		queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Terran_Goliath));
	}
	////ڶ
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) >= 1 &&
		self->getUpgradeLevel(BWAPI::UpgradeTypes::Charon_Boosters) == 0 &&
		!self->isUpgrading(BWAPI::UpgradeTypes::Charon_Boosters) &&
		!queue.anyInQueue(BWAPI::UpgradeTypes::Charon_Boosters) &&
		numMinerals > 100 && numGas > 100 &&
		numGoliaths > 0)
	{
		queue.queueAsHighestPriority(MacroAct(BWAPI::UpgradeTypes::Charon_Boosters));
	}
	//̹˼//
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) >= 1 &&
		!self->hasResearched(BWAPI::TechTypes::Tank_Siege_Mode) &&
		!self->isResearching(BWAPI::TechTypes::Tank_Siege_Mode) &&
		!queue.anyInNextN(BWAPI::TechTypes::Tank_Siege_Mode, 12) &&
		numTanks > 0)
	{
		queue.queueAsHighestPriority(MacroAct(BWAPI::TechTypes::Tank_Siege_Mode));
	}
	//̹//Ϊ̹//
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) > 0 &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode) &&
		numMinerals >= 150 && numGas >= 100 &&
		_numFreeFactory > 0 &&
		numTanks < 1 && numGoliaths > 3)
	{
		queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode));
	}

	////
	if (numFactorys > 1)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Engineering_Bay) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Engineering_Bay) &&
			!hasEBay && numMinerals > 125)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Engineering_Bay));
		}
	}
	//̨
	buildTurret(queue, 3, 1, 2);

	//оԺ//̨ǰ//
	if (UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Command_Center) > 1 &&
		numMinerals > 150)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Academy) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Academy) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Academy) == 0)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Academy));
		}
	}
	//̨//з//
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Academy) >= 1)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Comsat_Station) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Comsat_Station) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Comsat_Station) < 1 &&
			numMinerals > 50 && numGas > 50)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Comsat_Station));
		}
	}

	//ֻغͷֻصĵر//
	BWAPI::Unit choke2Bunker = BWAPI::Broodwar->getClosestUnit(
		InformationManager::Instance().getPosition("chokePoint2"),
		BWAPI::Filter::IsOwned && BWAPI::Filter::GetType == BWAPI::UnitTypes::Terran_Bunker,
		120);

	//if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Vulture) > 2)
	//{
	//	if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Bunker) &&
	//		!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Bunker) &&
	//		!choke2Bunker && numMinerals > 100)
	//	{
	//		_buildSecondBase = true;
	//		queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Bunker));
	//		BuildingPlacer::Instance().setBunkerBuildPlace(InformationManager::Instance().getPosition("chokePoint2"));
	//	}
	//}
	//if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Command_Center) &&
	//	!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Command_Center) &&
	//	choke2Bunker && numMinerals > 200 &&
	//	UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) > 1)
	//{
	//	queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Command_Center));
	//}

	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Vulture) > 2)
	{
		_buildSecondBaseIn = true;

		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Command_Center) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Command_Center) &&
			numMinerals > 200 &&
			UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) > 1)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Command_Center));
		}
	}

	//ʵͣ
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) == 0)
	{
		if (numGas - numMinerals > 100)
		{
			WorkerManager::Instance().setCollectGas(false);
		}
		else if (numGas - numMinerals < 0)
		{
			WorkerManager::Instance().setCollectGas(true);
		}
	}
	else
	{
		WorkerManager::Instance().setCollectGas(true);
	}
	//////////սλı////////////

	CombatCommander::Instance().updateBunkerSquad();
	CombatCommander::Instance().updateChokeSquads();
	CombatCommander::Instance().updateAttackSquads();
	//ڼſڲ
	if (numVultures > 0 && choke2Bunker)
	{
		BWAPI::Position aimPos = CombatCommander::Instance().getNearPosition(InformationManager::Instance().getPosition("chokePoint2"),
			InformationManager::Instance().getPosition("frontLine1"), false, 300);
		CombatCommander::Instance().updateMineAsRowFrontSquad(aimPos, InformationManager::Instance().getPosition("chokePoint2"), 60, 200, 0);
	}
	else
	{
		CombatCommander::Instance().clearSquad("mineRowFrontSquad");
	}
	//һֱ
	CombatCommander::Instance().setAggression(false);

	//////////ģл////////////
	if (numCenters > 1 || supplyUsed > 100)
	{
		doSwitchModule();
	}
}

/*
void StrategyManager::updateModuleB6(BuildOrderQueue & queue)
{
	int numSCVs = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_SCV);
	int numMarines = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Marine);
	int numBunker = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Bunker);
	int numBarracks = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Barracks);
	int numSupplys = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Supply_Depot);
	int numFactorys = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Factory);
	int numTanks = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode)
		+ UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Siege_Mode);
	int numMinerals = BWAPI::Broodwar->self()->minerals();
	int numGas = BWAPI::Broodwar->self()->gas();
	BWAPI::Player self = BWAPI::Broodwar->self();
	bool hasEBay = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Engineering_Bay) > 0;
	int numTurrets = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Missile_Turret);
	int numGoliaths = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Goliath);
	int numVultures = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Vulture);
	int numCenters = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Command_Center);
	int supplyTotal = BWAPI::Broodwar->self()->supplyTotal() / 2;
	int supplyUsed = BWAPI::Broodwar->self()->supplyUsed() / 2;

	////////////////ͽ/////////////////

	if (_lastModule == "A4")
	{
		//·һĵر
		BWAPI::Unit choke1Bunker = BWAPI::Broodwar->getClosestUnit(
			InformationManager::Instance().getPosition("bunkerPoint"),
			BWAPI::Filter::IsOwned && BWAPI::Filter::GetType == BWAPI::UnitTypes::Terran_Bunker,
			100);
		if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Barracks) > 0 && numCenters < 2)
		{
			if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Bunker) &&
				!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Bunker) &&
				!choke1Bunker)
			{
				queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Bunker));
				BuildingPlacer::Instance().setBunkerBuildPlace(InformationManager::Instance().getPosition("bunkerPoint"));
			}

			if (!choke1Bunker && numMinerals < 180) { return; }
		}
	}

	//SCV10ֹͣһȲSCV
	//10ͣSCV50ʱֹͣ
	if (numSCVs < 10)
	{
		if (!queue.anyInNextN(BWAPI::UnitTypes::Terran_SCV, 12))
		{
			queue.queueAsLowestPriority(BWAPI::UnitTypes::Terran_SCV);
		}
		return;
	}
	if (numSCVs < 45 &&
		UnitUtil::GetUncompletedUnitCount(BWAPI::UnitTypes::Terran_SCV) == 0)
	{
		if (!queue.anyInNextN(BWAPI::UnitTypes::Terran_SCV, 12))
		{
			queue.queueAsLowestPriority(BWAPI::UnitTypes::Terran_SCV);
		}
	}

	//Ӫݻ٣޽
	if (numBarracks == 0)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Barracks) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Barracks) &&
			numMinerals > 135)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Barracks));
		}
	}
	//ǹ䵽5
	//ÿ301ǹֱҷܹеλֹԷܱ
	if ((UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Barracks) >= 1 && numMarines < 2) ||
		(UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Barracks) >= 1 && numMarines < 5 && numFactorys > 0))
	{
		if (!queue.anyInNextN(BWAPI::UnitTypes::Terran_Marine, 12) &&
			UnitUtil::GetUncompletedUnitCount(BWAPI::UnitTypes::Terran_Marine) == 0)
		{
			queue.queueAsLowestPriority(BWAPI::UnitTypes::Terran_Marine);
		}
	}
	//
	if (numSCVs + numMarines >= 9)
	{
		BuildingPlacer::Instance().allowBuildSupply = true;
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Supply_Depot) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Supply_Depot) &&
			numSupplys == 0 && numMinerals > 90)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Supply_Depot));
		}
	}

	////ֹݻ
	if (numSCVs >= 11 && numMinerals > 100)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Refinery) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Refinery) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Refinery) < 1)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Refinery));
		}
	}
	////ͬ//A1A4û޹ﲹ
	if (numFactorys == 0 && numGas > 100)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Factory) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Factory))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Factory));
		}
	}
	//ڶ
	if (numFactorys == 1 && numMinerals > 200 && numGas > 100)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Factory) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Factory))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Factory));
		}
	}
	//е//޺úӻе
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) >
		UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop))
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Machine_Shop) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Machine_Shop))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Machine_Shop));
		}
	}

	//׳//̹2󣬿ʼ׳
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) > 0 &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Vulture) &&
		numMinerals >= 75 && numVultures < 4)
	{
		queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Terran_Vulture));
	}
	//׳֩
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) >= 1)
	{
		//B2Ϊ֩
		if (!self->hasResearched(BWAPI::TechTypes::Spider_Mines) &&
			!self->isResearching(BWAPI::TechTypes::Spider_Mines) &&
			numMinerals > 100 && numGas > 100 &&
			!queue.anyInNextN(BWAPI::TechTypes::Spider_Mines, 10))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::TechTypes::Spider_Mines));
		}
	}
	////Armory//˵ǰ//
	if (UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Factory) > 1)
	{
		if (!queue.numInQueue(BWAPI::UnitTypes::Terran_Armory) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Armory) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Armory) < 1 &&
			numMinerals > 100 && numGas > 50)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Armory));
		}
	}
	//еλ
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Armory) > 0 &&
		shouldUpgrade(BWAPI::UpgradeTypes::Terran_Vehicle_Plating) &&
		UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Command_Center) > 0)
	{
		if (!self->isUpgrading(BWAPI::UpgradeTypes::Terran_Vehicle_Plating) &&
			!queue.anyInQueue(BWAPI::UpgradeTypes::Terran_Vehicle_Plating) &&
			!queue.anyInQueue(BWAPI::UpgradeTypes::Terran_Vehicle_Weapons))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UpgradeTypes::Terran_Vehicle_Plating));
		}
	}
	////޺úʼ//ڷ//
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Armory) > 0 &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Goliath) &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode) &&
		numVultures > 3 &&
		UnitUtil::GetUncompletedUnitCount(BWAPI::UnitTypes::Terran_Goliath) == 0 &&
		numMinerals >= 100 && numGas >= 50)
	{
		queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Terran_Goliath));
	}
	////ڶ
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) >= 1 &&
		self->getUpgradeLevel(BWAPI::UpgradeTypes::Charon_Boosters) == 0 &&
		!self->isUpgrading(BWAPI::UpgradeTypes::Charon_Boosters) &&
		!queue.anyInQueue(BWAPI::UpgradeTypes::Charon_Boosters) &&
		numMinerals > 100 && numGas > 100 &&
		numGoliaths > 0)
	{
		queue.queueAsHighestPriority(MacroAct(BWAPI::UpgradeTypes::Charon_Boosters));
	}
	//̹˼//
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) >= 1 &&
		!self->hasResearched(BWAPI::TechTypes::Tank_Siege_Mode) &&
		!self->isResearching(BWAPI::TechTypes::Tank_Siege_Mode) &&
		!queue.anyInNextN(BWAPI::TechTypes::Tank_Siege_Mode, 12) &&
		numTanks > 0)
	{
		queue.queueAsHighestPriority(MacroAct(BWAPI::TechTypes::Tank_Siege_Mode));
	}
	//̹//Ϊ̹//
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) > 0 &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode) &&
		numMinerals >= 150 && numGas >= 100 &&
		numTanks < 1 && numGoliaths > 3)
	{
		queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode));
	}

	////
	if (numFactorys > 1)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Engineering_Bay) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Engineering_Bay) &&
			!hasEBay && numMinerals > 125)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Engineering_Bay));
		}
	}
	//̨
	buildTurret(queue, 3);

	//оԺ//̨ǰ//
	if (UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Command_Center) > 1 &&
		numMinerals > 150)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Academy) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Academy) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Academy) == 0)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Academy));
		}
	}
	//̨//з//
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Academy) >= 1)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Comsat_Station) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Comsat_Station) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Comsat_Station) < 1 &&
			numMinerals > 50 && numGas > 50)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Comsat_Station));
		}
	}

	//ֻغͷֻصĵر//
	BWAPI::Unit choke2Bunker = BWAPI::Broodwar->getClosestUnit(
		InformationManager::Instance().getPosition("chokePoint2"),
		BWAPI::Filter::IsOwned && BWAPI::Filter::GetType == BWAPI::UnitTypes::Terran_Bunker,
		120);

	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) > 1)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Bunker) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Bunker) &&
			!choke2Bunker && numMinerals > 100)
		{
			_buildSecondBase = true;
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Bunker));
			BuildingPlacer::Instance().setBunkerBuildPlace(InformationManager::Instance().getPosition("chokePoint2"));
		}
	}
	if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Command_Center) &&
		!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Command_Center) &&
		choke2Bunker && numMinerals > 400 &&
		UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) > 1)
	{
		queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Command_Center));
	}
	//ʵͣ
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) == 0)
	{
		if (numGas - numMinerals > 100)
		{
			WorkerManager::Instance().setCollectGas(false);
		}
		else if (numGas - numMinerals < 0)
		{
			WorkerManager::Instance().setCollectGas(true);
		}
	}
	else
	{
		WorkerManager::Instance().setCollectGas(true);
	}
	//////////սλı////////////

	CombatCommander::Instance().updateBunkerSquad();
	CombatCommander::Instance().updateChokeSquads();
	CombatCommander::Instance().updateAttackSquads();
	//ڼſڲ
	if (numVultures > 0 && choke2Bunker)
	{
		BWAPI::Position aimPos = CombatCommander::Instance().getNearPosition(InformationManager::Instance().getPosition("chokePoint2"),
			InformationManager::Instance().getPosition("frontLine1"), false, 300);
		CombatCommander::Instance().updateMineAsRowFrontSquad(aimPos, InformationManager::Instance().getPosition("chokePoint2"), 40, 200);
	}
	else
	{
		CombatCommander::Instance().clearSquad("mineRowFrontSquad");
	}
	//һֱ
	CombatCommander::Instance().setAggression(false);

	//////////ģл////////////
	if (numCenters > 1 || supplyUsed > 100)
	{
		doSwitchModule();
	}
}
*/

//B4//3VF Rush//
void StrategyManager::updateModuleB4(BuildOrderQueue & queue)
{
	int numSCVs = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_SCV);
	int numMarines = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Marine);
	int numBunker = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Bunker);
	int numBarracks = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Barracks);
	int numSupplys = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Supply_Depot);
	int numFactorys = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Factory);
	int numTanks = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode)
		+ UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Siege_Mode);
	int numMinerals = BWAPI::Broodwar->self()->minerals();
	int numGas = BWAPI::Broodwar->self()->gas();
	BWAPI::Player self = BWAPI::Broodwar->self();
	bool hasEBay = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Engineering_Bay) > 0;
	int numTurrets = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Missile_Turret);
	int numGoliaths = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Goliath);
	int numVultures = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Vulture);
	int numCenters = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Command_Center);
	int supplyTotal = BWAPI::Broodwar->self()->supplyTotal() / 2;
	int supplyUsed = BWAPI::Broodwar->self()->supplyUsed() / 2;

	////////////////ͽ/////////////////

	//SCV10ֹͣһȲSCV
	//SCVͣ30ʱֹͣ
	//ʱ¿ʼSCVֱ60
	if ((numSCVs < 30 ||
		(numSCVs < 60 && (queue.anyInQueue(BWAPI::UnitTypes::Terran_Command_Center) || UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Command_Center) == 2))) &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Vulture) &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Marine) &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Goliath) &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode))
	{
		if (!queue.anyInNextN(BWAPI::UnitTypes::Terran_SCV, 2) &&
			UnitUtil::GetUncompletedUnitCount(BWAPI::UnitTypes::Terran_SCV) == 0)
		{
			queue.queueAsHighestPriority(BWAPI::UnitTypes::Terran_SCV);
		}
	}
	//Ӫݻ٣޽
	if (numSCVs >= 10 && numBarracks == 0)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Barracks) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Barracks) &&
			numMinerals > 135)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Barracks));
		}
	}
	//ǹ䵽5
	//ÿ301ǹֱҷܹеλֹԷܱ
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Barracks) >= 1 &&
		UnitUtil::GetUncompletedUnitCount(BWAPI::UnitTypes::Terran_Marine) == 0 &&
		(numMarines < 5 || (BWAPI::Broodwar->getFrameCount() % (24 * 50) == 1 && numFactorys < 1)))
	{
		if (!queue.anyInNextN(BWAPI::UnitTypes::Terran_Marine, 2))
		{
			queue.queueAsHighestPriority(BWAPI::UnitTypes::Terran_Marine);
		}
	}
	//
	if (numSCVs + numMarines >= 9)
	{
		BuildingPlacer::Instance().allowBuildSupply = true;
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Supply_Depot) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Supply_Depot) &&
			numSupplys == 0 && numMinerals > 90)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Supply_Depot));
		}
	}

	////A2,A3,A4Ѿ޺ãֻA1ûޣҪﲹ
	if (numSCVs >= 12 && numMinerals > 100)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Refinery) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Refinery) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Refinery) < 1)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Refinery));
		}
	}
	
	////ͬϣֻA1Ҫ//
	if (numFactorys == 0 && numMinerals > 200 && numGas > 100)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Factory) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Factory))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Factory));
		}
	}
	//е//޺úӻе
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) >= 1)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Machine_Shop) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Machine_Shop) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) < 1)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Machine_Shop));
		}
	}
	//׳//е޺ú׳
	if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Vulture) &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Goliath) &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode) &&
		numMinerals >= 75)
	{
		if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) == 1 &&
			UnitUtil::GetUncompletedUnitCount(BWAPI::UnitTypes::Terran_Vulture) == 0)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Vulture));
		}
		else if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) > 1)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Vulture));
		}
	}
	//׳ٶȺ֩
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) >= 1)
	{
		//ٶ
		if (self->getUpgradeLevel(BWAPI::UpgradeTypes::Ion_Thrusters) == 0 &&
			!self->isUpgrading(BWAPI::UpgradeTypes::Ion_Thrusters) &&
			!queue.anyInQueue(BWAPI::UpgradeTypes::Ion_Thrusters) &&
			numMinerals > 100 && numGas > 100)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UpgradeTypes::Ion_Thrusters));
		}
		//Ȼ֩
		else if (!self->hasResearched(BWAPI::TechTypes::Spider_Mines) &&
			!self->isResearching(BWAPI::TechTypes::Spider_Mines) &&
			numMinerals > 100 && numGas > 100 &&
			!queue.anyInNextN(BWAPI::TechTypes::Spider_Mines, 2) &&
			UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) > 1)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::TechTypes::Spider_Mines));
		}
	}
	//Ǯ޵ڶ
	if (numFactorys == 1 && numMinerals > 200 && numGas > 100)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Factory))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Factory));
		}
	}
	//ڶе//ڶ޺ú
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) >= 2)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Machine_Shop) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Machine_Shop) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) < 2)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Machine_Shop));
		}
	}

	//Ǯ޵
	if (numFactorys == 2 && numMinerals > 200 && numGas > 100)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Factory) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Factory))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Factory));
		}
	}

	//̹//ڶ޺ú1̹
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) > 1 &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode) &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Goliath) &&
		numMinerals >= 150 && numGas >= 100 &&
		numTanks < 1 && numVultures > 3)
	{
		queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode));
	}
	////Armory//˵ǰ//
	if (UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Factory) > 2)
	{
		if (!queue.numInQueue(BWAPI::UnitTypes::Terran_Armory) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Armory) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Armory) < 1 &&
			numMinerals > 100 && numGas > 50)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Armory));
		}
	}
	////޺úʼ//ڷ//
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Armory) > 0 &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Goliath) &&
		!queue.anyInQueue(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode) &&
		UnitUtil::GetUncompletedUnitCount(BWAPI::UnitTypes::Terran_Goliath) < 2 &&
		numVultures > 2 &&
		numMinerals >= 100 && numGas >= 50)
	{
		queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Goliath));
	}
	//
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) >= 1 &&
		self->getUpgradeLevel(BWAPI::UpgradeTypes::Charon_Boosters) == 0 &&
		!self->isUpgrading(BWAPI::UpgradeTypes::Charon_Boosters) &&
		!queue.anyInQueue(BWAPI::UpgradeTypes::Charon_Boosters) &&
		numMinerals > 100 && numGas > 100 &&
		numGoliaths > 0)
	{
		queue.queueAsHighestPriority(MacroAct(BWAPI::UpgradeTypes::Charon_Boosters));
	}
	//̹˼//һЩ
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) >= 1 &&
		UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Armory) >= 1 &&
		!self->hasResearched(BWAPI::TechTypes::Tank_Siege_Mode) &&
		!self->isResearching(BWAPI::TechTypes::Tank_Siege_Mode) &&
		!queue.anyInNextN(BWAPI::TechTypes::Tank_Siege_Mode, 12) &&
		numMinerals > 150 && numGas > 150)
	{
		queue.queueAsHighestPriority(MacroAct(BWAPI::TechTypes::Tank_Siege_Mode));
	}
	
	////
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) == 2)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Engineering_Bay) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Engineering_Bay) &&
			!hasEBay && numMinerals > 125)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Engineering_Bay));
		}
	}
	////
	if (hasEBay && numTurrets < 3 && numFactorys > 2)
	{
		BuildingPlacer::Instance().allowBuildTuttet = true;

		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Missile_Turret) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Missile_Turret))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Missile_Turret));
			BuildingPlacer::Instance().setTurretBuildPlace(Bases::Instance().myStartingBase()->getPosition());
		}
	}
	else if (hasEBay && numTurrets >= 2 && numTurrets < 4 && numFactorys > 2)//choke12
	{
		BuildingPlacer::Instance().allowBuildTuttet = true;

		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Missile_Turret) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Missile_Turret))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Missile_Turret));
			BuildingPlacer::Instance().setTurretBuildPlace(InformationManager::Instance().getPosition("chokePoint1"));
		}
	}
	
	//оԺ//̨ǰ//
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) >= 3 &&
		numMinerals > 150)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Academy) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Academy) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Academy) == 0)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Academy));
		}
	}
	//̨//з//
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Academy) >= 1)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Comsat_Station) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Comsat_Station) &&
			UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Comsat_Station) < 1 &&
			numMinerals > 50 && numGas > 50)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Comsat_Station));
		}
	}

	//ֻغͷֻصĵر//
	BWAPI::Unit choke2Bunker = BWAPI::Broodwar->getClosestUnit(
		InformationManager::Instance().getPosition("chokePoint2"),
		BWAPI::Filter::IsOwned && BWAPI::Filter::GetType == BWAPI::UnitTypes::Terran_Bunker,
		100);
	if (numGoliaths > 3)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Bunker) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Bunker) &&
			!choke2Bunker && numMinerals > 100)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Bunker));
			BuildingPlacer::Instance().setBunkerBuildPlace(InformationManager::Instance().getPosition("chokePoint2"));
		}
	}
	if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Command_Center) &&
		!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Command_Center) &&
		choke2Bunker && numMinerals > 400)
	{
		queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Command_Center));
	}
	//ʵͣ
	if (numGas - numMinerals  > 100)
	{
		WorkerManager::Instance().setCollectGas(false);
	}
	else if (numGas - numMinerals  < 0)
	{
		WorkerManager::Instance().setCollectGas(true);
	}

	//////////սλı////////////

	CombatCommander::Instance().updateB3Squad();
	//ڶԷſ
	if (numVultures > 0 &&
		CombatCommander::Instance().getAggression() &&
		Bases::Instance().enemyStart())
	{
		CombatCommander::Instance().updateMineAsRowFrontSquad(InformationManager::Instance().getPosition("frontLine3"),
			InformationManager::Instance().getPosition("frontLine2"), 60, 260, 1);
	}
	else
	{
		CombatCommander::Instance().clearSquad("mineRowFrontSquad");
	}
	//㹻ʼѹƶԷ
	if (numVultures > 3 && numTanks > 0 && numMarines > 3)
	{
		CombatCommander::Instance().setAggression(true);
	}
	else
	{
		CombatCommander::Instance().setAggression(false);
	}

	//////////ģл////////////
	if (numCenters > 1 || supplyUsed > 100)
	{
		doSwitchModule();
	}
}

//C1//ƽ//7.29.2019
void StrategyManager::updateModuleC(BuildOrderQueue & queue)
{
	int numSCVs = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_SCV);
	int numCC = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Command_Center);
	int numRefineries = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Refinery);
	int numMarines = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Marine);
	int numMedics = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Medic);
	int numWraith = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Wraith);
	int numVultures = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Vulture);
	int numVessels = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Science_Vessel);
	int numGoliaths = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Goliath);
	int numTanks = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode)
		+ UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Siege_Mode);
	////////////
	int numBunker = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Bunker);
	int numTurret = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Missile_Turret);
	int comFactory = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory);
	int numBarracks = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Barracks);
	int numArmorys = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Armory);
	int numSupplys = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Supply_Depot);

	int numMinerals = BWAPI::Broodwar->self()->minerals();
	int numGas = BWAPI::Broodwar->self()->gas();
	int supplyUsed = BWAPI::Broodwar->self()->supplyUsed() / 2;

	/////////////
	bool hasEBay = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Engineering_Bay) > 0;
	bool hasAcademy = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Academy) > 0;
	bool hasArmory = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Armory) > 0;
	bool hasFactory = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) > 0;
	bool hasMachineShop = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) > 0;

	int maxSCVs = WorkerManager::Instance().getMaxWorkers();

	bool makeVessel = false;

	BWAPI::Player self = BWAPI::Broodwar->self();

	///////////////λ////////////////

	//õҪӵĵλ[0]׳[1]ˣ[2]̹
	std::vector<int> numAttackUnitAdd = StrategyManager::isMechanicalUnitAdd();
	//SCV
	if (numSCVs < std::min(maxSCVs, 66) && numMinerals < 1000)
	{
		if (queue.numInQueue(BWAPI::UnitTypes::Terran_SCV) < 2 &&
			UnitUtil::GetUncompletedUnitCount(BWAPI::UnitTypes::Terran_SCV) == 0)
		{
			queue.queueAsLowestPriority(BWAPI::UnitTypes::Terran_SCV);
		}
	}
	//ǹ//4פر
	if (numMarines < 4 && CombatCommander::Instance().getDefenseLine() == InformationManager::Instance().getPosition("frontLine1"))
	{
		if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Barracks) >= 1 &&
			UnitUtil::GetUncompletedUnitCount(BWAPI::UnitTypes::Terran_Marine) == 0 &&
			!queue.anyInQueue(BWAPI::UnitTypes::Terran_Marine))
		{
			queue.queueAsLowestPriority(BWAPI::UnitTypes::Terran_Marine);
		}
	}

	//еλ//
	bool anyInQueue = queue.anyInQueue(BWAPI::UnitTypes::Terran_Vulture) ||
		queue.anyInQueue(BWAPI::UnitTypes::Terran_Goliath) ||
		queue.anyInQueue(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode);
	//
	if (numAttackUnitAdd[1] && hasArmory && !anyInQueue)
	{
		queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Terran_Goliath));
	}
	//׳
	if (numAttackUnitAdd[0] && !anyInQueue)
	{
		queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Terran_Vulture));
	}
	//̹
	if (numAttackUnitAdd[2] && !anyInQueue)
	{
		queue.queueAsLowestPriority(MacroAct(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode));
	}

	//Ƽ
		// Maintain 1 vessel to spot for the ground squad and 1 to go with the recon squad.
		if (numVessels < 2 &&
			UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Starport) > 0 &&
			UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Science_Facility) > 0 &&
			UnitUtil::GetUncompletedUnitCount(BWAPI::UnitTypes::Terran_Science_Vessel) == 0 &&
			!queue.anyInQueue(BWAPI::UnitTypes::Terran_Science_Vessel) &&
			numMinerals > 100 && numGas > 225)
		{
			queue.queueAsLowestPriority(BWAPI::UnitTypes::Terran_Science_Vessel);
		}
	

	//////////////////////////////////////////

	//Ӫ//ݻʱ̲
	if (numBarracks < 1)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Barracks) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Barracks) &&
			numMinerals > 135)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Barracks));
		}
	}

	//壬һΪǰ
	if (UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Engineering_Bay) == 0)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Engineering_Bay) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Engineering_Bay) &&
			numMinerals > 125)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Engineering_Bay));
		}
	}
	//оԺһ//̨ǰ
	if (!hasAcademy)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Academy) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Academy) &&
			numMinerals > 150)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Academy));
		}
	}
	//2ͬʱͷ
	if ((numArmorys < 1 && hasFactory) ||
		(supplyUsed > 100 && numArmorys < 2))
	{
		if (!queue.numInQueue(BWAPI::UnitTypes::Terran_Armory) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Armory) &&
			numMinerals > 100 && numGas > 50)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Armory));
		}
	}
	//ر1Ҫ·ڣ
	if (shouldBuildBunker())
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Bunker) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Bunker) &&
			numMinerals > 100)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Bunker));
		}
	}
	// 
	//3Ͷελ
	//ÿ2
	//·2Ҫ2

	if (shouldBuildTurret())
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Missile_Turret) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Missile_Turret))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Missile_Turret));
		}
	}

	//е⣬̹˼֩
	if (comFactory > UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop))
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Machine_Shop) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Machine_Shop) &&
			BWAPI::Broodwar->getFrameCount() % (24 * 30) == 1)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Machine_Shop));
		}
	}
	//̨//Է
	if ((UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Command_Center) >
		UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Comsat_Station)) && 
		BWAPI::Broodwar->getFrameCount() % (24 * 60) == 1)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Comsat_Station) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Comsat_Station) &&
			numMinerals > 50 && numGas > 50)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Comsat_Station));
		}
	}
	//
	if (shouldBuildCenter())
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Command_Center) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Command_Center) &&
			numMinerals > 200)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Command_Center));
		}
 	}
	//
	if (UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Factory) < 6 && _numFreeFactory == 0)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Factory) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Factory) &&
			numMinerals > 300 && numGas > 160)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Factory));
		}
	}
	//ѧվ2еλǰ//Ƽǰ
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Starport) > 0 &&
		UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Science_Facility) == 0 &&
		supplyUsed > 100)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Science_Facility) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Science_Facility) &&
			numMinerals > 100 && numGas > 150)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Science_Facility));
		}
	}
	//
	if (comFactory > 2 && 
		UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Starport) < 1)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Starport) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Starport) &&
			numMinerals > 150 && numGas > 100)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Starport));
		}
	}
	//
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Command_Center) >
		UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Refinery))
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Refinery) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Refinery) &&
			numMinerals > 100)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Refinery));
		}
	}
	//̨//Ƽǰ
	if (UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Starport) > 0 &&
		UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Control_Tower) == 0)
	{
		if (!queue.anyInQueue(BWAPI::UnitTypes::Terran_Control_Tower) &&
			!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Control_Tower) &&
			BWAPI::Broodwar->getFrameCount() % (24 * 20) == 1)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Control_Tower));
		}
	}

	//ʵͣ
	if (numGas - numMinerals  > 1000)
	{
		WorkerManager::Instance().setCollectGas(false);
	}
	else if (numMinerals - numGas > 0)
	{
		WorkerManager::Instance().setCollectGas(true);
	}

	/////////////////////////Ƽ///////////////////////////
	if (hasMachineShop)
	{
		int numUsed = self->isResearching(BWAPI::TechTypes::Spider_Mines) +
			self->isResearching(BWAPI::TechTypes::Tank_Siege_Mode) +
			self->isUpgrading(BWAPI::UpgradeTypes::Ion_Thrusters) +
			self->isUpgrading(BWAPI::UpgradeTypes::Charon_Boosters);

		int numMachineShopLeft = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) - numUsed;
		int numFreeSpace = numMachineShopLeft -
			queue.anyInNextN(BWAPI::TechTypes::Spider_Mines, 10) -
			queue.anyInNextN(BWAPI::UpgradeTypes::Ion_Thrusters, 10) -
			queue.anyInNextN(BWAPI::UpgradeTypes::Charon_Boosters, 10) -
			queue.anyInNextN(BWAPI::TechTypes::Tank_Siege_Mode, 10);

		if (!self->hasResearched(BWAPI::TechTypes::Spider_Mines) &&
			!self->isResearching(BWAPI::TechTypes::Spider_Mines) &&
			numMinerals > 100 && numGas > 100 &&
			!queue.anyInNextN(BWAPI::TechTypes::Spider_Mines, 10) &&
			numFreeSpace > 0)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::TechTypes::Spider_Mines));
		}
		else if (!self->hasResearched(BWAPI::TechTypes::Tank_Siege_Mode) &&
			!self->isResearching(BWAPI::TechTypes::Tank_Siege_Mode) &&
			numMinerals > 100 && numGas > 100 &&
			!queue.anyInNextN(BWAPI::TechTypes::Tank_Siege_Mode, 10) &&
			numFreeSpace > 0)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::TechTypes::Tank_Siege_Mode));
		}
		else if (self->getUpgradeLevel(BWAPI::UpgradeTypes::Ion_Thrusters) == 0 &&
			!self->isUpgrading(BWAPI::UpgradeTypes::Ion_Thrusters) &&
			!queue.anyInQueue(BWAPI::UpgradeTypes::Ion_Thrusters) &&
			numMinerals > 100 && numGas > 100 &&
			numFreeSpace > 0)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UpgradeTypes::Ion_Thrusters));
		}
		else if (self->getUpgradeLevel(BWAPI::UpgradeTypes::Charon_Boosters) == 0 &&
			hasArmory &&
			!self->isUpgrading(BWAPI::UpgradeTypes::Charon_Boosters) &&
			!queue.anyInQueue(BWAPI::UpgradeTypes::Charon_Boosters) &&
			numMinerals > 100 && numGas > 100 &&
			numFreeSpace > 0)
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UpgradeTypes::Charon_Boosters));
		}
	}
	if (hasArmory && shouldUpgrade(BWAPI::UpgradeTypes::Terran_Vehicle_Weapons))
	{
		if (!self->isUpgrading(BWAPI::UpgradeTypes::Terran_Vehicle_Weapons) &&
			!queue.anyInQueue(BWAPI::UpgradeTypes::Terran_Vehicle_Plating) &&
			!queue.anyInQueue(BWAPI::UpgradeTypes::Terran_Vehicle_Weapons))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UpgradeTypes::Terran_Vehicle_Weapons));
		}
	}
	else if (hasArmory && shouldUpgrade(BWAPI::UpgradeTypes::Terran_Vehicle_Plating))
	{
		if (!self->isUpgrading(BWAPI::UpgradeTypes::Terran_Vehicle_Plating) &&
			!queue.anyInQueue(BWAPI::UpgradeTypes::Terran_Vehicle_Plating) &&
			!queue.anyInQueue(BWAPI::UpgradeTypes::Terran_Vehicle_Weapons))
		{
			queue.queueAsHighestPriority(MacroAct(BWAPI::UpgradeTypes::Terran_Vehicle_Plating));
		}
	}
}

const	MetaPairVector	StrategyManager::updateModuleC1()
{
	// the goal to return
	std::vector<MetaPair> goal;

	// These counts include uncompleted units.
	int numSCVs = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_SCV);
	int numCC = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Command_Center);
	int numRefineries = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Refinery);
	int numMarines = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Marine);
	int numMedics = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Medic);
	int numWraith = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Wraith);
	int numVultures = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Vulture);
	int numVessels = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Science_Vessel);
	int numGoliaths = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Goliath);
	int numTanks = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode)
		+ UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Siege_Tank_Siege_Mode);
	////////////
	int numBunker = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Bunker);
	int numTurret = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Missile_Turret);
	int numFactory = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Factory);
	int numBarracks = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Barracks);
	int numArmorys = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Armory);
	int numSupplys = UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Supply_Depot);

	int numMinerals = BWAPI::Broodwar->self()->minerals();
	int numGas = BWAPI::Broodwar->self()->gas();

	/////////////
	bool hasEBay = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Engineering_Bay) > 0;
	bool hasAcademy = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Academy) > 0;
	bool hasArmory = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Armory) > 0;
	bool hasFactory = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Factory) > 0;
	bool hasMachineShop = UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Machine_Shop) > 0;

	int maxSCVs = WorkerManager::Instance().getMaxWorkers();

	bool makeVessel = false;

	BWAPI::Player self = BWAPI::Broodwar->self();

	////////////////////////////////////////////////////////++	
	//ӵλ
	//õҪӵĵλ[0]Ϊǹ[1]׳[2]̹ˣ[3]
	std::vector<int> numAttackUnitAdd = StrategyManager::numAttackUnitAdd();
	//SCV
	goal.push_back(MetaPair(BWAPI::UnitTypes::Terran_SCV, std::min(maxSCVs + 3, numSCVs + 2 * numCC + 1)));

	//ǹ
	if (numAttackUnitAdd[0] && numBarracks > 0)
	{
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Marine, numAttackUnitAdd[0] + numMarines));
	}
	//׳
	if (numAttackUnitAdd[1] && hasFactory)
	{
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Vulture, numAttackUnitAdd[1] + numVultures));
	}
	//̹
	if (numAttackUnitAdd[2] && hasFactory)
	{
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Siege_Tank_Tank_Mode, numAttackUnitAdd[2] + numTanks));
	}
	//
	if (numAttackUnitAdd[3] && hasFactory && hasMachineShop)
	{
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Goliath, numAttackUnitAdd[3] + numGoliaths));
	}
	//Ƽ
	if (makeVessel || InformationManager::Instance().enemyHasCloakTech())
	{
		// Maintain 1 vessel to spot for the ground squad and 1 to go with the recon squad.
		if (numVessels < 2)
		{
			goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Science_Vessel, numVessels + 1));
		}
	}

	//
	//ӪֻҪһӪ
	if (numBarracks < 1)
	{
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Barracks, 1));
	}
	//壬һΪǰ
	if (shouldBuildEBay())
	{
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Engineering_Bay, 1));
	}
	//оԺһΪṇ̀˷ܼ//̨ǰ
	if (!hasAcademy && hasFactory)
	{
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Academy, 1));
	}
	//2ͬʱͷ
	if (numArmorys == 0 && hasFactory)
	{
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Academy, numArmorys + 1));
	}
	else if (numArmorys == 1 && numMinerals > 400 && numGas > 200)
	{
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Academy, numArmorys + 1));
	}

	//ر1Ҫ·ڣ
	if (shouldBuildBunker())
	{
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Bunker, numBunker + 1));
	}
	// 
	//3Ͷελ
	//ÿ2
	//·2Ҫ2
	if (shouldBuildTurret())
	{
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Missile_Turret, numTurret + 1));
	}
	//е⣬̹˼֩
	if (hasFactory && !hasMachineShop)
	{
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Machine_Shop, 1));
	}
	//̨//Է
	if (numCC > 0 && hasAcademy)
	{
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Comsat_Station, UnitUtil::GetCompletedUnitCount(BWAPI::UnitTypes::Terran_Command_Center)));
	}
	//
	if (numCC == 1)
	{
		if (shouldBuildCenter())
		{
			goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Command_Center, numCC + 1));
		}
	}
	else if (numCC < 3)
	{
		if (CombatCommander::Instance().getDefenseLine() != InformationManager::Instance().getPosition("chokePoint2"))
		{
			goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Command_Center, numCC + 1));
		}
	}
	//
	if (numFactory < 1)
	{
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Factory, 1));
	}

	//ѧվ2еλǰ
	if (self->getUpgradeLevel(BWAPI::UpgradeTypes::Terran_Vehicle_Weapons) == 1 &&
		UnitUtil::GetAllUnitCount(BWAPI::UnitTypes::Terran_Science_Facility) == 0)
	{
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UnitTypes::Terran_Science_Facility, 1));
	}

	//Ƽ
	if (hasMachineShop)
	{
		int numAttackUnits = numVultures + numTanks + numGoliaths;

		if (!self->hasResearched(BWAPI::TechTypes::Spider_Mines) && numVultures >= 2)
		{
			goal.push_back(std::pair<MacroAct, int>(BWAPI::TechTypes::Spider_Mines, 1));
		}

		if (!self->hasResearched(BWAPI::TechTypes::Tank_Siege_Mode))
		{
			goal.push_back(std::pair<MacroAct, int>(BWAPI::TechTypes::Tank_Siege_Mode, 1));
		}
		if (self->getUpgradeLevel(BWAPI::UpgradeTypes::Ion_Thrusters) == 0 && numAttackUnits > 15)
		{
			goal.push_back(std::pair<MacroAct, int>(BWAPI::UpgradeTypes::Ion_Thrusters, 1));
		}
		if (self->getUpgradeLevel(BWAPI::UpgradeTypes::Charon_Boosters) == 0 && numAttackUnits > 15)
		{
			goal.push_back(std::pair<MacroAct, int>(BWAPI::UpgradeTypes::Charon_Boosters, 1));
		}
	}
	if (hasArmory && shouldUpgrade(BWAPI::UpgradeTypes::Terran_Vehicle_Weapons))
	{
		int weaponsUps = self->getUpgradeLevel(BWAPI::UpgradeTypes::Terran_Vehicle_Weapons);
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UpgradeTypes::Terran_Vehicle_Weapons, weaponsUps + 1));
	}
	if (hasArmory && shouldUpgrade(BWAPI::UpgradeTypes::Terran_Vehicle_Plating))
	{
		int platUps = self->getUpgradeLevel(BWAPI::UpgradeTypes::Terran_Vehicle_Plating);
		goal.push_back(std::pair<MacroAct, int>(BWAPI::UpgradeTypes::Terran_Vehicle_Plating, platUps + 1));
	}


	return goal;
}

//7.24.2019//жǷезλſ//
bool StrategyManager::enemyUnitsFrontPosition(BWAPI::Position currentPos, BWAPI::Position referencePos, bool isFront)
{
	//ԸõΪģǷезλ
	BWAPI::Position center;
	if (isFront)
	{
		center = CombatCommander::Instance().getNearPosition(currentPos, referencePos, true, 300);
	}
	else
	{
		center = CombatCommander::Instance().getNearPosition(currentPos, referencePos, false, 300);
	}
	BWAPI::Unit enemyUnit = BWAPI::Broodwar->getClosestUnit(
		center,
		BWAPI::Filter::IsEnemy,
		20 * 30);
	if (enemyUnit)
	{
		return true;
	}
	return false;
}

void StrategyManager:: doSwitchModule()
{
	_lastModule = _currentModule;
	if (_currentModule == "A1")
	{
		if (Config::Strategy::StrategyName == "A1-B1-B2-C2")
		{
			_currentModule = "B1";
			CombatCommander::Instance().clearSquad("bunkerSquad2");
		}
		else if (Config::Strategy::StrategyName == "A1-B3-C2")
		{
			_currentModule = "B3";
			CombatCommander::Instance().clearSquad("bunkerSquad2");
		}
	}
	else if (_currentModule == "A2")
	{
		_currentModule = "B6";
		CombatCommander::Instance().clearSquad("bunkerSquad2");
	}
	else if (_currentModule == "A3")
	{
		if (Config::Strategy::StrategyName == "A3-B5-C1")
		{
			_currentModule = "B5";
		}
		else if (Config::Strategy::StrategyName == "A3-B7-C1")
		{
			_currentModule = "B7";
		}
		CombatCommander::Instance().clearSquad("bunkerSquad2");
	}
	else if (_currentModule == "A4")
	{
		_currentModule = "B2";
	}
	else if (_currentModule == "B1")
	{
		_currentModule = "B2";
	}
	else if (_currentModule == "B2")
	{
		_currentModule = "C1";
		///ģ鴴ս///
		CombatCommander::Instance().clearSquad("bunkerSquad2");
		CombatCommander::Instance().clearSquad("chokeSquad1");
		CombatCommander::Instance().clearSquad("chokeSquad2");
		CombatCommander::Instance().clearSquad("chokeSquad3");
		CombatCommander::Instance().clearSquad("chokeSquad4");
		CombatCommander::Instance().clearSquad("mineRowFrontSquad");
		_startModuleC = true;
	}
	else if (_currentModule == "B3")
	{
		_currentModule = "C2";
		///ģ鴴ս///
		CombatCommander::Instance().clearSquad("mineRowFrontSquad");
		CombatCommander::Instance().clearSquad("attackSquad");	
		_startModuleC = true;
	}
	else if (_currentModule == "B4")
	{
		_currentModule = "C2";
		///ģ鴴ս///
		CombatCommander::Instance().clearSquad("mineRowFrontSquad");
		CombatCommander::Instance().clearSquad("attackSquad");
		_startModuleC = true;
	}
	else if (_currentModule == "B5")
	{
		_currentModule = "C1";
		CombatCommander::Instance().clearSquad("chokeSquad1");
		CombatCommander::Instance().clearSquad("chokeSquad2");
		CombatCommander::Instance().clearSquad("chokeSquad3");
		CombatCommander::Instance().clearSquad("chokeSquad4");
		CombatCommander::Instance().clearSquad("mineRowFrontSquad");
		CombatCommander::Instance().clearSquad("harassSquad");

	}
	else if (_currentModule == "B6")
	{
		_currentModule = "C1";
		///ģ鴴ս///
		CombatCommander::Instance().clearSquad("bunkerSquad2");
		CombatCommander::Instance().clearSquad("chokeSquad1");
		CombatCommander::Instance().clearSquad("chokeSquad2");
		CombatCommander::Instance().clearSquad("chokeSquad3");
		CombatCommander::Instance().clearSquad("chokeSquad4");
		CombatCommander::Instance().clearSquad("mineRowFrontSquad");
		_startModuleC = true;
	}
	else if (_currentModule == "B7")
	{
		_currentModule = "C1";
		///ģ鴴ս///
		CombatCommander::Instance().clearSquad("bunkerSquad2");
		CombatCommander::Instance().clearSquad("chokeSquad1");
		CombatCommander::Instance().clearSquad("chokeSquad2");
		CombatCommander::Instance().clearSquad("chokeSquad3");
		CombatCommander::Instance().clearSquad("chokeSquad4");
		CombatCommander::Instance().clearSquad("mineRowFrontSquad");
		CombatCommander::Instance().clearSquad("harassSquad");

		_startModuleC = true;
	}
	return;
}

void StrategyManager::shouldSwitchModule()
{
	int interval = BWAPI::Broodwar->getFrameCount() % (24 * 5);

	if (interval != 1) { return; }

	bool flyingUnits = false;
	int numEnemyUnits = 0;
	for (auto unit : BWAPI::Broodwar->enemy()->getUnits())
	{
		//ԷǷгͼ
		if (unit->getType() == BWAPI::UnitTypes::Zerg_Spire ||
			unit->getType() == BWAPI::UnitTypes::Zerg_Greater_Spire ||
			unit->getType() == BWAPI::UnitTypes::Zerg_Mutalisk)
		{
			flyingUnits = true;
		}
		if (unit->getType() != BWAPI::UnitTypes::Zerg_Drone &&
			unit->getType() != BWAPI::UnitTypes::Terran_SCV &&
			unit->getType() != BWAPI::UnitTypes::Protoss_Probe &&
			unit->getType() != BWAPI::UnitTypes::Buildings)
		{
			numEnemyUnits++;
		}
	}

	int numSelfUnits = 0;

	for (auto unit : BWAPI::Broodwar->self()->getUnits())
	{
		if (unit->getType() != BWAPI::UnitTypes::Terran_SCV &&
			unit->getType() != BWAPI::UnitTypes::Buildings)
		{
			numSelfUnits++;
		}
	}

	//лӦԵзĿеλ
	if (_currentModule == "B5" && flyingUnits)
	{	
		_currentModule == "B7";
		_lastModule == "B5";
		return;
	}

	int score = numSelfUnits - numEnemyUnits;

	//C1C2ģ֮лݵ˫սл
	if (_currentModule == "C1" || _currentModule == "C2")
	{
		//BWAPI::Broodwar->printf("my score is : %d", score);

		if ((score > 80 && _currentModule == "C1") ||
			(score > 30 && _lastModule == "B5" && _currentModule == "C1") ||
			(score > 30 && _lastModule == "B7" && _currentModule == "C1"))
		{
			_currentModule = "C2";
			_lastModule = "C1";
			return;
		}
		else if (score < 36 && _currentModule == "C2" && 
			BWAPI::Broodwar->getFrameCount() / (24 * 60) > 14)
		{
			_currentModule = "C1";
			_lastModule = "C2";
			return;
		}
	}
}