/*
* Tyr is an AI for StarCraft: Broodwar, 
* 
* Please visit https://github.com/SimonPrins/Tyr for further information.
* 
* Copyright 2015 Simon Prins
*
* This file is part of Tyr.
* Tyr is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 2 of the License, or
* (at your option) any later version.
* Tyr is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with Tyr.  If not, see http://www.gnu.org/licenses/.
*/


package com.tyr.unitgroups;

import java.util.ArrayList;

import com.tyr.Tyr;
import com.tyr.agents.Agent;
import com.tyr.buildingplacement.RepairCommand;

import bwapi.Color;
import bwapi.Game;
import bwapi.Player;
import bwapi.Race;


/**
 * Unit group that manages all buildings that are currently under construction.
 * For instance, if the scv building the building is killed, a new one will be sent to finish the work.
 * @author Simon
 *
 */
public class UnderConstruction extends UnitGroup
{
	/**
	 * This list contains all the repair commands for buildings that are under construction whose building scv got killed.
	 */
	ArrayList<RepairCommand> emergencyCommands = new ArrayList<RepairCommand>();
	
	/**
	 * 
	 * Unit group that manages all buildings that are currently under construction.
	 * For instance, if the scv building the building is killed, a new one will be sent to finish the work.
	 * @param rejects The OutOfJob object to which all units will be sent that are no longer needed.
	 */
	public UnderConstruction(OutOfJob rejects)
	{
		super(rejects);
	}

	@Override
	public void onFrame(Game game, Player self, Tyr bot) 
	{
		// See if we can remove any of the commands for continuing construction of a building.
		ArrayList<RepairCommand> removeRepairs = new ArrayList<RepairCommand>();
		for(RepairCommand repair : emergencyCommands)
		{
			repair.target.drawCircle(Color.Teal);
			repair.worker.drawCircle(Color.Teal);
			
			// If the building has been destroyed, the command can be removed.
			if (repair.target.isDead())
			{
				removeRepairs.add(repair);
				continue;
			}
			
			// If the building has been completed or the worker building it has been killed we also remove the command.
			if (repair.target.unit.isCompleted() || repair.worker.isDead())
			{
				removeRepairs.add(repair);
				units.add(repair.target);
				continue;
			}
			
			if (repair.worker.unit.isIdle())
				repair.worker.unit.rightClick(repair.target.unit);
		}
		
		for(RepairCommand repair : removeRepairs)
		{
			emergencyCommands.remove(repair);
			if (!repair.worker.isDead())
				rejects.add(repair.worker);
		}
		
		ArrayList<Agent> completed = new ArrayList<Agent>();
		ArrayList<Agent> repairing = new ArrayList<Agent>();
		for(Agent building : units)
		{
			// If the building has been completed, we can remove it.
			if(building.unit.isCompleted())
			{
				completed.add(building);
				continue;
			}
			
			if (building.unit.getHitPoints() <= 75)
				building.drawCircle(Color.Red, 12);
			if (building.unit.isUnderAttack())
				building.drawCircle(Color.Orange, 16);
			
			if (building.unit.isUnderAttack() && building.unit.getHitPoints() <= 75)
			{
				building.drawCircle(Color.Purple, 20);
				building.unit.cancelConstruction();
				continue;
			}
			
			building.drawCircle(Color.White);
			
			// If we are terran and the constructing scv is dead, we need to send a new one.
			if(self.getRace() == Race.Terran && building.unit.getBuildUnit() == null)
			{
				Agent worker = bot.workForce.pop(building.unit.getPosition());
				if(worker == null)
					continue;
				
				worker.unit.rightClick(building.unit);
				RepairCommand emergencyCommand = new RepairCommand(worker, building);
				emergencyCommands.add(emergencyCommand);
				repairing.add(building);
			}
		}
		
		// We do not keep track of buildings being repaired in our units list, otherwise we would be doing unnecessary checks.
		for(Agent building : repairing)
			units.remove(building);
		
		for(Agent building : completed)
		{
			units.remove(building);
			rejects.add(building);
		}
	}
	
	@Override
	public int takeAgent(Agent agent)
	{
		if(agent.unit.getType().isBuilding() && !agent.unit.isCompleted())
			return 1;
		else
			return -1;
	}
}
