/*
* Tyr is an AI for StarCraft: Broodwar, 
* 
* Please visit https://github.com/SimonPrins/Tyr for further information.
* 
* Copyright 2015 Simon Prins
*
* This file is part of Tyr.
* Tyr is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 2 of the License, or
* (at your option) any later version.
* Tyr is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with Tyr.  If not, see http://www.gnu.org/licenses/.
*/


package com.tyr.tasks;

import java.util.ArrayList;

import com.tyr.DebugMessages;
import com.tyr.EnemyManager;
import com.tyr.Force;
import com.tyr.ForceTracker;
import com.tyr.StopWatch;
import com.tyr.Tyr;
import com.tyr.agents.Agent;
import com.tyr.agents.None;
import com.tyr.unitgroups.AttackGroup;
import com.tyr.unitgroups.UnitGroup;

import bwapi.Color;
import bwapi.Game;
import bwapi.Player;
import bwapi.Position;
import bwapi.Unit;
import bwapi.UnitType;

/**
 * This class implements a solution for sieging at the opponents natural.
 */
public class SiegeNaturalSolution extends Solution implements Force
{
	/**
	 *  The group of attacking units.
	 */
	private AttackGroup attackGroup;
	
	/**
	 *  The target of the attack.
	 */
	private Position attackTarget;
	
	/**
	 * A position at which we want to siege.
	 */
	private Position siegeTarget;
	
	/**
	 * The frame at which we decided to siege up.
	 */
	private  int siegeFrame = -10000;
	
	/**
	 * How long do the tanks remain sieged?
	 */
	public int siegeTime = 400;
	
	/**
	 * How long do the tanks remain sieged after destroying a building?
	 */
	public int siegeBuildingTime = 100;

	/**
	 * List of all tanks that are currently besieging the opponents natural expand.
	 */
	private ArrayList<Agent> besiegingTanks = new ArrayList<Agent>();
	
	/**
	 * The center of this force.
	 */
	private Position center;
	
	/**
	 * The frame when the center was last updated.
	 */
	private int updateFrame;
	
	/**
	 * Are we done with the attack? 
	 */
	private boolean done;
	
	/**
	 * This class implements a solution for sieging at the opponents natural.
	 * @param task The task that started this solution.
	 * @param attackTarget The position which we should attack.
	 */
	public SiegeNaturalSolution(Task task, Position attackTarget) 
	{
		super(task);
		this.attackTarget = attackTarget;
		this.attackGroup = new AttackGroup(attackTarget);
		
		ForceTracker.register(this);
	}
	
	/**
	 * Sets the target which the units will attack.
	 * @param attackTarget
	 */
	public void setTarget(Position attackTarget)
	{
		this.attackTarget = attackTarget;
	}

	@Override
	public void onFrame(Game game, Player self, Tyr bot)
	{
		StopWatch watch = new StopWatch();
		ArrayList<Long> times = new ArrayList<Long>();
		
		watch.start();
		
		// Draw some debug info.
		Tyr.drawCircle(attackTarget, Color.Orange, 64);

		if (besiegingTanks.size() == 0)
			siegeFrame = -10000;
		if (isSieging())
		{
			for (Unit enemy : EnemyManager.getEnemyUnits())
			{
				if (enemy.isCloaked() || enemy.getType().isFlyer())
					continue;
				if (siegeTarget.getDistance(enemy.getPosition()) <= UnitType.Terran_Siege_Tank_Siege_Mode.sightRange())
				{
					if (enemy.getType().isBuilding())
					{
						siegeFrame = Math.min(siegeFrame, game.getFrameCount() + siegeTime - siegeBuildingTime);
						continue;
					}
					siegeFrame = game.getFrameCount();
					break;
				}
			}
		}
		
		times.add(watch.time());
		
		for (Agent agent : attackGroup.units)
		{
			// We already have a target to siege at.
			if (siegeFrame >= game.getFrameCount())
				break;

			if (!agent.isTank())
				continue;
			for (Unit enemy : EnemyManager.getEnemyUnits())
			{
				if (enemy.isCloaked() || enemy.getType().isFlyer() || enemy.getType().isWorker())
					continue;
				if (agent.distanceSquared(enemy) <= UnitType.Terran_Siege_Tank_Siege_Mode.sightRange() * UnitType.Terran_Siege_Tank_Siege_Mode.sightRange())
				{
					siegeTarget = agent.unit.getPosition();
					siegeFrame = game.getFrameCount();
					break;
				}
			}
		}
		
		times.add(watch.time());
		
		// We siege at the siegeTarget.
		if (isSieging())
		{
			Tyr.drawCircle(siegeTarget, Color.Blue, 256);
			game.drawTextMap(siegeTarget.getX(), siegeTarget.getY(), (game.getFrameCount() - siegeFrame) + "");
			for (int i=attackGroup.units.size()-1; i >= 0; i--)
			{
				Agent agent = attackGroup.units.get(i);
				if (!agent.isTank())
					continue;
				
				int distanceSq = agent.distanceSquared(siegeTarget); 
				if ((distanceSq <= 256*256 && agent.unit.getType() == UnitType.Terran_Siege_Tank_Siege_Mode) 
						|| distanceSq <= 96*96)
				{
					attackGroup.units.remove(i);
					agent.order(new None(agent));
					besiegingTanks.add(agent);
				}
			}
		}
		
		times.add(watch.time());

		// When our tanks are close enough to the enemy to lay siege, we have them siege up.
		for (int i=attackGroup.units.size()-1; i >= 0; i--)
		{
			Agent agent = attackGroup.units.get(i);
			if (!agent.isTank())
				continue;
			if (agent.distanceSquared(attackTarget) <= UnitType.Terran_Siege_Tank_Siege_Mode.sightRange() * UnitType.Terran_Siege_Tank_Siege_Mode.sightRange())
			{
				attackGroup.units.remove(i);
				agent.order(new None(agent));
				besiegingTanks.add(agent);
			}
		}
		
		times.add(watch.time());
		
		for (int i = besiegingTanks.size()-1; i >= 0; i--)
		{
			Agent agent = besiegingTanks.get(i);
			if (agent.distanceSquared(attackTarget) > UnitType.Terran_Siege_Tank_Siege_Mode.sightRange() * UnitType.Terran_Siege_Tank_Siege_Mode.sightRange()
					&& (!isSieging()) || agent.distanceSquared(siegeTarget) > 256*256)
			{
				besiegingTanks.remove(i);
				attackGroup.add(agent);
				continue;
			}
			agent.drawCircle(Color.Green);
			if (agent.unit.getType() == UnitType.Terran_Siege_Tank_Tank_Mode)
				agent.unit.siege();
		}
		
		times.add(watch.time());
		
		if (!isSieging())
			for (Agent agent : attackGroup.units)
				if (agent.distanceSquared(attackTarget) <= 10000)
					done = true;
		
		times.add(watch.time());
		
		DebugMessages.addMessage("Siege " + attackGroup.units.size() + " " + besiegingTanks.size() + (isSieging()?(" " + (game.getFrameCount() - siegeFrame)):""));
		
		// Manage the attackGroup.
		attackGroup.cleanup();
		if (isSieging())
			attackGroup.setTarget(siegeTarget);
		else
			attackGroup.setTarget(attackTarget);
		attackGroup.onFrame(game, self, bot);
		
		times.add(watch.time());
		
		for (int i=0; i<times.size(); i++)
			if (times.get(i) > 50)
				DebugMessages.addMessage("SiegeNatural times(" + i + ") = " + times.get(i));
	}
	
	/**
	 * Add an agent to the attack group with the attacking agents.
	 * @param unit The agent to be added.
	 */
	public void add(Agent agent)
	{
		attackGroup.add(agent);
	}
	
	/**
	 * This method is called when this solution is no longer needed.
	 */
	public void done(UnitGroup unitGroup) 
	{
		for(Agent unit : attackGroup.units)
		{
			unitGroup.add(unit);
			unit.order(new None(unit));
		}
		attackGroup.units = new ArrayList<Agent>();
	}
	
	/**
	 * Clears the solution so that it can be removed.
	 */
	public void clear() 
	{
		attackGroup.clear();
	}
	
	/**
	 * Are we currently sieging up?
	 * @return Are we currently sieging up?
	 */
	private boolean isSieging()
	{
		return siegeFrame + siegeTime >= Tyr.game.getFrameCount();
	}

	/**
	 * Returns the number if agents sent out by this solution.
	 * @return Returns the number if agents sent out by this solution.
	 */
	public int size() 
	{
		return attackGroup.units.size();
	}

	@Override
	public Position getCenter()
	{
		if (updateFrame == Tyr.game.getFrameCount())
			return center;
		
		if (isSieging())
			return siegeTarget;
		
		int totalX = 0;
		int totalY = 0;
		for (Agent agent : attackGroup.units)
		{
			totalX += agent.unit.getX();
			totalY += agent.unit.getY();
		}
		center = new Position(totalX / attackGroup.units.size(), totalY / attackGroup.units.size());
		updateFrame = Tyr.game.getFrameCount();
		return center;
	}

	@Override
	public boolean disabled()
	{
		return attackGroup.units.size() + besiegingTanks.size() == 0;
	}

	@Override
	public boolean done()
	{
		return done;
	}
}
