/*
* Tyr is an AI for StarCraft: Broodwar, 
* 
* Please visit https://github.com/SimonPrins/Tyr for further information.
* 
* Copyright 2015 Simon Prins
*
* This file is part of Tyr.
* Tyr is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 2 of the License, or
* (at your option) any later version.
* Tyr is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with Tyr.  If not, see http://www.gnu.org/licenses/.
*/


package com.tyr.tasks;

import java.util.ArrayList;

import com.tyr.DebugMessages;
import com.tyr.EnemyManager;
import com.tyr.Tyr;
import com.tyr.agents.Agent;
import com.tyr.agents.None;
import com.tyr.unitgroups.AttackGroup;
import com.tyr.unitgroups.UnitGroup;

import bwapi.Game;
import bwapi.Player;
import bwapi.Position;


/**
 * This solution simply sends the entire group of units to attack the enemy.
 * AttackRegroupSolution was created to replace this.
 * @author Simon
 *
 */
public class SendTroopsSolution extends TargetableSolution 
{
	/**
	 * The group with all the attack units.
	 */
	private AttackGroup attackGroup;
	
	/**
	 * The target where we are attacking.
	 */
	private Position target;
	
	/**
	 * This solution simply sends the entire group of units to attack the enemy.
	 * AttackRegroupSolution was created to replace this.
	 * 
	 * @param task The task that is using this solution.
	 * @param target The initial target where we attack.
	 */
	public SendTroopsSolution(Task task, Position target) 
	{
		super(task);
		
		this.target = target;
		
		this.attackGroup = new AttackGroup(target);
	}
	
	/**
	 * Set the new target for the attack.
	 * @param target The target where we are going to attack.
	 */
	public void setTarget(Position target)
	{
		this.target = target;
	}

	@Override
	public void onFrame(Game game, Player self, Tyr bot)
	{
		attackGroup.setTarget(target);
		
		attackGroup.setIgnoreWorkers(EnemyManager.getManager().getInvader() != null && !EnemyManager.getManager().getInvader().getType().isWorker());
		
		DebugMessages.addMessage("Defending units size: " + attackGroup.units.size());

		attackGroup.cleanup();
		attackGroup.onFrame(game, self, bot);
	}

	/**
	 * Add an agent who will join the attack.
	 * @param unit The agent who will join the attack.
	 */
	public void add(Agent unit)
	{
		attackGroup.add(unit);
	}
	
	/**
	 * The number of units sent by this solution.
	 * @return The number of units sent by this solution.
	 */
	public int getCount()
	{
		return attackGroup.units.size();
	}
	
	/**
	 * This method is called when this solution is no longer needed.
	 * @param unitGroup The unit group to which we will add all the units.
	 */
	public void done(UnitGroup unitGroup) 
	{
		for(Agent unit : attackGroup.units)
		{
			unitGroup.add(unit);
			unit.order(new None(unit));
		}
		attackGroup.units = new ArrayList<Agent>();
	}
	
	/**
	 * Have we added our units to the scanner previously.
	 */
	private boolean addedToScanner = false;	

	/**
	 * Adds the attacking units to the scanner to start finding the remaining enemy buildings.
	 */
	public void addToScanner() 
	{
		// We do not want to add the units twice.
		if (addedToScanner)
			return;
		addedToScanner = true;
		
		
		for(Agent agent : attackGroup.units)
			Tyr.bot.scanner.add(agent);
	}

	/**
	 * Get the closest squared distance between any of the agents and this position.
	 * @param target The target position for which we want to know the distance.
	 * @return The minimum squared distance.
	 */
	public int distanceSq(Position target)
	{
		int distance = Integer.MAX_VALUE;
		for (Agent agent : attackGroup.units)
			distance = Math.min(distance, agent.distanceSquared(target));
		return distance;
	}

}
