/*
* Tyr is an AI for StarCraft: Broodwar, 
* 
* Please visit https://github.com/SimonPrins/Tyr for further information.
* 
* Copyright 2015 Simon Prins
*
* This file is part of Tyr.
* Tyr is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 2 of the License, or
* (at your option) any later version.
* Tyr is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with Tyr.  If not, see http://www.gnu.org/licenses/.
*/


package com.tyr.tasks;

import java.util.ArrayList;
import java.util.List;

import com.tyr.DebugMessages;
import com.tyr.EnemyManager;
import com.tyr.Tyr;
import com.tyr.agents.Agent;
import com.tyr.agents.None;

import bwapi.Color;
import bwapi.Game;
import bwapi.Player;
import bwapi.Position;
import bwapi.Unit;
import bwapi.WeaponType;

/**
 * This class implements a solution for constantly pushing the opponent.
 */
public class RunbySolution extends SolutionItem
{
	/**
	 * The agents that will be sent to attack.
	 */
	private List<Agent> agents;
	
	private int state = RUNBY;
	
	private static final int RUNBY = 0;
	private static final int HARASS = 1;

	private static int smallRange = 300;
	private static int largeRange = 500;
	
	/**
	 * This class implements a solution for constantly pushing the opponent.
	 * @param task The task that started this solution.
	 * @param agents The agents that will be sent to attack.
	 */
	public RunbySolution(Task task, List<Agent> agents) 
	{
		super(task);
		this.agents = agents;
	}

	@Override
	public void onFrame(Game game, Player self, Tyr bot)
	{
		DebugMessages.addMessage("RunbySolution onFrame.");
		for (int i=agents.size()-1; i>=0; i--)
			if (agents.get(i) == null || agents.get(i).isDead())
				agents.remove(i);
		
		if (agents.size() == 0)
			return;
		
		if (RunbyTask.completed)
		{
			for (Agent agent : agents)
				bot.hobos.add(agent);
			agents = new ArrayList<Agent>();
			return;
		}
		
		Position attackTarget = bot.suspectedEnemy.get(0).getPosition();
		Tyr.drawCircle(attackTarget, Color.Red, 64);
		
		
		for (Agent agent : agents)
			agent.drawCircle(Color.Orange);
		
		if (state == RUNBY)
		{
			DebugMessages.addMessage("Performing Runby.");
			for (Agent agent : agents)
				agent.unit.move(attackTarget);
			
			for (Agent agent : agents)
				if (agent.distanceSquared(attackTarget) <= smallRange * smallRange)
				{
					state = HARASS;
					break;
				}
		}
		else if (state == HARASS)
		{
			boolean targetExists = false;
			DebugMessages.addMessage("Performing Harass.");
			for (Agent agent : agents)
			{
				if (agent.distanceSquared(attackTarget) >= largeRange * largeRange)
				{
					agent.order(new None(agent));
					agent.unit.move(attackTarget);
					continue;
				}
				
				Unit killTarget = null;
				double distance = Double.MAX_VALUE;
				// Priority for attacking the chosen unit.
				// 0 means don't care
				// 1 is for regular buildings (i.e. not defensive structures)
				// 2 is for defensive structures
				// 3 is for workers
				// 4 is for enemy units which can attack back
				int priority = 0;
				
				for (Unit enemy : EnemyManager.getEnemyUnits())
				{
					if (!agent.unit.canAttack(enemy))
						continue;
					if ((enemy.getType().isFlyer() || enemy.isLifted()) 
							&& (agent.unit.getType().airWeapon() == WeaponType.None || agent.unit.getType().airWeapon() == null))
						continue;

					if (enemy.getDistance(attackTarget) >= largeRange - 64)
						continue;
					
					int newPriority = 0;
					if (enemy.getType().isBuilding())
						newPriority = enemy.getType().groundWeapon() != null && enemy.getType().groundWeapon() != WeaponType.None?2:1;
					else if (enemy.getType().isWorker())
						newPriority = 3;
					else if (enemy.getType().groundWeapon() != null && enemy.getType().groundWeapon() != WeaponType.None)
						newPriority = 4;
					else
						newPriority = 0;
					
					if (newPriority < priority || newPriority == 0)
						continue;

					double newDistance = agent.distanceSquared(enemy);
					if (newDistance >= smallRange * smallRange)
						continue;
					if (newPriority == priority && newDistance >= distance)
						continue;
					
					killTarget = enemy;
					distance = newDistance;
					priority = newPriority;
				}
					
				if (killTarget != null)
				{
					targetExists = true;
					if (agent.unit.isInWeaponRange(killTarget))
						agent.attack(killTarget);
					else
					{
						agent.order(new None(agent));
						agent.unit.move(killTarget.getPosition());
					}
				}
				else
				{
					agent.order(new None(agent));
					agent.unit.move(attackTarget);
				}
			}
			
			if (!targetExists)
				RunbyTask.completed = true;
		}
	}

	@Override
	public boolean done()
	{
		return agents.size() == 0;
	}

	@Override
	public void add(Agent agent)
	{
		agents.add(agent);
	}

	@Override
	public void clear()
	{
		agents = new ArrayList<Agent>();
	}

	@Override
	public int size()
	{
		return agents.size();
	}
}
