/*
* Tyr is an AI for StarCraft: Broodwar, 
* 
* Please visit https://github.com/SimonPrins/Tyr for further information.
* 
* Copyright 2015 Simon Prins
*
* This file is part of Tyr.
* Tyr is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 2 of the License, or
* (at your option) any later version.
* Tyr is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with Tyr.  If not, see http://www.gnu.org/licenses/.
*/


package com.tyr.tasks;

import com.tyr.EnemyManager;
import com.tyr.EnemyPosition;
import com.tyr.Settings;
import com.tyr.Tyr;
import com.tyr.agents.Agent;

import bwapi.Game;
import bwapi.Player;
import bwapi.Position;
import bwapi.UnitType;
import bwta.BaseLocation;


/**
 * A task that determines whether we are ready to attack the opponent.
 */
public class PokeTask extends Task
{
	/**
	 * The target position where we want the army to attack.
	 */
	private Position target;
	
	/**
	 * How many units do we send out?
	 */
	public static int required = 5;
	
	/**
	 * How many units must there be in the home group before we send out a harass?
	 */
	public static int defendingUnitsNeeded = 5;
	
	/**
	 * Have we sent out an attack yet?
	 */
	private static boolean attackSent = false;
	
	/**
	 * Is the poke task done?
	 */
	private static boolean done = false;
	
	@Override
	public boolean isRequired(Game game, Player self, Tyr bot) 
	{
		if (!done && attackSent && solution == null)
			done = true;
		
		if (solution != null)
		{
			if (((MarineHarassSolution)solution).size() == 0)
				solution = null;
			else return true;
		}
		
		if (solution == null && Settings.getBlockAttack())
			return false;
		
		if (attackSent)
			return false;
        
		return bot.homeGroup.units.size() >= defendingUnitsNeeded;
	}
	
	@Override
	public void solve(Game game, Player self, Tyr bot)
	{
		// If we find a new target to attack, we set that new target.
		boolean newTarget = acquireTarget(bot); 
		if(newTarget)
			((MarineHarassSolution)solution).setTarget(target);
		
		if (bot.homeGroup.units.size() >= defendingUnitsNeeded && !attackSent)
		{
			for(int i = required-1; i >= 0; i--)
			{
				Agent agent = bot.homeGroup.units.get(i);
				if (agent.unit.getType() != UnitType.Terran_Battlecruiser && agent.unit.getType() != UnitType.Terran_Science_Vessel)
				{
					((MarineHarassSolution)solution).add(agent);
					bot.homeGroup.units.remove(i);
				}
			}
			
			attackSent = true;
		}
		
		// We call the super class, which will send the units out to defend.
		super.solve(game, self, bot);
	}

	@Override
	public void findSolution(Game game, Player self, Tyr bot)
	{
		acquireTarget(bot);
		solution = new MarineHarassSolution(this, target);
	}
	
	/**
	 * See if we need to find a new target to attack.
	 * If so we return true, if the current target is already fine, we return false.
	 */
	public boolean acquireTarget(Tyr bot)
	{
		if (target != null)
		{
	    	for(EnemyPosition p : EnemyManager.getManager().enemyBuildingMemory)
	    	{
	    		if (p.pos.getX() == target.getX() && p.pos.getY() == target.getY())
	    			return false;
	    	}
		}
		
		Position newTarget = null;
    	for(EnemyPosition p : EnemyManager.getManager().enemyBuildingMemory)
    	{
    		newTarget = p.pos;
			break;
    	}

    	if (newTarget == null)
    		for (BaseLocation b : bot.suspectedEnemy) 
    		{
    			newTarget = b.getPosition();
    			break;
    		}
    	
    	if (newTarget == null)
    		return false;
    	
    	if (target == null)
    	{
    		target = newTarget;
    		return true;
    	}
    	
    	
    	if (target.getX() == newTarget.getX() && target.getY() == newTarget.getY())
    		return false;
    	
    	target = newTarget;
    	return true;
	}
	
	/**
	 * Have we sent out an attack yet?
	 * @return Have we sent out an attack yet?
	 */
	public static boolean getAttackSent()
	{
		return attackSent;
	}

	/**
	 * Is the poke task done?
	 * @return Is the poke task done?
	 */
	public static boolean getDone()
	{
		return done;
	}

	/**
	 * Stop this pokeTask.
	 */
	public void done()
	{
		done = true;
		((MarineHarassSolution)solution).done(Tyr.bot.hobos);
	}
	
	/**
	 * Reset the done and attackSent variables so that we can send another attack.
	 */
	public static void reset()
	{
		done = false;
		attackSent = false;
	}

	/**
	 * Disable the attack.
	 */
	public static void disable()
	{
		attackSent = false;
	}
}
