/*
* Tyr is an AI for StarCraft: Broodwar, 
* 
* Please visit https://github.com/SimonPrins/Tyr for further information.
* 
* Copyright 2015 Simon Prins
*
* This file is part of Tyr.
* Tyr is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 2 of the License, or
* (at your option) any later version.
* Tyr is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with Tyr.  If not, see http://www.gnu.org/licenses/.
*/


package com.tyr.tasks;

import java.util.ArrayList;

import com.tyr.Tyr;
import com.tyr.agents.Agent;
import com.tyr.agents.None;
import com.tyr.unitgroups.AttackGroup;
import com.tyr.unitgroups.AttackNaturalGroup;
import com.tyr.unitgroups.DistributeAttackGroup;
import com.tyr.unitgroups.IAttackGroup;
import com.tyr.unitgroups.PrioritizedAttackGroup;
import com.tyr.unitgroups.UnitGroup;
import com.tyr.unitgroups.WallAttackGroup;

import bwapi.Color;
import bwapi.Game;
import bwapi.Player;
import bwapi.Position;

/**
 * This class implements a solution for constantly pushing the opponent.
 */
public class ConstantPushSolution extends Solution
{
	/**
	 *  The group of attacking units.
	 */
	private IAttackGroup attackGroup;
	
	/**
	 *  The target of the attack.
	 */
	private Position attackTarget;

	public static final int ATTACK = 0;
	public static final int PRIORITIZE = 1;
	public static final int ATTACK_NATURAL = 2;
	public static final int DISTRIBUTE = 3;
	public static final int ATTACK_WALL = 4;
	
	
	
	/**
	 * This class implements a solution for constantly pushing the opponent.
	 * @param task The task that started this solution.
	 * @param attackTarget The position which we should attack.
	 */
	public ConstantPushSolution(Task task, Position attackTarget) 
	{
		this(task, attackTarget, ATTACK);
	}
	
	/**
	 * This class implements a solution for constantly pushing the opponent.
	 * @param task The task that started this solution.
	 * @param attackTarget The position which we should attack.
	 * @param mode The mode we use when attacking.
	 */
	public ConstantPushSolution(Task task, Position attackTarget, int mode) 
	{
		super(task);
		this.attackTarget = attackTarget;
		if (mode == PRIORITIZE)
			this.attackGroup = new PrioritizedAttackGroup(attackTarget);
		else if (mode == ATTACK_NATURAL)
			this.attackGroup = new AttackNaturalGroup(attackTarget);
		else if (mode == DISTRIBUTE)
			this.attackGroup = new DistributeAttackGroup(attackTarget);
		else if (mode == ATTACK_WALL)
			this.attackGroup = new WallAttackGroup(attackTarget);
		else
			this.attackGroup = new AttackGroup(attackTarget);
			
	}
	
	/**
	 * Sets the target which the units will attack.
	 * @param attackTarget
	 */
	public void setTarget(Position attackTarget)
	{
		this.attackTarget = attackTarget;
	}

	@Override
	public void onFrame(Game game, Player self, Tyr bot)
	{
		// Draw some debug info.
		Tyr.drawCircle(attackTarget, Color.Orange, 64);

		// Manage the attackGroup.
		attackGroup.cleanup();
		attackGroup.setTarget(attackTarget);
		attackGroup.onFrame(game, self, bot);
	}
	
	/**
	 * Add an agent to the attack group with the attacking agents.
	 * @param unit The agent to be added.
	 */
	public void add(Agent agent)
	{
		attackGroup.add(agent);
	}
	
	/**
	 * This method is called when this solution is no longer needed.
	 */
	public void done(UnitGroup unitGroup) 
	{
		for(Agent unit : attackGroup.units)
		{
			unitGroup.add(unit);
			unit.order(new None(unit));
		}
		attackGroup.units = new ArrayList<Agent>();
	}
	
	/**
	 * Have we added our units to the scanner previously.
	 */
	private boolean addedToScanner = false;	

	/**
	 * Adds the attacking units to the scanner to start finding the remaining enemy buildings.
	 */
	public void addToScanner() 
	{
		// We do not want to add the units twice.
		if (addedToScanner)
			return;
		addedToScanner = true;
		

		for(Agent agent : attackGroup.units)
			Tyr.bot.scanner.add(agent);
	}
	
	/**
	 * Clears the solution so that it can be removed.
	 */
	public void clear() 
	{
		attackGroup.clear();
	}

	/**
	 * Returns the number if agents sent out by this solution.
	 * @return Returns the number if agents sent out by this solution.
	 */
	public int size() 
	{
		return attackGroup.units.size();
	}
	
	public boolean done() 
	{
		return size() == 0;
	}
}
