/*
* Tyr is an AI for StarCraft: Broodwar, 
* 
* Please visit https://github.com/SimonPrins/Tyr for further information.
* 
* Copyright 2015 Simon Prins
*
* This file is part of Tyr.
* Tyr is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 2 of the License, or
* (at your option) any later version.
* Tyr is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with Tyr.  If not, see http://www.gnu.org/licenses/.
*/


package com.tyr.tasks;

import com.tyr.EnemyManager;
import com.tyr.Tyr;
import com.tyr.agents.Agent;
import com.tyr.unitgroups.AttackGroup;

import bwapi.Color;
import bwapi.Game;
import bwapi.Player;
import bwapi.Position;

/**
 * This class implements a solution for harassing the opponents expands with vultures.
 */
public class BoomerangSolution extends Solution 
{
	/**
	 * The unit group of units that perform the harass.
	 */
	private AttackGroup attackGroup;
	
	/**
	 * The position at which the boomerang is aimed.
	 */
	private Position boomerangTarget;
	
	/**
	 * The position of the base we currently intend to harass.
	 */
	private Position target;
	
	/**
	 * Is the boomerang on its way back?
	 */
	private boolean returning = false;
	
	/**
	 * At which frame do we return home, of we are not attacked before then?
	 */
	private int returnFrame;
	
	/**
	 * This class implements a solution for harassing the opponents expands with vultures.
	 * @param task The task that started this solution.
	 * @param boomerangTarget The position at which the boomerang is aimed.
	 */
	public BoomerangSolution(Task task, Position boomerangTarget) 
	{
		super(task);
		this.boomerangTarget = boomerangTarget;
		attackGroup = new AttackGroup(boomerangTarget);
		
		returnFrame = Tyr.game.getFrameCount() + 2000;
	}
	
	@Override
	public void onFrame(Game game, Player self, Tyr bot)
	{
		if (boomerangTarget != null)
			Tyr.drawCircle(boomerangTarget, Color.Blue, 200);
		
		if (!returning && Tyr.game.getFrameCount() >= returnFrame)
		{
			returning = true;
			target = bot.homeGroup.target;
			attackGroup.setTarget(target);
		}
		
		// See if we need to change the target, and if so give the new target to the harass group.
		if (!returning)
		{
			returning = EnemyManager.getManager().getInvaderCount() > 0;
			for (Agent agent : attackGroup.units)
			{
				if (!returning)
					break;
				if (agent.distanceSquared(boomerangTarget) >= 40000)
					returning = false;
			}
			
			if (returning)
			{
				target = bot.homeGroup.target;
				attackGroup.setTarget(target);
			}
		}
		else
		{
			if (target == null || bot.homeGroup.target.getDistance(target) > 100)
			{
				target = bot.homeGroup.target;
				attackGroup.setTarget(target);
			}

			boolean done = true;
			for (Agent agent : attackGroup.units)
			{
				if (target != null && agent.distanceSquared(target) >= 40000)
				{
					done = false;
					break;
				}
			}
			if (done)
				attackGroup.clear();
		}
		
		// Manage the harass group.
		attackGroup.cleanup();
		attackGroup.onFrame(game, self, bot);
	}

	/**
	 * Adds a unit to the group of harassing vultures.
	 * @param agent The unit to be added.
	 */
	public void add(Agent agent)
	{
		attackGroup.add(agent);
	}

	public int size()
	{
		return attackGroup.units.size();
	}	
}
