/*
* Tyr is an AI for StarCraft: Broodwar, 
* 
* Please visit https://github.com/SimonPrins/Tyr for further information.
* 
* Copyright 2015 Simon Prins
*
* This file is part of Tyr.
* Tyr is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 2 of the License, or
* (at your option) any later version.
* Tyr is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with Tyr.  If not, see http://www.gnu.org/licenses/.
*/


package com.tyr.tasks;

import com.tyr.EnemyManager;
import com.tyr.EnemyPosition;
import com.tyr.PositionUtil;
import com.tyr.ScannerListener;
import com.tyr.Settings;
import com.tyr.Tyr;
import com.tyr.agents.Agent;
import com.tyr.buildingplacement.SpaceManager;

import bwapi.Game;
import bwapi.Player;
import bwapi.Position;
import bwapi.UnitType;
import bwta.BaseLocation;


/**
 * A task that determines whether we are ready to attack the opponent.
 */
public class AttackTask extends Task implements ScannerListener
{
	/**
	 * The target position where we want the army to attack.
	 */
	private Position target;
	
	/**
	 * The size of the previous attack that was sent out.
	 */
	private static int previousAttackSize = 0;
	
	/**
	 * Have we sent out an attack yet?
	 */
	private static boolean attackSent = false;
	
	/**
	 * Do we use antiLurkerSolutions against the enemy?
	 */
	public static boolean useAntiLurker;
	
	private static AttackTask attackTask;
	
	/**
	 * Do we prioritize attacking enemy expansions?
	 */
	public static boolean prioritizeExpands = false;
	
	/**
	 * A task that determines whether we are ready to attack the opponent.
	 */
	public AttackTask()
	{
		Settings.addScannerListener(this);
		attackTask = this;
	}
	
	@Override
	public boolean isRequired(Game game, Player self, Tyr bot) 
	{
		if (solution != null)
		{
			if (((SolutionList)solution).size() == 0)
				solution = null;
			else return true;
		}
		
		if (solution == null && Settings.getBlockAttack())
			return false;
        
		return bot.homeGroup.units.size() >= getNextAttackSize() || (self.supplyUsed() >= 400 && bot.homeGroup.units.size() > 0);
	}
	
	/**
	 * Get the desired size for the next attack.
	 * Each consecutive attack is increased by 5 units, up to the determined maximum.
	 * @return
	 */
	private int getNextAttackSize()
	{
		return Math.min(Math.max(previousAttackSize + 5, Settings.getRequiredSize()), Settings.getMaximumSize());
	}
	
	/**
	 * Sets the size of the previously sent attack.
	 */
	private void setPreviousAttackSize(int val)
	{
		previousAttackSize = val;
	}
	
	/**
	 * Adds the attacking units to the scanner to start finding the remaining enemy buildings.
	 */
	public void addToScanner()
	{
		if (solution != null)
			for (Solution sol : ((SolutionList)solution).solutions)
			{
				if (useAntiLurker)
					((AntiLurkerSolution)sol).addToScanner();
				else
					((AttackRegroupSolution)sol).addToScanner();
			}
	}
	
	@Override
	public void solve(Game game, Player self, Tyr bot)
	{
		// If we find a new target to attack, we set that new target.
		boolean newTarget = acquireTarget(bot); 
		if(newTarget)
			for(Solution sol : ((SolutionList)solution).solutions)
			{
				if (useAntiLurker)
					((AntiLurkerSolution)sol).setTarget(target);
				else
					((AttackRegroupSolution)sol).setTarget(target);
			}
		
		if (bot.homeGroup.units.size() >= getNextAttackSize() || (self.supplyUsed() >= 400 && bot.homeGroup.units.size() > 0))
		{
			int i = bot.homeGroup.units.size() - 1;
			while(i >= 0)
			{
				SolutionItem sol;
				if (useAntiLurker)
					sol = new AntiLurkerSolution(this, target);
				else
					sol = new AttackRegroupSolution(this, target);
				for(; i >= 0 && sol.size() < 15; i--)
				{
					Agent agent = bot.homeGroup.units.get(i);
					if (agent.unit.getType() != UnitType.Terran_Battlecruiser && agent.unit.getType() != UnitType.Terran_Science_Vessel)
					{
						sol.add(agent);
						bot.homeGroup.units.remove(i);
					}
				}
				((SolutionList)solution).add(sol);
			}
			
			attackSent = true;
			
			this.setPreviousAttackSize(this.getNextAttackSize());
		}
		
		// We call the super class, which will send the units out to defend.
		super.solve(game, self, bot);
	}

	@Override
	public void findSolution(Game game, Player self, Tyr bot)
	{
		solution = new SolutionList(this);
	}
	
	/**
	 * See if we need to find a new target to attack.
	 * If so we return true, if the current target is already fine, we return false.
	 */
	public boolean acquireTarget(Tyr bot)
	{
		if (prioritizeExpands)
		{
			boolean enemyHasExpands = EnemyManager.getManager().getAllCount(UnitType.Protoss_Nexus)
					+ EnemyManager.getManager().getAllCount(UnitType.Terran_Command_Center)
					+ EnemyManager.getManager().getAllCount(UnitType.Zerg_Hatchery)
					+ EnemyManager.getManager().getAllCount(UnitType.Zerg_Hive)
					+ EnemyManager.getManager().getAllCount(UnitType.Zerg_Lair)
					>= 3;
			
			final Position enemyNatural;
			if (Tyr.bot.suspectedEnemy.size() == 1)
			{
				final BaseLocation enemyBase = SpaceManager.getEnemyNatural();
				if (enemyBase != null)
					enemyNatural = enemyBase.getPosition();
				else
					enemyNatural = null;
			}
			else enemyNatural = null;
			if (!enemyHasExpands && Tyr.bot.suspectedEnemy.size() == 1)
			{
		    	for(EnemyPosition p : EnemyManager.getManager().enemyBuildingMemory)
		    	{
		    		if (!p.type.isResourceDepot())
		    			continue;

		    		if (PositionUtil.distanceSq(Tyr.bot.suspectedEnemy.get(0).getPosition(), p.pos) <= 100 * 100)
		    			continue;
		    		
		    		if (PositionUtil.distanceSq(enemyNatural, p.pos) <= 100 * 100)
		    			continue;
		    		
		    		enemyHasExpands = true;
		    		break;
		    	}
				
			}


    		// We skip the enemy natural and main if he has other expands.
			if (target != null && enemyHasExpands && isNaturalOrMain(target, enemyNatural))
				target = null;
			
			if (target != null)
			{
		    	for(EnemyPosition p : EnemyManager.getManager().enemyBuildingMemory)
		    	{
		    		if (!p.type.isResourceDepot())
		    			continue;

		    		if (EnemyManager.getManager().getAllCount(p.type) > 1
		    				&& bot.suspectedEnemy.size() == 1
		    				&& PositionUtil.distanceSq(p.pos, bot.suspectedEnemy.get(0).getPosition()) <= 10000)
		    			continue;
		    		
		    		if (p.pos.getX() == target.getX() && p.pos.getY() == target.getY())
		    		{
		    			return false;
		    		}
		    	}
			}
	    	for(EnemyPosition p : EnemyManager.getManager().enemyBuildingMemory)
	    	{
	    		if (!p.type.isResourceDepot())
	    			continue;
	    		
	    		if (EnemyManager.getManager().getAllCount(p.type) > 1
	    				&& bot.suspectedEnemy.size() == 1
	    				&& PositionUtil.distanceSq(p.pos, bot.suspectedEnemy.get(0).getPosition()) <= 10000)
	    			continue;
	    		
	    		// We skip the enemy natural and main if he has other expands.
	    		if (enemyHasExpands && isNaturalOrMain(p.pos, enemyNatural))
	    			continue;
	    		
	    		target = p.pos;
	    		return true;
	    	}
		}
		
		if (target != null)
		{
	    	for(EnemyPosition p : EnemyManager.getManager().enemyBuildingMemory)
	    	{
	    		if (p.pos.getX() == target.getX() && p.pos.getY() == target.getY())
	    			return false;
	    	}
		}
		
		Position newTarget = null;
    	for(EnemyPosition p : EnemyManager.getManager().enemyBuildingMemory)
    	{
    		newTarget = p.pos;
			break;
    	}

    	if (newTarget == null)
    		for (BaseLocation b : bot.suspectedEnemy) 
    		{
    			newTarget = b.getPosition();
    			break;
    		}
    	
    	if (newTarget == null)
    		return false;
    	
    	if (target == null)
    	{
    		target = newTarget;
    		return true;
    	}
    	
    	
    	if (target.getX() == newTarget.getX() && target.getY() == newTarget.getY())
    		return false;
    	
    	target = newTarget;
    	return true;
	}

	@Override
	public void onScannerEliminating()
	{
		if (solution != null)
			for (Solution sol : ((SolutionList)solution).solutions)
			{
				if (useAntiLurker)
					((AntiLurkerSolution)sol).clear();
				else
					((AttackRegroupSolution)sol).clear();
			}
		solution = null;
	}
	
	private boolean isNaturalOrMain(Position pos, Position enemyNatural)
	{
		if (Tyr.bot.suspectedEnemy.size() != 1)
			return false;
		if (enemyNatural != null && PositionUtil.distanceSq(pos, enemyNatural) <= 100 * 100)
			return true;
		
		return (PositionUtil.distanceSq(Tyr.bot.suspectedEnemy.get(0).getPosition(), pos) <= 100 * 100);
	}
	
	/**
	 * Have we sent out an attack yet?
	 * @return Have we sent out an attack yet?
	 */
	public static boolean getAttackSent()
	{
		return attackSent;
	}
	
	/**
	 * Have we sent out an attack yet?
	 * @return Have we sent out an attack yet?
	 */
	public static void setAttackSent(boolean val)
	{
		attackSent = val;
	}
	
	/**
	 * Cancel all current attacks.
	 */
	public static void cancelAttacks()
	{
		if (attackTask != null && attackTask.solution != null)
			for (SolutionItem sol : ((SolutionList)attackTask.solution).solutions)
				sol.clear();
	}
}
