/*
 * Tyr is an AI for StarCraft: Broodwar, 
 * 
 * Please visit https://github.com/SimonPrins/Tyr for further information.
 * 
 * Copyright 2015 Simon Prins
 *
 * This file is part of Tyr.
 * Tyr is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * Tyr is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with Tyr.  If not, see http://www.gnu.org/licenses/.
 */


package com.tyr.builds;

import com.tyr.DebugMessages;
import com.tyr.EnemyManager;
import com.tyr.Settings;
import com.tyr.StopWatch;
import com.tyr.Tyr;
import com.tyr.UnitTracker;
import com.tyr.agents.Agent;
import com.tyr.agents.BunkerAgent;
import com.tyr.buildingplacement.DefensiveStructures;
import com.tyr.requirements.Requirement;
import com.tyr.requirements.TechRequirement;
import com.tyr.requirements.UnitRequirement;
import com.tyr.tasks.AssignScience;
import com.tyr.tasks.AttackTask;
import com.tyr.tasks.CorsairHarassTask;
import com.tyr.tasks.ScoutExpansionsTask;
import com.tyr.unitgroups.AttackGroup;
import com.tyr.unitgroups.HomeGroup;
import com.tyr.unitgroups.MineralWorkers;

import bwapi.Game;
import bwapi.Player;
import bwapi.TechType;
import bwapi.Unit;
import bwapi.UnitType;
import bwapi.UpgradeType;

/**
 * Implements a bio build designed to stop Killerbot.
 * 
 * @author Simon
 * 
 */
public class TvZFE extends CompositeBuildOrder {
	/**
	 * The minimum size the army needs to be for the first attack.
	 */
	private int minimumArmySize = 10;

	/**
	 * The maximum size the army needs to be for an attack.
	 */
	private int maximumArmySize = 50;
	
	/**
	 * Is the opponent going for a ling rush?
	 */
	private boolean lingRush = false;
	
	/**
	 * Do we turtle on one base for a while?
	 */
	private boolean turtle = false;

	/**
	 * Are we expecting lurkers.
	 */
	private boolean lurkersDetected = false;

	/**
	 * Have we encountered a large number of lurkers.
	 */
	private boolean massLurkers = false;
	
	/**
	 * Is the opponent going for some rush involving tech.
	 * E.G. lurker rush, muta rush.
	 */
	private boolean techRush = false;
	
	/**
	 * The highest number of lings that has ever existed for the opponent.
	 */
	private int mostOpponentLings;
	
	/**
	 * Is the first attack completed?
	 */
	boolean firstAttackDone = false;
	
	/**
	 * A task for scouting for enemy expansions.
	 */
	private ScoutExpansionsTask scoutExpansionsTask = null;

	@Override
	public void initialize(Game game, Player self, Tyr bot) {
		Settings.setRequiredSize(minimumArmySize);
		Settings.setMaximumSize(maximumArmySize);
		
		Settings.setLargeInvasionDist(720);
		Settings.setSmallInvasionDist(480);

		Settings.setWorkersPerGas(1);
		
		AttackTask.useAntiLurker = true;
		AttackTask.prioritizeExpands = true;
		AttackGroup.fearLurkers = true;
		
		HomeGroup.keepTanksSieged = true;

		this.add(new SupplyDepotPart());
		this.add(new WorkerScoutPart(1000).doubleScout());
		this.add(new ExpandPart(true));
		ExpandPart.maximumCcs = 2;

		this.add(new GenericPartFactory(UnitType.Terran_Barracks)
				.setMax(1)
				.setCost(100, 0)
				.create());

		this.add(new GenericPartFactory(UnitType.Terran_Barracks)
				.setMax(2)
				.setCost(100, 0)
				.add(new UnitRequirement(UnitType.Terran_Command_Center, 2))
				.create());

		this.add(new GenericPartFactory(UnitType.Terran_Barracks)
				.setMax(2)
				.setCost(100, 0)
				.add(new Requirement() {

					@Override
					public boolean met()
					{
						return lingRush && count(BUNKER) > 0;
					}
					
				})
				.create());
		
		this.add(new GenericPartFactory(UnitType.Terran_Barracks)
		.setMax(3)
		.add(new Requirement() {

			@Override
			public boolean met()
			{
				return turtle;
			}
			
		})
		.create());

		this.add(new GenericPartFactory(BARRACKS)
		.setMax(3)
		.add(new UnitRequirement(ACADEMY))
		.add(new UnitRequirement(UnitType.Terran_Command_Center, 2))
		.create());

		this.add(new GenericPartFactory(UnitType.Terran_Barracks)
		.setMax(5)
		.setCost(200, 0)
		.add(new UnitRequirement(ENGINEERING_BAY))
		.add(new UnitRequirement(ACADEMY))
		.add(new UnitRequirement(UnitType.Terran_Command_Center, 2))
		.create());
		
		this.add(new GenericPartFactory(UnitType.Terran_Barracks)
		.setMax(8)
		.setCost(400, 0)
		.add(new UnitRequirement(FACTORY))
		.add(new UnitRequirement(STARPORT))
		.add(new UnitRequirement(UnitType.Terran_Command_Center, 2))
		.create());
		
		this.add(new GenericPartFactory(UnitType.Terran_Barracks)
		.setMax(8)
		.setCost(250, 0)
		.add(new UnitRequirement(FACTORY))
		.add(new UnitRequirement(SCIENCE_VESSEL))
		.add(new UnitRequirement(UnitType.Terran_Command_Center, 2))
		.create());
		

		this.add(new GenericPartFactory(UnitType.Terran_Factory)
				.add(new UnitRequirement(BARRACKS, 2))
				.add(new UnitRequirement(ENGINEERING_BAY))
				.add(new UnitRequirement(MISSILE_TURRET))
				.add(new UnitRequirement(COMSAT))
				.add (new Requirement() {
					
					@Override
					public boolean met() {
						return lurkersDetected;
					}
				})
				.create());
		

		this.add(new GenericPartFactory(UnitType.Terran_Factory)
				.setMax(2)
				.add(new UnitRequirement(BARRACKS, 5))
				.add(new UnitRequirement(SCIENCE_VESSEL))
				.add(new UnitRequirement(UnitType.Terran_Command_Center, 2))
				.add (new Requirement() {
					
					@Override
					public boolean met() {
						return massLurkers;
					}
				})
				.create());
		
		this.add(new GenericPartFactory(UnitType.Terran_Factory)
		.add(new UnitRequirement(BARRACKS, 5))
		.add(new UnitRequirement(UnitType.Terran_Command_Center, 2))
		.create());
		
		this.add(new GenericPartFactory(UnitType.Terran_Starport)
				.add(new UnitRequirement(UnitType.Terran_Factory, 1, true))
				.add(new TechRequirement(TechType.Tank_Siege_Mode, true))
				.create());
		
		this.add(new GenericPartFactory(ACADEMY)
		.add(new UnitRequirement(BARRACKS, 2))
		.add(new UnitRequirement(REFINERY)).create());
		
		this.add(new GenericPartFactory(UnitType.Terran_Engineering_Bay)
				.add(new UnitRequirement(ACADEMY)).create());
		
		this.add(new GenericPartFactory(UnitType.Terran_Engineering_Bay)
				.add(new UnitRequirement(BARRACKS, 2))
				.add(new Requirement() {
					
					@Override
					public boolean met() {
						return lurkersDetected;
					}
				}).create());

		this.add(new GenericPartFactory(UnitType.Terran_Science_Facility).add(
				new UnitRequirement(UnitType.Terran_Starport, 1, true))
				.create());

		bot.taskManager.potentialTasks.add(new AssignScience());
		//bot.taskManager.potentialTasks.add(new WraithDefenseTask());
		bot.taskManager.potentialTasks.add(new CorsairHarassTask());
		
		Tyr.bot.groups.remove(Tyr.bot.swarm);

		super.initialize(game, self, bot);
	}

	@Override
	public void onFrame(Game game, Player self, Tyr bot) {
		DebugMessages.addMessage("TvZ build.");

		DebugMessages.addMessage("Most enemy zerglings: " + mostOpponentLings);
		
		StopWatch watch = new StopWatch();
		long time;
		watch.start();

		super.onFrame(game, self, bot);

		time = watch.time();
		if (time > 55)
			DebugMessages.addMessage("Bio super time: " + time);

		if (EnemyManager.getManager().getEnemyTypes().contains(ZERGLING)
				&& completed(CC) < 2
				&& completed(MARINE) < 5)
			lingRush = true;
		
		if (EnemyManager.getManager().getEnemyTypes().contains(POOL)
				&& game.getFrameCount() <= 3500)
			lingRush = true;
		
		if (lingRush && game.getFrameCount() <= 4500
				&& EnemyManager.getManager().getEnemyTypes().contains(EXTRACTOR))
			techRush = true;
		
		if (completed(BUNKER) > 0 
				&& completed(MISSILE_TURRET) > 0 
				&& completed(COMSAT) > 0)
			techRush = false;
		
		
		mostOpponentLings = Math.max(mostOpponentLings, EnemyManager.getManager().getAllCount(ZERGLING));
		
		if (mostOpponentLings >= 12 
				&& !firstAttackDone
				&& Settings.getRequiredSize() == 10)
		{
			firstAttackDone = true;
			Settings.setRequiredSize(25);
		}

		if (ExpandPart.maximumCcs <= 2 
				&& UnitTracker.getCcCount() >= 2
				&& UnitTracker.countTanks() > 0
				&& completed(SCIENCE_FACILITY) > 0
				&& completed(SCIENCE_VESSEL) > 0)
		{
			int patchCount = 0;
			for(MineralWorkers base : bot.workForce.mineralWorkers)
				patchCount += base.minerals.size();
			
			if (patchCount < 14)
				ExpandPart.maximumCcs = 100;
		} else if (ExpandPart.maximumCcs <= 2)
		{
			if (techRush
					|| (turtle && count(MARINE) < 12)
					|| (lurkersDetected && completedTanks() == 0)
					|| lurkersPresent())
				ExpandPart.maximumCcs = 1;
			else
				ExpandPart.maximumCcs = 2;
		}
		
		if (scoutExpansionsTask == null && firstAttackDone && AttackTask.getAttackSent())
		{
			scoutExpansionsTask = new ScoutExpansionsTask();
			bot.taskManager.potentialTasks.add(scoutExpansionsTask);
		}
		
		/*
		if (game.getFrameCount() <= 4100
				&& EnemyManager.getManager().getInvader() != null 
				&& EnemyManager.getManager().getInvader().getType() == ZERGLING 
			)
			turtle = true;
		*/
		
		boolean disableBunkers = (UnitTracker.getCcCount() > 1 && count(MARINE) >= 8) 
				|| (self.completedUnitCount(UnitType.Terran_Marine) + self.completedUnitCount(UnitType.Terran_Medic) >= 10 && !AttackTask.getAttackSent());
		
		DebugMessages.addMessage("Disabling bunkers: " + disableBunkers);
		for (Unit structure : bot.defensiveStructures.get(0).defences)
		{
			if (structure.getType() != UnitType.Terran_Bunker)
				continue;
			BunkerAgent bunker = (BunkerAgent)Tyr.bot.agentMap.get(structure.getID());
			bunker.disabled = disableBunkers;
		}

		if (!firstAttackDone) {
			
			if (!EnemyManager.getManager().enemyDefensiveStructures.isEmpty())
			{
				firstAttackDone = true;
				Settings.setRequiredSize(25);
				AttackTask.setAttackSent(false);
				AttackTask.cancelAttacks();
			}
		}
		
		
		
		if (EnemyManager.getManager().getEnemyTypes().contains(LURKER)
				|| EnemyManager.getManager().getEnemyTypes().contains(HYDRA)
				|| EnemyManager.getManager().getEnemyTypes().contains(HYDRA_DEN)
				|| EnemyManager.getManager().getEnemyTypes().contains(LURKER_EGG))
			lurkersDetected = true;
		
		
		if (EnemyManager.getManager().getAllCount(LURKER)
				+ EnemyManager.getManager().getAllCount(LURKER_EGG) >= 8)
			massLurkers = true;
		
		if (lingRush && count(MARINE) < 10)
			DebugMessages.addMessage("Ling rush detected.");
		
		if (lingRush && completed(CC) > 1 && completed(MARINE) < 5)
		{
			for (Unit unit : self.getUnits())
			{
				if (unit.isBeingConstructed() && unit.getType().isResourceDepot())
				{
					unit.cancelConstruction();
					break;
				}
			}
		}
		
		if (UnitTracker.count(FACTORY) > 0
				&& bot.getAvailableGas() < 400)
			Settings.setWorkersPerGas(3);
		else if (UnitTracker.count(UnitType.Terran_Factory) > 0
				|| UnitTracker.count(UnitType.Terran_Engineering_Bay) > 0)
			Settings.setWorkersPerGas(2);

		time = watch.time();
		if (time > 55)
			DebugMessages.addMessage("Bio tier 1 time: " + time);

		if (bot.getAvailableMinerals() >= 100
				&& UnitTracker.getGeyserCount() > 0
				&& UnitTracker.count(UnitType.Terran_Command_Center) >= 2
				&& UnitTracker.count(UnitType.Terran_Barracks) >= 2
				&& UnitTracker.count(UnitType.Terran_Marine) >= 6
				&& (UnitTracker.count(UnitType.Terran_Refinery) == 0 || UnitTracker
						.count(UnitType.Terran_Factory) > 0 || minerals() >= 500)) {
			bot.spaceManager.build(UnitType.Terran_Refinery);
		}

		if (bot.getAvailableMinerals() >= 100
				&& UnitTracker.getGeyserCount() > 0
				&& UnitTracker.count(UnitType.Terran_Barracks) >= 2
				&& count(BUNKER) > 0
				&& count(MARINE) >= 3
				&& techRush
				&& UnitTracker.count(UnitType.Terran_Refinery) == 0) {
			bot.spaceManager.build(UnitType.Terran_Refinery);
		}

		time = watch.time();
		if (time > 55)
			DebugMessages.addMessage("Bio refinery: " + time + " geyserCount: "
					+ UnitTracker.getGeyserCount());

		boolean buildDetectionInMain = (lurkersDetected || techRush) && completed(MISSILE_TURRET) == 0;
		
		
		// Build bunkers.
		if (bot.getAvailableMinerals() >= 50
				&& completed(BARRACKS) >= 1
				&& count(MARINE) >= 1
				&& count(BUNKER) < 2)
		{
			for(DefensiveStructures structures : bot.defensiveStructures)
			{
				int count = structures.getUnitCount(UnitType.Terran_Bunker);
				
				if(count == 0)
				{
					// We do not build a bunker at our main (unless we are getting ling rushed).
					if (structures.defendedPosition.getDistance(Tyr.tileToPosition(self.getStartLocation())) <= 200)
						continue;
					
					bot.spaceManager.buildDefensive(UnitType.Terran_Bunker, structures);
		  			break;
				}
			}
		}


		// Build a bunker in the main when getting ling rushed.
		if (bot.getAvailableMinerals() >= 50  
				&& self.completedUnitCount(UnitType.Terran_Barracks) >= 1
				&& UnitTracker.count(UnitType.Terran_Bunker) < 1
				&& (lingRush || buildDetectionInMain)
				&& bot.defensiveStructures.size() > 0)
		{
			bot.spaceManager.buildDefensive(UnitType.Terran_Bunker, bot.defensiveStructures.get(0));
		}
		
		time = watch.time();
		if (time > 55)
			DebugMessages.addMessage("Bio bunker time: " + time);
		
		if (bot.defensiveStructures.size() == 1
				&& techRush
				&& completed(ENGINEERING_BAY) >= 1
				&& minerals() >= 75
				&& count(BUNKER) > 0
				&& count(MISSILE_TURRET) == 0)
			bot.spaceManager.buildDefensive(MISSILE_TURRET, bot.defensiveStructures.get(0));

		if(completed(ENGINEERING_BAY) >= 1 && minerals() >= 75)
		{
			for(DefensiveStructures structures : bot.defensiveStructures)
			{
				// We do not build a missile turret at our main.
				if (!turtle && !lurkersDetected  && structures.defendedPosition.getDistance(Tyr.tileToPosition(self.getStartLocation())) <= 200)
					continue;
				
				int count = structures.getUnitCount(MISSILE_TURRET);
				
				if(count < 1 && structures.defences.size() > 0)
				{
		  			bot.spaceManager.buildDefensive(MISSILE_TURRET, structures);
		  			break;
				}
			}
		}

		time = watch.time();
		if (time > 55)
			DebugMessages.addMessage("Bio turrets time: " + time);
	}

	private boolean lurkersPresent() 
	{
		for (Unit enemy : EnemyManager.getEnemyUnits())
			if (enemy.getType() == LURKER)
				return true;
		return false;
	}

	@Override
	public boolean overrideStructureOrder(Game game, Player self, Tyr bot, Agent agent)
	{
		if(super.overrideStructureOrder(game, self, bot, agent))
			return true;
		
		if (agent.unit.getType() == UnitType.Terran_Barracks)
		{
			if (UnitTracker.count(UnitType.Terran_Command_Center) < 2 
					&& bot.getAvailableMinerals() < 400 
					&& !lingRush
					&& count(MARINE) > 0)
				return true;
			
			if (lingRush && count(CC) < 2)
			{
				if ((count(MARINE) + count(MEDIC) < 9 || turtle || mostOpponentLings > 6 || techRush)
					&& (count(MARINE) + count(MEDIC) < 13  || turtle) 
					&& count(MARINE) + count(MEDIC) < 15
					&& !agent.unit.isTraining() 
					&& minerals() >= 50)
				{
					if (completed(ACADEMY) > 0
						&& gas() >= 25
						&& count(MEDIC) * 4 < count(MARINE)
						&& count(MARINE) >= 6
						)
						agent.unit.train(MEDIC);
					else
						agent.unit.train(MARINE);
				}
				return true;
			}
			
			if ((UnitTracker.count(UnitType.Terran_Siege_Tank_Siege_Mode) + UnitTracker.count(UnitType.Terran_Siege_Tank_Siege_Mode) == 0
					|| (!self.isResearching(TechType.Tank_Siege_Mode) && !self.hasResearched(TechType.Tank_Siege_Mode)))
					&& bot.getAvailableMinerals() < 200
					&& UnitTracker.count(UnitType.Terran_Marine) >= 5
					&& lurkersDetected)
				return true;
			
			if (!agent.unit.isTraining() 
					&& Tyr.bot.getAvailableGas() >= 25 && Tyr.bot.getAvailableMinerals() >= 50
					&& self.completedUnitCount(UnitType.Terran_Academy) > 0)
			{
				if (UnitTracker.count(UnitType.Terran_Medic) * 4 < 
						UnitTracker.count(UnitType.Terran_Marine) - 4 * UnitTracker.count(UnitType.Terran_Bunker))
				{
					agent.unit.train(UnitType.Terran_Medic);
					return true;
				}
			}
		}
		else if (agent.unit.getType() == UnitType.Terran_Academy)
		{
			if (bot.getAvailableGas() >= 150 && bot.getAvailableMinerals() >= 150
					&& !agent.unit.isResearching() && self.getUpgradeLevel(UpgradeType.U_238_Shells) == 0)
			{
				agent.unit.upgrade(UpgradeType.U_238_Shells);
			} else if (bot.getAvailableMinerals() >= 100 && bot.getAvailableGas() >= 100 
					&& !agent.unit.isResearching() && !self.hasResearched(TechType.Stim_Packs))
			{
				agent.unit.research(TechType.Stim_Packs);
			}
			return true;
		}
		else if (agent.unit.getType() == UnitType.Terran_Starport)
		{
			if (UnitTracker.count(UnitType.Terran_Command_Center) < 2 && bot.getAvailableMinerals() < 400)
				return true;
			if (UnitTracker.count(UnitType.Terran_Science_Vessel) >= 2 && UnitTracker.countTanks() >= 4 && UnitTracker.count(UnitType.Terran_Armory) == 0)
				return true;
			if (agent.unit.getAddon() != null)
			{
				if (!agent.unit.isTraining()
						&& UnitTracker.count(UnitType.Terran_Science_Facility) >0
						&& bot.getAvailableMinerals() >= 100 
						&& bot.getAvailableGas() >= 225)
					agent.unit.train(UnitType.Terran_Science_Vessel);
				return true;
			}
			
			if (UnitTracker.count(UnitType.Terran_Control_Tower) == 0)
			{
				if (UnitTracker.count(UnitType.Terran_Science_Facility) > 0
						&& bot.getAvailableMinerals() >= 50 && bot.getAvailableGas() >= 50)
					agent.unit.buildAddon(UnitType.Terran_Control_Tower);
				return true;
			}
			
			return UnitTracker.count(UnitType.Terran_Science_Vessel) == 0;
		}
		else if (agent.unit.getType() == UnitType.Terran_Factory)
		{
			if (UnitTracker.count(UnitType.Terran_Command_Center) < 2 && bot.getAvailableMinerals() < 400)
				return true;
			if (UnitTracker.count(UnitType.Terran_Siege_Tank_Siege_Mode) + UnitTracker.count(UnitType.Terran_Siege_Tank_Tank_Mode) < 2)
				return false;
			if (!self.hasResearched(TechType.Irradiate)
					&& bot.getAvailableGas() < 350
					&& UnitTracker.countTanks() >= 4)
				return true;

			return UnitTracker.countTanks() >= (massLurkers ? 12 : 4);
		}
		else if (agent.unit.getType() == UnitType.Terran_Science_Facility)
		{
			if (!self.hasResearched(TechType.Irradiate)
					&& bot.getAvailableMinerals() >= 200 
					&& bot.getAvailableGas() >= 200
					&& UnitTracker.count(UnitType.Terran_Science_Vessel) > 0)
			{
				agent.unit.research(TechType.Irradiate);
				return true;
			}
			return false;
		}
		else if (agent.unit.getType() == UnitType.Terran_Engineering_Bay)
		{
			return false;
		}
		
		return false;
	}
}
