/*
* Tyr is an AI for StarCraft: Broodwar, 
* 
* Please visit https://github.com/SimonPrins/Tyr for further information.
* 
* Copyright 2015 Simon Prins
*
* This file is part of Tyr.
* Tyr is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 2 of the License, or
* (at your option) any later version.
* Tyr is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with Tyr.  If not, see http://www.gnu.org/licenses/.
*/


package com.tyr.builds;

import com.tyr.DebugMessages;
import com.tyr.EnemyManager;
import com.tyr.Settings;
import com.tyr.StopWatch;
import com.tyr.Tyr;
import com.tyr.UnitTracker;
import com.tyr.agents.Agent;
import com.tyr.buildingplacement.DefensiveStructures;
import com.tyr.tasks.ConstantPushTask;
import com.tyr.tasks.VultureHarassTask;
import com.tyr.unitgroups.AttackGroup;

import bwapi.Game;
import bwapi.Player;
import bwapi.Race;
import bwapi.TechType;
import bwapi.Unit;
import bwapi.UnitType;
import bwapi.UpgradeType;


/**
 * A build designed to counter an early zealot push.
 * @author Simon
 *
 */
public class TAntiZealot extends CompositeBuildOrder
{
	/**
	 * Do we commit completely to defending the zealot push?
	 */
	public boolean commit;
	
	/**
	 * Do we build a bunker?
	 */
	public boolean buildBunker = true;
	
	/**
	 * Do we expand?
	 */
	public boolean expand = false;
	
	/**
	 * What race is the enemy?
	 */
	public Race enemyRace = Race.Unknown;
	
	/**
	 * Have we encountered a dragoon?
	 */
	private boolean dragoonEncountered = false;
	
	/**
	 * Do we use tanks?
	 */
	public boolean goTank = false;
	
	/**
	 * The vulture harass task.
	 */
	private VultureHarassTask vultureHarass;
	
	/**
	 * The constant push task.
	 */
	private ConstantPushTask constantPushTask;
	
	/**
	 * A build designed to counter an early zealot push.
	 */
	public TAntiZealot() {}

	/**
	 * A build designed to counter an early zealot push.
	 * @param commit Do we commit completely to defending the zealot push?
	 */
	public TAntiZealot(boolean commit)
	{
		this.commit = commit;
	}

	/**
	 * A build designed to counter an early zealot push.
	 * @param commit Do we commit completely to defending the zealot push?
	 * @param buildBunker Do we build a bunker?
	 * @param expand Do we expand?
	 * @param goTank Do we build tanks?
	 */
	public TAntiZealot(boolean commit, boolean buildBunker, boolean expand, boolean goTank)
	{
		this.commit = commit;
		this.buildBunker = buildBunker;
		this.expand = expand;
		this.goTank = goTank;
	}
	
	@Override
	public void initialize(Game game, Player self, Tyr bot)
	{
		this.add(new SupplyDepotPart());
		this.add(new WorkerScoutPart(1600));
		if (expand)
		{
			this.add(new ExpandPart(true));
			ExpandPart.maximumCcs = 1;
		}
		
		if (commit)
		{
			constantPushTask = new ConstantPushTask();
			bot.taskManager.potentialTasks.add(constantPushTask);
			
			AttackGroup.offensiveMines = true;
			
			vultureHarass = new VultureHarassTask(true);
			bot.taskManager.potentialTasks.add(vultureHarass);
			VultureHarassTask.defendingUnitsNeeded = 4;
		}
		
		super.initialize(game, self, bot);
	}
	
	@Override
	public void onFrame(Game game, Player self, Tyr bot)
	{
		DebugMessages.addMessage("Countering Zealot push.");
		
		StopWatch watch = new StopWatch();
		long time;
		watch.start();
		
		if (commit)
		{
			if (UnitTracker.count(UnitType.Terran_Vulture) >= 12
					&& !vultureHarass.isHarassing())
			{
				DebugMessages.addMessage("Stopping vulture push!");
				ConstantPushTask.paused.add(UnitType.Terran_Vulture);
			}
			else
			{
				if (UnitTracker.count(UnitType.Terran_Vulture) < 12)
					DebugMessages.addMessage("Continueing vulture push, vultures: " + UnitTracker.count(UnitType.Terran_Vulture));
				else
					DebugMessages.addMessage("Continueing vulture push, already harassing.");
				ConstantPushTask.paused.remove(UnitType.Terran_Vulture);
			}
		}
		
		if (enemyRace == Race.Unknown)
		{
			for (Unit enemy : EnemyManager.getEnemyUnits())
			{
				if (enemy.getType().getRace() == Race.Protoss)
					enemyRace = Race.Protoss;
				else if (enemy.getType().getRace() == Race.Terran)
					enemyRace = Race.Terran;
				else if (enemy.getType().getRace() == Race.Zerg)
				{
					enemyRace = Race.Zerg;
					buildBunker = true;
				}
			}
		}
		
		if (enemyRace == Race.Protoss
				&& !dragoonEncountered)
		{
			for (Unit enemy : EnemyManager.getEnemyUnits())
			{
				if (enemy.getType() == UnitType.Protoss_Dragoon)
				{
					dragoonEncountered = true;
					break;
				}
			}
		}
		
		if (constantPushTask != null)
		{
			if (dragoonEncountered && !self.hasResearched(TechType.Tank_Siege_Mode))
			{
				VultureHarassTask.defendingUnitsNeeded = 200;
				constantPushTask.stop();
			}
			else
			{
				VultureHarassTask.defendingUnitsNeeded = 4;
				constantPushTask.go();
			}
		}
		
		super.onFrame(game, self, bot);
		
		if (UnitTracker.count(UnitType.Terran_Factory)  < 1 && (bot.getAvailableGas() >= 100 || bot.getAvailableMinerals() < 100))
			Settings.setWorkersPerGas(1);
		else if ( (UnitTracker.count(UnitType.Terran_Factory)  <= 2 || bot.getAvailableGas() >= 300 )  
				&& (bot.getAvailableGas() >= 100 || bot.getAvailableMinerals() < 150))
			Settings.setWorkersPerGas(2);
		else
			Settings.setWorkersPerGas(3);
		
		time = watch.time();
		if (time > 55)
			DebugMessages.addMessage("Zealot counter super time: " + time);
		
		if (ExpandPart.maximumCcs == 1 && bot.getAvailableMinerals() >= 600)
			ExpandPart.maximumCcs = 2;
		else if (ExpandPart.maximumCcs == 2 && bot.getAvailableMinerals() >= 600 && self.completedUnitCount(UnitType.Terran_Command_Center) >= 2)
			ExpandPart.maximumCcs = 3;
		
		//if we've the resources to build a barracks ...
		if(bot.getAvailableMinerals() >= 100 
				&& UnitTracker.count(UnitType.Terran_Refinery) > 0
				&& (bot.getAvailableMinerals() >= 150 || UnitTracker.getCcCount() > 1)
				&& (UnitTracker.count(UnitType.Terran_Barracks) == 0)) 
		{
			bot.spaceManager.build(UnitType.Terran_Barracks);
		}
		
		if(UnitTracker.count(UnitType.Terran_Refinery) < UnitTracker.count(UnitType.Terran_Command_Center)
				&& (UnitTracker.count(UnitType.Terran_Refinery) == 0 || UnitTracker.count(UnitType.Terran_Factory) >= 1)
				&& bot.getAvailableMinerals() >= 100 
				&& UnitTracker.getGeyserCount() > 0)
		{
			bot.spaceManager.build(UnitType.Terran_Refinery);
		}
		
		if(bot.getAvailableMinerals() >= 200 && bot.getAvailableGas() >= 100
				&& (!commit || UnitTracker.count(UnitType.Terran_Bunker) > 0 || !buildBunker)
				&& UnitTracker.count(UnitType.Terran_Barracks) >= 1 
				&& UnitTracker.count(UnitType.Terran_Factory) < Math.min(7, 3 * UnitTracker.count(UnitType.Terran_Command_Center))
				&& (enemyRace != Race.Protoss || UnitTracker.count(UnitType.Terran_Command_Center) > 1 || !commit || UnitTracker.count(UnitType.Terran_Factory) < 2)
				&& (UnitTracker.count(UnitType.Terran_Factory) == 0
						|| bot.getAvailableMinerals() >= 200 && bot.getAvailableGas() >= 100
					)
			)
		{
			bot.spaceManager.build(UnitType.Terran_Factory);
		}

		time = watch.time();
		if (time > 55)
			DebugMessages.addMessage("Zealot counter tier 1 time: " + time);
				time = watch.time();
		
		if (bot.getAvailableMinerals() >= 150 
				&& UnitTracker.count(UnitType.Terran_Barracks) != 0 && UnitTracker.count(UnitType.Terran_Refinery) != 0
				&& UnitTracker.count(UnitType.Terran_Academy) == 0
				&& (bot.homeGroup.units.size() >= 10 || UnitTracker.count(UnitType.Terran_Vulture) >= 6 || dragoonEncountered))
		{
			bot.spaceManager.build(UnitType.Terran_Academy);
		}
		
		if (game.enemy().getRace() != Race.Zerg
				&& UnitTracker.count(UnitType.Terran_Armory) < 2
				&& bot.getAvailableGas() >= 50
				&& bot.getAvailableMinerals() >= 100
				&& commit
				&& self.completedUnitCount(UnitType.Terran_Factory) > 0
				&& (vultureHarass.isHarassing() || game.enemy().getRace() == Race.Terran))
		{
			bot.spaceManager.build(UnitType.Terran_Armory);
		}
		
		time = watch.time();
		if (time > 55)
			DebugMessages.addMessage("Zealot counter tier 2 time: " + time);
		
		
		// Build bunkers.
		if (buildBunker
				&& bot.getAvailableMinerals() >= 50 
				&& (bot.getAvailableMinerals() >= 100 ||  UnitTracker.count(UnitType.Terran_Factory) > 1)
				&& (UnitTracker.count(UnitType.Terran_Factory) >= 1 || commit) 
				&& self.completedUnitCount(UnitType.Terran_Barracks) >= 1
				&& UnitTracker.count(UnitType.Terran_Bunker) < 2)
		{
			for(DefensiveStructures structures : bot.defensiveStructures)
			{
				int count = structures.getUnitCount(UnitType.Terran_Bunker);
				
				if(count < 1)
				{
					// We do not build a bunker at our main if there already is one at our natural.
					if (bot.defensiveStructures.size() > 1 && structures.defendedPosition.getDistance(Tyr.tileToPosition(self.getStartLocation())) <= 200)
					{
						if (UnitTracker.count(UnitType.Terran_Bunker) == 0)
							continue;
						else
							break;
					}
					
					bot.spaceManager.buildDefensive(UnitType.Terran_Bunker, structures);
		  			break;
				}
			}
		}

		time = watch.time();
		if (time > 55)
			DebugMessages.addMessage("Zealot counter bunker time: " + time);
	}

	@Override
	public boolean overrideStructureOrder(Game game, Player self, Tyr bot, Agent agent)
	{
		if(super.overrideStructureOrder(game, self, bot, agent))
			return true;
		
		if(agent.unit.getType() == UnitType.Terran_Barracks)
		{
			if (!buildBunker)
				return true;
			if (UnitTracker.count(UnitType.Terran_Factory) == 0 || UnitTracker.count(UnitType.Terran_Marine) >= Math.max(4, UnitTracker.count(UnitType.Terran_Bunker)*4))
				return true;
		}
		
		if(agent.unit.getType() == UnitType.Terran_Starport)
			return true;
		
		if (agent.unit.getType() == UnitType.Terran_Factory)
		{
			if (goTank && enemyRace == Race.Protoss)
			{
				if (agent.unit.getAddon() != null)
					return false;
			}
			
			if (self.completedUnitCount(UnitType.Terran_Factory) >= ((enemyRace == Race.Protoss && goTank)?2:3)
					&& UnitTracker.usableAddOns(UnitType.Terran_Machine_Shop) == 0
					&& !agent.unit.isTraining()
					&& commit)
			{
				UnitTracker.addAddOn(UnitType.Terran_Machine_Shop);
				agent.unit.buildAddon(UnitType.Terran_Machine_Shop);
			}
			else if(UnitTracker.count(UnitType.Terran_Armory) >= 1 && !agent.unit.isTraining()
						&& (UnitTracker.count(UnitType.Terran_Goliath) < 4 || UnitTracker.count(UnitType.Terran_Goliath) < UnitTracker.count(UnitType.Terran_Vulture) - 12) 
						&& bot.getAvailableMinerals() >= 100 && bot.getAvailableGas() >= 50)
				agent.unit.train(UnitType.Terran_Goliath);
			else if (bot.getAvailableMinerals() >= 75 && !agent.unit.isTraining())
				agent.unit.train(UnitType.Terran_Vulture);
			return true;
		}
		else if(agent.unit.getType() == UnitType.Terran_Armory)
		{
			if(bot.getAvailableMinerals() >= UpgradeType.Terran_Vehicle_Weapons.mineralPrice()
					&& bot.getAvailableGas() >= UpgradeType.Terran_Vehicle_Weapons.gasPrice())
				agent.unit.upgrade(UpgradeType.Terran_Vehicle_Weapons);
			
			if(bot.getAvailableMinerals() >= UpgradeType.Terran_Vehicle_Plating.mineralPrice()
					&& bot.getAvailableGas() >= UpgradeType.Terran_Vehicle_Plating.gasPrice())
				agent.unit.upgrade(UpgradeType.Terran_Vehicle_Plating);
			
			return true;
		}
		else if (agent.unit.getType() == UnitType.Terran_Engineering_Bay)
		{
			return true;
		}
		else if (agent.unit.getType() == UnitType.Terran_Machine_Shop)
		{
			if (goTank && enemyRace == Race.Protoss && ! self.hasResearched(TechType.Tank_Siege_Mode))
				return false;
			if (!agent.unit.isResearching() && (commit || self.hasResearched(TechType.Tank_Siege_Mode))
					&& !self.hasResearched(TechType.Spider_Mines)
					&& bot.getAvailableMinerals() >= 100
					&& bot.getAvailableGas() >= 100)
				agent.unit.research(TechType.Spider_Mines);
			else if (!agent.unit.isResearching() && self.hasResearched(TechType.Spider_Mines)
					&& bot.getAvailableMinerals() >= 100
					&& bot.getAvailableGas() >= 100)
				agent.unit.upgrade(UpgradeType.Charon_Boosters);
			else if (UnitTracker.count(UnitType.Terran_Armory) > 0
					&& !agent.unit.isResearching() && self.hasResearched(TechType.Spider_Mines)
					&& bot.getAvailableMinerals() >= 100
					&& bot.getAvailableGas() >= 100)
				agent.unit.upgrade(UpgradeType.Charon_Boosters);
			if (!goTank || enemyRace != Race.Protoss)
				return true;
		}
		return false;
	}
}
