/*
* Tyr is an AI for StarCraft: Broodwar, 
* 
* Please visit https://github.com/SimonPrins/Tyr for further information.
* 
* Copyright 2015 Simon Prins
*
* This file is part of Tyr.
* Tyr is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 2 of the License, or
* (at your option) any later version.
* Tyr is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with Tyr.  If not, see http://www.gnu.org/licenses/.
*/


package com.tyr.builds;

import com.tyr.requirements.ConjRequirement;
import com.tyr.requirements.Requirement;

import bwapi.UnitType;


/**
 * This class can be used to create GenericPart objects.
 * @author Simon
 *
 */
public class GenericPartFactory
{
	/**
	 * The type of building to build.
	 */
	private UnitType building;

	/**
	 * The maximum number of this type of building to build.
	 */
	private int max = 1;
	
	/**
	 * The minerals that should be available before we build the building.
	 */
	private int minerals;
	
	/**
	 * The gas that should be available before we build the building.
	 */
	private int gas;
	
	/**
	 * Additional requirements that need to be met for us to build the building.
	 */
	private ConjRequirement requirements = new ConjRequirement();
	
	/**
	 * This class can be used to create GenericPart objects.
	 * @param building The type of building to build.
	 */
	public GenericPartFactory(UnitType building)
	{
		this.building = building;
		minerals = building.mineralPrice();
		gas = building.gasPrice();
	}
	
	/**
	 * Set the maximum number of this type of building to build.
	 * @param max The maximum number of this type of building to build.
	 * @return The factory for setting further settings.
	 */
	public GenericPartFactory setMax(int max)
	{
		this.max = max;
		return this;
	}
	
	/**
	 * Set the amount of minerals and gas needed before we build this building.
	 * @param minerals The minerals that should be available before we build the building
	 * @param gas The gas that should be available before we build the building
	 * @return The factory for setting further settings.
	 */
	public GenericPartFactory setCost(int minerals, int gas)
	{
		this.minerals = minerals;
		this.gas = gas;
		return this;
	}
	
	/**
	 * Set the amount of additional minerals and gas needed before we build this building.
	 * @param minerals The additional minerals that should be available before we build the building
	 * @param gas The additional gas that should be available before we build the building
	 * @return The factory for setting further settings.
	 */
	public GenericPartFactory additionalCost(int minerals, int gas)
	{
		this.minerals = minerals + building.mineralPrice();
		this.gas = gas + building.gasPrice();
		return this;
	}
	
	/**
	 * Add an requirement for building the building.
	 * @param req Additional requirement needed before we construct the building. 
	 * @return The factory for setting further settings.
	 */
	public GenericPartFactory add(Requirement req)
	{
		requirements.addRequirement(req);
		return this;
	}
	
	/**
	 * Creates the resulting GenericPart.
	 * @return The resulting GenericPart
	 */
	public GenericPart create()
	{
		final GenericPart result = new GenericPart(building, max, minerals, gas, requirements);
		
		return result;
	}
}
