/*
* Tyr is an AI for StarCraft: Broodwar, 
* 
* Please visit https://github.com/SimonPrins/Tyr for further information.
* 
* Copyright 2015 Simon Prins
*
* This file is part of Tyr.
* Tyr is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 2 of the License, or
* (at your option) any later version.
* Tyr is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with Tyr.  If not, see http://www.gnu.org/licenses/.
*/


package com.tyr.builds;

import com.tyr.Tyr;
import com.tyr.UnitTracker;
import com.tyr.agents.Agent;
import com.tyr.tasks.Task;

import bwapi.Game;
import bwapi.Player;
import bwapi.UnitType;


/**
 * Base class that represents a build order.
 * Manages the building of structures, but also things related to the build, such as number of workers in gas, sending out a worker scout, etc..
 * @author Simon
 *
 */
public abstract class BuildOrder
{
	/**
	 * This method performs the initalization for the buildorder.
	 * This method is expected to be called once at the start of the game.
	 * @param game The game.
	 * @param self The player controlled by our bot.
	 * @param bot The instance of Tyr that is currently playing the game.
	 */
	public abstract void initialize(Game game, Player self, Tyr bot);

	/**
	 * This method is expected to be called every frame.
	 * @param game The game.
	 * @param self The player controlled by our bot.
	 * @param bot The instance of Tyr that is currently playing the game.
	 */
	public abstract void onFrame(Game game, Player self, Tyr bot);
	
	/**
	 * All production structures follow standard rules for training units, researching upgrades, etc..
	 * This method is used by the build to override that order.
	 * If the standard order is overriden, this method should return true, in which case the standard action will not be executed.
	 * @param game The game.
	 * @param self The player controlled by our bot.
	 * @param bot The instance of Tyr that is currently playing the game.
	 * @param agent The building for which we want to know what it will build.
	 */
	public boolean overrideStructureOrder(Game game, Player self, Tyr bot, Agent agent)
	{
		return false;
	}
	
	public static int count(UnitType type)
	{
		return UnitTracker.count(type);
	}
	
	public static int completed(UnitType type)
	{
		return Tyr.self.completedUnitCount(type);
	}

	public static int completedTanks()
	{
		return completed(UnitType.Terran_Siege_Tank_Siege_Mode) + completed(UnitType.Terran_Siege_Tank_Tank_Mode);
	}
	
	public static int minerals()
	{
		return Tyr.bot.getAvailableMinerals();
	}
	
	public static int gas()
	{
		return Tyr.bot.getAvailableGas();
	}
	
	public static void build(UnitType type)
	{
		Tyr.bot.spaceManager.build(type);
	}
	
	public void addTask(Task task)
	{
		Tyr.bot.taskManager.potentialTasks.add(task);
	}
	
	/*
	 * Terran unit types.
	 */
	public static final UnitType CC = UnitType.Terran_Command_Center;
	public static final UnitType REFINERY = UnitType.Terran_Refinery;
	public static final UnitType MARINE = UnitType.Terran_Marine;
	public static final UnitType SCV = UnitType.Terran_SCV;
	public static final UnitType MEDIC = UnitType.Terran_Medic;
	public static final UnitType DEPOT = UnitType.Terran_Supply_Depot;
	public static final UnitType BARRACKS = UnitType.Terran_Barracks;
	public static final UnitType ENGINEERING_BAY = UnitType.Terran_Engineering_Bay;
	public static final UnitType ACADEMY = UnitType.Terran_Academy;
	public static final UnitType COMSAT = UnitType.Terran_Comsat_Station;
	public static final UnitType FACTORY = UnitType.Terran_Factory;
	public static final UnitType BUNKER = UnitType.Terran_Bunker;
	public static final UnitType MISSILE_TURRET = UnitType.Terran_Missile_Turret;
	public static final UnitType SCIENCE_VESSEL = UnitType.Terran_Science_Vessel;
	public static final UnitType STARPORT = UnitType.Terran_Starport;
	public static final UnitType SCIENCE_FACILITY = UnitType.Terran_Science_Facility;
	
	

	/*
	 * Protoss unit types.
	 */
	public static final UnitType NEXUS = UnitType.Protoss_Nexus;
	public static final UnitType PYLON = UnitType.Protoss_Pylon;
	public static final UnitType ASSIMILATOR = UnitType.Protoss_Assimilator;
	public static final UnitType GATEWAY = UnitType.Protoss_Gateway;
	public static final UnitType FORGE = UnitType.Protoss_Forge;
	public static final UnitType CYBERNETICS_CORE = UnitType.Protoss_Cybernetics_Core;
	public static final UnitType CANNON = UnitType.Protoss_Photon_Cannon;
	public static final UnitType CITADEL = UnitType.Protoss_Citadel_of_Adun;
	public static final UnitType TEMPLAR_ARCHIVES = UnitType.Protoss_Templar_Archives;
	public static final UnitType SHIELD_BATTERY = UnitType.Protoss_Shield_Battery;
	public static final UnitType STARGATE = UnitType.Protoss_Stargate;
	public static final UnitType FLEET_BEACON = UnitType.Protoss_Fleet_Beacon;
	public static final UnitType ARBITER_TRIBUNAL = UnitType.Protoss_Arbiter_Tribunal;
	public static final UnitType ROBOTICS_FACILITY = UnitType.Protoss_Robotics_Facility;
	public static final UnitType OBSERVATORY = UnitType.Protoss_Observatory;
	public static final UnitType ROBO_SUPPORT = UnitType.Protoss_Robotics_Support_Bay;
	public static final UnitType PROBE = UnitType.Protoss_Probe;
	public static final UnitType ZEALOT = UnitType.Protoss_Zealot;
	public static final UnitType DRAGOON = UnitType.Protoss_Dragoon;
	public static final UnitType HIGH_TEMPLAR = UnitType.Protoss_High_Templar;
	public static final UnitType DARK_TEMPLAR = UnitType.Protoss_Dark_Templar;
	public static final UnitType ARCHON = UnitType.Protoss_Archon;
	public static final UnitType DARK_ARCHON = UnitType.Protoss_Dark_Archon;
	public static final UnitType REAVER = UnitType.Protoss_Reaver;
	public static final UnitType SCARAB = UnitType.Protoss_Scarab;
	public static final UnitType OBSERVER = UnitType.Protoss_Observer;
	public static final UnitType SHUTTLE = UnitType.Protoss_Shuttle;
	public static final UnitType SCOUT = UnitType.Protoss_Scout;
	public static final UnitType ARBITER = UnitType.Protoss_Arbiter;
	public static final UnitType CARRIER = UnitType.Protoss_Carrier;
	public static final UnitType INTERCEPTOR = UnitType.Protoss_Interceptor;
	public static final UnitType CORSAIR = UnitType.Protoss_Corsair;
	

	/*
	 * Zerg unit types.
	 */
	public static final UnitType EXTRACTOR = UnitType.Zerg_Extractor;
	public static final UnitType ZERGLING = UnitType.Zerg_Zergling;
	public static final UnitType HYDRA = UnitType.Zerg_Hydralisk;
	public static final UnitType LURKER = UnitType.Zerg_Lurker;
	public static final UnitType POOL = UnitType.Zerg_Spawning_Pool;
	public static final UnitType HYDRA_DEN = UnitType.Zerg_Hydralisk_Den;
	public static final UnitType LURKER_EGG = UnitType.Zerg_Lurker_Egg;
}
