/*
* Tyr is an AI for StarCraft: Broodwar, 
* 
* Please visit https://github.com/SimonPrins/Tyr for further information.
* 
* Copyright 2015 Simon Prins
*
* This file is part of Tyr.
* Tyr is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 2 of the License, or
* (at your option) any later version.
* Tyr is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with Tyr.  If not, see http://www.gnu.org/licenses/.
*/


package com.tyr.builds;

import com.tyr.DebugMessages;
import com.tyr.EnemyManager;
import com.tyr.Settings;
import com.tyr.StopWatch;
import com.tyr.Tyr;
import com.tyr.UnitTracker;
import com.tyr.agents.Agent;
import com.tyr.buildingplacement.DefensiveStructures;
import com.tyr.requirements.LinearMax;
import com.tyr.requirements.UnitRequirement;
import com.tyr.tasks.AttackTask;
import com.tyr.unitgroups.MineralWorkers;

import bwapi.Game;
import bwapi.Player;
import bwapi.Race;
import bwapi.TechType;
import bwapi.UnitType;
import bwapi.UpgradeType;


/**
 * Implements a bio build.
 * @author Simon
 *
 */
public class AntiPool extends CompositeBuildOrder
{
	/**
	 * Are we ready to take expands?
	 */
	private boolean takeExpands = false;
	
	/**
	 * Index of the base where we last tried to place a turret.
	 */
	private int turretPos;
	
	/**
	 * How many turrets do we build near our supply depots?
	 */
	private int depotTurrets = 10;
	
	/**
	 * Is the first attack completed?
	 */
	boolean firstAttackDone = false;
	
	
	@Override
	public void initialize(Game game, Player self, Tyr bot)
	{
		Settings.setLargeInvasionDist(720);
		Settings.setRequiredSize(10);
		Settings.setMaximumSize(60);
		
		Settings.setWorkersPerGas(1);
		bot.bunkers.maximumRepair = 1;
		
		this.add(new SupplyDepotPart());
		this.add(new WorkerScoutPart(1600));

		this.add(new GenericPartFactory(UnitType.Terran_Barracks)
			.setCost(100, 0)
			.create());
		
		GenericPartFactory barracks2Factory = new GenericPartFactory(UnitType.Terran_Barracks)
			.setMax(2)
			.setCost(100, 0);
		this.add(barracks2Factory.create());
		barracks2Factory.add(new UnitRequirement(UnitType.Terran_Bunker));
		
		GenericPartFactory barracksFactory = new GenericPartFactory(UnitType.Terran_Barracks)
			.setMax(8)
			.setCost(100, 0)
			.add(new LinearMax(UnitType.Terran_Command_Center, UnitType.Terran_Barracks, 3, 0, false))
			.add(new UnitRequirement(UnitType.Terran_Academy));
		barracksFactory.add(new UnitRequirement(UnitType.Terran_Bunker));
		this.add(barracksFactory.create());
		
		GenericPartFactory academyFactory = new GenericPartFactory(UnitType.Terran_Academy)
			.add(new UnitRequirement(UnitType.Terran_Barracks))
			.add(new UnitRequirement(UnitType.Terran_Refinery));
		this.add(academyFactory.create());
		
		this.add(new GenericPartFactory(UnitType.Terran_Engineering_Bay)
			.setMax(2)
			.add(new UnitRequirement(UnitType.Terran_Marine, 10))
			.add(new UnitRequirement(UnitType.Terran_Academy))
			.create());
		
		this.add(new GenericPartFactory(UnitType.Terran_Factory)
			.add(new UnitRequirement(UnitType.Terran_Barracks, 6))
			.add(new UnitRequirement(UnitType.Terran_Engineering_Bay, 2))
			.create());
		
		super.initialize(game, self, bot);
	}
	
	@Override
	public void onFrame(Game game, Player self, Tyr bot)
	{
		DebugMessages.addMessage("Stopping zergling rush.");
		DebugMessages.addMessage("Total lings detected: " + EnemyManager.getManager().getAllCount(UnitType.Zerg_Zergling));
		
		StopWatch watch = new StopWatch();
		long time;
		watch.start();
		
		super.onFrame(game, self, bot);
		
		time = watch.time();
		if (time > 55)
			DebugMessages.addMessage("Bio super time: " + time); 
		
		if (UnitTracker.count(UnitType.Terran_Barracks) > 5 || UnitTracker.count(UnitType.Terran_Engineering_Bay) > 0)
			Settings.setWorkersPerGas(2);
		
		// We only take an expand after the first attack.
		if(!takeExpands && AttackTask.getAttackSent())
		{
			this.add(new ExpandPart(true));
			takeExpands = true;
		}
		
		bot.bunkers.disabled = self.completedUnitCount(UnitType.Terran_Marine) + self.completedUnitCount(UnitType.Terran_Medic) >= (firstAttackDone?20:10) && !AttackTask.getAttackSent();
		
		if (!firstAttackDone && !EnemyManager.getManager().enemyDefensiveStructures.isEmpty())
		{
			firstAttackDone = true;
			Settings.setRequiredSize(20);
			AttackTask.setAttackSent(false);
			AttackTask.cancelAttacks();
		}
		
		time = watch.time();
		if (time > 55)
			DebugMessages.addMessage("Bio tier 1 time: " + time);
		
		if (bot.getAvailableMinerals()>= 100 
				&& UnitTracker.getGeyserCount() > 0 
				&& UnitTracker.count(UnitType.Terran_Barracks) >= 1
				&& (UnitTracker.count(UnitType.Terran_Refinery) == 0 || UnitTracker.count(UnitType.Terran_Starport) > 0)
				&& (UnitTracker.count(UnitType.Terran_Bunker) > 0)
			)
		{
			bot.spaceManager.build(UnitType.Terran_Refinery);
		}

		time = watch.time();
		if (time > 55)
			DebugMessages.addMessage("Bio refinery: " + time + " geyserCount: " + UnitTracker.getGeyserCount());
		
		if (bot.getAvailableMinerals() >= 75 && self.completedUnitCount(UnitType.Terran_Engineering_Bay) > 0)
		{
			for(DefensiveStructures structures : bot.defensiveStructures)
			{
				if (structures.defendedPosition.getDistance(Tyr.tileToPosition(self.getStartLocation())) <= 200)
					continue;
				if (structures.getUnitCount(UnitType.Terran_Missile_Turret) == 0)
					bot.spaceManager.buildDefensive(UnitType.Terran_Missile_Turret, structures);
				break;
			}
		}
		
		// Build bunkers, unless we are in greedy mode.
		if (bot.getAvailableMinerals() >= 100 && UnitTracker.count(UnitType.Terran_Barracks) >= 1)
		{
			for(DefensiveStructures structures : bot.defensiveStructures)
			{
				int count = structures.getUnitCount(UnitType.Terran_Bunker);

				// We do not build a bunker at our main if there already is one at our natural.
				if (bot.defensiveStructures.size() > 1 
						&& structures.defendedPosition.getDistance(Tyr.tileToPosition(self.getStartLocation())) <= 200)
					continue;
				
				// Ultra defensive mode goes for double bunkers.
				if(count < 1)
				{
					bot.spaceManager.buildDefensive(UnitType.Terran_Bunker, structures);
		  			break;
				}
			}
		}

		time = watch.time();
		if (time > 55)
			DebugMessages.addMessage("Bio bunker time: " + time);
		
		if(self.completedUnitCount(UnitType.Terran_Engineering_Bay) >= 1 && bot.getAvailableMinerals() >= 75)
		{
			boolean built = false;
			
			for (int i=0; i<bot.workForce.mineralWorkers.size(); i++)
			{
				MineralWorkers workers = bot.workForce.mineralWorkers.get((i + turretPos)%bot.workForce.mineralWorkers.size());
				if (bot.spaceManager.build(UnitType.Terran_Missile_Turret, null, workers.turretSite))
				{
					built = true;
					turretPos = (i + turretPos + 1)%bot.workForce.mineralWorkers.size();
					break;
				}
			}
			if (!built)
				DebugMessages.addMessage("No placement found for any turret.");
		}
		
		if (bot.getAvailableMinerals() >= 75
				&& self.completedUnitCount(UnitType.Terran_Missile_Turret) <= depotTurrets
				&& UnitTracker.count(UnitType.Terran_Marine) >= 15)
		{
			bot.spaceManager.build(UnitType.Terran_Missile_Turret, null, bot.spaceManager.depotBuildSite);
		}

		time = watch.time();
		if (time > 55)
			DebugMessages.addMessage("Bio turrets time: " + time);
	}

	@Override
	public boolean overrideStructureOrder(Game game, Player self, Tyr bot, Agent agent)
	{
		if(super.overrideStructureOrder(game, self, bot, agent))
			return true;
		
		if (agent.unit.getType() == UnitType.Terran_Barracks)
		{
			if (!agent.unit.isTraining() 
					&& Tyr.bot.getAvailableGas() >= 25 && Tyr.bot.getAvailableMinerals() >= 50
					&& self.completedUnitCount(UnitType.Terran_Academy) > 0)
			{
				if (UnitTracker.count(UnitType.Terran_Medic) * 4 < 
						UnitTracker.count(UnitType.Terran_Marine) - 4 * UnitTracker.count(UnitType.Terran_Bunker))
				{
					agent.unit.train(UnitType.Terran_Medic);
					return true;
				}
			}
		}
		else if (agent.unit.getType() == UnitType.Terran_Academy)
		{
			if (bot.getAvailableMinerals() >= 100 && bot.getAvailableGas() >= 100 
					&& !agent.unit.isResearching() && !self.hasResearched(TechType.Stim_Packs))
			{
				agent.unit.research(TechType.Stim_Packs);
				return true;
			}
			else if (bot.getAvailableGas() >= 150 && bot.getAvailableMinerals() >= 150
					&& !agent.unit.isResearching())
			{
				agent.unit.upgrade(UpgradeType.U_238_Shells);
			}
		}
		else if (agent.unit.getType() == UnitType.Terran_Factory)
		{
			if (!agent.unit.isTraining() && bot.getAvailableMinerals() >= 100 && Tyr.game.enemy().getRace() == Race.Protoss)
				agent.unit.train(UnitType.Terran_Vulture);
			return true;
		}
		
		return false;
	}
}
