/*
* Tyr is an AI for StarCraft: Broodwar, 
* 
* Please visit https://github.com/SimonPrins/Tyr for further information.
* 
* Copyright 2015 Simon Prins
*
* This file is part of Tyr.
* Tyr is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 2 of the License, or
* (at your option) any later version.
* Tyr is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with Tyr.  If not, see http://www.gnu.org/licenses/.
*/


package com.tyr.builds;

import com.tyr.DebugMessages;
import com.tyr.EnemyManager;
import com.tyr.Settings;
import com.tyr.StopWatch;
import com.tyr.Tyr;
import com.tyr.UnitTracker;
import com.tyr.agents.Agent;
import com.tyr.buildingplacement.DefensiveStructures;
import com.tyr.requirements.CostRequirement;
import com.tyr.requirements.DisjRequirement;
import com.tyr.requirements.LinearMax;
import com.tyr.requirements.Requirement;
import com.tyr.requirements.UnitRequirement;
import com.tyr.tasks.AssignScience;
import com.tyr.tasks.PokeTask;
import com.tyr.tasks.SalesmenMarinesTask;
import com.tyr.unitgroups.AttackGroup;
import com.tyr.unitgroups.MarineHarassGroup;
import com.tyr.unitgroups.MineralWorkers;

import bwapi.Game;
import bwapi.Player;
import bwapi.Race;
import bwapi.TechType;
import bwapi.Unit;
import bwapi.UnitType;
import bwapi.UpgradeType;


/**
 * Implements a bio build.
 * @author Simon
 *
 */
public class AntiMuta extends CompositeBuildOrder
{
	/**
	 * Index of the base where we last tried to place a turret.
	 */
	private int turretPos;
	
	/**
	 * How many turrets do we build near our supply depots?
	 */
	private int depotTurrets = 4;
	
	/**
	 * Is the opponent performing a zergling rush?
	 */
	private boolean lingRush = false;
	
	/**
	 * Is the opponent using lurkers?
	 */
	private boolean lurkers = false;
	
	@Override
	public void initialize(Game game, Player self, Tyr bot)
	{
		Settings.setRequiredSize(201);
		Settings.setMaximumSize(201);
		
		Settings.setWorkersPerGas(1);
		
		bot.bunkers.maximumRepair = 1;
		
		this.add(new SupplyDepotPart());
		this.add(new WorkerScoutPart(1600));

		this.add(new GenericPartFactory(UnitType.Terran_Barracks)
			.setCost(100, 0)
			.create());
		
		GenericPartFactory barracks2Factory = new GenericPartFactory(UnitType.Terran_Barracks)
			.setMax(2)
			.setCost(150, 0)
			.add(new UnitRequirement(UnitType.Terran_Command_Center, 2));

		this.add(barracks2Factory.create());
		
		GenericPartFactory barracksFactory = new GenericPartFactory(UnitType.Terran_Barracks)
			.setMax(6)
			.setCost(150, 0)
			.add(new LinearMax(UnitType.Terran_Command_Center, UnitType.Terran_Barracks, 3, -1, false))
			.add(new UnitRequirement(UnitType.Terran_Academy))
			.add(new UnitRequirement(UnitType.Terran_Command_Center, 2))
			.add(new UnitRequirement(UnitType.Terran_Bunker))
			.add(new DisjRequirement()
				.addRequirement(new UnitRequirement(UnitType.Terran_Science_Facility))
				.addRequirement(new CostRequirement(350, 0)));

		this.add(barracksFactory.create());
		final AntiMuta thisBuild = this;
			
		barracksFactory = new GenericPartFactory(UnitType.Terran_Barracks)
				.setMax(8)
				.setCost(150, 0)
				.add(new LinearMax(UnitType.Terran_Command_Center, UnitType.Terran_Barracks, 3, 0, false))
				.add(new UnitRequirement(UnitType.Terran_Academy))
				.add(new UnitRequirement(UnitType.Terran_Command_Center, 2))
				.add(new UnitRequirement(UnitType.Terran_Bunker))
				.add(new DisjRequirement()
					.addRequirement(new UnitRequirement(UnitType.Terran_Science_Facility))
					.addRequirement(new CostRequirement(350, 0)))
				.add(new Requirement() {
						@Override
						public boolean met() {
							return !thisBuild.lurkers;
						}
				});
		
		this.add(barracksFactory.create());
		
		GenericPartFactory academyFactory = new GenericPartFactory(UnitType.Terran_Academy)
			.add(new UnitRequirement(UnitType.Terran_Barracks))
			.add(new UnitRequirement(UnitType.Terran_Refinery))
			.add(new UnitRequirement(UnitType.Terran_Command_Center, 2));
		this.add(academyFactory.create());
		
		this.add(new GenericPartFactory(UnitType.Terran_Engineering_Bay)
			.setMax(2)
			.add(new UnitRequirement(UnitType.Terran_Marine, 10))
			.add(new UnitRequirement(UnitType.Terran_Academy))
			.create());
		
		this.add(new GenericPartFactory(UnitType.Terran_Factory)
			.add(new UnitRequirement(UnitType.Terran_Barracks, 2))
			.add(new UnitRequirement(UnitType.Terran_Engineering_Bay, 2))
			.create());
		
		this.add(new GenericPartFactory(UnitType.Terran_Starport)
			.add(new UnitRequirement(UnitType.Terran_Barracks, 2))
			.add(new UnitRequirement(UnitType.Terran_Engineering_Bay, 2))
			.add(new UnitRequirement(UnitType.Terran_Factory, 1, true))
			.create());
		
		this.add(new GenericPartFactory(UnitType.Terran_Science_Facility)
			.add(new UnitRequirement(UnitType.Terran_Barracks, 2))
			.add(new UnitRequirement(UnitType.Terran_Engineering_Bay, 2))
			.add(new UnitRequirement(UnitType.Terran_Starport, 1, true))
			.create());
		

		this.add(new ExpandPart(true));

		bot.taskManager.potentialTasks.add(new AssignScience());
		bot.taskManager.potentialTasks.add(new SalesmenMarinesTask(null, 25, 8, true, 0));
		bot.taskManager.potentialTasks.add(new PokeTask());
		
		AttackGroup.fearLurkers = true;
		AttackGroup.fearDefenses = true;
		
		super.initialize(game, self, bot);
	}
	
	@Override
	public void onFrame(Game game, Player self, Tyr bot)
	{
		DebugMessages.addMessage("Going bio.");
		
		StopWatch watch = new StopWatch();
		long time;
		watch.start();

		if (!lingRush)
		{
			int lingCount = 0;
			for (Unit enemy : EnemyManager.getEnemyUnits())
				if (enemy.getType() == UnitType.Zerg_Zergling);
					lingCount++;
			if (lingCount >= 8)
				lingRush = true;
		}
		
		if (!lurkers)
			for (Unit enemy : EnemyManager.getEnemyUnits())
				if (enemy.getType() == UnitType.Zerg_Lurker 
				|| enemy.getType() == UnitType.Zerg_Hydralisk 
				|| enemy.getType() == UnitType.Zerg_Hydralisk_Den);
					lurkers = true;
		
		if (PokeTask.getDone() && PokeTask.defendingUnitsNeeded < 20)
		{
			PokeTask.defendingUnitsNeeded = 20;
			PokeTask.required = 20;
			MarineHarassGroup.retreatTypes = new UnitType[] {UnitType.Zerg_Lurker};
			PokeTask.reset();
		}
		
		SalesmenMarinesTask.paused = lurkers && self.completedUnitCount(UnitType.Terran_Science_Vessel) == 0;
		
		super.onFrame(game, self, bot);
		
		time = watch.time();
		if (time > 55)
			DebugMessages.addMessage("Bio super time: " + time);
		
		bot.bunkers.disabled = (self.completedUnitCount(UnitType.Terran_Marine) >= 5 && !PokeTask.getAttackSent()) 
				|| bot.defensiveStructures.size() > 1; 
		
		if (UnitTracker.count(UnitType.Terran_Engineering_Bay) > 0 && bot.getAvailableGas() < 400)
			Settings.setWorkersPerGas(3);
		else if (UnitTracker.count(UnitType.Terran_Barracks) > 5 || UnitTracker.count(UnitType.Terran_Engineering_Bay) > 0)
			Settings.setWorkersPerGas(2);
		
		time = watch.time();
		if (time > 55)
			DebugMessages.addMessage("Bio tier 1 time: " + time);
		
		if (bot.getAvailableMinerals()>= 100 
				&& UnitTracker.getGeyserCount() > 0 
				&& UnitTracker.count(UnitType.Terran_Barracks) >= 1
				&& (UnitTracker.count(UnitType.Terran_Refinery) == 0 || UnitTracker.count(UnitType.Terran_Starport) > 0)
				&& (UnitTracker.getCcCount() >= 2)
				&& (UnitTracker.count(UnitType.Terran_Bunker) > 0)
			)
		{
			bot.spaceManager.build(UnitType.Terran_Refinery);
		}

		time = watch.time();
		if (time > 55)
			DebugMessages.addMessage("Bio refinery: " + time + " geyserCount: " + UnitTracker.getGeyserCount());
		
		// Build bunkers, unless we are in greedy mode.
		if (bot.getAvailableMinerals() >= 100 && UnitTracker.count(UnitType.Terran_Barracks) >= 1
				&& UnitTracker.count(UnitType.Terran_Bunker) == 0)
		{
			for(DefensiveStructures structures : bot.defensiveStructures)
			{
				bot.spaceManager.buildDefensive(UnitType.Terran_Bunker, structures);
	  			break;
	  		}
		}

		time = watch.time();
		if (time > 55)
			DebugMessages.addMessage("Bio bunker time: " + time);
		
		if(self.completedUnitCount(UnitType.Terran_Engineering_Bay) >= 1 && bot.getAvailableMinerals() >= 75)
		{
			boolean built = false;
			
			for (int i=0; i<bot.workForce.mineralWorkers.size(); i++)
			{
				MineralWorkers workers = bot.workForce.mineralWorkers.get((i + turretPos)%bot.workForce.mineralWorkers.size());
				if (bot.spaceManager.build(UnitType.Terran_Missile_Turret, null, workers.turretSite))
				{
					built = true;
					turretPos = (i + turretPos + 1)%bot.workForce.mineralWorkers.size();
					break;
				}
			}
			if (!built)
				DebugMessages.addMessage("No placement found for any turret.");
		}
		
		if (bot.getAvailableMinerals() >= 75
				&& self.completedUnitCount(UnitType.Terran_Missile_Turret) >= 4
				&& depotTurrets > 0)
		{
			bot.spaceManager.build(UnitType.Terran_Missile_Turret, null, bot.spaceManager.depotBuildSite);
			depotTurrets--;
		}

		time = watch.time();
		if (time > 55)
			DebugMessages.addMessage("Bio turrets time: " + time);
	}

	@Override
	public boolean overrideStructureOrder(Game game, Player self, Tyr bot, Agent agent)
	{
		if(super.overrideStructureOrder(game, self, bot, agent))
			return true;
		
		if (agent.unit.getType() == UnitType.Terran_Barracks)
		{
			// After we have enough marines for our bunker, we do not train marines until we have our expand started.
			if(UnitTracker.count(UnitType.Terran_Marine) >= 5 && UnitTracker.getCcCount() <= 1 && !lingRush)
				return true;
			if (!agent.unit.isTraining() 
					&& Tyr.bot.getAvailableGas() >= 25 && Tyr.bot.getAvailableMinerals() >= 50
					&& self.completedUnitCount(UnitType.Terran_Academy) > 0)
			{
				if (UnitTracker.count(UnitType.Terran_Medic) * 4 < 
						UnitTracker.count(UnitType.Terran_Marine) - 4 * UnitTracker.count(UnitType.Terran_Bunker))
				{
					agent.unit.train(UnitType.Terran_Medic);
					return true;
				}
			}
		}
		else if (agent.unit.getType() == UnitType.Terran_Academy)
		{
			if (bot.getAvailableMinerals() >= 100 && bot.getAvailableGas() >= 100 
					&& !agent.unit.isResearching() && !self.hasResearched(TechType.Stim_Packs))
			{
				agent.unit.research(TechType.Stim_Packs);
				return true;
			}
			else if (bot.getAvailableGas() >= 150 && bot.getAvailableMinerals() >= 150
					&& !agent.unit.isResearching())
			{
				agent.unit.upgrade(UpgradeType.U_238_Shells);
			}
		}
		else if (agent.unit.getType() == UnitType.Terran_Starport)
		{
			if (agent.unit.getAddon() == null 
					&& bot.getAvailableMinerals() >= 50 && bot.getAvailableGas() >= 50)
				agent.unit.buildAddon(UnitType.Terran_Control_Tower);
			if (!agent.unit.isTraining()
					&& bot.getAvailableMinerals() >= 100 
					&& bot.getAvailableGas() >= 225)
			{
				agent.unit.train(UnitType.Terran_Science_Vessel);
			}
			return true;
		}
		else if (agent.unit.getType() == UnitType.Terran_Factory)
		{
			if (!agent.unit.isTraining() && bot.getAvailableMinerals() >= 100 && Tyr.game.enemy().getRace() == Race.Protoss)
				agent.unit.train(UnitType.Terran_Vulture);
			return !lurkers;
		}
		else if (agent.unit.getType() == UnitType.Terran_Science_Facility)
		{
			if (!self.hasResearched(TechType.Irradiate)
					&& bot.getAvailableMinerals() >= 200 
					&& bot.getAvailableGas() >= 200)
			{
				agent.unit.research(TechType.Irradiate);
				return true;
			}
			return false;
		}
		
		return false;
	}
}
