/*
* Tyr is an AI for StarCraft: Broodwar, 
* 
* Please visit https://github.com/SimonPrins/Tyr for further information.
* 
* Copyright 2015 Simon Prins
*
* This file is part of Tyr.
* Tyr is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 2 of the License, or
* (at your option) any later version.
* Tyr is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with Tyr.  If not, see http://www.gnu.org/licenses/.
*/


package com.tyr.agents;

import com.tyr.DebugMessages;
import com.tyr.EnemyManager;
import com.tyr.EnemyPosition;
import com.tyr.Tyr;
import com.tyr.unitgroups.ScoutGroup;

import bwapi.Color;
import bwapi.Game;
import bwapi.Player;
import bwapi.Position;
import bwapi.TechType;
import bwapi.Unit;
import bwapi.UnitType;


/**
 * Order a unit to attack toward a target position.
 * @author Simon
 *
 */
public class Attack extends Command
{
	/**
	 * The position where the unit will attack.
	 */
	private Position target;
	
	/**
	 * The frame during which we last updated the waitingAtCannon variable.
	 * Used to determine if we need to update the value.
	 */
	private static int waitFrame;
	
	/**
	 * How many units need to wait at a cannon before attacking?
	 */
	public static int requiredAtCannon = 10;
	
	/**
	 * The number of units waiting at a photon cannon.
	 */
	private static int waitingAtCannon;
	
	/**
	 * The number of units who waited at a photon cannon last frame.
	 */
	private static int waitingAtCannonPrev;
	
	/**
	 * Do we stutterstep closer if the enemy is far away?
	 */
	private boolean moveCloser;
	
	/**
	 * Frame at which we last sieged, if this is a siegetank.
	 */
	public int siegeFrame = -1000000;
	
	/**
	 * Refuse to wait at cannons.
	 */
	public static boolean dontWaitAtCannon = false;

	/**
	 * Order a unit to attack toward a target position.
	 * @param agent The unit who will attack the enemy.
	 * @param target The position where the unit will attack.
	 */
	public Attack(Agent agent, Position target)
	{
		super(agent);
		this.target = target;
	}
	
	/**
	 * Order a unit to attack toward a target position.
	 * @param agent The unit who will attack the enemy.
	 * @param target The position where the unit will attack.
	 * @param moveCloser Do we stutterstep closer if the enemy is far away?
	 */
	public Attack(Agent agent, Position target, boolean moveCloser)
	{
		super(agent);
		this.target = target;
		this.moveCloser = moveCloser;
	}
	
	@Override
	public void execute(Game game, Player self, Tyr bot) 
	{
		agent.drawCircle(Color.Red);
		
		// Do we need to update the waiting at cannon variable.
		if (waitFrame != game.getFrameCount())
		{
			waitingAtCannonPrev = waitingAtCannon;
			DebugMessages.addMessage("Waiting at cannon: " + waitingAtCannon);
			waitingAtCannon = 0;
			waitFrame = game.getFrameCount();
		}
		
		// Are we in range of a photon cannon?
		boolean inrange = false;
		
		// If we are in range of a photon cannon, will we proceed to attack?
		boolean proceed = (bot.scout.opponentStrategy != ScoutGroup.cannons && waitingAtCannonPrev >= requiredAtCannon) 
				|| agent.unit.getType() == UnitType.Terran_Siege_Tank_Siege_Mode || agent.unit.getType() == UnitType.Terran_Siege_Tank_Tank_Mode || agent.unit.getType() == UnitType.Protoss_Dark_Templar || dontWaitAtCannon;


		int dist = UnitType.Terran_Siege_Tank_Siege_Mode.sightRange() + 64;
		int smallDist = UnitType.Terran_Siege_Tank_Siege_Mode.sightRange() - 32;
		if (bot.scout.opponentStrategy != ScoutGroup.cannons)
			smallDist += 32;
		
		// Determine if we are in range of a photon cannon.
		for(EnemyPosition enemy : EnemyManager.getManager().enemyDefensiveStructures)
		{
			if (agent.unit.getType() == UnitType.Terran_Battlecruiser)
				break;
			
			if(enemy.type == UnitType.Protoss_Photon_Cannon
					&& Math.abs(agent.unit.getX() - enemy.pos.getX()) <= dist
					&& Math.abs(agent.unit.getY() - enemy.pos.getY()) <= dist
					&& agent.distanceSquared(enemy.pos)<= dist * dist)
			{
				inrange = agent.distanceSquared(enemy.pos)<= (smallDist) * (smallDist);
				waitingAtCannon++;
				break;
			}
		}
		
		if (inrange && !agent.unit.isHoldingPosition() && !proceed)
			agent.unit.holdPosition();
		else if(proceed || !inrange)
			attack();
		else
			Tyr.drawCircle(agent.unit.getPosition(), Color.Blue, 6);
	}

	/**
	 * Method to manage the actual attack.
	 */
	private void attack()
	{
		if (!Agent.isRanged(agent.unit))
		{
			agent.orderAttack(target);
			return;
		}
		if (agent.unit.getType() == UnitType.Terran_Siege_Tank_Tank_Mode)
		{
			if (!Tyr.self.hasResearched(TechType.Tank_Siege_Mode))
			{
				if (!agent.unit.isHoldingPosition())
					agent.unit.holdPosition();
				agent.drawCircle(Color.Green, 12);
				return;
			}
			int radius = UnitType.Terran_Siege_Tank_Siege_Mode.groundWeapon().maxRange();
			int minRadius = UnitType.Terran_Siege_Tank_Siege_Mode.groundWeapon().minRange();
			// Do not siege if there are melee units in range.
			if (Tyr.self.hasResearched(TechType.Tank_Siege_Mode))
			{
				for(Unit unit : EnemyManager.getEnemyUnits())
				{
					if (unit.getType().isFlyer() || unit.isLifted() || unit.isCloaked() 
							|| agent.distanceSquared(unit) >= minRadius * minRadius || unit.getType() == UnitType.Protoss_Scarab)
						continue;
					
					// Do not siege when melee units are nearby.
					if (Agent.isRanged(unit) && agent.unit.getType() != UnitType.Zerg_Lurker)
						continue;
	
					agent.stutterstep(target, moveCloser);
					return;
				}
			}
			for(Unit unit : EnemyManager.getEnemyUnits())
			{
				if (unit.getType().isFlyer() || unit.isLifted() || unit.isCloaked() 
						|| agent.distanceSquared(unit) <= minRadius * minRadius || unit.getType() == UnitType.Protoss_Scarab)
					continue;
				
				// Only siege for ranged units.
				if (!Agent.isRanged(unit) || agent.unit.getType() == UnitType.Zerg_Lurker)
					continue;
				if (agent.distanceSquared(unit) <= radius * radius)
				{
					Tyr.game.drawLineMap(agent.unit.getX(), agent.unit.getY(), unit.getX(), unit.getY(), Color.Red);
					siegeFrame = Tyr.game.getFrameCount();
					if (Tyr.self.hasResearched(TechType.Tank_Siege_Mode))
						agent.unit.siege();
					else if (agent.unit.isMoving() || agent.unit.isAttacking())
						agent.unit.stop();
					agent.drawCircle(Color.Red, 6);
					return;
					
				}
			}

			if (Tyr.self.hasResearched(TechType.Tank_Siege_Mode))
				agent.stutterstep(target, moveCloser);
			
			return;
		}
		else if (agent.unit.getType() == UnitType.Terran_Siege_Tank_Siege_Mode)
		{
			int radius = UnitType.Terran_Siege_Tank_Siege_Mode.groundWeapon().maxRange();
			int minRadius = UnitType.Terran_Siege_Tank_Siege_Mode.groundWeapon().minRange();
			

			// Unsiege if there are melee units in range.
			for(Unit unit : EnemyManager.getEnemyUnits())
			{
				if (unit.getType().isFlyer() || unit.isLifted() || unit.isCloaked() 
						|| agent.distanceSquared(unit) >= (minRadius-96) * (minRadius - 96) || unit.getType() == UnitType.Protoss_Scarab)
					continue;
				
				// Only unsiege for melee units.
				if (Agent.isRanged(unit) && agent.unit.getType() != UnitType.Zerg_Lurker)
					continue;

				agent.drawCircle(Color.White, 6);
				agent.unit.unsiege();
				return;
			}
			for(Unit unit : EnemyManager.getEnemyUnits())
			{
				if (unit.getType().isFlyer() || unit.isLifted() || unit.isCloaked()
						|| agent.distanceSquared(unit) <= minRadius * minRadius || unit.getType() == UnitType.Protoss_Scarab)
					continue;
				
				// Only siege for ranged enemies.
				if (!Agent.isRanged(unit) || agent.unit.getType() == UnitType.Zerg_Lurker)
					continue;
				
				if (agent.distanceSquared(unit) <= radius * radius)
				{
					Tyr.game.drawLineMap(agent.unit.getX(), agent.unit.getY(), unit.getX(), unit.getY(), Color.Red);
					agent.drawCircle(Color.Red, 6);
					Unit currentTarget = agent.unit.getTarget();
					siegeFrame = Tyr.game.getFrameCount();
					if (agent.unit.isIdle() || (!agent.unit.isAttackFrame() && !agent.unit.isAttacking()) || (currentTarget != null && !Agent.isRanged(currentTarget)))
						agent.unit.attack(unit);
					return;
					
				}
			}
			if (Tyr.game.getFrameCount() - siegeFrame >= 250)
			{
				agent.drawCircle(Color.White, 6);
				agent.unit.unsiege();
			}
			else
				agent.drawCircle(Color.Green, 6);
			return;
		}
		
		agent.stutterstep(target, moveCloser);
	}
	
	@Override
	public boolean replace(Command command) 
	{
		if (!command.getClass().equals(Attack.class))
			return true;
		
		Attack atkCom = ((Attack)command);
		
		if (atkCom.target == null)
			return true;
		
		if (this.target == null)
			return false;

		atkCom.moveCloser = moveCloser;
		//this.moveCloser = atkCom.moveCloser;
		
		if (target.getX() == atkCom.target.getX() && target.getY() == atkCom.target.getY())
		{
			return false;
		}
		//this.target = atkCom.target;
		atkCom.target = target;
		
		attack();
		
		return false;
	}
}
