#include "Common.h"
#include "WorkerManager.h"
#include "Micro.h"
#include "UnitUtil.h"

using namespace UAlbertaBot;

WorkerManager::WorkerManager() 
{
    previousClosestWorker = nullptr;
	previousClosestBlockingMinWorker = nullptr;
	_collectGas = false;
}

WorkerManager & WorkerManager::Instance() 
{
	static WorkerManager instance;
	return instance;
}

void WorkerManager::update() 
{
	updateWorkerStatus();
	handleGasWorkers();
	handleIdleWorkers();
	handleReturnCargoWorkers();
	handleMoveWorkers();
	handleRepairWorkers();

	drawResourceDebugInfo();
	drawWorkerInformation(500,20);

	workerData.drawDepotDebugInfo();
}

// Adjust worker jobs. This is done first, before handling each job.
// NOTE A mineral worker may go briefly idle after collecting minerals.
// That's OK; we don't change its status then.
void WorkerManager::updateWorkerStatus()
{
	// for each of our Workers
	for (auto & worker : workerData.getWorkers())
	{
		if (!worker->isCompleted())
		{
			continue;     // the worker list includes drones in the egg
		}

		// TODO temporary debugging - see Micro::SmartMove
		// UAB_ASSERT(UnitUtil::IsValidUnit(worker), "bad worker");

		// If it's supposed to be on minerals but is actually collecting gas, fix it.
		// This can happen when we stop collecting gas; the worker can be mis-assigned.
		if (workerData.getWorkerJob(worker) == WorkerData::Minerals &&
			(worker->getOrder() == BWAPI::Orders::MoveToGas ||
			worker->getOrder() == BWAPI::Orders::WaitForGas ||
			worker->getOrder() == BWAPI::Orders::ReturnGas))
		{
			workerData.setWorkerJob(worker, WorkerData::Idle, nullptr);
		}

		// Idleness.
		// Order can be PlayerGuard for a drone that tries to build and fails.
		// There are other causes.
		if ((worker->isIdle() || worker->getOrder() == BWAPI::Orders::PlayerGuard) &&
			workerData.getWorkerJob(worker) != WorkerData::Minerals &&
			workerData.getWorkerJob(worker) != WorkerData::Build &&
			workerData.getWorkerJob(worker) != WorkerData::Move &&
			workerData.getWorkerJob(worker) != WorkerData::Scout)
		{
			workerData.setWorkerJob(worker, WorkerData::Idle, nullptr);
		}

		// If the worker is busy mining gas and an enemy comes near, maybe fight it.
		else if (workerData.getWorkerJob(worker) == WorkerData::Gas)
		{
			// Special micro eg. storm dodge
			//if (Micro::CheckSpecialCases(worker)) continue;

			BWAPI::Unit refinery = workerData.getWorkerResource(worker);

			// If the refinery is gone.
			// A missing resource depot is dealt with in handleGasWorkers().
			if (!refinery || !refinery->exists() || refinery->getHitPoints() <= 0)
			{
				workerData.setWorkerJob(worker, WorkerData::Idle, nullptr);
			}
			else
			{
				// Self-defense.
				BWAPI::Unit target = getClosestEnemyUnit(worker);

				if (target &&
					(!target->isMoving() || target->isBraking() || target->isStuck()) &&
					worker->getDistance(target) <= 64)
				{
					Micro::CatchAndAttackUnit(worker, target);
				}
				else if (worker->getOrder() != BWAPI::Orders::MoveToGas &&
					worker->getOrder() != BWAPI::Orders::WaitForGas &&
					worker->getOrder() != BWAPI::Orders::HarvestGas &&
					worker->getOrder() != BWAPI::Orders::ReturnGas &&
					worker->getOrder() != BWAPI::Orders::ResetCollision)
				{
					workerData.setWorkerJob(worker, WorkerData::Idle, nullptr);
				}
			}
		}

		// If the worker is busy mining minerals and an enemy comes near, maybe fight it.
		else if (workerData.getWorkerJob(worker) == WorkerData::Minerals)
		{
			// Special micro eg. storm dodge
			//if (Micro::CheckSpecialCases(worker)) continue;

			// Self-defense.
			BWAPI::Unit target = getClosestEnemyUnit(worker);

			if (target &&
				(!target->isMoving() || target->isBraking() || target->isStuck()) &&
				worker->getDistance(target) <= 64)
			{
				Micro::CatchAndAttackUnit(worker, target);
			}
			else if (worker->getOrder() != BWAPI::Orders::MoveToMinerals &&
				worker->getOrder() != BWAPI::Orders::WaitForMinerals &&
				worker->getOrder() != BWAPI::Orders::MiningMinerals &&
				worker->getOrder() != BWAPI::Orders::ReturnMinerals &&
				worker->getOrder() != BWAPI::Orders::ResetCollision)
			{
				workerData.setWorkerJob(worker, WorkerData::Idle, nullptr);
			}
		}
	}
}

void WorkerManager::setRepairWorker(BWAPI::Unit worker, BWAPI::Unit unitToRepair)
{
    workerData.setWorkerJob(worker, WorkerData::Repair, unitToRepair);
}

void WorkerManager::stopRepairing(BWAPI::Unit worker)
{
    workerData.setWorkerJob(worker, WorkerData::Idle, nullptr);
}

void WorkerManager::handleGasWorkers() 
{
	// Don't collect gas if gas collection is off, or if the resource depot is missing.
	if (_collectGas)
	{
		for (auto unit : BWAPI::Broodwar->self()->getUnits())
		{
			// if that unit is a refinery
			if (unit->getType().isRefinery() && unit->isCompleted() && refineryHasDepot(unit))
			{
				// Gather gas: If too few are assigned to this refinery, add more.
				int numAssigned = workerData.getNumAssignedWorkers(unit);
				for (int i = 0; i < (Config::Macro::WorkersPerRefinery - numAssigned); ++i)
				{
					BWAPI::Unit gasWorker = getGasWorker(unit);
					if (gasWorker)
					{
						workerData.setWorkerJob(gasWorker, WorkerData::Gas, unit);
					}
					else {
						return;    // won't find any more
					}
				}
			}
			else if (unit->getType().isRefinery() && unit->isCompleted() && !refineryHasDepot(unit))
			{
				// Don't gather gas: if this refinery does not have a depot, remove workers from it
				std::set<BWAPI::Unit> gasWorkers;
				workerData.getGasWorkers(gasWorkers);
				for (auto gasWorker : gasWorkers)
				{
					if (unit == workerData.getWorkerResource(gasWorker) && 
						gasWorker->getOrder() != BWAPI::Orders::HarvestGas)    // not inside the refinery
					{
						workerData.setWorkerJob(gasWorker, WorkerData::Idle, nullptr);
					}
				}
			}
		}
	}
	else
	{
		// Don't gather gas: If any workers are assigned, take them off.
		std::set<BWAPI::Unit> gasWorkers;
		workerData.getGasWorkers(gasWorkers);
		for (auto gasWorker : gasWorkers)
		{
			if (gasWorker->getOrder() != BWAPI::Orders::HarvestGas)    // not inside the refinery
			{
				workerData.setWorkerJob(gasWorker, WorkerData::Idle, nullptr);
				// An idle worker carrying gas will become a ReturnCargo worker,
				// so gas will not be lost needlessly.
			}
		}
	}
}

// Is our refinery near a resource depot that it can deliver gas to?
bool WorkerManager::refineryHasDepot(BWAPI::Unit refinery)
{
	// Iterate through units, not bases, because even if the main hatchery is destroyed
	// (so the base is considered gone), a macro hatchery may be close enough.
	// TODO could iterate through bases (from InfoMan) instead of units
	for (auto unit : BWAPI::Broodwar->self()->getUnits())
	{
		if (unit->getType().isResourceDepot() &&
			(unit->isCompleted() || 
			unit->getType() == BWAPI::UnitTypes::Zerg_Lair ||
			unit->getType() == BWAPI::UnitTypes::Zerg_Hive) &&
			unit->getDistance(refinery) < 600)
		{
			return true;
		}
	}

	return false;
}

void WorkerManager::handleIdleWorkers() 
{
	for (auto & worker : workerData.getWorkers())
	{
        UAB_ASSERT(worker != nullptr, "Worker was null");

		// if it is idle
		if (worker->isCompleted() && 
			workerData.getWorkerJob(worker) == WorkerData::Idle)
		{
			if (worker->isCarryingMinerals() || worker->isCarryingGas()) {
				// It's carrying something, set it to hand in its cargo.
				setReturnCargoWorker(worker);         // only happens if there's a resource depot
			}
			else {
				// Otherwise send it to mine minerals.
				setMineralWorker(worker);             // only happens if there's a resource depot
			}
		}
	}
}

void WorkerManager::handleReturnCargoWorkers()
{
	for (auto & worker : workerData.getWorkers())
	{
		UAB_ASSERT(worker != nullptr, "Worker was null");

		if (workerData.getWorkerJob(worker) == WorkerData::ReturnCargo)
		{
			// Special micro eg. storm dodge
			//if (Micro::CheckSpecialCases(worker)) continue;

			// If it still needs to return cargo, return it; otherwise go idle.
			// We have to make sure it has a resource depot to return cargo to.
			BWAPI::Unit depot;
			if ((worker->isCarryingMinerals() || worker->isCarryingGas()) &&
				(depot = getClosestDepot(worker)) &&
				worker->getDistance(depot) < 600)
			{
				Micro::SmartReturnCargo(worker);
			}
			else
			{
				// Can't return cargo. Let's be a mineral worker instead--if possible.
				setMineralWorker(worker);
			}
		}
	}
}

void WorkerManager::handleMoveWorkers()
{
	for (auto & worker : workerData.getWorkers())
	{
		UAB_ASSERT(worker != nullptr, "Worker was null");

		// if it is a move worker
		if (workerData.getWorkerJob(worker) == WorkerData::Move)
		{
			BWAPI::Unit depot;
			if ((worker->isCarryingMinerals() || worker->isCarryingGas()) &&
				(depot = getClosestDepot(worker)) &&
				worker->getDistance(depot) <= 100)
			{
				// A move worker is being sent to build or something.
				// Don't let it carry minerals or gas around wastefully.
				Micro::SmartReturnCargo(worker);
			}
			else
			{
				WorkerMoveData data = workerData.getWorkerMoveData(worker);
				//Micro::SmartMovePath(worker, data.position);
				Micro::SmartMove(worker, data.position);
			}
		}
	}
}

void WorkerManager::handleRepairWorkers()
{
    if (BWAPI::Broodwar->self()->getRace() != BWAPI::Races::Terran)
    {
        return;
    }

    for (auto & unit : BWAPI::Broodwar->self()->getUnits())
    {
        if (unit->getType().isBuilding() && (unit->getHitPoints() < unit->getType().maxHitPoints()))
        {
            BWAPI::Unit repairWorker = getClosestMineralWorkerTo(unit);
            setRepairWorker(repairWorker, unit);
            break;
        }
    }
}

// Used for combat workers
// Only count enemy units that can be targeted by workers.
BWAPI::Unit WorkerManager::getBestEnemyUnit(BWAPI::Unit worker)
{
	UAB_ASSERT(worker != nullptr, "Worker was null");

	BWAPI::Unit closestUnit = nullptr;
	int dist = 9999;
	int closestDist = 9999;         // ignore anything farther away

	for (auto & unit : BWAPI::Broodwar->enemy()->getUnits())
	{
		if (unit->isFlying())
		{
			continue;
		}

		dist = unit->getDistance(worker);
		if (!unit->isMoving() || unit->isBraking() || unit->isStuck())
		{
			dist -= 128;
		}
		if (unit->getType().topSpeed() >= worker->getType().topSpeed())
		{
			dist += 64;
		}
		if (dist < closestDist)
		{
			closestUnit = unit;
			closestDist = dist;
		}
	}

	return closestUnit;
}

// Used for worker self-defense.
// Only count enemy units that can be targeted by workers.
BWAPI::Unit WorkerManager::getClosestEnemyUnit(BWAPI::Unit worker)
{
	UAB_ASSERT(worker != nullptr, "Worker was null");

	BWAPI::Unit closestUnit = nullptr;
	int closestDist = 1000;         // ignore anything farther away

	for (auto & unit : BWAPI::Broodwar->enemy()->getUnits())
	{
		if (unit->isFlying())
		{
			continue;
		}

		int dist = unit->getDistance(worker);
		if (dist < closestDist)
		{
			closestUnit = unit;
			closestDist = dist;
		}
	}

	return closestUnit;
}

BWAPI::Unit WorkerManager::getClosestMineralWorkerTo(BWAPI::Unit unit)
{
    UAB_ASSERT(unit != nullptr, "Unit was null");

    BWAPI::Unit closestMineralWorker = nullptr;
    int closestDist = 100000;

	// Former closest worker may have died or (if zerg) morphed into a building.
	if (UnitUtil::IsValidUnit(previousClosestWorker) && previousClosestWorker->getType().isWorker())
	{
		return previousClosestWorker;
	}

    // for each of our workers
	for (auto & worker : workerData.getWorkers())
	{
        UAB_ASSERT(worker != nullptr, "Worker was null");

		// if it is a mineral worker
		if (worker->isCompleted() &&
			(workerData.getWorkerJob(worker) == WorkerData::Minerals || 
			workerData.getWorkerJob(worker) == WorkerData::Idle))
		{
			int dist = worker->getDistance(unit);
			if (worker->isCarryingMinerals() || worker->isCarryingGas()) 
			{
				// If it has cargo, pretend it is farther away.
				// That way we prefer empty workers and lose less cargo.
				dist += 64;
			}

            if (dist < closestDist)
            {
                closestMineralWorker = worker;
                dist = closestDist;
            }
		}
	}

    previousClosestWorker = closestMineralWorker;
    return closestMineralWorker;
}

BWAPI::Unit WorkerManager::getClosestBlockingMinWorkerTo(BWAPI::Unit unit)
{
	UAB_ASSERT(unit != nullptr, "Unit was null");

	BWAPI::Unit closestWorker = nullptr;
	int closestDist = 100000;

	// Former closest worker may have died or (if zerg) morphed into a building.
	if (UnitUtil::IsValidUnit(previousClosestBlockingMinWorker) && previousClosestBlockingMinWorker->getType().isWorker())
	{
		return previousClosestBlockingMinWorker;
	}

	// for each of our workers
	for (auto & worker : workerData.getWorkers())
	{
		UAB_ASSERT(worker != nullptr, "Worker was null");

		// if it is a mineral worker
		if (worker->isCompleted() &&
			(workerData.getWorkerJob(worker) == WorkerData::Minerals ||
			workerData.getWorkerJob(worker) == WorkerData::Idle))
		{
			int dist = worker->getDistance(unit);
			if (!worker->isCarryingMinerals() && !worker->isCarryingGas() && 
				dist < closestDist && dist < 1200)
			{
				closestWorker = worker;
				closestDist = dist;
			}
		}
	}

	previousClosestBlockingMinWorker = closestWorker;
	return closestWorker;
}

BWAPI::Unit WorkerManager::getWorkerScout()
{
	for (auto & worker : workerData.getWorkers())
	{
		UAB_ASSERT(worker != nullptr, "Worker was null");
		if (workerData.getWorkerJob(worker) == WorkerData::Scout)
		{
			return worker;
		}
	}

	return nullptr;
}

// Send the worker to mine minerals at the closest resource depot, if any.
void WorkerManager::setMineralWorker(BWAPI::Unit unit)
{
    UAB_ASSERT(unit != nullptr, "Unit was null");

	BWAPI::Unit depot = getClosestDepot(unit);

	if (depot)
	{
		workerData.setWorkerJob(unit, WorkerData::Minerals, depot);
	}
	else
	{
		// BWAPI::Broodwar->printf("No valid depot for mineral worker");
	}
}

// Worker is carrying minerals or gas. Tell it to hand them in.
void WorkerManager::setReturnCargoWorker(BWAPI::Unit unit)
{
	UAB_ASSERT(unit != nullptr, "Unit was null");

	BWAPI::Unit depot = getClosestDepot(unit);

	if (depot)
	{
		workerData.setWorkerJob(unit, WorkerData::ReturnCargo, depot);
	}
	else
	{
		// BWAPI::Broodwar->printf("No valid depot to accept return cargo");
	}
}

BWAPI::Unit WorkerManager::getClosestDepot(BWAPI::Unit worker)
{
	UAB_ASSERT(worker != nullptr, "Worker was null");

	BWAPI::Unit closestDepot = nullptr;
	int closestDistance = 0;

	for (auto & unit : BWAPI::Broodwar->self()->getUnits())
	{
        UAB_ASSERT(unit != nullptr, "Unit was null");

		if (unit->getType().isResourceDepot() && (unit->isCompleted() || 
			unit->getType() == BWAPI::UnitTypes::Zerg_Lair || 
			unit->getType() == BWAPI::UnitTypes::Zerg_Hive) && !workerData.depotIsFull(unit))
		{
			int distance = unit->getDistance(worker);
			if (!closestDepot || distance < closestDistance)
			{
				closestDepot = unit;
				closestDistance = distance;
			}
		}
	}

	return closestDepot;
}

// other managers that need workers call this when they're done with a unit
void WorkerManager::finishedWithWorker(BWAPI::Unit unit) 
{
	UAB_ASSERT(unit != nullptr, "Unit was null");

	//BWAPI::Broodwar->printf("finished with worker %d", unit->getID());
	workerData.setWorkerJob(unit, WorkerData::Idle, nullptr);
}

// Find a worker to be reassigned to gas duty.
BWAPI::Unit WorkerManager::getGasWorker(BWAPI::Unit refinery)
{
	UAB_ASSERT(refinery != nullptr, "Refinery was null");

	BWAPI::Unit closestWorker = nullptr;
	int closestDistance = 0;

	for (auto & unit : workerData.getWorkers())
	{
		UAB_ASSERT(unit != nullptr, "Unit was null");

		if (unit->isCompleted() &&
			(workerData.getWorkerJob(unit) == WorkerData::Minerals ||
			workerData.getWorkerJob(unit) == WorkerData::Idle))
		{
			// Don't waste minerals. It's OK (and unlikely) to already be carrying gas.
			if (unit->isCarryingMinerals() &&                       // doesn't have minerals and
				unit->getOrder() != BWAPI::Orders::MiningMinerals)  // isn't about to get them
			{
				continue;
			}

			int distance = unit->getDistance(refinery);
			if (!closestWorker || distance < closestDistance)
			{
				closestWorker = unit;
				closestDistance = distance;
			}
		}
	}

	return closestWorker;
}

void WorkerManager::setBuildingWorker(BWAPI::Unit worker, Building & b)
{
     UAB_ASSERT(worker != nullptr, "Worker was null");

     workerData.setWorkerJob(worker, WorkerData::Build, b.type);
}

// Get a builder for BuildingManager.
// if setJobAsBuilder is true (default), it will be flagged as a builder unit
// set 'setJobAsBuilder' to false if we just want to see which worker will build a building
BWAPI::Unit WorkerManager::getBuilder(const Building & b, bool setJobAsBuilder)
{
	// variables to hold the closest worker of each type to the building
	BWAPI::Unit closestMovingWorker = nullptr;
	BWAPI::Unit closestMiningWorker = nullptr;
	int closestMovingWorkerDistance = 0;
	int closestMiningWorkerDistance = 0;

	// look through each worker that had moved there first
	for (auto & unit : workerData.getWorkers())
	{
        UAB_ASSERT(unit != nullptr, "Unit was null");

        // gas steal building uses scout worker
        if (b.isGasSteal && (workerData.getWorkerJob(unit) == WorkerData::Scout))
        {
            if (setJobAsBuilder)
            {
                workerData.setWorkerJob(unit, WorkerData::Build, b.type);
            }
            return unit;
        }

		// mining worker check
		if (unit->isCompleted() &&
			(workerData.getWorkerJob(unit) == WorkerData::Minerals ||
			workerData.getWorkerJob(unit) == WorkerData::Idle))
		{
			// if it is a new closest distance, set the pointer
			int distance = unit->getDistance(BWAPI::Position(b.finalPosition));
			if (unit->isCarryingMinerals() || unit->isCarryingGas() ||
				unit->getOrder() == BWAPI::Orders::MiningMinerals)
			{
				// If it has cargo or is busy getting some, pretend it is farther away.
				// That way we prefer empty workers and lose less cargo.
				distance += 96;
			}
			if (!closestMiningWorker || distance < closestMiningWorkerDistance)
			{
				closestMiningWorker = unit;
				closestMiningWorkerDistance = distance;
			}
		}

		// moving worker check
		if (unit->isCompleted() && (workerData.getWorkerJob(unit) == WorkerData::Move))
		{
			// if it is a new closest distance, set the pointer
			int distance = unit->getDistance(BWAPI::Position(b.finalPosition));
			if (unit->isCarryingMinerals() || unit->isCarryingGas() ||
				unit->getOrder() == BWAPI::Orders::MiningMinerals) {
				// If it has cargo or is busy getting some, pretend it is farther away.
				// That way we prefer empty workers and lose less cargo.
				distance += 96;
			}
			if (!closestMovingWorker || distance < closestMovingWorkerDistance)
			{
				closestMovingWorker = unit;
				closestMovingWorkerDistance = distance;
			}
		}
	}

	// if we found a moving worker, use it, otherwise using a mining worker
	BWAPI::Unit chosenWorker = closestMovingWorker ? closestMovingWorker : closestMiningWorker;

	// if the worker exists (one may not have been found in rare cases)
	if (chosenWorker && setJobAsBuilder)
	{
		workerData.setWorkerJob(chosenWorker, WorkerData::Build, b.type);
	}

	return chosenWorker;
}

// sets a worker as a scout
void WorkerManager::setScoutWorker(BWAPI::Unit worker)
{
	UAB_ASSERT(worker != nullptr, "Worker was null");

	workerData.setWorkerJob(worker, WorkerData::Scout, nullptr);
}

// gets a worker which will move to a current location
BWAPI::Unit WorkerManager::getMoveWorker(BWAPI::Position p)
{
	// set up the pointer
	BWAPI::Unit closestWorker = nullptr;
	int closestDistance = 0;

	// for each worker we currently have
	for (auto & unit : workerData.getWorkers())
	{
        UAB_ASSERT(unit != nullptr, "Unit was null");

		// only consider it if it's a mineral worker
		if (unit->isCompleted() &&
			(workerData.getWorkerJob(unit) == WorkerData::Minerals ||
			workerData.getWorkerJob(unit) == WorkerData::Idle))
		{
			// if it is a new closest distance, set the pointer
			int distance = unit->getDistance(p);
			if (unit->isCarryingMinerals() || unit->isCarryingGas() ||
				unit->getOrder() == BWAPI::Orders::MiningMinerals) {
				// If it has cargo or is busy getting some, pretend it is farther away.
				// That way we prefer empty workers and lose less cargo.
				distance += 96;
			}
			if (!closestWorker || distance < closestDistance)
			{
				closestWorker = unit;
				closestDistance = distance;
			}
		}
	}

	// return the worker
	return closestWorker;
}

// Sets a worker to move to a given location. Use getMoveWorker() to choose the worker.
void WorkerManager::setMoveWorker(BWAPI::Unit worker, int mineralsNeeded, int gasNeeded, BWAPI::Position & p)
{
	UAB_ASSERT(worker && p.isValid(), "bad call");
	workerData.setWorkerJob(worker, WorkerData::Move, WorkerMoveData(mineralsNeeded, gasNeeded, p));
	//BWAPI::Broodwar->printf("set move worker %d", worker->getID());
}
/*
// sets a worker to move to a given location
void WorkerManager::setMoveWorker(int mineralsNeeded, int gasNeeded, BWAPI::Position p)
{
	// set up the pointer
	BWAPI::Unit closestWorker = nullptr;
	int closestDistance = 0;

	// for each worker we currently have
	for (auto & unit : workerData.getWorkers())
	{
        UAB_ASSERT(unit != nullptr, "Unit was null");

		// only consider it if it's a mineral worker
		if (unit->isCompleted() &&
			(workerData.getWorkerJob(unit) == WorkerData::Minerals ||
			workerData.getWorkerJob(unit) == WorkerData::Idle))
		{
			// if it is a new closest distance, set the pointer
			int distance = unit->getDistance(p);
			if (unit->isCarryingMinerals() || unit->isCarryingGas()) {
				// If it has cargo, pretend it is farther away.
				// That way we prefer empty workers and lose less cargo.
				distance += 64;
			}
			if (!closestWorker || distance < closestDistance)
			{
				closestWorker = unit;
				closestDistance = distance;
			}
		}
	}

	if (closestWorker)
	{
		//BWAPI::Broodwar->printf("Setting worker job Move for worker %d", closestWorker->getID());
		workerData.setWorkerJob(closestWorker, WorkerData::Move, WorkerMoveData(mineralsNeeded, gasNeeded, p));
	}
	else
	{
		//BWAPI::Broodwar->printf("no worker found");
	}
}
*/
// will we have the required resources by the time a worker can travel a certain distance
bool WorkerManager::willHaveResources(int mineralsRequired, int gasRequired, double distance)
{
	// if we don't require anything, we will have it
	if (mineralsRequired <= 0 && gasRequired <= 0)
	{
		return true;
	}

	// the speed of the worker unit
	double speed = BWAPI::Broodwar->self()->getRace().getWorker().topSpeed();

    UAB_ASSERT(speed > 0, "Speed is negative");

	// how many frames it will take us to move to the building location
	// add a second to account for worker getting stuck. better early than late
	double framesToMove = (distance / speed) + 24 + std::min(0.0, 0.1 * (distance - 100) / speed);

	// magic numbers to predict income rates
	double mineralRate = getNumMineralWorkers() * 0.045;
	double gasRate     = getNumGasWorkers() * 0.07;

	// calculate if we will have enough by the time the worker gets there
	return
		mineralRate * framesToMove >= mineralsRequired &&
		gasRate * framesToMove >= gasRequired;
}

void WorkerManager::setCombatWorker(BWAPI::Unit worker)
{
	UAB_ASSERT(worker != nullptr, "Worker was null");

	//BWAPI::Broodwar->printf("set combat worker %d", worker->getID());
	workerData.setWorkerJob(worker, WorkerData::Combat, nullptr);
}

void WorkerManager::onUnitMorph(BWAPI::Unit unit)
{
	UAB_ASSERT(unit != nullptr, "Unit was null");

	// if something morphs into a worker, add it
	if (unit->getType().isWorker() && unit->getPlayer() == BWAPI::Broodwar->self() && unit->getHitPoints() >= 0)
	{
		workerData.addWorker(unit);
	}

	// if something morphs into a building, was it a drone?
	if (unit->getType().isBuilding() && unit->getPlayer() == BWAPI::Broodwar->self() && unit->getPlayer()->getRace() == BWAPI::Races::Zerg)
	{
		//BWAPI::Broodwar->printf("A Drone started building");
		workerData.workerDestroyed(unit);
	}
}

void WorkerManager::onUnitShow(BWAPI::Unit unit)
{
	UAB_ASSERT(unit != nullptr, "Unit was null");

	// add the depot if it exists
	if (unit->getType().isResourceDepot() && unit->getPlayer() == BWAPI::Broodwar->self())
	{
		workerData.addDepot(unit);
	}

	// if something morphs into a worker, add it
	if (unit->getType().isWorker() && unit->getPlayer() == BWAPI::Broodwar->self() && unit->getHitPoints() >= 0)
	{
		//BWAPI::Broodwar->printf("A worker was shown %d", unit->getID());
		workerData.addWorker(unit);
	}
}

void WorkerManager::rebalanceWorkers()
{
	// for each worker
	for (auto & worker : workerData.getWorkers())
	{
        UAB_ASSERT(worker != nullptr, "Worker was null");

		if (!workerData.getWorkerJob(worker) == WorkerData::Minerals)
		{
			continue;
		}

		BWAPI::Unit depot = workerData.getWorkerDepot(worker);

		if (depot && workerData.depotIsFull(depot))
		{
			workerData.setWorkerJob(worker, WorkerData::Idle, nullptr);
		}
		else if (!depot)
		{
			workerData.setWorkerJob(worker, WorkerData::Idle, nullptr);
		}
	}
}

void WorkerManager::onUnitDestroy(BWAPI::Unit unit) 
{
	UAB_ASSERT(unit != nullptr, "Unit was null");

	if (unit->getType().isResourceDepot() && unit->getPlayer() == BWAPI::Broodwar->self())
	{
		workerData.removeDepot(unit);
	}

	if (unit->getType().isWorker() && unit->getPlayer() == BWAPI::Broodwar->self()) 
	{
		workerData.workerDestroyed(unit);
	}

	if (unit->getType() == BWAPI::UnitTypes::Resource_Mineral_Field)
	{
		rebalanceWorkers();
	}
}

void WorkerManager::drawResourceDebugInfo() 
{
    if (!Config::Debug::DrawResourceInfo)
    {
        return;
    }

	for (auto & worker : workerData.getWorkers()) 
    {
        UAB_ASSERT(worker != nullptr, "Worker was null");

		char job = workerData.getJobCode(worker);

		BWAPI::Position pos = worker->getTargetPosition();

		BWAPI::Broodwar->drawTextMap(worker->getPosition().x, worker->getPosition().y - 5, "\x07%c", job);
		BWAPI::Broodwar->drawTextMap(worker->getPosition().x, worker->getPosition().y + 5, "\x03%s", worker->getOrder().getName().c_str());

		BWAPI::Broodwar->drawLineMap(worker->getPosition().x, worker->getPosition().y, pos.x, pos.y, BWAPI::Colors::Cyan);

		BWAPI::Unit depot = workerData.getWorkerDepot(worker);
		if (depot)
		{
			BWAPI::Broodwar->drawLineMap(worker->getPosition().x, worker->getPosition().y, depot->getPosition().x, depot->getPosition().y, BWAPI::Colors::Orange);
		}
	}
}

void WorkerManager::drawWorkerInformation(int x, int y) 
{
    if (!Config::Debug::DrawWorkerInfo)
    {
        return;
    }

	BWAPI::Broodwar->drawTextScreen(x, y, "\x04 Workers %d", workerData.getNumWorkers());
	BWAPI::Broodwar->drawTextScreen(x, y+20, "\x04 UnitID");
	BWAPI::Broodwar->drawTextScreen(x+50, y+20, "\x04 State");

	int yspace = 0;

	for (auto & unit : workerData.getWorkers())
	{
        UAB_ASSERT(unit != nullptr, "Worker was null");

		BWAPI::Broodwar->drawTextScreen(x, y+40+((yspace)*10), "\x03 %d", unit->getID());
		BWAPI::Broodwar->drawTextScreen(x+50, y+40+((yspace++)*10), "\x03 %c", workerData.getJobCode(unit));
	}
}

bool WorkerManager::isFree(BWAPI::Unit worker)
{
    UAB_ASSERT(worker != nullptr, "Worker was null");

	return worker->isCompleted() && 
		(workerData.getWorkerJob(worker) == WorkerData::Minerals || 
		workerData.getWorkerJob(worker) == WorkerData::Idle);
}

bool WorkerManager::isWorkerScout(BWAPI::Unit worker)
{
    UAB_ASSERT(worker != nullptr, "Worker was null");

	return (workerData.getWorkerJob(worker) == WorkerData::Scout);
}

bool WorkerManager::isBuilder(BWAPI::Unit worker)
{
    UAB_ASSERT(worker != nullptr, "Worker was null");

	return (workerData.getWorkerJob(worker) == WorkerData::Build);
}

int WorkerManager::getNumMineralWorkers() const
{
	return workerData.getNumMineralWorkers();	
}

int WorkerManager::getNumIdleWorkers() const
{
	return workerData.getNumIdleWorkers();	
}

int WorkerManager::getNumGasWorkers() const
{
	return workerData.getNumGasWorkers();
}

int WorkerManager::getNumReturnCargoWorkers() const
{
	return workerData.getNumReturnCargoWorkers();
}

int WorkerManager::getNumCombatWorkers() const
{
	return workerData.getNumCombatWorkers();
}

// The largest number of workers that it is efficient to have right now.
// NOTE Does not take into account possible preparations for future expansions.
int WorkerManager::getMaxWorkers() const
{
	int patches = InformationManager::Instance().getMyNumMineralPatches();
	int refineries = InformationManager::Instance().getMyNumRefineries();

	// Never let the max number of workers fall to 0!
	// Set aside 1 for future opportunities.
	return std::max(1, int((Config::Macro::WorkersPerPatch * patches) + (Config::Macro::WorkersPerRefinery * refineries) + 1));
}


void WorkerManager::handleBlockingMineralsWorkers() {

	BWAPI::Unit worker;
	int x = 0;

	// Get all mineral patches
	BWAPI::Unitset minerals;
	for (auto & unit : BWAPI::Broodwar->getAllUnits())
	{
		//if (UnitUtil::IsValidUnit(unit))
		//{
			// if the minerals have <8 resouces, mine them
			if (unit->getType().isMineralField() && unit->getResources() < 8)
			{
				//BWAPI::Broodwar->printf("found blocking minerals");

				// get closest worker to this mineral patch
				worker = getClosestBlockingMinWorkerTo(unit);

				if (UnitUtil::IsValidUnit(worker))
				{
					x++;
					//BWAPI::Broodwar->printf("found blocking minerals worker %d: %d (%d,%d) dist = %d", x, unit->getResources(), unit->getPosition().x, unit->getPosition().y, unit->getDistance(worker->getPosition()));

					if (unit->getDistance(worker) < 1200)
					{
						//BWAPI::Broodwar->printf("mine blocking minerals worker %d: %d", x, unit->getResources());

						if (unit->getDistance(worker) < 32)
						{
							return;
						}

						worker->gather(unit);
						return;
					}
				}
			}
		//}
	}
}

bool WorkerManager::handleBlockingMineralsBuilders(BWAPI::Unit worker) {

	UAB_ASSERT(worker != nullptr, "worker was null");

	int x = 0;

	// Get all mineral patches
	BWAPI::Unitset minerals;
	for (auto & unit : BWAPI::Broodwar->getAllUnits())
	{
		//if (UnitUtil::IsValidUnit(unit))
		//{
			// if the minerals have <8 resouces, mine them
			if (unit->getType().isMineralField() && unit->getResources() < 8)
			{
				//BWAPI::Broodwar->printf("found blocking minerals");

				if (UnitUtil::IsValidUnit(worker) && unit->getDistance(worker) < 1200)
				{
					x++;
					//BWAPI::Broodwar->printf("found blocking minerals builder %d: %d (%d,%d) dist = %d", x, unit->getResources(), unit->getPosition().x, unit->getPosition().y, unit->getDistance(worker->getPosition()));

					if (unit->getDistance(worker) < 100)
					{
						//BWAPI::Broodwar->printf("mine blocking minerals builder %d: %d", x, unit->getResources());

						if (worker->isGatheringMinerals())
						{
							return true;
						}

						worker->gather(unit);
						return true;
					}
				}
			}
		//}
	}

	return false;
}


