#pragma once

#include "Common.h"
#include "BWTA.h"
#include "UnitData.h"

namespace UAlbertaBot
{
struct BaseInfo
{
	BWTA::BaseLocation *	location;
	BWAPI::Unit				resourceDepot;

	BaseInfo(BWTA::BaseLocation * loc, BWAPI::Unit depot)
		: location(loc)
		, resourceDepot(depot)
	{
	}
};

struct BaseStatus
{
public:
	BWAPI::Unit  	resourceDepot;		// hatchery, etc.; valid iff depotSeen
	BWAPI::Player	owner;              // self, enemy, neutral
	bool			reserved;			// if this is our planned expansion

	// The resourceDepot pointer is set for a base if the depot has been seen.
	// It is possible to infer a base location without seeing the depot.

	BaseStatus()
		: resourceDepot(nullptr)
		, owner(BWAPI::Broodwar->neutral())
		, reserved(false)
	{
	}

	BaseStatus(BWAPI::Unit depot, BWAPI::Player player)
		: resourceDepot(depot)
		, owner(player)
		, reserved(false)
	{
	}

	BaseStatus(BWAPI::Unit depot, BWAPI::Player player, bool res)
		: resourceDepot(depot)
		, owner(player)
		, reserved(res)
	{
	}
};

class InformationManager 
{
	BWAPI::Player _self;
    BWAPI::Player _enemy;
	BWAPI::Playerset _allies;

    std::map<BWAPI::Player, UnitData>                   _unitData;
    std::map<BWAPI::Player, BWTA::BaseLocation *>       _mainBaseLocations;
	BWTA::BaseLocation *								_myNaturalBaseLocation;  // whether taken yet or not
	std::map<BWAPI::Player, std::set<BWTA::Region *> >  _occupiedRegions;        // contains any building
	std::map<BWTA::BaseLocation *, BaseStatus>			_theBases;

	bool												_enemyProxy;
	bool												_weHaveCombatUnits;
	bool												_enemyHasAirUnits;
	bool												_enemyHasAntiAir;
	bool												_enemyHasAirTech;
	bool												_enemyHasCloakTech;
	bool												_enemyHasMobileCloakTech;
	bool												_enemyHasOverlordHunters;
	bool												_enemyHasLiftedBuildings;
	bool												_enemyHasVultures;
	int													_lastUpdateFrame;

	InformationManager();

	void                    initializeRegionInformation();
	void					initializeNaturalBase();
	void					initializeNaturalChokepoint();

	int                     getIndex(BWAPI::Player player) const;

	void					baseInferred(BWTA::BaseLocation * base);
	void					baseFound(BWAPI::Unit depot);
	void					baseFound(BWTA::BaseLocation * base, BWAPI::Unit depot);
	void					baseLost(BWAPI::TilePosition basePosition);
	void					baseLost(BWTA::BaseLocation * base);
	void					maybeAddBase(BWAPI::Unit unit);
	bool					closeEnough(BWAPI::TilePosition a, BWAPI::TilePosition b);
	void					chooseNewMainBase();

	void                    updateUnit(BWAPI::Unit unit);
    void                    updateUnitInfo();
    void                    updateBaseLocationInfo();
	void					updateTheBases();
    void                    updateOccupiedRegions(BWTA::Region * region,BWAPI::Player player);
	void					updateGoneFromLastPosition();
	void					updateBulletInfo();
    bool                    isValidUnit(BWAPI::Unit unit);

public:
	BWAPI::Unitset			scarabs;
	BWAPI::Unitset			nukes;
	BWAPI::Unitset			hazards;
	BWAPI::Bulletset		storms;
	BWAPI::Bulletset		emps; //unused
	BWAPI::Bulletset		ensnares; //unused

    void                    update();

    // event driven stuff
	void					onUnitShow(BWAPI::Unit unit)        { updateUnit(unit); maybeAddBase(unit); }
    void					onUnitHide(BWAPI::Unit unit)        { updateUnit(unit); }
	void					onUnitCreate(BWAPI::Unit unit)		{ updateUnit(unit); maybeAddBase(unit); }
    void					onUnitComplete(BWAPI::Unit unit)    { updateUnit(unit); }
	void					onUnitMorph(BWAPI::Unit unit)       { updateUnit(unit); maybeAddBase(unit); }
    void					onUnitRenegade(BWAPI::Unit unit)    { updateUnit(unit); }
    void					onUnitDestroy(BWAPI::Unit unit);
	
	bool					isEnemyBuildingInRegion(BWTA::Region * region);
    int						getNumUnits(BWAPI::UnitType type,BWAPI::Player player);
    bool					nearbyForceHasCloaked(BWAPI::Position p,BWAPI::Player player,int radius);
    bool					isCombatUnit(BWAPI::UnitType type) const;

    void                    getNearbyForce(std::vector<UnitInfo> & unitInfo,BWAPI::Position p,BWAPI::Player player,int radius);

    const UIMap &           getUnitInfo(BWAPI::Player player) const;
	const UnitData &        getUnitData(BWAPI::Player player) const;
	const UnitInfo &		getUnit(BWAPI::Player player, BWAPI::Unit unit) const;

    std::set<BWTA::Region *> &  getOccupiedRegions(BWAPI::Player player);
    BWTA::BaseLocation *    getMainBaseLocation(BWAPI::Player player);
	BWTA::BaseLocation *	getMyMainBaseLocation();
	BWTA::BaseLocation *	getEnemyMainBaseLocation();
	BWAPI::Player			getBaseOwner(BWTA::BaseLocation * base);
	BWAPI::Unit 			getBaseDepot(BWTA::BaseLocation * base);
	BWTA::BaseLocation *	getMyNaturalLocation();
	int						getNumBases(BWAPI::Player player);
	int						getNumFreeLandBases();
	int						getMyNumMineralPatches();
	int						getMyNumGeysers();
	int						getAir2GroundSupply(BWAPI::Player player) const;
	bool					getGroundOverAir() const;
	int						getMyNumRefineries();

	bool					isBaseReserved(BWTA::BaseLocation * base);
	void					reserveBase(BWTA::BaseLocation * base);
	void					unreserveBase(BWTA::BaseLocation * base);
	void					unreserveBase(BWAPI::TilePosition baseTilePosition);

	bool					weHaveCombatUnits();

	bool					enemyHasAirUnits();
	bool					enemyHasAntiAir();			// TODO
	bool					enemyHasAirTech();
	bool                    enemyHasCloakTech();
	bool                    enemyHasMobileCloakTech();
	bool					enemyHasOverlordHunters();
	bool					enemyHasLiftedBuildings();
	bool					enemyHasVultures();

	// Defense logic
	BWTA::BaseLocation *	getLeastDefendedBaseGround(BWAPI::Player player);
	BWTA::BaseLocation *	getLeastDefendedBaseAir(BWAPI::Player player);
	int						getNumEnemyUnitsVsGround(BWTA::BaseLocation *base);
	int						getNumBuildings(BWTA::BaseLocation *base, BWAPI::UnitType unittype);
	int						getNumSpores(BWTA::BaseLocation *base);
	int						getNumSunkens(BWTA::BaseLocation *base);
	int						getNumCreep(BWTA::BaseLocation *base);
	int						getNumWorkersIncoming();
	int						getEnemyPowerGroundUnitsIncoming();
	int						getEnemyPowerGroundUnits();
	int						getMyPowerGroundWeapon(bool withCreep);

	// Unit mix logic
	int						getPower(BWAPI::Player player);
	int						getCostArmy(BWAPI::Player player);
	int						getCostUnit(BWAPI::Player player, BWAPI::UnitType unittype);
	int						getCostUnit(BWAPI::UnitType unittype);
	int						getCostAir(BWAPI::Player player);
	int						getCostGround(BWAPI::Player player);
	int						getCostStaticDefenseAir(BWAPI::Player player);
	int						getCostStaticDefenseGround(BWAPI::Player player);
	int						getCostAntiAir(BWAPI::Player player);
	int						getCostAntiGround(BWAPI::Player player);
	int						getCostGroundAoE(BWAPI::Player player);
	int						getCostAntiMeleeAoE(BWAPI::Player player);
	int						getCostSmall(BWAPI::Player player);
	int						getCostMedium(BWAPI::Player player);
	int						getCostLarge(BWAPI::Player player);

	int						nScourgeNeeded();           // zerg specific

    void                    drawExtendedInterface();
    void                    drawUnitInformation(int x,int y);
    void                    drawMapInformation();
	void					drawBaseInformation(int x, int y);

	// yay for singletons!
	static InformationManager & Instance();
};
}
