#include "Dll.h"


MyBase::MyBase(Ximp *x, BWTA::BaseLocation* baseLoc): ximp(x), 
                                                      baseLocation(baseLoc),
                                                      defendingGroup(NULL),
                                                      isCapital(false),
                                                      isNatural(false),
                                                      isMinedOut(false),
                                                      isUnderAttack(false),
                                                      isGasSteal(false),
                                                      hasUnpoweredBuildings(false),
                                                      lastAttackedFrame(0),
                                                      lastSeenMineralCount(0),
                                                      waitPylon(NULL), 
                                                      buildingSides(0), 
                                                      chokepointSpaceDist(7), 
                                                      checkMineralsDist(4), 
                                                      pylonFreeSpace(1), 
                                                      buildingsFreeSpace(1), 
                                                      pylonEnergySpace(7),
                                                      isFull(false) { 
  regions.insert(baseLocation->getRegion()); 

  // Update regions
  set<BWTA::Region*> allRegions = BWTA::getRegions();
  for (set<BWTA::Region*>::iterator it = allRegions.begin(); it != allRegions.end(); it++) {
    // Electric Circuit
    if (Broodwar->mapHash() == "9505d618c63a0959f0c0bfe21c253a2ea6e58d26") {
      if (baseLocation->getPosition() == Position(288, 272) && (*it)->getCenter() == Position(722, 802)) {
        regions.insert(*it);
      }
    }
    // Judgement Day
    else if (Broodwar->mapHash() == "2f69eaa1a73bb743934d55e7ea12186fe340e656") {
      if (baseLocation->getPosition() == Position(3744, 2992) && (*it)->getCenter() == Position(3130, 3252)) {
        regions.insert(*it);
      }
      else if (baseLocation->getPosition() == Position(3744, 1136) && (*it)->getCenter() == Position(3139, 874)) {
        regions.insert(*it);
      }
      else if (baseLocation->getPosition() == Position(352, 1136) && (*it)->getCenter() == Position(945, 873)) {
        regions.insert(*it);
      }
      else if (baseLocation->getPosition() == Position(352, 2992) && (*it)->getCenter() == Position(953, 3253)) {
        regions.insert(*it);
      }
    }
  }

  // Update chokepoints
  for (set<BWTA::Region*>::iterator rt = regions.begin(); rt != regions.end(); rt++) {  
    set<BWTA::Chokepoint*> ch = (*rt)->getChokepoints();

    for (set<BWTA::Chokepoint*>::iterator it = ch.begin(); it != ch.end(); it++) {
      bool add = true;

      // Benzene
      if (Broodwar->mapHash() == "af618ea3ed8a8926ca7b17619eebcb9126f0d8b1") {
        if ( (*it)->getCenter() == Position(44, 1064) ||
             (*it)->getCenter() == Position(4044, 1088) ) {
          add = false;
        }
      }
      // Destination
      else if (Broodwar->mapHash() == "4e24f217d2fe4dbfa6799bc57f74d8dc939d425b") {
        if ( (*it)->getCenter() == Position(1309, 3851) ||
             (*it)->getCenter() == Position(1730, 226) ) {
          add = false;
        }
      }
      // Andromeda
      else if (Broodwar->mapHash() == "1e983eb6bcfa02ef7d75bd572cb59ad3aab49285") {
        if ( (*it)->getCenter() == Position(1300, 3584) ||
             (*it)->getCenter() == Position(2780, 3604) ||
             (*it)->getCenter() == Position(1292, 436) ||
             (*it)->getCenter() == Position(2776, 448) ) {
          add = false;
        }
      }
      // Electric Circuit
      else if (Broodwar->mapHash() == "9505d618c63a0959f0c0bfe21c253a2ea6e58d26" && !baseLocation->isStartLocation()) {
        if ( (*it)->getCenter() == Position(564, 520) ||
             (*it)->getCenter() == Position(271, 686) ||
             (*it)->getCenter() == Position(693, 3863) ||
             (*it)->getCenter() == Position(3842, 3374) || 
             (*it)->getCenter() == Position(3379, 208) ) {
          add = false;
        }
      }

      if (add) {
        chokepoints.insert(*it); 
      }
    }
  }

  int distant = 0;
  for (set<BWTA::Region*>::iterator it = regions.begin(); it != regions.end(); it++) {
    BWTA::Polygon poly = (*it)->getPolygon();

    for (int i = 0; i < poly.size(); i++) {
      int current = baseLocation->getPosition().getDistance(poly[i]);

      if (current > distant) {
        distant = current; 
      }
    }
  }
  maxBasePointDist = distant;

  int vecX = (*chokepoints.begin())->getCenter().x() - baseLocation->getPosition().x();
  int vecY = (*chokepoints.begin())->getCenter().y() - baseLocation->getPosition().y();

  buildingSides |= vecX <= 0 ? 1 : 4;
  buildingSides |= vecY <= 0 ? 8 : 2;

  set<BWTA::BaseLocation*> startLocations = BWTA::getStartLocations();
  set<Position> chokepointCenter;

  for (set<BWTA::Chokepoint*>::iterator it = chokepoints.begin(); it != chokepoints.end(); it++) {
    bool count = true;

    if (!baseLocation->isStartLocation()) {
      for (set<BWTA::BaseLocation*>::iterator locIt = startLocations.begin(); locIt != startLocations.end(); locIt++) {
        set<BWTA::Chokepoint*> locChokepoints = (*locIt)->getRegion()->getChokepoints();

        for (set<BWTA::Chokepoint*>::iterator jt = locChokepoints.begin(); jt != locChokepoints.end(); jt++) {
          if (*jt == *it) {
            count = false;
            break;
          }
        }

        if (!count) {
          break;
        }
      }
    }

    // Electric Circuit
    if (Broodwar->mapHash() == "9505d618c63a0959f0c0bfe21c253a2ea6e58d26") {
      if ((*it)->getCenter() == Position(916, 448)) {
        count = false;
      }
    }
    // Heartbreak Ridge
    else if (Broodwar->mapHash() == "6f8da3c3cc8d08d9cf882700efa049280aedca8c") {
      if ((*it)->getCenter() == Position(4056, 784) ||
          (*it)->getCenter() == Position(290, 1910)) {
        count = false;
      }
    }
    // Python
    else if (Broodwar->mapHash() == "de2ada75fbc741cfa261ee467bf6416b10f9e301") {
      if ((*it)->getCenter() == Position(1950, 203)) {
        count = true;
      }
    }
    // Destination
    else if (Broodwar->mapHash() == "4e24f217d2fe4dbfa6799bc57f74d8dc939d425b") {
      if ((*it)->getCenter() == Position(1331, 3681)) {
        count = true;
      }
    }
    // MoonGlaive
    else if (Broodwar->mapHash() == "c8386b87051f6773f6b2681b0e8318244aa086a6") {
      if ((*it)->getCenter() == Position(1612, 540)) {
        count = true;
      }
    }

    if (count) {
      chokepointCenter.insert((*it)->getCenter());
    }
  }
 
  // Judgement day
  if (Broodwar->mapHash() != "2f69eaa1a73bb743934d55e7ea12186fe340e656") {
    chokepointCenter.insert(baseLocation->getPosition());
  }

  defenseCentroid = ximp->computeCentroid(chokepointCenter);
  
  // Python
  if (Broodwar->mapHash() == "de2ada75fbc741cfa261ee467bf6416b10f9e301") {
    if (baseLocation->getPosition() == Position(1216, 3024)) {
      defenseCentroid = Position(1408, 2848);
    }
    else if (baseLocation->getPosition() == Position(2816, 1136)) {
      defenseCentroid = Position(2624, 1312);
    }
  }

  pylonsCentroid = Position((defenseCentroid.x() + baseLocation->getPosition().x()) / 2, (defenseCentroid.y() + baseLocation->getPosition().y()) / 2);


  // Judgement day
  if (Broodwar->mapHash() == "2f69eaa1a73bb743934d55e7ea12186fe340e656") {
    pylonsCentroid = defenseCentroid;
  }

  set<Unit*> minerals = baseLocation->getStaticMinerals(); 

  for (set<Unit*>::iterator it = minerals.begin(); it != minerals.end(); it++) {
    set<Unit*> workers;
    mineralFields.insert(pair<Unit*, set<Unit*>>(*it, workers));
  }

  buildingPositions.clear();
  buildingWorkers.clear();
}

// sides 0 - 0000, 1 - 0001, 2 - 0010, 3 - 0011, 4 - 0100, 5 - 0101, 6 - 0110, 7 - 0111, 8 - 1000, 9 - 1001, 10 - 1010, 11 - 1011, 12 - 1100, 13 - 1101, 14 - 1110, 15 - 1111
bool MyBase::isFreeSpace(UnitType unitType, TilePosition pos, int freeSpace, int sides) {
  set<TilePosition> positionsBeignBuilt = getPositionsBeignBuilt();

  TilePosition posMax(pos.x()+unitType.tileWidth()-1, pos.y()+unitType.tileHeight()-1);
  
  for (int i = 1; i <= freeSpace; i++) {
    TilePosition first(pos.x()-i, pos.y()-i);
    TilePosition last(posMax.x()+i, posMax.y()+i);
    
    int width = last.x() - first.x();
    int height = last.y() - first.y();

    for (int x = 0; x <= width; x++) {
      for (int y = 0; y <= height; y++) {
        if ((((x == 0 && (sides & 1) != 0) || (x == width && (sides & 4) != 0)) && (y > 0 && y < height)) || 
             ((y == 0 && (sides & 8) != 0) || (y == height && (sides & 2) != 0))) {
          TilePosition current(first.x()+x, first.y()+y);

          if (!Broodwar->isBuildable(current, true) || positionsBeignBuilt.find(current) != positionsBeignBuilt.end()) {
            return false;
          }
        }
      }
    }
  }

  return true;
}

TilePosition MyBase::getSuitableBuildLocation(UnitType unitType, TilePosition pos, int freeSpace, int sides, int maxDist) {
  if (maxDist == -1) {
    maxDist = maxBasePointDist / TILE_SIZE;
  }

  for (int dist = 0; dist <= maxDist; dist++) {
    for (int dx = -dist; dx <= dist; dx++) {
      for (int dy = -dist; dy <= dist; dy++) {

        if (((dx == -dist || dx == dist) && (dy > -dist && dy < dist)) || 
             (dy == -dist || dy == dist)) {

          TilePosition currentTilePos(pos.x() + dx - unitType.tileWidth()/2, pos.y() + dy - unitType.tileHeight()/2);
          Position currentPos(currentTilePos);

          Unit* nearestMineral = ximp->getNearestMineralField(currentPos, false, true);
          Unit* nearestGeyser = ximp->getNearestGeyser(currentPos);
          Unit* nearestAssimilator = ximp->getNearestUnitType(currentPos, UnitTypes::Protoss_Assimilator);
          BWTA::BaseLocation* nearestBaseLocation = BWTA::getNearestBaseLocation(currentPos);

          bool distOk = true;
          if ((nearestMineral != NULL && currentPos.getDistance(nearestMineral->getInitialPosition()) <= 5*TILE_SIZE && nearestBaseLocation != NULL && currentPos.getDistance(nearestBaseLocation->getPosition()) <= 7*TILE_SIZE) 
            || (nearestGeyser != NULL && currentPos.getDistance(nearestGeyser->getInitialPosition()) <= 4*TILE_SIZE && nearestBaseLocation != NULL && currentPos.getDistance(nearestBaseLocation->getPosition()) <= 5*TILE_SIZE)
            || (nearestAssimilator != NULL && currentPos.getDistance(nearestAssimilator->getPosition()) <= 4*TILE_SIZE && nearestBaseLocation != NULL && currentPos.getDistance(nearestBaseLocation->getPosition()) <= 5*TILE_SIZE)) {
            distOk = false;
          }

          if (distOk && !ximp->colony.isPositionBeignBuilt(currentTilePos, unitType) && Broodwar->canBuildHere(NULL, currentTilePos, unitType, true) && isFreeSpace(unitType, currentTilePos, freeSpace, sides) && distOk && isInBaseRegions(currentPos)) {
            return currentTilePos;
          }
        }
      }
    }
  }

  return TilePosition(-1, -1);
}

bool MyBase::build(UnitType unitType, TilePosition tilePos, Unit* worker) {
  if (worker == NULL) {
    return false;
  }

  if (ximp->isEnoughResources(unitType, false)) {
    return worker->build(tilePos, unitType);
  }
  else {
    map<UnitType, TilePosition>::iterator buildingPositionsIt = buildingPositions.find(unitType);

    if (buildingPositionsIt == buildingPositions.end()) {
      buildingPositions.insert(pair<UnitType, TilePosition>(unitType, tilePos));
      
      double distance = worker->getPosition().getDistance(Position(tilePos));
            
      if (ximp->analyzed) {
        distance = BWTA::getGroundDistance(worker->getTilePosition(), tilePos);
      }

      if (ximp->willHaveEnoughResources(unitType, distance, UnitTypes::Protoss_Probe.topSpeed())) {
        buildingWorkers.insert(pair<UnitType, Unit*>(unitType, worker));

        Position movePos = Position(tilePos.x()*TILE_SIZE + unitType.tileWidth()*TILE_SIZE/2, tilePos.y()*TILE_SIZE + unitType.tileHeight()*TILE_SIZE/2);

        worker->move(movePos);
      }
    }

    return false;
  }
}

bool MyBase::buildNewPylon(Position pos) { 
  TilePosition startPosition = TilePosition(baseLocation->getPosition());
  if (!isCapital) {
    startPosition = TilePosition(pylonsCentroid);
  }

  if (pos != Position(-1, -1)) {
    startPosition = TilePosition(pos);
  }

  TilePosition tilePos = getSuitableBuildLocation(UnitTypes::Protoss_Pylon, startPosition, pylonFreeSpace);
  
  if (tilePos != TilePosition(-1, -1)) {
    Unit* worker = getNearestFreeWorker(Position(tilePos)); 

    return build(UnitTypes::Protoss_Pylon, tilePos, worker);;
  }
  else {
    isFull = true;
  }

  return false;
}

bool MyBase::buildAssimilator() {
  map<UnitType, set<Unit*>>::iterator assimilatorsIt = buildings.find(UnitTypes::Protoss_Assimilator);
  set<Unit*> geysers = baseLocation->getGeysers();

  if (assimilatorsIt == buildings.end() || assimilatorsIt->second.size() < geysers.size()) {
    for (set<Unit*>::iterator it = geysers.begin(); it != geysers.end(); it++) {
      bool resourcesOk = (*it)->getResources() > 0;

      if (!resourcesOk) {
        set<Unit*> allUnits = Broodwar->getAllUnits();

        for (set<Unit*>::iterator jt = allUnits.begin(); jt != allUnits.end(); jt++) {
          if ((*jt)->getType() == UnitTypes::Resource_Vespene_Geyser) {
            if ((*jt)->getPosition() == (*it)->getInitialPosition() && (*jt)->getResources() > 0) {
              resourcesOk = true;
              break;
            }
          }
        }
      }

      if (resourcesOk) {
        bool isFree = true;

        if (assimilatorsIt != buildings.end()) {
          for (set<Unit*>::iterator jt = assimilatorsIt->second.begin(); jt != assimilatorsIt->second.end(); jt++) {
            if ((*it)->getDistance(*jt) <= 2*TILE_SIZE) {
              isFree = false;
            }
          }
        }

        if (isFree) {
          bool geyserIsBeignBuilt = false;
          set<Unit*> workers = ximp->getMyUnits(UnitTypes::Protoss_Probe);
          
          for (set<Unit*>::iterator jt = workers.begin(); jt != workers.end(); jt++) {
            if ((*jt)->getBuildType() == UnitTypes::Protoss_Assimilator && (*jt)->getTargetPosition().getDistance((*it)->getPosition()) <= 3*TILE_SIZE) {
              geyserIsBeignBuilt = true;
              break;
            }
          }

          if (!geyserIsBeignBuilt) {
            Unit* worker = getNearestFreeWorker((*it)->getPosition());

            return build(UnitTypes::Protoss_Assimilator, (*it)->getInitialTilePosition(), worker);;
          }
        }
      }
    }
  }

  return false;
}

bool MyBase::buildNearPylon(UnitType unitType) {
  if (unitType == UnitTypes::Protoss_Assimilator) {
    return buildAssimilator();
  }

  map<UnitType, set<Unit*>>::iterator pylons = buildings.find(UnitTypes::Protoss_Pylon);

  int freeSpace = buildingsFreeSpace;
  if (unitType == UnitTypes::Protoss_Photon_Cannon) {
    freeSpace = 0;
  }

  if (pylons != buildings.end()) {
    for (set<Unit*>::iterator it = pylons->second.begin(); it != pylons->second.end(); it++) {
      TilePosition tilePos = getSuitableBuildLocation(unitType, TilePosition((*it)->getPosition()), freeSpace, buildingSides, pylonEnergySpace);

      if (tilePos != TilePosition(-1, -1)) {
        Unit* worker = getNearestFreeWorker(Position(tilePos));

        return build(unitType, tilePos, worker);
      }
    }
  }

  map<UnitType, set<Unit*>>::iterator allPylons = buildings.find(UnitTypes::Protoss_Pylon);

  if (allPylons != buildings.end()) {
    for (set<Unit*>::iterator it = allPylons->second.begin(); it != allPylons->second.end(); it++) {
      if (!(*it)->isCompleted()) {
        waitPylon = *it;
        return false;
      }
    }
  }
  
  if (isBeignBuild(UnitTypes::Protoss_Pylon)) {
    return false;
  }

  return buildNewPylon();
}

bool MyBase::buildNearPosition(UnitType unitType, Position pos) {
  int freeSpace = buildingsFreeSpace;
  if (unitType == UnitTypes::Protoss_Photon_Cannon) {
    freeSpace = 0;
  }

  TilePosition tilePos = getSuitableBuildLocation(unitType, TilePosition(pos), freeSpace, buildingSides, pylonEnergySpace);

  if (tilePos != TilePosition(-1, -1)) {
    Unit* worker = getNearestFreeWorker(Position(tilePos));

    return build(unitType, tilePos, worker);
  }
  
  map<UnitType, set<Unit*>>::iterator allPylons = buildings.find(UnitTypes::Protoss_Pylon);

  if (allPylons != buildings.end()) {
    for (set<Unit*>::iterator it = allPylons->second.begin(); it != allPylons->second.end(); it++) {
      if (!(*it)->isCompleted()) {
        waitPylon = *it;
        return false;
      }
    }
  }
  
  if (isBeignBuild(UnitTypes::Protoss_Pylon)) {
    return false;
  }

  return buildNewPylon(pos);
}

Unit* MyBase::getNearestUnassignedMineralField(Position pos) {
  Unit* closest = NULL;

  for (int i = 0; i <= 3; i++) {
	  for (map<Unit*, set<Unit*>>::iterator it = mineralFields.begin(); it != mineralFields.end(); it++) {
      if (it->first->exists() && it->second.size() <= i) {
        if (closest == NULL || pos.getDistance(it->first->getInitialPosition()) < pos.getDistance(closest->getInitialPosition())) {
          closest = it->first;
        }
      }
	  }

    if (closest != NULL) {
      return closest;
    }
  }

  return closest;
}

Unit* MyBase::getAssignedMineralField(Unit* worker) {
  for (map<Unit*, set<Unit*>>::iterator it = mineralFields.begin(); it != mineralFields.end(); it++) {
    for (set<Unit*>::iterator jt = it->second.begin(); jt != it->second.end(); jt++) {
      if (*jt == worker) {
        return it->first;
      }
    }
  }

  return NULL;
}

void MyBase::assignToMineralField(Unit* mineral, Unit* worker) {
  map<Unit*, set<Unit*>>::iterator mineralFieldsIt = mineralFields.find(mineral);

  if (mineralFieldsIt != mineralFields.end()) {
    if (mineralFieldsIt->first->exists()) {
      mineralFieldsIt->second.insert(worker);
    }
  }
}

void MyBase::unassignFromMineralField(Unit* worker) {
  for (map<Unit*, set<Unit*>>::iterator it = mineralFields.begin(); it != mineralFields.end(); it++) {
    bool deleted = false;
    
    for (set<Unit*>::iterator jt = it->second.begin(); jt != it->second.end(); jt++) {
      if (*jt == worker) {
        it->second.erase(jt);
        deleted = true;
        break;
      }
    }

    if (deleted) {
      break;
    }
  }
}

bool MyBase::isMyBase(bool mustHaveNexus) {
  if (!mustHaveNexus) {
    if (!buildings.empty()) {
      for (map<UnitType, set<Unit*>>::iterator unitTypeIterator = buildings.begin(); unitTypeIterator != buildings.end(); unitTypeIterator++) {
        if (unitTypeIterator->second.size() > 0) {
          return true;
        }
      }
    }
  }

  return hqs.size() > 0;
}

void MyBase::addBuilding(Unit* building, bool completed) {
  int countBaseLocationsInCurrentRegion = baseLocation->getRegion()->getBaseLocations().size();

  if ((countBaseLocationsInCurrentRegion == 1 && BWTA::getRegion(building->getPosition()) == baseLocation->getRegion()) || 
      (countBaseLocationsInCurrentRegion >  1 && ximp->colony.getNearestBaseLocationAir(building->getPosition()) == baseLocation)) {

    if (!building->getType().isResourceDepot()) {
      map<UnitType, set<Unit*>> *currentBuildings = &buildings;

      if (completed) {
        currentBuildings = &buildingsCompleted;
        if (building == waitPylon) {
          waitPylon = NULL;
        }
      }

      map<UnitType, set<Unit*>>::iterator buildingsIterator = currentBuildings->find(building->getType());

      if (buildingsIterator != currentBuildings->end()) {
        buildingsIterator->second.insert(building);
      }
      else {
        set<Unit*> newBuildingType;
        newBuildingType.insert(building);
        currentBuildings->insert(pair<UnitType, set<Unit*>>(building->getType(), newBuildingType));
      }
    }
    else if (building->getType().isResourceDepot()) {
      hqs.insert(building);

      if (ximp->colony.getCapitalMyBase() == NULL) {
        isCapital = true;
      }
    }
  }
}

void MyBase::deleteBuilding(Unit* building) {
  if (!building->getType().isResourceDepot()) {
    map<UnitType, set<Unit*>>::iterator buildingsIterator = buildings.find(building->getType());

    if (building->getType() == UnitTypes::Resource_Vespene_Geyser) {
      buildingsIterator = buildings.find(UnitTypes::Protoss_Assimilator);
    }

    if (buildingsIterator != buildings.end()) {
      for (set<Unit*>::iterator it = buildingsIterator->second.begin(); it != buildingsIterator->second.end(); it++) {
        if (*it == building) {
          buildingsIterator->second.erase(it);
          isUnderAttack = true;
          lastAttackedFrame = Broodwar->getFrameCount();
          break;
        }
      }
    }

    buildingsIterator = buildingsCompleted.find(building->getType());

    if (building->getType() == UnitTypes::Resource_Vespene_Geyser) {
      buildingsIterator = buildingsCompleted.find(UnitTypes::Protoss_Assimilator);
    }

    if (buildingsIterator != buildingsCompleted.end()) {
      for (set<Unit*>::iterator it = buildingsIterator->second.begin(); it != buildingsIterator->second.end(); it++) {
        if (*it == building) {
          buildingsIterator->second.erase(it);
          isUnderAttack = true;
          lastAttackedFrame = Broodwar->getFrameCount();
          break;
        }
      }
    }
  } 
  else if (building->getType().isResourceDepot()) {
    for (set<Unit*>::iterator it = hqs.begin(); it != hqs.end(); it++) {
      if (*it == building) {
        hqs.erase(it);
        isUnderAttack = true;
        lastAttackedFrame = Broodwar->getFrameCount();
        break;
      }
    }

    if (hqs.empty()) {
      isCapital = false;
    }
  }

  isFull = false;
}

bool MyBase::addWorker(Unit* worker) {
  map<UnitType, set<Unit*>>::iterator assimilators = buildingsCompleted.find(UnitTypes::Protoss_Assimilator);

  if (assimilators != buildingsCompleted.end() && getMaxGasWorkers() > gasWorkers.size()) {
    gasWorkers.insert(worker);
    return true;
  }
  else if (getMaxMineralWorkers() > mineralWorkers.size()) {
    mineralWorkers.insert(worker);

    Unit* nearest = getAssignedMineralField(worker);

    if (nearest == NULL) {
      nearest = getNearestUnassignedMineralField(baseLocation->getPosition());

      if (nearest != NULL) {
        assignToMineralField(nearest, worker);
      }
    }
    return true;
  }

  return false;
}

bool MyBase::deleteWorker(Unit* worker) {
  for (set<Unit*>::iterator mineralWorkersIterator = mineralWorkers.begin(); mineralWorkersIterator != mineralWorkers.end(); mineralWorkersIterator++) {
    if (*mineralWorkersIterator == worker) {
      unassignFromMineralField(worker);
      mineralWorkers.erase(mineralWorkersIterator);
      return true;
    }
  }

  for (set<Unit*>::iterator gasWorkersIterator = gasWorkers.begin(); gasWorkersIterator != gasWorkers.end(); gasWorkersIterator++) {
    if (*gasWorkersIterator == worker) {
      gasWorkers.erase(gasWorkersIterator);
      return true;
    }
  }

  return false;
}

set<Unit*> MyBase::getBaseWorkers() {
  set<Unit*> baseWorkers;

  for (set<Unit*>::iterator mineralWorkersIterator = mineralWorkers.begin(); mineralWorkersIterator != mineralWorkers.end(); mineralWorkersIterator++) {
    baseWorkers.insert(*mineralWorkersIterator);
  }

  for (set<Unit*>::iterator gasWorkersIterator = gasWorkers.begin(); gasWorkersIterator != gasWorkers.end(); gasWorkersIterator++) {
    baseWorkers.insert(*gasWorkersIterator);
  }

  return baseWorkers;
}

Unit* MyBase::getWorkerNotAssignedToMinerals() {
  for (set<Unit*>::iterator worker = mineralWorkers.begin(); worker != mineralWorkers.end(); worker++) {
    bool assigned = false;

    for (map<Unit*, set<Unit*>>::iterator it = mineralFields.begin(); it != mineralFields.end(); it++) {
      if (it->first->exists() && it->first->getResources() > 0 && it->second.find(*worker) != it->second.end()) {
        assigned = true;
        break;
      }
    }

    if (!assigned) {
      return *worker;
    }
  }

  return NULL;
}

Unit* MyBase::getNearestFreeWorker(Position pos, bool onlyThisBase) {
  Unit* closest = NULL;

  if (!mineralWorkers.empty()) {
    for (set<Unit*>::iterator it = mineralWorkers.begin(); it != mineralWorkers.end(); it++) {
      if (closest == NULL || pos.getDistance((*it)->getPosition()) < pos.getDistance(closest->getPosition())) {
        if ((*it) != ximp->observation.myBaseScoutWorker && !ximp->colony.isInNewNexusWorkers(*it) /*&& !(*it)->isCarryingMinerals() && (*it)->getOrder() != Orders::PlaceBuilding*/ && (*it)->isCompleted()) {
          closest = *it;
        }
      }
    }
  }
  else {
    for (set<Unit*>::iterator it = gasWorkers.begin(); it != gasWorkers.end(); it++) {
      if (closest == NULL || pos.getDistance((*it)->getPosition()) < pos.getDistance(closest->getPosition())) {
        if ((*it) != ximp->observation.myBaseScoutWorker && !ximp->colony.isInNewNexusWorkers(*it) && !(*it)->isCarryingGas() /*&& (*it)->getOrder() != Orders::PlaceBuilding*/ && (*it)->isCompleted() && *it != ximp->observation.myBaseScoutWorker) {
          closest = *it;
        }
      }
    }
  }

  if (!onlyThisBase) {
    if (Broodwar->getFrameCount() < 3000 && ximp->observation.earlyScoutWorker != NULL && (closest == NULL || ximp->observation.earlyScoutWorker->getPosition().getDistance(pos) < closest->getPosition().getDistance(pos)) && (ximp->colony.getCapitalMyBase()->isInBaseRegions(ximp->observation.earlyScoutWorker->getPosition()) || isInBaseRegions(ximp->observation.earlyScoutWorker->getPosition()) || ximp->observation.earlyScoutWorker->getPosition().getDistance(baseLocation->getPosition()) < 40 * TILE_SIZE)) {
      return ximp->observation.earlyScoutWorker;
    }
    
    if (closest != NULL) {
      return closest;
    }

    map<UnitType, set<Unit*>>::iterator workers = ximp->myUnits.find(UnitTypes::Protoss_Probe);

    if (workers != ximp->myUnits.end()) {
      for (int i = 0; i < 2; i++) {
        for (set<Unit*>::iterator it = workers->second.begin(); it != workers->second.end(); it++) {
          if (closest == NULL || pos.getDistance((*it)->getPosition()) < pos.getDistance(closest->getPosition())) {
            if ((*it) != ximp->observation.myBaseScoutWorker && (*it) != ximp->observation.earlyScoutWorker && !ximp->colony.isInNewNexusWorkers(*it) && !(*it)->isCarryingGas() && (i == 1 || !(*it)->isGatheringGas()) /*&& (*it)->getOrder() != Orders::PlaceBuilding*/ && (*it)->isCompleted()) {
              closest = *it;
            }
          }
        }

        if (closest != NULL) {
          break;
        }
      }
    }
  }

  return closest;
}

void MyBase::taskIdleWorkers() {
  for (set<Unit*>::iterator it = mineralWorkers.begin(); it != mineralWorkers.end(); it++) {
    if (!(*it)->exists()) {
      deleteWorker(*it);
      break;
    }
    
    if ((*it)->isCompleted() && (*it)->getOrder() == Orders::PlayerGuard) {
      if (isInBuildingWorkers(*it) || ximp->colony.isInNewNexusWorkers(*it) || ximp->observation.myBaseScoutWorker == *it) {
        continue;
      }

      Unit* nearest = getAssignedMineralField(*it);

      if (nearest == NULL) {
        nearest = getNearestUnassignedMineralField(baseLocation->getPosition());

        if (nearest != NULL) {
          assignToMineralField(nearest, *it);
        }
        else {
          //Unit* nearest = ximp->getNearestMineralField(baseLocation->getPosition(), true);
        }
      }

      if ((*it)->isCarryingMinerals()) {
        Unit* nearestNexus = ximp->getNearestUnitType(baseLocation->getPosition(), UnitTypes::Protoss_Nexus);

        if (nearestNexus != NULL) {
          nearest = nearestNexus;
        }
      }

      (*it)->rightClick(nearest);
    }
  }

  for (set<Unit*>::iterator it = gasWorkers.begin(); it != gasWorkers.end(); it++) {
    if (!(*it)->exists()) {
      deleteWorker(*it);
      break;
    }

    if ((*it)->isCompleted() && (*it)->getOrder() == Orders::PlayerGuard) {
      if (isInBuildingWorkers(*it) || ximp->colony.isInNewNexusWorkers(*it) || ximp->observation.myBaseScoutWorker == *it) {
        continue;
      }

      map<UnitType, set<Unit*>>::iterator assimilators = buildingsCompleted.find(UnitTypes::Protoss_Assimilator);

      if (assimilators != buildingsCompleted.end()) {
        set<Unit*> tasked;  

        for (set<Unit*>::iterator assimilatorsIterator = assimilators->second.begin(); assimilatorsIterator != assimilators->second.end(); assimilatorsIterator++) {
          int count = 0;

          for (set<Unit*>::iterator jt = gasWorkers.begin(); jt != gasWorkers.end(); jt++) {
            if (tasked.find(*jt) == tasked.end() && (*jt)->getOrder() != Orders::PlaceBuilding) {
              (*jt)->rightClick(*assimilatorsIterator);
              tasked.insert(*jt);
              if (++count == 3) {
                break; 
              }
            }
          }
        }

      }

      break;
    }
  }
}

void MyBase::mineAssignedMinerals() {
  for (map<Unit*, set<Unit*>>::iterator it = mineralFields.begin(); it != mineralFields.end(); it++) {
    for (set<Unit*>::iterator jt = it->second.begin(); jt != it->second.end(); jt++) {
      if (isInBuildingWorkers(*jt) || ximp->colony.isInNewNexusWorkers(*jt) || ximp->observation.myBaseScoutWorker == *jt) {
        continue;
      }

      if ((*jt)->getOrder() != Orders::AttackUnit && (*jt)->getOrder() == Orders::MoveToMinerals && (*jt)->getOrderTarget() != it->first && (*jt)->getOrderTarget() != NULL && (*jt)->getOrderTarget()->getResources() > 0) {
        if (it->first->exists() && it->first->getResources() > 0) {
          (*jt)->rightClick(it->first);
        }
        else {
          unassignFromMineralField(*jt);
          break;
        }
      }
    }
  }  
}

void MyBase::checkFull() {
  if (isFull && getSuitableBuildLocation(UnitTypes::Protoss_Pylon, TilePosition(baseLocation->getPosition()), pylonFreeSpace) != TilePosition(-1,-1)) {
    isFull = false;
  }
}

void MyBase::checkMinerals() {
  if (ximp->analyzed) {
    set<Unit*> unitsInRadius = Broodwar->getUnitsInRadius(baseLocation->getPosition(), TILE_SIZE*checkMineralsDist);

    bool myUnit = false;
    for (set<Unit*>::iterator it = unitsInRadius.begin(); it != unitsInRadius.end(); it++) {
      if ((*it)->getPlayer() == Broodwar->self() && (*it)->isCompleted()) {
        myUnit = true;
      }
    }

    if (myUnit) {
      if (baseLocation->getMinerals().empty()) {
        isMinedOut = true;
      }
      else {
        isMinedOut = false;
        lastSeenMineralCount = baseLocation->minerals();
      }
    }
  }
}

void MyBase::checkUnderAttack() {
  if (Broodwar->getFrameCount() - lastAttackedFrame >= 200) {
    if (!isMyBase()) {
      isUnderAttack = false;
    }

    if (!buildings.empty()) {
      for (map<UnitType, set<Unit*>>::iterator it = buildings.begin(); it != buildings.end(); it++) {
        for (set<Unit*>::iterator jt = it->second.begin(); jt != it->second.end(); jt++) {
          
          if ((*jt)->isUnderAttack()) {
            isUnderAttack = true;
            lastAttackedFrame = Broodwar->getFrameCount();
            return;
          }
        }
      }
    }

    if (!hqs.empty()) {
      for (set<Unit*>::const_iterator it = hqs.begin(); it != hqs.end(); it++) {
        if ((*it)->isUnderAttack()) {
          isUnderAttack = true;
          lastAttackedFrame = Broodwar->getFrameCount();
          return;
        }
      }
    }

    isUnderAttack = false;
  }
}

void MyBase::checkGasSteal() {
  set<Unit*> geysers = baseLocation->getGeysers();
  set<Unit*> allUnits = Broodwar->getAllUnits();

  isGasSteal = false;

  for (set<Unit*>::iterator it = geysers.begin(); it != geysers.end(); it++) {
    for (set<Unit*>::iterator jt = allUnits.begin(); jt != allUnits.end(); jt++) {
      if ((*jt)->getType().isRefinery() && Broodwar->self()->isEnemy((*jt)->getPlayer())) {
        if ((*jt)->getPosition() == (*it)->getInitialPosition()) {
          isGasSteal = true;
          break;
        }
      }
    }
  }
}

void MyBase::checkUnpoweredBuildings() {
  for (map<UnitType, set<Unit*>>::iterator it = buildingsCompleted.begin(); it != buildingsCompleted.end(); it++) {
    for (set<Unit*>::iterator jt = it->second.begin(); jt != it->second.end(); jt++) {
      if ((*jt)->isUnpowered()) {
        hasUnpoweredBuildings = true;
        return;
      }
    }
  }

  hasUnpoweredBuildings = false;
}

int MyBase::getMaxMineralWorkers() {
  if (!hasCompletedNexus()) {
    return 0;
  }

  return (int)(baseLocation->getMinerals().size() * ximp->colony.countWorkersPerMineralField);
}

int MyBase::getMaxGasWorkers() {
  if (!hasCompletedNexus()) {
    return 0;
  }

  int maxCount = 0;

  map<UnitType, set<Unit*>>::iterator assimilators = buildingsCompleted.find(UnitTypes::Protoss_Assimilator);

  if (assimilators != buildingsCompleted.end()) {
    for (set<Unit*>::iterator it = assimilators->second.begin(); it != assimilators->second.end(); it++) {
      maxCount += (*it)->getResources() > 0 ? ximp->colony.countWorkersPerGas : ximp->colony.countWorkersPerGasDepleted;
    }
  }

  return maxCount;
}

bool MyBase::isInBuildingWorkers(Unit* worker) {
  for (map<UnitType, Unit*>::iterator it = buildingWorkers.begin(); it != buildingWorkers.end(); it++) {
    if (it->second == worker) {
      return true;
    }
  }

  for (map<BWTA::BaseLocation*, MyBase>::iterator it = ximp->colony.myBases.begin(); it != ximp->colony.myBases.end(); it++) {
    for (map<UnitType, Unit*>::iterator jt = it->second.buildingWorkers.begin(); jt != it->second.buildingWorkers.end(); jt++) {
      if (jt->second == worker) {
        return true;
      }
    }  
  }

  return false;
}

void MyBase::resolveGasSteal() {
  set<Unit*> geysers = baseLocation->getGeysers();
  set<Unit*> allUnits = Broodwar->getAllUnits();

  for (set<Unit*>::iterator it = geysers.begin(); it != geysers.end(); it++) {
    for (set<Unit*>::iterator jt = allUnits.begin(); jt != allUnits.end(); jt++) {
      if ((*jt)->getType().isRefinery() && Broodwar->self()->isEnemy((*jt)->getPlayer())) {
        if ((*jt)->getPosition() == (*it)->getInitialPosition()) {
          set<Unit*> workers = getBaseWorkers();
          
          int defending = 0, maxDefending = 6;
          for (set<Unit*>::iterator worker = workers.begin(); worker != workers.end(); worker++) {
            (*worker)->attack(*jt);
            defending++;
            if (defending >= maxDefending) {
              break;
            }
          }
          
          break;
        }
      }
    }
  }
}

void MyBase::resolveUnpoweredBuildings() {
  if (!isUnderAttack && getEnemyUnitsInBaseRegions().empty()) {
    for (map<UnitType, set<Unit*>>::iterator it = buildingsCompleted.begin(); it != buildingsCompleted.end(); it++) {
      for (set<Unit*>::iterator jt = it->second.begin(); jt != it->second.end(); jt++) {
        if ((*jt)->isUnpowered()) {
          map<UnitType, TilePosition>::iterator buildingPositionsIt = buildingPositions.find(UnitTypes::Protoss_Pylon);

          map<UnitType, set<Unit*>>::iterator buildingsIt = buildings.find(UnitTypes::Protoss_Pylon);

          bool isPylonNearPositionBeignBuilt = false;
          for (set<Unit*>::iterator pylonsIt = buildingsIt->second.begin(); pylonsIt != buildingsIt->second.end(); pylonsIt++) {
            if (!(*pylonsIt)->isCompleted() && (*pylonsIt)->getPosition().getDistance((*jt)->getPosition()) < pylonEnergySpace * TILE_SIZE) {
              isPylonNearPositionBeignBuilt = true;
              break;
            }
          }
          
          if (buildingPositionsIt == buildingPositions.end() && !isBeignBuild(UnitTypes::Protoss_Pylon) && !isPylonNearPositionBeignBuilt) {
            TilePosition tilePos = getSuitableBuildLocation(UnitTypes::Protoss_Pylon, TilePosition((*jt)->getPosition()), 0, 0, pylonEnergySpace);

            if (tilePos != TilePosition(-1, -1) && Position(tilePos).getDistance((*jt)->getPosition()) < pylonEnergySpace * TILE_SIZE) {
              Unit* worker = getNearestFreeWorker(Position(tilePos));
              build(UnitTypes::Protoss_Pylon, tilePos, worker);
            }
          }
        }
      }
    }
  }
}

double MyBase::getWorkersRatio() {
  if (mineralWorkers.size() + gasWorkers.size() > 0 && getMaxMineralWorkers() + getMaxGasWorkers() > 0) {
    return (double)(mineralWorkers.size() + gasWorkers.size()) / (double)(getMaxMineralWorkers() + getMaxGasWorkers());
  }
  else {
    if (mineralWorkers.size() + gasWorkers.size() == 1) {
      return 1.1;
    }
    return mineralWorkers.size() + gasWorkers.size();
  }
}

Unit* MyBase::getFirstBuildingUnderAttack() {
  if (!isMyBase()) {
    return NULL;
  }

  if (!buildings.empty()) {
    for (map<UnitType, set<Unit*>>::iterator it = buildings.begin(); it != buildings.end(); it++) {
      for (set<Unit*>::iterator jt = it->second.begin(); jt != it->second.end(); jt++) {
        
        if ((*jt)->isUnderAttack()) {
          return *jt;
        }
      }
    }
  }

  if (!hqs.empty()) {
    for (set<Unit*>::const_iterator it = hqs.begin(); it != hqs.end(); it++) {
      if ((*it)->isUnderAttack()) {
        return *it;
      }
    }
  }

  return NULL;
}

bool MyBase::isAnyWorkerUnderAttack() {
  if (!isMyBase()) {
    return false;
  }

  for (set<Unit*>::iterator it = mineralWorkers.begin(); it != mineralWorkers.end(); it++) {
    if ((*it)->isUnderAttack()) {
      return true;
    }
  }

  for (set<Unit*>::iterator it = gasWorkers.begin(); it != gasWorkers.end(); it++) {
    if ((*it)->isUnderAttack()) {
      return true;
    }
  }

  return false;
}

bool MyBase::hasCompletedNexus() {
  for (set<Unit*>::iterator it = hqs.begin(); it != hqs.end(); it++) {
    if ((*it)->isCompleted()) {
      return true;
    }
  }

  return false;
}

int MyBase::getNeededAssimilators() {
  int count = 0;

  map<UnitType, set<Unit*>>::iterator assimilatorsIt = buildings.find(UnitTypes::Protoss_Assimilator);
  set<Unit*> geysers = baseLocation->getGeysers();

  if (assimilatorsIt == buildings.end() || assimilatorsIt->second.size() < geysers.size()) {
    for (set<Unit*>::iterator it = geysers.begin(); it != geysers.end(); it++) {
      bool isFree = true;

      if (assimilatorsIt != buildings.end()) {
        for (set<Unit*>::iterator jt = assimilatorsIt->second.begin(); jt != assimilatorsIt->second.end(); jt++) {
          if ((*it)->getDistance(*jt) <= 2*TILE_SIZE) {
            isFree = false;
          }
        }
      }

      bool resourcesOk = (*it)->getResources() > 0;

      if (!resourcesOk) {
        set<Unit*> allUnits = Broodwar->getAllUnits();

        for (set<Unit*>::iterator jt = allUnits.begin(); jt != allUnits.end(); jt++) {
          if ((*jt)->getType() == UnitTypes::Resource_Vespene_Geyser) {
            if ((*jt)->getPosition() == (*it)->getInitialPosition() && (*jt)->getResources() > 0) {
              resourcesOk = true;
              break;
            }
          }
        }
      }

      if (isFree && resourcesOk) {
        count++;
      }
    }
  }

  return count;
}


bool MyBase::isInBaseRegions(Position pos) {
  for (set<BWTA::Region*>::iterator it = regions.begin(); it != regions.end(); it++) {
    if ((*it)->getPolygon().isInside(pos)) {
      return true;
    }
  }

  return false;
}

void MyBase::moveOneMineralWorkerToGasWorkers() {
  if (mineralWorkers.size() > 0) {
    Unit *worker = *mineralWorkers.begin();
    unassignFromMineralField(worker);
    mineralWorkers.erase(mineralWorkers.begin());
    gasWorkers.insert(worker);
    worker->stop();
  }
}

vector<UnitType> MyBase::getBuildingsBeignBuilt() {
  vector<UnitType> result;

  //////////////
  for (map<UnitType, Unit*>::iterator it = buildingWorkers.begin(); it != buildingWorkers.end(); it++) {
    result.push_back(it->first);
  }
  //////////////

  set<Unit*> workers = ximp->getMyUnits(UnitTypes::Protoss_Probe);

  for (set<Unit*>::iterator wt = workers.begin(); wt != workers.end(); wt++) {
    if ((*wt)->getBuildType() != UnitTypes::None && isInBaseRegions((*wt)->getTargetPosition())) {
      result.push_back((*wt)->getBuildType());
    }
  }

  return result;
}

set<TilePosition> MyBase::getPositionsBeignBuilt() {
  set<TilePosition> result;

  UnitType buildingNexus = UnitTypes::Protoss_Nexus;
      
  TilePosition first(baseLocation->getPosition().x()/TILE_SIZE - buildingNexus.tileWidth()/2, baseLocation->getPosition().y()/TILE_SIZE - buildingNexus.tileHeight()/2);
  TilePosition last(first.x() + buildingNexus.tileWidth() - 1, first.y() + buildingNexus.tileHeight() - 1);

  for (int x = first.x(); x <= last.x(); x++) {
    for (int y = first.y(); y <= last.y(); y++) {
      result.insert(TilePosition(x, y));
    }
  }

  //////////////
  for (map<UnitType, TilePosition>::iterator it = buildingPositions.begin(); it != buildingPositions.end(); it++) {
    if (isInBaseRegions(Position(it->second))) {
      UnitType buildingType = it->first;
      Position pos = Position(it->second);
        
      TilePosition first(pos.x()/TILE_SIZE, pos.y()/TILE_SIZE);
      TilePosition last(first.x() + buildingType.tileWidth() - 1, first.y() + buildingType.tileHeight() - 1);

      for (int x = first.x(); x <= last.x(); x++) {
        for (int y = first.y(); y <= last.y(); y++) {
          result.insert(TilePosition(x, y));
        }
      }
    }
  }
  //////////////

  set<Unit*> workers = ximp->getMyUnits(UnitTypes::Protoss_Probe);

  for (set<Unit*>::iterator wt = workers.begin(); wt != workers.end(); wt++) {
    if ((*wt)->getBuildType() != UnitTypes::None && isInBaseRegions((*wt)->getTargetPosition())) {
      UnitType buildingType = (*wt)->getBuildType();
      Position pos = (*wt)->getTargetPosition();
      
      TilePosition first(pos.x()/TILE_SIZE - buildingType.tileWidth()/2, pos.y()/TILE_SIZE - buildingType.tileHeight()/2);
      TilePosition last(first.x() + buildingType.tileWidth() - 1, first.y() + buildingType.tileHeight() - 1);

      for (int x = first.x(); x <= last.x(); x++) {
        for (int y = first.y(); y <= last.y(); y++) {
          result.insert(TilePosition(x, y));
        }
      }
    }
  }

  return result;
}

vector<UnitType> MyBase::getAllBuildingTypes() {
  vector<UnitType> result;

  for (set<Unit*>::iterator it = hqs.begin(); it != hqs.end(); it++) {
    result.push_back((*it)->getType());
  }

  for (map<UnitType, set<Unit*>>::iterator it = buildings.begin(); it != buildings.end(); it++) {
    int count = it->second.size();
    for (int i = 0; i < count; i++) {
      result.push_back(it->first);
    }
  }

  vector<UnitType> beignBuilt = getBuildingsBeignBuilt();
  for (vector<UnitType>::iterator it = beignBuilt.begin(); it != beignBuilt.end(); it++) {
    vector<UnitType>::iterator first = std::find(result.begin(), result.end(), *it);
    result.insert(first, *it);
  }

  return result;
}

set<Unit*> MyBase::getAllBuildings() {
  set<Unit*> result;

  result.insert(hqs.begin(), hqs.end());

  for (map<UnitType, set<Unit*>>::iterator it = buildings.begin(); it != buildings.end(); it++) {
    result.insert(it->second.begin(), it->second.end());
  }

  return result;
}

bool MyBase::isPositionBeignBuilt(TilePosition pos, UnitType buildingType) {
  set<TilePosition> positionsBeignBuilt = getPositionsBeignBuilt();

  TilePosition last(pos.x() + buildingType.tileWidth() - 1, pos.y() + buildingType.tileHeight() - 1);

  for (int x = pos.x(); x <= last.x(); x++) {
    for (int y = pos.y(); y <= last.y(); y++) {
      if (positionsBeignBuilt.find(TilePosition(x, y)) != positionsBeignBuilt.end()) {
        return true;
      }
    }
  }

  return false;
}

bool MyBase::isBeignBuild(UnitType unitType) {
  vector<UnitType> beignBuild = getBuildingsBeignBuilt();

  for (int i = 0; i < beignBuild.size(); i++) {
    if (beignBuild[i] == unitType) {
      return true;
    }
  }

  return false;
}

bool MyBase::isAnyWorkerPlacingBuilding() {
  set<Unit*> workers = getBaseWorkers();

  for (set<Unit*>::iterator wt = workers.begin(); wt != workers.end(); wt++) {
    if ((*wt)->getOrder() == Orders::PlaceBuilding) {
      return true;
    }
  }

  return buildingWorkers.size() > 0;
}

bool MyBase::isTrainingQueueEmpty(UnitType unitType) {
  UnitType whatBuilds = unitType.whatBuilds().first;
  map<UnitType, set<Unit*>>::iterator buildingsCompletedIt = buildingsCompleted.find(whatBuilds);

  set<Unit*> productionBuildings;

  if (buildingsCompletedIt != buildingsCompleted.end()) {
    productionBuildings = buildingsCompletedIt->second;
  }

  for (set<Unit*>::iterator it = productionBuildings.begin(); it != productionBuildings.end(); it++) {
    if ((*it)->getTrainingQueue().empty()) {
      return true;
    }
  }

  return false;
}

bool MyBase::isBaseSafe() {
  set<Unit*> units = Broodwar->getAllUnits();
  set<Unit*> enemyUnits = ximp->getEnemyUnits(units);

  for (set<Unit*>::iterator it = enemyUnits.begin(); it != enemyUnits.end(); ) { 
    if (isInBaseRegions((*it)->getPosition())) {
      if ((*it)->getType().isWorker() || (*it)->getType() == UnitTypes::Zerg_Overlord || (*it)->getType() == UnitTypes::Protoss_Observer) {
        it = enemyUnits.erase(it);
      }
      else {
        it++;
      }
    }
    else {
      it = enemyUnits.erase(it);
    }
  }

  return enemyUnits.empty();
}

set<Unit*> MyBase::getEnemyUnitsInBaseRegions(bool detected) {
  set<Unit*> units = Broodwar->getAllUnits();
  set<Unit*> enemyUnits = ximp->getEnemyUnits(units);

  for (set<Unit*>::iterator it = enemyUnits.begin(); it != enemyUnits.end(); ) { 
    if (isInBaseRegions((*it)->getPosition()) && (!detected || (detected && (*it)->isDetected()))) {
      it++;
    }
    else {
      it = enemyUnits.erase(it);
    }
  }

  return enemyUnits;
}

bool MyBase::buildBuilding(UnitType unitType, Position pos) {
  if (isFull || 
      waitPylon != NULL || 
      isAnyWorkerPlacingBuilding() || 
      isBeignBuild(unitType)) {
    return false;
  }

  if (pos != Position(-1, -1)) {
    return buildNearPosition(unitType, pos);
  }
  else {
    return buildNearPylon(unitType);
  }
}

bool MyBase::trainUnit(UnitType unitType) {
  if (!ximp->buildRequirementsMet(unitType) || !ximp->isEnoughResources(unitType, false)) {
    return false;
  }

  UnitType whatBuilds = unitType.whatBuilds().first;
  map<UnitType, set<Unit*>>::iterator buildingsCompletedIt = buildingsCompleted.find(whatBuilds);

  set<Unit*> productionBuildings;

  if (buildingsCompletedIt != buildingsCompleted.end()) {
    productionBuildings = buildingsCompletedIt->second;
  }

  for (set<Unit*>::iterator it = productionBuildings.begin(); it != productionBuildings.end(); it++) {
    if ((*it)->getTrainingQueue().empty()) {
      (*it)->train(unitType);
      return true;
    }
  }

  return false;
}

bool MyBase::cancelCarriers() {
  map<UnitType, set<Unit*>>::iterator buildingsCompletedIt = buildingsCompleted.find(UnitTypes::Protoss_Stargate);

  if (buildingsCompletedIt != buildingsCompleted.end()) {
    for (set<Unit*>::iterator it = buildingsCompletedIt->second.begin(); it != buildingsCompletedIt->second.end(); it++) {
      if (!(*it)->getTrainingQueue().empty() && (*(*it)->getTrainingQueue().begin()) == UnitTypes::Protoss_Carrier && (*it)->getRemainingTrainTime() > UnitTypes::Protoss_Carrier.buildTime() / 2 && (!isTrainingQueueEmpty(UnitTypes::Protoss_Corsair) || !ximp->isEnoughResources(UnitTypes::Protoss_Corsair, false))) {
        (*it)->cancelTrain();
        return true;
      }
    }
  }

  return false;
}

void MyBase::buildBuildings() {
  for (map<UnitType, TilePosition>::iterator buildingPositionsIt = buildingPositions.begin(); buildingPositionsIt != buildingPositions.end(); ) {
    map<UnitType, Unit*>::iterator buildingWorkersIt = buildingWorkers.find(buildingPositionsIt->first);
   
    Unit* worker = NULL;
    if (buildingWorkersIt != buildingWorkers.end()) {
      worker = buildingWorkersIt->second;
    }
    else {
      worker = getNearestFreeWorker(Position(buildingPositionsIt->second));
    }
    
    if (worker != NULL) {
      if (ximp->isEnoughResources(buildingPositionsIt->first, true)) {
        if (ximp->buildRequirementsMet(buildingPositionsIt->first)) {
          if (buildingWorkersIt != buildingWorkers.end()) {
            buildingWorkers.erase(buildingWorkersIt);
          }

          if (!worker->build(buildingPositionsIt->second, buildingPositionsIt->first)) {
            buildBuilding(buildingPositionsIt->first);
          }

          buildingPositionsIt = buildingPositions.erase(buildingPositionsIt);
        }
        else {
          buildingPositionsIt++;
        }
      }
      else {
        double distance = worker->getPosition().getDistance(Position(buildingPositionsIt->second));
            
        if (ximp->analyzed) {
          distance = BWTA::getGroundDistance(worker->getTilePosition(), buildingPositionsIt->second);
        }

        if (ximp->willHaveEnoughResources(buildingPositionsIt->first, distance, UnitTypes::Protoss_Probe.topSpeed())) {
          Position movePos = Position(buildingPositionsIt->second.x()*TILE_SIZE + buildingPositionsIt->first.tileWidth()*TILE_SIZE/2, buildingPositionsIt->second.y()*TILE_SIZE + buildingPositionsIt->first.tileHeight()*TILE_SIZE/2);

          if (!worker->isMoving() || worker->getOrderTargetPosition() != movePos) {
            worker->move(movePos);
              
            if (buildingWorkersIt == buildingWorkers.end()) {
              buildingWorkers.insert(pair<UnitType, Unit*>(buildingPositionsIt->first, worker));
            }
          }
        }

        buildingPositionsIt++;
      }
    }
    else {
      buildingPositionsIt++;
    }
  }
}