#include "StrategyManager.h"
#include "CombatCommander.h"
#include "ProductionManager.h"
#include "StrategyBossZerg.h"
#include "UnitUtil.h"

using namespace CDBot;

StrategyManager::StrategyManager() 
	: _selfRace(BWAPI::Broodwar->self()->getRace())
	, _enemyRace(BWAPI::Broodwar->enemy()->getRace())
    , _emptyBuildOrder(BWAPI::Broodwar->self()->getRace())
	, _openingGroup("")
{
}

StrategyManager & StrategyManager::Instance() 
{
	static StrategyManager instance;
	return instance;
}

bool StrategyManager::diagonallyBorn()
{
	auto enemyName = BWAPI::Broodwar->enemy()->getName();
	if (enemyName == "PurpleWave")
	//BWAPI::Broodwar->drawTextScreen(150, 60, "Enemy is PurpleWave");
	{
		BWTA::BaseLocation * enemyBase = InformationManager::Instance().getEnemyMainBaseLocation();

		BWTA::BaseLocation * ourBase = InformationManager::Instance().getMyMainBaseLocation();
		BWAPI::Position myMainPos = ourBase->getPosition();

		if (enemyBase)
		{
			BWAPI::Position enemyMainPos = enemyBase->getPosition();
			//BWAPI::Broodwar->drawTextScreen(150, 30, "%f", myMainPos.getDistance(enemyMainPos));
			if (myMainPos.getDistance(enemyMainPos) > 4500)
			{
				BWAPI::Broodwar->drawTextScreen(150, 10, "Distance is %f\nToo far. Change 4d rush strategey.", myMainPos.getDistance(enemyMainPos));
				return true;
			}
			else
			{
				return false;
			}
		}
		return false;
	}
	return false;
}

const BuildOrder & StrategyManager::getOpeningBookBuildOrder() const
{
    auto buildOrderIt = _strategies.find(Config::Strategy::StrategyName);

    // look for the build order in the build order map
	if (buildOrderIt != std::end(_strategies))
    {
        return (*buildOrderIt).second._buildOrder;
    }
    else
    {
        UAB_ASSERT_WARNING(false, "Strategy not found: %s, returning empty initial build order", Config::Strategy::StrategyName.c_str());
        return _emptyBuildOrder;
    }
}

void StrategyManager::addStrategy(const std::string & name, Strategy & strategy)
{
    _strategies[name] = strategy;
}

// Set _openingGroup depending on the current strategy, which in principle
// might be from the config file or from opening learning.
// This is part of initialization; it happens early on.
void StrategyManager::setOpeningGroup()
{
	auto buildOrderItr = _strategies.find(Config::Strategy::StrategyName);

	if (buildOrderItr != std::end(_strategies))
	{
		_openingGroup = (*buildOrderItr).second._openingGroup;
	}
}

const std::string & StrategyManager::getOpeningGroup() const
{
	return _openingGroup;
}

void StrategyManager::onEnd(const bool isWinner)
{
	// Nothing here for now.
}

void StrategyManager::handleUrgentProductionIssues(BuildOrderQueue & queue)
{
	if (_selfRace == BWAPI::Races::Zerg)
	{
		StrategyBossZerg::Instance().handleUrgentProductionIssues(queue);
	}
	else
	{
		// detect if there's a supply block once per second
		if ((BWAPI::Broodwar->getFrameCount() % 24 == 1) && detectSupplyBlock(queue))
		{
			if (Config::Debug::DrawBuildOrderSearchInfo)
			{
				BWAPI::Broodwar->printf("Supply block, building supply!");
			}

			queue.queueAsHighestPriority(MacroAct(BWAPI::Broodwar->self()->getRace().getSupplyProvider()));
		}

		// If we need gas, make sure it is turned on.
		// NOTE Nothing for protoss or terran turns off gas if we get too much.
		if (!queue.isEmpty())
		{
			const MacroAct nextInQueue = queue.getHighestPriorityItem().macroAct;
			if (nextInQueue.gasPrice() > BWAPI::Broodwar->self()->gas())
			{
				WorkerManager::Instance().setCollectGas(true);
			}
		}

		// If they have mobile cloaked units, get some static detection.
		if (InformationManager::Instance().enemyHasMobileCloakTech())
		{
			if (BWAPI::Broodwar->self()->getRace() == BWAPI::Races::Protoss)
			{
				if (BWAPI::Broodwar->self()->allUnitCount(BWAPI::UnitTypes::Protoss_Photon_Cannon) < 2 &&
					!queue.anyInQueue(BWAPI::UnitTypes::Protoss_Photon_Cannon) &&
					!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Protoss_Photon_Cannon))
				{
					queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Protoss_Photon_Cannon));
					queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Protoss_Photon_Cannon));

					if (BWAPI::Broodwar->self()->allUnitCount(BWAPI::UnitTypes::Protoss_Forge) == 0 &&
						!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Protoss_Forge))
					{
						queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Protoss_Forge));
					}
				}
			}
			else if (BWAPI::Broodwar->self()->getRace() == BWAPI::Races::Terran)
			{
				if (BWAPI::Broodwar->self()->allUnitCount(BWAPI::UnitTypes::Terran_Missile_Turret) < 3 &&
					!queue.anyInQueue(BWAPI::UnitTypes::Terran_Missile_Turret) &&
					!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Missile_Turret))
				{
					queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Missile_Turret));
					queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Missile_Turret));
					queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Missile_Turret));

					if (BWAPI::Broodwar->self()->allUnitCount(BWAPI::UnitTypes::Terran_Engineering_Bay) == 0 &&
						!BuildingManager::Instance().isBeingBuilt(BWAPI::UnitTypes::Terran_Engineering_Bay))
					{
						queue.queueAsHighestPriority(MacroAct(BWAPI::UnitTypes::Terran_Engineering_Bay));
					}
				}
			}
		}
	}
}

// Called to refill the production queue when it is empty.
void StrategyManager::freshProductionPlan()
{
	if (_selfRace == BWAPI::Races::Zerg)
	{
		ProductionManager::Instance().setBuildOrder(StrategyBossZerg::Instance().freshProductionPlan());
	}
}

// Return true if we're supply blocked and should build supply.
// Note: This understands zerg supply but is not used when we are zerg.
bool StrategyManager::detectSupplyBlock(BuildOrderQueue & queue)
{
	// If the _queue is empty or supply is maxed, there is no block.
	if (queue.isEmpty() || BWAPI::Broodwar->self()->supplyTotal() >= 400)
	{
		return false;
	}

	// If supply is being built now, there's no block. Return right away.
	// Terran and protoss calculation:
	if (BuildingManager::Instance().isBeingBuilt(BWAPI::Broodwar->self()->getRace().getSupplyProvider()))
	{
		return false;
	}

	// Terran and protoss calculation:
	int supplyAvailable = BWAPI::Broodwar->self()->supplyTotal() - BWAPI::Broodwar->self()->supplyUsed();

	// Zerg calculation:
	// Zerg can create an overlord that doesn't count toward supply until the next check.
	// To work around it, add up the supply by hand, including hatcheries.
	if (BWAPI::Broodwar->self()->getRace() == BWAPI::Races::Zerg) {
		supplyAvailable = -BWAPI::Broodwar->self()->supplyUsed();
		for (auto & unit : BWAPI::Broodwar->self()->getUnits())
		{
			if (unit->getType() == BWAPI::UnitTypes::Zerg_Overlord)
			{
				supplyAvailable += 16;
			}
			else if (unit->getType() == BWAPI::UnitTypes::Zerg_Egg &&
				unit->getBuildType() == BWAPI::UnitTypes::Zerg_Overlord)
			{
				return false;    // supply is building, return immediately
				// supplyAvailable += 16;
			}
			else if ((unit->getType() == BWAPI::UnitTypes::Zerg_Hatchery && unit->isCompleted()) ||
				unit->getType() == BWAPI::UnitTypes::Zerg_Lair ||
				unit->getType() == BWAPI::UnitTypes::Zerg_Hive)
			{
				supplyAvailable += 2;
			}
		}
	}

	int supplyCost = queue.getHighestPriorityItem().macroAct.supplyRequired();
	// Available supply can be negative, which breaks the test below. Fix it.
	supplyAvailable = std::max(0, supplyAvailable);

	// if we don't have enough supply, we're supply blocked
	if (supplyAvailable < supplyCost)
	{
		// If we're zerg, check to see if a building is planned to be built.
		// Only count it as releasing supply very early in the game.
		if (BWAPI::Broodwar->self()->getRace() == BWAPI::Races::Zerg
			&& BuildingManager::Instance().buildingsQueued().size() > 0
			&& BWAPI::Broodwar->self()->supplyTotal() <= 18)
		{
			return false;
		}
		return true;
	}

	return false;
}


// this will return true if any unit is on the first frame of its training time remaining
// this can cause issues for the build order search system so don't plan a search on these frames
bool StrategyManager::canPlanBuildOrderNow() const
{
	for (const auto unit : BWAPI::Broodwar->self()->getUnits())
	{
		if (unit->getRemainingTrainTime() == 0)
		{
			continue;
		}

		BWAPI::UnitType trainType = unit->getLastCommand().getUnitType();

		if (unit->getRemainingTrainTime() == trainType.buildTime())
		{
			return false;
		}
	}

	return true;
}

//write result to writeDir
void StrategyManager::writeResults()
{
	if (!Config::Modules::UsingStrategyIO)//need to add "Modules" namespace in Config, or use IO by default;
	{
		return;
	}

	std::string enemyName = BWAPI::Broodwar->enemy()->getName();
	std::replace(enemyName.begin(), enemyName.end(), ' ', '_');

	std::string enemyResultsFile = Config::Strategy::WriteDir + enemyName + ".txt";//bwapi-data/write/***.txt

	std::stringstream ss;

	for (auto & kv : _strategies)
	{
		const Strategy & strategy = kv.second;

		ss << strategy._name << " " << strategy._wins << " " << strategy._losses << "\n";
	}

	Logger::LogOverwriteToFile(enemyResultsFile, ss.str());
}
//read result from readDir
void StrategyManager::readResults()
{
	if (!Config::Modules::UsingStrategyIO)//need to add "Modules" namespace in Config, or use IO by default;
	{
		return;
	}

	std::string enemyName = BWAPI::Broodwar->enemy()->getName();
	std::replace(enemyName.begin(), enemyName.end(), ' ', '_');

	std::string enemyResultsFile = Config::Strategy::ReadDir + enemyName + ".txt";//bwapi-data/read/***.txt

	std::string strategyName;
	int wins = 0;
	int losses = 0;

	FILE *file = fopen(enemyResultsFile.c_str(), "r");
	if (file != nullptr)
	{
		char line[4096]; /* or other suitable maximum line size */
		while (fgets(line, sizeof line, file) != nullptr) /* read a line */
		{
			std::stringstream ss(line);

			ss >> strategyName;
			ss >> wins;
			ss >> losses;

			//BWAPI::Broodwar->printf("Results Found: %s %d %d", strategyName.c_str(), wins, losses);

			if (_strategies.find(strategyName) == _strategies.end())
			{
				//BWAPI::Broodwar->printf("Warning: Results file has unknown Strategy: %s", strategyName.c_str());
			}
			else
			{
				_strategies[strategyName]._wins = wins;
				_strategies[strategyName]._losses = losses;
			}
		}

		fclose(file);
	}
	else
	{
		BWAPI::Broodwar->printf("No results file found: %s", enemyResultsFile.c_str());
	}
}
//set online learning using data from readDir
void StrategyManager::setOnLineLearning()
{
	// turn it off if we don't need onLine Learning;
	//return;

	if (!Config::Modules::UsingStrategyIO)//need to add "Modules" namespace in Config, or use IO by default;
	{
		return;
	}

	const std::string & strategyName = Config::Strategy::StrategyName;
	Strategy & currentStrategy = _strategies[strategyName];

	int totalGamesPlayed = 0;
	int strategyGamesPlayed = currentStrategy._wins + currentStrategy._losses;
	double winRate = strategyGamesPlayed > 0 ? currentStrategy._wins / static_cast<double>(strategyGamesPlayed) : 0;

	// if we are using an enemy specific strategy
	if (Config::Strategy::FoundEnemySpecificStrategy)
	{
		return;
	}

	//Give a large base for a strategy if winRate is no less than 0.5
	if (strategyGamesPlayed < 5 || (strategyGamesPlayed > 0 && winRate > 0.49))
	{
		BWAPI::Broodwar->sendText("Using default strategy selection: games = %d  winrate = %.3lf", strategyGamesPlayed, winRate);
		return;
	}

	// get the total number of games played so far with this race
	for (auto & kv : _strategies)
	{
		Strategy & strategy = kv.second;
		if (strategy._race == _selfRace)
		{
			totalGamesPlayed += strategy._wins + strategy._losses;
		}
	}

	// calculate the UCB value and store the highest
	double C = 0.5;
	std::string bestUCBStrategy;
	double bestUCBStrategyVal = std::numeric_limits<double>::lowest();
	double bestUCBWinRate = std::numeric_limits<double>::lowest();
	int bestUCBStrategyGames = 0;
	for (auto & kv : _strategies)
	{
		// Do not consider strategies for another race
		Strategy & strategy = kv.second;
		if (strategy._race != _selfRace)
		{
			continue;
		}

		// Do not consider strategies that we have not played
		int sGamesPlayed = strategy._wins + strategy._losses;
		if (sGamesPlayed == 0)
		{
			continue;
		}

		double sWinRate = sGamesPlayed > 0 ? strategy._wins / static_cast<double>(sGamesPlayed) : 0;
		double ucbVal = C * sqrt(log((double)totalGamesPlayed / sGamesPlayed));
		double val = sWinRate + ucbVal;

		if (val > bestUCBStrategyVal)
		{
			bestUCBStrategy = strategy._name;
			bestUCBStrategyVal = val;
			bestUCBStrategyGames = sGamesPlayed;
			bestUCBWinRate = sWinRate;
		}
	}

	BWAPI::Broodwar->sendText("Adapting strategy selection: usb = %.3lf  games = %d  winrate = %.3lf", bestUCBStrategyVal, bestUCBStrategyGames, bestUCBWinRate);
	Config::Strategy::StrategyName = bestUCBStrategy;
}