#include "MicroZergling.h"
#include "MapTools.h"
#include "UnitUtil.h"

using namespace CDBot;

MicroZerglings::MicroZerglings()
{
}

void MicroZerglings::executeMicro(const BWAPI::Unitset & targets)
{
	const BWAPI::Unitset & zerglings = getUnits();

	BWAPI::Unitset ZerglingTargets;
	std::copy_if(targets.begin(), targets.end(), std::inserter(ZerglingTargets,ZerglingTargets.end()),
		[](BWAPI::Unit u){ return u->isVisible() && !u->isFlying() && u->getPosition().isValid(); });

	const int zerglingRange = BWAPI::UnitTypes::Zerg_Zergling.groundWeapon().maxRange();

	for (const auto zergling : zerglings)
	{
		const bool inOrderRange = zergling->getDistance(order.getPosition()) <= 3 * 32;
		BWAPI::Unit target = getTarget(zergling, ZerglingTargets);

		if (target)
		{
			// If our target is a threat, burrow at max range.
			// If our target is safe, get close first.
			// Count a turret as a threat, but not a spore colony. Zerg has overlords.
			const bool isThreat =
				target->getType() == BWAPI::UnitTypes::Terran_Missile_Turret ||
				UnitUtil::CanAttackGround(target) && !target->getType().isWorker();

			const int dist = zergling->getDistance(target);

			if (Config::Debug::DrawUnitTargetInfo) {
				BWAPI::Broodwar->drawLineMap(zergling->getPosition(), target->getPosition(), BWAPI::Colors::Purple);
				BWAPI::Broodwar->drawCircleMap(zergling->getPosition(), 12, BWAPI::Colors::Red);
				BWAPI::Broodwar->drawTextMap(zergling->getPosition() + BWAPI::Position(20, -10), "%c%d/192", white, dist);
			}

			
			const bool cannonDanger =
				target->getType() == BWAPI::UnitTypes::Protoss_Photon_Cannon &&
				zerglings.size() < 5;
			const int cannonRange = (BWAPI::UnitTypes::Protoss_Photon_Cannon).groundWeapon().maxRange();

			if (dist <= 64 ||
				isThreat && dist <= zerglingRange ||
				cannonDanger && dist > cannonRange + 32 && dist < cannonRange + 128)
			{
				// Burrow or stay burrowed.
				if (zergling->canBurrow())
				{
					zergling->burrow();
				}
				else if (zergling->isBurrowed())
				{
					if (dist <= zerglingRange)
					{
						Micro::SmartAttackUnit(zergling, target);
					}
				}
			}
			else if (!isThreat && dist > zerglingRange ||
				isThreat && dist > std::max(zerglingRange, 32 + UnitUtil::GetAttackRange(target, zergling)))
			{
				// Possibly unburrow and move.
				if (zergling->canUnburrow() && !inOrderRange && zergling->getHitPoints() > 20 && !zergling->isIrradiated())
				{
					// Unburrow only at set intervals. Reduces the burrow-unburrow frenzy.
					if (BWAPI::Broodwar->getFrameCount() % 24 == 0)
					{
						zergling->unburrow();
					}
				}
				else if (zergling->isBurrowed())
				{
					if (dist <= zerglingRange)
					{
						Micro::SmartAttackUnit(zergling, target);
					}
				}
				else
				{
					Micro::SmartMove(zergling, target->getPosition());
				}
			}
			else
			{
				// In between "close enough to burrow" and "far enough to unburrow".
				// Keep doing whatever we were doing.
				if (zergling->isBurrowed())
				{
					if (dist <= zerglingRange)
					{
						Micro::SmartAttackUnit(zergling, target);
					}
				}
				else
				{
					Micro::SmartMove(zergling, target->getPosition());
				}
			}
		}
		else
		{
			if (Config::Debug::DrawUnitTargetInfo) {
				BWAPI::Broodwar->drawLineMap(zergling->getPosition(), order.getPosition(), BWAPI::Colors::White);
			}

			// No target assigned.
			// Move toward the order position and burrow there.
			if (inOrderRange) {
				if (zergling->canBurrow())
				{
					zergling->burrow();
				}
			}
			else
			{
				if (zergling->canUnburrow())
				{
					zergling->unburrow();
				}
				else
				{
					Micro::SmartMove(zergling, order.getPosition());
				}
			}
		}
	}
}
BWAPI::Unit MicroZerglings::getTarget(BWAPI::Unit zergling, const BWAPI::Unitset & targets)
{
	if (targets.empty())
	{
		return nullptr;
	}

	const int zerglingRange = BWAPI::UnitTypes::Zerg_Zergling.groundWeapon().maxRange();

	BWAPI::Unitset targetsInRange;
	for (const auto target : targets)
	{
		if (zergling->getDistance(target) <= zerglingRange)
		{
			targetsInRange.insert(target);
		}
	}


	const BWAPI::Unitset & newTargets = targetsInRange.empty() ? targets : targetsInRange;

	int highPriority = 0;
	int closestDist = 99999;
	BWAPI::Unit bestTarget = nullptr;

	for (const auto target : newTargets)
	{
		int distance = zergling->getDistance(target);
		int priority = getAttackPriority(target);

		if ((priority > highPriority) || (priority == highPriority && distance < closestDist))
		{
			closestDist = distance;
			highPriority = priority;
			bestTarget = target;
		}
	}

	return bestTarget;
}

//  Only ground units are passed in as potential targets.
int MicroZerglings::getAttackPriority(BWAPI::Unit target) const
{
	BWAPI::UnitType targetType = target->getType();

	// A spider mine on the attack is a time-critical target.
	if (targetType == BWAPI::UnitTypes::Terran_Vulture_Spider_Mine && !target->isBurrowed())
	{
		return 10;
	}
	if (targetType == BWAPI::UnitTypes::Protoss_High_Templar ||
		targetType == BWAPI::UnitTypes::Protoss_Reaver)
	{
		return 10;
	}

	// highest priority is something that can attack us or aid in combat
	if (UnitUtil::CanAttackGround(target) && !target->getType().isWorker())
	{
		return 9;
	}
	// turrets are just as bad (cannons are threats, zerg has overlords so spores are meh)
	if (targetType == BWAPI::UnitTypes::Terran_Missile_Turret)
	{
		return 9;
	}
	if (targetType.isWorker())
	{
		if (target->isConstructing() || target->isRepairing())
		{
			return 9;
		}
		return 8;
	}
	if (targetType == BWAPI::UnitTypes::Terran_Medic)
	{
		return 8;
	}
	// next is special buildings
	if (targetType == BWAPI::UnitTypes::Zerg_Spore_Colony)
	{
		return 6;
	}
	if (targetType == BWAPI::UnitTypes::Terran_Comsat_Station)
	{
		return 6;
	}
	if (targetType == BWAPI::UnitTypes::Zerg_Spawning_Pool)
	{
		return 5;
	}
	if (targetType == BWAPI::UnitTypes::Protoss_Pylon || targetType == BWAPI::UnitTypes::Protoss_Observatory)
	{
		return 5;
	}
	// next is buildings that cost gas
	if (targetType.gasPrice() > 0)
	{
		return 4;
	}
	if (targetType.mineralPrice() > 0)
	{
		return 3;
	}
	// then everything else
	return 1;
}