/*
* Tyr is an AI for StarCraft: Broodwar, 
* 
* Please visit https://github.com/SimonPrins/Tyr for further information.
* 
* Copyright 2015 Simon Prins
*
* This file is part of Tyr.
* Tyr is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 2 of the License, or
* (at your option) any later version.
* Tyr is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with Tyr.  If not, see http://www.gnu.org/licenses/.
*/


package com.tyr.tasks;

import java.util.ArrayList;

import com.tyr.DebugMessages;
import com.tyr.Tyr;

import bwapi.Game;
import bwapi.Player;
import bwapi.Race;


/**
 * This class keeps track of all tasks that may need our attention.
 * It will try to find an appropriate solution to all tasks that need some attention.
 */
public class TaskManager 
{
	/**
	 * List of all the task that may be performed.
	 */
	public ArrayList<Task> potentialTasks = new ArrayList<Task>();
	
	/**
	 * The scan task that may be performed.
	 */
	private ScanTask scanTask;
	
	/**
	 * The observer task that may be performed.
	 */
	private ObserverTask observerTask;
	
	/**
	 * This class keeps track of all tasks that may need our attention.
 	 * It will try to find an appropriate solution to all tasks that need some attention.
	 */
	public TaskManager()
	{
		// Initialize all standard tasks.
		potentialTasks.add(new DefendMainTask(false));
		potentialTasks.add(new DefendMainTask(true));
		potentialTasks.add(new AttackTask());
		potentialTasks.add(new ClearMineralTask());
		//potentialTasks.add(new BCTask());
		
		scanTask = new ScanTask();
		if (Tyr.self.getRace() == Race.Protoss)
			observerTask = new ObserverTask();
	}

	/**
	 * This method should be called each frame.
	 * @param game The game.
	 * @param self The player controlled by our bot.
	 * @param bot The instance of Tyr that is currently playing the game.
	 */
	public void onFrame(Game game, Player self, Tyr bot)
	{
		// Do we need to run our observer task?
		if (observerTask != null && observerTask.isRequired(game, self, bot))
		{
			if (!observerTask.isBeingSolved())
				observerTask.findSolution(game, self, bot);
			
			if (observerTask.isBeingSolved())
				observerTask.solve(game, self, bot);
		}
		
		// Do we need to start our scanner?
		if (scanTask != null && scanTask.isRequired(game, self, bot))
		{
			if (!scanTask.isBeingSolved())
				scanTask.findSolution(game, self, bot);
			
			if (scanTask.isBeingSolved())
			{
				scanTask.solve(game, self, bot);
				DebugMessages.addMessage("Scanning.");
				
				// We want the scanner to have full control of our units.
				return;
			}
		}
		
		for(Task task : potentialTasks)
		{
			// Check if the task needs to be executed.
			boolean isRequired = task.isRequired(game, self, bot);
			
			// If the task does need to be executed, the task is asked to find an appropriate solution.
			if (isRequired && !task.isBeingSolved())
				task.findSolution(game, self, bot);
			
			
			// If the task needs to be execute and a solution has been found, we require the task to solve itself.
			if (isRequired && task.isBeingSolved())
				task.solve(game, self, bot);
		}
	}
	
	/**
	 * Disables the scanner.
	 */
	public void disableScanner()
	{
		scanTask = null;
	}
	
}
