/*
* Tyr is an AI for StarCraft: Broodwar, 
* 
* Please visit https://github.com/SimonPrins/Tyr for further information.
* 
* Copyright 2015 Simon Prins
*
* This file is part of Tyr.
* Tyr is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 2 of the License, or
* (at your option) any later version.
* Tyr is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
* You should have received a copy of the GNU General Public License
* along with Tyr.  If not, see http://www.gnu.org/licenses/.
*/


package com.tyr.tasks;

import com.tyr.Tyr;

import bwapi.Game;
import bwapi.Player;

/**
 * This class implements a task that may have to be performed by the bot.
 * These include tasks such as defending a base or harassing an enemy base.
 * 
 * @author Simon
 *
 */
public abstract class Task
{
	/**
	 * The solution to solve this task.
	 * This is null if the task has no solution.
	 */
	protected Solution solution;
	
	/**
	 * Determines if this is a task that has to be performed right now.
	 * For instance, if a base is not being attacked, then it doesn't have to be defended either.
	 * @param game The game.
	 * @param self The player controlled by our bot.
	 * @param bot The instance of Tyr that is currently playing the game.
	 * @return
	 */
	public abstract boolean isRequired(Game game, Player self, Tyr bot);
	
	/**
	 * Determines whether the task is being solved.
	 * This is true if there is an active solution to solve the task.
	 */
	public boolean isBeingSolved()
	{
		return solution != null;
	}

	/**
	 * When the solution has to be performed, this method will be called each frame.
	 * @param game The game.
	 * @param self The player controlled by our bot.
	 * @param bot The instance of Tyr that is currently playing the game.
	 */
	public void solve(Game game, Player self, Tyr bot)
	{
		if (solution != null)
			solution.onFrame(game, self, bot);
	}

	/*
	 * This method will try to find an appropriate solution to this task.
	 * This method may only be called when isRequired returns true and isBeingSolved returns false.
	 */
	public abstract void findSolution(Game game, Player self, Tyr bot);
	
}
